#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "edv_recycle_bin_sync.h"
#include "edv_utils_gtk.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_recycle_bin_sync_get_error(EDVCore *core);
static void edv_recycle_bin_sync_copy_error_message(
        EDVCore *core,
        const gchar *msg
);

static gboolean edv_recycle_bin_sync_is_string_all_digits(const gchar *s);

static gint edv_recycle_bin_sync_index_file(
	EDVCore *core,
	const gchar *index_path,
	const gchar *recycle_bin_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status_rtn,
	void (*status_message_cb)(
			GtkWidget *,
			const gchar *,
			gpointer
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
			GtkWidget *,
			const gfloat,
			gpointer
	),
	gpointer status_progress_data
);
static gint edv_recycle_bin_sync_objects(
	EDVCore *core,
	const gchar *index_path,
	const gchar *recycle_bin_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status_rtn,
	void (*status_message_cb)(
			GtkWidget *,
			const gchar *,
			gpointer
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
			GtkWidget *,
			const gfloat,
			gpointer
	),
	gpointer status_progress_data
);

gint edv_recycle_bin_sync(
	EDVCore *core,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	void (*status_message_cb)(
			GtkWidget *,		/* NULL */
			const gchar *,		/* Message */
			gpointer		/* status_message_data */
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
			GtkWidget *,		/* NULL */
			gfloat,			/* Progress */
			gpointer		/* status_progress_data */
	),
	gpointer status_progress_data
);


#define STATUS_MESSAGE(_msg_)	{		\
 if(status_message_cb != NULL) {		\
  status_message_cb(				\
   NULL,					\
   (_msg_),					\
   status_message_data				\
  );						\
  gtk_events_process();				\
 }						\
}
#define STATUS_PROGRESS(_vf_)	{		\
 if(status_progress_cb != NULL) {		\
  status_progress_cb(				\
   NULL,					\
   (_vf_),					\
   status_progress_data				\
  );						\
  gtk_events_process();				\
 }						\
}

#define MESSAGE_ERROR(_title_,_msg_)	{	\
 if(interactive) {				\
  edv_play_sound_error(core);			\
  edv_message_error(				\
   (_title_), (_msg_), NULL, toplevel		\
  );						\
} }

#define MESSAGE_WARNING(_title_,_msg_)	{	\
 if(interactive) {				\
  edv_play_sound_warning(core);			\
  edv_message_warning(				\
   (_title_), (_msg_), NULL, toplevel		\
  );						\
} }

#define MESSAGE(_title_,_msg_)	{		\
 if(interactive) {				\
 edv_play_sound_info(core);			\
 CDialogSetTransientFor(toplevel);		\
 CDialogGetResponseIconData(			\
  (_title_), (_msg_), NULL,			\
  edv_get_recycle_bin_icon_data(			\
   EDV_ICON_SIZE_32,				\
   1,						\
   NULL						\
  ),						\
  CDIALOG_BTNFLAG_OK,				\
  CDIALOG_BTNFLAG_OK				\
 );						\
 CDialogSetTransientFor(NULL);			\
} }


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message.
 */
const gchar *edv_recycle_bin_sync_get_error(EDVCore *core)
{
        return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Sets the last error message.
 */
static void edv_recycle_bin_sync_copy_error_message(
        EDVCore *core,
        const gchar *msg
)
{
        if(core == NULL)
                return;

        g_free(core->last_error_buf);
        core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Checks if all the characters in the string are digits.
 *
 *	If the string is empty or NULL then FALSE will be returned.
 */
static gboolean edv_recycle_bin_sync_is_string_all_digits(const gchar *s)
{
	if(s == NULL)
		return(FALSE);

	if(*s == '\0')
		return(FALSE);

	while(*s != '\0')
	{
		if(!isdigit((int)*s))
			return(FALSE);
		s++;
	}

	return(TRUE);
}


/*
 *	Scans the index file, fixes it as needed, and removes any
 *	invalid recycled object entries from it.
 *
 *	Invalid recycled object entries include those that refer to
 *	non-existant VFS objects or VFS objects that are not files.
 *
 *	Returns the number of problems encountered.
 */
static gint edv_recycle_bin_sync_index_file(
	EDVCore *core,
	const gchar *index_path,
	const gchar *recycle_bin_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status_rtn,
	void (*status_message_cb)(
			GtkWidget *,
			const gchar *,
			gpointer
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
			GtkWidget *,
			const gfloat,
			gpointer
	),
	gpointer status_progress_data
)
{

typedef struct {
	gulong		index;
	gchar		*name;
	gint		error_code;		/* errno describing problem */
	gchar		*details;
} EDVRecycleBinSyncProblem;
#define EDV_RECYCLE_BIN_SYNC_PROBLEM(p)	((EDVRecycleBinSyncProblem *)(p))

	gint nproblems = 0;
	gulong		index,
			index_size;
	gfloat coeff;
	gchar *recycled_obj_vfs_path;
	GList *problems_list;
	EDVRecycleBinIndex *rp = edv_recycle_bin_index_open(index_path);
	EDVVFSObject *vfs_obj;
	EDVRecycledObject *obj;

	/* Unable to open the recycle bin index file? */
	if(rp == NULL)
	{
		const gchar *error_msg = edv_recycle_bin_index_get_error();
		if(error_msg != NULL)
		{
			gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de abrir el archivo del ndice de cajn de recirculacin:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable d'ouvrir recycler le fichier d'index d'huche:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig offen, behlter index akte wiederzuverwerten:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per aprire il file di indice di contenitore per la raccolta differenziata:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam open bak index dossier te recyclen:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de abrir recicla arquivo de ndice de caixa:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls pne resirkulasjons beholder indeksarkiv:\n\
\n\
    %s\n\
\n\
%s."
#else
"Unable to open the recycle bin index file:\n\
\n\
    %s\n\
\n\
%s."
#endif
				,
				index_path,
				error_msg
			);
			MESSAGE_ERROR(
"Recycle Bin Error",
				msg
			);
			g_free(msg);
		}
		if(*status_rtn == 0)
			*status_rtn = -1;
		nproblems++;
		return(nproblems);
	}

	/* Get the index file's statistics */
	vfs_obj = edv_recycle_bin_index_stat(rp);
	if(vfs_obj != NULL)
	{
		index_size = vfs_obj->size;
		edv_vfs_object_delete(vfs_obj);
	}
	else
	{
		index_size = 0l;
	}


	STATUS_MESSAGE(
"Reading recycle bin index file..."
	);

	problems_list = NULL;

/* Adds the problem to the problems list */
#define RECORD_PROBLEM(_index_,_name_,_error_code_,_details_) {	\
 EDVRecycleBinSyncProblem *p = EDV_RECYCLE_BIN_SYNC_PROBLEM(	\
  g_malloc0(sizeof(EDVRecycleBinSyncProblem))			\
 );								\
 if(p != NULL) {						\
  p->index = (_index_);						\
  p->name = STRDUP(_name_);					\
  p->error_code = (_error_code_);				\
  p->details = STRDUP(_details_);				\
  problems_list = g_list_append(				\
   problems_list,						\
   p								\
  );								\
 }								\
}

	/* Iterate through all the recycled objects referenced in
	 * the recycle bin index file and check for problems with
	 * each one, record any problems in the problems list
	 */
	for(obj = edv_recycle_bin_index_next(rp);
	    obj != NULL;
	    obj = edv_recycle_bin_index_next(rp)
	)
	{
		index = obj->index;

		coeff = (index_size > 0l) ?
			((gfloat)edv_recycle_bin_index_tell(rp) / (gfloat)index_size) : 0.0f;
		STATUS_PROGRESS((0.0f / 4.0f) + (coeff / 4.0f));

		/* Generate the full path to this recycled object's
		 * VFS object
		 *
		 * This should work identically to calling
		 * edv_recycle_bin_index_get_recycled_object_path()
		 * but faster
		 */
		recycled_obj_vfs_path = g_strdup_printf(
			"%s%c%ld",
			recycle_bin_path,
			G_DIR_SEPARATOR,
			index
		);

		/* Get the VFS object of the recycled object's statistics */
		vfs_obj = edv_vfs_object_lstat(recycled_obj_vfs_path);
		if(vfs_obj != NULL)
		{
			/* Is a file? */
			if(vfs_obj->type == EDV_OBJECT_TYPE_FILE)
			{
				const EDVPermissionFlags permissions = vfs_obj->permissions;

				/* Not owned by user? */
				if(vfs_obj->owner_id != core->effective_user_id)
				{
					/* Attempt to set the ownership
					 * to the user of this process
					 */
					(void)edv_lchown(
						recycled_obj_vfs_path,
						core->effective_user_id,
						vfs_obj->group_id
					);
				}

				/* Not only readable and writable by the owner? */
				if(!(permissions & EDV_PERMISSION_UR) ||
				   !(permissions & EDV_PERMISSION_UW) ||
				   (permissions & EDV_PERMISSION_UX) ||
				   (permissions & EDV_PERMISSION_GR) ||
				   (permissions & EDV_PERMISSION_GW) ||
				   (permissions & EDV_PERMISSION_GX) ||
				   (permissions & EDV_PERMISSION_OR) ||
				   (permissions & EDV_PERMISSION_OW) ||
				   (permissions & EDV_PERMISSION_OX) ||
				   (permissions & EDV_PERMISSION_SETUID) ||
				   (permissions & EDV_PERMISSION_SETGID) ||
				   (permissions & EDV_PERMISSION_STICKY)
				)
				{
					/* Attempt to set the object
					 * only readable and writable
					 * by the owner
					 */
					(void)edv_permissions_set(
						recycled_obj_vfs_path,
						EDV_PERMISSION_UR | EDV_PERMISSION_UW
					);
				}

				/* File with inconsistent size? */
				if((obj->type == EDV_OBJECT_TYPE_FILE) &&
				   (obj->size != vfs_obj->size)
				)
				{
					gchar	*size1 = STRDUP(edv_str_size_delim(obj->size)),
						*size2 = STRDUP(edv_str_size_delim(vfs_obj->size)),
						*details = g_strdup_printf(
"Has a size of %s %s in the index file\n\
but %s %s of data on the VFS.",
						size1,
						(obj->size == 1l) ? "byte" : "bytes",
						size2,
						(vfs_obj->size == 1l) ? "byte" : "bytes"
					);
					g_free(size1);
					g_free(size2);
					RECORD_PROBLEM(
						index,
						obj->name,
						EINVAL,
						details
					);
					g_free(details);
				}
			}
			else
			{
				/* This recycled object's VFS object
				 * is not a file, all recycled
				 * objects must be represented by
				 * a file on the VFS, so add its
				 * index and name to the list of
				 * missing objects
				 */
				RECORD_PROBLEM(
					index,
					obj->name,
					EINVAL,
"Has a corresponding VFS object that is not a file."
				);
			}

			edv_vfs_object_delete(vfs_obj);
		}
		else
		{
			/* This recycled object's VFS object does not
			 * exist, add its index and name to the list
			 * of missing objects
			 */
			RECORD_PROBLEM(
				index,
				obj->name,
				ENOENT,
"Has no corresponding file on the VFS."
			);
		}

		g_free(recycled_obj_vfs_path);
	}

#undef RECORD_PROBLEM

	/* Close the recycle bin index file */
	edv_recycle_bin_index_close(rp);

 
	/* At this point we now have a list of missing recycled objects
	 * refered to by their index numbers, begin querying user for
	 * their removal
	 */
	if(problems_list != NULL)
	{
		gboolean got_cancel = FALSE;
		gint	i,
			response;
		const gint m = g_list_length(problems_list);
		GList *glist;
		EDVRecycleBinSyncProblem *problem;

		nproblems += m;

		STATUS_MESSAGE(
"Handling recycled objects with problems..."
		);

		/* Handle each problem */
		for(glist = problems_list, i = 0;
		    glist != NULL;
		    glist = g_list_next(glist), i++
		)
		{
			problem = EDV_RECYCLE_BIN_SYNC_PROBLEM(glist->data);
			if(problem == NULL)
				continue;

			/* Report the progress */
			if(m > 0)
				STATUS_PROGRESS(
					(1.0f / 4.0f) +
					(((gfloat)i / (gfloat)m) / 4.0f)
				);

			/* Query user for removal of this index */
			if(interactive && !(*yes_to_all))
			{
				gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Refered objetivo reciclado de \"%s\" a por el ndice:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
Usted quiere quitar su referencia del archivo del ndice?"
#elif defined(PROG_LANGUAGE_FRENCH)
"L'objet recycl \"%s\" refered  par l'index:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
Voulez-vous enlever sa rfrence du fichier d'index?"
#elif defined(PROG_LANGUAGE_GERMAN)
"Wiederverwertet Objekt \"%s\" refered zu durch Index:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
Sie wollen herausnehmen seine verweisung von der Index akte nicht?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il refered di \"%s\" di oggetto riciclato a dall'indice:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
Lei vuole togliere il suo riferimento dal file di indice?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Gerecyclde voorwerp \"%s\" refered te door index:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
U wil verwijderen zijne referentie van het index dossier?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Refered reciclado de \"%s\" de objeto a por ndice:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
Quer retirar seua referncia do arquivo de ndice?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Resirkuleredd objekt \"%s\" refered til ved indeks:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
Gjr De fjerner dets referanse fra indeksarkivet?"
#else
"Recycled object:\n\
\n\
    %s\n\
\n\
Refered to by index:\n\
\n\
    #%ld\n\
\n\
%s\n\
\n\
Do you want to remove its reference from the index file?"
#endif
					,
					problem->name,
					problem->index,
					problem->details
				);
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					"Recycle Bin Sync Problem",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES |
						CDIALOG_BTNFLAG_YES_TO_ALL |
						CDIALOG_BTNFLAG_NO |
						CDIALOG_BTNFLAG_CANCEL,
					CDIALOG_BTNFLAG_YES
				);
				CDialogSetTransientFor(NULL);
				g_free(msg);
			}
			else
			{
				response = CDIALOG_RESPONSE_YES_TO_ALL;
			}
			/* Handle response */
			switch(response)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
				*yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
				/* Yes, remove the index from the index file */
				(void)edv_recycle_bin_index_remove(
					index_path,
					problem->index
				);
				break;
			  case CDIALOG_RESPONSE_NO:
				/* No, do not remove the index from the index file */
				break;
			  default:
				/* Cancel */
				if(*status_rtn == 0)
					*status_rtn = -4;
				got_cancel = TRUE;
				break;
			}
			if(got_cancel)
				break;
		}

		if(*status_rtn == 0)
			*status_rtn = -1;
	}

	STATUS_MESSAGE(NULL);

	/* Delete the problems list */
	if(problems_list != NULL)
	{
		GList *glist;
		EDVRecycleBinSyncProblem *problem;
		for(glist = problems_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			problem = EDV_RECYCLE_BIN_SYNC_PROBLEM(glist->data);
			if(problem == NULL)
				continue;

			g_free(problem->name);
			g_free(problem->details);
			g_free(problem);
		}
		g_list_free(problems_list);
	}

	return(nproblems);
}


/*
 *	Scans the recycle bin directory and checks if any recycled
 *	objects are not listed in the index file, adds non-referenced
 *	recycled objects to the index file.
 *
 *	Returns the number of problems encountered.
 */
static gint edv_recycle_bin_sync_objects(
	EDVCore *core,
	const gchar *index_path,
	const gchar *recycle_bin_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status_rtn,
	void (*status_message_cb)(
			GtkWidget *,
			const gchar *,
			gpointer
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
			GtkWidget *,
			const gfloat,
			gpointer
	),
	gpointer status_progress_data
)
{
	gint		i,
			nproblems = 0,
			nobjs;
	const gchar	*obj_name,
			*idx_file_name = g_basename(index_path);
	GList		*glist,
			*names_list,
			*idx_list = NULL;
	EDVRecycleBinIndex *rp = edv_recycle_bin_index_open(index_path);

	if(STRISEMPTY(idx_file_name))
		idx_file_name = "";

	/* Able to open the recycle bin index file? */
	if(rp != NULL)
	{
		gulong index_size;
		gfloat coeff;
		EDVRecycledObject *obj;

		EDVVFSObject *vfs_obj = edv_recycle_bin_index_stat(rp);
		if(vfs_obj != NULL)
		{
			index_size = vfs_obj->size;
			edv_vfs_object_delete(vfs_obj);
		}
		else
		{
			index_size = 01;
		}

		STATUS_MESSAGE(
"Reading recycle bin index file..."
		);

		/* Get the list of indices */
		for(obj = edv_recycle_bin_index_next(rp);
		    obj != NULL;
		    obj = edv_recycle_bin_index_next(rp)
		)
		{
			coeff = (index_size > 0l) ?
				((gfloat)edv_recycle_bin_index_tell(rp) / (gfloat)index_size) : 0.0f;
			STATUS_PROGRESS((2.0f / 4.0f) + (coeff / 4.0f));
			idx_list = g_list_append(
				idx_list,
				(gpointer)obj->index
			);
		}

		/* Close the recycle bin index file */
		edv_recycle_bin_index_close(rp);
	}

	STATUS_MESSAGE(
"Checking for inert objects..."
	);

	/* Get the listing of every object in the recycled bin
	 * directory
	 */
	names_list = edv_directory_list(
		recycle_bin_path,
		FALSE,				/* Unsorted */
		FALSE				/* Exclude notations */
	);
	nobjs = g_list_length(names_list);
	for(glist = names_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		STATUS_PROGRESS(
			(3.0f / 4.0f) +
			(((gfloat)i / (gfloat)nobjs) / 4.0f)
		);

		obj_name = (const gchar *)glist->data;
		if(obj_name == NULL)
			continue;

		/* Skip the index file */
		if(!strcmp((const char *)obj_name, (const char *)idx_file_name))
			continue;

		/* Skip the EDV_NAME_DIRECTORY_PROPERTIES_FILE file */
		if(!strcmp((const char *)obj_name, EDV_NAME_DIRECTORY_PROPERTIES_FILE))
			continue;

		/* Is this object's name not all digits? */
		if(!edv_recycle_bin_sync_is_string_all_digits(obj_name))
		{
			/* This object is probably not a recycled object and
			 * does not belong in this directory since it does
			 * not start with a number
			 *
			 * Query the user to move this object to the user's
			 * home directory
			 */
			gint response;
			gboolean got_cancel = FALSE;

			nproblems++;
			if(*status_rtn == 0)
				*status_rtn = -1;

			if(interactive && !(*yes_to_all))
			{
				gchar *s = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Un objeto denomin \"%s\" no aparece pertenecer\n\
en la gua reciclada de objetos.\n\
\n\
Mueve este objeto a la gua del hogar?"
#elif defined(PROG_LANGUAGE_FRENCH)
"Un objet a nomm \"%s\" n'apparat pas d'appartenir\n\
dans l'annuaire d'objets recycl.\n\
\n\
Transfre cet objet  l'annuaire de maison?"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ein objekt erscheint \"%s\" nicht hat genannt,\n\
im wiederverwerteten objekten verzeichnis zu gehren.\n\
\n\
Bewegen sie dieses objekt zum heim verzeichnis?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Un oggetto ha dato un nome a \"%s\" non appare\n\
appartenere nell'elenco di oggetti riciclato.\n\
\n\
Muove quest'oggetto all'elenco di casa?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Een voorwerp verschijnt \"%s\" niet noemde\n\
om in de gerecyclde voorwerpen gids te horen.\n\
\n\
Beweeg deze voorwerp aan de huis gids?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Um objeto nomeou \"%s\" no aparece pertencer\n\
no guia reciclado de objetos.\n\
\n\
Mova este objeto ao guia de lar?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Et objekt kalt \"%s\" kommer ikke fram hre\n\
til i den resirkuleredde objektkatalog.\n\
\n\
Flytt dette objektet til hjemkatalogen?"
#else
"An inert object named \"%s\" was found in the recycle bin.\n\
\n\
Move this object to your home directory?"
#endif
					,
					obj_name
				);
				edv_play_sound_question(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"El Objeto Inerte Encontr",
s,
"Un objeto se ha encontrado en la gua reciclada de\n\
objetos que no aparece para pertenecer all. Para ayudar\n\
a mantener la gua reciclada de objetos lo limpia es\n\
recomienda que el objeto se sea mudado de la gua\n\
reciclada de objetos a la gua del hogar.",
#elif defined(PROG_LANGUAGE_FRENCH)
"L'Objet Inerte A Trouv",
s,
"Un objet a t trouv dans l'annuaire d'objets recycl\n\
qui n'apparat pas d'appartenir l-bas. Pour aider garder\n\
les objets recycl que l'annuaire nettoie c'est recommandent\n\
que l'objet ait sorti de l'annuaire d'objets recycl \n\
l'annuaire de maison.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Trges Objekt Hat Gefunden",
s,
"Ein objekt ist im wiederverwerteten objekten\n\
verzeichnis gefunden worden, das nicht erscheint, dort zu\n\
gehren. Zu helfen, um das wiederverwertete objekte\n\
verzeichnis zu behalten, um es zu reinigen, empfiehlt ist,\n\
da das Objekt aus dem wiederverwerteten objekten verzeichnis\n\
zum heim verzeichnis bewegt ist.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Oggetto Inerte Ha Trovato",
s,
"Un oggetto  stato trovato nell'elenco di oggetti\n\
riciclato che non appare appartenere l. Per aiutare tenere\n\
l'elenco di oggetti riciclato pulisce  raccomanda che\n\
l'oggetto  fuori mosso dell'elenco di oggetti riciclato\n\
all'elenco di casa.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Inert Voorwerp Vond",
s,
"Een voorwerp is in de gerecyclde voorwerpen gids gevonden\n\
dat niet verschijnt om daar te horen. Om kost de gerecyclde\n\
voorwerpen gids volkomen het is, aanraadt te helpen dat het\n\
voorwerp uit de gerecyclde voorwerpen gids aan de huis gids\n\
bewogen is.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto Inerte Achou",
s,
"Um objeto foi achado no guia reciclado de objetos que no\n\
aparece pertencer a. Ajudar mantem o guia reciclado de\n\
objetos limpar  recomenda que o objeto  movido para fora do\n\
guia reciclado de objetos ao guia de lar.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nytralt Objekt Funnet",
s,
"Et objekt finner i den resirkuleredde objektkatalog som\n\
ikke kommer fram hre til der. Hjelpe beholder den\n\
resirkuleredde objektkatalog rengjr det er anbefaler at\n\
objektet flyttet ut av den resirkuleredde objektkatalog til\n\
hjemkatalogen.",
#else
"Inert Object Found",
s,
"An object has been found in the recycle bin directory that\n\
does not appear to belong there. To help keep the recycle\n\
bin directory clean it is recommend that the object be moved\n\
out of the recycle bin directory and into your home directory\n\
for further examination.",
#endif
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES |
					CDIALOG_BTNFLAG_YES_TO_ALL |
					CDIALOG_BTNFLAG_NO |
					CDIALOG_BTNFLAG_CANCEL |
					CDIALOG_BTNFLAG_HELP,
					CDIALOG_BTNFLAG_YES
				);
				CDialogSetTransientFor(NULL);
				g_free(s);
			}
			else
			{
				response = CDIALOG_RESPONSE_YES_TO_ALL;
			}
			/* Handle response */
			switch(response)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
				*yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
				/* Yes, move the object */
				if(TRUE)
				{
					gchar	*src_path = edv_paths_join(
						recycle_bin_path,
						obj_name
					),
						*tar_path = edv_paths_join(
						core->home_path,
						obj_name
					);
					if(edv_rename(
						src_path,
						tar_path
					))
					{
						const gint error_code = (gint)errno;
						gchar *s = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de mover objeto:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable de dplacer l'objet:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, objekt zu bewegen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per muovere l'oggetto:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam voorwerp te bewegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de mover objeto:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls flytte objekt:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n\
\n\
%s."
#else
"Unable to move object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s."
#endif
							,
							obj_name,
							core->home_path,
							g_strerror(error_code)
						);
						MESSAGE_ERROR(
"Move Error",
							s
						);
						g_free(s);
						if(*status_rtn == 0)
							*status_rtn = -1;
					}
					g_free(tar_path);
					g_free(src_path);
				}
				break;
			  case CDIALOG_RESPONSE_NO:
				/* No, do not move the object */
				break;
			  default:
				/* Cancel */
				if(*status_rtn == 0)
					*status_rtn = -4;
				got_cancel = TRUE;
				break;
			}
			if(got_cancel)
				break;

			continue;
		}

		/* Each recycled object's name is a number, so get the
		 * numeric value of the object's name as the recycled
		 * object's index
		 */
		if(TRUE)
		{
			const gulong index = (gulong)atol((const char *)obj_name);
			gboolean in_list = FALSE;
			GList *glist;

			/* Check if this recycled object is in the list */
			for(glist = idx_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				if((gulong)glist->data == index)
				{
					in_list = TRUE;
					break;
				}
			}

			/* Recycled object in the index file? */
			if(in_list)
			{
				/* This object belongs in the recycle
				 * bin directory
				 *
				 * Generate the full path to this
				 * recycled object's VFS object
				 *
				 * This should work identically to
				 * calling
				 * edv_recycle_bin_index_get_recycled_object_path()
				 * but faster
				 */
				gchar *path = g_strconcat(
					recycle_bin_path,
					G_DIR_SEPARATOR_S,
					obj_name,
					NULL
				);

				/* Set the recycled object to be only readable and
				 * writable by the owner
				 */
				if(edv_permissions_set(
					path,
					EDV_PERMISSION_UR | EDV_PERMISSION_UW
				))
				{
					const gint error_code = (gint)errno;
					gchar *s = g_strdup_printf(
"%s:\n\
\n\
    %s",
						g_strerror(error_code),
						obj_name
					);
					MESSAGE_ERROR(
						"CHMod Failed",
						s
					);
					g_free(s);
					if(*status_rtn == 0)
						*status_rtn = -1;
				}

				g_free(path);
			}
			else
			{
				/* This object probably does not belong
				 * in the recycle bin directory even
				 * though its name is numeric
				 *
				 * Query the user to move this object
				 * to the user's home directory
				 */
				gint response;
				gboolean got_cancel = FALSE;

				nproblems++;
				if(*status_rtn == 0)
					*status_rtn = -1;

				if(interactive && !(*yes_to_all))
				{
					gchar *s = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Un objeto denomin \"%s\" no aparece pertenecer\n\
en la gua reciclada de objetos.\n\
\n\
Mueve este objeto a la gua del hogar?"
#elif defined(PROG_LANGUAGE_FRENCH)
"Un objet a nomm \"%s\" n'apparat pas d'appartenir\n\
dans l'annuaire d'objets recycl.\n\
\n\
Transfre cet objet  l'annuaire de maison?"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ein objekt erscheint \"%s\" nicht hat genannt,\n\
im wiederverwerteten objekten verzeichnis zu gehren.\n\
\n\
Bewegen sie dieses objekt zum heim verzeichnis?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Un oggetto ha dato un nome a \"%s\" non appare\n\
appartenere nell'elenco di oggetti riciclato.\n\
\n\
Muove quest'oggetto all'elenco di casa?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Een voorwerp verschijnt \"%s\" niet noemde\n\
om in de gerecyclde voorwerpen gids te horen.\n\
\n\
Beweeg deze voorwerp aan de huis gids?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Um objeto nomeou \"%s\" no aparece pertencer\n\
no guia reciclado de objetos.\n\
\n\
Mova este objeto ao guia de lar?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Et objekt kalt \"%s\" kommer ikke fram hre\n\
til i den resirkuleredde objektkatalog.\n\
\n\
Flytt dette objektet til hjemkatalogen?"
#else
"An object named \"%s\" in the recycle bin\n\
directory does not have an entry in the recycle bin\n\
index file.\n\
\n\
Move this object to the home directory?"
#endif
						,
						obj_name
					);
					edv_play_sound_warning(core);
					CDialogSetTransientFor(toplevel);
					response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"El Objeto Inerte Encontr",
s,
"Un objeto se ha encontrado en la gua reciclada de\n\
objetos que no aparece para pertenecer all. Para ayudar\n\
a mantener la gua reciclada de objetos lo limpia es\n\
recomienda que el objeto se sea mudado de la gua\n\
reciclada de objetos a la gua del hogar.",
#elif defined(PROG_LANGUAGE_FRENCH)
"L'Objet Inerte A Trouv",
s,
"Un objet a t trouv dans l'annuaire d'objets recycl\n\
qui n'apparat pas d'appartenir l-bas. Pour aider garder\n\
les objets recycl que l'annuaire nettoie c'est recommandent\n\
que l'objet ait sorti de l'annuaire d'objets recycl \n\
l'annuaire de maison.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Trges Objekt Hat Gefunden",
s,
"Ein objekt ist im wiederverwerteten objekten\n\
verzeichnis gefunden worden, das nicht erscheint, dort zu\n\
gehren. Zu helfen, um das wiederverwertete objekte\n\
verzeichnis zu behalten, um es zu reinigen, empfiehlt ist,\n\
da das Objekt aus dem wiederverwerteten objekten verzeichnis\n\
zum heim verzeichnis bewegt ist.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Oggetto Inerte Ha Trovato",
s,
"Un oggetto  stato trovato nell'elenco di oggetti\n\
riciclato che non appare appartenere l. Per aiutare tenere\n\
l'elenco di oggetti riciclato pulisce  raccomanda che\n\
l'oggetto  fuori mosso dell'elenco di oggetti riciclato\n\
all'elenco di casa.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Inert Voorwerp Vond",
s,
"Een voorwerp is in de gerecyclde voorwerpen gids gevonden\n\
dat niet verschijnt om daar te horen. Om kost de gerecyclde\n\
voorwerpen gids volkomen het is, aanraadt te helpen dat het\n\
voorwerp uit de gerecyclde voorwerpen gids aan de huis gids\n\
bewogen is.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto Inerte Achou",
s,
"Um objeto foi achado no guia reciclado de objetos que no\n\
aparece pertencer a. Ajudar mantem o guia reciclado de\n\
objetos limpar  recomenda que o objeto  movido para fora do\n\
guia reciclado de objetos ao guia de lar.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nytralt Objekt Funnet",
s,
"Et objekt finner i den resirkuleredde objektkatalog som\n\
ikke kommer fram hre til der. Hjelpe beholder den\n\
resirkuleredde objektkatalog rengjr det er anbefaler at\n\
objektet flyttet ut av den resirkuleredde objektkatalog til\n\
hjemkatalogen.",
#else
"Inert Object Found",
s,
"An object has been found in the recycle bin directory that\n\
does not appear to belong there. To help keep the recycle\n\
bin directory clean it is recommend that the object be moved\n\
out of the recycle bin directory and into your home directory\n\
for further examination.",
#endif
						CDIALOG_ICON_WARNING,
						CDIALOG_BTNFLAG_YES |
						CDIALOG_BTNFLAG_YES_TO_ALL |
						CDIALOG_BTNFLAG_NO |
						CDIALOG_BTNFLAG_CANCEL |
						CDIALOG_BTNFLAG_HELP,
						CDIALOG_BTNFLAG_YES
					);
					CDialogSetTransientFor(NULL);
					g_free(s);
				}
				else
				{
					response = CDIALOG_RESPONSE_YES_TO_ALL;
				}
				/* Handle response */
				switch(response)
				{
				  case CDIALOG_RESPONSE_YES_TO_ALL:
					*yes_to_all = TRUE;
				  case CDIALOG_RESPONSE_YES:
					/* Yes, move the object */
					if(TRUE)
					{
						gchar	*src_path = edv_paths_join(
							recycle_bin_path,
							obj_name
						),
									*tar_path = edv_paths_join(
							core->home_path,
							obj_name
						);
						if(edv_rename(
							src_path,
							tar_path
						))
						{
							const gint error_code = (gint)errno;
							gchar *s = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de mover objeto:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable de dplacer l'objet:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, objekt zu bewegen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per muovere l'oggetto:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam voorwerp te bewegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de mover objeto:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n\
\n\
%s."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls flytte objekt:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n\
\n\
%s."
#else
"Unable to move object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s."
#endif
								,
								obj_name,
								core->home_path,
								g_strerror(error_code)
							);
							MESSAGE_ERROR(
"Move Error",
								s
							);
							g_free(s);
							nproblems++;
							if(*status_rtn == 0)
								*status_rtn = -1;
						}
						g_free(tar_path);
						g_free(src_path);
					}
					break;
				  case CDIALOG_RESPONSE_NO:
					/* No, do not move the object */
					break;
				  default:
					/* Cancel */
					if(*status_rtn == 0)
						*status_rtn = -4;
					got_cancel = TRUE;
					break;
				}
				if(got_cancel)
					break;

				continue;
			}
		}
	}


	STATUS_MESSAGE(NULL);

	/* Delete the indices list */
	g_list_free(idx_list);

	/* Delete list of recycled object names */
	if(names_list != NULL)
	{
		g_list_foreach(
			names_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(names_list);
	}

	return(nproblems);
}


/*
 *	Checks the Recycle Bin for errors, fixes any errors, and
 *	compacts the Recycled Objects.
 */
gint edv_recycle_bin_sync(
	EDVCore *core,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	void (*status_message_cb)(
			GtkWidget *,		/* NULL */
			const gchar *,		/* Message */
			gpointer		/* status_message_data */
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
			GtkWidget *,		/* NULL */
			gfloat,			/* Progress */
			gpointer		/* status_progress_data */
	),
	gpointer status_progress_data
)
{
	gint		status = 0,
			nproblems = 0,
			lock_pid = 0;
	const gulong time_start = edv_time();
	const gchar *index_path;
	gchar *recycle_bin_path = NULL;
	CfgList *cfg_list;
	EDVPermissionFlags permissions;
	EDVVFSObject *obj;

	if((core == NULL) || (yes_to_all == NULL))
	{
		errno = EINVAL;
		return(-2);
	}

	if(core->op_level > 0)
	{
		errno = EBUSY;
		return(-6);
	}

	/* Clear the last error message */
	edv_recycle_bin_sync_copy_error_message(core, NULL);

	core->op_level++;

#define CLEANUP_RETURN(rtn_val)	{		\
 const gint error_code = (gint)errno;		\
						\
 /* Report the final progress */		\
 if(status_progress_cb != NULL)			\
  status_progress_cb(				\
   NULL,					\
   1.0f,					\
   status_progress_data				\
  );						\
						\
 if(lock_pid != 0)				\
  (void)edv_recycle_bin_index_unlock(		\
   index_path,					\
   core->pid					\
  );						\
						\
 g_free(recycle_bin_path);			\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(rtn_val);				\
}

	cfg_list = core->cfg_list;

	/* Report the initial progress */
	STATUS_PROGRESS(0.0f);


	/* Get and check if the path to the recycle bin index file
	 * was set properly in the configuration
	 */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(STRISEMPTY(index_path))
	{
		/* Recycle bin index file was not set in the
		 * configuration, attempt to fix it by setting it
		 * in the configuration
		 *
		 * Get the location of the local data directory to
		 * generate the recycle bin index file path with
		 */
		gint response;
		gchar	*new_index_path,
			*data_path = STRDUP(EDV_GET_S(EDV_CFG_PARM_DIR_LOCAL));

		nproblems++;

		if(STRISEMPTY(data_path))
		{
			/* Local data directory was not set in the
			 * configuration, attempt to fix it by setting
			 * it in the configuration
			 */
			const gchar *home_path = (core->home_path != NULL) ?
				core->home_path : g_getenv(ENV_VAR_NAME_HOME);
			if(home_path == NULL)
				home_path = "";

			nproblems++;

			/* Generate the new local data directory path */
			g_free(data_path);
			data_path = g_strconcat(
				home_path,
				G_DIR_SEPARATOR_S,
				EDV_NAME_DEF_USER_DATA_DIR,
				NULL
			);

			/* Confirm set */
			if(interactive)
			{
				gchar *msg = g_strdup_printf(
"The local data directory was not set in the configuration,\n\
would you like to set it to:\n\
\n\
    %s",
					data_path
				);
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
"Local Data Directory Path Not Set",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
					CDIALOG_BTNFLAG_YES
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);
			}
			else
			{
				response = CDIALOG_RESPONSE_YES;
			}
			if(response != CDIALOG_RESPONSE_YES)
			{
				g_free(data_path);
				if(status == 0)
					status = -5;
				CLEANUP_RETURN(status);
			}

			/* Set the new local data directory path */
			EDV_SET_S(EDV_CFG_PARM_DIR_LOCAL, data_path);
		}
		/* Unable to obtain or set the local data directory? */
		if(STRISEMPTY(data_path))
		{
			g_free(data_path);
			MESSAGE_ERROR(
"Recycle Bin Error",
"Unable to obtain or generate the local data directory path"
			);
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to obtain or generate the local data directory path."
			);
			if(status == 0)
				status = -1;
			CLEANUP_RETURN(status);
		}

		/* Generate the new recycle bin index file path */
		new_index_path = g_strconcat(
			data_path,
			G_DIR_SEPARATOR_S,
			EDV_NAME_DEF_RECYCLE_BIN_INDEX_FILE,
			NULL
		);
		if(STRISEMPTY(new_index_path))
		{
			g_free(new_index_path);
			g_free(data_path);
			MESSAGE_ERROR(
"Recycle Bin Error",
"Unable to obtain or generate the index file path"
			);
			nproblems++;
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to obtain or generate the index file path."
			);
			if(status == 0)
				status = -1;
			CLEANUP_RETURN(status);
		}

		/* Confirm set */
		if(interactive)
		{
			gchar *msg = g_strdup_printf(
"The recycled objects index file path was not set in the\n\
configuration, would you like to set it to:\n\
\n\
    %s",
				new_index_path
			);
			edv_play_sound_warning(core);
			CDialogSetTransientFor(toplevel);
			response = CDialogGetResponse(
"Index File Path Not Set",
				msg,
				NULL,
				CDIALOG_ICON_WARNING,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_YES
			);
			g_free(msg);
			CDialogSetTransientFor(NULL);
		}
		else
		{
			response = CDIALOG_RESPONSE_YES;
		}
		if(response != CDIALOG_RESPONSE_YES)
		{
			g_free(data_path);
			g_free(new_index_path);
			if(status == 0)
				status = -5;
			CLEANUP_RETURN(status);
		}

		/* Set the new recycle bin index file path in the
		 * configuration
		 */
		EDV_SET_S(
			EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX,
			new_index_path
		);

		g_free(new_index_path);
		g_free(data_path);

		/* Reget the recycle bin inedx file path from the
		 * configuration to confirm that it was set properly
		 */
		index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
		if(STRISEMPTY(index_path))
		{
			MESSAGE_ERROR(
"Recycle Bin Error",
"Unable to obtain or generate the index file path"
			);
			nproblems++;
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to obtain or generate the index file path."
			);
			if(status == 0)
				status = -1;
			CLEANUP_RETURN(status);
		}
	}


	/* Get and check if the path to the recycle bin directory
	 * can be properly obtained from the recycle bin index file
	 * path
	 */
	recycle_bin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(STRISEMPTY(recycle_bin_path))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to determine the recycle bin directory path\n\
from the index file path:\n\
\n\
    %s\n\
\n\
%s.",
			index_path,
			g_strerror(error_code)
		);
		MESSAGE_ERROR(
"Recycle Bin Error",
			msg
		);
		g_free(msg);
		nproblems++;
		edv_recycle_bin_sync_copy_error_message(
			core,
"Unable to determine the recycle bin directory path from the index file path."
		);
		if(status == 0)
			status = -1;
		errno = (int)error_code;
		CLEANUP_RETURN(status);
	}


	/* Check if the Recycle Bin is locked and attempy to lock it */
	status = edv_recycle_bin_index_lock(
		index_path,
		core->pid
	);
	if(status != 0)
	{
		/* Unable to lock the Recycle Bin, check to see what
		 * the problem was
		 */
		gchar *msg;
		nproblems++;
		switch(status)
		{
		    /* Another process has locked the Recycle Bin */
		    case -6:
			if(interactive)
			{
				/* Query the user to forcefully
				 * unlock the Recycle Bin
				 */
				gint response;
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
"Recycle Bin Locked",
"The Recycled Bin appears to be locked and in use\n\
by another program.\n\
\n\
You can either wait for the program to finish its\n\
use of the recycle bin and unlock it or forcefully\n\
remove the lock if you suspect that the program\n\
has stopped responding.\n\
\n\
Forcefully remove the lock and continue?",
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
					CDIALOG_BTNFLAG_NO
				);
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_YES)
				{
					/* Forcefully remove the lock
					 * and then lock it again
					 */
					(void)edv_recycle_bin_index_unlock(
						index_path,
						core->pid
					);
					status = edv_recycle_bin_index_lock(
						index_path,
						core->pid
					);
					if(status != 0)
					{
						/* Unable to forcefully
						 * unlock and relock the
						 * Recycle Bin
						 */
						const gint error_code = (gint)errno;
						gchar *msg = g_strdup_printf(
"Unable to forcefully unlock the recycle bin.\n\
\n\
%s.",
							edv_recycle_bin_index_get_error()
						);
						MESSAGE_ERROR(
"Recycle Bin Error",
							msg
						);
						g_free(msg);
						edv_recycle_bin_sync_copy_error_message(
							core,
"Unable to forcefully unlock the recycle bin."
						);
						errno = (int)error_code;
						CLEANUP_RETURN(status);
					}
					else
					{
						/* Mark that we have
						 * successfully locked
						 * the Recycle Bin
						 */
						lock_pid = core->pid;
					}
				}
				else
				{
					edv_recycle_bin_sync_copy_error_message(
						core,
"Unable to lock the recycle bin."
					);
					errno = EBUSY;
					CLEANUP_RETURN(status);
				}
			}
			else
			{
				/* Not interactive, do not forcefully
				 * remove the lock
				 */
				edv_recycle_bin_sync_copy_error_message(
					core,
"Unable to lock the recycle bin."
				);
				errno = EBUSY;
				CLEANUP_RETURN(status);
			}
			break;

		    /* General error */
		    default:
			msg = g_strdup_printf(
"Unable to lock the recycle bin.\n\
\n\
%s.",
				edv_recycle_bin_index_get_error()
			);
			MESSAGE_ERROR(
"Recycle Bin Error",
				msg
			);
			g_free(msg);
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to lock the recycle bin."
			);
			CLEANUP_RETURN(status);
			break;
		}
	}
	else
	{
		/* Mark that we have locked the Recycle Bin */
		lock_pid = core->pid;
	}


	/* Begin checking for errors */

	/* Check if the Recycle Bin directory does not exist (which
	 * is not considered an error)
	 */
	obj = edv_vfs_object_stat(recycle_bin_path);
	if(obj == NULL)
	{
		GList *new_paths_list;

		/* Create the Recycle Bin directory just to see if
		 * it is possible to create one
		 */
		if(edv_directory_create(
			recycle_bin_path,
			TRUE,			/* Create parents */
			&new_paths_list
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to create the recycle bin directory:\n\
\n\
    %s\n\
\n\
%s.",
				recycle_bin_path,
				g_strerror(error_code)
			);
			MESSAGE_ERROR(
"Create Directory Error",
				msg
			);
			g_free(msg);
			if(new_paths_list != NULL)
			{
				g_list_foreach(
					new_paths_list,
					(GFunc)g_free,
					NULL
				);
				g_list_free(new_paths_list);
			}		    
			nproblems++;
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to create the recycle bin directory."
			);
			if(status == 0)
				status = -1;
			errno = (int)error_code;
			CLEANUP_RETURN(status);
		}

		/* Notify about the new directories created */
		if(new_paths_list != NULL)
		{
			EDVVFSObject *obj;
			const gchar *path;
			GList *glist;
			for(glist = new_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				path = (const gchar *)glist->data;
				if(path == NULL)
					continue;

				/* Set the new directory's permission
				 * so that only the owner can read,
				 * write, or execute it
				 */
				(void)edv_permissions_set(
					path,
					EDV_PERMISSION_UR |
					EDV_PERMISSION_UW |
					EDV_PERMISSION_UX
				);

				/* Notify about this new directory */
				obj = edv_vfs_object_lstat(path);
				if(obj != NULL)
				{
					edv_emit_vfs_object_added(
						core,
						path,
						obj
					);
					edv_vfs_object_delete(obj);
				}
			}
			g_list_foreach(
				new_paths_list,
				(GFunc)g_free,
				NULL
			);
			g_list_free(new_paths_list);
		}

		/* Get the newly created Recycle Bin directory's
		 * statistics
		 */
		obj = edv_vfs_object_stat(recycle_bin_path);
		if(obj == NULL)
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to create the recycle bin directory:\n\
\n\
    %s\n\
\n\
%s.",
				recycle_bin_path,
				g_strerror(error_code)
			);
			MESSAGE_ERROR(
"Create Directory Error",
				msg
			);
			g_free(msg);
			nproblems++;
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to create the recycle bin directory."
			);
			if(status == 0)
				status = -1;
			errno = (int)error_code;
			CLEANUP_RETURN(status);
		}
	}

	/* Check if the Recycle Bin directory is not a directory */
	if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
	{
		/* The recycle bin directory not a directory,
		 * query the user to move that object elsewhere
		 */
		gint response;
		gchar	*msg,
			*safe_bad_recycle_bin_path;

		nproblems++;

		/* Generate the target path to move the object that
		 * is in place of the Recycle Bin directory to
		 */
		safe_bad_recycle_bin_path = g_strconcat(
			recycle_bin_path,
			".oth",
			NULL
		);
		if(edv_path_lexists(safe_bad_recycle_bin_path))
		{
			gint i;
			for(i = 0; i < (guint16)-1; i++)
			{
				g_free(safe_bad_recycle_bin_path);
				safe_bad_recycle_bin_path = g_strdup_printf(
					"%s.%i",
					recycle_bin_path,
					i
				);
				if(!edv_path_lexists(safe_bad_recycle_bin_path))
					break;
			}
		}

		/* Query the user */
		msg = g_strdup_printf(
"Recycle bin directory path:\n\
\n\
    %s\n\
\n\
Is not a directory.\n\
\n\
The recycle bin directory must be a directory object.\n\
\n\
Move this object to:\n\
\n\
%s\n\
\n\
To allow the recycle bin directory to be created as a\n\
directory?",
			recycle_bin_path,
			safe_bad_recycle_bin_path
		);
		edv_play_sound_warning(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
"Invalid Recycle Bin Directory",
			msg,
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		if(response == CDIALOG_RESPONSE_YES)
		{
			/* Move this object */
			if(edv_rename(
				recycle_bin_path,
				safe_bad_recycle_bin_path
			))
			{
				/* Move failed */
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to move:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
					recycle_bin_path,
					safe_bad_recycle_bin_path,
					g_strerror(error_code)
				);
				MESSAGE_ERROR(
"Move Failed",
					msg
				);
				g_free(msg);
				g_free(safe_bad_recycle_bin_path);
				nproblems++;
				edv_recycle_bin_sync_copy_error_message(
					core,
"Unable to fix an invalid recycle bin directory."
				);
				if(status == 0)
					status = -1;
				errno = (int)error_code;
				CLEANUP_RETURN(status);
			}
		}

		g_free(safe_bad_recycle_bin_path);

		/* Do not continue checking because there is nothing
		 * further to check if the Recycle Bin directory
		 * was not a directory
		 */
		CLEANUP_RETURN(status);
	}

	/* Is the owner of the Recycle Bin directory not the user of
	 * this process?
	 */
	if(obj->owner_id != core->effective_user_id)
	{
		if(edv_chown(
			recycle_bin_path,
			core->effective_user_id,
			obj->group_id
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to take ownership of the recycle bin directory:\n\
\n\
    %s\n\
\n\
%s.\n\
\n\
This may pose a security concearn since other users\n\
may be able to access the recycled objects.",
				recycle_bin_path,
				g_strerror(error_code)
			);
			MESSAGE_ERROR(
"Change Ownership Error",
				msg
			);
			g_free(msg);
			nproblems++;
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to set the ownership of the recycle bin directory."
			);
			if(status == 0)
				status = -1;
			errno = (int)error_code;
			CLEANUP_RETURN(status);
		}
	}

	permissions = obj->permissions;

	/* Is the recycle bin directory not only readable, writable,
	 * and executable by the owner?
	 */
	if(!(permissions & EDV_PERMISSION_UR) ||
	   !(permissions & EDV_PERMISSION_UW) ||
	   !(permissions & EDV_PERMISSION_UX) ||
	   (permissions & EDV_PERMISSION_GR) ||
	   (permissions & EDV_PERMISSION_GW) ||
	   (permissions & EDV_PERMISSION_GX) ||
	   (permissions & EDV_PERMISSION_OR) ||
	   (permissions & EDV_PERMISSION_OW) ||
	   (permissions & EDV_PERMISSION_OX) ||
	   (permissions & EDV_PERMISSION_SETUID) ||
	   (permissions & EDV_PERMISSION_SETGID) ||
	   (permissions & EDV_PERMISSION_STICKY)
	)
	{
		/* Set the recycle bin directory to be only readable,
		 * writable, and executable by the owner
		 */
		if(edv_permissions_set(
			recycle_bin_path,
			EDV_PERMISSION_UR | EDV_PERMISSION_UW |
			EDV_PERMISSION_UX
		))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to set the permissions on the recycle bin directory:\n\
\n\
    %s\n\
\n\
%s.\n\
\n\
This may pose a security concearn since other users\n\
may be able to access the recycled objects.",
				recycle_bin_path,
				g_strerror(error_code)
			);
			MESSAGE_ERROR(
"Change Permissions Error",
				msg
			);
			g_free(msg);
			nproblems++;
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to set the permissions of the recycle bin directory."
			);
			if(status == 0)
				status = -1;
			errno = (int)error_code;
			CLEANUP_RETURN(status);
		}
	}

	edv_vfs_object_delete(obj);

	/* Get the recycled objects index file's statistics */
	obj = edv_vfs_object_stat(index_path);
	if(obj != NULL)
	{
		/* Recycled objects index file exists
		 *
		 * Update the owner of the recycle bin directory
		 */
		if(obj->owner_id != core->effective_user_id)
		{
			if(edv_chown(
				index_path,
				core->effective_user_id,
				obj->group_id
			))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to take ownership of the index file:\n\
\n\
    %s\n\
\n\
%s.\n\
\n\
This may pose a security concearn since other users\n\
may be able to access the recycled objects.",
					index_path,
					g_strerror(error_code)
				);
				MESSAGE_ERROR(
"Change Ownership Error",
					msg
				);
				g_free(msg);
				nproblems++;
				edv_recycle_bin_sync_copy_error_message(
					core,
"Unable to set the ownership of the index file."
				);
				if(status == 0)
					status = -1;
				errno = (int)error_code;
				CLEANUP_RETURN(status);
			}
		}

		permissions = obj->permissions;

		/* Is the recycle bin index file not only readable and
		 * writable by the owner?
		 */
		if(!(permissions & EDV_PERMISSION_UR) ||
		   !(permissions & EDV_PERMISSION_UW) ||
		   (permissions & EDV_PERMISSION_UX) ||
		   (permissions & EDV_PERMISSION_GR) ||
		   (permissions & EDV_PERMISSION_GW) ||
		   (permissions & EDV_PERMISSION_GX) ||
		   (permissions & EDV_PERMISSION_OR) ||
		   (permissions & EDV_PERMISSION_OW) ||
		   (permissions & EDV_PERMISSION_OX) ||
		   (permissions & EDV_PERMISSION_SETUID) ||
		   (permissions & EDV_PERMISSION_SETGID) ||
		   (permissions & EDV_PERMISSION_STICKY)
		)
		{
			/* Set the recycle bin index file to be only readable
			 * and writable by the owner
			 */
			if(edv_permissions_set(
				index_path,
				EDV_PERMISSION_UR | EDV_PERMISSION_UW
			))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to set the permissions on the index file:\n\
\n\
    %s\n\
\n\
%s.\n\
\n\
This may pose a security concearn since other users\n\
may be able to access the recycled objects.",
					index_path,
					g_strerror(error_code)
				);
				MESSAGE_ERROR(
"Change Permissions Error",
					msg
				);
				g_free(msg);
				nproblems++;
				edv_recycle_bin_sync_copy_error_message(
					core,
"Unable to set the permissions of the index file."
				);
				if(status == 0)
					status = -1;
				errno = (int)error_code;
				CLEANUP_RETURN(status);
			}
		}

		/* Check for indices that refer to non-existant recycled
		 * objects
		 */
		nproblems += edv_recycle_bin_sync_index_file(
			core,
			index_path,
			recycle_bin_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			&status,
			status_message_cb, status_message_data,
			status_progress_cb, status_progress_data
		);

		/* If the index file is of 0 size then remove it */
		if(obj->size == 0l)
			(void)edv_unlink(obj->path);

		edv_vfs_object_delete(obj);
	}
	else
	{
		/* Unable to get the recycled objects index file's
		 * statistics
		 *
		 * Report error only if it was not because it does not
		 * exist
		 */
		const gint error_code = (gint)errno;
		if(error_code != ENOENT)
		{
			gchar *msg = g_strdup_printf(
"Unable to obtain the statistics of the index file:\n\
\n\
    %s\n\
\n\
%s.",
				index_path,
				g_strerror(error_code)
			);
			MESSAGE_ERROR(
"Recycle Bin Error",
				msg
			);
			g_free(msg);
			nproblems++;
			edv_recycle_bin_sync_copy_error_message(
				core,
"Unable to obtain the statistics of the index file."
			);
			if(status == 0)
				status = -1;
			errno = (int)error_code;
			CLEANUP_RETURN(status);
		}
	}
	if(status != -4)
	{
		/* Check for recycled objects that are not listed
		 * in the index file
		 */
		nproblems += edv_recycle_bin_sync_objects(
			core,
			index_path,
			recycle_bin_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			&status,
			status_message_cb, status_message_data,
			status_progress_cb, status_progress_data
		);
	}


	/* Errors found? */
	if((status != 0) && (status != -4))
	{
		MESSAGE(
"Recycle Bin Sync Results",
"Errors in the recycle bin were encountered and fixed.\n\
\n\
The recycle bin has been compacted."
		);
	}


	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_SYNC_RECBIN,
		time_start,
		edv_time(),
		status,
		NULL,				/* No source */
		NULL,				/* No target */
		core->last_error_ptr		/* Comments */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
