#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "edv_progress.h"
#include "edv_utils_gtk.h"
#include "edv_recover_obj.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


const gchar *edv_recover_object_get_error(EDVCore *core);
static void edv_recover_object_set_error(
	EDVCore *core,
	const gchar *msg
);

static gint edv_recover_object_progress_cb(
	gpointer data, const gulong pos, const gulong total
);

gint edv_recover_object(
	EDVCore *core,
	const gulong index,
	const gchar *alternate_recovery_path,
 	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message.
 */
const gchar *edv_recover_object_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *      Coppies the error message to the core's last_error_buf
 *      and sets the core's last_error_ptr to point to it.
 */
static void edv_recover_object_set_error(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Recycle recover or delete progress callback.
 */
static gint edv_recover_object_progress_cb(
	gpointer data, const gulong pos, const gulong total
)
{
	gint status = 0;
/*	EDVCore *core = EDV_CORE(data); */

	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			((total > 0) && (pos >= 0)) ?
				((gfloat)pos / (gfloat)total) : -1.0f,
			EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			status = -1;
	}

	return(status);
}


/*
 *	Recovers the recycled object.
 *
 *	The index specifies the recycled object to recover.
 *
 *	The alternate_recovery_path specifies the full path to the
 *	directory that is to be used as the alternate recovery location
 *	for the recycled object. If alternate_recovery_path is NULL
 *	then the recycled object's original location will be used
 *	as the recovery location.
 *
 *	The new_path_rtn specifies the pointer to a gchar * that will
 *	be set to a dynamically allocated string describing the full
 *	path to the recovered object. The calling function must
 *	delete the returned string.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	shown during this operation.
 *
 *	If interactive is TRUE then the user will be notified and
 *	queried if any problems arise.
 *
 *	The yes_to_all specifies a pointer to the preinitialized
 *	current gboolean "yes to all" value. This value may be
 *	set to TRUE during this call if the user responds with
 *	"yes to all" to a query.
 */
gint edv_recover_object(
	EDVCore *core,
	const gulong index,
	const gchar *alternate_recovery_path,
 	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	const gchar *index_path;
	gchar *recovered_path = NULL;
	CfgList *cfg_list;
	EDVRecycledObject *obj = NULL;
	EDVVFSObject *vfs_obj;

	/* Reset the return values */
	if(new_path_rtn != NULL)
		*new_path_rtn = NULL;

	/* Clear the last error message */
	edv_recover_object_set_error(core, NULL);

	if((core == NULL) || (index == 0l))
	{
		edv_recover_object_set_error(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get path to recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
	{
		core->last_error_ptr = "Unable to get the recycled objects index file.";
		errno = EINVAL;
		return(-2);
	}

	/* Attempt to lock the recycle bin */
	while(TRUE)
	{
		status = edv_recycle_bin_index_lock(
			index_path,
			core->pid
		);
		if(status == 0)
			break;

		/* Already locked? */
		if(status == -6)
		{
			/* Query the user to retry or abort */
			if(interactive)
			{
				gint response;
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
"Recover Failed",
"The Recycle Bin is currently locked and in use.\n\
\n\
Try again?",
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_RETRY |
						CDIALOG_BTNFLAG_ABORT,
					CDIALOG_BTNFLAG_RETRY
				);
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_RETRY)
				{
					/* Retry */
					continue;
				}
				else
				{
					/* Abort */
					errno = EINTR;
					return(-4);
				}
			}
			else
			{
				core->last_error_ptr =
"The Recycle Bin is currently locked.";
				errno = EBUSY;
				return(-6);
			}
		}

		/* Other error occured while attempting to lock */
		core->last_error_ptr = "Unable to lock the Recycle Bin.";
		return(status);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 /* Unlock the recycle bin */			\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
						\
 g_free(recovered_path);			\
 edv_recycled_object_delete(obj);		\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Get the statistics of recycled object */
	obj = edv_recycled_object_stat(
		index_path,
		index
	);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to obtain the statistics of the recycled object:\n\
\n\
    #%ld",
			index
		);
		edv_recover_object_set_error(core, msg);
		g_free(msg);
		errno = (int)error_code;
		CLEANUP_RETURN(-1);
	}

	/* Generate the full path to the recovered object
	 *
	 * Alternate recovery location specified?
	 */
	if(alternate_recovery_path != NULL)
	{
		recovered_path = edv_paths_join(
			alternate_recovery_path,
			obj->name
		);
	}
	else
	{
		/* Use the original location */
		if((obj->original_path == NULL) || (obj->name == NULL))
		{
			gchar *msg = g_strdup_printf(
"Unable to obtain the original path or name of the recycled object:\n\
\n\
    #%ld",
				index
			);
			edv_recover_object_set_error(core, msg);
			g_free(msg);
			errno = EINVAL;
			CLEANUP_RETURN(-1);
		}

		recovered_path = edv_paths_join(
			obj->original_path,
			obj->name
		);
	}
	if(recovered_path == NULL)
	{
		core->last_error_ptr =
"Unable to generate the path to the recovered object.";
		errno = EINVAL;
		CLEANUP_RETURN(-1);
	}


	/* The target object must not exist */
	vfs_obj = edv_vfs_object_lstat(recovered_path);
	if(vfs_obj != NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"An object with the same name already exists at the recovery location:\n\
\n\
    %s",
			recovered_path
		);
		edv_recover_object_set_error(core, msg);
		g_free(msg);
		edv_vfs_object_delete(vfs_obj);
		errno = (int)error_code;
		CLEANUP_RETURN(-1);
	}

	/* Map the progress dialog? */
	if(show_progress)
	{
		gchar	*name_shortened = edv_path_shorten(
				obj->name,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*recovered_path_shortened = edv_path_shorten(
				recovered_path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Convaleciente:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Retrouver:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Wiedererlangen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ricuperare:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Terugkrijgen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Recuperar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvinning:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Recovering:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
			,
			name_shortened,
			recovered_path_shortened
		);
		g_free(name_shortened);
		g_free(recovered_path_shortened);
		edv_progress_dialog_map_recover_animated(
			cfg_list,
			msg,
			0.0f,
			toplevel,
			FALSE
		);
		g_free(msg);
	}

	/* Recover */
	status = edv_recycle_bin_index_recover(
		index_path,
		index,
		recovered_path,
		show_progress ? edv_recover_object_progress_cb : NULL,
		core
	);
	if(status == -4)
	{
		/* User aborted recovery */
	}
	else if(status != 0)
	{
		/* Recovery failed */
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to recover:\n\
\n\
    %s\n\
\n\
%s.",
			obj->name,
			edv_recycle_bin_index_get_error()
		);
		edv_recover_object_set_error(core, msg);
		g_free(msg);
		errno = (int)error_code;
	}
	else
	{
		/* Remove the entry from the recycled objects index file */
		(void)edv_recycle_bin_index_remove(
			index_path,
			index
		);

		/* Add the path to the return string of the recovered object */
		if(new_path_rtn != NULL)
		{
			g_free(*new_path_rtn);
			*new_path_rtn = STRDUP(recovered_path);
		}
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_RECOVER,
		time_start,
		edv_time(),
		status,
		obj->name,			/* Source */
		recovered_path,			/* Target */
		core->last_error_ptr		/* Comment */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
