#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "edv_progress.h"
#include "edv_utils_gtk.h"
#include "edv_purge_obj.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


const gchar *edv_purge_object_get_error(EDVCore *core);
static void edv_purge_object_copy_error_message(
	EDVCore *core, const gchar *msg
);

static gint edv_purge_object_progress_coefficient_cb(
	gpointer data,
	const gfloat coeff
);
static gint edv_purge_object_progress_cb(
	gpointer data,
	const gulong pos, const gulong total
);

gint edv_purge_object(
	EDVCore *core,
	const gulong index,
	GtkWidget *toplevel,
	const gfloat progress_value,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
gint edv_purge_all_objects(
	EDVCore *core,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message.
 */
const gchar *edv_purge_object_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Coppies the error message to the core's last_error_buf
 *	and sets the core's last_error_ptr to point to it.
 */
static void edv_purge_object_copy_error_message(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Purge progress coefficient callback.
 */
static gint edv_purge_object_progress_coefficient_cb(
	gpointer data,
	const gfloat coeff
)
{
	gint status = 0;
/*	EDVCore *core = EDV_CORE(data); */

	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			coeff,
			EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			status = -1;
	}

	return(status);
}

/*
 *	Purge progress callback.
 */
static gint edv_purge_object_progress_cb(
	gpointer data,
	const gulong pos, const gulong total
)
{
	gint status = 0;
/*	EDVCore *core = EDV_CORE(data); */

	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			((total > 0) && (pos >= 0)) ?
				((gfloat)pos / (gfloat)total) : -1.0f,
			EDV_PROGRESS_BAR_NTICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
			status = -1;
	}

	return(status);
}


/*
 *	Purges the recycled object.
 *
 *	The index specifies the recycled object to purge.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	The progress_value specifies the progress value to display
 *	on the progress dialog if show_progress is TRUE. If
 *	progress_value is negative then the progress dialog will
 *	be updated with an internally calculated value.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	shown during this operation.
 *
 *	If interactive is TRUE then the user will be notified and
 *	queried if any problems arise.
 */
gint edv_purge_object(
	EDVCore *core,
	const gulong index,
	GtkWidget *toplevel,
	const gfloat progress_value,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	const gchar *index_path;
	CfgList *cfg_list;
	EDVRecycledObject *obj = NULL;

	/* Clear the last error message */
	edv_purge_object_copy_error_message(core, NULL);

	if((core == NULL) || (index == 0l) || (yes_to_all == NULL))
	{
		edv_purge_object_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get path to recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
	{
		core->last_error_ptr =
"Unable to get the recycled objects index file.";
		errno = EINVAL;
		return(-2);
	}

	/* Attempt to lock the recycle bin */
	while(TRUE)
	{
		status = edv_recycle_bin_index_lock(
			index_path,
			core->pid
		);
		if(status == 0)
			break;

		/* Already locked? */
		if(status == -6)
		{
			/* Query the user to retry or abort */
			if(interactive)
			{
				gint response;
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
"Purge Failed",
"The Recycle Bin is currently locked and in use.\n\
\n\
Try again?",
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_RETRY |
						CDIALOG_BTNFLAG_ABORT,
					CDIALOG_BTNFLAG_RETRY
				);
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_RETRY)
				{
					/* Retry */
					continue;
				}
				else
				{
					/* Abort */
					errno = EINTR;
					return(-4);
				}
			}
			else
			{
				core->last_error_ptr =
"The Recycle Bin is currently locked.";
				errno = EBUSY;
				return(-6);
			}
		}

		/* Other error occured while attempting to lock */
		core->last_error_ptr = "Unable to lock the Recycle Bin.";
		return(status);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 /* Unlock the recycle bin */			\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
						\
 edv_recycled_object_delete(obj);		\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* Get the statistics of recycled object */
	obj = edv_recycled_object_stat(index_path, index);

	/* Map the progress dialog? */
	if(show_progress)
	{
		gchar *msg;
		if((obj != NULL) ? (obj->name != NULL) : FALSE)
		{
			gchar *name_shortened = edv_path_shorten(
				obj->name,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS
			);
			msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Purgando:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Purger:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Reinigung:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Prosciogliere:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Reinigen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Descartar:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rensing:\n\
\n\
    %s\n"
#else
"Purging:\n\
\n\
    %s\n"
#endif
				,
				name_shortened
			);
			g_free(name_shortened);
		}
		else
		{
			msg = g_strdup_printf(
"Purging:\n\
\n\
    #%ld\n",
				index
			);
		}
		edv_progress_dialog_map_purge_animated(
			cfg_list,
			msg,
			MAX(progress_value, 0.0f),
			toplevel,
			FALSE
		);
		g_free(msg);

		/* If the given progress value is not negative then explicitly
		 * call the progress callback with values simulating the given
		 * progress value
		 */
		if(progress_value >= 0.0f)
		{
			if(edv_purge_object_progress_coefficient_cb(
				core,
				progress_value
			))
			{
				status = -4;
				CLEANUP_RETURN(status);
			}
		}
	}


	/* Remove the recycled object from the recycle bin */
	status = edv_recycle_bin_index_purge(
		index_path,
		index,
		((progress_value < 0.0f) && show_progress) ?
			edv_purge_object_progress_cb : NULL,
		core
	);
	if(status != 0)
	{
		gchar *msg;
		if((obj != NULL) ? (obj->name != NULL) : FALSE)
			msg = g_strdup_printf(
"Unable to purge:\n\
\n\
    %s\n\
\n\
%s.",
				obj->name, edv_recycle_bin_index_get_error()
			);
		else
			msg = g_strdup_printf(
"Unable to purge:\n\
\n\
    #%ld\n\
\n\
%s.",
			index,
			edv_recycle_bin_index_get_error()
		);
		edv_purge_object_copy_error_message(core, msg);
		g_free(msg);
	}
	else
	{
		/* Remove recycled object entry from the recycled objects
		 * index file
		 */
		(void)edv_recycle_bin_index_remove(
			index_path,
			index
		);
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_PURGE,
		time_start,
		edv_time(),
		status,
		(obj != NULL) ? obj->name : NULL,	/* Source */
		NULL,				/* No target */
		core->last_error_ptr		/* Comment */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Purges all the recycled objects from the recycle bin.
 *
 *	The index specifies the recycled object to purge.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	shown during this operation.
 *
 *	If interactive is TRUE then the user will be notified and
 *	queried if any problems arise.
 *
 *	The yes_to_all specifies a pointer to the preinitialized
 *	current gboolean "yes to all" value. This value may be
 *	set to TRUE during this call if the user responds with
 *	"yes to all" to a query.
 */
gint edv_purge_all_objects(
	EDVCore *core,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	const gchar *index_path;
	CfgList *cfg_list;

	/* Clear the last error message */
	edv_purge_object_copy_error_message(core, NULL);

	if((core == NULL) || (yes_to_all == NULL))
	{
		edv_purge_object_copy_error_message(core, "Invalid value.");
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get path to recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
	{
		core->last_error_ptr = "Unable to get the recycled objects index file.";
		errno = EINVAL;
		return(-2);
	}

	/* Attempt to lock the recycle bin */
	while(TRUE)
	{
		status = edv_recycle_bin_index_lock(
			index_path,
			core->pid
		);
		if(status == 0)
			break;

		/* Already locked? */
		if(status == -6)
		{
			/* Query the user to retry or abort */
			if(interactive)
			{
				gint response;
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
"Purge Failed",
"The Recycle Bin is currently locked and in use.\n\
\n\
Try again?",
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_RETRY |
						CDIALOG_BTNFLAG_ABORT,
					CDIALOG_BTNFLAG_RETRY
				);
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_RETRY)
				{
					/* Retry */
					continue;
				}
				else
				{
					/* Abort */
					errno = EINTR;
					return(-4);
				}
			}
			else
			{
				core->last_error_ptr =
"The Recycle Bin is currently locked.";
				errno = EBUSY;
				return(-6);
			}
		}

		/* Other error occured while attempting to lock */
		core->last_error_ptr = "Unable to lock the Recycle Bin.";
		return(status);
	}

	/* Is there an operation already in progress? */
	if(core->op_level > 0)
	{
		core->last_error_ptr =
"An operation is already in progress, please try again later.";
		errno = EBUSY;
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 /* Unlock the recycle bin */			\
 (void)edv_recycle_bin_index_unlock(		\
  index_path,					\
  core->pid					\
 );						\
						\
 core->op_level--;				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	if(show_progress)
	{
#if defined(PROG_LANGUAGE_SPANISH)
		gchar *msg = g_strdup(
"Purgando todo contenido del Cajn de la Recirculacin...\n"
		);
#elif defined(PROG_LANGUAGE_FRENCH)
		gchar *msg = g_strdup(
"Purger tous contenus du Recycle l'Huche...\n"
		);
#elif defined(PROG_LANGUAGE_GERMAN)
		gchar *msg = g_strdup(
"Reinigung alles Inhalts vom Verwertet Behlter wieder...\n"
		);
#elif defined(PROG_LANGUAGE_ITALIAN)
		gchar *msg = g_strdup(
"Prosciogliere tutto il contenuto dal Contenitore per\n\
la raccolta differenziata...\n"
		);
#elif defined(PROG_LANGUAGE_DUTCH)
		gchar *msg = g_strdup(
"Reinigen van alle inhoud van het Recyclt Bak...\n"
		);
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		gchar *msg = g_strdup(
"O Purging todo contedo da Caixa de Recycle...\n"
		);
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		gchar *msg = g_strdup(
"Rensing av all innhold fra Recycle Bin...\n"
		);
#else
		gchar *msg = g_strdup(
"Purging all contents from the Recycle Bin...\n"
		);
#endif
		edv_progress_dialog_map_purge_animated(
			cfg_list,
			msg,
			0.0f,
			toplevel,
			FALSE
		);
		g_free(msg);
	}


	/* Purge all objects found in the recycled objects directory,
	 * including the recycled objects index file.
	 */
	status = edv_recycle_bin_index_purge_all(
		index_path, 
		show_progress ? edv_purge_object_progress_cb : NULL,
		core
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to purge all the recycled objects.\n\
\n\
%s.",
			edv_recycle_bin_index_get_error()
		);
		edv_purge_object_copy_error_message(core, msg);
		g_free(msg);
		errno = (int)error_code;
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL,
		time_start,
		edv_time(),
		status,
		NULL,				/* No source */
		NULL,				/* No target */
		core->last_error_ptr		/* Comment */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
