#include <gtk/gtk.h>

#include "cfg.h"

#include "progressdialog.h"

#include "edv_types.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"

#include "edv_cfg_list.h"
#include "config.h"


static gchar *EDVProgressGetAnimationDirectoryPath(CfgList *cfg_list);

/* VFS Operations */
void edv_progress_dialog_map_move_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);
void edv_progress_dialog_map_copy_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);
void edv_progress_dialog_map_delete_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap,
	const EDVDeleteMethod delete_method
);
void edv_progress_dialog_map_recover_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);
void edv_progress_dialog_map_purge_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);

/* Archive Operations */
void edv_progress_dialog_map_archive_add_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);
void edv_progress_dialog_map_archive_extract_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);
void edv_progress_dialog_map_archive_delete_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);
void edv_progress_dialog_map_archive_fix_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the animation directory from the configuration.
 */
static gchar *EDVProgressGetAnimationDirectoryPath(CfgList *cfg_list)
{
	const gchar *data_path = EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL);
	if(STRISEMPTY(data_path))
		data_path = EDV_PATH_DEF_GLOBAL_DATA_DIR;

	return(g_strconcat(
		data_path,
		G_DIR_SEPARATOR_S,
		EDV_NAME_ANIMATIONS_SUBDIR,
		NULL
	));
}


/*
 *	Maps the progress dialog in animation mode for move.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_move_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "folder_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "file01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Mover",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Dmnagement",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Bewegen",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"Il Trasloco",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Bewegen",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Mover",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Flytting",
		label,
		"Stans",
#else
		"Moving",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog in animation mode for copy.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_copy_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "file01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Copiar",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Copier",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Kopieren",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"Copiare",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Kopiren",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Copiar",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Kopiering",
		label,
		"Stans",
#else
		"Copying",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog in animation mode for delete.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 *
 *	The delete_method specifies the delete method to display on
 *	the progress dialog.
 */
void edv_progress_dialog_map_delete_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap,
	const EDVDeleteMethod delete_method
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again.
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	if(delete_method == EDV_DELETE_METHOD_PURGE)
	{
		start_icon_paths_list = NULL;
		APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
		APPEND_PATH(start_icon_paths_list, "folder_32x32.xpm");
		APPEND_PATH(start_icon_paths_list, "folder_32x32.xpm");
		icon_paths_list = NULL;
		APPEND_PATH(icon_paths_list, "trash01_20x20.xpm");
		APPEND_PATH(icon_paths_list, "trash02_20x20.xpm");
		APPEND_PATH(icon_paths_list, "trash03_20x20.xpm");
		APPEND_PATH(icon_paths_list, "trash04_20x20.xpm");
		APPEND_PATH(icon_paths_list, "trash05_20x20.xpm");
		APPEND_PATH(icon_paths_list, "trash06_20x20.xpm");
		end_icon_paths_list = NULL;
		APPEND_PATH(end_icon_paths_list, NULL);
		APPEND_PATH(end_icon_paths_list, NULL);
		APPEND_PATH(end_icon_paths_list, NULL);
	}
	else
	{
		start_icon_paths_list = NULL;
		APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
		APPEND_PATH(start_icon_paths_list, "folder_32x32.xpm");
		APPEND_PATH(start_icon_paths_list, "folder_32x32.xpm");
		icon_paths_list = NULL;
		APPEND_PATH(icon_paths_list, "file01_20x20.xpm");
		APPEND_PATH(icon_paths_list, "file02_20x20.xpm");
		APPEND_PATH(icon_paths_list, "file03_20x20.xpm");
		APPEND_PATH(icon_paths_list, "file04_20x20.xpm");
		APPEND_PATH(icon_paths_list, "file05_20x20.xpm");
		APPEND_PATH(icon_paths_list, "file06_20x20.xpm");
		end_icon_paths_list = NULL;
		APPEND_PATH(end_icon_paths_list, "recycle_bin_full_32x32.xpm");
		APPEND_PATH(end_icon_paths_list, "recycle_bin_full_32x32.xpm");
		APPEND_PATH(end_icon_paths_list, "recycle_bin_full_32x32.xpm");
	}
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Borrar",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Effacer",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Lschen",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"Cancellare",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Schrappen",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Anular",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Stryking",
		label,
		"Stans",
#else
		"Deleting",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog in animation mode for recover.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_recover_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "recycle_bin_full_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "recycle_bin_full_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "recycle_bin_full_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "file01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Convaleciente",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Retrouver",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Wiedererlangen",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"Ricuperare",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Terugkrijgen",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Recuperar",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Utvinning",
		label,
		"Stans",
#else
		"Recovering",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog in animation mode for purge.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_purge_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "recycle_bin_full_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "recycle_bin_full_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "recycle_bin_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "trash01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, NULL);
	APPEND_PATH(end_icon_paths_list, NULL);
	APPEND_PATH(end_icon_paths_list, NULL);
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Purgando",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Purger",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Reinigung",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"Prosciogliere",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Reinigen",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Purging",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Rensing",
		label,
		"Stans",
#else
		"Purging",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}


/*
 *	Maps the progress dialog in animation mode for archive add.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_archive_add_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "folder_file_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "file01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, "package_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "package_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "package_file_32x32.xpm");
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Agregar",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Addition",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Hinzufgen",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"L'Aggiunta",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Toevoegen",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Adicionar",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Tilfying",
		label,
		"Stans",
#else
		"Adding",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog in animation mode for archive extract.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_archive_extract_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "package_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "package_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "package_file_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "file01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "file06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_32x32.xpm");
	APPEND_PATH(end_icon_paths_list, "folder_file_32x32.xpm");
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Extraer",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Extraire",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Extrahieren",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"Estrarre",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Onttrekken",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Extrair",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Trekking Ut",
		label,
		"Stans",
#else
		"Extracting",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog in animation mode for archive delete.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_archive_delete_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar *animations_path;
	GList		*start_icon_paths_list,
			*icon_paths_list,
			*end_icon_paths_list;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
#define APPEND_PATH(_glist_,_xpm_file_name_)	{	\
 (_glist_) = g_list_append(				\
  (_glist_),						\
  ((_xpm_file_name_) != NULL) ? g_strconcat(		\
   animations_path,					\
   G_DIR_SEPARATOR_S,					\
   (_xpm_file_name_),					\
   NULL							\
  ) : NULL						\
 );							\
}
	start_icon_paths_list = NULL;
	APPEND_PATH(start_icon_paths_list, "package_file_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "package_32x32.xpm");
	APPEND_PATH(start_icon_paths_list, "package_32x32.xpm");
	icon_paths_list = NULL;
	APPEND_PATH(icon_paths_list, "trash01_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash02_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash03_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash04_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash05_20x20.xpm");
	APPEND_PATH(icon_paths_list, "trash06_20x20.xpm");
	end_icon_paths_list = NULL;
	APPEND_PATH(end_icon_paths_list, NULL);
	APPEND_PATH(end_icon_paths_list, NULL);
	APPEND_PATH(end_icon_paths_list, NULL);
#undef APPEND_PATH
	g_free(animations_path);

	ProgressDialogMapAnimationFile(
#if defined(PROG_LANGUAGE_SPANISH)
		"Borrar",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Effacer",
		label,
		"Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
		"Lschen",
		label,
		"Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
		"Cancellare",
		label,
		"Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
		"Schrappen",
		label,
		"Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
		"Anular",
		label,
		"Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
		"Stryking",
		label,
		"Stans",
#else
		"Deleting",
		label,
		"Stop",
#endif
		start_icon_paths_list,
		icon_paths_list,
		end_icon_paths_list,
		EDV_PROGRESS_DLG_ANIM_INT,
		EDV_PROGRESS_DLG_ANIM_INC
	);

	g_list_foreach(start_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(start_icon_paths_list);
	g_list_foreach(icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(icon_paths_list);
	g_list_foreach(end_icon_paths_list, (GFunc)g_free, NULL);
	g_list_free(end_icon_paths_list);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog in animation mode for archive fix.
 *
 *	If label is not NULL then it specifies the string describing
 *	the progress message.
 *
 *	If progress is zero or positive then it specifies the progress
 *	value from 0.0 to 1.0. If progress is negative then it specifies
 *	to display an unknown progress.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If force_remap is TRUE and the progress dialog is already mapped
 *	then the current progress dialog's query will be broken and the
 *	the progress dialog will be remapped. If force_remap is FALSE
 *	and the progress dialog is already mapped then the progress
 *	dialog's current message and progress will be updated.
 */
void edv_progress_dialog_map_archive_fix_animated(
	CfgList *cfg_list,
	const gchar *label,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	gchar		*animations_path,
			*icon_path;

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
		/* Check if the progress dialog needs to be unmapped and
		 * remapped again
		 */
		if(force_remap)
		{
			ProgressDialogBreakQuery(FALSE);
		}
		else
		{
			/* Already mapped and does not need unmapping, so just
			 * update the progress message and value
			 */
			if(progress < 0.0f)
				ProgressDialogUpdateUnknown(
					NULL,
					label,
					NULL,
					NULL,
					TRUE
			   );
			else
				ProgressDialogUpdate(
					NULL,
					label,
					NULL,
					NULL,
					progress,
					EDV_PROGRESS_BAR_NTICKS,
					TRUE
				);
			return;
		}
	}

	ProgressDialogSetTransientFor(toplevel);

	animations_path = EDVProgressGetAnimationDirectoryPath(cfg_list);
	icon_path = g_strconcat(
		animations_path,
		G_DIR_SEPARATOR_S,
		"package_file_32x32.xpm",
		NULL
	);
	g_free(animations_path);

	ProgressDialogMapFile(
		"Fixing",
		label,
		icon_path,
		"Stop"
	);

	g_free(icon_path);

	if(progress < 0.0f)
		ProgressDialogUpdateUnknown(
			NULL,
			NULL,
			NULL,
			NULL,
			TRUE
		);
	else
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			progress,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}
