#include <string.h>
#include <gdk/gdk.h>

#include "guiutils.h"
#include "guirgbimg.h"

#include "edv_pixmap.h"


/* EDVPixmap Lists */
GList *edv_pixmaps_list_append_from_data(
	GList *pixmaps_list,
	edv_pixmap_data *data,
	const gchar *name,
	EDVPixmap **p_rtn
);
GList *edv_pixmaps_list_append_from_file(
	GList *pixmaps_list,
	const gchar *path,
	EDVPixmap **p_rtn
);

GList *edv_pixmaps_list_clean(GList *pixmaps_list);

EDVPixmap *edv_pixmaps_list_match_name(
	GList *pixmaps_list,
	const gchar *name  
);
EDVPixmap *edv_pixmaps_list_match_path(
	GList *pixmaps_list,
	const gchar *path
);

/* EDVPixmap */
EDVPixmap *edv_pixmap_new(void);
EDVPixmap *edv_pixmap_new_ref(void);
EDVPixmap *edv_pixmap_new_from_data(
	edv_pixmap_data *data,
	const gchar *name
);
EDVPixmap *edv_pixmap_new_from_file(const gchar *path);
gboolean edv_pixmap_resize(
	EDVPixmap *p,
	const gint new_width, const gint new_height
);
gboolean edv_pixmap_is_loaded(EDVPixmap *p);
gint edv_pixmap_ref_count(EDVPixmap *p);
EDVPixmap *edv_pixmap_ref(EDVPixmap *p);
static void EDVPixmapDelete(EDVPixmap *p);
EDVPixmap *edv_pixmap_unref(EDVPixmap *p);

/* EDVPixmap GDK/GTK */
GtkWidget *edv_pixmap_new_gtk_pixmap(EDVPixmap *p);
void edv_pixmap_set_gtk_pixmap(EDVPixmap *p, GtkWidget *w);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVPixmap of source type EDV_PIXMAP_SOURCE_DATA
 *	with two reference counts (one for the list and one for the
 *	calling function) or gets an existing matching EDVPixmap from
 *	the list with one reference count added.
 *
 *	The pixmaps_list specifies the pixmaps list, a GList of
 *	EDVPixmap * pixmaps.
 *
 *	The data specifies the XPM data.
 *
 *	The name specifies the C declaration name of the XPM data.
 *
 *	If p_rtn is not NULL then the new or matching EDVPixmap will
 *	be set to *p_rtn.
 *
 *	Returns the new pixmaps_list or NULL on error.
 */
GList *edv_pixmaps_list_append_from_data(
	GList *pixmaps_list,
	edv_pixmap_data *data,
	const gchar *name,
	EDVPixmap **p_rtn
)
{
	/* Check if an EDVPixmap already exists with the specified name */
	EDVPixmap *p = edv_pixmaps_list_match_name(
		pixmaps_list,
		name
	);
	if(p != NULL)
	{
		/* Increment the reference count on the existing EDVPixmap
		 * in the list for the calling function
		 */
		p = edv_pixmap_ref(p);
		if(p_rtn != NULL)
			*p_rtn = p;

		return(pixmaps_list);
	}
	else
	{
		/* Create a new EDVPixmap with one reference count */
		p = edv_pixmap_new_from_data(
			data,
			name
		);
		if(p == NULL)
		{
			if(p_rtn != NULL)
				*p_rtn = NULL;

			return(pixmaps_list);
		}

		/* Append the EDVPixmap to the list and add a reference
		 * count to it so that it has one reference count for the
		 * list and one reference count for the calling function
		 */
		pixmaps_list = g_list_append(
			pixmaps_list,
			edv_pixmap_ref(p)
		);

		if(p_rtn != NULL)
			*p_rtn = p;

		return(pixmaps_list);
	}
}

/*
 *	Creates a new EDVPixmap of source type EDV_PIXMAP_SOURCE_FILE
 *	with two reference counts (one for the list and one for the
 *	calling function) or gets an existing matching EDVPixmap from
 *	the list with one reference count added.
 *
 *	The pixmaps_list specifies the pixmaps list, a GList of
 *	EDVPixmap * pixmaps.
 *
 *	The path specifies the XPM file.
 *
 *	If p_rtn is not NULL then the new or matching EDVPixmap will
 *	be set to *p_rtn.
 *
 *	Returns the new pixmaps_list or NULL on error.
 */
GList *edv_pixmaps_list_append_from_file(
	GList *pixmaps_list,
	const gchar *path,
	EDVPixmap **p_rtn
)
{
	/* Check if an EDVPixmap already exists with the specified path */
	EDVPixmap *p = edv_pixmaps_list_match_path(
		pixmaps_list,
		path
	);
	if(p != NULL)
	{
		/* Increment the reference count on the existing EDVPixmap
		 * in the list for the calling function
		 */
		p = edv_pixmap_ref(p);
		if(p_rtn != NULL)
			*p_rtn = p;

		return(pixmaps_list);
	}
	else
	{
		/* Create a new EDVPixmap with one reference count */
		p = edv_pixmap_new_from_file(path);
		if(p == NULL)
		{
			if(p_rtn != NULL)
				*p_rtn = NULL;

			return(pixmaps_list);
		}

		/* Append the EDVPixmap to the list and add a reference
		 * count to it so that it has one reference count for the
		 * list and one reference count for the calling function
		 */
		pixmaps_list = g_list_append(
			pixmaps_list,
			edv_pixmap_ref(p)
		);

		if(p_rtn != NULL)
			*p_rtn = p;

		return(pixmaps_list);
	}
}


/*
 *	Unrefs all EDVPixmaps in the list with only one reference
 *	count remaining.
 */
GList *edv_pixmaps_list_clean(GList *pixmaps_list)
{
	EDVPixmap *p;
	GList *glist = pixmaps_list;
	while(glist != NULL)
	{
		p = EDV_PIXMAP(glist->data);
		if(p == NULL)
		{
			pixmaps_list = g_list_remove(
				pixmaps_list,
				NULL
			);
			glist = pixmaps_list;	/* Go back to the beginning */
			continue;
		}

		/* Does this EDVPixmap have only one reference count remaining? */
		if(edv_pixmap_ref_count(p) == 1)
		{
			pixmaps_list = g_list_remove(
				pixmaps_list,
				p
			);
			edv_pixmap_unref(p);
			glist = pixmaps_list;	/* Go back to the beginning */
			continue;
		}

		glist = g_list_next(glist);
	}

	return(pixmaps_list);
}


/*
 *	Finds the EDVPixmap of source type EDV_PIXMAP_SOURCE_DATA by
 *	name.
 */
EDVPixmap *edv_pixmaps_list_match_name(
	GList *pixmaps_list,
	const gchar *name
)
{
	GList *glist;
	EDVPixmap *p;

	if(STRISEMPTY(name))
		return(NULL);

	for(glist = pixmaps_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		p = EDV_PIXMAP(glist->data);
		if(p == NULL)
			continue;

		if((p->source_type != EDV_PIXMAP_SOURCE_DATA) ||
		   (p->name == NULL)
		)
			continue;

		if(!strcmp((const char *)p->name, (const char *)name))
			return(p);
	}

	return(NULL);
}

/*
 *	Finds the EDVPixmap of source type EDV_PIXMAP_SOURCE_FILE by
 *	path.
 */
EDVPixmap *edv_pixmaps_list_match_path(
	GList *pixmaps_list,
	const gchar *path
)
{
	GList *glist;
	EDVPixmap *p;

	if(STRISEMPTY(path))
		return(NULL);

	for(glist = pixmaps_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		p = EDV_PIXMAP(glist->data);
		if(p == NULL)
			continue;

		if((p->source_type != EDV_PIXMAP_SOURCE_FILE) ||
		   (p->path == NULL)
		)
			continue;

		if(!strcmp((const char *)p->path, (const char *)path))
			return(p);
	}

	return(NULL);
}


/*
 *	Creates a new EDVPixmap.
 *
 *	Returns the new EDVPixmap with zero reference counts or NULL
 *	on error.
 */
EDVPixmap *edv_pixmap_new(void)
{
	return(EDV_PIXMAP(g_malloc0(sizeof(EDVPixmap))));
}

/*
 *	Creates a new EDVPixmap with exactly one reference count.
 *
 *	Returns the new EDVPixmap with one reference count or NULL on
 *	error.
 */
EDVPixmap *edv_pixmap_new_ref(void)
{
	return(edv_pixmap_ref(edv_pixmap_new()));
}

/*
 *	Creates a new EDVPixmap of source type EDV_PIXMAP_SOURCE_DATA
 *	with one reference count from XPM data.
 *
 *	Returns the new EDVPixmap with one reference count.
 */
EDVPixmap *edv_pixmap_new_from_data(
	edv_pixmap_data *data,
	const gchar *name
)
{
	EDVPixmap *p = edv_pixmap_new_ref();
	if(p == NULL)
		return(NULL);

	p->source_type = EDV_PIXMAP_SOURCE_DATA;

	/* XPM data specified? */
	if(data != NULL)
	{
		GdkPixmap *pixmap;

		/* Load the pixmap and mask pair from the XPM data */
		p->pixmap = pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
			&p->mask,
			(guint8 **)data
		);
		if(pixmap != NULL)
		{
			/* Get the size */
			gdk_window_get_size(
				(GdkWindow *)pixmap,
				&p->width, &p->height
			);

			/* Record the XPM data pointer */
			p->data = data;
		}
	}

#if 0
	/* Set the name */
	p->name = STRDUP(name);
	if(STRISEMPTY(p->name))
		g_printerr(
"edv_pixmap_new_from_data(): Warning:\
 Creating EDVPixmap %p from data %p with no name.\n",
			p,
			data
		);
#endif

	/* Failed to load the pixmap? */
	if((data != NULL) && (p->pixmap == NULL))
	{
		g_printerr(
"edv_pixmap_new_from_data(): Warning:\
 Failed to load the GdkPixmap for EDVPixmap \"%s\" from data %p.\n",
			p->name,
			data
		);
		(void)edv_pixmap_unref(p);
		return(NULL);
	}

	return(p);
}

/*
 *	Creates a new EDVPixmap of source type EDV_PIXMAP_SOURCE_FILE
 *	with one reference count from an XPM file.
 *
 *	Returns the new EDVPixmap with one reference count.
 */
EDVPixmap *edv_pixmap_new_from_file(const gchar *path)
{
	EDVPixmap *p = edv_pixmap_new_ref();
	if(p == NULL)
		return(NULL);

	p->source_type = EDV_PIXMAP_SOURCE_FILE;

	/* XPM file path? */
	if(path != NULL)
	{
		GdkPixmap *pixmap;

		/* Load the pixmap and mask pair from the XPM data */
		p->pixmap = pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
			&p->mask,
			path
		);
		if(pixmap != NULL)
		{
			/* Get the size */
			gdk_window_get_size(
				(GdkWindow *)pixmap,
				&p->width, &p->height
			);

			/* Record the XPM file path */
			p->path = STRDUP(path);
		}
	}

	/* Failed to open the pixmap? */
	if((path != NULL) && (p->pixmap == NULL))
	{
		(void)edv_pixmap_unref(p);
		return(NULL);
	}

	return(p);
}

/*
 *	Resizes the EDVPixmap.
 *
 *	The p specifies the EDVPixmap to resize.
 *
 *	The new_width and new_height specifies the new size in pixels.
 *
 *	Returns TRUE if the EDVPixmap was resized or FALSE if there
 *	was no change in size because the new size and the current size
 *	are the same or error.
 */
gboolean edv_pixmap_resize(
	EDVPixmap *p,
	const gint new_width, const gint new_height
)
{
	const gint	bpp = 4;		/* RGBA */
	gint		src_width,
			src_height,
			src_bpl,
			tar_bpl;
	guint8		*tar_data,
			*src_data;
	GdkBitmap *tar_mask;
	GdkPixmap *tar_pixmap;
	GdkGC *gc;

	if((p == NULL) || (new_width <= 0) || (new_height <= 0))
		return(FALSE);

	/* No change in size? */
	if((new_width == p->width) && (new_height == p->height))
		return(FALSE);

	/* No GdkPixmap loaded? */
	if(p->pixmap == NULL)
		return(FALSE);

	/* Get the source RGBA image data from the source GdkPixmap */
	src_data = gdk_get_rgba_image(
		(GdkDrawable *)p->pixmap,
		p->mask,
		NULL,				/* Entire area */
		&src_width, &src_height,
		&src_bpl
	);
	if(src_data == NULL)
		return(FALSE);

	/* Create the resized target RGBA image data */
	tar_bpl = new_width * bpp;
	tar_data = (guint8 *)g_malloc(
		tar_bpl * new_height * sizeof(guint8)
	);
	if(tar_data == NULL)
	{
		g_free(src_data);
		return(FALSE);
	}

	/* Copy/resize the RGBA image data */
	GUIImageBufferResize(
		bpp,
		src_data,
		src_width, src_height,
		src_bpl,
		tar_data,
		new_width, new_height,
		tar_bpl,
		NULL, NULL
	);

	/* Delete the source RGBA image data */
	g_free(src_data);

	/* Create the resized GdkPixmap */
	tar_pixmap = GDK_PIXMAP_NEW(new_width, new_height);
	if(tar_pixmap == NULL)
	{
		g_free(tar_data);
		return(FALSE);
	}

	/* Create the GdkGC for the resized target GdkPixmap */
	gc = gdk_gc_new((GdkWindow *)tar_pixmap);
	if(gc == NULL)
	{
		g_free(tar_data);
		(void)GDK_PIXMAP_UNREF(tar_pixmap);
		return(FALSE);
	}

	/* Put the resized target RGBA image data to the target GdkPixmap */
	gdk_draw_rgb_32_image(
		(GdkDrawable *)tar_pixmap,
		gc,
		0, 0,
		new_width, new_height,
		GDK_RGB_DITHER_NORMAL,
		tar_data,
		tar_bpl
	);

	/* Resize the mask */
	if(p->mask != NULL)
	{
		/* Create the resized mask from the RGBA image data */
		tar_mask = GUICreateBitmapFromDataRGBA(
			new_width, new_height,
			tar_bpl,
			tar_data,
			0x80,			/* Alpha threshold */
			NULL
		);
	}
	else
	{
		tar_mask = NULL;
	}

	/* Delete the resized target RGBA image data */
	g_free(tar_data);

	/* Unref the target GdkPixmap's GdkGC */
	(void)GDK_GC_UNREF(gc);

	/* Set the new resized target GdkPixmap and GdkBitmap on to
	 * the EDVPixmap
	 */
	(void)GDK_PIXMAP_UNREF(p->pixmap);
	p->pixmap = tar_pixmap;
	(void)GDK_BITMAP_UNREF(p->mask);
	p->mask = tar_mask;

	return(TRUE);
}

/*
 *	Check if the EDVPixmap's GdkPixmap is loaded (not NULL).
 */
gboolean edv_pixmap_is_loaded(EDVPixmap *p)
{
	if(p == NULL)
		return(FALSE);

	if(p->pixmap != NULL)
		return(TRUE);
	else
		return(FALSE);
}

/*
 *	Gets the number of reference counts on the EDVPixmap.
 */
gint edv_pixmap_ref_count(EDVPixmap *p)
{
	if(p == NULL)
		return(0);

	return(p->ref_count);
}

/*
 *	Adds a reference count to the EDVPixmap.
 *
 *	Returns the EDVPixmap or NULL on error.
 */
EDVPixmap *edv_pixmap_ref(EDVPixmap *p)
{
	if(p == NULL)
		return(NULL);

	p->ref_count++;

	return(p);
}

/*
 *	Deletes the EDVPixmap.
 */
static void EDVPixmapDelete(EDVPixmap *p)
{
	if(p == NULL)
		return;

	(void)GDK_PIXMAP_UNREF(p->pixmap);
	(void)GDK_BITMAP_UNREF(p->mask);
	g_free(p->name);
	g_free(p->path);
	g_free(p);
}

/*
 *	Removes a reference count from the EDVPixmap.
 *
 *	If no reference counts remain then the EDVPixmap will be deleted.
 *
 *	This function always returns NULL.
 */
EDVPixmap *edv_pixmap_unref(EDVPixmap *p)
{
	if(p == NULL)
		return(NULL);

	p->ref_count--;

	/* If there are no reference counts remaining then delete
	 * this EDVPixmap
	 */
	if(p->ref_count <= 0)
		EDVPixmapDelete(p);

	return(NULL);
}


/*
 *	Creates a new GtkPixmap from the EDVPixmap.
 */
GtkWidget *edv_pixmap_new_gtk_pixmap(EDVPixmap *p)
{
	if(p == NULL)
		return(NULL);

	if(p->pixmap == NULL)
		return(NULL);

	return(gtk_pixmap_new(p->pixmap, p->mask));
}

/*
 *	Sets the GtkPixmap's pixmap and mask from the EDVPixmap.
 */
void edv_pixmap_set_gtk_pixmap(
	EDVPixmap *p,
	GtkWidget *w
)
{
	if((p == NULL) || (w == NULL))
		return;

	if(p->pixmap == NULL)
		return;

	gtk_pixmap_set(
		GTK_PIXMAP(w),
		p->pixmap, p->mask
	);
}
