#include <gtk/gtk.h>
#include "guiutils.h"

#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_mime_type.h"
#include "edv_open_to_menu.h"
#include "edv_open_cb.h"


static void edv_open_to_menu_data_destroy_cb(gpointer data);

GtkWidget *edv_open_to_menu_new_from_mime_type(
	EDVCore *core,
	EDVMIMEType *m,
	GList *(*paths_list_get_cb)(gpointer),
	gpointer paths_list_get_data,
	void (*goto_directory_cb)(gpointer, const gchar *),
	gpointer goto_directory_data,
	GtkWidget *toplevel,
	const gboolean verbose
);


#define EDV_OPEN_TO_MENU_ITEM_KEY	"EDV/OpenToMenu/Item"

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Data destroy callback.
 */
static void edv_open_to_menu_data_destroy_cb(gpointer data)
{
	edv_open_cb_data_delete(EDV_OPEN_CB_DATA(data));
}


/*
 *	Creates a new Open To menu from the specified MIME Type.
 *
 *	The m specifies the MIME Type who's command names are to be
 *	added to the new menu. If m is NULL then an new empty menu
 *	will be returned.
 *
 *	The paths_list_get_cb specifies the callback function to
 *	obtain a list of paths. This function needs to return a GList
 *	which member data pointing to a dynamically allocated string
 *	describing the full path to the object.
 *
 *	Returns the new Open To menu.
 */
GtkWidget *edv_open_to_menu_new_from_mime_type(
	EDVCore *core,
	EDVMIMEType *m,
	GList *(*paths_list_get_cb)(gpointer),
	gpointer paths_list_get_data,
	void (*goto_directory_cb)(gpointer, const gchar *),
	gpointer goto_directory_data,                    
	GtkWidget *toplevel,
	const gboolean verbose
)
{
	const gchar *command_name;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget	*w,
			*menu = GUIMenuCreate();
	EDVOpenCBData *d;


#define NEW_MENU_ITEM_LABEL(_name_,_icon_data_,_data_) { \
 w = GUIMenuItemCreate(					\
  menu,							\
  GUI_MENU_ITEM_TYPE_LABEL,				\
  accelgrp,						\
  (_icon_data_),					\
  (_name_),						\
  0, 0,							\
  edv_open_cb, (_data_)					\
 );							\
 if((_data_) != NULL)					\
  gtk_object_set_data_full(				\
   GTK_OBJECT(w),					\
   EDV_OPEN_TO_MENU_ITEM_KEY,				\
   (_data_),						\
   edv_open_to_menu_data_destroy_cb			\
  );							\
}

#define NEW_MENU_SEPARATOR {				\
 w = GUIMenuItemCreate(					\
  menu,							\
  GUI_MENU_ITEM_TYPE_SEPARATOR,				\
  NULL,							\
  NULL,							\
  NULL,							\
  0, 0,							\
  NULL, NULL						\
 );							\
}

	/* No MIME Type specified? */
	if(m == NULL)
	{
		/* Call the paths list get callback to obtain a list
		 * of paths to create the Open To menu
		 */
		if(paths_list_get_cb != NULL)
		{
			/* Get the list of selected paths */
			GList *paths_list = paths_list_get_cb(
				paths_list_get_data
			);
			if(paths_list != NULL)
			{
				/* Create the menu item based on the last selected
				 * object's type and permissions
				 */
				const gchar *path;
				GList *glist = g_list_last(paths_list);
				EDVVFSObject *obj;

				if(glist == NULL)
					glist = paths_list;

				/* Obtain the command name by examining the last
				 * selected object's type and permissions
				 */

				command_name = "Unknown";

				path = (const gchar *)glist->data;

				/* Get the destination object's statistics */
				obj = edv_vfs_object_stat(path);
				if(obj != NULL)
				{
					/* File */
					if(EDV_VFS_OBJECT_IS_FILE(obj))
					{
						const EDVPermissionFlags p = obj->permissions;
						if((p & EDV_PERMISSION_UX) ||
						   (p & EDV_PERMISSION_GX) ||
						   (p & EDV_PERMISSION_OX)
						)
							command_name = "Execute";
						else
							command_name = "View";
					}
					/* Directory */
					else if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
					{
						command_name = "Go To";
					}

					edv_vfs_object_delete(obj);
				}
				g_list_foreach(
					paths_list,
					(GFunc)g_free,
					NULL
				);
				g_list_free(paths_list);

				d = edv_open_cb_data_new();
				if(d != NULL)
				{
					d->core = core;
					d->paths_list_get_cb = paths_list_get_cb;
					d->paths_list_get_data = paths_list_get_data;
					d->goto_directory_cb = goto_directory_cb;
					d->goto_directory_data = goto_directory_data;
					d->command_name = STRDUP(command_name);
					d->toplevel = toplevel;
					d->verbose = TRUE;
					NEW_MENU_ITEM_LABEL(
						command_name,
						NULL,
						d
					);
				}
			}
			else
			{
				w = GUIMenuItemCreate(
					menu,
					GUI_MENU_ITEM_TYPE_LABEL,
					accelgrp,
					NULL,
					"(None)",
					0, 0,
					NULL, NULL
				);
				gtk_widget_set_sensitive(w, FALSE);
			}
		}
		else
		{
			w = GUIMenuItemCreate(
				menu,
				GUI_MENU_ITEM_TYPE_LABEL,
				accelgrp,
				NULL,
				"(None)",
				0, 0,
				NULL, NULL
			);
			gtk_widget_set_sensitive(w, FALSE);
		}
		return(menu);
	}

	/* MIME Type specified, create the menu items by the MIME
	 * Type's class
	 */
	switch(m->mt_class)
	{
	  case EDV_MIME_TYPE_CLASS_SYSTEM:
		if(paths_list_get_cb != NULL)
		{
			/* Create a menu item based on the last selected
			 * object's type and permissions
			 */
			GList *paths_list = paths_list_get_cb(
				paths_list_get_data
			);
			if(paths_list != NULL)
			{
				const gchar *path;
				GList *glist = g_list_last(paths_list);
				EDVVFSObject *obj;

				if(glist == NULL)
					glist = paths_list;

				command_name = "Unknown";

				path = (const gchar *)glist->data;

				/* Get this object's destination statistics */
				obj = edv_vfs_object_stat(path);
				if(obj != NULL)
				{
					/* File? */
					if(EDV_VFS_OBJECT_IS_FILE(obj))
					{
						const EDVPermissionFlags p = obj->permissions;
						if((p & EDV_PERMISSION_UX) ||
						   (p & EDV_PERMISSION_GX) ||
						   (p & EDV_PERMISSION_OX)
						)
							command_name = "Execute";
						else
							command_name = "View";
					}
					/* Directory? */
					else if(EDV_VFS_OBJECT_IS_DIRECTORY(obj))
					{
						command_name = "Go To";
					}

					edv_vfs_object_delete(obj);
				}
				g_list_foreach(
					paths_list,
					(GFunc)g_free,
					NULL
				);
				g_list_free(paths_list);

				d = edv_open_cb_data_new();
				if(d != NULL)
				{
					d->core = core;
					d->paths_list_get_cb = paths_list_get_cb;
					d->paths_list_get_data = paths_list_get_data;
					d->goto_directory_cb = goto_directory_cb;
					d->goto_directory_data = goto_directory_data;
					d->command_name = STRDUP(command_name);
					d->toplevel = toplevel;
					d->verbose = TRUE;
					NEW_MENU_ITEM_LABEL(
						command_name,
						NULL,
						d
					);
				}
			}
			else
			{
				w = GUIMenuItemCreate(
					menu,
					GUI_MENU_ITEM_TYPE_LABEL,
					accelgrp,
					NULL,
					"(None)",
					0, 0,
					NULL, NULL
				);
				gtk_widget_set_sensitive(w, FALSE);
			}
		}
		else
		{
			w = GUIMenuItemCreate(
				menu,
				GUI_MENU_ITEM_TYPE_LABEL,
				accelgrp,
				NULL,
				"(None)",
				0, 0,
				NULL, NULL
			);
			gtk_widget_set_sensitive(w, FALSE);
		}
		break;

	  case EDV_MIME_TYPE_CLASS_FORMAT:
	  case EDV_MIME_TYPE_CLASS_PROGRAM:
	  case EDV_MIME_TYPE_CLASS_UNIQUE:
		switch(m->handler)
		{
		  case EDV_MIME_TYPE_HANDLER_COMMAND:
			if(m->commands_list != NULL)
			{
				const gchar *command_name;
				GList *glist;
				EDVMIMETypeCommand *cmd;

				for(glist = m->commands_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					cmd = EDV_MIME_TYPE_COMMAND(glist->data);
					if(cmd == NULL)
						continue;

					command_name = cmd->name;
					d = edv_open_cb_data_new();
					if(d != NULL)
					{
						d->core = core;
						d->paths_list_get_cb = paths_list_get_cb;
						d->paths_list_get_data = paths_list_get_data;
						d->goto_directory_cb = goto_directory_cb;
						d->goto_directory_data = goto_directory_data;
						d->command_name = STRDUP(command_name);
						d->toplevel = toplevel;
						d->verbose = TRUE;
						NEW_MENU_ITEM_LABEL(
							command_name,
							NULL,
							d
						);
					}
				}
			}
			else
			{
				w = GUIMenuItemCreate(
					menu,
					GUI_MENU_ITEM_TYPE_LABEL,
					accelgrp,
					NULL,
					"(None)",
					0, 0,
					NULL, NULL
				);
				gtk_widget_set_sensitive(w, FALSE);
			}
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER:
			command_name = "Archiver";
			d = edv_open_cb_data_new();
			if(d != NULL)
			{
				d->core = core;
				d->paths_list_get_cb = paths_list_get_cb;
				d->paths_list_get_data = paths_list_get_data;
				d->goto_directory_cb = goto_directory_cb;
				d->goto_directory_data = goto_directory_data;
				d->command_name = STRDUP(command_name);
				d->toplevel = toplevel;
				d->verbose = TRUE;
				NEW_MENU_ITEM_LABEL(
					command_name,
					NULL,
					d
				);
			}
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER:
			command_name = "Image Browser";
			d = edv_open_cb_data_new();
			if(d != NULL)
			{
				d->core = core;
				d->paths_list_get_cb = paths_list_get_cb;
				d->paths_list_get_data = paths_list_get_data;
				d->goto_directory_cb = goto_directory_cb;
				d->goto_directory_data = goto_directory_data;
				d->command_name = STRDUP(command_name);
				d->toplevel = toplevel;
				d->verbose = TRUE;
				NEW_MENU_ITEM_LABEL(
					command_name,
					NULL,
					d
				);
			}
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN:
			command_name = "Recycle Bin";
			d = edv_open_cb_data_new();
			if(d != NULL)
			{
				d->core = core;
				d->paths_list_get_cb = paths_list_get_cb;
				d->paths_list_get_data = paths_list_get_data;
				d->goto_directory_cb = goto_directory_cb;
				d->goto_directory_data = goto_directory_data;
				d->command_name = STRDUP(command_name);
				d->toplevel = toplevel;
				d->verbose = TRUE;
				NEW_MENU_ITEM_LABEL(
					command_name,
					NULL,
					d
				);
			}
			break;
		}
		break;
	}

#undef NEW_MENU_ITEM_LABEL
#undef NEW_MENU_SEPARATOR

	return(menu);
}
