#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "pulist.h"
#include "cdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_utils_gtk.h"
#include "edv_open.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


static gchar *edv_open_strip_extension(const gchar *s);
static gboolean edv_open_is_file_executable(const gchar *path);

/* Command Formatting */
gchar *edv_open_format_command(
	EDVCore *core,
	const gchar *cmd,
	GList *paths_list
);

/* Open */
static gint edv_open_iteration(
	EDVCore *core,
	const gchar *path,
	const gchar *command_name,
	GtkWidget *toplevel,
	const gboolean verbose
);
gint edv_open(
	EDVCore *core,
	GList *paths_list,
	const gchar *command_name,
	GtkWidget *toplevel,
	const gboolean verbose
);

/* Open With */
gint edv_open_with(
	EDVCore *core,
	GList *paths_list,
	const gchar *command_name,
	GtkWidget *toplevel,
	const gboolean verbose
);

/* Open Command */
gint edv_open_command(
	EDVCore *core,
	GList *paths_list,
	const gchar *command,
	GtkWidget *toplevel,
	const gboolean verbose
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns a copy of the path or name without the extension.
 */
static gchar *edv_open_strip_extension(const gchar *s)
{
	gint len;
	gchar *path_wo_ext;
	const gchar	*name,
			*ext;

	if(s == NULL)
		return(NULL);

	/* Get the name portion which starts after the directory
	 * deliminator (if any)
	 */
	name = (const gchar *)strrchr(
		(const char *)s,
		G_DIR_SEPARATOR
	);
	if(name != NULL)
		name++;
	else
		name = s;

	/* If the name starts with a '.' character (hidden names) then
	 * seek past the '.' character
	 */
	if(*name == '.') 
		name++;

	/* Get the extension */
	ext = (const gchar *)strchr((const char *)name, '.');

	/* No extension? */
	if(ext == NULL)
		return(STRDUP(s));

	/* Get the path without the extension */
	len = (gint)(ext - s);
	path_wo_ext = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	if(path_wo_ext == NULL)
		return(NULL);

	if(len > 0)
		(void)memcpy(
			path_wo_ext,
			s,
			len * sizeof(gchar)
		);
	path_wo_ext[len] = '\0';

	/* Need to tack on the ending double quote character? */
	if(*path_wo_ext == '"')
	{
		gchar *s2 = g_strconcat(
			path_wo_ext,
			"\"",
			NULL
		);
		if(s2 != NULL)
		{
			g_free(path_wo_ext);
			path_wo_ext = s2;
		}
	}

	return(path_wo_ext);
}

/*
 *	Checks if the object exists, is a file, and is set executable.
 */
static gboolean edv_open_is_file_executable(const gchar *path)
{
	EDVPermissionFlags permissions;
	EDVVFSObject *obj = edv_vfs_object_stat(path);
	if(obj == NULL)
		return(FALSE);

	/* Not a file? */
	if(!EDV_VFS_OBJECT_IS_FILE(obj))
	{
		edv_vfs_object_delete(obj);
		return(FALSE);
	}

	/* Is this file executable? */
	permissions = obj->permissions;
	if((permissions & EDV_PERMISSION_UX) ||
	   (permissions & EDV_PERMISSION_GX) ||
	   (permissions & EDV_PERMISSION_OX)
	)
	{
		edv_vfs_object_delete(obj);
		return(TRUE);
	}

	edv_vfs_object_delete(obj);

	return(FALSE);
}

/*
 *	Performs token substitutions on the command string.
 *
 *	The core specifies the core, who's values will be used in the
 *	substitutions.
 *
 *	The cmd specifies the original (unsubstituted) command string
 *	with tokens.
 *
 *	The paths_list specifies the list of gchar * full paths to
 *	the selected objects.
 *
 *	Returns a dynamically allocated string describing the command
 *	with all the substitutions made.
 *
 *	Reminder: There is a sister function in lib/edv_open.c which
 *	needs to perform equvilently to this function.
 */
gchar *edv_open_format_command(
	EDVCore *core,
	const gchar *cmd,
	GList *paths_list
)
{
	const gchar	*path,
			*first_path,
			*last_path;
	gchar		*s,
			*cmd_rtn,
			*cwd_quoted,
			*display_quoted,
			*home_quoted,
			*names_quoted,
			*paths_quoted,
			*first_name_quoted, *first_name_wo_ext_quoted,
			*last_name_quoted, *last_name_wo_ext_quoted,
			*first_path_quoted, *first_path_wo_ext_quoted,
			*last_path_quoted, *last_path_wo_ext_quoted,
			*pid_quoted;
	GList *glist;

	if((core == NULL) || (cmd == NULL))
		return(NULL);

	/* Begin creating substitution values */

	/* Current the current working directory */
	s = edv_getcwd();
	cwd_quoted = g_strdup_printf(
		"\"%s\"",
		(s != NULL) ? s : ""
	);
	g_free(s);

	/* Display address */
	display_quoted = g_strdup_printf(
		"\"%s\"",
		g_getenv(ENV_VAR_NAME_DISPLAY)
	);

	/* Home directory */
	home_quoted = g_strdup_printf(
		"\"%s\"",
		core->home_path
	);

	/* All selected object names */
	names_quoted = g_strdup("");
	for(glist = paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		path = (gchar *)glist->data;
		if(path == NULL)
			continue;

		s = g_strconcat(
			names_quoted,
			"\"",
			g_basename(path),
			(glist->next != NULL) ? "\" " : "\"",
			NULL
		);
		g_free(names_quoted);
		names_quoted = s;
	}

	/* All selected object paths */
	paths_quoted = g_strdup("");
	for(glist = paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		path = (gchar *)glist->data;
		if(path == NULL)
			continue;

		s = g_strconcat(
			paths_quoted,
			"\"",
			path,
			(glist->next != NULL) ? "\" " : "\"",
			NULL
		);
		g_free(paths_quoted);
		paths_quoted = s;
	}

	/* First selected object name and path */
	glist = g_list_first(paths_list);
	first_path = (glist != NULL) ? (gchar *)glist->data : NULL;   
	if(first_path == NULL)                                        
		first_path = "";
	first_name_quoted = g_strdup_printf("\"%s\"", g_basename(first_path));
	first_name_wo_ext_quoted = edv_open_strip_extension(first_name_quoted);
	first_path_quoted = g_strdup_printf("\"%s\"", first_path);
	first_path_wo_ext_quoted = edv_open_strip_extension(first_path_quoted);

	/* Last selected object name and path */
	glist = g_list_last(paths_list);
	last_path = (glist != NULL) ? (gchar *)glist->data : NULL;
	if(last_path == NULL)
		last_path = "";
	last_name_quoted = g_strdup_printf("\"%s\"", g_basename(last_path));
	last_name_wo_ext_quoted = edv_open_strip_extension(last_name_quoted);
	last_path_quoted = g_strdup_printf("\"%s\"", last_path);
	last_path_wo_ext_quoted = edv_open_strip_extension(last_path_quoted);
		


	/* Process ID */
	pid_quoted = g_strdup_printf("\"%i\"", core->pid);


	/* Begin formatting the command, always use the quoted string
	 * as the value and substitute for the quoted token first then
	 * the unquoted token
	 */
	cmd_rtn = STRDUP(cmd);

#define CMDSUB(_token_,_val_)	{			\
 gchar *s = edv_strsub(cmd_rtn, (_token_), (_val_));	\
 g_free(cmd_rtn);					\
 cmd_rtn = s;						\
}

	/* "%cwd" - Current working directory */
	CMDSUB("\"%cwd\"", cwd_quoted);
	CMDSUB("%cwd", cwd_quoted);

	/* "%display" - Display address */
	CMDSUB("\"%display\"", display_quoted);
	CMDSUB("%display", display_quoted);    

	/* "%home" - Home directory */
	CMDSUB("\"%home\"", home_quoted);
	CMDSUB("%home", home_quoted);


	/* "%names" - Names of all the selected objects (without
	 * paths)
	 */
	CMDSUB("\"%names\"", names_quoted);
	CMDSUB("%names", names_quoted);

	/* "%paths" - Full paths of all the selected objects */
	CMDSUB("\"%paths\"", paths_quoted);
	CMDSUB("%paths", paths_quoted);


	/* "%first_name_wo_ext" - Name of the first selected object
	 * (without path or extension)
	 */
	CMDSUB("\"%first_name_wo_ext\"", first_name_wo_ext_quoted);
	CMDSUB("%first_name_wo_ext", first_name_wo_ext_quoted);

	/* "%first_name" - Name of the first selected object (without
	 * path)
	 */
	CMDSUB("\"%first_name\"", first_name_quoted);
	CMDSUB("%first_name", first_name_quoted);

	/* "%first_path_wo_ext" - Full path to the last selected
	 * object without the extension
	 */
	CMDSUB("\"%first_path_wo_ext\"", first_path_wo_ext_quoted);
	CMDSUB("%first_path_wo_ext", first_path_wo_ext_quoted);

	/* "%first_path" - Full path to the last selected object */
	CMDSUB("\"%first_path\"", first_path_quoted);
	CMDSUB("%first_path", first_path_quoted);


	/* "%name_wo_ext" - Name of the last selected object (without
	 * path or ext)
	 */
	CMDSUB("\"%name_wo_ext\"", last_name_wo_ext_quoted);
	CMDSUB("%name_wo_ext", last_name_wo_ext_quoted);

	/* "%name" - Name of last selected object (without path) */
	CMDSUB("\"%name\"", last_name_quoted);
	CMDSUB("%name", last_name_quoted);

	/* "%path_wo_ext" - Full path to the last selected object
	 * without the extension
	 */
	CMDSUB("\"%path_wo_ext\"", last_path_wo_ext_quoted);
	CMDSUB("%path_wo_ext", last_path_wo_ext_quoted);

	/* "%path" - Full path to the last selected object */
	CMDSUB("\"%path\"", last_path_quoted);
	CMDSUB("%path", last_path_quoted);


	/* "%pid" - This program's process id */
	CMDSUB("\"%pid\"", pid_quoted);
	CMDSUB("%pid", pid_quoted);


	/* "%pe" - Full path to the last selected object without
	 * the extension
	 */
	CMDSUB("\"%pe\"", last_path_wo_ext_quoted);
	CMDSUB("%pe", last_path_wo_ext_quoted);

	/* "%p" - Full path to the last selected object */
	CMDSUB("\"%p\"", last_path_quoted);
	CMDSUB("%p", last_path_quoted);

	/* "%s" - Full path to the last selected object */
	CMDSUB("\"%s\"", last_path_quoted);
	CMDSUB("%s", last_path_quoted);

#undef CMDSUB

	g_free(cwd_quoted);
	g_free(display_quoted);
	g_free(home_quoted);
	g_free(names_quoted);
	g_free(paths_quoted);
	g_free(first_name_quoted);
	g_free(first_name_wo_ext_quoted);
	g_free(last_name_quoted);
	g_free(last_name_wo_ext_quoted);
	g_free(first_path_quoted);
	g_free(first_path_wo_ext_quoted);
	g_free(last_path_quoted);
	g_free(last_path_wo_ext_quoted);
	g_free(pid_quoted);

	return(cmd_rtn);
}


/*
 *	Opens the object specified by path.
 *
 *	The MIME Type appropriate for the object will be looked up and
 *	if it is found then the program or method specified by the MIME
 *	Type to open the object will be used.
 *
 *	The command_name specifies which MIME Type command must be used
 *	to open the object.
 *
 *	If stdout_path_rtn or stderr_path_rtn are not NULL, then paths
 *	to the stdout and stderr files will be returned. The calling
 *	function must delete the returned strings.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value or no MIME Type found.
 *	-3	Systems error or memory allocation error.
 */
static gint edv_open_iteration(
	EDVCore *core,
	const gchar *path,
	const gchar *command_name,
	GtkWidget *toplevel,
	const gboolean verbose
)
{
	gint status;
	CfgList *cfg_list = core->cfg_list;
	EDVMIMEType *m;
	EDVMIMETypeCommand *cmd;

	if(STRISEMPTY(path))
		return(-2);

	/* Match a MIME Type of class EDV_MIME_TYPE_CLASS_FORMAT or
	 * EDV_MIME_TYPE_CLASS_UNIQUE with the specified path
	 */
	m = edv_mime_types_list_match_path(
		core->mime_types_list,
		path
	);

	/* Unable to find a MIME Type appropriate for this object? */
	if(m == NULL)
	{
		const gchar *def_viewer_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_DEF_VIEWER);

		/* If this object is a file and is set executable, then
		 * attempt to run it
		 */
		if(edv_open_is_file_executable(path))
		{
			gint pid;
			gchar *parent, *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);

			/* Record the previous working dir */
			gchar *pwd = edv_getcwd();

			/* Set the new working dir as the parent of the
			 * specified path
			 */
			parent = g_dirname(path);
			if(parent != NULL)
			{
				edv_setcwd(parent);
				g_free(parent);
			}

			/* Execute the executable object */
			pid = edv_system_shell(
				path,
				shell_prog,
				shell_args
			);
			if(pid < 0)
			{
				const gint error_code = (gint)errno;
				if(verbose)
				{
					gchar *msg = g_strdup_printf(
"Unable to execute:\n\
\n\
    %s\n\
\n\
%s.",
						path,
						g_strerror(error_code)
				    );
					edv_play_sound_error(core);
					edv_message_error(
						"Open Failed",
						msg,
"Please check to make sure that the path to the program\n\
is correct and that the program is set executable.",
						toplevel
					);
					g_free(msg);
				}
				status = -1;
			}
			else
			{
				status = 0;
			}

			g_free(shell_prog);

			/* Change back to the previous working dir */
			if(pwd != NULL)
			{
				edv_setcwd(pwd);
				g_free(pwd);
			}

			return(status);
		}
		/* Object is not executable, use the default viewer to view
		 * this object
		 */
		else if(!STRISEMPTY(def_viewer_cmd))
		{
			gint pid;
			gchar	*parent,
				*cmd_s,
				*shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);
			GList *paths_list = NULL;

			/* Record the previous working dir */
			gchar *pwd = edv_getcwd();

			/* Set new working dir as the parent of the specified
			 * path
			 */
			parent = g_dirname(path);
			if(parent != NULL)
			{
				edv_setcwd(parent);
				g_free(parent);
			}

			/* Format the command by substituting the tokens */
			paths_list = g_list_append(
				paths_list,
				STRDUP(path)
			);
			cmd_s = edv_open_format_command(
				core,
				def_viewer_cmd,
				paths_list
			);
			if(paths_list != NULL)
			{
				g_list_foreach(
					paths_list,
					(GFunc)g_free,
					NULL
				);
				g_list_free(paths_list);
			}

			/* Execute the command */
			pid = edv_system_shell(
				cmd_s,
				shell_prog,
				shell_args
			);
			if(pid < 0)
			{
				const gint error_code = (gint)errno;
				if(verbose)
				{
					gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
						cmd_s,
						g_strerror(error_code)
				    );
					edv_play_sound_error(core);
					edv_message_error(
						"Open Failed",
						msg,
"Please check to make sure that the path to the program\n\
is correct and that the program is set executable.",
						toplevel
					);
					g_free(msg);
				}
				status = -1;
			}
			else
			{
				status = 0;
			}
			g_free(cmd_s);
			g_free(shell_prog);

			/* Change back to the previous working dir */
			if(pwd != NULL)
			{
				edv_setcwd(pwd);
				g_free(pwd);
			}

			return(status);
		}
		/* All else print warning about not finding a MIME Type,
		 * not being able to execute, and not being able to use a
		 * default viewer for this object
		 */
		else if(verbose)
		{
			gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de encontrar de tipo MIME asociado con objeto:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable pour trouver le Type de MIME associ avec l'objet:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, MIME Typ zu finden, mit objekt hat verbunden:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per trovare il Tipo di MIME  frequentato l'oggetto:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam MIME Type te vinden met voorwerp associeerde:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de achar Tipo de MIMO associado com objeto:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls finne MIME Type vrer som tilknyttet med objekt:\n\
\n\
    %s\n"
#else
"Unable to find a MIME Type associated with object:\n\
\n\
    %s\n"
#endif
				, path
			);
			edv_play_sound_warning(core);
			edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Abra Fallado",
msg,
"Para definir un de tipo MIME para este objeto,\n\
va a Ver->De Tipo MIME...\n\
\n\
Para la ayuda adicional a establecer de tipo MIME,\n\
vea Ayuda->De Tipo MIME...",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir A Echou",
msg,
"Pour dfinir un Type de MIME pour cet objet,\n\
Aller Regarder->Les Types De MIME...\n\
\n\
Pour l'aide supplmentaire sur monte les Types de MIME,\n\
L'Aide D'vch -> Les Types De MIME...",
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Versagt",
msg,
"Um einen MIME Typ fr dieses objekt zu definieren,\n\
gehen sie, Um->MIME Typen Anzusehen...\n\
\n\
Fr zustzliche Hilfe stellend auf MIME Typen auf,\n\
sieht Hilfe->MIME Typen...",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aprire Fallito",
msg,
"Per definire un Tipo di MIME per quest'oggetto,\n\
andare Osservare->I Tipi Di MIME...\n\
\n\
Per l'aiuto ulteriore su installa i Tipi di MIME,\n\
vedere L'Aiuto->I Tipi Di MIME...",
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Verzuimde",
msg,
"Om een MIME Type voor deze voorwerp te definiren,\n\
Ga Om->MIME Typen Te Bekijken...\n\
\n\
Voor bijkomende hulp op opstellen van MIME Typen,\n\
ziet Hulp->MIME Typen...",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Abra Fracassado",
msg,
"Definir um Tipo de MIMO para este objeto,\n\
vai Ver->Tipos de MIMO...\n\
\n\
Para ajuda adicional em armar Tipos de MIMO,\n\
vem Ajuda->Tipos De MIMO...",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen Failed",
msg,
"Definere en MIME Type for dette objektet,\n\
dra til View->MIME Types...\n\
\n\
For ytterligere hjelp p innstilling ser opp MIME Types,\n\
Help->MIME Types...",
#else
"Open Failed",
msg,
"To define a MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
				toplevel
			);
			g_free(msg);
		}
		return(-2);
	}

	/* At this point a MIME Type has been matched, find out how we
	 * should open it by checking the MIME Type's handler code
	 *
	 * Any handler code specifying an internal method will return
	 * within this block
	 */
	switch(m->handler)
	{
	  case EDV_MIME_TYPE_HANDLER_COMMAND:
		/* Open this object with a command, so go on with the
		 * rest of this function which deals with opening by the
		 * MIME Type's command
		 */
		break;

	  case EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER:
		edv_new_archiver(
			core,
			path,
			NULL,			/* No password */
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		return(0);
		break;

	  case EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER:
		edv_new_image_browser(
			core,
			path,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		return(0);
		break;

	  case EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN:
		edv_map_recycle_bin(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);
		return(0);
		break;

	  default:
		edv_play_sound_warning(core);
		edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"El Tratante No Sostuvo",
"El tratante definido para el de tipo MIME asociado\n\
con este objeto no es sostenido.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"L'Agent N'A Pas Soutenu",
"L'agent dfini pour le Type de MIME associ avec cet\n\
objet n'est pas soutenu.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Behandler Hat Nicht Untersttzt",
"Der Behandler ist fr den MIME Typ, der mit diesem\n\
objekt verbunden worden ist, nicht untersttzt hat\n\
definiert.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Addestratore Non Ha Sostenuto",
"L'Addestratore definito per il Tipo di MIME  frequentato\n\
quest'oggetto non  sostenuto.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Handler Steunde Niet",
"De handler is voor het MIME Type, die met deze voorwerp\n\
geassocieerd is niet gesteund definieerde.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Manipulador Nao Apoiou",
"O Manipulador definido para o Tipo de MIMO associado\n\
com este objeto nao  apoiado.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Behandler Not Supported",
"Det behandler-definerte for den MIME Type som vrer\n\
tilknyttet med dette objektet ikke sttter.\n",
#else
"Handler Not Supported",
"The handler specified for the MIME Type associated\n\
with this object is not supported.\n",
#endif
			NULL,
			toplevel
		);
		return(-2);
		break;
	}

	/* If this point has been reached then it means that an
	 * external command should be executed in order to open the
	 * object
	 *
	 * If the command name is specified then get the MIME Type's
	 * command that matches that command name, otherwise use the
	 * default (first) command defined on the MIME Type
	 */
	cmd = NULL;
	if(!STRISEMPTY(command_name))
	{
		/* The command name is specified, find a command on this
		 * MIME Type who's command name matches the specified
		 * command name
		 */
		cmd = edv_mime_type_match_command_by_name(
			m,
			command_name
		);
	}
	else
	{
		/* No command name specified, so use the default (the
		 * first) command defined on the MIME Type (if any)
		 */
		GList *glist = m->commands_list;
		if(glist != NULL)
			cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	}

	/* Matched MIME Type has no command? */
	if((cmd != NULL) ? STRISEMPTY(cmd->command) : TRUE)
	{
		if(verbose)
		{
			gchar *msg;
			if(!STRISEMPTY(command_name))
				msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type \"%s\" no tiene una orden de \"%s\".\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME Type \"%s\" n'a pas un ordre de \"%s\".\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type \"%s\" hat keinen \"%s\" Befehl.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type \"%s\" non ha un comando di \"%s\".\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type \"%s\" heeft geen \"%s\" bevel.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"MIME Type \"%s\" no tem um comando de \"%s\".\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type \"%s\" har ikke en \"%s\" kommando.\n"
#else
"The MIME Type \"%s\" does not have a \"%s\" command.\n"
#endif
					,
					m->type,
					command_name
				);
			else
				msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type \"%s\" no tiene ninguna orden definida.\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME Type \"%s\" n'a pas d'ordres dfinis.\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type \"%s\" hat irgendeine definierten Befehle nicht.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type \"%s\" non ha qualunque comandi definiti.\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type \"%s\" heeft geen gedefinieerde bevelen.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"MIME Type \"%s\" no tem qualquer comandos definidos.\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type \"%s\" har ikke noen definerte kommandoer.\n"
#else
"The MIME Type \"%s\" does not have any defined commands.\n"
#endif
					,
					m->type
				);
			edv_play_sound_warning(core);
			edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Abra Fallado",
msg,
"Para redactar el de tipo MIME para este objeto,\n\
va a Ver->De Tipo MIME...\n\
\n\
Para la ayuda adicional a establecer de tipo MIME,\n\
Vea Ayuda->De Tipo MIME...",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir A Echou",
msg,
"Pour diter le Type de MIME pour cet objet,\n\
aller Regarder->Les Types De MIME...\n\
\n\
Pour l'aide supplmentaire sur monte les Types de MIME,\n\
L'Aide D'vch->Les Types De MIME...",
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Versagt",
msg,
"Um den MIME Typ fr dieses Objekt zu redigieren,\n\
gehen Sie, Um->MIME Typen Anzusehen...\n\
\n\
Fr zustzliche Hilfe stellend auf MIME Typen auf,\n\
sieht Hilfe->MIME Typen...",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aprire Fallito",
msg,
"Per redigere il Tipo di MIME per quest'oggetto,\n\
andare Osservare->I Tipi Di MIME...\n\
\n\
Per l'aiuto ulteriore su installa i Tipi di MIME,\n\
vedere L'Aiuto->I Tipi Di MIME...",
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Verzuimde",
msg,
"Om het MIME Type voor deze voorwerp te redigeren,\n\
ga Om->MIME Typen Te Bekijken...\n\
\n\
Voor bijkomende hulp op opstellen van MIME Typen,\n\
ziet Hulp->MIME Typen...",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Abra Fracassado",
msg,
"Editar o Tipo de MIMO para este objeto,\n\
vai Ver->Tipos De MIMO...\n\
\n\
Para ajuda adicional em armar Tipos de MIMO,\n\
vem Ajuda->Tipos De MIMO...",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen Failed",
msg,
"Redigere MIME Type for dette objektet,\n\
dra til View->MIME Types...\n\
\n\
For ytterligere hjelp p innstilling ser opp MIME Types,\n\
Help->MIME Types...",
#else
"Open Failed",
msg,
"To edit the MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
				toplevel
			);
			g_free(msg);
		}
		return(-2);
	}


	/* If the matched command's command is not an absolute path
	 * then it implies that it refers to another MIME Type
	 */
	if(*cmd->command != G_DIR_SEPARATOR)
	{
		/* Get the referenced MIME Type and command */
		gboolean got_match = FALSE;

		/* Record and use the current command value as the
		 * reference MIME Type
		 */
		const gchar *ref_type = cmd->command;

		GList *glist;

		/* Reset the contexts */
		m = NULL;
		cmd = NULL;

		/* Find the referenced MIME Type */
		for(glist = core->mime_types_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			m = EDV_MIME_TYPE(glist->data);
			if(m == NULL)
				continue;

			/* Check only MIME Types who's class is
			 * EDV_MIME_TYPE_CLASS_PROGRAM
			 */
			if(m->mt_class == EDV_MIME_TYPE_CLASS_PROGRAM)
			{
				if(m->type != NULL)
				{
					if(!strcmp(
						(const char *)m->type,
						(const char *)ref_type
					))
					{
						/* Found a referenced MIME Type, now get
						 * its first command and update cmd to
						 * reference it
						 *
						 * Note that only the first command will
						 * be obtained from a MIME Type of class
						 * EDV_MIME_TYPE_CLASS_PROGRAM
						 */
						GList *glist = m->commands_list;
						if(glist != NULL)
							cmd = EDV_MIME_TYPE_COMMAND(glist->data);
						got_match = TRUE;
						break;
					}
				}
			}
		}
		/* Unable to find the referenced MIME Type? */
		if(!got_match || (m == NULL))
		{
			/* No MIME Type associated for the given path */
			if(verbose)
			{
				gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de encontrar la clase de la Aplicacin \"%s\" de tipo MIME."
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable pour trouver \"%s\" de Type de MIME de classe d'Application."
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, Anwendung Klasse MIME Typ \"%s\" zu finden."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per trovare \"%s\" di Tipo di MIME di classe di Domanda."
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam Toepassing klas MIME Type \"%s\" te vinden."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de achar \"%s\" de Tipo de MIMO de classe de Aplicao."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls finne Application klasse MIME Type \"%s\"."
#else
"Unable to find an Application class MIME Type \"%s\"."
#endif
					,
					ref_type
				);
				edv_play_sound_warning(core);
				edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Abra Fallado",
msg,
"Para redactar el de tipo MIME para este objeto,\n\
va a Ver->De Tipo MIME...\n\
\n\
Para la ayuda adicional a establecer de tipo MIME,\n\
Vea Ayuda->De Tipo MIME...",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir A Echou",
msg,
"Pour diter le Type de MIME pour cet objet,\n\
aller Regarder->Les Types De MIME...\n\
\n\
Pour l'aide supplmentaire sur monte les Types de MIME,\n\
L'Aide D'vch->Les Types De MIME...",
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Versagt",
msg,
"Um den MIME Typ fr dieses Objekt zu redigieren,\n\
gehen Sie, Um->MIME Typen Anzusehen...\n\
\n\
Fr zustzliche Hilfe stellend auf MIME Typen auf,\n\
sieht Hilfe->MIME Typen...",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aprire Fallito",
msg,
"Per redigere il Tipo di MIME per quest'oggetto,\n\
andare Osservare->I Tipi Di MIME...\n\
\n\
Per l'aiuto ulteriore su installa i Tipi di MIME,\n\
vedere L'Aiuto->I Tipi Di MIME...",
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Verzuimde",
msg,
"Om het MIME Type voor deze voorwerp te redigeren,\n\
ga Om->MIME Typen Te Bekijken...\n\
\n\
Voor bijkomende hulp op opstellen van MIME Typen,\n\
ziet Hulp->MIME Typen...",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Abra Fracassado",
msg,
"Editar o Tipo de MIMO para este objeto,\n\
vai Ver->Tipos De MIMO...\n\
\n\
Para ajuda adicional em armar Tipos de MIMO,\n\
vem Ajuda->Tipos De MIMO...",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen Failed",
msg,
"Redigere MIME Type for dette objektet,\n\
dra til View->MIME Types...\n\
\n\
For ytterligere hjelp p innstilling ser opp MIME Types,\n\
Help->MIME Types...",
#else
"Open Failed",
msg,
"To edit the MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
					toplevel
				);
				g_free(msg);
			}
			return(-2);
		}
	}

	/* Referenced MIME Type has no defined command? */
	if((cmd != NULL) ? STRISEMPTY(cmd->command) : TRUE)
	{
		if(verbose)
		{
			gchar *msg;
			if(m != NULL)
				msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"\"%s\" de tipo MIME no tiene una orden definida."
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME \"%s\" de Type n'a pas un ordre dfini."
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Typ \"%s\" hat keinen definierten Befehl."
#elif defined(PROG_LANGUAGE_ITALIAN)
"\"%s\" di Tipo di MIME non ha un comando definito."
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type \"%s\" heeft geen gedefinieerd bevel."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"\"%s\" de Tipo de MIMO nao tem um comando definido."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type \"%s\" har ikke en definert kommando."
#else
"The MIME Type \"%s\" does not have a defined command."
#endif
					, m->type
				);
			else
				msg = STRDUP(
"Unable to find a MIME Type associated with this object."
				);
			edv_play_sound_warning(core);
			edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Abra Fallado",
msg,
"Para redactar el de tipo MIME para este objeto,\n\
va a Ver->De Tipo MIME...\n\
\n\
Para la ayuda adicional a establecer de tipo MIME,\n\
Vea Ayuda->De Tipo MIME...",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir A Echou",
msg,
"Pour diter le Type de MIME pour cet objet,\n\
aller Regarder->Les Types De MIME...\n\
\n\
Pour l'aide supplmentaire sur monte les Types de MIME,\n\
L'Aide D'vch->Les Types De MIME...",
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Versagt",
msg,
"Um den MIME Typ fr dieses Objekt zu redigieren,\n\
gehen Sie, Um->MIME Typen Anzusehen...\n\
\n\
Fr zustzliche Hilfe stellend auf MIME Typen auf,\n\
sieht Hilfe->MIME Typen...",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aprire Fallito",
msg,
"Per redigere il Tipo di MIME per quest'oggetto,\n\
andare Osservare->I Tipi Di MIME...\n\
\n\
Per l'aiuto ulteriore su installa i Tipi di MIME,\n\
vedere L'Aiuto->I Tipi Di MIME...",
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Verzuimde",
msg,
"Om het MIME Type voor deze voorwerp te redigeren,\n\
ga Om->MIME Typen Te Bekijken...\n\
\n\
Voor bijkomende hulp op opstellen van MIME Typen,\n\
ziet Hulp->MIME Typen...",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Abra Fracassado",
msg,
"Editar o Tipo de MIMO para este objeto,\n\
vai Ver->Tipos De MIMO...\n\
\n\
Para ajuda adicional em armar Tipos de MIMO,\n\
vem Ajuda->Tipos De MIMO...",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen Failed",
msg,
"Redigere MIME Type for dette objektet,\n\
dra til View->MIME Types...\n\
\n\
For ytterligere hjelp p innstilling ser opp MIME Types,\n\
Help->MIME Types...",
#else
"Open Failed",
msg,
"To edit the MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
				toplevel
			);
			g_free(msg);
		}
		return(-2);
	}

	/* We now have sufficient information to run the program */
	if(TRUE)
	{
		gint pid;
		gchar	*parent,
			*cmd_s,
			*shell_prog;
		const gchar	*shell_cmd = cmd->shell_command,
				*shell_args;
		GList *paths_list = NULL;

		/* Record the previous working dir */
		gchar *pwd = edv_getcwd();  

		/* If no shell command was specified by the MIME Type then
		 * use the shell command specified by the configuration
		 */
		if(STRISEMPTY(shell_cmd))
			shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL);

		/* Get the shell program and arguments */
		shell_args = edv_strarg(
			shell_cmd,
			&shell_prog,
			TRUE,			/* Parse escapes */
			TRUE			/* Parse quotes */
		);

		/* Set new working dir as the parent of the specified
		 * path
		 */
		parent = g_dirname(path);
		if(parent != NULL)
		{
			edv_setcwd(parent);
			g_free(parent);
		}

		/* Format the command by substituting the tokens */
		paths_list = g_list_append(
			paths_list,
			STRDUP(path)
		);
		cmd_s = edv_open_format_command(
			core,
			cmd->command,
			paths_list
		);
		if(paths_list != NULL)
		{
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		}

		/* Prepend the run in terminal command? */
		if(cmd->flags & EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL)
		{
			const gchar *cmd_terminal_run = EDV_GET_S(
				EDV_CFG_PARM_PROG_TERMINAL_RUN
			);
			if(!STRISEMPTY(cmd_terminal_run))
			{
				/* Format the command for run in terminal */
				gchar *s = g_strconcat(
					cmd_terminal_run,
					" ",
					cmd_s,
					NULL
				);
				g_free(cmd_s);
				cmd_s = s;
			}
			else
			{
				edv_play_sound_warning(core);
				edv_message_warning(
					"Open Failed",
"The \"Run In Terminal\" command is not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
					NULL,
					toplevel
				);
				g_free(cmd_s);
				g_free(shell_prog);

				/* Change back to the previous working dir */
				if(pwd != NULL)
				{
					edv_setcwd(pwd);
					g_free(pwd);
				}

				return(-1);
			}
		}

		/* Execute the command */
		pid = edv_system_shell(
			cmd_s,
			shell_prog,
			shell_args
		);
		if(pid < 0)
		{
			const gint error_code = (gint)errno;
			if(verbose)
			{
				gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
					cmd_s,
					g_strerror(error_code)
				);
				edv_play_sound_error(core);
				edv_message_error(
					"Open Failed",
					msg,
"Please check to make sure that the path to the program\n\
is correct and that the program is set executable.",
					toplevel
				);
				g_free(msg);
			}
			status = -1;
		}
		else
		{
			status = 0;
		}
		g_free(cmd_s);
		g_free(shell_prog);

		/* Change back to the previous working dir */
		if(pwd != NULL)
		{
			edv_setcwd(pwd);
			g_free(pwd);
		}
	}

	return(status);
}

/*
 *	Opens the object(s).
 *
 *	The core specifies the core.
 *
 *	The paths_list specifies a list of gchar * full paths of
 *	the objects to open.
 *
 *	The command_name specifies the name of the command on the
 *	matched MIME Type. If command_name is NULL then the default
 *	command on the matched MIME Type will be used.
 *
 *	If verbose is TRUE then error and warning messages will be
 *	displayed in a dialog.
 *
 *	The toplevel specifies the toplevel GtkWidget for the dialogs.
 *
 *	If stdout_path_rtn or stderr_path_rtn are not NULL then they
 *	will be set with a dynamically allocated string describing the
 *	stdout and stderr output file paths. The calling function must
 *	delete these returned strings.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	System error or memory allocation error.
 *	-4	User aborted.
 */
gint edv_open(
	EDVCore *core,
	GList *paths_list,
	const gchar *command_name,
	GtkWidget *toplevel,
	const gboolean verbose
)
{
	gint status;
	GList *glist;

	if((core == NULL) || (paths_list == NULL))
		return(-2);

	for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
	{
		status = edv_open_iteration(
			core,
			(gchar *)glist->data,
			command_name,
			toplevel,
			verbose
		);
		if(status != 0)
			return(status);
	}

	return(0);
}


/*
 *	Opens the objects by prompting the user to select a method from
 *	the Open With Popup List.
 *
 *	The core specifies the core.
 *
 *	The paths_list specifies a list of gchar * full paths of
 *	the objects to open.
 *
 *	The command_name specifies the name of the command on the
 *	matched MIME Type. If command_name is NULL then the default
 *	command on the matched MIME Type will be used.
 *
 *	If verbose is TRUE then error and warning messages will be
 *	displayed in a dialog.
 *
 *	The toplevel specifies the toplevel GtkWidget for the dialogs.
 *
 *	If stdout_path_rtn or stderr_path_rtn are not NULL then they
 *	will be set with a dynamically allocated string describing the
 *	stdout and stderr output file paths. The calling function must
 *	delete these returned strings.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	System error or memory allocation error.
 *	-4	User aborted.
 *	-6	An operation is already in progress.
 */
gint edv_open_with(
	EDVCore *core,
	GList *paths_list,
	const gchar *command_name,
	GtkWidget *toplevel,
	const gboolean verbose
)
{
	gint		mt_num,
			status;
	const gchar *value;
	GList *glist;
	GtkWidget *pulist;
	CfgList *cfg_list;
	EDVMIMEType *m;
	EDVMIMETypeCommand *cmd;

	if((core == NULL) || (paths_list == NULL))
		return(-2);

	cfg_list = core->cfg_list;

	/* Get the Open With Popup List */
	pulist = core->open_with_pulist;
	if(pulist == NULL)
		return(-1);
	if(PUListIsQuery(pulist))
		return(-6);

	/* Map the Open With Popup List */
	value = PUListMapQuery(
		pulist,
		NULL,				/* Use the last value */
		-1,				/* Default number of items visible */
		PULIST_RELATIVE_DOWN,
		NULL,				/* No relative widget, use
						 * pointer coordinates */
		NULL				/* No map widget */
	);
	/* User canceled? */
	if(value == NULL)
		return(-4);

	/* Get the MIME Type from the matched value */
	mt_num = (gint)PUListGetDataFromValue(
		pulist,
		value
	);
	if(mt_num < 0)
		return(-2);

	m = EDV_MIME_TYPE(g_list_nth_data(
		core->mime_types_list,
		(guint)mt_num
	));
	if(m == NULL)
		return(-2);

	/* MIME Type must be of class EDV_MIME_TYPE_CLASS_PROGRAM */
	if(m->mt_class != EDV_MIME_TYPE_CLASS_PROGRAM)
		return(-2);

	/* Get the first command on the MIME Type */
	glist = m->commands_list;
	if(glist != NULL)
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	else
		cmd = NULL;

	/* MIME Type has no defined command? */
	if((cmd != NULL) ? STRISEMPTY(cmd->command) : TRUE)
	{
		if(verbose)
		{
			gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"\"%s\" de tipo MIME no tiene una orden definida."
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME \"%s\" de Type n'a pas un ordre dfini."
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Typ \"%s\" hat keinen definierten Befehl."
#elif defined(PROG_LANGUAGE_ITALIAN)
"\"%s\" di Tipo di MIME non ha un comando definito."
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type \"%s\" heeft geen gedefinieerd bevel."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"\"%s\" de Tipo de MIMO nao tem um comando definido."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type \"%s\" har ikke en definert kommando."
#else
"The MIME Type \"%s\" does not have a defined command."
#endif
				,
				m->type
			);
			edv_play_sound_warning(core);
			edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Abra Fallado",
msg,
"Para redactar el de tipo MIME para este objeto,\n\
va a Ver->De Tipo MIME...\n\
\n\
Para la ayuda adicional a establecer de tipo MIME,\n\
Vea Ayuda->De Tipo MIME...",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir A Echou",
msg,
"Pour diter le Type de MIME pour cet objet,\n\
aller Regarder->Les Types De MIME...\n\
\n\
Pour l'aide supplmentaire sur monte les Types de MIME,\n\
L'Aide D'vch->Les Types De MIME...",
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Versagt",
msg,
"Um den MIME Typ fr dieses Objekt zu redigieren,\n\
gehen Sie, Um->MIME Typen Anzusehen...\n\
\n\
Fr zustzliche Hilfe stellend auf MIME Typen auf,\n\
sieht Hilfe->MIME Typen...",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aprire Fallito",
msg,
"Per redigere il Tipo di MIME per quest'oggetto,\n\
andare Osservare->I Tipi Di MIME...\n\
\n\
Per l'aiuto ulteriore su installa i Tipi di MIME,\n\
vedere L'Aiuto->I Tipi Di MIME...",
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Verzuimde",
msg,
"Om het MIME Type voor deze voorwerp te redigeren,\n\
ga Om->MIME Typen Te Bekijken...\n\
\n\
Voor bijkomende hulp op opstellen van MIME Typen,\n\
ziet Hulp->MIME Typen...",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Abra Fracassado",
msg,
"Editar o Tipo de MIMO para este objeto,\n\
vai Ver->Tipos De MIMO...\n\
\n\
Para ajuda adicional em armar Tipos de MIMO,\n\
vem Ajuda->Tipos De MIMO...",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen Failed",
msg,
"Redigere MIME Type for dette objektet,\n\
dra til View->MIME Types...\n\
\n\
For ytterligere hjelp p innstilling ser opp MIME Types,\n\
Help->MIME Types...",
#else
"Open Failed",
msg,
"To edit the MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
				toplevel
			);
			g_free(msg);
		}
		return(-2);
	}


	/* We now have sufficient information to run the program */
	if(TRUE)
	{
		gint pid;
		gchar	*cmd_s,
			*shell_prog;
		const gchar	*shell_cmd = cmd->shell_command,
				*shell_args;

		/* Record the previous working dir */
		gchar *pwd = edv_getcwd();

		/* If no shell command was specified by the MIME Type then
		 * use the shell command specified by the configuration
		 */
		if(STRISEMPTY(shell_cmd))
			shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL);

		/* Get the shell program and arguments */
		shell_args = edv_strarg(
			shell_cmd,
			&shell_prog,
			TRUE,			/* Parse escapes */
			TRUE			/* Parse quotes */
		);

		/* Set the working directory as the parent of the first
		 * path in the list
		 */
		if(paths_list != NULL)
		{
			const gchar *path = (const gchar *)paths_list->data;
			if(path != NULL)
			{
				gchar *parent = g_dirname(path);
				if(parent != NULL)
				{
					edv_setcwd(parent);
					g_free(parent);
				}
			}
		}

		/* Format the command by substituting the tokens */
		cmd_s = edv_open_format_command(
			core,
			cmd->command,
			paths_list
		);

		/* Prepend the run in terminal command? */
		if(cmd->flags & EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL)
		{
			const gchar *cmd_terminal_run = EDV_GET_S(
				EDV_CFG_PARM_PROG_TERMINAL_RUN
			);
			if(!STRISEMPTY(cmd_terminal_run))
			{
				/* Format the command for run in terminal */
				gchar *s = g_strconcat(
					cmd_terminal_run,
					" ",
					cmd_s,
					NULL
				);
				g_free(cmd_s);
				cmd_s = s;
			}
			else
			{
				edv_play_sound_warning(core);
				edv_message_warning(
					"Open Failed",
"The \"Run In Terminal\" command is not set.\n\
\n\
To set the \"Run In Terminal\" command go to\n\
Settings->Options...->Programs.",
					NULL,
					toplevel
				);
				g_free(cmd_s);
				g_free(shell_prog);

				/* Change back to the previous working dir */
				if(pwd != NULL)
				{
					edv_setcwd(pwd);
					g_free(pwd);
				}

				return(-1);
			}
		}

		/* Execute the command */
		pid = edv_system_shell(
			cmd_s,
			shell_prog,
			shell_args
		);
		if(pid < 0)
		{
			const gint error_code = (gint)errno;
			if(verbose)
			{
				gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
					cmd_s,
					g_strerror(error_code)
				);
				edv_play_sound_error(core);
				edv_message_error(
					"Open Failed",
					msg,
"Please check to make sure that the path to the program\n\
is correct and that the program is set executable.",
					toplevel
				);
				g_free(msg);
			}
			status = -1;
		}
		else
		{
			status = 0;
		}
		g_free(cmd_s);
		g_free(shell_prog);

		/* Change back to the previous working dir */
		if(pwd != NULL)
		{
			edv_setcwd(pwd);
			g_free(pwd);
		}
	}

	return(status);
}

/*
 *	Opens the objects with the command.
 *
 *	The core specifies the core.
 *
 *	The paths_list specifies a list of gchar * full paths of
 *	the objects to open.
 *
 *	The command specifies the unsubstituted command string.
 *
 *	If verbose is TRUE then error and warning messages will be
 *	displayed in a dialog.
 *
 *	The toplevel specifies the toplevel GtkWidget for the dialogs.
 *
 *	If stdout_path_rtn or stderr_path_rtn are not NULL then they
 *	will be set with a dynamically allocated string describing the
 *	stdout and stderr output file paths. The calling function must
 *	delete these returned strings.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	System error or memory allocation error.
 *	-4	User responded with `Cancel'.
 *	-6	Call is a renterent to an existing call to this function.
 */
gint edv_open_command(
	EDVCore *core,
	GList *paths_list,
	const gchar *command,
	GtkWidget *toplevel,
	const gboolean verbose
)
{
	gint		status,
			pid;
	const gchar	*path,
			*shell_cmd,
			*shell_args;
	gchar		*cmd_s,
			*pwd,
			*shell_prog;
	CfgList *cfg_list;

	if((core == NULL) || (paths_list == NULL) || STRISEMPTY(command))
		return(-2);

	cfg_list = core->cfg_list;

	/* Record the previous working dir */
	pwd = edv_getcwd();

	/* Get the shell program and arguments */
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL);
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Set the working directory as the parent of the first
	 * path in the list
	 */
	path = (const gchar *)paths_list->data;
	if(path != NULL)
	{
		gchar *parent = g_dirname(path);
		edv_setcwd(parent);
		g_free(parent);
	}

	/* Format the command by substituting the tokens */
	cmd_s = edv_open_format_command(
		core,
		command,
		paths_list
	);

	/* Execute the command */
	pid = edv_system_shell(
		cmd_s,
		shell_prog,
		shell_args
	);
	if(pid < 0)
	{
		const gint error_code = (gint)errno;
		if(verbose)
		{
			gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
				cmd_s,
				g_strerror(error_code)
			);
			edv_play_sound_error(core);
			edv_message_error(
				"Open Failed",
				msg,
"Please check to make sure that the path to the program\n\
is correct and that the program is set executable.",
				toplevel
			);
			g_free(msg);
		}
		status = -1;
	}
	else
	{
		status = 0;
	}
	g_free(cmd_s);
	g_free(shell_prog);

	/* Change back to the previous working dir */
	if(pwd != NULL)
	{
		edv_setcwd(pwd);
		g_free(pwd);
	}

	return(status);
}
