#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "url.h"
#include "cfg.h"
#include "cfg_fio.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"
#include "cfg_win.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_id.h"
#include "libendeavour2-base/edv_id_fio.h"
#include "edv_ids_list.h"
#include "edv_purge_obj.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_confirm.h"
#include "edv_device_mount.h"
#include "edv_history.h"
#include "edv_utils_gtk.h"
#include "vfs_browser.h"
#include "vfs_browser_tree.h"
#include "vfs_browser_op.h"
#include "image_browser.h"
#include "image_browser_op.h"
#include "archiver.h"
#include "archiver_list.h"
#include "archiver_op.h"
#include "recycle_bin.h"
#include "recycle_bin_list.h"
#include "recycle_bin_op.h"
#include "prop_dlg.h"
#include "find_win.h"
#include "history_win.h"
#include "options_win.h"
#include "customize_win.h"
#include "run_dlg.h"
#include "devices_list_win.h"
#include "mime_types_list_win.h"
#include "about_dlg.h"
#include "edv_help.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/* New/Map Window Nexus */
gint edv_new_window(
	EDVCore *core,
	const EDVWindowType win_type,
	const gchar *path,
	const gchar *extra,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);
gint edv_new_window_name(
	EDVCore *core,
	const gchar *win_name,
	const gchar *path,
	const gchar *extra,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);


/* New/Delete Map/Unmap Windows
 *
 * About Dialog
 */
void edv_map_about_dialog_page(
	EDVCore *core,
	const gchar *page_name,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);
void edv_map_about_dialog(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);

/* VFS Browser */
gint edv_new_vfs_browser(
	EDVCore *core,
	const gchar *path,
	const gchar *extra,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
void edv_delete_vfs_browser(
	EDVCore *core,
	const gint i
);

/* Image Browser */
gint edv_new_image_browser(
	EDVCore *core,
	const gchar *path,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
void edv_delete_image_browser(
	EDVCore *core,
	const gint i
);

/* Archiver */
gint edv_new_archiver(
	EDVCore *core,
	const gchar *path,
	const gchar *password,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);
void edv_delete_archiver(
	EDVCore *core,
	const gint i
);

/* Recycle Bin & Recycle Bin Desktop Icon */
void edv_map_recycle_bin_desktop_icon(EDVCore *core);
void edv_map_recycle_bin(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
);

/* MIME Types List Window */
void edv_map_mime_types(
	EDVCore *core,
	const gchar *type,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);

/* Devices List Window */
void edv_map_devices(
	EDVCore *core,
	const gchar *path,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);

/* History List Window */
void edv_map_history(
	EDVCore *core,
	const gint event_index,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);

/* Options Window */
void edv_map_options_page(
	EDVCore *core,
	const gchar *page_name,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);
void edv_map_options(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);

/* Customize Window */
void edv_map_customize_page(
	EDVCore *core,
	const gchar *page_name,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);
void edv_map_customize(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);

/* Properties Dialog */
static gint edv_new_properties_dialog_nexus(
	EDVCore *core,
	gint *existing_dialogs
);
static void edv_new_properties_dialog_select_page_move(
	EDVCore *core,
	EDVPropDlg *d,
	const gchar *page_name,
	const gint existing_dialogs,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);
gint edv_new_properties_dialog_vfs(
	EDVCore *core,
	const gchar *path,
	const gchar *page_name,
	GList *extended_properites_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);
gint edv_new_properties_dialog_recycle_bin(
	EDVCore *core,
	const gulong index,
	const gchar *page_name,
	GList *extended_properites_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);
gint edv_new_properties_dialog_archive(
	EDVCore *core,
	const gchar *path,
	const gchar *arch_path,
	const gchar *page_name,
	GList *extended_properites_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
);


/* Running */
void edv_run_device_check(
	EDVCore *core, EDVDevice *dev,
	GtkWidget *toplevel
);
void edv_run_device_tools(
	EDVCore *core, EDVDevice *dev,
	GtkWidget *toplevel
);
void edv_run_device_format(
	EDVCore *core, EDVDevice *dev,
	GtkWidget *toplevel
);

void edv_run_terminal(
	EDVCore *core,
	const gchar *cmd,
	const gchar *wd,
	GtkWidget *toplevel
);


/* Object Operations Dialog */
edv_obj_op_dlg_struct *edv_get_object_operations_dialog(EDVCore *core);


/* Window Mapping */
void edv_map_run_dialog_command(
	EDVCore *core,
	const gchar *command,
	const gchar *working_dir,
	GtkWidget *toplevel
);
void edv_map_run_dialog(
	EDVCore *core,
	GtkWidget *toplevel
);

void edv_map_find(
	EDVCore *core,
	const EDVLocationType location_type,
	const gchar *location,
	const gchar *value
);
void edv_map_find_vfs(
	EDVCore *core,
	EDVVFSBrowser *browser
);
void edv_map_find_image_browser(
	EDVCore *core,
	EDVImageBrowser *imbr
);
void edv_map_find_archive(
	EDVCore *core,
	EDVArchiver *archiver
);
void edv_map_find_recycle_bin(
	EDVCore *core,
	edv_recbin_struct *recbin
);


/* Update Resources & Widgets */
void edv_update_rc_styles(EDVCore *core);
void edv_update_id_popup_lists(EDVCore *core);
void edv_update_devices_popup_list(EDVCore *core);
void edv_update_mime_type_hint_indicies(EDVCore *core);
void edv_update_open_with_popup_list(EDVCore *core);


/* Refresh & Reset */
void edv_refresh(EDVCore *core);
void edv_reset(EDVCore *core);


/* Sync */
void edv_sync_edv(EDVCore *core);


/* Clearing */
void edv_clear_error_messages(EDVCore *core);
void edv_clear_all_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
);
void edv_clear_events_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
);
void edv_clear_location_bars_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
);
void edv_clear_run_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
);

/* Recycle Bin Size Check */
void edv_recycle_bin_size_check(
	EDVCore *core,
	GtkWidget *toplevel
);

/* Purge Recycle Bin */
void edv_purge_all_recycled_objects(
	EDVCore *core,
	const gboolean map_recbin,
	const gboolean show_progress,
	const gboolean interactive,
	GtkWidget *toplevel
);

/* Query Unmount Before Exit */
void edv_query_unmount_before_exit(EDVCore *core);

/* Network */
gint edv_internet_download_object(
	EDVCore *core,
	const URLStruct *url,			/* Source */
	const gchar *tar_path,			/* Target */
	GtkWidget *toplevel
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Creates a new window.
 *
 *	The win_type specifies the EDVWindowType which can be any of
 *	EDV_WINDOW_*.
 *
 *	The path specifies the initial location for most window types.
 *	If path is NULL then no initial location is set, some window
 *	types require a path be specified otherwise -2 will be returned.
 *
 *	The extra specifies the page or secondary argument, depending
 *	on the window's type, this value may or may not apply.
 *
 *	The toplevel specifies the relative toplevel GtkWidget.
 *
 *	Returns the window's index on success or negative on error.
 */
gint edv_new_window(
	EDVCore *core,
	const EDVWindowType win_type,
	const gchar *path,
	const gchar *extra,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	if((core == NULL) || (win_type == EDV_WINDOW_NONE))
	{
		errno = EINVAL;
		return(-2);
	}

	switch(win_type)
	{
	  case EDV_WINDOW_NONE:
		errno = EINVAL;
		return(-2);
		break;

	  case EDV_WINDOW_ABOUT_DIALOG:
		edv_map_about_dialog_page(
			core,
			extra,
			geometry_flags,
			geometry,
			toplevel
		);
		return(0);
		break;

	  case EDV_WINDOW_VFS_BROWSER:
		return(edv_new_vfs_browser(
			core,
			path,
			extra,
			geometry_flags,
			geometry
		));
		break;
	  case EDV_WINDOW_IMAGE_BROWSER:
		return(edv_new_image_browser(
			core,
			path,
			geometry_flags,
			geometry
		));
		break;
	  case EDV_WINDOW_ARCHIVER:
		return(edv_new_archiver(
			core,
			path,
			extra,			/* Password */
			geometry_flags,
			geometry
		));
		break;
	  case EDV_WINDOW_RECYCLE_BIN:
		edv_map_recycle_bin(
			core,
			geometry_flags,
			geometry
		);
		return(0);
		break;

	  case EDV_WINDOW_MIME_TYPES_LIST:
		edv_map_mime_types(
			core,
			extra,			/* MIME Type type */
			geometry_flags,
			geometry,
			toplevel
		);
		return(0);
		break;
	  case EDV_WINDOW_DEVICES_LIST:
		edv_map_devices(
			core,
			path,
			geometry_flags,
			geometry,
			toplevel
		);
		return(0);
		break;
	  case EDV_WINDOW_HISTORY_LIST:
		edv_map_history(
			core,
			STRISEMPTY(extra) ? -1 : (gint)atoi(extra),
			geometry_flags,
			geometry,
			toplevel
		);
		return(0);
		break;

	  case EDV_WINDOW_OPTIONS:
		edv_map_options_page(
			core,
			extra,			/* Page */
			geometry_flags,
			geometry,
			toplevel
		);
		return(0);
		break;
	  case EDV_WINDOW_CUSTOMIZE:
		edv_map_customize_page(
			core,
			extra,			/* Page */
			geometry_flags,
			geometry,
			toplevel
		);
		return(0);
		break;

	  case EDV_WINDOW_PROPERTIES_DIALOG:
		if(path != NULL)
		{
			const gchar *name = g_basename(path);
			if((name != NULL) ? (*name == '#') : FALSE)
				return(edv_new_properties_dialog_recycle_bin(
					core,
					(gulong)ATOL(name + 1),
					extra,			/* Page */
					NULL,			/* No extended properties list */
					geometry_flags,
					geometry,
					toplevel
				));
		}
		return(edv_new_properties_dialog_vfs(
			core,
			path,
			extra,			/* Page */
			NULL,			/* No extended properties list */
			geometry_flags,
			geometry,
			toplevel
		));
		break;
	  case EDV_WINDOW_FIND:
		edv_map_find(
			core,
			EDV_LOCATION_TYPE_VFS,
			((path == NULL) && (extra != NULL)) ?
				extra : path,
			extra
		);
		return(0);
		break;
	  case EDV_WINDOW_OBJECT_OPERATIONS_DIALOG:
		errno = EOPNOTSUPP;
		return(-9);			/* Not supported */
		break;
	  case EDV_WINDOW_RUN_DIALOG:
		edv_map_run_dialog_command(
			core,
			((path == NULL) && (extra != NULL)) ?
				extra : path,
			extra,
			toplevel
		);
		return(0);
		break;
	  case EDV_WINDOW_HELP:
		edv_help(
			core,
			extra,
			toplevel
		);
		return(0);
		break;
	}

	errno = EINVAL;

	return(-2);
}

/*
 *	Creates a new window by name.
 *
 *	The win_name specifies the window's name.
 *
 *	The path specifies the initial location for most window types.
 *	If path is NULL then no initial location is set, some window
 *	types require a path be specified otherwise -2 will be returned.
 *
 *	The extra specifies the page or secondary argument, depending
 *	on the window's type, this value may or may not apply.
 *
 *	The toplevel specifies the relative toplevel GtkWidget.
 *
 *	Returns the window's index on success or negative on error.
 */
gint edv_new_window_name(
	EDVCore *core,
	const gchar *win_name,
	const gchar *path,
	const gchar *extra,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	return(edv_new_window(
		core,
		edv_window_name_to_window_type(win_name),
		path,
		extra,
		geometry_flags,
		geometry,
		toplevel
	));
}


/*
 *	Creates/maps the About Dialog.
 *
 *	The page_name specifies the initial page to display. If
 *	page_name is NULL then the default page is displayed.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_about_dialog_page(
	EDVCore *core,
	const gchar *page_name,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	gboolean invalid_page = FALSE;
	AboutDlg *d;

	if(core == NULL)
		return;

	/* Get/create the About Dialog */
	d = core->about_dlg;
	if(d != NULL)
	{
		/* Unmap the About Dialog if it is already mapped */
		if(about_dlg_is_mapped(d))
			about_dlg_unmap(d);

		/* Set the geometry position as needed */
		if((geometry_flags != 0) && (geometry != NULL))
		{
			if((geometry_flags & GDK_GEOMETRY_X) ||
			   (geometry_flags & GDK_GEOMETRY_Y)
			)
				gtk_widget_set_uposition(
					d->toplevel,
					geometry->x, geometry->y
				);
		}
	}
	else
	{
		/* Create a new About Dialog */
		core->about_dlg = d = about_dlg_new(
			core,
			geometry_flags,
			geometry
		);
		if(d == NULL)
			return;
	}

	/* Select page? */
	if(!STRISEMPTY(page_name))
	{
		GtkNotebook *notebook = GTK_NOTEBOOK(d->notebook);
		if(notebook != NULL)
		{
			gint page_num;

			/* Convert the page name to the page number */
			if(!g_strcasecmp(page_name, "Credits"))
				page_num = 0;
			else if(!g_strcasecmp(page_name, "Copyright"))
				page_num = 1;
			else if(!g_strcasecmp(page_name, "Version"))
				page_num = 2;
			else
				page_num = -1;

			/* Got a page number? */
			if(page_num > -1)
			{
				gtk_notebook_set_page(notebook, page_num);
			}
			else
			{
				invalid_page = TRUE;
			}
		}
	}

	/* Map the About Dialog */
	about_dlg_map(d);

	/* Was an invalid page name specified? */
	if(!STRISEMPTY(page_name) && invalid_page)
	{
		gchar *msg = g_strdup_printf(
"Invalid About Dialog page name \"%s\".",
			page_name
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"About Dialog Page Switch Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
	}
}

void edv_map_about_dialog(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	edv_map_about_dialog_page(
		core,
		NULL,				/* No page */
		geometry_flags,
		geometry,
		toplevel
	);
}


/*
 *	Creates a new File Browser.
 *
 *	The new File Browser will be added to the core and mapped.
 *
 *	The path specifies the initial location.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	Returns the index of the new File Browser or negative on error.
 */
gint edv_new_vfs_browser(
	EDVCore *core,
	const gchar *path,
	const gchar *extra,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	gint		i,
					existing_windows;
	CfgList *cfg_list;
	EDVVFSBrowser *browser;

	if(core == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;
	existing_windows = core->total_browsers;

	/* Exceeded the maximum number of File Browsers? */
	i = EDV_GET_I(EDV_CFG_PARM_MAX_BROWSERS);
	if(i > -1)
	{
		if(existing_windows >= i)
		{
			errno = EDQUOT;			/* Quota exceeded */
			return(-8);
		}
	}

	/* Shift the window position based on the number of existing
	 * File Browsers
	 */
	if(existing_windows > 0)
	{
		gint	offset_x = EDV_GET_I(
				EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X
			),
			offset_y = EDV_GET_I(
				EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y
			),
			x = EDV_GET_I(
				EDV_CFG_PARM_BROWSER_X
			) + offset_x,
			y = EDV_GET_I(
				EDV_CFG_PARM_BROWSER_Y
			) + offset_y;
		EDV_SET_I(EDV_CFG_PARM_BROWSER_X, x);
		EDV_SET_I(EDV_CFG_PARM_BROWSER_Y, y);
	}

	/* Allocate more pointers */
	i = MAX(core->total_browsers, 0);
	core->total_browsers = i + 1;

	core->browser = (EDVVFSBrowser **)g_realloc(
		core->browser,
		core->total_browsers * sizeof(EDVVFSBrowser *)
	);
	if(core->browser == NULL)
	{
		core->total_browsers = 0;
		return(-3);
	}

	/* Create a new File Browser */
	core->browser[i] = browser = edv_vfs_browser_new(
		core,
		geometry_flags,
		geometry
	);
	if(browser != NULL)
	{
		gchar *startup_dir = edv_path_evaluate(
			NULL,
			STRISEMPTY(path) ?
				EDV_GET_S(EDV_CFG_PARM_DIR_START_UP) :
				path
		);
		GtkWidget *toplevel = browser->toplevel;

		/* If the startup directory is not a directory then replace
		 * it with its parent
		 */
		if(startup_dir != NULL)
		{
			EDVVFSObject *obj = edv_vfs_object_stat(startup_dir);
			if(obj != NULL)
			{
				if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
				{
					gchar *parent = g_dirname(startup_dir);
					if(parent != NULL)
					{
						g_free(startup_dir);
						startup_dir = parent;
					}
				}
				edv_vfs_object_delete(obj);
			}
			else
			{
				gchar *parent = g_dirname(startup_dir);
				if(parent != NULL)
				{
					g_free(startup_dir);
					startup_dir = parent;
				}
			}
		}

		edv_vfs_browser_set_busy(browser, TRUE);
		GUIBlockInput(toplevel, TRUE);

		/* Need to map it immediatly after creation, before any
		 * timeouts check on it
		 */
		edv_vfs_browser_map(browser);

		edv_vfs_browser_set_title(browser, NULL);
		edv_vfs_browser_update_display(browser);

		/* Was a directory tree origin specified? */
		if(!STRISEMPTY(extra))
		{
			/* Set the directory tree origin path, but do not create
			 * the toplevels or select it, this will be done as
			 * needed below
			 */
			g_free(browser->directory_ctree_origin_path);
			browser->directory_ctree_origin_path = edv_path_evaluate(
				NULL,
				extra
			);
		}

		/* Is the specified path within the Directory Tree? */
		if(edv_vfs_browser_is_path_from_tree_origin(browser, startup_dir))
		{
			/* Create the toplevels */
			edv_vfs_browser_tree_create_toplevel_nodes(browser);

			/* Select the specified path */
			edv_vfs_browser_tree_select_path(browser, startup_dir);
		}
		else
		{
			/* Set the directory tree origin, create the toplevels,
			 * and select the specified path
			 */
			edv_vfs_browser_tree_set_origin_path(browser, startup_dir);
		}

		g_free(startup_dir);

		GUIBlockInput(toplevel, FALSE);
		edv_vfs_browser_set_busy(browser, FALSE);

		/* Notify about the new window being created */
		edv_emit_window_created(
			core,
			EDV_WINDOW_VFS_BROWSER,
			browser
		);

		return(i);
	}
	else
		return(-1);
}

/*
 *	Deletes the File Browser.
 *
 *	The i specifies the File Browser's index in the list.
 */
void edv_delete_vfs_browser(
	EDVCore *core,
	const gint i
)
{
	gint j, n;
	gpointer win;

	if(core == NULL)
		return;

	if((i < 0) || (i >= core->total_browsers))
		return;

	win = core->browser[i];

	/* Delete this File Browser */
	edv_vfs_browser_delete(core->browser[i]);
/*	core->browser[i] = NULL; */

	/* Shift pointers */
	n = core->total_browsers - 1;
	for(j = i; j < n; j++)
		core->browser[j] = core->browser[j + 1];

	/* Reduce the total and reallocate pointers */
	core->total_browsers = n;
	if(core->total_browsers > 0)
	{
		core->browser = (EDVVFSBrowser **)g_realloc(
			core->browser,
			core->total_browsers * sizeof(EDVVFSBrowser *)
		);
		if(core->browser == NULL)
			core->total_browsers = 0;
	}
	else
	{
		g_free(core->browser);
		core->browser = NULL;
	}

	/* Notify about the window being deleted */
	edv_emit_window_deleted(
		core,
		EDV_WINDOW_VFS_BROWSER,
		win
	);
}

/*
 *	Creates a new Image Browser.
 *
 *	The new Image Browser will be added to the core and mapped.
 *
 *	The path specifies the initial location.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	Returns the index of the new Image Browser or negative on error.
 */
gint edv_new_image_browser(
	EDVCore *core,
	const gchar *path,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	gint i, existing_windows;
	CfgList *cfg_list;
	EDVImageBrowser *imbr;

	if(core == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;
	existing_windows = core->total_imbrs;

	/* Exceeded the maximum number of Image Browsers? */
	i = EDV_GET_I(EDV_CFG_PARM_MAX_IMBRS);
	if(i > -1)
	{
		if(existing_windows >= i)
		{
			errno = EDQUOT;			/* Quota exceeded */
			return(-8);
		}
	}

	/* Shift the window position based on the number of existing
	 * Image Browsers
	 */
	if(existing_windows > 0)
	{
		gint	offset_x = EDV_GET_I(
				EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X
			),
			offset_y = EDV_GET_I(
				EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y
			),
			x = EDV_GET_I(
				EDV_CFG_PARM_IMBR_X
			) + offset_x,
			y = EDV_GET_I(
				EDV_CFG_PARM_IMBR_Y
			) + offset_y;
		EDV_SET_I(EDV_CFG_PARM_IMBR_X, x);
		EDV_SET_I(EDV_CFG_PARM_IMBR_Y, y);
	}

	/* Allocate more pointers */
	i = MAX(core->total_imbrs, 0);
	core->total_imbrs = i + 1;

	core->imbr = (EDVImageBrowser **)g_realloc(
		core->imbr,
		core->total_imbrs * sizeof(EDVImageBrowser *)
	);
	if(core->imbr == NULL)
	{
		core->total_imbrs = 0;
		return(-3);
	}

	/* Create a new Image Browser */
	core->imbr[i] = imbr = edv_image_browser_new(
		core,
		geometry_flags,
		geometry
	);
	if(imbr != NULL)
	{
		gchar *startup_dir = edv_path_evaluate(
			NULL,
			!STRISEMPTY(path) ?
				path : EDV_GET_S(EDV_CFG_PARM_DIR_START_UP)
		);
		GtkWidget *toplevel = imbr->toplevel;

		/* If the startup directory is not a directory then replace
		 * it with its parent
		 */
		if(startup_dir != NULL)
		{
			EDVVFSObject *obj = edv_vfs_object_stat(startup_dir);
			if(obj != NULL)
			{
				if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj))
				{
					gchar *parent = g_dirname(startup_dir);
					if(parent != NULL)
					{
						g_free(startup_dir);
						startup_dir = parent;
					}
				}
				edv_vfs_object_delete(obj);
			}
			else
			{
				gchar *parent = g_dirname(startup_dir);
				if(parent != NULL)
				{
					g_free(startup_dir);
					startup_dir = parent;
				}
			}
		}

		edv_image_browser_set_busy(imbr, TRUE);
		GUIBlockInput(toplevel, TRUE);

		/* Need to map it immediatly after creation, before any
		 * timeouts check on it
		 */
		edv_image_browser_map(imbr);

		edv_image_browser_set_title(imbr, NULL);
		edv_image_browser_update_display(imbr);

		/* Get the initial listing of the startup directory */
		edv_image_browser_select_path(imbr, startup_dir);

		edv_image_browser_update_display(imbr);

		g_free(startup_dir);

		GUIBlockInput(toplevel, FALSE);
		edv_image_browser_set_busy(imbr, FALSE);

		/* Notify about the new window being created */
		edv_emit_window_created(
			core,
			EDV_WINDOW_IMAGE_BROWSER,
			imbr
		);

		return(i);
	}
	else
		return(-1);
}

/*
 *	Deletes the Image Browser.
 *
 *	The i specifies the Image Browser's index in the list.
 */
void edv_delete_image_browser(
	EDVCore *core,
	const gint i
)
{
	gint j, n;
	gpointer win;

	if(core == NULL)
		return;

	if((i < 0) || (i >= core->total_imbrs))
		return;

	win = core->imbr[i];

	/* Delete this Image Browser */
	edv_image_browser_delete(core->imbr[i]);
/*	core->imbr[i] = NULL; */

	/* Shift pointers */
	n = core->total_imbrs - 1;
	for(j = i; j < n; j++)
		core->imbr[j] = core->imbr[j + 1];

	/* Reduce the total and reallocate pointers */
	core->total_imbrs = n;
	if(core->total_imbrs > 0)
	{
		core->imbr = (EDVImageBrowser **)g_realloc(
			core->imbr,
			core->total_imbrs * sizeof(EDVImageBrowser *)
		);
		if(core->imbr == NULL)
			core->total_imbrs = 0;
	}
	else
	{
		g_free(core->imbr);
		core->imbr = NULL;
	}

	/* Notify about the window being deleted */
	edv_emit_window_deleted(
		core,
		EDV_WINDOW_IMAGE_BROWSER,
		win
	);
}

/*
 *	Creates a new Archiver.
 *
 *	The new Archiver will be added to the core and mapped.
 *
 *	The path specifies the archive that is to be opened on the
 *	Archiver after it has been mapped.
 *
 *	The password specifies the password.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	Returns the index of the new Archiver or negative on error.
 */
gint edv_new_archiver(
	EDVCore *core,
	const gchar *path,
	const gchar *password,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	gint		i,
			existing_windows;
	CfgList *cfg_list;
	EDVArchiver *archiver;

	if(core == NULL)
	{
		errno = EINVAL;
		return(-1);
	}

	cfg_list = core->cfg_list;
	existing_windows = core->total_archivers;

	/* Exceeded the maximum number of Archivers? */
	i = EDV_GET_I(EDV_CFG_PARM_MAX_ARCHIVERS);
	if(i > -1)
	{
		if(existing_windows >= i)
		{
			errno = EDQUOT;		/* Quota exceeded */
			return(-8);
		}
	}

	/* Shift the window position based on the number of existing
	 * Archivers
	 */
	if(existing_windows > 0)
	{
		const gint	offset_x = EDV_GET_I(
				EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X
			),
			offset_y = EDV_GET_I(
				EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y
			),
			x = EDV_GET_I(
				EDV_CFG_PARM_ARCHIVER_X
				) + offset_x,
			y = EDV_GET_I(
				EDV_CFG_PARM_ARCHIVER_Y
			) + offset_y;
		EDV_SET_I(EDV_CFG_PARM_ARCHIVER_X, x);
		EDV_SET_I(EDV_CFG_PARM_ARCHIVER_Y, y);
	}

	/* Allocate more pointers */
	i = MAX(core->total_archivers, 0);
	core->total_archivers = i + 1;

	core->archiver = (EDVArchiver **)g_realloc(
		core->archiver,
		core->total_archivers * sizeof(EDVArchiver *)
	);
	if(core->archiver == NULL)
	{
		core->total_archivers = 0;
		return(-3);
	}

	/* Create a new Archiver */
	core->archiver[i] = archiver = edv_archiver_new(
		core,
		geometry_flags,
		geometry
	);
	if(archiver != NULL)
	{
		/* Need to map it immediatly after creation, before any
		 * timeouts check on it
		 */
		edv_archiver_map(archiver);

		edv_archiver_set_title(archiver, NULL);
		edv_archiver_update_display(archiver);

		/* Notify about the new window being created */
		edv_emit_window_created(
			core,
			EDV_WINDOW_ARCHIVER,
			archiver
		);

		/* No archive to open at startup? */
		if(STRISEMPTY(path))
		{
			edv_archiver_list_reset_columns(archiver);
		}
		else
		{
			/* Open the specified archive */
			edv_archiver_open_archive(
				archiver,
				path,
				password,
				TRUE		/* List passively */
			);
			edv_archiver_update_display(archiver);
		}

		return(i);
	}
	else
		return(-1);
}

/*
 *	Deletes the Archiver.
 *
 *	The i specifies the Archiver's index in the list.
 */
void edv_delete_archiver(
	EDVCore *core,
	const gint i
)
{
	gint j, n;
	gpointer win;

	if(core == NULL)
		return;

	if((i < 0) || (i >= core->total_archivers))
		return;

	win = core->archiver[i];

	/* Delete this Archiver */
	edv_archiver_delete(core->archiver[i]);
/*	core->archiver[i] = NULL; */

	/* Shift pointers */
	n = core->total_archivers - 1;
	for(j = i; j < n; j++)
		core->archiver[j] = core->archiver[j + 1];

	/* Reduce the total and reallocate pointers */
	core->total_archivers = n;
	if(core->total_archivers > 0)
	{
		core->archiver = (EDVArchiver **)g_realloc(
			core->archiver,
			core->total_archivers * sizeof(EDVArchiver *)
		);
		if(core->archiver == NULL)
			core->total_archivers = 0;
	}
	else
	{
		g_free(core->archiver);
		core->archiver = NULL;
	}

	/* Notify about the window being deleted */
	edv_emit_window_deleted(
		core,
		EDV_WINDOW_ARCHIVER,
		win
	);
}


/*
 *	Creates/maps the Recycle Bin Desktop Icon.
 */
void edv_map_recycle_bin_desktop_icon(EDVCore *core)
{
	if(core == NULL)
		return;

	if(core->recbin_deskicon == NULL)
		core->recbin_deskicon = edv_recycle_bin_desktop_icon_new(core);
	edv_recycle_bin_desktop_icon_map(core->recbin_deskicon);
	edv_recycle_bin_desktop_icon_update_display(core->recbin_deskicon);
}

/*
 *	Creates/maps the Recycle Bin.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 */
void edv_map_recycle_bin(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry
)
{
	GtkWidget *toplevel;
	GtkCList *clist;
	const CfgList *cfg_list;
	edv_recbin_struct *recbin;

	if(core == NULL)
	{
		errno = EINVAL;
		return;
	}

	cfg_list = core->cfg_list;

	/* Create/get the Recycle Bin */
	recbin = core->recbin;
	if(recbin == NULL)
		core->recbin = recbin = edv_recycle_bin_new(
			core,
			geometry_flags,
			geometry
		);
	if(recbin == NULL)
		return;

	/* Unmap the Recycle Bin if it is already mapped */
	if(edv_recycle_bin_is_mapped(recbin))
		edv_recycle_bin_unmap(recbin);

	toplevel = recbin->toplevel;

	/* Need to map it immediatly after it is created so that it does
	 * not get deleted in the timeout callback when the recycle bin
	 * is updated further below
	 */
	edv_recycle_bin_map(recbin);

	edv_recycle_bin_set_busy(recbin, TRUE);
	GUIBlockInput(toplevel, TRUE);

	clist = GTK_CLIST(recbin->contents_clist);
	if(clist != NULL)
	{
		gtk_clist_freeze(clist);
		edv_recycle_bin_list_get_listing(
			recbin,
			EDV_GET_B(EDV_CFG_PARM_LISTS_SHOW_PROGRESS)
		);
		gtk_clist_thaw(clist);
	}

	GUIBlockInput(toplevel, FALSE);
	edv_recycle_bin_set_busy(recbin, FALSE);

	edv_recycle_bin_set_title(recbin);
	edv_recycle_bin_update_display(recbin);
}


/*
 *	Creates/maps the MIME Types List Window.
 *
 *	If type is not NULL then it specifies the initial MIME Type to
 *	display.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_mime_types(
	EDVCore *core,
	const gchar *type,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	edv_mime_types_list_win_struct *lw;

	if(core == NULL)
	{
		errno = EINVAL;
		return;
	}

	/* Get/create the MIME Types List Window */
	lw = core->mime_types_list_win;
	if(lw != NULL)
	{
		/* Unmap the MIME Types List Window if it is already mapped */
		if(EDVMimeTypesListWinIsMapped(lw))
			EDVMimeTypesListWinUnmap(lw);

		/* Set the geometry position as needed */
		if((geometry_flags != 0) && (geometry != NULL))
		{
			if((geometry_flags & GDK_GEOMETRY_X) ||
			   (geometry_flags & GDK_GEOMETRY_Y)
			)
				gtk_widget_set_uposition(
					lw->toplevel,
					geometry->x, geometry->y
				);
		}
	}
	else
	{
		/* Create a new MIME Types List Window */
		core->mime_types_list_win = lw = EDVMimeTypesListWinNew(core);
		if(lw == NULL)
			return;
	}

	/* Center the MIME Types List Window to the specified toplevel
	 * window
	 */
	if(!(geometry_flags & GDK_GEOMETRY_X) &&
	   !(geometry_flags & GDK_GEOMETRY_Y) &&
	   (toplevel != NULL)
	)
		edv_center_window_to_window(toplevel, lw->toplevel);

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Map the MIME Types List Window */
	EDVMimeTypesListWinMap(lw);

	/* Load the MIME Types List to the MIME Types List Window */
	EDVMimeTypesListWinGetList(
		lw,
		TRUE				/* Verbose */
	);

	/* Select the MIME Type if one was specified */
	if(type != NULL)
		EDVMimeTypesListWinSelectByType(
			lw,
			type
		);

	EDVMimeTypesListWinSetBusy(lw, FALSE);

}


/*
 *	Creates/maps the Devices List Window.
 *
 *	If path is not NULL then it specifies the initial Device to
 *	display.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_devices(
	EDVCore *core,
	const gchar *path,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	edv_devices_list_win_struct *lw;

	if(core == NULL)
	{
		errno = EINVAL;
		return;
	}

	/* Get/create the Devices List Window */
	lw = core->devices_list_win;
	if(lw != NULL)
	{
		/* Unmap the Devices List Window if it is already mapped */
		if(EDVDevicesListWinIsMapped(lw))
			EDVDevicesListWinUnmap(lw);

		/* Set the geometry position as needed */
		if((geometry_flags != 0) && (geometry != NULL))
		{
			if((geometry_flags & GDK_GEOMETRY_X) ||
			   (geometry_flags & GDK_GEOMETRY_Y)
			)
				gtk_widget_set_uposition(
					lw->toplevel,
					geometry->x, geometry->y
				);
		}
	}
	else
	{
		/* Create a new Devices List Window */
		core->devices_list_win = lw = EDVDevicesListWinNew(core);
		if(lw == NULL)
			return;
	}

	/* Center the Devices List Window to the specified toplevel
	 * window
	 */
	if(!(geometry_flags & GDK_GEOMETRY_X) &&
	   !(geometry_flags & GDK_GEOMETRY_Y) &&
	   (toplevel != NULL)
	)
		edv_center_window_to_window(toplevel, lw->toplevel);

	EDVDevicesListSetBusy(lw, TRUE);

	/* Map the Devices List Window */
	EDVDevicesListWinMap(lw);

	/* Load the Devices List to the Devices List Window */
	EDVDevicesListWinGetList(
		lw,
		TRUE				/* Verbose */
	);

	/* Select the Device if one was specified */
	if(path != NULL)
		EDVDevicesListSelectByPath(
			lw,
			path
		);

	EDVDevicesListSetBusy(lw, FALSE);
}


/*
 *	Creates/maps the History List Window.
 *
 *	If event_index is not negative then it specifies the initial
 *	event to display.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_history(
	EDVCore *core,
	const gint event_index,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	edv_history_win_struct *lw;

	if(core == NULL)
	{
		errno = EINVAL;
		return;
	}

	/* Get/create the History List Window */
	lw = core->history_list_win;
	if(lw != NULL)
	{
		/* Unmap the History Window if it is already mapped */
		if(EDVHistoryWinIsMapped(lw))
			EDVHistoryWinUnmap(lw);

		/* Set the geometry position as needed */
		if((geometry_flags != 0) && (geometry != NULL))
		{
			if((geometry_flags & GDK_GEOMETRY_X) ||
			   (geometry_flags & GDK_GEOMETRY_Y)
			)
				gtk_widget_set_uposition(
					lw->toplevel,
					geometry->x, geometry->y
				);
		}
	}
	else
	{
		/* Create a new History List Window */
		core->history_list_win = lw = EDVHistoryWinNew(core);
		if(lw == NULL)
			return;
	}

	/* Center the History List Window to the specified toplevel
	 * window
	 */
	if(!(geometry_flags & GDK_GEOMETRY_X) &&
	   !(geometry_flags & GDK_GEOMETRY_Y) &&
	   (toplevel != NULL)
	)
		edv_center_window_to_window(toplevel, lw->toplevel);

	EDVHistoryWinSetBusy(lw, TRUE);

	/* Map the History List Window */
	EDVHistoryWinMap(lw);

	/* Load the History List to the History List Window */
	EDVHistoryWinUpdateList(lw);

	/* Select the initial event to display */
	if(event_index > -1)
		EDVHistoryWinSelect(
			lw,
			event_index
		);

	EDVHistoryWinUpdateMenus(lw);

	EDVHistoryWinSetBusy(lw, FALSE);
}


/*
 *	Creates/maps the Options Window.
 *
 *	The page_name specifies the initial page to display. If
 *	page_name is NULL then the default page is displayed.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_options_page(
	EDVCore *core,
	const gchar *page_name,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	gboolean invalid_page = FALSE;
	GtkWidget *cfgwin;

	if(core == NULL)
	{
		errno = EINVAL;
		return;
	}

	/* Get/create the Options Window */
	cfgwin = core->options_cfg_win;
	if(cfgwin != NULL)
	{
		/* Unmap the Options Window if it is already mapped */
		if(GTK_WIDGET_MAPPED(cfgwin))
			gtk_widget_hide(cfgwin);

		/* Set the geometry position as needed */
		if((geometry_flags != 0) && (geometry != NULL))
		{
			if((geometry_flags & GDK_GEOMETRY_X) ||
			   (geometry_flags & GDK_GEOMETRY_Y)
			)
				gtk_widget_set_uposition(
					cfgwin,
					geometry->x, geometry->y
				);
		}
	}
	else
	{
		/* Create a new Options Window */
		core->options_cfg_win = cfgwin = EDVOptionsWindowNew(
			core,
			geometry_flags,
			geometry
		);
		if(cfgwin == NULL)
			return;
	}

	/* Realize */
	if(!GTK_WIDGET_REALIZED(cfgwin))
		gtk_widget_realize(cfgwin);

	/* Center the Options Window to the specified toplevel window */
	if(!(geometry_flags & GDK_GEOMETRY_X) &&
	   !(geometry_flags & GDK_GEOMETRY_Y) &&
	   (toplevel != NULL)
	)
		edv_center_window_to_window(toplevel, cfgwin);

	/* Get the values from the configuration */
	CfgWinGetValues(cfgwin, FALSE);

	/* Page specified? */
	if(!STRISEMPTY(page_name))
	{
		/* Switch to the specified page */
		if(!EDVOptionsWindowSelectTab(cfgwin, page_name))
			invalid_page = TRUE;
	}

	gtk_widget_show(cfgwin);
	CfgWinSetHasChanges(cfgwin, FALSE);
	CfgWinUpdateMenus(cfgwin);

	/* Was an invalid page name specified? */
	if(!STRISEMPTY(page_name) && invalid_page)
	{
		gchar *msg = g_strdup_printf(
"Invalid Options Window page name \"%s\".",
			page_name
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Options Window Page Switch Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = EINVAL;
	}
}

/*
 *	Creates/maps the Options Window.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_options(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	edv_map_options_page(
		core,
		NULL,				/* No page name */
		geometry_flags,
		geometry,
		toplevel
	);
}

/*
 *	Creates/maps the Customize Window.
 *
 *	The page_name specifies the initial page to display. If
 *	page_name is NULL then the default page is displayed.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_customize_page(
	EDVCore *core,
	const gchar *page_name,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	gboolean invalid_page = FALSE;
	gint		browser_num,
			imbr_num,
			archiver_num,
			recbin_num;
	GtkWidget *cfgwin;

	if(core == NULL)
	{
		errno = EINVAL;
		return;
	}

	/* Get/create the Customize Window */
	cfgwin = core->customize_cfg_win;
	if(cfgwin != NULL)
	{
		/* Unmap the Customize Window if it is already mapped */
		if(GTK_WIDGET_MAPPED(cfgwin))
			gtk_widget_hide(cfgwin);

		/* Set the geometry position as needed */
		if((geometry_flags != 0) && (geometry != NULL))
		{
			if((geometry_flags & GDK_GEOMETRY_X) ||
			   (geometry_flags & GDK_GEOMETRY_Y)
			)
				gtk_widget_set_uposition(
					cfgwin,
					geometry->x, geometry->y
				);
		}
	}
	else
	{
		core->customize_cfg_win = cfgwin = EDVCustomizeWindowNew(
			core,
			geometry_flags,
			geometry
		);
		if(cfgwin == NULL)
			return;
	}

	/* Realize */
	if(!GTK_WIDGET_REALIZED(cfgwin))
		gtk_widget_realize(cfgwin);

	/* Center the Customize Window to the specified toplevel window */
	if(!(geometry_flags & GDK_GEOMETRY_X) &&
	   !(geometry_flags & GDK_GEOMETRY_Y) &&
	   (toplevel != NULL)
	)
		edv_center_window_to_window(toplevel, cfgwin);

	/* Get the values from the configuration */
	CfgWinGetValues(cfgwin, FALSE);

	/* If the page name was not specified then match the page name
	 * with the specified toplevel GtkWidget
	 */
	if(page_name == NULL)
	{
		if(edv_match_window_from_toplevel(
			core,
			toplevel,
			&browser_num,
			&imbr_num,
			&archiver_num,
			&recbin_num
		))
		{
			if(browser_num > -1)
				page_name = "File Browser";
			else if(imbr_num > -1)
				page_name = "Image Browser";
			else if(archiver_num > -1)
				page_name = "Archiver";
			else if(recbin_num > -1)
				page_name = "Recycle Bin";
		}
	}

	/* Page specified? */
	if(!STRISEMPTY(page_name))
	{
		/* Switch to the specified page */
		if(!EDVCustomizeWindowSelectTab(cfgwin, page_name))
			invalid_page = TRUE;
	}

	gtk_widget_show(cfgwin);
	CfgWinSetHasChanges(cfgwin, FALSE);
	CfgWinUpdateMenus(cfgwin);

	/* Was an invalid page name specified? */
	if(!STRISEMPTY(page_name) && invalid_page)
	{
		gchar *msg = g_strdup_printf(
"Invalid Customize Window page name \"%s\".",
			page_name
		);
		edv_play_sound_warning(core);
		edv_message_warning(
			"Customize Window Page Switch Failed",
			msg,
			NULL,
			toplevel
		);
		g_free(msg);
		errno = EINVAL;
	}
}

/*
 *	Creates/maps the Customize Window.
 *
 *	The geometry_flags and geometry specifies the initial window
 *	position and size. If geometry_flags is 0 or geometry is NULL
 *	then the position and size specified by the configuration will
 *	be used instead.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 */
void edv_map_customize(
	EDVCore *core,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	edv_map_customize_page(
		core,
		NULL,				/* No page name */
		geometry_flags,
		geometry,
		toplevel
	);
}


/*
 *	Allocates a new properties dialog entry in the list.
 *
 *	Does not create the actual properties dialog.
 *
 *	Returns the properties dialog index in the list or NULL on
 *	error.
 */
static gint edv_new_properties_dialog_nexus(
	EDVCore *core,
	gint *existing_dialogs
)
{
	gint i;

	/* Count the number of existing dialogs */
	*existing_dialogs = 0;
	for(i = 0; i < core->total_prop_dlgs; i++)
	{
		if(core->prop_dlg[i] != NULL)
			*existing_dialogs = (*existing_dialogs) + 1;
	}

	/* Look for an available pointer in the list */
	for(i = 0; i < core->total_prop_dlgs; i++)
	{
		if(core->prop_dlg[i] == NULL)
			break;
	}
	if(i >= core->total_prop_dlgs)
	{
		/* Allocate more pointers */
		i = MAX(core->total_prop_dlgs, 0);
		core->total_prop_dlgs = i + 1;

		core->prop_dlg = (EDVPropDlg **)g_realloc(
			core->prop_dlg,
			core->total_prop_dlgs * sizeof(EDVPropDlg *)
		);
		if(core->prop_dlg == NULL)
		{
			core->total_prop_dlgs = 0;
			return(-3);
		}

		core->prop_dlg[i] = NULL;
	}

	return(i);
}

/*
 *	Selects the page on the properties dialog, shifts the
 *	dialog based on the number of existing dialogs, and
 *	maps the dialog relative to the toplevel GtkWidget.
 */
static void edv_new_properties_dialog_select_page_move(
	EDVCore *core,
	EDVPropDlg *d,
	const gchar *page_name,
	const gint existing_dialogs,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	CfgList *cfg_list = core->cfg_list;

	/* Set page? */
	if(page_name != NULL)
	{
		(void)edv_prop_dlg_set_page(
			d,
			page_name
		);
	}

	/* Shift the window position if there are existing windows and
	 * the geometry was not specified
	 */
	if((existing_dialogs > 0) && (geometry_flags == 0))
	{
		const gint	offset_x =
			EDV_GET_I(EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_X) * existing_dialogs,
					offset_y =
			EDV_GET_I(EDV_CFG_PARM_WINDOW_CASCADE_OFFSET_Y) * existing_dialogs;

		edv_center_window_to_window_offset(
			toplevel,
			edv_prop_dlg_get_toplevel(d),
			offset_x, offset_y
		);
	}
	/* Center the Properties Dialog to the toplevel GtkWidget if
	 * the geometry was not specified
	 */
	else if(geometry_flags == 0)
	{
		edv_center_window_to_window(
			toplevel,
			edv_prop_dlg_get_toplevel(d)
		);
	}

	edv_prop_dlg_map(d);
	edv_prop_dlg_set_has_changes(d, FALSE);
	edv_prop_dlg_update_display(d);
}

/*
 *	Creates a new Properties Dialog to display a VFS object.
 *
 *	The path specifies the VFS object.
 *
 *	The page_name specifies the name of the page to switch to
 *	after the dialog has been mapped.
 *
 *	Returns the index of the new Properties Dialog or -1 on
 *	error.
 */
gint edv_new_properties_dialog_vfs(
	EDVCore *core,
	const gchar *path,
	const gchar *page_name,
	GList *extended_properites_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	gint		i,
			existing_dialogs;
	EDVPropDlg *d;

	if((core == NULL) || STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Allocate a new properties dialog entry in the list */
	i = edv_new_properties_dialog_nexus(core, &existing_dialogs);
	if(i < 0)
		return(i);

	/* Create a new Properties Dialog and add it to the list */
	core->prop_dlg[i] = d = edv_prop_dlg_new(
		core,
		EDV_LOCATION_TYPE_VFS,
		path,
		0l,				/* Not used */
		NULL,				/* Not used */
		extended_properites_list,
		geometry_flags,
		geometry
	);
	if(d == NULL)
		return(-1);

	/* Select the page, move, and map the Properties Dialog */
	edv_new_properties_dialog_select_page_move(
		core,
		d,
		page_name,
		existing_dialogs,
		geometry_flags,
		geometry,
		toplevel
	);

	return(i);
}

gint edv_new_properties_dialog_recycle_bin(
	EDVCore *core,
	const gulong index,
	const gchar *page_name,
	GList *extended_properites_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	gint		i,
					existing_dialogs;
	EDVPropDlg *d;

	if((core == NULL) || (index == 0))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Allocate a new properties dialog entry in the list */
	i = edv_new_properties_dialog_nexus(core, &existing_dialogs);
	if(i < 0)
		return(i);

	/* Create a new properties dialog and add it to the list */
	core->prop_dlg[i] = d = edv_prop_dlg_new(
		core,
		EDV_LOCATION_TYPE_RECYCLE_BIN,
		NULL,				/* Not used */
		index,
		NULL,
		extended_properites_list,
		geometry_flags,
		geometry
	);
	if(d == NULL)
		return(-1);

	/* Select the page, move, and map the properties dialog */
	edv_new_properties_dialog_select_page_move(
		core,
		d,
		page_name,
		existing_dialogs,
		geometry_flags,
		geometry,
		toplevel
	);

	return(i);
}

gint edv_new_properties_dialog_archive(
	EDVCore *core,
	const gchar *path,
	const gchar *arch_path,
	const gchar *page_name,
	GList *extended_properites_list,
	const GdkGeometryFlags geometry_flags,
	const GdkRectangle *geometry,
	GtkWidget *toplevel
)
{
	gint		i,
			existing_dialogs;
	EDVPropDlg *d;

	if(core == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	/* Allocate a new properties dialog entry in the list */
	i = edv_new_properties_dialog_nexus(core, &existing_dialogs);
	if(i < 0)
		return(i);

	/* Create a new properties dialog and add it to the list */
	core->prop_dlg[i] = d = edv_prop_dlg_new(
		core,
		EDV_LOCATION_TYPE_ARCHIVE,
		path,
		0l,				/* Not used */
		arch_path,
		extended_properites_list,
		geometry_flags,
		geometry
	);
	if(d == NULL)
		return(-1);

	/* Select the page, move, and map the properties dialog */
	edv_new_properties_dialog_select_page_move(
		core,
		d,
		page_name,
		existing_dialogs,
		geometry_flags,
		geometry,
		toplevel
	);

	return(i);
}


/*
 *	Runs the device check command on the given device reference.
 */
void edv_run_device_check(
	EDVCore *core,
	EDVDevice *dev,
	GtkWidget *toplevel
)
{
	gint pid;
	gchar *shell_prog;
	const gchar	*shell_args,
			*shell_cmd,
			*cmd;
	CfgList *cfg_list;

	if((core == NULL) || (dev == NULL))
		return;

	cfg_list = core->cfg_list;
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),

	cmd = dev->command_check;
	if(cmd == NULL)
		return;

	/* Seek past spaces in command string and make sure it is not
	 * empty
	 */
	while(ISBLANK(*cmd))
		cmd++;
	if(*cmd == '\0')
		return;

	/* Get the shell program and the shell arguments */
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Execute the check command */
	pid = edv_system_shell(
		cmd,
		shell_prog,
		shell_args
	);
	if(pid >= 0)
	{
		dev->last_check_time = edv_time();
	}
	else
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n\
\n\
%s.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n\
\n\
%s.",
#endif
			cmd,
			g_strerror(error_code)
		);
		edv_play_sound_error(core);
		edv_message_error(
#ifdef PROG_LANGUAGE_ENGLISH
"Device Check Failed",
				msg,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
				msg,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
				msg,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
			toplevel
		);
		g_free(msg);
	}

	g_free(shell_prog);
}

/*
 *      Runs the device tools command on the given device reference.
 */
void edv_run_device_tools(
	EDVCore *core,
	EDVDevice *dev,
	GtkWidget *toplevel
)
{
	gint pid;
	gchar *shell_prog;
	const gchar	*shell_args,
			*shell_cmd,
			*cmd;
	CfgList *cfg_list;

	if((core == NULL) || (dev == NULL))
		return;

	cfg_list = core->cfg_list;
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),

	cmd = dev->command_tools;
	if(cmd == NULL)
		return;

	/* Seek past spaces in command string and make sure it is not
	 * empty.
	 */
	while(ISBLANK(*cmd))
		cmd++;
	if(*cmd == '\0')
		return;

	/* Get the shell program and the shell arguments */
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Execute the tools command */
	pid = edv_system_shell(
		cmd,
		shell_prog,
		shell_args
	);
	if(pid < 0)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n\
\n\
%s.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n\
%s.",
#endif
			cmd,
			g_strerror(error_code)
		);
		edv_play_sound_error(core);
		edv_message_error(
#ifdef PROG_LANGUAGE_ENGLISH
"Device Tool Failed",
				msg,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
				msg,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
				msg,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
			toplevel
		);
		g_free(msg);
	}

	g_free(shell_prog);
}

/*
 *      Runs the device format command on the given device reference.
 */
void edv_run_device_format(
	EDVCore *core,
	EDVDevice *dev,
	GtkWidget *toplevel
)
{
	gint pid;
	gchar *shell_prog;
	const gchar	*shell_args,
			*shell_cmd,
			*cmd;
	CfgList *cfg_list;

	if((core == NULL) || (dev == NULL))
		return;

	cfg_list = core->cfg_list;
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),

	cmd = dev->command_format;
	if(cmd == NULL)
		return;

	/* Seek past spaces in command string and make sure it is not
	 * empty.
	 */
	while(ISBLANK(*cmd))
		cmd++;
	if(*cmd == '\0')
		return;

	/* Get the shell program and the shell arguments */
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Execute the format command */
	pid = edv_system_shell(
		cmd,
		shell_prog,
		shell_args
	);
	if(pid < 0)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n\
\n\
%s.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n\
\n\
%s.",
#endif
			cmd,
			g_strerror(error_code)
		);
		edv_play_sound_error(core);
		edv_message_error(
#ifdef PROG_LANGUAGE_ENGLISH
"Device Format Failed",
				msg,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
				msg,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
				msg,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
			toplevel
		);
		g_free(msg);
	}

	g_free(shell_prog);
}

/*
 *	Runs the terminal defined in the configuration list.
 *
 *	If cmd is not NULL then the command specified by cmd will be
 *	appended to the terminal run command.
 *
 *	The wd specifies the working directory for the terminal. If this
 *	is NULL then the current working directory will be used.
 */
void edv_run_terminal(
	EDVCore *core,
	const gchar *cmd,
	const gchar *wd,
	GtkWidget *toplevel
)
{
	gchar *cwd;
	CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	if(!STRISEMPTY(wd))
	{
		cwd = edv_getcwd();
		edv_setcwd(wd);
	}
	else
	{
		cwd = NULL;
	}

	/* Execute the run terminal command
	 *
	 * If the cmd is specified then append cmd to the run
	 * terminal command defined in the configuration,
	 * otherwise just run the terminal command defined in the
	 * configuration
	 */
	if(!STRISEMPTY(cmd))
	{
		/* Get terminal command for running a program with a
		 * terminal, format the command, and then run the command
		 */
		gchar *cmd_terminal = STRDUP(EDV_GET_S(
			EDV_CFG_PARM_PROG_TERMINAL_RUN
		));
		if(!STRISEMPTY(cmd_terminal))
		{
			gchar	*shell_prog,
				*cmd_tar = g_strconcat(
				cmd_terminal,
				" ",
				cmd,
				NULL
			);
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);

			/* Execute the terminal command */
			const gint pid = edv_system_shell(
				cmd,
				shell_prog,
				shell_args
			);
			if(pid < 0)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
					cmd_tar,
					g_strerror(error_code)
				);
				edv_play_sound_error(core);
				edv_message_error(
"Run Terminal Failed",
					msg,
"Make sure that the run terminal command is\n\
defined correctly, please go to\n\
Settings->Options->Programs.",
					toplevel
				);
				g_free(msg);
			}
			g_free(cmd_tar);
			g_free(shell_prog);
		}
		g_free(cmd_terminal);
	}
	else
	{
		/* No command specified, just run the terminal */
		gchar *cmd_terminal = STRDUP(EDV_GET_S(
			EDV_CFG_PARM_PROG_TERMINAL
		));
		if(!STRISEMPTY(cmd_terminal))
		{
			gchar *shell_prog;
			const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);

			/* Execute the terminal command */
			const gint pid = edv_system_shell(
				cmd_terminal,
				shell_prog,
				shell_args
			);
			if(pid < 0)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
					cmd_terminal,
					g_strerror(error_code)
				);
				edv_play_sound_error(core);
				edv_message_error(
"Run Terminal Failed",
					msg,
"Make sure that the run terminal command is\n\
defined correctly, please go to\n\
Settings->Options->Programs.",
					toplevel
				);
				g_free(msg);
			}
			g_free(shell_prog);
		}
		g_free(cmd_terminal);
	}

	edv_setcwd(cwd);

	g_free(cwd);
}


/*
 *	Gets the Object Operations Dialog, creating it as needed.
 */
edv_obj_op_dlg_struct *edv_get_object_operations_dialog(EDVCore *core)
{
	if(core == NULL)
		return(NULL);

	if(core->obj_op_dlg == NULL)
		core->obj_op_dlg = EDVObjOpDlgNew(core);

	return(core->obj_op_dlg);
}


/*
 *	Maps the Run Dialog (creating it as needed) with the specified
 *	command and working directory.
 */
void edv_map_run_dialog_command(
	EDVCore *core,
	const gchar *command,
	const gchar *working_dir,
	GtkWidget *toplevel
)
{
	CfgList *cfg_list;
	EDVRunDlg *dlg;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Get the Run Dialog, create it as needed */
	dlg = core->run_dlg;
	if(dlg == NULL)
	{
		core->run_dlg = dlg = edv_run_dlg_new(core);
		edv_run_dlg_get_values(dlg);
	}
	if(dlg == NULL)
		return;

	/* Unmap the Run Dialog if it is already mapped */
	if(edv_run_dlg_is_mapped(dlg))
		edv_run_dlg_unmap(dlg);

	/* Move the run dialog to its last position */
	gtk_widget_set_uposition(
		dlg->toplevel,
		EDV_GET_I(EDV_CFG_PARM_RUN_DLG_X),
		EDV_GET_I(EDV_CFG_PARM_RUN_DLG_Y)
	);

	/* Set the command as needed */
	if(!STRISEMPTY(command))
		edv_run_dlg_set_command(dlg, command);

	/* Set the working directory as needed */
	if(!STRISEMPTY(working_dir))
		edv_run_dlg_set_working_directory(dlg, working_dir);

	edv_run_dlg_map(dlg);
}

void edv_map_run_dialog(
	EDVCore *core,
	GtkWidget *toplevel
)
{
	edv_map_run_dialog_command(core, NULL, NULL, NULL);
}


/*
 *	Maps the Find Window.
 */
void edv_map_find(
	EDVCore *core,
	const EDVLocationType location_type,
	const gchar *location,
	const gchar *value
)
{
	EDVFindWin *fw;

	if(core == NULL)
		return;

	/* Get the Find Window, create it as needed */
	fw = core->find_win;
	if(fw == NULL)
		core->find_win = fw = edv_find_win_new(core);
	if(fw == NULL)
		return;

	/* Unmap the Find Window if it is already mapped */
	if(edv_find_win_is_mapped(fw))
		edv_find_win_unmap(fw);

	/* Set the reference window */
	edv_find_win_set_reference_window(
		fw,
		-1,
		-1,
		-1,
		-1
	);

	/* Set the location */
	if(!STRISEMPTY(location))
	{
		edv_find_win_set_location_type(
			fw,
			EDV_LOCATION_TYPE_VFS
		);
		edv_find_win_set_location(
			fw,
			location,
			FALSE
		);
	}

	/* Set the search value */
	if(!STRISEMPTY(value))
		edv_find_win_set_search(
			fw,
			value,
			FALSE
		);

	edv_find_win_update_display(fw);

	/* Map the Find Window */
	edv_find_win_map(fw);
}

/*
 *	Maps the find window relative to the given browser.
 */
void edv_map_find_vfs(
	EDVCore *core,
	EDVVFSBrowser *browser
)
{
	gint i, browser_num = -1;
	EDVFindWin *fw;

	if((core == NULL) || (browser == NULL))
		return;

	/* Get the Find Window, create it as needed */
	fw = core->find_win;
	if(fw == NULL)
		core->find_win = fw = edv_find_win_new(core);
	if(fw == NULL)
		return;

	/* Unmap the Find Window if it is already mapped */
	if(edv_find_win_is_mapped(fw))
		edv_find_win_unmap(fw);

	/* Get this File Browser's index */
	for(i = 0; i < core->total_browsers; i++)
	{
		if(core->browser[i] == browser)
		{
			browser_num = i;
			break;
		}
	}

	/* Set the File Browser as the Find Window's reference */
	edv_find_win_set_reference_window(
		fw,
		browser_num,
		-1,
		-1,
		-1
	);

	/* Set initial location of find window to match the current
	 * location of the given browser.
	 */
	edv_find_win_set_location(
		fw,
		edv_vfs_browser_get_location(browser),
		FALSE
	);

	edv_center_window_to_window(
		browser->toplevel,
		fw->toplevel
	);
	edv_find_win_update_display(fw);
	edv_find_win_map(fw);
}

/*
 *      Maps the find window relative to the given image browser.
 */
void edv_map_find_image_browser(
	EDVCore *core,
	EDVImageBrowser *imbr
)
{
	gint		i,
			imbr_num = -1;
	EDVFindWin *fw;

	if((core == NULL) || (imbr == NULL))
		return;

	/* Get the Find Window, create it as needed */
	fw = core->find_win;
	if(fw == NULL)
		core->find_win = fw = edv_find_win_new(core);
	if(fw == NULL)
		return;

	/* Unmap the Find Window if it is already mapped */
	if(edv_find_win_is_mapped(fw))
		edv_find_win_unmap(fw);

	/* Get this Image Browser's index */
	for(i = 0; i < core->total_imbrs; i++)
	{
		if(core->imbr[i] == imbr)
		{
			imbr_num = i;
			break;
		}
	}

	/* Set the Image Browser as the Find Window's reference */
	edv_find_win_set_reference_window(
		fw,
		-1,
		imbr_num,
		-1,
		-1
	);

	/* Set initial location of find window to match the current
	 * location of the given image browser.
	 */
	edv_find_win_set_location(
		fw,
		edv_image_browser_get_location(imbr),
		FALSE
	);

	edv_center_window_to_window(imbr->toplevel, fw->toplevel);
	edv_find_win_update_display(fw);
	edv_find_win_map(fw);
}

/*
 *	Maps the find window relative to the given archiver.
 */
void edv_map_find_archive(
	EDVCore *core,
	EDVArchiver *archiver
)
{
	gint		i,
			archiver_num = -1;
	EDVFindWin *fw;

	if((core == NULL) || (archiver == NULL))
		return;

	/* Get the Find Window, create it as needed */
	fw = core->find_win;
	if(fw == NULL)
		core->find_win = fw = edv_find_win_new(core);
	if(fw == NULL)
		return;

	/* Unmap the Find Window if it is already mapped */
	if(edv_find_win_is_mapped(fw))
		edv_find_win_unmap(fw);

	/* Get this Archiver's index */
	for(i = 0; i < core->total_archivers; i++)
	{
		if(core->archiver[i] == archiver)
		{
			archiver_num = i;
			break;
		}
	}

	/* Set the Archiver as the Find Window's reference */
	edv_find_win_set_reference_window(
		fw,
		-1,
		-1,
		-1,
		archiver_num
	);

	/* Set initial location of find window to match the current
	 * location of the given archiver.
	 */
	edv_find_win_set_location(
		fw,
		edv_archiver_get_location(archiver),
		FALSE
	);

	edv_center_window_to_window(
		archiver->toplevel,
		fw->toplevel
	);
	edv_find_win_update_display(fw);
	edv_find_win_map(fw);
}

/*
 *      Maps the find window relative to the given recycle bin.
 */
void edv_map_find_recycle_bin(
	EDVCore *core,
	edv_recbin_struct *recbin
)
{
	const gchar *index_path;
	gchar *recycled_dir;
	CfgList *cfg_list;
	EDVFindWin *fw;

	if((core == NULL) || (recbin == NULL))
		return;

	cfg_list = core->cfg_list;

	/* Get the Find Window, create it as needed */
	fw = core->find_win;
	if(fw == NULL)
		core->find_win = fw = edv_find_win_new(core);
	if(fw == NULL)
		return;

	/* Unmap the Find Window if it is already mapped */
	if(edv_find_win_is_mapped(fw))
		edv_find_win_unmap(fw);

	/* Set the Recycle Bin as the Find Window's reference */
	edv_find_win_set_reference_window(
		fw,
		-1,
		-1,
		0,		/* First and only Recycle Bin */
		-1
	);

	/* Get the path to the Recycled Objects Index File */
	index_path = EDV_GET_S(
		EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX
	);

	/* Get the path to the Recycled Objects Directory */
	recycled_dir = (index_path != NULL) ?
		g_dirname(index_path) : NULL;

	/* Set initial location of find window to be the Recycled
	 * Objects Directory
	 */
	if(recycled_dir != NULL)
		edv_find_win_set_location(
			fw,
			recycled_dir,
			FALSE
		);

	edv_center_window_to_window(
		recbin->toplevel,
		fw->toplevel
	);
	edv_find_win_update_display(fw);
	edv_find_win_map(fw);

	g_free(recycled_dir);
}


/*
 *	Updates the RC styles.
 */
void edv_update_rc_styles(EDVCore *core)
{
	CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Override GTK style? */
	if(EDV_GET_B(EDV_CFG_PARM_GTK_STYLE_OVERRIDE))
	{
		GtkRcStyle *rcstyle = edv_new_rc_style_from_cfg(
			CFGItemListGetValueStyle(
				cfg_list,
				EDV_CFG_PARM_STYLE_STANDARD
			)
		);
		GTK_RC_STYLE_UNREF(core->standard_rcstyle);
		core->standard_rcstyle = rcstyle;

		rcstyle = edv_new_rc_style_from_cfg(
			CFGItemListGetValueStyle(
				cfg_list,
				EDV_CFG_PARM_STYLE_LISTS
			)
		);
		GTK_RC_STYLE_UNREF(core->lists_rcstyle);
		core->lists_rcstyle = rcstyle;
	}
	else
	{
		gboolean style_defined;
		GtkStateType state;
		GtkRcStyle *rcstyle;
		const gchar	*font_name,
				*fg_color_name[5],
				*bg_color_name[5],
				*text_color_name[5],
				*base_color_name[5],
				*bg_pixmap_path[5];

		font_name = core->font_name;

		state = GTK_STATE_NORMAL;
		fg_color_name[state] = core->fg_color_name;
		bg_color_name[state] = core->bg_color_name;
		text_color_name[state] = core->fg_color_name;
		base_color_name[state] = core->bg_color_name;
		bg_pixmap_path[state] = core->bg_pixmap_path;

		state = GTK_STATE_ACTIVE;
		fg_color_name[state] = core->fg_color_name;
		bg_color_name[state] = core->bg_color_name;
		text_color_name[state] = core->fg_color_name;
		base_color_name[state] = core->bg_color_name;
		bg_pixmap_path[state] = core->bg_pixmap_path;

		state = GTK_STATE_PRELIGHT;
		fg_color_name[state] = core->fg_color_name;
		bg_color_name[state] = core->bg_color_name;
		text_color_name[state] = core->fg_color_name;
		base_color_name[state] = core->bg_color_name;
		bg_pixmap_path[state] = core->bg_pixmap_path;

		state = GTK_STATE_SELECTED;
		fg_color_name[state] = core->sfg_color_name;
		bg_color_name[state] = core->sbg_color_name;
		text_color_name[state] = core->sfg_color_name;
		base_color_name[state] = core->sbg_color_name;
		bg_pixmap_path[state] = core->sbg_pixmap_path;

		state = GTK_STATE_INSENSITIVE;
		fg_color_name[state] = core->fg_color_name;
		bg_color_name[state] = core->bg_color_name;
		text_color_name[state] = core->fg_color_name;
		base_color_name[state] = core->bg_color_name;
		bg_pixmap_path[state] = core->bg_pixmap_path;

#define SET_RCSTYLE(_s_)	{			\
 gint i;						\
 const gchar *s;					\
							\
 style_defined = FALSE;					\
							\
 s = font_name;						\
 if(!STRISEMPTY(s)) {					\
  (_s_)->font_name = STRDUP(s);				\
  style_defined = TRUE;					\
 }							\
							\
 for(i = 0; i < 5; i++) {				\
  s = fg_color_name[i];					\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_FG;			\
   gdk_color_parse(s, &((_s_)->fg[i]));			\
   style_defined = TRUE;				\
  }							\
  s = bg_color_name[i];					\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_BG;			\
   gdk_color_parse(s, &((_s_)->bg[i]));			\
   style_defined = TRUE;				\
  }                                                     \
  s = text_color_name[i];				\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_TEXT;		\
   gdk_color_parse(s, &((_s_)->text[i]));		\
   style_defined = TRUE;				\
  }							\
  s = base_color_name[i];				\
  if(!STRISEMPTY(s)) {					\
   (_s_)->color_flags[i] |= GTK_RC_BASE;		\
   gdk_color_parse(s, &((_s_)->base[i]));		\
   style_defined = TRUE;				\
  }							\
							\
  s = bg_pixmap_path[i];				\
  if(!STRISEMPTY(s)) {					\
   (_s_)->bg_pixmap_name[i] = STRDUP(s);		\
   style_defined = TRUE;				\
  }							\
 }							\
}

		/* Standard RC style */
		rcstyle = gtk_rc_style_new();
		SET_RCSTYLE(rcstyle);
		GTK_RC_STYLE_UNREF(core->standard_rcstyle);
		if(style_defined)
		{
			core->standard_rcstyle = rcstyle;
		}
		else
		{
			GTK_RC_STYLE_UNREF(rcstyle);
			core->standard_rcstyle = NULL;
		}

		/* Lists RC style */
		rcstyle = gtk_rc_style_new();
		SET_RCSTYLE(rcstyle);
		GTK_RC_STYLE_UNREF(core->lists_rcstyle);
		if(style_defined)
		{
			core->lists_rcstyle = rcstyle;
		}
		else
		{
			GTK_RC_STYLE_UNREF(rcstyle);
			core->lists_rcstyle = NULL;
		}

#undef SET_RCSTYLE
	}
}


/*
 *	Updates the Users & Groups Popup Lists.
 *
 *	If the Popup Lists do not exist then they will be created.
 */
void edv_update_id_popup_lists(EDVCore *core)
{
	gint i, n;
	GList *glist;
	GtkRcStyle *rcstyle;
	GtkWidget	*w,
			*pulist;
	EDVUID *uid;
	EDVGID *gid;

	if(core == NULL)
		return;

	/* Get the Users popup list, create it as needed */
	pulist = core->users_pulist;
	if(pulist == NULL)
		core->users_pulist = pulist = PUListNew();
	if(pulist == NULL)
		return;

	/* Update style */
	rcstyle = core->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
		gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add Users to the Popup List */
	for(glist = core->uids_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		uid = EDV_UID(glist->data);
		if(uid == NULL)
			continue;

		if(STRISEMPTY(uid->name))
			continue;

		n = PUListAddItem(
			pulist,
			uid->name
		);
		if(n < 0)
			continue;

		PUListSetItemData(
			pulist,
			n,
			(gpointer)i
		);
	}


	/* Get the Groups popup list, create it as needed */
	pulist = core->groups_pulist;
	if(pulist == NULL)
		core->groups_pulist = pulist = PUListNew();
	if(pulist == NULL)
		return;

	/* Update style */
	rcstyle = core->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
		gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add Groups to the Popup List */
	for(glist = core->gids_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		gid = EDV_GID(glist->data);
		if(gid == NULL)
			continue;

		if(STRISEMPTY(gid->name))
			continue;

		n = PUListAddItem(
			pulist,
			gid->name
		);
		if(n < 0)
			continue;

		PUListSetItemData(
			pulist,
			n,
			(gpointer)i
		);
	}
}

/*
 *	Updates the Devices Popup List.
 *
 *	If the Popup List does not exist then it will be created.
 */
void edv_update_devices_popup_list(EDVCore *core)
{
	gint i, n;
	const gchar *text;
	GList *glist;
	GtkRcStyle *rcstyle;
	GtkWidget	*w,
			*pulist;
	EDVPixmap *icon;
	EDVDevice *dev;

	if(core == NULL)
		return;

	/* Get the Devices popup list, create it as needed */
	pulist = core->devices_pulist;
	if(pulist == NULL)
		core->devices_pulist = pulist = PUListNew();
	if(pulist == NULL)
		return;

	/* Update style */
	rcstyle = core->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
		gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add the Devices to the Popup List */
	for(glist = core->devices_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		dev = EDV_DEVICE(glist->data);
		if(dev == NULL)
			continue;

		/* Skip devices who are marked as unlisted, in which case 
		 * they should not be listed on the devices popup list
		 */
		if(EDV_DEVICE_IS_UNLISTED(dev))
			continue;

		/* Realize device, loading icons as needed */
		edv_device_realize(dev, FALSE);

		/* Get the text as the device's name or device path */
		text = dev->name;
		if((text == NULL) && (dev->device_path != NULL))
			text = g_basename(dev->device_path);
		if(text == NULL)
			text = "";

		/* Add new item to devices popup list */
		icon = edv_pixmap_ref(dev->small_icon[
			EDV_DEVICE_ICON_STATE_STANDARD
		]);
		if(edv_pixmap_is_loaded(icon))
			n = PUListAddItemPixText(
				pulist,
				text,
				icon->pixmap, icon->mask
			);
		else
			n = PUListAddItem(
				pulist,
				text
			);
		icon = edv_pixmap_unref(icon);
		if(n < 0)
			continue;

		PUListSetItemData(
			pulist,
			n,
			(gpointer)i
		);
	}
}

/*
 *	Updates the MIME Type class hint indices.
 */
void edv_update_mime_type_hint_indicies(EDVCore *core)
{
	gint i;
	GList *glist;
	EDVMIMEType *m;

	if(core == NULL)
		return;

	/* Reset all indexes for all MIME Type classes */
	core->mimetype_system_index_first = -1;
	core->mimetype_system_index_last = -1;
	core->mimetype_format_index_first = -1;
	core->mimetype_format_index_last = -1;
	core->mimetype_program_index_first = -1;
	core->mimetype_program_index_last = -1;
	core->mimetype_unique_index_first = -1;
	core->mimetype_unique_index_last = -1;

	/* Iterate through all MIME Types and update each class hint
	 * index
	 */
	for(glist = core->mime_types_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
			if(core->mimetype_system_index_first < 0)
				core->mimetype_system_index_first = i;
			core->mimetype_system_index_last = i;
			break;

		  case EDV_MIME_TYPE_CLASS_FORMAT:
			if(core->mimetype_format_index_first < 0)
				core->mimetype_format_index_first = i;
			core->mimetype_format_index_last = i;
			break;

		  case EDV_MIME_TYPE_CLASS_PROGRAM:
			if(core->mimetype_program_index_first < 0)
				core->mimetype_program_index_first = i;
			core->mimetype_program_index_last = i;
			break;

		  case EDV_MIME_TYPE_CLASS_UNIQUE:
			if(core->mimetype_unique_index_first < 0)
				core->mimetype_unique_index_first = i;
			core->mimetype_unique_index_last = i;
			break;
		}
	}
}

/*
 *	Updates the Open With Popup List.
 *
 *	If the Popup List does not exist then it will be created.
 */
void edv_update_open_with_popup_list(EDVCore *core)
{
	gint i, n;
	GList *glist;
	GtkRcStyle *rcstyle;
	GtkWidget	*w,
					*pulist;
	EDVPixmap *icon;
	EDVMIMEType *m;

	if(core == NULL)
		return;

	/* Get the Open With popup list, create it as needed */
	pulist = core->open_with_pulist;
	if(pulist == NULL)
		core->open_with_pulist = pulist = PUListNew();
	if(pulist == NULL)
		return;

	/* Update style */
	rcstyle = core->lists_rcstyle;
	w = PUListGetCList(pulist);
	if((w != NULL) && (rcstyle != NULL))
		gtk_widget_modify_style(w, rcstyle);

	/* Delete all existing items from the Popup List */
	PUListClear(pulist);

	/* Add MIME Types of class application to the Popup List */
	for(glist = core->mime_types_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		if(STRISEMPTY(m->type))
			continue;

		/* Skip MIME Type if it is not of class application */
		if(m->mt_class != EDV_MIME_TYPE_CLASS_PROGRAM)
			continue;

		/* Realize this MIME Type since we need to use its icons */
		edv_mime_type_realize(m, FALSE);

		/* Add the new item to the Popup List */
		icon = edv_pixmap_ref(m->small_icon[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		]);
		if(edv_pixmap_is_loaded(icon))
			n = PUListAddItemPixText(
				pulist,
				(m->description != NULL) ?
					m->description : m->type,
				icon->pixmap, icon->mask
			);
		else
			n = PUListAddItem(
				pulist,
				(m->description != NULL) ?
					m->description : m->type
			);
		icon = edv_pixmap_unref(icon);
		if(n < 0)
			continue;

		PUListSetItemData(
			pulist,
			n,
			(gpointer)i
		);
	}
}


/*
 *	Refreshes all of Endeavour's resources to refresh.
 *
 *	Updates the Devices statistics and mount states.
 *
 *	Instructs all windows to refresh.
 */
void edv_refresh(EDVCore *core)
{
	gint i;
	EDVVFSBrowser *browser;
	EDVImageBrowser *imbr;
	EDVArchiver *archiver;
	edv_recbin_struct *recbin;
	edv_devices_list_win_struct *device_list;
	edv_mime_types_list_win_struct *mimetype_list;
	GtkWidget *cfgwin;

	if(core == NULL)
		return;

	/* Update the Device mount states and stats */
	edv_devices_list_update_mount_states(
		core->devices_list
	);
	edv_devices_list_update_statistics(
		core->devices_list
	);

	/* Refresh the File Browsers */
	for(i = 0; i < core->total_browsers; i++)
	{
		browser = core->browser[i];
		if(browser == NULL)
			continue;

		edv_vfs_browser_op_refresh(browser);
	}

	/* Refresh the Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
	{
		imbr = core->imbr[i];
		if(imbr == NULL)
			continue;

		edv_image_browser_op_refresh(imbr);
	}

	/* Refresh the Archivers */
	for(i = 0; i < core->total_archivers; i++)
	{
		archiver = core->archiver[i];
		if(archiver == NULL)
			continue;

		edv_archiver_op_refresh(archiver);
	}

	/* Refresh the Recycle Bin */
	recbin = core->recbin;
	if(recbin != NULL)
	{
		edv_recycle_bin_op_refresh(recbin);
	}

	/* Devices List */
	device_list = core->devices_list_win;
	if(EDVDevicesListWinIsMapped(device_list))
	{
		EDVDevicesListWinGetList(device_list, FALSE);
	}

	/* MIME Types List */
	mimetype_list = core->mime_types_list_win;
	if(EDVMimeTypesListWinIsMapped(mimetype_list))
	{
		EDVMimeTypesListWinGetList(mimetype_list, FALSE);
	}

	/* Options Window */
	cfgwin = core->options_cfg_win;
	if(cfgwin != NULL)
	{
		if(GTK_WIDGET_MAPPED(cfgwin))
			CfgWinGetValues(cfgwin, FALSE);
	}

	/* Customize Window */
	cfgwin = core->customize_cfg_win;
	if(cfgwin != NULL)
	{
		if(GTK_WIDGET_MAPPED(cfgwin))
			CfgWinGetValues(cfgwin, FALSE);
	}   
}

/*
 *	Reloads all of Endeavour's configurations from file and
 *	refreshes resources.
 *
 *	Any unsaved configuration in memory will be discarded.
 *
 *	edv_refresh() will be automatically called at the end.
 */
void edv_reset(EDVCore *core)
{
	gchar		*sd,
					*hostname;
	CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Reget the Host's Name */
	hostname = edv_get_host_name();

	/* Reget the User & Group ID lists */
	if(core->uids_list != NULL)
	{
		g_list_foreach(
			core->uids_list, (GFunc)edv_uid_delete, NULL
		);
		g_list_free(core->uids_list);
		core->uids_list = NULL;
	}
	core->uids_list = edv_uids_list_open_from_system(
		core->uids_list,
		-1				/* Append */
	);

	if(core->gids_list != NULL)
	{
		g_list_foreach(
			core->gids_list, (GFunc)edv_gid_delete, NULL
		);
		g_list_free(core->gids_list);
		core->gids_list = NULL;
	}
	core->gids_list = edv_gids_list_open_from_system(
		core->gids_list,
		-1				/* Append */
	);

	/* Reget the Effective User ID string */
	sd = edv_uid_uid_to_name(
		core->uids_list,
		core->effective_user_id, NULL
	);

	g_free(core->effective_user_id_str);
	core->effective_user_id_str = STRDUP(
		STRISEMPTY(sd) ? "anonymous" : sd
	);

	/* Reget the Effective User ID & Hostname string */
	g_free(core->effective_user_id_host_str);
	core->effective_user_id_host_str = g_strconcat(
		STRISEMPTY(sd) ? "anonymous" : sd,
		"@",
		STRISEMPTY(hostname) ? "unknown" : hostname,
		NULL
	);

	g_free(sd);

	/* Reopen the configuration list from the configuration file */
	if(core->cfg_path != NULL)
		CFGFileOpen(
			core->cfg_path,
			cfg_list
		);

	/* Reopen the Devices list from the Devices file */
	if(core->devices_list != NULL)
	{
		g_list_foreach(
			core->devices_list,
			(GFunc)edv_device_delete,
			NULL
		);
		g_list_free(core->devices_list);
		core->devices_list = NULL;
	}
	core->devices_list = edv_devices_list_open_from_system(
		core->devices_list,
		NULL, core
	);
	core->devices_list = edv_devices_list_open(
		core->devices_list,
		EDV_GET_S(EDV_CFG_PARM_FILE_DEVICES),
		NULL, core
	);

	/* Reopen the MIME Types list from the MIME Types file */
	if(core->mime_types_list != NULL)
	{
		g_list_foreach(
			core->mime_types_list,
			(GFunc)edv_mime_type_delete,
			NULL
		);
		g_list_free(core->mime_types_list);
		core->mime_types_list = NULL;
	}
	core->mime_types_list = edv_mime_types_list_file_open_system(
		core->mime_types_list,
		EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
		-1,					/* Append */
		NULL, core,
		NULL, core,
		TRUE				/* Read only */
	);
	core->mime_types_list = edv_mime_types_list_file_open(
		core->mime_types_list,
		EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES),
		-1,					/* Append */
		FALSE,				/* Do not update */
		FALSE,				/* Not only newer */
		NULL, core,
		NULL, core,
		NULL, core,
		FALSE				/* Not read only */
	);
	core->mime_types_list = edv_mime_types_list_file_open(
		core->mime_types_list,
		EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES_GLOBAL),
		-1,					/* Append */
		FALSE,				/* Do not update */
		FALSE,				/* Not only newer */
		NULL, core,
		NULL, core,
		NULL, core,
		TRUE				/* Read only */
	);
	/* Update the MIME Types class list index hints */
	edv_update_mime_type_hint_indicies(core);

	/* Notify all resources about the reset
	 *
	 * This will cause all the windows to update their
	 * customizable attributes based on the just reloaded
	 * configuration
	 */
	edv_emit_reconfigured(core);

	/* Refresh all the windows */
	edv_refresh(core);

	g_free(hostname);
}


/*
 *	Saves all of Endeavour's configuration in memory to file and
 *	commits buffer caches to disk.
 */
void edv_sync_edv(EDVCore *core)
{
	const gulong time_start = edv_time();
	const gchar *s;
	CfgList *cfg_list;
	EDVVFSObject *obj;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Begin saving all of Endeavour's configuration in memory to
	 * file
	 */

	/* MIME Types */
	s = EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES);
	if(!STRISEMPTY(s))
	{
		gchar *path = STRDUP(s);
		if(path != NULL)
		{
			const gboolean object_existed = edv_path_lexists(path);
			edv_mime_types_list_file_save(
				core->mime_types_list,
				path,
				FALSE,			/* Do not save read only
											 * MIME Types */
				NULL, core
			);
			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				if(object_existed)
					edv_emit_vfs_object_modified(
						core,
						path,
						path,
						obj
					);
				else
					edv_emit_vfs_object_added(
						core,
						path,
						obj
					);
				edv_vfs_object_delete(obj);
			}
			g_free(path);
		}
	}

	/* Devices */
	s = EDV_GET_S(EDV_CFG_PARM_FILE_DEVICES);
	if(!STRISEMPTY(s))
	{
		gchar *path = STRDUP(s);
		if(path != NULL)
		{
			const gboolean object_existed = edv_path_lexists(path);
			edv_devices_list_file_save(
				core->devices_list,
				path,
				NULL, core
			);
			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				if(object_existed)
					edv_emit_vfs_object_modified(
						core,
						path,
						path,
						obj
					);
				else
					edv_emit_vfs_object_added(
						core,
						path,
						obj
					);
				edv_vfs_object_delete(obj);
			}
			g_free(path);
		}
	}

	/* Configuration */
	if(core->cfg_path != NULL)
	{
		gchar *path = STRDUP(core->cfg_path);
		if(path != NULL)
		{
			const gboolean object_existed = edv_path_lexists(path);
			CFGFileSave(
				path,
				cfg_list
			);
			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				if(object_existed)
					edv_emit_vfs_object_modified(
						core,
						path,
						path,
						obj
					);
				else
					edv_emit_vfs_object_added(
						core,
						path,
						obj
					);
				edv_vfs_object_delete(obj);
			}
			g_free(path);
		}
	}

	/* Commit buffer caches to disk */
	(void)edv_sync();

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_SYNC_DISKS,
		time_start,
		edv_time(),
		0,					/* Status */
		NULL,				/* Source */
		NULL,				/* Target */
		NULL				/* Comment */
	);
}


/*
 *	Clears all the error message buffers and pointers.
 */
void edv_clear_error_messages(EDVCore *core)
{
	if(core == NULL)
		return;

	core->last_error_ptr = NULL;
	g_free(core->last_error_buf);
	core->last_error_buf = NULL;
}

/*
 *	Deletes all history.
 */
void edv_clear_all_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
)
{
	if(core == NULL)
		return;

	/* Confirm */
	if(confirm)
	{
		gint response;

		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
"Confirm Clear",
"Are you sure you want to clear all of " PROG_NAME "'s\n\
history from memory and delete all of " PROG_NAME "'s\n\
history files?",
			NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		);
		CDialogSetTransientFor(NULL);
		if(response != CDIALOG_RESPONSE_YES)
			return;
	}

	edv_clear_events_history(
		core,
		FALSE,
		toplevel
	);

	edv_clear_location_bars_history(
		core,
		FALSE,
		toplevel
	);

	edv_clear_run_history(
		core,
		FALSE,
		toplevel
	);
}

/*
 *	Clears the history events.
 *
 *	Clears the History List Window and deletes the events history
 *	log file.
 */
void edv_clear_events_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
)
{
	if(core == NULL)
		return;

	EDVHistoryWinClear(
		core,
		core->history_list_win,
		confirm,
		toplevel
	);
}

/*
 *	Clears the locations history on all windows.
 */
void edv_clear_location_bars_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
)
{
	gint i;
	const gchar *path;
	CfgList *cfg_list;
	EDVVFSBrowser *browser;
	EDVImageBrowser *imbr;
	EDVArchiver *archiver;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Confirm */
	if(confirm)
	{
		gint response;

		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Claro",
"Usted est seguro que usted quiere borrar la historia de\n\
ubicaciones en todas las ventanas?",
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Clair",
"Etes-vous sr que vous voulez effacer l'histoire d'emplacements\n\
sur toutes les fentres?",
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Klar",
"Sind sie sicher sie die orte geschichte auf allen fenstern\n\
wollen lschen?",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Chiaro",
"Lei sono sicuro che lei vuole cancellare la storia di posizioni\n\
su tutte le finestre?",
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Helder",
"ent u zeker u de plaatzen geschiedenis op alle ramen wil\n\
schrappen?",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Claro",
"Esto seguro quer anular a histria de localidades em todas\n\
as janelas?",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Klar",
"Er de sikker de stryker plasserings historien p alle vinduer?",
#else
"Confirm Clear",
"Are you sure you want to clear all the Location Bar Histories\n\
from memory and delete all the Location Bar History Files?",
#endif
			NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		);
		CDialogSetTransientFor(NULL);
		if(response != CDIALOG_RESPONSE_YES)
			return;
	}

	/* Clear the File Browser's Locations List */
	for(i = 0; i < core->total_browsers; i++)
	{
		browser = core->browser[i];
		if(browser == NULL)
			continue;

		GUIComboSetList(browser->location_combo, NULL);
	}

	/* Delete the File Browser Locations History File */
	path = EDV_GET_S(
		EDV_CFG_PARM_FILE_BROWSER_LOCATION_HISTORY
	);
	if(!STRISEMPTY(path))
	{
		if(!edv_unlink(path))
			edv_emit_vfs_object_removed(core, path);
	}


	/* Clear the Image Browser Locations List */
	for(i = 0; i < core->total_imbrs; i++)
	{
		imbr = core->imbr[i];
		if(imbr == NULL)
			continue;

		GUIComboSetList(imbr->location_combo, NULL);
	}

	/* Delete the Image Browser Locations History File */
	path = EDV_GET_S(
		EDV_CFG_PARM_FILE_IMBR_LOCATION_HISTORY
	);
	if(!STRISEMPTY(path))
	{
		if(!edv_unlink(path))
			edv_emit_vfs_object_removed(core, path);
	}


	/* Clear the Archiver Locations List */
	for(i = 0; i < core->total_archivers; i++)
	{
		archiver = core->archiver[i];
		if(archiver == NULL)
			continue;

		GUIComboSetList(archiver->location_combo, NULL);
	}

	/* Delete the Image Browser Locations History File */
	path = EDV_GET_S(
		EDV_CFG_PARM_FILE_ARCHIVER_LOCATION_HISTORY
	);
	if(!STRISEMPTY(path))
	{
		if(!edv_unlink(path))
			edv_emit_vfs_object_removed(core, path);
	}

	edv_play_sound_completed(core);
}

/*
 *	Clear the Run Dialog history and deletes the run history log
 *	file.
 */
void edv_clear_run_history(
	EDVCore *core,
	const gboolean confirm,
	GtkWidget *toplevel
)
{
	const gchar *path;
	CfgList *cfg_list;
	EDVRunDlg *d;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Confirm */
	if(confirm)
	{
		gint response;

		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Claro",
"Usted est seguro que usted quiere borrar el corre la historia?",
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Clair",
"Etes-vous sr que vous voulez effacer l'histoire de course?",
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Klar",
"Sind Sie sicher Sie die Lauf Geschichte wollen lschen?",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Chiaro",
"Lei sono sicuro che lei vuole cancellare il ha corso la storia?",
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Helder",
"Bent u zeker u de tocht geschiedenis wil schrappen?",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Claro",
"Esto seguro quer anular a histria de corrida?",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Klar",
"Er De sikker De stryker lphistorien?",
#else
"Confirm Clear",
"Are you sure you want to clear the Run History from memory\n\
and delete the Run History File?",
#endif
			NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		);
		CDialogSetTransientFor(NULL);
		if(response != CDIALOG_RESPONSE_YES)
			return;
	}

	/* Clear Run Dialog Run History */
	d = core->run_dlg;
	if(d != NULL)
	{
		GUIComboSetList(d->run_combo, NULL);
	}

	/* Delete Run Dialog Run History File */
	path = CFGItemListGetValueS(
		cfg_list, EDV_CFG_PARM_FILE_RUNDLG_HISTORY
	);
	if(!STRISEMPTY(path))
	{
		if(!edv_unlink(path))
			edv_emit_vfs_object_removed(core, path);
	}

	edv_play_sound_completed(core);
}


/*
 *	Tabulates the total size of all the recycled objects in the
 *	recycle bin and checks if it has exceeded the size in which
 *	to warn the user and warns the user if the size has been
 *	exceeded.
 */
void edv_recycle_bin_size_check(
	EDVCore *core,
	GtkWidget *toplevel
)
{
	const gchar *index_path;
	gulong		recbin_size_warn,
					total_size;
	CfgList *cfg_list;
	EDVRecycleBinIndex *rp;
	EDVRecycledObject *obj;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Get path to the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
		return;

	/* Get the recycle bin size warning size */
	recbin_size_warn = EDV_GET_UL(EDV_CFG_PARM_RECBIN_SIZE_WARN);

	/* If the warning size is 0 then no warning should ever be
	 * given
	 */
	if(recbin_size_warn == 0l)
		return;

	/* Calculate the size of all the recycled objects in the
	 * recycle bin
	 */
	total_size = 0l;
	rp = edv_recycle_bin_index_open(index_path);
	for(obj = edv_recycle_bin_index_next(rp);
		obj != NULL;
		obj = edv_recycle_bin_index_next(rp)
	)
		total_size += obj->size;
	edv_recycle_bin_index_close(rp);

	/* Has the total size meet or exceeded the recycle bin warn
	 * size?
	 */
	if(total_size >= recbin_size_warn)
	{
		gchar	*total_size_s = STRDUP(edv_str_size_delim(total_size)),
					*recbin_size_warn_s = STRDUP(edv_str_size_delim(
						recbin_size_warn
					)),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El tamao del cajn de la recirculacin es ahora %s bytes,\n\
ha excedido al usuario el tamao especificado de\n\
%s bytes en que este advertir deber ser mostrado.\n\
\n\
Al purge todos los objetos existentes de recycled, van\n\
primero al Cajn de Reciclaje y entonces van a\n\
File->Purge All.",
#elif defined(PROG_LANGUAGE_FRENCH)
"La taille du recycle l'huche est maintenant %s bytes,\n\
il a dpass l'utilisateur la taille spcifie de\n\
%s bytes  qui cet avertissement devrait tre montr.\n\
\n\
Purger tout l'existant recycl objets, premirement va\n\
au Recycle Huche et alors va File->Purge All.",
#else
"The size of the recycle bin is now %s bytes,\n\
it has exceeded the user specified size of %s bytes\n\
at which this warning is to be shown.\n\
\n\
To purge all the existing recycled objects, first\n\
go to the Recycle Bin and then go to File->Purge All.",
#endif
			total_size_s, recbin_size_warn_s
		);

		edv_play_sound_warning(core);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponseIconData(
#ifdef PROG_LANGUAGE_ENGLISH
			"Recycle Bin Size Warning",
#endif
#ifdef PROG_LANGUAGE_SPANISH
			"Advertir De Cajn De Reciclaje",
#endif
#ifdef PROG_LANGUAGE_FRENCH
			"Recycler l'Avertissement d'Huche",
#endif
			msg,
			NULL,
			edv_get_recycle_bin_icon_data(
				EDV_ICON_SIZE_32,
				1,
				NULL
			),
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		);
		g_free(msg);
		g_free(total_size_s);
		g_free(recbin_size_warn_s);
		CDialogSetTransientFor(NULL);
	}
}


/*
 *	Purges all the recycled objects in the recycle bin.
 */
void edv_purge_all_recycled_objects(
	EDVCore *core,
	const gboolean map_recbin,
	const gboolean show_progress,
	const gboolean interactive,
	GtkWidget *toplevel
)
{
	gboolean yes_to_all = FALSE;
	gint		status,
			nobjs,
			nobjs_purged = 0;
	gulong index;
	const gchar	*error_msg,
			*index_path;
	GList		*glist,
			*index_list = NULL;
	CfgList *cfg_list;
	EDVRecycledObject *obj;
	edv_recbin_struct *recbin;
	EDVRecycleBinIndex *rp;

	if(core == NULL)
		return;

#define CLEANUP_RETURN	{		\
 g_list_free(index_list);		\
					\
 return;				\
}

	cfg_list = core->cfg_list;

	/* Map the recycle bin? */
	if(map_recbin)
	{
		/* Create the Recycle Bin as needed and map it */
		edv_map_recycle_bin(
			core,
			core->geometry_flags,
			(core->geometry_flags != 0) ? &core->geometry : NULL
		);

		/* Get the pointer to the Recycle Bin so we know that is
		 * has been mapped
		 */
		recbin = core->recbin;

		/* If the toplevel GtkWidget is not specified then set the
		 * toplevel GtkWidget as the Recycle Bin's toplevel
		 * GtkWidget
		 */
		if((toplevel == NULL) && (recbin != NULL))
			toplevel = recbin->toplevel;
	}
	else
	{
		recbin = NULL;
	}

	/* Get the path to the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
	{
		CLEANUP_RETURN;
	}

	/* Get the list of recycled objects */
	rp = edv_recycle_bin_index_open(index_path);
	for(obj = edv_recycle_bin_index_next(rp);
		obj != NULL;
		obj = edv_recycle_bin_index_next(rp)
	)
	{
		if(obj->index != 0)
			index_list = g_list_append(
				index_list,
				(gpointer)obj->index
			);
	}
	edv_recycle_bin_index_close(rp);

	/* No recycled objects to purge? */
	if(index_list == NULL)
	{
		CLEANUP_RETURN;
	}

	nobjs = g_list_length(index_list);

	edv_recycle_bin_set_busy(recbin, TRUE);

	/* Confirm purge */
	if(interactive)
	{
		const gint response = edv_confirm_purge(
			core,
			toplevel,
			NULL,
			nobjs
		);
		switch(response)
		{
		  case CDIALOG_RESPONSE_YES:
		  case CDIALOG_RESPONSE_YES_TO_ALL:
			break;

		  default:
			edv_recycle_bin_set_busy(recbin, FALSE);
			CLEANUP_RETURN;
			break;
		}
	}

	/* Iterate through the list of recycled objects and purge
	 * each one
	 *
	 * Note that edv_purge_all_objects() is not used here because
	 * it does not allow the reporting of each recycled object
	 * purged during the purge
	 */
	status = 0;
	for(glist = index_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		index = (gulong)glist->data;

		/* Purge this recycled object */
		status = edv_purge_object(
			core,
			index,                  /* Recycled object to purge */
			toplevel,
			(nobjs > 0) ?
				((gfloat)nobjs_purged / (gfloat)nobjs) : -1.0f,
			TRUE,                   /* Show progress */
			TRUE,                   /* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_purge_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Purge Recycled Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

		/* Was the recycled object purged successfully? */
		if(status == 0)
		{
			/* Notify about this recycled object being purged */
			edv_emit_recycled_object_removed(core, index);
			nobjs_purged++;
		}

		/* User aborted? */
		if(status == -4)
			break;
	}

	/* If no errors occured or no user abort then call
	 * edv_purge_all_objects() to remove any stray recycled object
	 * files and the recycled objects index file
	 */
	if(status == 0)
	{
		/* Remove any stray recycled object files and the recycled
		 * objects index file
		 */
		status = edv_purge_all_objects(
			core,
			toplevel,
			FALSE,                  /* Do not show progress */
			TRUE,                   /* Interactive */
			&yes_to_all
		);

		/* Check for errors */
		error_msg = edv_purge_object_get_error(core);
		if(!STRISEMPTY(error_msg))
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Purge Recycled Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
		edv_play_sound_completed(core);

	edv_recycle_bin_set_busy(recbin, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}


/*
 *	Queries the user to unmount each mounted removable device.
 *
 *	This function is intended to be called just before exiting.
 */
void edv_query_unmount_before_exit(EDVCore *core)
{
	gboolean yes_to_all = FALSE;
	GList *glist;
	EDVDevice *d;

	for(glist = core->devices_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(EDV_DEVICE_IS_NO_UNMOUNT(d))
			continue;

		edv_device_update_mount_state(d);
		if(EDV_DEVICE_IS_MOUNTED(d))
		{
			gint response;

			if(yes_to_all)
			{
				response = CDIALOG_RESPONSE_YES;
			}
			else
			{
				gchar *msg = g_strdup_printf(
"Removable device \"%s\" is still mounted.\n\
\n\
Unmount the device?",
					STRISEMPTY(d->name) ? d->mount_path : d->name
				);
				CDialogSetTransientFor(NULL);
				response = CDialogGetResponse(
					"Unmount Device",
					msg,
					NULL,
					CDIALOG_ICON_QUESTION,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
						CDIALOG_BTNFLAG_NO,
					CDIALOG_BTNFLAG_NO
				);
				g_free(msg);
			}
			switch(response)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
				yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
				edv_device_unmount(
					core,
					d,
					TRUE,			/* Show progress */
					TRUE,			/* Verbose */
					NULL			/* No toplevel */
				);
				break;
			}
		}
	}
}


/*
 *	Calls the Endeavour Download Front End program to download the
 *	given URL described in dnd_obj to the directory target_path.
 *
 *	If the dnd_obj is not given then the download program will still
 *	be runned but used to query where the user wants to download
 *	to.
 */
gint edv_internet_download_object(
	EDVCore *core,
	const URLStruct *url,			/* Source */
	const gchar *tar_path,			/* Target */
	GtkWidget *toplevel
)
{
	gboolean confirm;
	const gchar *prog;
	CfgList *cfg_list;

	if((core == NULL) || STRISEMPTY(tar_path))
	{
		errno = EINVAL;
		return(-2);
	}

	cfg_list = core->cfg_list;

	/* Get configuration */
	confirm = EDV_GET_B(EDV_CFG_PARM_CONFIRM_DOWNLOAD);
	prog = EDV_GET_S(EDV_CFG_PARM_PROG_NET_DOWNLOAD);

	/* Download Front End program not set? */
	if(STRISEMPTY(prog))
	{
		edv_play_sound_error(core);
		edv_message_error(
			"No Download Program",
"Download Front End program not set",
"To set the network download program, please go\n\
to Settings->Options->Programs.",
			toplevel
		);
		errno = EINVAL;
		return(-2);
	}
	else
	{
		gint pid;
		gchar	*url_str,
					*shell_prog,
					*cmd;
		const gchar	*shell_args,
					*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL);
 
		/* Format the url */
		if(url == NULL)
			url_str = g_strdup("");
		else if(url->port > 0)
			url_str = g_strdup_printf(
				"%s://%s%s%s%s%s:%i%s",
				(url->protocol != NULL) ? url->protocol : "file",
				(url->user != NULL) ? url->user : "",
				(url->password != NULL) ? ":" : "",
				(url->password != NULL) ? url->password : "",
				(url->user != NULL) ? "@" : "",
				(url->host != NULL) ? url->host : "localhost",
				url->port,
				url->path
			);
		else
			url_str = g_strdup_printf(
				"%s://%s%s%s%s%s%s",
				(url->protocol != NULL) ? url->protocol : "file",
				(url->user != NULL) ? url->user : "",
				(url->password != NULL) ? ":" : "",
				(url->password != NULL) ? url->password : "",
				(url->user != NULL) ? "@" : "",
				(url->host != NULL) ? url->host : "localhost",
				url->path
			);

		/* Format the command
		 *
		 * The program must accept arguments where the first
		 * argument(s) are the URLs to download and the last
		 * argument is the destination path
		 */
		cmd = g_strdup_printf(
			"\"%s\"%s%s \"%s\" \"%s\"",
			prog,
			" -b",
			confirm ? " -c" : "",
			url_str,
			tar_path
		);

		/* Get the shell program and the shell arguments */
		shell_args = edv_strarg(
			shell_cmd,
			&shell_prog,
			TRUE,				/* Parse escapes */
			TRUE				/* Parse quotes */
		);

		/* Execute the download command */
		pid = edv_system_shell(
			cmd,
			shell_prog,
			shell_args
		);
		if(pid < 0)
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to execute the command:\n\
\n\
    %s\n\
\n\
%s.",
				cmd,
				g_strerror(error_code)
			);
			edv_play_sound_error(core);
			edv_message_error(
				"Download Failed",
				msg,
"Make sure that the network download program is\n\
defined correctly, please go to\n\
Settings->Options->Programs.",
				toplevel
			);
			g_free(msg);
			g_free(cmd);
			g_free(shell_prog);
			g_free(url_str);
			errno = (int)error_code;
			return(-1);
		}

		g_free(cmd);
		g_free(shell_prog);
		g_free(url_str);

		return(0);
	}
}
