#include <gtk/gtk.h>

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_property_directory.h"
#include "edv_pixmap.h"
#include "edv_device.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_obj_info_match.h"


EDVMatchObjectRelevency edv_match_object_icon(
	GList *devices_list,
	GList *mime_types_list,
	const EDVObjectType type,
	const gchar *path,
	const gboolean link_valid,
	const EDVPermissionFlags permissions,
	const EDVIconSize icon_size,
	EDVPixmap **icon_closed,
	EDVPixmap **icon_opened,
	EDVPixmap **icon_extended,
	EDVPixmap **icon_hidden
);
EDVMatchObjectRelevency edv_match_object_type_string(
	GList *mime_types_list,
	const EDVObjectType type,
	const gchar *path,
	const EDVPermissionFlags permissions,
	gchar **type_string_rtn
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the EDVPixmaps that best matches the object information.
 *
 *	The devices_list specifies the devices list. If
 *	devices_list is NULL then no devices will be matched.
 *
 *	The mime_types_list specifies the MIME types list. If
 *	mime_types_list is NULL then no MIME types will be matched.
 *
 *	The type specifies the object's type, one of EDV_OBJECT_TYPE_*.
 *
 *	The path specifies a string describing the object's full path
 *	or name. If path is not a full path then specific object
 *	matching will not be made.
 *
 *	If link_valid is TRUE then it hints that, if type is
 *	EDV_OBJECT_TYPE_LINK, then it is a valid link.
 *
 *	The permissions specifies the object's EDVPermissionFlags
 *	permissions.
 *
 *	The icon_size specifies the requested icon size to be returned.
 *
 *	If icon_closed, icon_opened, icon_extended, and/or icon_hidden
 *	are not NULL then they specify the return values for their
 *	respective icons. The returned icons will have a reference
 *	count added and therefore must be unref'ed by the calling
 *	function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
EDVMatchObjectRelevency edv_match_object_icon(
	GList *devices_list,
	GList *mime_types_list,
	const EDVObjectType type,
	const gchar *path,
	const gboolean link_valid,
	const EDVPermissionFlags permissions,
	const EDVIconSize icon_size,
	EDVPixmap **icon_closed,
	EDVPixmap **icon_opened,
	EDVPixmap **icon_extended,
	EDVPixmap **icon_hidden
)
{
	EDVMatchObjectRelevency match_status = EDV_MATCH_OBJECT_INFO_NONE;
	const gboolean is_file = (type == EDV_OBJECT_TYPE_FILE) ? TRUE : FALSE;
	EDVPixmap *icons_list[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Reset the local icon list */
	(void)memset(
		icons_list,
		0x00,
		sizeof(icons_list)
	);

	/* Reset the return values */
	if(icon_closed != NULL)
		*icon_closed = NULL;
	if(icon_opened != NULL)
		*icon_opened = NULL;
	if(icon_extended != NULL)
		*icon_extended = NULL;
	if(icon_hidden != NULL)
		*icon_hidden = NULL;

/* Sets the icons from the MIME Type, realizing the MIME type as needed
 *
 * If one or more pixmaps were obtained then match_status will be set
 */
#define GET_MIMETYPE_ICONS(_m_)	{			\
 EDVMIMETypeIconState state;				\
 const gint nstates = EDV_MIME_TYPE_TOTAL_ICON_STATES;	\
							\
 /* Realize MIME Type so that its icon pixmap and mask	\
  * pairs are loaded					\
  */							\
 edv_mime_type_realize((_m_), FALSE);			\
							\
 /* Get icons by the requested size */			\
 switch(icon_size) {					\
  case EDV_ICON_SIZE_16:				\
   /* TODO, add support for size 16 (MINI) */		\
   break;						\
  case EDV_ICON_SIZE_20:				\
   for(state = 0; state < nstates; state++)		\
    icons_list[state] = (_m_)->small_icon[state];	\
   break;						\
  case EDV_ICON_SIZE_32:				\
   for(state = 0; state < nstates; state++)		\
    icons_list[state] = (_m_)->medium_icon[state];	\
   break;						\
  case EDV_ICON_SIZE_48:				\
   for(state = 0; state < nstates; state++)		\
    icons_list[state] = (_m_)->large_icon[state];	\
   break;						\
 }							\
							\
 /* If we got a valid pixmap then set the		\
  * EDV_MATCH_OBJECT_INFO_GENERAL on the match_status	\
  * so we know that we have matched at least something	\
  */							\
 for(state = 0; state < nstates; state++) {		\
  if(edv_pixmap_is_loaded(icons_list[state])) {		\
   match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;	\
   break;						\
  }							\
 }							\
}

	/* If the object's type is a link then its icon should
	 * specifically display it as a link
	 */
	if(type == EDV_OBJECT_TYPE_LINK)
	{
		const gchar *mime_type_str = EDV_MIME_TYPE_TYPE_INODE_LINK;
		GList *glist;
		EDVMIMEType *m;

		/* Object is a link, now iterate through MIME Types list and
		 * find the MIME Type of class EDV_MIME_TYPE_CLASS_SYSTEM
		 * and use its icons
		 */
		for(glist = mime_types_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			m = EDV_MIME_TYPE(glist->data);
			if(m == NULL)
				continue;

			/* Only handle if MIME Type class is a systems object */
			if((m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM) &&
			   !STRISEMPTY(m->type)
			)
			{
				if(!strcmp((const char *)m->type, (const char *)mime_type_str))
				{
					GET_MIMETYPE_ICONS(m);
					match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
					break;
				}
			}
		}
	}

	/* If the object's type is a directory then check if its
	 * directory properties specify a specific icon
	 */
	if(type == EDV_OBJECT_TYPE_DIRECTORY)
	{
		/* The specified path must be a full path in order to get
		 * the directory properties
		 */
		if((path != NULL) ? g_path_is_absolute(path) : FALSE)
		{
			/* Closed */
			gchar *icon_path = edv_property_directory_get_icon_path(
				path,
				icon_size
			);
			if(icon_path != NULL)
			{
				EDVPixmap *icon = edv_pixmap_new_from_file(icon_path);
				if(edv_pixmap_is_loaded(icon))
				{
					if(icon_closed != NULL)
						*icon_closed = edv_pixmap_ref(icon);
					match_status |= EDV_MATCH_OBJECT_INFO_SPECIFIC;
				}

				(void)edv_pixmap_unref(icon);
				g_free(icon_path);
			}

			/* Opened */
			icon_path = edv_property_directory_get_icon_opened_path(
				path,
				icon_size
			);
			if(icon_path != NULL)
			{
				EDVPixmap *icon = edv_pixmap_new_from_file(icon_path);
				if(edv_pixmap_is_loaded(icon))
				{
					if(icon_opened != NULL)
						*icon_opened = edv_pixmap_ref(icon);
					match_status |= EDV_MATCH_OBJECT_INFO_SPECIFIC;
				}

				(void)edv_pixmap_unref(icon);
				g_free(icon_path);
			}
		}

		if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
			return(match_status);
	}

	/* Check the devices list to see if this object matches
	 * one of the devices' mount path
	 */
	if((devices_list != NULL) &&
	   ((path != NULL) ? g_path_is_absolute(path) : FALSE) &&
	   (match_status == EDV_MATCH_OBJECT_INFO_NONE)
	)
	{
		GList *glist;
		EDVDevice *d;

		/* Iterate through the list of devices */
		for(glist = devices_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			d = EDV_DEVICE(glist->data);
			if(d == NULL)
				continue;

			if(STRISEMPTY(d->mount_path))
				continue;

			/* Specified path matches device's mount path? */
			if(!strcmp((const char *)d->mount_path, (const char *)path))
			{
				EDVDeviceIconState device_icon_state;
				EDVMIMETypeIconState state;

				/* Realize this device first to ensure that the
				 * icons are loaded
				 */
				edv_device_realize(d, FALSE);

				/* Get the appropriate device icon state
				 *
				 * Is this device mounted?
				 */
				if(EDV_DEVICE_IS_MOUNTED(d))
				{
					device_icon_state = EDV_DEVICE_ICON_STATE_STANDARD;
				}
				else
				{
					device_icon_state = EDV_DEVICE_ICON_STATE_UNMOUNTED;
				}

				/* Get the device icon */
				switch(icon_size)
				{
				  case EDV_ICON_SIZE_16:
					break;
				  case EDV_ICON_SIZE_20:
					state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
					icons_list[state] = d->small_icon[device_icon_state];
					if(!edv_pixmap_is_loaded(icons_list[state]))
						icons_list[state] = d->small_icon[
							EDV_DEVICE_ICON_STATE_STANDARD
						];
					state = EDV_MIME_TYPE_ICON_STATE_OPENED;
					icons_list[state] = d->small_icon[
						EDV_DEVICE_ICON_STATE_SELECTED
					];
					if(!edv_pixmap_is_loaded(icons_list[state]))
						icons_list[state] = d->small_icon[
							EDV_DEVICE_ICON_STATE_STANDARD
						];
					break;
				  case EDV_ICON_SIZE_32:
					state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
					icons_list[state] = d->medium_icon[device_icon_state];
					if(!edv_pixmap_is_loaded(icons_list[state]))
						icons_list[state] = d->medium_icon[
							EDV_DEVICE_ICON_STATE_STANDARD
						];
					state = EDV_MIME_TYPE_ICON_STATE_OPENED;
					icons_list[state] = d->medium_icon[
						EDV_DEVICE_ICON_STATE_SELECTED
					];
					if(!edv_pixmap_is_loaded(icons_list[state]))
						icons_list[state] = d->medium_icon[
							EDV_DEVICE_ICON_STATE_STANDARD
						];
					break;
				  case EDV_ICON_SIZE_48:
					state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
					icons_list[state] = d->large_icon[device_icon_state];
					if(!edv_pixmap_is_loaded(icons_list[state]))
						icons_list[state] = d->large_icon[
							EDV_DEVICE_ICON_STATE_STANDARD
						];
					state = EDV_MIME_TYPE_ICON_STATE_OPENED;
					icons_list[state] = d->large_icon[
						EDV_DEVICE_ICON_STATE_SELECTED
					];
					if(!edv_pixmap_is_loaded(icons_list[state]))
						icons_list[state] = d->large_icon[
							EDV_DEVICE_ICON_STATE_STANDARD
						];
					break;
				}
				/* Got match? */
				if(edv_pixmap_is_loaded(icons_list[EDV_MIME_TYPE_ICON_STATE_STANDARD]))
				{
					match_status |= EDV_MATCH_OBJECT_INFO_EXACT;
					break;
				}
				break;
			}
		}	/* Iterate through devices */

		if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
		{
			/* Set the return values */
			if(icon_closed != NULL)
				*icon_closed = edv_pixmap_ref(icons_list[
					EDV_MIME_TYPE_ICON_STATE_STANDARD
				]);
			if(icon_opened != NULL)
				*icon_opened = edv_pixmap_ref(icons_list[
					EDV_MIME_TYPE_ICON_STATE_OPENED
				]);
			if(icon_extended != NULL)
				*icon_extended = edv_pixmap_ref(icons_list[
					EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
				]);
			if(icon_hidden != NULL)
				*icon_hidden = edv_pixmap_ref(icons_list[
					EDV_MIME_TYPE_ICON_STATE_HIDDEN
				]);

			return(match_status);
		}
	}

	/* Check the MIME Types list to see if this object matches
	 * one of the MIME Types
	 */
	if((mime_types_list != NULL) &&
	   (match_status == EDV_MATCH_OBJECT_INFO_NONE)
	)
	{
		const gchar	*name = (path != NULL) ? g_basename(path) : NULL,
				*value;
		GList *glist;
		EDVMIMEType *m;

		/* Iterate through the MIME Types list */
		for(glist = mime_types_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			m = EDV_MIME_TYPE(glist->data);
			if(m == NULL)
				continue;

			value = m->value;
			if(STRISEMPTY(value))
				continue;

			/* Handle by MIME Type class */
			switch(m->mt_class)
			{
			  case EDV_MIME_TYPE_CLASS_SYSTEM:
				break;
			  case EDV_MIME_TYPE_CLASS_FORMAT:
				if((name != NULL) &&
				   (match_status == EDV_MATCH_OBJECT_INFO_NONE) &&
				   is_file
				)
				{
					if(edv_name_has_extension(name, value))
					{
						GET_MIMETYPE_ICONS(m);
						if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
							match_status |= EDV_MATCH_OBJECT_INFO_SPECIFIC;
					}
				}
				break;
			  case EDV_MIME_TYPE_CLASS_PROGRAM:
				if((path != NULL) ? g_path_is_absolute(path) : FALSE)
				{
					if(!strcmp(value, path))
					{
						GET_MIMETYPE_ICONS(m);
						if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
							match_status |= EDV_MATCH_OBJECT_INFO_EXACT;
					}
				}
				break;
			  case EDV_MIME_TYPE_CLASS_UNIQUE:
				if((path != NULL) ? g_path_is_absolute(path) : FALSE)
				{
					if(!strcmp(value, path))
					{
						GET_MIMETYPE_ICONS(m);
						if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
							match_status |= EDV_MATCH_OBJECT_INFO_EXACT;
					}
				}
				break;
			}

			if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
				break;

		}	/* Iterate through the MIME Types list */
	}



	/* If still did not get match, then use the basic system MIME
	 * types for the specified object
	 */
	if(match_status == EDV_MATCH_OBJECT_INFO_NONE)
	{
		const gchar *mime_type_str = "";
		GList *glist;
		EDVMIMEType *m;

		/* Get MIME Type type string used for matching of system
		 * object type
		 */
		switch(type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_UNKNOWN;
			break;

		  case EDV_OBJECT_TYPE_FILE:
			/* Check the file's permissions allow execution, in
			 * which case we use the file/executable MIME Type
			 * instead of file/regular
			 */
			if(permissions & (EDV_PERMISSION_UX |
				EDV_PERMISSION_GX | EDV_PERMISSION_OX)
			)
				mime_type_str = EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE;
			else
				mime_type_str = EDV_MIME_TYPE_TYPE_INODE_FILE;
			break;

		  case EDV_OBJECT_TYPE_DIRECTORY:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DIRECTORY;
			break;

		  case EDV_OBJECT_TYPE_LINK:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_LINK;
			break;

		  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK;
			break;

		  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER;
			break;

		  case EDV_OBJECT_TYPE_FIFO:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_FIFO;
			break;

		  case EDV_OBJECT_TYPE_SOCKET:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_SOCKET;
			break;

		  case EDV_OBJECT_TYPE_ERROR:
			mime_type_str = EDV_MIME_TYPE_TYPE_INODE_ERROR;
			break;
		}

		/* Iterate through the MIME Types list */
		for(glist = mime_types_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			m = EDV_MIME_TYPE(glist->data);
			if(m == NULL)
				continue;

			/* Only handle if the MIME Type class is system */
			if((m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM) &&
			   !STRISEMPTY(m->type)
			)
			{
				if(!strcmp((const char *)m->type, (const char *)mime_type_str))
				{
					GET_MIMETYPE_ICONS(m);
					match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
					break;
				}
			}
		}

		/* If this object does not match any of the system
		 * MIME Types then return with the unknown icons
		 */
		if(match_status == EDV_MATCH_OBJECT_INFO_NONE)
		{
			m = edv_mime_types_list_match_type(
				mime_types_list,
				NULL,
				EDV_MIME_TYPE_TYPE_INODE_UNKNOWN,
				FALSE		/* Not case sensitive */
			);
			if(m != NULL)
			{
				GET_MIMETYPE_ICONS(m);
				match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
			}
		}
	}

#undef GET_MIMETYPE_ICONS


	/* Set the return values */
	if(icon_closed != NULL)
		*icon_closed = edv_pixmap_ref(icons_list[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		]);
	if(icon_opened != NULL)
		*icon_opened = edv_pixmap_ref(icons_list[
			EDV_MIME_TYPE_ICON_STATE_OPENED
		]);
	if(icon_extended != NULL)
		*icon_extended = edv_pixmap_ref(icons_list[
			EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
		]);
	if(icon_hidden != NULL)
		*icon_hidden = edv_pixmap_ref(icons_list[
			EDV_MIME_TYPE_ICON_STATE_HIDDEN
		]);

	return(match_status);
}

/*
 *	Gets the MIME Type type string that best matches the object
 *	information.
 *
 *	The mime_types_list specifies the MIME Types list.
 *
 *	The type specifies the object's type.
 *
 *	The path specifies either the full path or just the name of
 *	the object.
 *
 *	The permissions specifies the object's permissions.
 *
 *	If type_string_rtn is not NULL then it specifies the return
 *	value for the type string. The returned type string must be
 *	deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
EDVMatchObjectRelevency edv_match_object_type_string(
	GList *mime_types_list,
	const EDVObjectType type,
	const gchar *path,
	const EDVPermissionFlags permissions,
	gchar **type_string_rtn
)
{
	const gboolean is_file = (type == EDV_OBJECT_TYPE_FILE) ? TRUE : FALSE;
	EDVMatchObjectRelevency match_status = EDV_MATCH_OBJECT_INFO_NONE;
	const gchar	*type_string = NULL,
			*name = (path != NULL) ? g_basename(path) : NULL;
	EDVMIMEType *m;

	if(type_string_rtn != NULL)
		*type_string_rtn = NULL;

	/* First check if the object is a link */
	if(type == EDV_OBJECT_TYPE_LINK)
	{
		type_string = EDV_MIME_TYPE_TYPE_INODE_LINK;
		match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
	}

	/* Check the MIME Types list for a MIME Type that matches
	 * this object object
	 */
	if((match_status == EDV_MATCH_OBJECT_INFO_NONE) &&
	   (mime_types_list != NULL)
	)
	{
		const gchar *value;
		GList *glist;

		/* Iterate through the MIME Types list */
		for(glist = mime_types_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			m = EDV_MIME_TYPE(glist->data);
			if(m == NULL)
				continue;

			value = m->value;
			if(STRISEMPTY(value))
				continue;

			/* Handle by MIME Type class */
			switch(m->mt_class)
			{
			  case EDV_MIME_TYPE_CLASS_SYSTEM:
				break;
			  case EDV_MIME_TYPE_CLASS_FORMAT:
				if((name != NULL) &&
				   (match_status == EDV_MATCH_OBJECT_INFO_NONE) &&
				   is_file
				)
				{
					if(edv_name_has_extension(name, value))
					{
						type_string = m->type;
						if(type_string != NULL)
							match_status |= EDV_MATCH_OBJECT_INFO_GENERAL |
								EDV_MATCH_OBJECT_INFO_SPECIFIC;
					}
				}
				break;
			  case EDV_MIME_TYPE_CLASS_PROGRAM:
				if((path != NULL) ? g_path_is_absolute(path) : FALSE)
				{
					if(!strcmp((const char *)value, (const char *)path))
					{
						type_string = m->type;
						if(type_string != NULL)
							match_status |= EDV_MATCH_OBJECT_INFO_GENERAL |
								EDV_MATCH_OBJECT_INFO_SPECIFIC |
								EDV_MATCH_OBJECT_INFO_EXACT;
					}
				}
				break;
			  case EDV_MIME_TYPE_CLASS_UNIQUE:
				if((path != NULL) ? g_path_is_absolute(path) : FALSE)
				{
					if(!strcmp((const char *)value, (const char *)path))
					{
						type_string = m->type;
						if(type_string != NULL)
							match_status |= EDV_MATCH_OBJECT_INFO_GENERAL |
								EDV_MATCH_OBJECT_INFO_SPECIFIC |
								EDV_MATCH_OBJECT_INFO_EXACT;
					}
				}
				break;
			}

			if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
				break;

		}	/* Iterate through the MIME Types list */
	}

	/* If there was no match then set the MIME Type string to
	 * one based on the object's type
	 */
	if(match_status == EDV_MATCH_OBJECT_INFO_NONE)
	{
		const gchar *s = "";

		/* Get MIME Type type string used for matching of system
		 * object type
		 */
		switch(type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
			s = EDV_MIME_TYPE_TYPE_INODE_UNKNOWN;
			break;
		  case EDV_OBJECT_TYPE_FILE:
			/* Check the file's permissions allow execution, in which
			 * case we use the file/executable MIME Type instead of
			 * file/regular
			 */
			if(permissions & (EDV_PERMISSION_UX |
				EDV_PERMISSION_GX | EDV_PERMISSION_OX)
			)
				s = EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE;
			else
				s = EDV_MIME_TYPE_TYPE_INODE_FILE;
			break;
		  case EDV_OBJECT_TYPE_DIRECTORY:
			s = EDV_MIME_TYPE_TYPE_INODE_DIRECTORY;
			break;
		  case EDV_OBJECT_TYPE_LINK:
			s = EDV_MIME_TYPE_TYPE_INODE_LINK;
			break;
		  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			s = EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK;
			break;
		  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			s = EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER;
			break;
		  case EDV_OBJECT_TYPE_FIFO:
			s = EDV_MIME_TYPE_TYPE_INODE_FIFO;
			break;
		  case EDV_OBJECT_TYPE_SOCKET:
			s = EDV_MIME_TYPE_TYPE_INODE_SOCKET;
			break;
		  case EDV_OBJECT_TYPE_ERROR:
			s = EDV_MIME_TYPE_TYPE_INODE_ERROR;
			break;
		}

		type_string = s;

		match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
	}

	/* Set the return values */
	if(type_string_rtn != NULL)
		*type_string_rtn = STRDUP(type_string);

	return(match_status);
}
