/*
	  Endeavour Mark II - Moving Picture Experts Group Audio

	Simple reading of MPEG audio streams.

	Designed to obtain basic MPEG stream information for display
	purposes.

	No encoding or decoding support.
 */

#ifndef EDV_MPEG_AUDIO_H
#define EDV_MPEG_AUDIO_H

#include <stdio.h>				/* FILE * */
#include <glib.h>


typedef struct _EDVMPEGAudioFrameHeader	EDVMPEGAudioFrameHeader;
#define EDV_MPEG_AUDIO_FRAME_HEADER(p)	((EDVMPEGAudioFrameHeader *)(p))
typedef struct _EDVMPEGAudioVBRHeader	EDVMPEGAudioVBRHeader;
#define EDV_MPEG_AUDIO_VBR_HEADER(p)	((EDVMPEGAudioVBRHeader *)(p))


/*
 *	Channel Modes:
 */
typedef enum {
	EDV_MPEG_AUDIO_CHANNEL_MODE_MONO,
	EDV_MPEG_AUDIO_CHANNEL_MODE_DUAL_MONO,
	EDV_MPEG_AUDIO_CHANNEL_MODE_STEREO,
	EDV_MPEG_AUDIO_CHANNEL_MODE_JOINT_STEREO
} EDVMPEGAudioChannelMode;


/*
 *	Emphasis:
 */
typedef enum {
	EDV_MPEG_AUDIO_EMPHASIS_NONE,
	EDV_MPEG_AUDIO_EMPHASIS_50_15MS,
	EDV_MPEG_AUDIO_EMPHASIS_RESERVED,
	EDV_MPEG_AUDIO_EMPHASIS_CCIT_J_17
} EDVMPEGAudioEmphasis;


/*
 *	Frame Header:
 */
struct _EDVMPEGAudioFrameHeader {
	gint		version_major,
			version_minor,
			layer;
	gboolean	crc_16_protection;	/* 16 bits CRC protection
						 * follows the frame header */
	gint		bitrate_kbps,		/* Bit rate in kilo bits
						 * per second */
			sample_rate_hz;		/* Sample rate in Hz */
	gboolean	padding,
			private;
	EDVMPEGAudioChannelMode	channel_mode;

	/* Channel mode extension, this is only used if channel_mode
	 * is set to EDV_MPEG_AUDIO_CHANNEL_MODE_JOINT_STEREO
	 *
	 * Value	Layer I and II		Layer III Stereo
	 * 0		bands 4 to 31		None		None
	 * 1		bands 8 to 31		Intensity	None
	 * 2		bands 12 to 31		None		MS
	 * 3		bands 16 to 31		Intensity	MS
	 */
	gint		mode_extension;

	gboolean	copyrighted,		/* FALSE = Not copyrighted,
						 * TRUE = Copyrighted */
			original;		/* FALSE = Copy of original
						 * TRUE = Original */
	EDVMPEGAudioEmphasis emphasis;

	/* Variable Bit Rate (VBR) Header */
	EDVMPEGAudioVBRHeader	*vbr_header;
};


/*
 *	Variable Bit Rate (VBR) Header:
 */
struct _EDVMPEGAudioVBRHeader {

	gint		quality;		/* 0 to 100 or -1
						 * 0 = Best
						 * 100 = Worst
						 * -1 = Not set */

	gulong		nframes,
			nbytes;

	guint8		*toc;			/* Table of contents */
	gint		toc_nentries,
			toc_entry_size,		/* 1, 2, or 4 bytes */
			toc_len,		/* toc_nentries * toc_entry_size */
			toc_entry_scale_factor,
			toc_frames_per_entry;

};


/* Calculation */
extern gulong edv_mpeg_audio_frame_samples(EDVMPEGAudioFrameHeader *header);
extern gulong edv_mpeg_audio_frame_length(EDVMPEGAudioFrameHeader *header);
extern gulong edv_mpeg_audio_frame_calculate_length_ms(
	EDVMPEGAudioFrameHeader *header,
	const gulong length_bytes
);

/* EDVMPEGAudioFrameHeader */
extern EDVMPEGAudioFrameHeader *edv_mpeg_audio_frame_header_new(void);
extern void edv_mpeg_audio_frame_header_delete(EDVMPEGAudioFrameHeader *header);

/* EDVMPEGAudioVBRHeader */
extern EDVMPEGAudioVBRHeader *edv_mpeg_audio_vbr_header_new(void);
extern void edv_mpeg_audio_vbr_header_delete(EDVMPEGAudioVBRHeader *header);

/* Stream */
extern EDVMPEGAudioFrameHeader *edv_mpeg_audio_stream_next_frame(FILE *fp);
extern gulong edv_mpeg_audio_scan_stream_length(
	FILE *fp,
	gulong *nframes_rtn,
	gulong *ms_rtn,
	gint *bitrate_min_rtn,
	gint *bitrate_max_rtn
);


#endif	/* EDV_MPEG_AUDIO_H */
