#include <stdio.h>
#include <errno.h>
#include <glib.h>

#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"

#include "edv_id3.h"
#include "edv_mpeg_audio.h"

/* Calculation */
static guint16 edv_mpeg_audio_swap_i16(const guint16 x);
static guint32 edv_mpeg_audio_swap_i32(const guint32 x);
gulong edv_mpeg_audio_frame_samples(EDVMPEGAudioFrameHeader *header);
gulong edv_mpeg_audio_frame_length(EDVMPEGAudioFrameHeader *header);
gulong edv_mpeg_audio_frame_calculate_length_ms(
	EDVMPEGAudioFrameHeader *header,
	const gulong length_bytes
);

/* EDVMPEGAudioFrameHeader */
EDVMPEGAudioFrameHeader *edv_mpeg_audio_frame_header_new(void);
void edv_mpeg_audio_frame_header_delete(EDVMPEGAudioFrameHeader *header);

/* EDVMPEGAudioVBRHeader */
EDVMPEGAudioVBRHeader *edv_mpeg_audio_vbr_header_new(void);
void edv_mpeg_audio_vbr_header_delete(EDVMPEGAudioVBRHeader *header);

/* Stream */
static gint edv_mpeg_audio_data_parse_bitrate(
	const guint8 v,
	const gint version_major,
	const gint layer
);
static gint edv_mpeg_audio_data_parse_sample_rate(
	const guint8 v,
	const gint version_major,
	const gint version_minor
);
static EDVMPEGAudioFrameHeader *edv_mpeg_audio_data_parse_frame_header(
	const guint8 *buf,
	const gint buf_len
);
static EDVMPEGAudioVBRHeader *edv_mpeg_audio_data_parse_xing_header(
	const guint8 *buf,
        const gint buf_len
);
static EDVMPEGAudioVBRHeader *edv_mpeg_audio_data_parse_vbri_header(
	const guint8 *buf,
        const gint buf_len
);
static EDVMPEGAudioVBRHeader *edv_mpeg_audio_data_parse_vbr_header(
	const guint8 *buf,
        const gint buf_len
);
EDVMPEGAudioFrameHeader *edv_mpeg_audio_stream_next_frame(FILE *fp);
gulong edv_mpeg_audio_scan_stream_length(
	FILE *fp,
	gulong *nframes_rtn,
	gulong *ms_rtn,
	gint *bitrate_min_rtn,
	gint *bitrate_max_rtn
);


/*
 *	Bitrate Tables:
 *
 *	Different versions and layers have different bitrate conversions
 *	from raw data.
 */
#define EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V1L1	{	\
	0,						\
	32,						\
	64,						\
	96,						\
	128,						\
	160,						\
	192,						\
	224,						\
	256,						\
	288,						\
	320,						\
	352,						\
	384,						\
	416,						\
	448,						\
	0						\
}
#define EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V1L2	{	\
	0,						\
	32,						\
	48,						\
	56,						\
	64,						\
	80,						\
	96,						\
	112,						\
	128,						\
	160,						\
	192,						\
	224,						\
	256,						\
	320,						\
	384,						\
	0						\
}
#define EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V1L3	{	\
	0,						\
	32,						\
	40,						\
	48,						\
	56,						\
	64,						\
	80,						\
	96,						\
	112,						\
	128,						\
	160,						\
	192,						\
	224,						\
	256,						\
	320,						\
	0						\
}
#define EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V2L1	{	\
	0,						\
	32,						\
	48,						\
	56,						\
	64,						\
	80,						\
	96,						\
	112,						\
	128,						\
	144,						\
	160,						\
	176,						\
	192,						\
	224,						\
	256,						\
	0						\
}
#define EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V2L23	{	\
	0,						\
	8,						\
	16,						\
	24,						\
	32,						\
	40,						\
	48,						\
	56,						\
	64,						\
	80,						\
	96,						\
	112,						\
	128,						\
	144,						\
	160,						\
	0						\
}


/*
 *	Swaps the 16 bit WORD value from MPEG big endiness to host
 *	endiness.
 */
static guint16 edv_mpeg_audio_swap_i16(const guint16 x)
{
	guint16 xs = (x & 0x00FF);
	xs = ((x & 0xFF00) >> 8) | (xs << 8);
	return(xs);
}

/*
 *	Swaps the 32 bit DWORD value from MPEG big endiness to host
 *	endiness.
 */
static guint32 edv_mpeg_audio_swap_i32(const guint32 x)
{
	guint32 xs = (x & 0x000000FF);
	xs = ((x & 0x0000FF00) >> 8) | (xs << 8);
	xs = ((x & 0x00FF0000) >> 16) | (xs << 8);
	xs = ((x & 0xFF000000) >> 24) | (xs << 8);
	return(xs);
}

/*
 *	Gets the number of samples in the frame.
 *
 *	Returns the number of samples or 0 on error.
 */
gulong edv_mpeg_audio_frame_samples(EDVMPEGAudioFrameHeader *header)
{
	if(header == NULL)
	{
		errno = EINVAL;
		return(0l);
	}

	switch(header->layer)
	{
	    case 1:
		return(384l);
		break;
	    case 2:
	    case 3:
		return(1152l);
		break;
	}

	errno = EINVAL;

	return(0l);
}

/*
 *	Calculates the compressed length of the MPEG frame in bytes.
 *
 *	Returns the compressed length in bytes or 0 on error.
 */
gulong edv_mpeg_audio_frame_length(EDVMPEGAudioFrameHeader *header)
{
	gulong padding;

	if(header == NULL)
	{
		errno = EINVAL;
		return(0l);
	}

	if((header->bitrate_kbps <= 0) || (header->sample_rate_hz == 0l))
	{
		errno = EINVAL;
		return(0l);
	}

	padding = (header->padding) ? 1 : 0;

	switch(header->layer)
	{
	    case 1:
		return(
			(24000l * 2l * (gulong)header->bitrate_kbps /
				header->sample_rate_hz) + (padding * 4)
		);
		break;
	    case 2:
	    case 3:
		return(
			(72000l * 2l * (gulong)header->bitrate_kbps /
				header->sample_rate_hz) + padding
		);
		break;
	}

	errno = EINVAL;

	return(0l);
}

/*
 *	Calculates the length of the MPEG frame in milliseconds.
 *
 *	Returns the length in milliseconds or 0 on error.
 */
gulong edv_mpeg_audio_frame_calculate_length_ms(
	EDVMPEGAudioFrameHeader *header,
	const gulong length_bytes
)
{
	if(header == NULL)
	{
		errno = EINVAL;
		return(0l);
	}

	if(header->bitrate_kbps <= 0)
	{
		errno = EINVAL;
		return(0l);
	}

	return(
		length_bytes * 8l /
			(gulong)header->bitrate_kbps
	);
}


/*
 *	Creates a new EDVMPEGAudioFrameHeader.
 *
 *	Returns a new dynamically allocated EDVMPEGAudioFrameHeader with
 *	all of its values zero'ed or NULL on error.
 */
EDVMPEGAudioFrameHeader *edv_mpeg_audio_frame_header_new(void)
{
	return(EDV_MPEG_AUDIO_FRAME_HEADER(g_malloc0(sizeof(EDVMPEGAudioFrameHeader))));
}

/*
 *	Deletes the EDVMPEGAudioFrameHeader.
 *
 *	The obj specifies the EDVMPEGAudioFrameHeader to delete.
 */
void edv_mpeg_audio_frame_header_delete(EDVMPEGAudioFrameHeader *header)
{
	if(header == NULL)
		return;

	edv_mpeg_audio_vbr_header_delete(header->vbr_header);
	g_free(header);
}


/*
 *	Creates a new EDVMPEGAudioVBRHeader.
 *
 *	Returns a new dynamically allocated EDVMPEGAudioVBRHeader with
 *	all of its values zero'ed or NULL on error.
 */
EDVMPEGAudioVBRHeader *edv_mpeg_audio_vbr_header_new(void)
{
	return(EDV_MPEG_AUDIO_VBR_HEADER(g_malloc0(sizeof(EDVMPEGAudioVBRHeader))));
}

/*
 *	Deletes the EDVMPEGAudioVBRHeader.
 *
 *	The obj specifies the EDVMPEGAudioVBRHeader to delete.
 */
void edv_mpeg_audio_vbr_header_delete(EDVMPEGAudioVBRHeader *header)
{
	if(header == NULL)
		return;

	g_free(header->toc);
	g_free(header);
}


/*
 *	Parses the bitrate in kilo bits per second from the raw data.
 */
static gint edv_mpeg_audio_data_parse_bitrate(
	const guint8 v,
	const gint version_major,
	const gint layer
)
{
	const guint8	mask = ((1 << 7) | (1 << 6) | (1 << 5) | (1 << 4)),
			vms = ((v & mask) >> 4);
	const gint	bitrate_table_v1l1[16] = EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V1L1,
			bitrate_table_v1l2[16] = EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V1L2,
			bitrate_table_v1l3[16] = EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V1L3,
			bitrate_table_v2l1[16] = EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V2L1,
			bitrate_table_v2l23[16] = EDV_MPEG_AUDIO_BITRATE_KBPS_LIST_V2L23,
			*bitrate_table = NULL;

	switch(version_major)
	{
	    case 1:
		switch(layer)
		{
		    case 1:
			bitrate_table = bitrate_table_v1l1;
			break;
		    case 2:
			bitrate_table = bitrate_table_v1l2;
			break;
		    case 3:
			bitrate_table = bitrate_table_v1l3;
			break;
		}
		break;
	    case 2:
		switch(layer)
		{
		    case 1:
			bitrate_table = bitrate_table_v2l1;
			break;
		    case 2:
			bitrate_table = bitrate_table_v2l23;
			break;
		}
		break;
	}
	if(bitrate_table == NULL)
		return(0);

	if(vms == 0)
		return(bitrate_table[0]);
	else if(vms == (1 << 0))
		return(bitrate_table[1]);
	else if(vms == (1 << 1))
		return(bitrate_table[2]);
	else if(vms == ((1 << 1) | (1 << 0)))
		return(bitrate_table[3]);
	else if(vms == (1 << 2))
		return(bitrate_table[4]);
	else if(vms == ((1 << 2) | (1 << 0)))
		return(bitrate_table[5]);
	else if(vms == ((1 << 2) | (1 << 1)))
		return(bitrate_table[6]);
	else if(vms == ((1 << 2) | (1 << 1) | (1 << 0)))
		return(bitrate_table[7]);
	else if(vms == (1 << 3))
		return(bitrate_table[8]);
	else if(vms == ((1 << 3) | (1 << 0)))
		return(bitrate_table[9]);
	else if(vms == ((1 << 3) | (1 << 1)))
		return(bitrate_table[10]);
	else if(vms == ((1 << 3) | (1 << 1) | (1 << 0)))
		return(bitrate_table[11]);
	else if(vms == ((1 << 3) | (1 << 2)))
		return(bitrate_table[12]);
	else if(vms == ((1 << 3) | (1 << 2) | (1 << 0)))
		return(bitrate_table[13]);
	else if(vms == ((1 << 3) | (1 << 2) | (1 << 1)))
		return(bitrate_table[14]);
	else if(vms == ((1 << 3) | (1 << 2) | (1 << 1) | (1 << 0)))
		return(bitrate_table[15]);

	return(0);
}

/*
 *	Parses the sample rate in Hz from the raw data.
 */
static gint edv_mpeg_audio_data_parse_sample_rate(
	const guint8 v,
	const gint version_major,
	const gint version_minor
)
{
	const guint8	mask = ((1 << 3) | (1 << 2)),
			vms = ((v & mask) >> 2);

	switch(version_major)
	{
	    case 1:
		if(vms == 0)
			return(44100);
		else if(vms == (1 << 0))
			return(48000);
		else if(vms == (1 << 1))
			return(32000);
		break;
	    case 2:
		switch(version_minor)
		{
		    case 0:
			if(vms == 0)
				return(22050);
			else if(vms == (1 << 0))
				return(24000);
			else if(vms == (1 << 1))
				return(16000);
			break;
		    case 5:
			if(vms == 0)
				return(11025);
			else if(vms == (1 << 0))
				return(12000);
			else if(vms == (1 << 1))
				return(8000);
			break;
		}
		break;
	}

	return(0);
}

/*
 *	Parses the MPEG frame header from the buffer.
 *
 *	Does not search for or parses the VBR header.
 *
 *	The buf specifies the compressed data buffer positioned at
 *	the potential start of the MPEG frame header at the MPEG sync
 *	word.
 */
static EDVMPEGAudioFrameHeader *edv_mpeg_audio_data_parse_frame_header(
	const guint8 *buf,
	const gint buf_len
)
{
	guint8 v8;
	EDVMPEGAudioFrameHeader *header;

	if((buf == NULL) || (buf_len < 4))
		return(NULL);

	/* Create a new EDVMPEGAudioFrameHeader */
	header = edv_mpeg_audio_frame_header_new();
	if(header == NULL)
		return(NULL);

	/* Skip the first 0xFF byte */

	/* Get the next byte */
	v8 = buf[1];

	/* Version */
	if((v8 & (1 << 4)) &&
	   (v8 & (1 << 3))
	)
	{
		header->version_major = 1;
		header->version_minor = 0;
	}
	else if(v8 & (1 << 4))
	{
		header->version_major = 2;
		header->version_minor = 0;
	}
	else if(v8 & (1 << 3))
	{
		/* Reserved */
		header->version_major = 0;
		header->version_minor = 0;
	}
	else
	{
		/* 2.5 (MPEG 2.0 extension) */
		header->version_major = 2;
		header->version_minor = 5;
	}

	/* Layer */
	if((v8 & (1 << 2)) &&
	   (v8 & (1 << 1))
	)
		header->layer = 1;
	else if(v8 & (1 << 2))
		header->layer = 2;
	else if(v8 & (1 << 1))
		header->layer = 3;
	else
		header->layer = 0;		/* Reserved */

	/* CRC 16 Bit Protection (note that unprotected is when the bit
	 * is set) */
	header->crc_16_protection = (v8 & (1 << 0)) ? FALSE : TRUE;

	/* Get the next byte */
	v8 = buf[2];

	/* Bitrate KBPS */
	header->bitrate_kbps = edv_mpeg_audio_data_parse_bitrate(
		v8,
		header->version_major,
		header->layer
	);

	/* Sample Rate Hz */
	header->sample_rate_hz = edv_mpeg_audio_data_parse_sample_rate(
		v8,
		header->version_major,
		header->version_minor
	);

	/* Frames are padded with one extra byte */
	header->padding = (v8 & (1 << 1)) ? TRUE : FALSE;

	/* Private bit for informative purposes */
	header->private = (v8 & (1 << 0)) ? TRUE : FALSE;

	/* Get the next byte */
	v8 = buf[3];

	/* Channel Mode */
	if((v8 & (1 << 7)) &&
	   (v8 & (1 << 6))
	)
		header->channel_mode = EDV_MPEG_AUDIO_CHANNEL_MODE_MONO;
	else if(v8 & (1 << 7))
		header->channel_mode = EDV_MPEG_AUDIO_CHANNEL_MODE_DUAL_MONO;
	else if(v8 & (1 << 6))
		header->channel_mode = EDV_MPEG_AUDIO_CHANNEL_MODE_JOINT_STEREO;
	else
		header->channel_mode = EDV_MPEG_AUDIO_CHANNEL_MODE_STEREO;

	/* Joint Stereo Mode Extension */
	if(header->channel_mode == EDV_MPEG_AUDIO_CHANNEL_MODE_JOINT_STEREO)
	{
		if((v8 & (1 << 5)) &&
		   (v8 & (1 << 4))
		)
			header->mode_extension = 3;
		else if(v8 & (1 << 5))
			header->mode_extension = 2;
		else if(v8 & (1 << 4))
			header->mode_extension = 1;
		else
			header->mode_extension = 0;
	}

	/* Copyrighted */
	header->copyrighted = (v8 & (1 << 3)) ? TRUE : FALSE;

	/* Original */
	header->original = (v8 & (1 << 2)) ? TRUE : FALSE;

	/* EDVMPEGAudioEmphasis */
	if((v8 & (1 << 1)) &&
	   (v8 & (1 << 0))
	)
		header->emphasis = EDV_MPEG_AUDIO_EMPHASIS_CCIT_J_17;
	else if(v8 & (1 << 1))
		header->emphasis = EDV_MPEG_AUDIO_EMPHASIS_RESERVED;
	else if(v8 & (1 << 0))
		header->emphasis = EDV_MPEG_AUDIO_EMPHASIS_50_15MS;
	else
		header->emphasis = EDV_MPEG_AUDIO_EMPHASIS_NONE;

	return(header);
}

/*
 *	Parses the XING header from the buffer.
 *
 *	The buf specifies the compressed data buffer positioned at
 *	the start of the XING header (at the ID).
 *
 *	Returns a new dynamically allocated EDVMPEGAudioVBRHeader
 *	describing the XING header or NULL on error.
 */
static EDVMPEGAudioVBRHeader *edv_mpeg_audio_data_parse_xing_header(
	const guint8 *buf,
        const gint buf_len
)
{
	guint32 flags = 0x00000000;
	const guint8	*buf_ptr, *buf_end;
	EDVMPEGAudioVBRHeader *header;

	if((buf == NULL) || (buf_len < 4))
		return(NULL);

	buf_ptr = buf;
	buf_end = buf_ptr + buf_len;

	/* Create a new EDVMPEGAudioVBRHeader */
	header = edv_mpeg_audio_vbr_header_new();
	if(header == NULL)
		return(NULL);

	/* Seek past the ID */
	buf_ptr += 4;

	/* Flags (4 bytes long) */
	if((buf_ptr + 4) <= buf_end)
	{
		flags = edv_mpeg_audio_swap_i32(*(guint32 *)buf_ptr);
		buf_ptr += 4;
	}

	/* Number Of Frames (4 bytes long, big endian DWORD)*/
	if((flags & (1 << 0)) && ((buf_ptr + 4) <= buf_end))
	{
		header->nframes = (gulong)edv_mpeg_audio_swap_i32(
			*(guint32 *)buf_ptr
		);
		buf_ptr += 4;
	}

	/* Number Of Bytes (4 bytes long, big endian DWORD)*/
	if((flags & (1 << 1)) && ((buf_ptr + 4) <= buf_end))
	{
		header->nbytes = (gulong)edv_mpeg_audio_swap_i32(
			*(guint32 *)buf_ptr
		);
		buf_ptr += 4;
	}

	/* Table Of Contents (TOC) (100 bytes) */
	if((flags & (1 << 2)) && ((buf_ptr + 100) <= buf_end))
	{
		header->toc_nentries = 100;
		header->toc_entry_size = 1;
		header->toc_len = header->toc_nentries * header->toc_entry_size;
		header->toc_entry_scale_factor = 1;
		header->toc_frames_per_entry = 1;	/* ??? */
		header->toc = g_memdup(
			buf_ptr,
			header->toc_len * sizeof(guint8)
		);
		buf_ptr += 100;
	}

	/* Quality (4 bytes long, big endian word)*/
	if((flags & (1 << 3)) && ((buf_ptr + 4) <= buf_end))
	{
		header->quality = (gint)edv_mpeg_audio_swap_i32(
			*(guint32 *)buf_ptr
		);
		buf_ptr += 4;
	}
	else
	{
		header->quality = -1;
	}

	return(header);
}

/*
 *	Parses the VBRI header from the buffer.
 *
 *	The buf specifies the compressed data buffer positioned at
 *	the start of the VBRI header (at the ID).
 *
 *	Returns a new dynamically allocated EDVMPEGAudioVBRHeader
 *	describing the VBRI header or NULL on error.
 */
static EDVMPEGAudioVBRHeader *edv_mpeg_audio_data_parse_vbri_header(
	const guint8 *buf,
        const gint buf_len
)
{
	guint16 version = 0x0000;
	gulong len;
	gfloat delay = 0.0f;
	const guint8	*buf_ptr, *buf_end;
	EDVMPEGAudioVBRHeader *header;

	if((buf == NULL) || (buf_len < 4))
		return(NULL);

	buf_ptr = buf;
	buf_end = buf_ptr + buf_len;

	/* Create a new EDVMPEGAudioVBRHeader */
	header = edv_mpeg_audio_vbr_header_new();
	if(header == NULL)
		return(NULL);

	/* Seek past the ID */
	buf_ptr += 4;

	/* Version (2 bytes long) */
	if((buf_ptr + 2) <= buf_end)
	{
		version = edv_mpeg_audio_swap_i16(*(guint16 *)buf_ptr);
		buf_ptr += 2;
	}

	/* Delay (2 bytes long) */
	if((buf_ptr + 2) <= buf_end)
	{
		delay = (gfloat)edv_mpeg_audio_swap_i16(*(guint16 *)buf_ptr);
/* Not sure if this is parsed correctly as a gfloat but its value is
 * not used so it does not matter
 */
		buf_ptr += 2;
	}

	/* Quality (2 bytes long) */
	if((buf_ptr + 2) <= buf_end)
	{
		header->quality = (gint)edv_mpeg_audio_swap_i16(*(guint16 *)buf_ptr);
		buf_ptr += 2;
	}
	else
	{
		header->quality = -1;
	}

	/* Number Of Bytes (4 bytes long, big endian DWORD)*/
	if((buf_ptr + 4) <= buf_end)
	{
		header->nbytes = (gulong)edv_mpeg_audio_swap_i32(
			*(guint32 *)buf_ptr
		);
		buf_ptr += 4;
	}

	/* Number Of Frames (4 bytes long, big endian DWORD)*/
	if((buf_ptr + 4) <= buf_end)
	{
		header->nframes = (gulong)edv_mpeg_audio_swap_i32(
			*(guint32 *)buf_ptr
		);
		buf_ptr += 4;
	}

	/* Table Of Contents (TOC) Entries (2 bytes long) */
	if((buf_ptr + 2) <= buf_end)
	{
		header->toc_nentries = (gint)edv_mpeg_audio_swap_i16(*(guint16 *)buf_ptr);
		/* Should we impose a limit? */
		buf_ptr += 2;
	}

	/* TOC Entry Scale Factor (2 bytes long) */
	if((buf_ptr + 2) <= buf_end)
	{
		header->toc_entry_scale_factor = (gint)edv_mpeg_audio_swap_i16(*(guint16 *)buf_ptr);
		buf_ptr += 2;
	}

	/* TOC Entry Size (2 bytes long) */
	if((buf_ptr + 2) <= buf_end)
	{
		header->toc_entry_size = (gint)edv_mpeg_audio_swap_i16(*(guint16 *)buf_ptr);
		if(header->toc_entry_size > 4)
			header->toc_entry_size = 4;
		else if(header->toc_entry_size < 1)
			header->toc_entry_size = 1;
		buf_ptr += 2;
	}

	/* TOC Frames Per Entry (2 bytes long) */
	if((buf_ptr + 2) <= buf_end)
	{
		header->toc_frames_per_entry = (gint)edv_mpeg_audio_swap_i16(*(guint16 *)buf_ptr);
		buf_ptr += 2;
	}

	/* TOC */
	header->toc_len = len = header->toc_nentries * header->toc_entry_size;
	if(((buf_ptr + len) <= buf_end) && (len > 0l))
	{
		header->toc = (guint8 *)g_malloc(len * sizeof(guint8));
		if(header->toc != NULL)
		{
			const gint entry_size = header->toc_entry_size;
			gint i = 0;
			const guint8	*toc_ptr = buf_ptr,
					*toc_end = toc_ptr + len;
			while(toc_ptr < toc_end)
			{
				switch(entry_size)
				{
				    case 1:
					header->toc[i] = *toc_ptr;
					break;
				    case 2:
					((guint16 *)header->toc)[i] = edv_mpeg_audio_swap_i16(*(guint16 *)toc_ptr);
					break;
				    case 4:
					((guint32 *)header->toc)[i] = edv_mpeg_audio_swap_i32(*(guint32 *)toc_ptr);
					break;
				}
				i++;
				toc_ptr += entry_size;
			}
		}

		buf_ptr += len;
	}
	if(header->toc == NULL)
	{
		header->toc_nentries = 0;
		header->toc_entry_size = 0;
		header->toc_len = 0;
	}

	return(header);
}

/*
 *	Searches for any Variable Bit Rate (VBR) headers in the buffer
 *	and parses the first one encountered.
 *
 *	The buf specifies the compressed data buffer positioned at
 *	the start of the MPEG frame header (at the MPEG sync word).
 *
 *	Returns a new dynamically allocated EDVMPEGAudioVBRHeader
 *	describing the VBR header that was encountered or NULL on error.
 */
static EDVMPEGAudioVBRHeader *edv_mpeg_audio_data_parse_vbr_header(
	const guint8 *buf,
        const gint buf_len
)
{
	const guint8	*buf_ptr, *buf_end;

	if((buf == NULL) || (buf_len < 1))
		return(NULL);

	/* Search through the buffer for the VBR header ID */
	for(buf_ptr = buf,
	    buf_end = buf_ptr + buf_len;
	    buf_ptr < buf_end;
	    buf_ptr++
	)
	{
		/* Xing */
		if(*buf_ptr == 'X')
		{
			if((buf_ptr + 4) > buf_end)
				continue;

			if(memcmp(
				buf_ptr + 1,
				"ing",
				3
			))
				continue;

			return(edv_mpeg_audio_data_parse_xing_header(
				buf_ptr,
				buf_end - buf_ptr
			));
		}
		/* Info */
		else if(*buf_ptr == 'I')
		{
			if((buf_ptr + 4) > buf_end)
				continue;

			if(memcmp(
				buf_ptr + 1,
				"nfo",
				3
			))
				continue;

			return(edv_mpeg_audio_data_parse_xing_header(
				buf_ptr,
				buf_end - buf_ptr
			));

			break;
		}
		/* VBRI */
		else if(*buf_ptr == 'V')
		{
			if((buf_ptr + 4) > buf_end)
				continue;

			if(memcmp(
				buf_ptr + 1,
				"BRI",
				3
			))
				continue;

			return(edv_mpeg_audio_data_parse_vbri_header(
				buf_ptr,
				buf_end - buf_ptr
			));

			break;
		}
	}

	return(NULL);
}

/*
 *	Seeks through the stream and gets the next MPEG frame header.
 *
 *	Any ID3v2 tags encountered in the stream will be skipped.
 *
 *	The fp specifies the stream.
 *
 *	Returns a dynamically allocated EDVMPEGAudioFrameHeader
 *	describing the MPEG frame header found at or past the specified
 *	stream position and the stream position will be seeked to the
 *	assumed start of the next frame or NULL if no MPEG frame header
 *	was found and the stream position will be seeked to the end of
 *	the stream.
 *
 *	If a Variable Bit Rate (VBR) header was found in the MPEG
 *	frame's compressed data then the EDVMPEGAudioFrameHeader's
 *	vbr_header will be set.
 */
EDVMPEGAudioFrameHeader *edv_mpeg_audio_stream_next_frame(FILE *fp)
{
	size_t units_read;
	guint8		v8,
			*io_buf;
	const guint8	*buf_ptr, *buf_end;
	gulong		io_buf_len,
			fp_pos,
			fp_buf_pos;
	EDVMPEGAudioFrameHeader *header = NULL;
	EDVVFSObject *obj;

	if(fp == NULL)
	{
		errno = EINVAL;
		return(header);
	}

	/* Get the stream's statistics */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len < 64l)
		io_buf_len = 64l;

	/* Allocate the read buffer */
	io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
	if(io_buf == NULL)
		return(header);

	while(!feof(fp) && (header == NULL))
	{
#define READ_BLOCK(_units_to_read_)	{		\
 const size_t units_to_read = (size_t)(_units_to_read_) / sizeof(guint8); \
 if(units_to_read > 0l)					\
  units_read = fread(					\
   io_buf,						\
   sizeof(guint8),					\
   units_to_read,					\
   fp							\
  );							\
 else							\
  units_read = 0l;					\
}

		fp_buf_pos = fp_pos = (gulong)ftell(fp);
		READ_BLOCK(io_buf_len);
		fp_pos += (gulong)(units_read * sizeof(guint8));
		if((units_read <= 0l) || ferror(fp))
			break;

		/* Iterate through the read block, searching for
		 * the next MPEG sync word and skipping any
		 * ID3v2 tags
		 */
		for(buf_ptr = io_buf,
		    buf_end = buf_ptr + (units_read * sizeof(guint8));
		    buf_ptr < buf_end;
		    buf_ptr++
		)
		{
			/* Possible start of an ID3v2 tag? */
			if(*buf_ptr == 0x49)
			{
				const gulong	required_buf_len = 3l,
						fp_tag_pos = fp_buf_pos + (gulong)(
					buf_ptr - io_buf
				);
				gulong	position,
					length;

				/* Need to read more data in order to
				 * make determination?
				 */
				if((buf_ptr + required_buf_len) > buf_end)
				{
					if(fseek(
						fp,
						(long)fp_tag_pos,
						SEEK_SET
					) != 0)
						break;

					fp_buf_pos = fp_pos = fp_tag_pos;
					READ_BLOCK(io_buf_len);
					fp_pos += (gulong)(units_read * sizeof(guint8));
					if((units_read <= 0l) || ferror(fp))
						break;

					buf_ptr = io_buf;
					buf_end = buf_ptr + (units_read * sizeof(guint8));
					if((buf_ptr + required_buf_len) > buf_end)
						continue;
				}

				/* Check the rest of the data for
				 * the ID3v2 tag header
				 */
				if(buf_ptr[1] != 0x44)
					continue;

				if(buf_ptr[2] != 0x33)
					continue;

				/* Seek the stream to the start of this
				 * ID3v2 tag for
				 * edv_id3_stream_locate_id3v2()
				 */
				if(fseek(
					fp,
					(long)fp_tag_pos,
					SEEK_SET
				) != 0)
					break;

				/* Get the position and length of this
				 * ID3v2 tag
				 */
				if(edv_id3_stream_locate_id3v2(
					fp,
					NULL,
					NULL,
					&position,
					&length
				))
				{
					gulong fp_new_pos = position + length;
					if(fp_new_pos <= fp_tag_pos)
						fp_new_pos = fp_tag_pos + 1l;

					/* Seek past this ID3v2 tag */
					if(fseek(
						fp,
						(long)fp_new_pos,
						SEEK_SET
					) != 0)
						break;

					/* Trigger the reading of the
					 * next block at the top of
					 * this loop
					 */
					buf_ptr = buf_end - 1;
				}
				else
				{
					/* Misidentified ID3v2 tag,
					 * restore the stream position
					 */
					(void)fseek(
						fp,
						(long)fp_pos,
						SEEK_SET
					);
				}
			}
			/* Start of an MPEG sync word? */
			else if(*buf_ptr == 0xFF)
			{
				const gulong	required_buf_len = 4l,
						fp_frame_pos = fp_buf_pos + (gulong)(
					buf_ptr - io_buf
				);

				/* Need to read more data in order to
				 * make determination?
				 */
				if((buf_end - buf_ptr) < required_buf_len)
				{
					if(fseek(
						fp,
						(long)fp_frame_pos,
						SEEK_SET
					) != 0)
						break;

					fp_buf_pos = fp_pos = fp_frame_pos;
					READ_BLOCK(io_buf_len);
					fp_pos += (gulong)(units_read * sizeof(guint8));
					if((units_read <= 0l) || ferror(fp))
						break;

					buf_ptr = io_buf;
					buf_end = buf_ptr + (units_read * sizeof(guint8));
					if((buf_end - buf_ptr) < required_buf_len)
						continue;
				}

				/* Get the next byte */
				v8 = buf_ptr[1];

				/* Check if the rest of the sync bits
				 * are present
				 */
				if(!(v8 & (1 << 7)) ||
				   !(v8 & (1 << 6)) ||
				   !(v8 & (1 << 5))
				)
					continue;

				/* Create a new EDVMPEGAudioFrameHeader,
				 * marking that the next MPEG sync
				 * word was found, and parse the data
				 */
				header = edv_mpeg_audio_data_parse_frame_header(
					buf_ptr,
					buf_end - buf_ptr
				);
				if(header != NULL)
				{
                                        /* Calculate the length of this
                                         * frame
                                         */
                                        gulong frame_len = edv_mpeg_audio_frame_length(header);
                                        if(frame_len < required_buf_len)
                                                frame_len = required_buf_len;

					/* Need to read more data in order
					 * to make determination?
					 */
					if((buf_ptr + frame_len) > buf_end)
					{
						if(fseek(
							fp,
							(long)fp_frame_pos,
							SEEK_SET
						) != 0)
							break;

						/* Adjust the read buffer's
						 * allocation to accomidate
						 * this frame
						 */
						if(io_buf_len < frame_len)
						{
							io_buf_len = frame_len;
							io_buf = (guint8 *)g_realloc(
								io_buf,
								io_buf_len * sizeof(guint8)
							);
							if(io_buf == NULL)
								io_buf_len = 0l;
						}

						fp_buf_pos = fp_pos = fp_frame_pos;
						READ_BLOCK(io_buf_len);
						fp_pos += (gulong)(units_read * sizeof(guint8));
						if((units_read <= 0l) || ferror(fp))
							break;

						buf_ptr = io_buf;
						buf_end = buf_ptr + (units_read * sizeof(guint8));
						if((buf_ptr + frame_len) > buf_end)
							break;
					}

					/* Search for and parse the VBR header	*/
					header->vbr_header = edv_mpeg_audio_data_parse_vbr_header(
						buf_ptr,
	                                        buf_end - buf_ptr
					);

					/* Seek the stream to the start
					 * of the next MPEG frame for
					 * the caller
					 */
					fp_pos = fp_frame_pos + frame_len;
					if(fseek(
						fp,
						(long)fp_pos,
						SEEK_SET
					) != 0)
						break;
				}

				break;
			}
		}
#undef READ_BLOCK
	}

	/* Delete the read buffer */
	g_free(io_buf);

	return(header);
}

/*
 *	Scans through the stream and calculates the total length of
 *	uncompressed MPEG data.
 *
 *	Any ID3v2 tags encountered in the stream will be skipped.
 *
 *	The fp specifies the stream.
 *
 *	Returns the total length of MPEG data found in the stream in
 *	bytes.
 */
gulong edv_mpeg_audio_scan_stream_length(
	FILE *fp,
	gulong *nframes_rtn,
	gulong *ms_rtn,
	gint *bitrate_min_rtn,
	gint *bitrate_max_rtn
)
{
	size_t units_read;
	guint8		v8,
			*io_buf;
	const guint8	*buf_ptr, *buf_end;
	gulong		io_buf_len,
			fp_pos,
			fp_buf_pos,
			fp_last_frame_pos,
			mpeg_data_length = 0l,
			nframes,
			ms,
			bitrate_min, bitrate_max;
	EDVVFSObject *obj;
	EDVMPEGAudioFrameHeader *header;

	if(nframes_rtn != NULL)
		*nframes_rtn = 0l;
	if(ms_rtn != NULL)
		*ms_rtn = 0l;
	if(bitrate_min_rtn != NULL)
		*bitrate_min_rtn = 0;
	if(bitrate_max_rtn != NULL)
		*bitrate_max_rtn = 0;

	if(fp == NULL)
	{
		errno = EINVAL;
		return(mpeg_data_length);
	}

	/* Get the stream's statistics */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len < 64l)
		io_buf_len = 64l;

	/* Allocate the read buffer */
	io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
	if(io_buf == NULL)
		return(mpeg_data_length);

	nframes = 0l;
	ms = 0l;
	bitrate_min = 0;
	bitrate_max = 0;

	fp_pos = 0l;
	fp_last_frame_pos = 0l;
	header = NULL;
	while(!feof(fp))
	{
#define READ_BLOCK(_units_to_read_)	{		\
 const size_t units_to_read = (size_t)(_units_to_read_) / sizeof(guint8); \
 if(units_to_read > 0l)					\
  units_read = fread(					\
   io_buf,						\
   sizeof(guint8),					\
   units_to_read,					\
   fp							\
  );							\
 else							\
  units_read = 0l;					\
}
		/* Read the next block */
		fp_buf_pos = fp_pos = (gulong)ftell(fp);
		READ_BLOCK(io_buf_len);
		fp_pos += (gulong)(units_read * sizeof(guint8));
		if((units_read <= 0l) || ferror(fp))
			break;

		/* Iterate through the read block, searching for
		 * the next MPEG sync word and skipping any
		 * ID3v2 tags
		 */
		for(buf_ptr = io_buf,
		    buf_end = buf_ptr + (units_read * sizeof(guint8));
		    buf_ptr < buf_end;
		    buf_ptr++
		)
		{
			/* Possible start of an ID3v2 tag? */
			if(*buf_ptr == 0x49)
			{
				gulong	position,
					length;
				const gulong	required_buf_len = 3l,
						fp_tag_pos = fp_buf_pos + (gulong)(
                                        buf_ptr - io_buf
                                );

				/* Need to read more data in order to
				 * make determination?
				 */
				if((buf_ptr + required_buf_len) > buf_end)
				{
					if(fseek(
						fp,
						(long)fp_tag_pos,
						SEEK_SET
					) != 0)
						break;

					fp_buf_pos = fp_pos = fp_tag_pos;
					READ_BLOCK(io_buf_len);
					fp_pos += (gulong)(units_read * sizeof(guint8));
					if((units_read <= 0l) || ferror(fp))
						break;

					buf_ptr = io_buf;
					buf_end = buf_ptr + (units_read * sizeof(guint8));
					if((buf_end - buf_ptr) < required_buf_len)
						continue;
				}

				/* Check the rest of the data for
				 * the ID3v2 tag header
				 */
				if(buf_ptr[1] != 0x44)
					continue;

				if(buf_ptr[2] != 0x33)
					continue;

				/* Seek the stream to the start of this
				 * ID3v2 tag for
				 * edv_id3_stream_locate_id3v2()
				 */
				if(fseek(
					fp,
					(long)fp_tag_pos,
					SEEK_SET
				) != 0)
					break;

				/* Get the position and length of this
				 * ID3v2 tag
				 */
				if(edv_id3_stream_locate_id3v2(
					fp,
					NULL,
					NULL,
					&position,
					&length
				))
				{
					gulong fp_new_pos = position + length;
					if(fp_new_pos <= fp_tag_pos)
						fp_new_pos = fp_tag_pos + 1l;

					/* Seek past this ID3v2 tag */
					if(fseek(
						fp,
						(long)fp_new_pos,
						SEEK_SET
					) != 0)
						break;

					/* Was there a previous MPEG frame? */
					if((header != NULL) &&
					   (fp_tag_pos > fp_last_frame_pos)
					)
					{
						const gulong len = (gulong)(
							fp_tag_pos - fp_last_frame_pos
						);

						/* Add to the total */
						mpeg_data_length += len;
						ms += edv_mpeg_audio_frame_calculate_length_ms(
							header,
							len
						);

						/* Reset the frame context */
						fp_last_frame_pos = 0l;
						edv_mpeg_audio_frame_header_delete(header);
						header = NULL;
					}

					/* Trigger the reading of the
					 * next block at the top of
					 * this loop
					 */
					buf_ptr = buf_end - 1;
				}
				else
				{
					/* Misidentified ID3v2 tag,
					 * restore the stream position
					 */
					(void)fseek(
						fp,
						(long)fp_pos,
						SEEK_SET
					);
				}
			}
			/* Start of an MPEG sync word? */
			else if(*buf_ptr == 0xFF)
			{
				const gulong	required_buf_len = 4l,
						fp_frame_pos = fp_buf_pos + (gulong)(
					buf_ptr - io_buf
				);

				/* Need to read more data in order to
				 * make determination?
				 */
				if((buf_ptr + required_buf_len) > buf_end)
				{
					if(fseek(
						fp,
						(long)fp_frame_pos,
						SEEK_SET
					) != 0)
						break;

					fp_buf_pos = fp_pos = fp_frame_pos;
					READ_BLOCK(io_buf_len);
					fp_pos += (gulong)(units_read * sizeof(guint8));
					if((units_read <= 0l) || ferror(fp))
						break;

					buf_ptr = io_buf;
					buf_end = buf_ptr + (units_read * sizeof(guint8));
					if((buf_end - buf_ptr) < required_buf_len)
						continue;
				}

				/* Get the next byte */
				v8 = buf_ptr[1];

				/* Check if the rest of the sync bits
				 * are present
				 */
				if(!(v8 & (1 << 7)) ||
				   !(v8 & (1 << 6)) ||
				   !(v8 & (1 << 5))
				)
					continue;


				/* Was there a previous MPEG frame? */
				if((header != NULL) &&
				   (fp_frame_pos > fp_last_frame_pos)
				)
				{
					const gulong len = (gulong)(
						fp_frame_pos - fp_last_frame_pos
					);

					/* Add to the total */
					mpeg_data_length += len;
					ms += edv_mpeg_audio_frame_calculate_length_ms(
						header,
						len
					);
				}

				/* Record the new frame's position */
				fp_last_frame_pos = fp_frame_pos;

				/* Parse the MPEG frame header data */
				edv_mpeg_audio_frame_header_delete(header);
				header = edv_mpeg_audio_data_parse_frame_header(
					buf_ptr,
					buf_end - buf_ptr
				);
				if(header != NULL)
				{
					/* Calculate the length of this
					 * frame
					 */
					gulong len = edv_mpeg_audio_frame_length(header);
					if(len < required_buf_len)
						len = required_buf_len;

					/* Cound this frame */
					nframes++;

					/* Update the bitrate ranges */
					if((bitrate_min == 0) ||
					   (bitrate_min > header->bitrate_kbps)
					)
						bitrate_min = header->bitrate_kbps;
					if((bitrate_max == 0) ||
					   (bitrate_max < header->bitrate_kbps)
					)
						bitrate_max = header->bitrate_kbps;

					/* Increment the buffer position
					 * past this frame
					 */
					buf_ptr += len;
					if(buf_ptr >= buf_end)
					{
						/* Not enough data in
						 * the read buffer for
						 * subsequent reads,
						 * seek to the next
						 * frame's position in
						 * the stream and read
						 * the next block
						 */
						const gulong fp_new_pos = fp_frame_pos + len;
						if(fseek(
							fp,
							(long)fp_new_pos,
							SEEK_SET
						) != 0)
							break;
					}
					else
					{
						/* To negate the
						 * buf_ptr++ at the
						 * end of this loop
						 */
						buf_ptr--;
					}
				}
				else
				{
					/* Unable to parse this MPEG
					 * frame header
					 *
					 * Seek the buffer pointer past
					 * this MPEG frame header
					 */
					buf_ptr += required_buf_len - 1;
				}
			}
		}
#undef READ_BLOCK
	}

	/* Delete the read buffer */
	g_free(io_buf);

	/* Was there a previous MPEG frame? */
	if(header != NULL)
	{
		if(fp_pos > fp_last_frame_pos)
		{
			const gulong len = (gulong)(fp_pos - fp_last_frame_pos);

			/* Add to the total */
			mpeg_data_length += len;
			ms += edv_mpeg_audio_frame_calculate_length_ms(
				header,
				len
			);
		}

		edv_mpeg_audio_frame_header_delete(header);
	}

	/* Set the return values */
	if(nframes_rtn != NULL)
		*nframes_rtn = nframes;
	if(ms_rtn != NULL)
		*ms_rtn = ms;
	if(bitrate_min_rtn != NULL)
		*bitrate_min_rtn = bitrate_min;
	if(bitrate_max_rtn != NULL)
		*bitrate_max_rtn = bitrate_max;

	return(mpeg_data_length);
}
