#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "pulist.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_fs_type.h"
#include "edv_pixmap.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_mount_bar.h"
#include "edv_device_mount.h"
#include "edv_dnd.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "config.h"

#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_goto_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_fsck_20x20.xpm"
#include "images/icon_tools_20x20.xpm"
#include "images/icon_floppy_20x20.xpm"
#include "images/icon_devices_list_20x20.xpm"


typedef struct _EDVMountBar		EDVMountBar;
#define EDV_MOUNT_BAR(p)		((EDVMountBar *)(p))
#define EDV_MOUNT_BAR_KEY		"EDV/MountBar"


/*
 *	Flags:
 */
typedef enum {
	EDV_MOUNT_BAR_MAPPED		= (1 << 0),
	EDV_MOUNT_BAR_REALIZED		= (1 << 1)
} EDVMountBarFlags;


/* Callbacks */
static void edv_mount_bar_destroy_cb(gpointer data);
static void edv_mount_bar_realize_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_show_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_hide_cb(GtkWidget *widget, gpointer data);
static gint edv_mount_bar_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static void edv_mount_bar_device_realize_cb(GtkWidget *widget, gpointer data);
static gint edv_mount_bar_device_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void edv_mount_bar_stats_realize_cb(GtkWidget *widget, gpointer data);
static gint edv_mount_bar_stats_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint edv_mount_bar_drag_data_received_nexus(
	EDVMountBar *bar,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	EDVDevice *dev
);
static void edv_mount_bar_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
);
static void edv_mount_bar_map_popup_list_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_mount_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_eject_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_goto_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_properties_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_fsck_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_tools_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_format_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_devices_list_cb(GtkWidget *widget, gpointer data);
static void edv_mount_bar_refresh_stats_cb(GtkWidget *widget, gpointer data);

/* Device Drawing */
static void edv_mount_bar_device_draw(EDVMountBar *bar);

/* Stats Drawing & Setting */
static void edv_mount_bar_stats_draw(EDVMountBar *bar);
static void edv_mount_bar_set_stats(EDVMountBar *bar);

/* Mount Bar */
static EDVMountBar *edv_mount_bar_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *edv_mount_bar_new(
	EDVCore *core,
	void (*mount_cb)(
		GtkWidget *,			/* Mount Bar */
		const gint,			/* Device index */
		EDVDevice *,			/* Device */
		gpointer			/* mount_data */
	),
	gpointer mount_data,
	void (*eject_cb)(
		GtkWidget *,			/* Mount Bar */
		const gint,			/* Device index */
		EDVDevice *,			/* Device */
		gpointer			/* eject_data */
	),
	gpointer eject_data,
	void (*goto_cb)(
		GtkWidget *,			/* Mount Bar */
		const gint,			/* Device index */
		EDVDevice *,			/* Device */
		gpointer			/* goto_data */
	),
	gpointer goto_data,
	void (*status_message_cb)(
		GtkWidget *,			/* Mount Bar */
		const gchar *,			/* Message */
		gpointer			/* status_message_data */
	),
	gpointer status_message_data
);
void edv_mount_bar_update_display(GtkWidget *w);
gint edv_mount_bar_current_device(GtkWidget *w);
void edv_mount_bar_select(
	GtkWidget *w,
	const gint dev_num
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Mount Bar:
 */
struct _EDVMountBar {

	GtkWidget	*toplevel;		/* GtkBox */
	gint		freeze_count;
	EDVCore		*core;
	EDVMountBarFlags	flags;
	GdkGC		*dev_gc,
			*stats_gc;

	GtkWidget	*dev_da,		/* Selected device GtkDrawingArea */
			*map_btn,		/* Map devices Popup List GtkButton */
			*mount_btn,
			*eject_btn,
			*refresh_btn,
			*goto_btn,
			*properties_btn,
			*fsck_btn,
			*tools_btn,
			*format_btn,
			*devices_list_btn,
			*stats_da;

	/* Selected device index or -1 for none */
	gint		selected_dev_num;

	/* Selected device values */
	EDVPixmap	*selected_device_icon;
	gchar		*selected_device_text;

	/* Selected Device Statistics Values */
	gint		dev_stats_page;
	gfloat		dev_stats_used_coeff;
	gchar		*device_stats_label;

	/* Mount/Unmount callback */
	void	(*mount_cb)(
		GtkWidget *,			/* Mount Bar */
		const gint,			/* Device index */
		EDVDevice *,			/* Device */
		gpointer			/* Data */
	);
	gpointer	mount_data;

	/* Eject callback */
	void	(*eject_cb)(
		GtkWidget *,			/* Mount Bar */
		const gint,			/* Device index */
		EDVDevice *,			/* Device */
		gpointer			/* Data */
	);
	gpointer	eject_data;

	/* Goto mount path callback */
	void	(*goto_cb)(
		GtkWidget *,			/* Mount Bar */
		const gint,			/* Device index */
		EDVDevice *,			/* Device */
		gpointer			/* Data */
	);
	gpointer	goto_data;

	/* Status message callback */
	void	(*status_message_cb)(
		GtkWidget *,			/* Mount Bar */
		const gchar *,			/* Message */
		gpointer			/* Data */
	);
	gpointer	status_message_data;

};


/*
 *	Toplevel GtkObject data "destroy" signal callback.
 */
static void edv_mount_bar_destroy_cb(gpointer data)
{
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	(void)GDK_GC_UNREF(bar->dev_gc);
	(void)GDK_GC_UNREF(bar->stats_gc);
	(void)edv_pixmap_unref(bar->selected_device_icon);
	g_free(bar->selected_device_text);
	g_free(bar->device_stats_label);

	bar->freeze_count--;

	g_free(bar);
}

/*
 *	Toplevel GtkWidget "realize" signal callback.
 */
static void edv_mount_bar_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	if(bar->flags & EDV_MOUNT_BAR_REALIZED)
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	window = widget->window;
	if(window == NULL)
	{

	}

	/* Mark the EDVMountBar as realized */
	bar->flags |= EDV_MOUNT_BAR_REALIZED;

	bar->freeze_count--;
}

/*
 *	Toplevel GtkWidget "show" signal callback.
 */
static void edv_mount_bar_show_cb(GtkWidget *widget, gpointer data)
{
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	/* Mark the EDVMountBar as mapped */
	bar->flags |= EDV_MOUNT_BAR_MAPPED;
}

/*
 *	Toplevel GtkWidget "hide" signal callback.
 */
static void edv_mount_bar_hide_cb(GtkWidget *widget, gpointer data)
{
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	/* Mark the EDVMountBar as not mapped */
	bar->flags &= ~EDV_MOUNT_BAR_MAPPED;
}


/*
 *	Any GtkWidget "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
static gint edv_mount_bar_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	const gchar *msg = NULL;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (crossing == NULL) || (bar == NULL))
		return(status);

	/* Handle this signal even if currently frozen */

	if(crossing->type == GDK_ENTER_NOTIFY)
	{
		if(bar->mount_btn == widget)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Mount/unmount dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Monter/Dmonter un composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Mount/unmount vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Mount/unmount congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Mount/unmount apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mount/unmount artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Mount/unmount innretning"
#else
"Mount/unmount the device"
#endif
			;
		else if(bar->eject_btn == widget)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Expulse medios del dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ejecter le support du composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Werfen sie medien von vorrichtung aus"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Espellere la stampa dal congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stoot media van apparaat uit"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Expulse imprensa de artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kast ut medier fra innretning"
#else
"Eject the media from the device"
#endif
			;
		else if(bar->refresh_btn == widget)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Refresque toda estadstica de dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rafrachir toutes les statistique composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Erfrischen sie alle gertestatistik"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Rinfrescare tutto la statistica di congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verfris alle apparaat statistieken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Refresque-se toda estatstica de artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forfrisk all innretnings statistikk"
#else
"Refresh all device statistics"
#endif
			;
		else if(bar->goto_btn == widget)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya a montar sendero"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller au rpertoire mont"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen sie, um pfad aufzustellen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare montare il sentiero"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga om pad te bestijgen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V montar caminho"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra montere sti"
#else
"Go to the mount path"
#endif
			;
		else if(bar->properties_btn == widget)
			msg =
"Modify the properties of the mount path"
			;
		else if(bar->fsck_btn == widget)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique el sistema del archivo de dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier le systme de fichier du composan"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie das dateisystem der vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare il sistema di file del congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer het dossier van het apparaat systeem"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique o sistema de arquivo do artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk innretningens arkivsystem"
#else
"Check the device's file system"
#endif
			;
		else if(bar->devices_list_btn == widget)
			msg = "Devices list";
		else if(bar->tools_btn == widget)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Corra el programa de instrumentos de dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Lancer le panneau d'outil du composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laufen sie die werkzeuge der vorrichtung programmieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha correto il programma di attrezzi del congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Loop de werktuigen van het apparaat programma"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corra o programa de ferramentas do artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kjr innretningens redskapprogram"
#else
"Run the device's tools program"
#endif
			;
		else if(bar->format_btn == widget)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Formatear los medios en el dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Formatter le support dans le composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Formatieren sie die medien in der vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il formato la stampa nel congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Formatteer de media in het apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O formato a imprensa no artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Formater mediene i innretningen"
#else
"Format the media in the device"
#endif
			;
		else if(bar->stats_da == widget)
		{ /* Ignore */ }
	}

	if(bar->status_message_cb != NULL)
		bar->status_message_cb(
			bar->toplevel,			/* Mount Bar */
			msg,				/* Message */
			bar->status_message_data	/* Data */
		);

	return(status);
}

/*
 *	Device GtkDrawingArea "realize" signal callback.
 */
static void edv_mount_bar_device_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	window = widget->window;
	if(window != NULL)
	{
		/* Create the device GdkGC */
		if(bar->dev_gc == NULL)
			bar->dev_gc = gdk_gc_new(window);
	}

	bar->freeze_count--;
}

/*
 *	Device GtkDrawingArea any event signal callback.
 */
static gint edv_mount_bar_device_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	guint keyval, state;
	gboolean press;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (event == NULL) || (bar == NULL))
		return(status);

	/* Handle this signal even if currently frozen */

	core = bar->core;

	switch((gint)event->type)
	{
	  case GDK_EXPOSE:
		edv_mount_bar_device_draw(bar);
		status = TRUE;
		break;

	  case GDK_FOCUS_CHANGE:
		focus = (GdkEventFocus *)event;
		if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
		{
			GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
			status = TRUE;
		}
		else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
		{   
			GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
			status = TRUE;
		}
		break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
		key = (GdkEventKey *)event;
		press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
 	    keyval = key->keyval;
		state = key->state;
#define STOP_KEY_SIGNAL_EMIT(_w_) {		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}
		/* Handle by key value */
		switch(keyval)
		{
		  case GDK_Return:
		  case GDK_KP_Enter:
		  case GDK_ISO_Enter:
		  case GDK_3270_Enter:
		  case GDK_space:
		  case GDK_KP_Space:
			if(press)
			{
				edv_mount_bar_mount_cb(NULL, bar);
			}
			STOP_KEY_SIGNAL_EMIT(widget);
			status = TRUE;
			break;

		  case GDK_Up:
		  case GDK_KP_Up:
		  case GDK_Page_Up:
		  case GDK_KP_Page_Up:
			if(press)
			{
				gint i = bar->selected_dev_num;
				EDVDevice *dev;

				/* Current selection in bounds? */
				if(i > 0)
				{
					/* Check the previous device and see if it is
					 * selectable, iterate on to previous device
					 * if the current device is not selectable
					 */
					GList *glist;

					i--;

					for(glist = g_list_nth(
							core->devices_list,
							(guint)i
						);
						glist != NULL;
						glist = g_list_previous(glist), i--
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				else
				{
					/* Current selection is out of
					 * bounds, so select the first
					 * valid device
					 */
					GList *glist;

					for(glist = core->devices_list, i = 0;
						glist != NULL;
						glist = g_list_next(glist), i++
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				edv_mount_bar_update_display(bar->toplevel);
			}
			STOP_KEY_SIGNAL_EMIT(widget);
			status = TRUE;
			break;

		  case GDK_Down:
		  case GDK_KP_Down:
		  case GDK_Page_Down:
		  case GDK_KP_Page_Down:
			if(press)
			{
				gint i = bar->selected_dev_num;
				EDVDevice *dev = NULL;

				/* Current selection in bounds? */
				if(i >= 0)
				{
					/* Check the next device and see if it is selectable.
					 * Iterate on to next device if the current is
					 * not selectable.
					 */
					GList *glist;

					i++;

					for(glist = g_list_nth(
							core->devices_list,
							(guint)i
						);
						glist != NULL;
						glist = g_list_next(glist), i++
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				else
				{
					/* Current selection is out of bounds, so select
					 * the first valid device.
					 */
					GList *glist;

					for(glist = core->devices_list, i = 0;
						glist != NULL;
						glist = g_list_next(glist), i++
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				edv_mount_bar_update_display(bar->toplevel);
			}
			STOP_KEY_SIGNAL_EMIT(widget);
			status = TRUE;
			break;

		  case GDK_Home:
			if(press)
			{
				gint i;
				GList *glist;
				EDVDevice *dev;

				/* Look for the first device that is selectable */
				for(glist = core->devices_list, i = 0;
					glist != NULL;
					glist = g_list_next(glist), i++
				)
				{
					dev = EDV_DEVICE(glist->data);
					if(dev == NULL)
						continue;

					if(EDV_DEVICE_IS_UNLISTED(dev))
						continue;

					bar->selected_dev_num = i;
					edv_mount_bar_update_display(bar->toplevel);
					break;
				}
			}
			STOP_KEY_SIGNAL_EMIT(widget);
			status = TRUE;
			break;

		  case GDK_End:
			if(press)
			{
				gint i;
				GList *glist;
				EDVDevice *dev;

				/* Look for the last device that is selectable */
				for(glist = g_list_last(core->devices_list),
					i = g_list_length(core->devices_list) - 1;
				    glist != NULL;
				    glist = g_list_previous(glist), i--
				)
				{
					dev = EDV_DEVICE(glist->data);
					if(dev == NULL)
						continue;

					if(EDV_DEVICE_IS_UNLISTED(dev))
						continue;

					bar->selected_dev_num = i;
					edv_mount_bar_update_display(bar->toplevel);
					break;
				}
			}
			STOP_KEY_SIGNAL_EMIT(widget);
			status = TRUE;
			break;
		}
#undef STOP_KEY_SIGNAL_EMIT
		break;

	  case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		if(!GTK_WIDGET_HAS_FOCUS(widget))
			gtk_widget_grab_focus(widget);
		switch(button->button)
		{
		  case GDK_BUTTON1:
			edv_mount_bar_map_popup_list_cb(bar->map_btn, bar);
			status = TRUE;
			break;

		  case GDK_BUTTON4:
			if(core->devices_list != NULL)
			{
				gint i = bar->selected_dev_num;
				EDVDevice *dev;

				/* Current selection in bounds? */
				if(i > 0)
				{
					/* Check the previous device and see if it is
					 * selectable, iterate on to previous device
					 * if the current device is not selectable
					 */
					GList *glist;

					i--;

					for(glist = g_list_nth(
							core->devices_list,
							(guint)i
						);
						glist != NULL;
						glist = g_list_previous(glist), i--
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				else
				{
					/* Current selection is out of bounds, so select
					 * the first valid device
					 */
					GList *glist;

					for(glist = core->devices_list, i = 0;
						glist != NULL;
						glist = g_list_next(glist), i++
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				edv_mount_bar_update_display(bar->toplevel);
			}
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			if(core->devices_list != NULL)
			{
				gint i = bar->selected_dev_num;
				EDVDevice *dev = NULL;

				/* Current selection in bounds? */
				if(i >= 0)
				{
					/* Check the next device and see if it is selectable.
					 * Iterate on to next device if the current is
					 * not selectable.
					 */
					GList *glist;

					i++;

					for(glist = g_list_nth(
							core->devices_list,
							(guint)i
						);
						glist != NULL;
						glist = g_list_next(glist), i++
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				else
				{
					/* Current selection is out of bounds, so select
					 * the first valid device.
					 */
					GList *glist;

					for(glist = core->devices_list, i = 0;
						glist != NULL;
						glist = g_list_next(glist), i++
					)
					{
						dev = EDV_DEVICE(glist->data);
						if(dev == NULL)
							continue;

						if(EDV_DEVICE_IS_UNLISTED(dev))
							continue;

						bar->selected_dev_num = i;
						break;
					}
				}
				edv_mount_bar_update_display(bar->toplevel);
			}
			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		break;
	}

	return(status);
}

/*
 *	Stats GtkDrawingArea "realize" signal callback.
 */
static void edv_mount_bar_stats_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	window = widget->window;
	if(window != NULL)
	{
		/* Create the stats GdkGC */
		if(bar->stats_gc == NULL)
			bar->stats_gc = gdk_gc_new(window);
	}

	bar->freeze_count--;
}

/*
 *	Stats GtkDrawingArea event signal callback.
 */
static gint edv_mount_bar_stats_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventFocus *focus;
	GdkEventButton *button;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (event == NULL) || (bar == NULL))
		return(status);

	core = bar->core;

	switch((gint)event->type)
	{
	  case GDK_EXPOSE:
		edv_mount_bar_stats_draw(bar);
		status = TRUE;
		break;

	  case GDK_FOCUS_CHANGE:
		focus = (GdkEventFocus *)event;
		if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
		{
			GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
			status = TRUE;
		}
		else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
		{   
			GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
			gtk_widget_queue_draw(widget);
			status = TRUE;
		}
		break;

	  case GDK_BUTTON_PRESS:
		button = (GdkEventButton *)event;
		if(!GTK_WIDGET_HAS_FOCUS(widget))
			gtk_widget_grab_focus(widget);
		switch(button->button)
		{
		  case GDK_BUTTON1:
			if(button->state & GDK_SHIFT_MASK)
			{
				bar->dev_stats_page--;
				if(bar->dev_stats_page < 0)
					bar->dev_stats_page = 2;
			}
			else
			{
				bar->dev_stats_page++;
				if(bar->dev_stats_page >= 3)
					bar->dev_stats_page = 0;
			}
			edv_mount_bar_set_stats(bar);
			status = TRUE;
			break;

		  case GDK_BUTTON4:
			bar->dev_stats_page--;
			if(bar->dev_stats_page < 0)
				bar->dev_stats_page = 2;
			edv_mount_bar_set_stats(bar);
			status = TRUE;
			break;

		  case GDK_BUTTON5:
			bar->dev_stats_page++;
			if(bar->dev_stats_page >= 3)
				bar->dev_stats_page = 0;
			edv_mount_bar_set_stats(bar);
			status = TRUE;
			break;
		}
		break;

	  case GDK_BUTTON_RELEASE:
		break;
	}

	return(status);
}

/*
 *	Drag data received nextus.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *	The tar_obj specifies the target location. If tar_obj is NULL
 *	then the current location will be used as the target location.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_mount_bar_drag_data_received_nexus(
	EDVMountBar *bar,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	EDVDevice *dev
)
{
	gint status;
	gchar *tar_path;
	EDVCore *core = bar->core;

	if((dev == NULL) && (bar->selected_dev_num > -1))
		dev = EDV_DEVICE(g_list_nth_data(
			core->devices_list,
			bar->selected_dev_num
		));
	if(dev == NULL)
		return(-2);

	tar_path = STRDUP(dev->mount_path);

	/* Process a DND operation to the VFS */
	status = edv_dnd_any_to_vfs(
		core,
		dc,
		info,				/* Source location type */
		selection_data,			/* Source objects */
		tar_path,			/* Target location */
		TRUE,				/* Verbose */
		gtk_widget_get_toplevel(bar->toplevel),
		NULL				/* No status bar */
	);

	g_free(tar_path);

	return(status);
}

/*
 *	Any GtkWidget "drag_data_received" signal callback.
 */
static void edv_mount_bar_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info,
	guint t,
	gpointer data
)
{
	GtkWidget *toplevel;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if((widget == NULL) || (dc == NULL) || (bar == NULL))
		return;

	if(bar->freeze_count > 0)
		return;

	if(selection_data == NULL)
		return;
	if(selection_data->length <= 0)
		return;

	toplevel = gtk_widget_get_toplevel(bar->toplevel);
	core = bar->core;

	/* Perform the drag operation */
	(void)edv_mount_bar_drag_data_received_nexus(
		bar,
		dc,
		info,
		selection_data,
		NULL				/* Use the current device */
	);
}

/*
 *	Map Devices Popup List callback.
 */
static void edv_mount_bar_map_popup_list_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num, nitems, nitems_visible;
	const gchar *value;
	GtkWidget *pulist;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	core = bar->core;
	pulist = core->devices_pulist;
	if(pulist == NULL)
		return;

	if(PUListIsQuery(pulist))
		return;

	nitems = PUListGetTotalItems(pulist);
	nitems_visible = MIN(10, nitems);

	/* Block input and get value */
	value = PUListMapQuery(
		pulist,
		bar->selected_device_text,
		nitems_visible,
		PULIST_RELATIVE_BELOW,
		bar->dev_da,			/* Relative widget */
		widget				/* Map widget */
	);

	/* User canceled? */
	if(value == NULL)
		return;

	gtk_widget_grab_focus(bar->dev_da);

	/* Get the Device from the selected value */
	dev_num = (gint)PUListGetDataFromValue(pulist, value);
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev != NULL)
	{
		/* Set new selected device on the mount bar structure */
		bar->selected_dev_num = dev_num;

		edv_mount_bar_update_display(bar->toplevel);
	}
}

/*
 *	Mount/Unmount callback.
 */
static void edv_mount_bar_mount_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	core = bar->core;
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev != NULL)
	{
		if(bar->mount_cb != NULL)
			bar->mount_cb(
				bar->toplevel,
				dev_num,			/* Device Number */
				dev,			/* Device */
				bar->mount_data
			);
	}
}

/*
 *	Eject callback.
 */
static void edv_mount_bar_eject_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	core = bar->core;
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev != NULL)
	{
		if(bar->eject_cb != NULL)
			bar->eject_cb(
				bar->toplevel,
				dev_num,			/* Device Number */
				dev,			/* Device */
				bar->eject_data
			);
	}
}

/*
 *	Go To Mount Path callback.
 */
static void edv_mount_bar_goto_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	core = bar->core;
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev != NULL)
	{
		if(bar->goto_cb != NULL)
			bar->goto_cb(
				bar->toplevel,
				dev_num,			/* Device Number */
				dev,			/* Device */
				bar->goto_data
			);
	}
}

/*
 *	Properties callback.
 *
 *	Maps the Properties Dialog with the selected Device's mount
 *	path.
 */
static void edv_mount_bar_properties_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	const gchar *mount_path;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	toplevel = gtk_widget_get_toplevel(bar->toplevel);
	core = bar->core;

	/* Get the selected device */
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	mount_path = dev->mount_path;
	if(STRISEMPTY(mount_path))
		return;

	/* Create a new Properties Dialog to display the object */
	edv_new_properties_dialog_vfs(
		core,
		mount_path,
		"Device",				/* Page name */
		NULL,				/* No extended properties list */
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);
}

/*
 *	FSCK callback.
 */
static void edv_mount_bar_fsck_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	toplevel = gtk_widget_get_toplevel(bar->toplevel);
	core = bar->core;
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	/* Run the device check */
	edv_run_device_check(core, dev, toplevel);
}

/*
 *	Tools callback.
 */
static void edv_mount_bar_tools_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	toplevel = gtk_widget_get_toplevel(bar->toplevel);
	core = bar->core;
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)  
		return;

	/* Run the device tools */
	edv_run_device_tools(core, dev, toplevel);
}

/*
 *	Format callback.
 */
static void edv_mount_bar_format_cb(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	GtkWidget *toplevel;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	toplevel = gtk_widget_get_toplevel(bar->toplevel);
	core = bar->core;
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev == NULL)
		return;

	/* Run the device format */
	edv_run_device_format(core, dev, toplevel);
}

/*
 *	Devices list callback.
 */
static void edv_mount_bar_devices_list_cb(GtkWidget *widget, gpointer data)
{
	EDVDevice *dev;
	GtkWidget *toplevel;
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	toplevel = gtk_widget_get_toplevel(bar->toplevel);
	core = bar->core;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)bar->selected_dev_num
	));

	edv_map_devices(
		core,
		(dev != NULL) ? dev->mount_path : NULL,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);
}


/*
 *	Refresh callback.
 *
 *	Updates the mount states and stats of all Devices and updates
 *	the selected Device's stats.
 */
static void edv_mount_bar_refresh_stats_cb(GtkWidget *widget, gpointer data)
{
	EDVCore *core;
	EDVMountBar *bar = EDV_MOUNT_BAR(data);
	if(bar == NULL)
		return;

	core = bar->core;

	/* Refresh all device mount states and device stats, then get
	 * device path or mount path that matches the given disk
	 * object's path
	 */
	edv_devices_list_update_mount_states(
		core->devices_list
	);
	edv_devices_list_update_statistics(
		core->devices_list
	);

	edv_mount_bar_update_display(bar->toplevel);
}


/*
 *	Draws the Mount Bar's device GtkDrawingArea.
 */
static void edv_mount_bar_device_draw(EDVMountBar *bar)
{
	const gint	border_minor = 2;
	gint		width, height,
			font_height,
			icon_width = 0, icon_height = 0;
	GdkFont *font;
	GdkDrawable *drawable;
	GtkWidget *w = bar->dev_da;
	GdkWindow *window = w->window;
	const gboolean has_focus = GTK_WIDGET_HAS_FOCUS(w) ||
		GTK_WIDGET_HAS_FOCUS(bar->map_btn);
	GdkGC *gc = bar->dev_gc;
	const GtkStateType state = GTK_WIDGET_STATE(w);
	GtkStyle *style = gtk_widget_get_style(w);

	if((window == NULL) || (gc == NULL) || (style == NULL))
		return;

	font = style->font;
	font_height = (font != NULL) ?
		(font->ascent + font->descent) : 0;

	drawable = (GdkDrawable *)window;
	gdk_window_get_size(
		(GdkWindow *)drawable,
		&width, &height
	);
	if((width <= 0) || (height <= 0))
		return;

	/* Draw the background */
	gdk_draw_rectangle(
		drawable,
		has_focus ?
			style->bg_gc[GTK_STATE_SELECTED] :
			style->base_gc[state],
		TRUE,				/* Fill */
		0, 0,
		width, height
	);

	/* Draw the icon? */
	if(edv_pixmap_is_loaded(bar->selected_device_icon))
	{
		gint	x = border_minor + border_minor,
			y;
		EDVPixmap *icon = bar->selected_device_icon;
		GdkBitmap *mask = icon->mask;
		GdkPixmap *pixmap = icon->pixmap;

		icon_width = icon->width;
		icon_height = icon->height;

		y = (height - icon_height) / 2;

		gdk_gc_set_clip_mask(gc, mask);
		gdk_gc_set_clip_origin(gc, x, y);
		gdk_draw_pixmap(
			drawable,
			gc,
			pixmap,
			0, 0,
			x, y,
			icon_width, icon_height
		);
		gdk_gc_set_clip_mask(gc, NULL);
	}

	/* Draw the label? */
	if(!STRISEMPTY(bar->selected_device_text) && (font != NULL))
	{
		gint	x = border_minor + border_minor + icon_width +
				border_minor,
			y;
		const gchar *s = bar->selected_device_text;
		GdkTextBounds b;

		gdk_string_bounds(font, s, &b);
		y = (height - font_height) / 2;

		gdk_draw_string(
			drawable,
			font,
			has_focus ?
				style->text_gc[GTK_STATE_SELECTED] :
				style->text_gc[state],
			x - b.lbearing,
			y + font->ascent,
			s
		);
	}

	/* Draw the focus */
	if(has_focus && GTK_WIDGET_SENSITIVE(w))
	{
		gdk_gc_set_function(gc, GDK_INVERT);
		gdk_gc_set_foreground(gc, &style->fg[state]);
		gdk_draw_rectangle(
			drawable,
			gc,
			FALSE,			/* Outline */
			0, 0,
			width - 1, height - 1
		);
		gdk_gc_set_function(gc, GDK_COPY);
	}

	/* Send the drawable to the window if it is not the window */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			(GdkDrawable *)window,
			style->white_gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}


/*
 *	Draws the Mount Bar's stats GtkDrawingArea.
 */
static void edv_mount_bar_stats_draw(EDVMountBar *bar)
{
	const gint	border_minor = 2;
	gint		x, y,
			width, height,
			font_height,
			dev_num;
	GdkFont *font;
	GdkDrawable *drawable;
	GdkGC *gc = bar->stats_gc;
	GtkWidget *w = bar->stats_da;
	GdkWindow *window = w->window;
	const GtkStateType state = GTK_WIDGET_STATE(w);
	GtkStyle *style = gtk_widget_get_style(w);
	EDVDevice *dev;
	EDVCore *core = bar->core;

	if((window == NULL) || (gc == NULL) || (style == NULL))
		return;

	font = style->font;
	font_height = (font != NULL) ? (font->ascent + font->descent) : 0;

	drawable = (GdkDrawable *)window;
	gdk_window_get_size(
		(GdkWindow *)drawable,
		&width, &height
	);
	if((width <= 0) || (height <= 0))
		return;

	/* Get the selected device */
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));

	/* Draw the background */
	gdk_draw_rectangle(
		drawable,
		style->base_gc[state],
		TRUE,				/* Fill */
		0, 0,
		width, height
	);

	/* Set the starting drawing coordinates */
	x = border_minor + border_minor;

	switch(bar->dev_stats_page)
	{
	    case 2:
		/* Device Icon */
		if(EDV_DEVICE_IS_MOUNTED(dev) &&
		   edv_pixmap_is_loaded(bar->selected_device_icon)
		)
		{
			EDVPixmap *icon = bar->selected_device_icon;
			const gint	icon_width = icon->width,
					icon_height = icon->height;
			GdkBitmap *mask = icon->mask;
			GdkPixmap *pixmap = icon->pixmap;

			y = (height - icon_height) / 2;
			gdk_gc_set_clip_mask(gc, mask);
			gdk_gc_set_clip_origin(gc, x, y);
			gdk_draw_pixmap(
				drawable,
				gc,
				pixmap,
				0, 0,
				x, y,
				icon_width, icon_height
			);
			gdk_gc_set_clip_mask(gc, NULL);
			x += icon_width + border_minor;
		}
		/* Label */
		if(!STRISEMPTY(bar->device_stats_label) && (font != NULL))
		{
			const gchar *s = bar->device_stats_label;
			GdkTextBounds b;
			gdk_string_bounds(font, s, &b);
			y = (height - font_height) / 2;
			gdk_draw_string(
				drawable, font, style->text_gc[state],
				x - b.lbearing,
				y + font->ascent,
				s
			);
			x += b.width + border_minor + border_minor;
		}
		break;
	    case 1:
		/* Label */
		if(!STRISEMPTY(bar->device_stats_label) && (font != NULL))
		{
			const gchar *s = bar->device_stats_label;
			GdkTextBounds b;
			gdk_string_bounds(font, s, &b);
			y = (height - font_height) / 2;
			gdk_draw_string(
				drawable, font, style->text_gc[state],
				x - b.lbearing,
				y + font->ascent,
				s
			);
			x += b.width + border_minor + border_minor;
		}
		break;
	    default:
		/* Used bar */
		if(EDV_DEVICE_IS_MOUNTED(dev))
		{
			const gint	bar_width = 80,
					bar_height = MAX(
						(gint)((height - (2 * border_minor)) * 0.60f),
						4
					);
			const gint bar_used_width = (bar_width - 4)
				* bar->dev_stats_used_coeff;

			y = (height - bar_height) / 2;
			if(bar_used_width > 0)
				gdk_draw_rectangle(
					drawable,
					style->bg_gc[GTK_STATE_SELECTED],
					TRUE,
					x + 2, y + 2,
					bar_used_width, bar_height - 4
				);
			gdk_draw_rectangle(
				drawable,
				style->fg_gc[state],
				FALSE,
				x, y,
				bar_width - 1, bar_height - 1
			);
			x += bar_width + border_minor + border_minor;
		}
		/* Label */
		if(!STRISEMPTY(bar->device_stats_label) && (font != NULL))
		{
			const gchar *s = bar->device_stats_label;
			GdkTextBounds b;
			gdk_string_bounds(font, s, &b);
			y = (height - font_height) / 2;
			gdk_draw_string(
				drawable, font, style->text_gc[state],
				x - b.lbearing,
				y + font->ascent,
				s
			);
			x += b.width + border_minor + border_minor;
		}
		break;
	}     


	/* Draw the frame */
	gtk_paint_shadow(
		style,
		(GdkWindow *)drawable,
		state,
		GTK_SHADOW_IN,
		NULL,				/* Entire area */
		w,
		NULL,				/* No detail */
		0, 0,
		width - 1, height - 1
	);

	/* Draw the focus */
	if(GTK_WIDGET_HAS_FOCUS(w) &&
	   GTK_WIDGET_SENSITIVE(w)
	)
		gtk_paint_focus(
			style,
			(GdkWindow *)drawable,
			NULL,			/* Entire area */
			w,
			NULL,			/* No detail */
			0, 0,
			width - 1, height - 1
		);

	/* Send the drawable to the window if it is not the window */
	if(drawable != (GdkDrawable *)window)
		gdk_draw_pixmap(
			(GdkDrawable *)window,
			style->white_gc,
			drawable,
			0, 0,
			0, 0,
			width, height
		);
}

/*
 *	Updates the Mount Bar's stats with the selected device.
 */
static void edv_mount_bar_set_stats(EDVMountBar *bar)
{
	const gint dev_num = bar->selected_dev_num;
	EDVCore *core = bar->core;
	EDVDevice *d = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));

	/* Got selected device? */
	if(d != NULL)
	{
		const gulong	used = d->blocks_total - d->blocks_free,
				total = d->blocks_total;

		/* Update stats used coefficient */
		bar->dev_stats_used_coeff = (total > 0l) ?
			CLIP(((gfloat)used / (gfloat)total), 0.0f, 1.0f) : 0.0f;

		/* Update stats label */
		if(EDV_DEVICE_IS_MOUNTED(d))
		{
			const gchar *fmt;
			gchar	*s_used = STRDUP(edv_str_size_delim(used)),
				*s_total = STRDUP(edv_str_size_delim(total)),
				*s_free_avail = STRDUP(edv_str_size_delim(d->blocks_available)),
				*s_free = STRDUP(edv_str_size_delim(d->blocks_free));

			switch(bar->dev_stats_page)
			{
			  case 2:
				g_free(bar->device_stats_label);
				bar->device_stats_label = g_strdup_printf(
					"%s   %s   %s",
					d->mount_path,
					d->fs_type_name,
					EDV_DEVICE_IS_READ_ONLY(d) ?
						"(ro)" : "(rw)"
				);
				break;
			  case 1:
				fmt = "Total: %s kb  \
Free & Available: %s kb  Free: %s kb";
				g_free(bar->device_stats_label);
				bar->device_stats_label = g_strdup_printf(
					fmt,
					s_total,
					s_free_avail,
					s_free
				);
				break;
			  default:
				fmt = "%i%% of %s kb used";
				g_free(bar->device_stats_label);
				bar->device_stats_label = g_strdup_printf(
					fmt,
					(gint)(bar->dev_stats_used_coeff * 100.0f),
					s_total
				);
				break;
			}

			g_free(s_used);
			g_free(s_total);
			g_free(s_free_avail);
			g_free(s_free);
		}
		else
		{
			const gchar *fmt;
#if defined(PROG_LANGUAGE_SPANISH)
			fmt = "El Artefacto \"%s\" No Mont";
#elif defined(PROG_LANGUAGE_FRENCH)
			fmt = "Composant \"%s\" Non Mont";
#else
			fmt = "Device \"%s\" Not Mounted";
#endif
			g_free(bar->device_stats_label);
			bar->device_stats_label = g_strdup_printf(
				fmt,
				!STRISEMPTY(d->name) ? d->name : "(Untitled)"
			);
		}
	}
	else
	{
		/* No device selected */
		bar->dev_stats_used_coeff = 0.0f;
		g_free(bar->device_stats_label);
		bar->device_stats_label = NULL;
	}

	gtk_widget_queue_draw(bar->stats_da);
}


/*
 *	Gets the EDVMountBar data from the GtkWidget.
 */
static EDVMountBar *edv_mount_bar_get_widget_data(
	GtkWidget *w,
	const gchar *func_name
)
{
	const gchar *key = EDV_MOUNT_BAR_KEY;
	EDVMountBar *bar;

	if(w == NULL)
		return(NULL);

	bar = EDV_MOUNT_BAR(gtk_object_get_data(
		GTK_OBJECT(w),
		key
	));
	if(bar == NULL)
	{
		g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
			func_name,
			w,
			key
		);
		return(NULL);
	}

	return(bar);
}

/*
 *	Creates a new EDVMountBar.
 */
GtkWidget *edv_mount_bar_new(
	EDVCore *core,
	void (*mount_cb)(
			GtkWidget *,			/* Mount Bar */
			const gint,			/* Device index */
			EDVDevice *,		/* Device */
			gpointer			/* mount_data */
	),
	gpointer mount_data,
	void (*eject_cb)(
			GtkWidget *,			/* Mount Bar */
			const gint,			/* Device index */
			EDVDevice *,		/* Device */
			gpointer			/* eject_data */
	),
	gpointer eject_data,
	void (*goto_cb)(
			GtkWidget *,			/* Mount Bar */
			const gint,			/* Device index */
			EDVDevice *,		/* Device */
			gpointer			/* goto_data */
	),
	gpointer goto_data,
	void (*status_message_cb)(
			GtkWidget *,			/* Mount Bar */
			const gchar *,			/* Message */
			gpointer			/* status_message_data */
	),
	gpointer status_message_data
)
{
	const gint border_minor = 2;
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING},
{EDV_DND_TARGET_RECYCLED_OBJECT, 0,	EDV_DND_INFO_RECYCLED_OBJECT},
{EDV_DND_TARGET_ARCHIVE_OBJECT,	0,	EDV_DND_INFO_ARCHIVE_OBJECT}
	};
	GtkWidget	*w,
					*toplevel,
					*parent, *parent2, *parent3;
	EDVMountBar *bar;

	if(core == NULL)
		return(NULL);

	bar = EDV_MOUNT_BAR(g_malloc0(sizeof(EDVMountBar)));
	if(bar == NULL)
		return(NULL);

	bar->toplevel = toplevel = gtk_hbox_new(FALSE, border_minor);
/*
	bar->freeze_count = 0;
	bar->dev_gc = NULL;
	bar->stats_gc = NULL;
 */
	bar->core = core;
/*
	bar->flags = 0;
	bar->selected_dev_num = 0;
	bar->selected_device_icon = NULL;
	bar->selected_device_text = NULL;
	bar->dev_stats_page = 0;
	bar->dev_stats_used_coeff = 0.0f;
	bar->device_stats_label = NULL;
 */
	bar->mount_cb = mount_cb;
	bar->mount_data = mount_data;
	bar->eject_cb = eject_cb;
	bar->eject_data = eject_data;
	bar->goto_cb = goto_cb;
	bar->goto_data = goto_data;
	bar->status_message_cb = status_message_cb;
	bar->status_message_data = status_message_data;

	bar->freeze_count++;

	/* Set the first device as the initially selected device,
	 * this first device is not always index 0, but rather the
	 * first one that is not marked as unlisted
	 */
	if(core != NULL)
	{
		gint i;
		GList *glist;
		EDVDevice *dev;

		for(glist = core->devices_list, i = 0;
			glist != NULL;
			glist = g_list_next(glist), i++
		)
		{
			dev = EDV_DEVICE(glist->data);
			if(dev == NULL)
				continue;

			if(EDV_DEVICE_IS_UNLISTED(dev))
				continue;

			bar->selected_dev_num = i;
			break;
		}
	}

	/* Toplevel GtkHBox */
	w = toplevel;
	gtk_widget_set_name(w, EDV_WIDGET_NAME_MOUNT_BAR);
	gtk_object_set_data_full(
		GTK_OBJECT(w), EDV_MOUNT_BAR_KEY,
		bar, edv_mount_bar_destroy_cb
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_mount_bar_realize_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "show",
		GTK_SIGNAL_FUNC(edv_mount_bar_show_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "hide",
		GTK_SIGNAL_FUNC(edv_mount_bar_hide_cb), bar
	);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	parent = w;

	/* Devices list GtkHbox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Device GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_widget_set_usize(
		w,
		180,
		20 + (2 * border_minor)
	);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Device GtkDrawingArea */
	bar->dev_da = w = gtk_drawing_area_new();
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	gtk_widget_add_events(
		w,
		GDK_EXPOSURE_MASK | GDK_STRUCTURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_realize_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_device_event_cb), bar
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		edv_mount_bar_drag_data_received_cb,
		bar,
		TRUE				/* Highlight */
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);

	/* Popup List Map GtkButton */
	bar->map_btn = w = PUListNewMapButtonArrow(
		GTK_ARROW_DOWN, GTK_SHADOW_OUT,
		edv_mount_bar_map_popup_list_cb, bar
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Mount/unmount GtkButton */
	bar->mount_btn = w = GUIButtonPixmap(
		(guint8 **)icon_mount_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_mount_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"El monte/artefacto de unmount"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Monte/dmonte le composant"
#else
		"Mount/unmount the device"
#endif
	);
	gtk_widget_show(w);

	/* Eject GtkButton */
	bar->eject_btn = w = GUIButtonPixmap(
		(guint8 **)icon_eject_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_eject_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"Expulse medios del artefacto"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Ejecter le support du composant"
#else
		"Eject the media from the device"
#endif
	);
	gtk_widget_show(w);

	/* Go To Mount Path GtkButton */
	bar->goto_btn = w = GUIButtonPixmap(
		(guint8 **)icon_goto_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_goto_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"Vaya a montar sendero"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Aller au point de montage"
#else
		"Go to the mount path"
#endif
	);
	gtk_widget_show(w);

	/* GtkSeparator */
	w = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Properties GtkButton */
	bar->properties_btn = w = GUIButtonPixmap(
		(guint8 **)icon_properties_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_properties_cb), bar
	);
	GUISetWidgetTip(
		w,
		"Modify the properties of the mount path"
	);
	gtk_widget_show(w);

	/* FSCK GtkButton */
	bar->fsck_btn = w = GUIButtonPixmap(
		(guint8 **)icon_fsck_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_fsck_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique el sistema del archivo de dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier le systme de fichier du composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie das dateisystem der vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare il sistema di file del congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer het dossier van het apparaat systeem"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique o sistema de arquivo do artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk innretningens arkivsystem"
#else
"Check the device's file system"
#endif
	);
	gtk_widget_show(w);

	/* Tools GtkButton */
	bar->tools_btn = w = GUIButtonPixmap(
		(guint8 **)icon_tools_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_tools_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Corra el programa de instrumentos de dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Lancer le panneau de contrle du composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laufen sie die werkzeuge der vorrichtung programmieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha correto il programma di attrezzi del congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Loop de werktuigen van het apparaat programma"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Corra o programa de ferramentas do artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kjr innretningens redskapprogram"
#else
"Run the device's tools program"
#endif
	);
	gtk_widget_show(w);

	/* Format GtkButton */
	bar->format_btn = w = GUIButtonPixmap(
		(guint8 **)icon_floppy_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_format_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Formatear los medios en el dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Formatte le support dans le composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Formatieren sie die medien in der vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il formato la stampa nel congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Formatteer de media in het apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O formato a imprensa no artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Formater mediene i innretningen"
#else
"Format the media in the device"
#endif
	);
	gtk_widget_show(w);

	/* GtkSeparator */
	w = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Refresh GtkButton */
	bar->refresh_btn = w = GUIButtonPixmap( 
		(guint8 **)icon_reload_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_refresh_stats_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
		"Refresque toda estadstica de artefacto"
#elif defined(PROG_LANGUAGE_FRENCH)
		"Rafrachir toute statistique composant"
#else
		"Refresh all device statistics"
#endif
	);
	gtk_widget_show(w);

	/* Stats GtkDrawingArea */
	bar->stats_da = w = gtk_drawing_area_new();
	gtk_widget_set_usize(
		w,
		-1, 20 + (2 * border_minor)
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	gtk_widget_add_events(
		w,
		GDK_EXPOSURE_MASK | GDK_STRUCTURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_mount_bar_stats_realize_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_stats_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_stats_event_cb), bar
	);
	gtk_signal_connect(  
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_stats_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_stats_event_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_stats_event_cb), bar
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		edv_mount_bar_drag_data_received_cb,
		bar,
		TRUE				/* Highlight */
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	/* Devices List GtkButton */
	bar->devices_list_btn = w = GUIButtonPixmap(
		(guint8 **)icon_devices_list_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_mount_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_mount_bar_devices_list_cb), bar
	);
	GUISetWidgetTip(
		w,
		"Devices list"
	);
	gtk_widget_show(w);


	edv_mount_bar_update_display(bar->toplevel);

	bar->freeze_count--;

	return(bar->toplevel);
}

/*
 *	Updates the Mount Bar's widgets to reflect current values.
 */
void edv_mount_bar_update_display(GtkWidget *w)
{
	gint dev_num;
	EDVDevice *dev;
	EDVCore *core;
	EDVMountBar *bar = edv_mount_bar_get_widget_data(
		w,
		"edv_mount_bar_update_display"
	);
	if(bar == NULL)
		return;

	core = bar->core;

	/* Get the selected Device */
	dev_num = bar->selected_dev_num;
	dev = EDV_DEVICE(g_list_nth_data(
		core->devices_list,
		(guint)dev_num
	));
	if(dev != NULL)
	{
		/* Update the device icon */
		const gint n = EDV_DEVICE_TOTAL_ICON_STATES;
		gint i;
		EDVPixmap *icon = NULL;
		edv_device_realize(dev, FALSE);
		for(i = 0; i < n; i++)
		{
			icon = dev->small_icon[i];
			if(edv_pixmap_is_loaded(icon))
				break;
		}
		if(edv_pixmap_is_loaded(icon))
		{
			(void)edv_pixmap_unref(bar->selected_device_icon);
			bar->selected_device_icon = edv_pixmap_ref(icon);
		}
	}

	/* Update the device label */
	if(dev != NULL)
	{
		const gchar *s = dev->name;
		if((s == NULL) && (dev->device_path != NULL))
			s = g_basename(dev->device_path);
		if(s == NULL)
			s = "(null)";

		g_free(bar->selected_device_text);
		bar->selected_device_text = STRDUP(s);
	}

	GUIButtonPixmapUpdate(
		bar->mount_btn,
		(guint8 **)(
			EDV_DEVICE_IS_MOUNTED(dev) ?
			icon_unmount_20x20_xpm : icon_mount_20x20_xpm
		),
		NULL
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->mount_btn,
		(dev != NULL) ? !EDV_DEVICE_IS_NO_UNMOUNT(dev) : FALSE
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->eject_btn,
		(dev != NULL) ? (!STRISEMPTY(dev->command_eject) &&
			!EDV_DEVICE_IS_NO_UNMOUNT(dev)) : FALSE
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->refresh_btn,
		TRUE
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->goto_btn,
		((dev != NULL) && (bar->goto_cb != NULL)) ? TRUE : FALSE
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->properties_btn,
		(dev != NULL) ? !STRISEMPTY(dev->mount_path) : FALSE
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->fsck_btn,
		(dev != NULL) ? !STRISEMPTY(dev->command_check) : FALSE
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->tools_btn,
		(dev != NULL) ? !STRISEMPTY(dev->command_tools) : FALSE
	);
	GTK_WIDGET_SET_SENSITIVE(
		bar->format_btn,
		(dev != NULL) ? !STRISEMPTY(dev->command_format) : FALSE
	);


	/* Redraw the device icon and label */
	gtk_widget_queue_draw(bar->dev_da);

	/* Update and redraw the stats label */
	edv_mount_bar_set_stats(bar);
}

/*
 *	Gets the current selected device.
 */
gint edv_mount_bar_current_device(GtkWidget *w)
{
	EDVMountBar *bar = edv_mount_bar_get_widget_data(
		w,
		"edv_mount_bar_current_device"
	);
	if(bar == NULL)
		return(-1);

	return(bar->selected_dev_num);
}

/*
 *	Selects the device.
 */
void edv_mount_bar_select(
	GtkWidget *w,
	const gint dev_num
)
{
	EDVMountBar *bar = edv_mount_bar_get_widget_data(
		w,
		"edv_mount_bar_select"
	);
	if(bar == NULL)
		return;

	if(dev_num == bar->selected_dev_num)
		return;

	bar->selected_dev_num = dev_num;

	edv_mount_bar_update_display(bar->toplevel);
}
