#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "../include/fio.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"

#include "config.h"


/* MIME Types List Matching */
EDVMIMEType *edv_mime_types_list_match_type(
	GList *mime_types_list,
	gint *n,
	const gchar *type,
	const gboolean case_sensitive
);
EDVMIMEType *edv_mime_types_list_match_path(
	GList *mime_types_list,
	const gchar *path
);

/* MIME Types List DDE Buffer */
GList *edv_mime_types_list_decode_buffer(
	GList *mime_types_list,
	const guint8 *buf,
	const gint buf_len
);
guint8 *edv_mime_types_list_encode_buffer(
	GList *mime_types_list,
	GList *selection_list,
	gint *buf_len_rtn
);

/* Commands List DDE Buffer */
GList *edv_mime_type_commands_list_decode_buffer(
	const guint8 *buf,
	const gint buf_len
);
guint8 *edv_mime_type_commands_list_encode_buffer(
	GList *commands_list,
	gint *buf_len_rtn
);

/* MIME Types List Open From System */
GList *edv_mime_types_list_file_open_system(
	GList *mime_types_list,
	const gchar *edv_data_path,
	const gint insert_index,
	gint (*progress_cb)(
		gpointer,
		const gulong, const gulong
	),
	gpointer progress_data,
	void (*added_cb)(
		const gint,
		EDVMIMEType *,
		gpointer
	),
	gpointer added_data,
	const gboolean mark_all_loaded_read_only
);

/* MIME Types List Open & Save */
GList *edv_mime_types_list_file_open(
	GList *mime_types_list,
	const gchar *path,
	const gint insert_index,
	const gboolean update,
	const gboolean only_newer,
	gint (*progress_cb)(
		gpointer,
		const gulong, const gulong
	),
	gpointer progress_data,
	void (*added_cb)(
		const gint,
		EDVMIMEType *,
		gpointer
	),
	gpointer added_data,
	void (*modified_cb)(
		const gint,
		EDVMIMEType *,
		gpointer
	),
	gpointer modified_data,
	const gboolean mark_all_loaded_read_only
);
void edv_mime_types_list_file_save(
	GList *mime_types_list,
	const gchar *path,
	const gboolean include_read_only,
	gint (*progress_cb)(
		gpointer,
		const gulong, const gulong
	),
	gpointer progress_data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Matches a MIME Type by type name.
 *
 *	The type specifies the MIME Type's type name.
 */
EDVMIMEType *edv_mime_types_list_match_type(
	GList *mime_types_list,
	gint *n,
	const gchar *type,
	const gboolean case_sensitive
)
{
	gint i;
	GList *glist;
	EDVMIMEType *m;

	if(n != NULL)
		*n = -1;

	if(type == NULL)
		return(NULL);

	for(glist = mime_types_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		if(m->type == NULL)
			continue;

		if(m->type == type)
		{
			if(n != NULL)
			   *n = i;
			return(m);
		}

		if(case_sensitive)
		{
			if(!strcmp((const char *)m->type, (const char *)type))
			{
				if(n != NULL)
				   *n = i;
				return(m);
			}
		}
		else
		{
			if(!g_strcasecmp(m->type, type))
			{
				if(n != NULL)
				   *n = i;
				return(m);
			}
		}
	}

	return(NULL);
}

/*
 *	Matches a MIME Type by path.
 *
 *	The path specifies the full path to the object.
 *
 *	The returned MIME Type will be of class
 *	EDV_MIME_TYPE_CLASS_FORMAT, EDV_MIME_TYPE_CLASS_PROGRAM, or
 *	EDV_MIME_TYPE_CLASS_UNIQUE. If no MIME Type is matched then NULL
 *	is returned.
 */
EDVMIMEType *edv_mime_types_list_match_path(
	GList *mime_types_list,
	const gchar *path
)
{
	const gchar *name;
	GList *glist;
	EDVMIMEType *m;

	if(STRISEMPTY(path))
		return(NULL);

	name = g_basename(path);

	for(glist = mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		/* Handle by the MIME Type's class */
		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
			break;
		  case EDV_MIME_TYPE_CLASS_FORMAT:
			if(edv_name_has_extension(name, m->value))
				return(m);
			break;
		  case EDV_MIME_TYPE_CLASS_PROGRAM:
			if(m->value != NULL)
			{
				if(path == m->value)
					return(m);
				else if(!strcmp((const char *)path, (const char *)m->value))
					return(m);
			}
			break;
		  case EDV_MIME_TYPE_CLASS_UNIQUE:
			if(m->value != NULL)
			{
				if(path == m->value)
					return(m);
				else if(!strcmp((const char *)path, (const char *)m->value))
					return(m);
			}
			break;
		}
	}

	return(NULL);
}


/*
 *	Creates a list of MIME Types from the DDE Buffer.
 *
 *	The mime_types_list specifies the MIME Types list.
 *
 *	The buf and buf_len specifies the DDE Buffer created by
 *	edv_mime_types_list_encode_buffer().
 *
 *	Returns a GList of EDVMIMEType * that must be
 *	deleted by the calling function. Each MIME Type in the
 *	returned list is a copy (not a reference) to a MIME Type in
 *	the MIME Types list.
 */
GList *edv_mime_types_list_decode_buffer(
	GList *mime_types_list,
	const guint8 *buf,
	const gint buf_len
)
{
	gint len;
	gchar *type;
	const guint8 *buf_ptr, *buf_end, *buf_next;
	GList *list_rtn;

	if((mime_types_list == NULL) ||
	   (buf == NULL) || (buf_len <= 0)
	)
		return(NULL);

	list_rtn = NULL;

	/* Begin parsing the DDE buffer
	 *
	 * Format of each segment is as follows:
	 *
	 *	<type>			String
	 */
	buf_ptr = buf;
	buf_end = buf_ptr + buf_len;
	while(buf_ptr < buf_end)
	{
		/* Get the type */
		buf_next = buf_ptr;
		while(buf_next < buf_end)
		{
			if(*((const gchar *)buf_next) == '\0')
				break;

			buf_next++;
		}
		len = buf_next - buf_ptr;
		if(len > 0)
		{
			type = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			if(type != NULL)
			{
				memcpy(
					type,
					buf_ptr,
					len * sizeof(gchar)
				);
				type[len] = '\0';
			}
		}
		else
		{
			type = NULL;
		}

		/* Look for a MIME Type in the list that matches this
		 * type
		 */
		if(type != NULL)
		{
			EDVMIMEType *m = edv_mime_types_list_match_type(
				mime_types_list,
				NULL,		/* No index return */
				type,
				FALSE		/* Not case sensitive */
			);
			if(m != NULL)
				list_rtn = g_list_append(
					list_rtn,
					edv_mime_type_copy(m)
				);

			g_free(type);
		}

		buf_ptr = buf_next + 1;	/* Seek past the null character */
	}

	return(list_rtn);
}

/*
 *	Creates a DDE Buffer from selected objects in the
 *	MIME Types list.
 *
 *	The mime_types_list specifies the MIME Types list.
 *
 *	The selection_list specifies a list of gints that specify
 *	the indices of which MIME Types in the list that are to
 *	be added to the DDE Buffer. If selection_list is NULL
 *	then all the MIME Types in the list will be added to the
 *	DDE Buffer.
 *
 *	The buf_len_rtn specifies the length of the DDE buffer.
 *
 *	Returns the DDE Buffer which describes a null character
 *	deliminated list of MIME Type types. This buffer must be
 *	deleted by the calling function.
 */
guint8 *edv_mime_types_list_encode_buffer(
	GList *mime_types_list,
	GList *selection_list,
	gint *buf_len_rtn
)
{
	gint buf_len;
	guint8 *buf;

	if(buf_len_rtn == NULL)
		return(NULL);

	*buf_len_rtn = 0;

	if(mime_types_list == NULL)
		return(NULL);

	buf_len = 0;
	buf = NULL;

#define APPEND_BUF(_type_)	{		\
 if((_type_) != NULL) {				\
  const gint	buf_i = buf_len,		\
		type_len = STRLEN(_type_);	\
						\
  buf_len += type_len + 1;	/* Include the null character */ \
  buf = (guint8 *)g_realloc(			\
   buf,						\
   buf_len * sizeof(guint8)			\
  );						\
  if(buf != NULL) {				\
   if(type_len > 0)				\
    (void)memcpy(				\
     buf + buf_i,				\
     (_type_),					\
     type_len					\
    );						\
   buf[buf_i + type_len] = '\0';	/* Set the null character */ \
  } else {					\
   buf_len = 0;					\
  }						\
 }						\
}

	/* Selection list specified? */
	if(selection_list != NULL)
	{
		gint i;
		GList *glist;
		EDVMIMEType *m;

		for(glist = selection_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			i = (gint)glist->data;
			if(i < 0)
				continue;

			m = EDV_MIME_TYPE(g_list_nth_data(
				mime_types_list,
				(guint)i
			));
			if(m == NULL)
				continue;

			APPEND_BUF(m->type);
		}
	}
	else
	{
		GList *glist;
		EDVMIMEType *m;

		/* Iterate through the MIME Types list */
		for(glist = mime_types_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			m = EDV_MIME_TYPE(glist->data);
			if(m == NULL)
				continue;

			APPEND_BUF(m->type);
		}
	}

#undef APPEND_BUF

	*buf_len_rtn = buf_len;

	return(buf);
}


/*
 *	Creates a list of MIME Type Commands from the DDE Buffer.
 *
 *	The buf and buf_len specifies the DDE Buffer created by
 *	edv_mime_type_commands_list_encode_buffer().
 *
 *	Returns a GList of EDVMIMETypeCommand * that must
 *	be deleted by the calling function.
 */
GList *edv_mime_type_commands_list_decode_buffer(
	const guint8 *buf,
	const gint buf_len
)
{
	gint len;
	const guint8	*buf_ptr,
			*buf_end,
			*buf_next;
	GList *list_rtn;
	EDVMIMETypeCommand *cmd;

	if((buf == NULL) || (buf_len <= 0))
		return(NULL);

	list_rtn = NULL;

	/* Begin parsing the DDE buffer
	 *
	 * Format of each segment is as follows:
	 *
	 *	<name>			String
	 *	<command>		String
	 *	<shell_command>		String
	 *	<flags>			Number string
	 */
	buf_ptr = buf;
	buf_end = buf_ptr + buf_len;
	while(buf_ptr < buf_end)
	{
		/* Create a new command */
		cmd = edv_mime_type_command_new();
		if(cmd == NULL)
			break;

		/* Append the new command to the return list */
		list_rtn = g_list_append(
			list_rtn,
			cmd
		);

		/* Begin parsing this segment */

		/* Get the name */
		buf_next = buf_ptr;
		while(buf_next < buf_end)
		{
			if(*buf_next == 0x00)
				break;
			buf_next++;
		}
		len = buf_next - buf_ptr;
		if(len > 0)
		{
			cmd->name = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			if(cmd->name != NULL)
			{
				memcpy(
					cmd->name,
					buf_ptr,
					len * sizeof(gchar)
				);
				cmd->name[len] = '\0';
			}
		}
		buf_ptr = buf_next + 1;	/* Seek past the null character */
		if(buf_ptr >= buf_end)
			break;

		/* Get the command */
		buf_next = buf_ptr;
		while(buf_next < buf_end)
		{
			if(*buf_next == 0x00)
				break;
			buf_next++;
		}
		len = buf_next - buf_ptr;
		if(len > 0)
		{
			cmd->command = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			if(cmd->command != NULL)
			{
				memcpy(
					cmd->command,
					buf_ptr,
					len * sizeof(gchar)
				);
				cmd->command[len] = '\0';
			}
		}
		buf_ptr = buf_next + 1;	/* Seek past the null character */
		if(buf_ptr >= buf_end)
			break;

		/* Get the shell command */
		buf_next = buf_ptr;
		while(buf_next < buf_end)
		{
			if(*buf_next == 0x00)
				break;
			buf_next++;
		}
		len = buf_next - buf_ptr;
		if(len > 0)
		{
			cmd->shell_command = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			if(cmd->shell_command != NULL)
			{
				memcpy(
					cmd->shell_command,
					buf_ptr,
					len * sizeof(gchar)
				);
				cmd->shell_command[len] = '\0';
			}
		}
		buf_ptr = buf_next + 1;	/* Seek past the null character */
		if(buf_ptr >= buf_end)
			break;

		/* Get the flags */
		buf_next = buf_ptr;
		while(buf_next < buf_end)
		{
			if(*buf_next == 0x00)
				break;
			buf_next++;
		}
/*	    len = buf_next - buf_ptr; */
		cmd->flags = ATOI((const gchar *)buf_ptr);
		buf_ptr = buf_next + 1;	/* Seek past the null character */
	}

	return(list_rtn);
}

/*
 *	Creates a DDE Buffer from the MIME Type commands list.
 *
 *	The commands_list specifies the MIME Type commands list,
 *	which is a GList of EDVMIMETypeCommand *.
 *
 *	The buf_len_rtn specifies the length of the DDE buffer.
 *
 *	Returns the DDE buffer which describes the MIME Type
 *	commands. This buffer must be deleted by the calling
 *	function.
 */
guint8 *edv_mime_type_commands_list_encode_buffer(
	GList *commands_list,
	gint *buf_len_rtn
)
{
	gint buf_len;
	gchar *s;
	guint8 *buf;
	GList *glist;
	EDVMIMETypeCommand *cmd;

	if(buf_len_rtn == NULL)
		return(NULL);

	*buf_len_rtn = 0;

	if(commands_list == NULL)
		return(NULL);

	buf_len = 0;
	buf = NULL;

#define APPEND_BUF_S(_s_)	{		\
 const gint	buf_i = buf_len,		\
		s_len = STRLEN(_s_);		\
						\
 buf_len += s_len + 1;		/* Include the null character */ \
 buf = (guint8 *)g_realloc(			\
  buf,						\
  buf_len * sizeof(guint8)			\
 );						\
 if(buf != NULL) {				\
  if(s_len > 0)					\
   memcpy(					\
    buf + buf_i,				\
    (_s_),					\
    s_len * sizeof(guint8)			\
   );						\
  buf[buf_i + s_len] = '\0';	/* Set the null character */ \
 } else {					\
  buf_len = 0;					\
 }						\
}

	/* Append each command to the buffer */
	for(glist = commands_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
			continue;

		/* Append this command as a segment
		 *
		 * Format of each segment is as follows:
		 *
		 *	<name>			String
		 *	<command>		String
		 *	<shell_command>		String
		 *	<flags>			Number string
		 */
		APPEND_BUF_S(cmd->name);
		APPEND_BUF_S(cmd->command);
		APPEND_BUF_S(cmd->shell_command);
		s = g_strdup_printf("%u", (guint)cmd->flags);
		APPEND_BUF_S(s);
		g_free(s);
	}

#undef APPEND_BUF_S

	*buf_len_rtn = buf_len;

	return(buf);
}


/*
 *	Opens the MIME Types that are specified on the system.
 *
 *	The edv_data_path specifies the full path to the data directury
 *	where the icons are to be loaded from. The EDV_NAME_ICONS_SUBDIR
 *	will be postfixed to this value in which all the icons will
 *	be loaded from this directory.
 *
 *	The list and total specifies the pointers to the MIME Types
 *	list.
 *
 *	The insert_index specifies the position at which to insert the
 *	MIME Types read from file to the list, or it can be -1 to
 *	append to the list.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.   
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	If added_cb is not NULL then it will be called after each new
 *	MIME Type is added to the list.
 *
 *	The added_data specifies the user data which will be passed to
 *	the added_cb function.
 *
 *	If mark_all_loaded_read_only is TRUE then all MIME Types read
 *	from this file will be marked as read only.
 */
GList *edv_mime_types_list_file_open_system(
	GList *mime_types_list,
	const gchar *edv_data_path,
	const gint insert_index,
	gint (*progress_cb)(
		gpointer,
		const gulong, const gulong
	),
	gpointer progress_data,
	void (*added_cb)(
		const gint,
		EDVMIMEType *,
		gpointer
	),
	gpointer added_data,
	const gboolean mark_all_loaded_read_only
)
{
	const int nstates = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	const gchar     *icon_path_s[nstates],
			*icon_path_m[nstates],
			*icon_path_l[nstates];
	EDVMIMEType *m;
	gint cur_insert_index = insert_index;
	gchar *icons_dir = g_strconcat(
		edv_data_path,
		G_DIR_SEPARATOR_S,
		EDV_NAME_ICONS_SUBDIR,
		NULL
	);

	if(icons_dir == NULL)
		return(mime_types_list);

	(void)memset(
		icon_path_s,
		0x00,
		sizeof(icon_path_s)
	);
	(void)memset(
		icon_path_m,
		0x00,
		sizeof(icon_path_m)
	);
	(void)memset(
		icon_path_l,
		0x00,
		sizeof(icon_path_l)
	);

#define APPEND_MIMETYPE(				\
 _val_,_type_,_desc_,					\
 _icon_path_s_,						\
 _icon_path_m_,						\
 _icon_path_l_						\
) {							\
 gint mt_num;						\
													\
 if(progress_cb != NULL) {				\
  if(progress_cb(progress_data, 0l, 0l)) {		\
   g_free(icons_dir);					\
   return(mime_types_list);				\
  }							\
 }							\
													\
 /* Create a new MIME Type */				\
 m = edv_mime_type_new_values();					\
 if(m != NULL) {					\
  m->type = EDV_MIME_TYPE_CLASS_SYSTEM;			\
  m->value = STRDUP(_val_);				\
  m->type = STRDUP(_type_);				\
  m->description = STRDUP(_desc_);			\
 }							\
													\
 /* Append/insert the new MIME Type into the list */	\
 if(cur_insert_index < 0) {				\
  mt_num = g_list_length(mime_types_list);		\
  mime_types_list = g_list_append(			\
   mime_types_list,					\
   m							\
  );							\
 } else {						\
  mt_num = cur_insert_index;				\
  mime_types_list = g_list_insert(			\
   mime_types_list,					\
   m,							\
   cur_insert_index					\
  );							\
  /* Increment the insert position */			\
  cur_insert_index++;					\
 }							\
 if(m != NULL) {					\
  /* Set the new MIME Type's values */			\
  gint i;						\
  const gchar *path;					\
													\
  m->read_only = mark_all_loaded_read_only;		\
  m->realized = FALSE;					\
													\
  for(i = 0; i < nstates; i++) {			\
   path = (_icon_path_s_)[i];				\
   if(path != NULL)					\
    m->small_icon_path[i] = g_strconcat(		\
     icons_dir,						\
     G_DIR_SEPARATOR_S,					\
     path,						\
     NULL						\
    );							\
													\
   path = (_icon_path_m_)[i];				\
   if(path != NULL)					\
    m->medium_icon_path[i] = g_strconcat(		\
     icons_dir,						\
     G_DIR_SEPARATOR_S,					\
     path,						\
     NULL						\
    );							\
													\
   path = (_icon_path_l_)[i];				\
   if(path != NULL)					\
    m->large_icon_path[i] = g_strconcat(		\
     icons_dir,						\
     G_DIR_SEPARATOR_S,					\
     path,						\
     NULL						\
    );							\
  }							\
													\
  /* Notify about this MIME Type being added */		\
  if(added_cb != NULL)					\
   added_cb(mt_num, m, added_data);			\
 }							\
}
	/* Unknown */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_object_misc_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_object_misc_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_object_misc_20x20.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_object_misc_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_object_misc_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_object_misc_32x32.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_object_misc_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_object_misc_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_object_misc_48x48.xpm";
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_UNKNOWN, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* Error */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_object_error_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_object_error_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_object_error_20x20.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_object_error_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_object_error_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_object_error_32x32.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_object_error_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_object_error_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_object_error_48x48.xpm";
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_ERROR, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* File */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_file_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_file_hidden_20x20.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =      
		"icon_file_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_file_hidden_32x32.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =      
		"icon_file_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_file_hidden_48x48.xpm";
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_FILE, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* Executable */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_executable_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_executable_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_executable_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* Directory */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_folder_closed_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_folder_opened_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] =
		"icon_folder_noaccess_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_folder_hidden_20x20.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_folder_closed_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] =      
		"icon_folder_opened_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] =
		"icon_folder_noaccess_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_folder_hidden_32x32.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_folder_closed_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] =
		"icon_folder_opened_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] =
		"icon_folder_noaccess_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_folder_hidden_48x48.xpm";
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_DIRECTORY, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* Link */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_link_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = 
		"icon_link_broken_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_link_hidden_20x20.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_link_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] =
		"icon_link_broken_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_link_hidden_32x32.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_link_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] =  
		"icon_link_broken_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] =
		"icon_link_hidden_48x48.xpm";
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_LINK, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* Device Block */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_device_block_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =      
		"icon_device_block_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_device_block_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* Device Character */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_device_character_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_device_character_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_device_character_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* FIFO Pipe */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_pipe_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_pipe_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_pipe_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_FIFO, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

	/* Socket */
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_socket_20x20.xpm";
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_s[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_socket_32x32.xpm";
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_m[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_STANDARD] =
		"icon_socket_48x48.xpm";
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_OPENED] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE] = NULL;
	icon_path_l[EDV_MIME_TYPE_ICON_STATE_HIDDEN] = NULL;
	APPEND_MIMETYPE(
		NULL, EDV_MIME_TYPE_TYPE_INODE_SOCKET, NULL,
		icon_path_s, icon_path_m, icon_path_l
	);

#undef APPEND_MIMETYPE

	g_free(icons_dir);

	return(mime_types_list);
}


/*
 *	Opens the MIME Types from the MIME Types file.
 *
 *	The mime_types_list specifies the MIME Types list.
 *
 *	The path specifies the MIME Types file.
 *
 *      The insert_index specifies the position at which to insert the
 *	MIME Types read from file to the list, or it can be -1 to
 *	append to the list.
 *
 *	If update is TRUE then if a MIME Type from the file already
 *	exists in the MIME Types list then it will be updated instead
 *	of a new MIME Type being added.
 *
 *	If only_newer is TRUE and update is TRUE then if a MIME Type
 *	from the file already exists in the MIME Types list then it
 *	will only be updated if the one from the file has a newer
 *	modified time.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.   
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	If added_cb is not NULL then it will be called after each new
 *	MIME Type is added to the list.
 *
 *	The added_data specifies the user data which will be passed to
 *	the added_cb function.
 *
 *	If mark_all_loaded_read_only is TRUE then all MIME Types read
 *	from this file will be marked as read only.
 *
 *	Reminder: There is a sister function in
 *	lib/edv_mime_types_list.c which needs to perform equvilently
 *	to this function.
 */
GList *edv_mime_types_list_file_open(
	GList *mime_types_list,
	const gchar *path,
	const gint insert_index,
	const gboolean update,
	const gboolean only_newer,
	gint (*progress_cb)(
		gpointer,
		const gulong, const gulong
	),
	gpointer progress_data,
	void (*added_cb)(
		const gint,
		EDVMIMEType *,
		gpointer
	),
	gpointer added_data,
	void (*modified_cb)(
		const gint,
		EDVMIMEType *,
		gpointer
	),
	gpointer modified_data,
	const gboolean mark_all_loaded_read_only
)
{
	FILE *fp;
	gboolean aborted = FALSE;
	gint cur_insert_index = insert_index;
	gulong file_size = 0l;
	gchar *parm;
	EDVVFSObject *obj;
	EDVMIMEType *m = NULL;
	EDVMIMETypeCommand *cmd = NULL;

	if(STRISEMPTY(path))
		return(mime_types_list);

	/* Open the MIME Types file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(mime_types_list);

	/* Get the file's stats */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		file_size = obj->size;
		edv_vfs_object_delete(obj);
	}

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
		if(progress_cb(
			progress_data,
			0l, file_size
		))
			aborted = TRUE;
	}

	/* Begin reading the MIME Types file */
	parm = NULL;
	while(!aborted)
	{
		/* Report the progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				progress_data,
				(gulong)ftell(fp), file_size
			))
			{
				aborted = TRUE;
				break;
			}
		}

		/* Read the next parameter */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Handle by parameter */

		/* BeginMIMEType */
		if(!g_strcasecmp(parm, "BeginMIMEType"))
		{
			gint value[1];
			EDVMIMETypeClass mt_class;

			/* Get the class, which is the value for BeginMIMEType */
			FGetValuesI(fp, (int *)value, 1);
			mt_class = (EDVMIMETypeClass)value[0];

			/* Create a new MIME Type */
			if(m == NULL)
			{
				m = edv_mime_type_new_values();
				if(m != NULL)
				{
					m->mt_class = mt_class;
				}
			}
			if(m != NULL)
			{
				/* Mark as read only? */
				if(mark_all_loaded_read_only)
					m->read_only = TRUE;
			}
		}
		/* Type */
		else if(!g_strcasecmp(parm, "Type"))
		{
			gchar *type = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				g_free(m->type);
				m->type = STRDUP(type);
			}
			g_free(type);
		}
		/* Value */
		else if(!g_strcasecmp(parm, "Value"))
		{
			gchar *value = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				g_free(m->value);
				m->value = STRDUP(value);
			}
			g_free(value);
		}
		/* Description */
		else if(!g_strcasecmp(parm, "Description"))
		{
			gchar *description = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				g_free(m->description);
				m->description = STRDUP(description);
			}
			g_free(description);
		}

		/* IconSmallStandard */
		else if(!g_strcasecmp(parm, "IconSmallStandard"))
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->small_icon_path[
					EDV_MIME_TYPE_ICON_STATE_STANDARD
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconSmallOpened */
		else if(!g_strcasecmp(parm, "IconSmallOpened") ||
			!g_strcasecmp(parm, "IconSmallSelected")
		)
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->small_icon_path[
					EDV_MIME_TYPE_ICON_STATE_OPENED
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconSmallInaccessible */
		else if(!g_strcasecmp(parm, "IconSmallInaccessible") ||
				!g_strcasecmp(parm, "IconSmallExtended")
		)
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->small_icon_path[
					EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconSmallHidden */
		else if(!g_strcasecmp(parm, "IconSmallHidden"))
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->small_icon_path[
					EDV_MIME_TYPE_ICON_STATE_HIDDEN
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}

		/* IconMediumStandard */
		else if(!g_strcasecmp(parm, "IconMediumStandard"))
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->medium_icon_path[
					EDV_MIME_TYPE_ICON_STATE_STANDARD
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconMediumOpened */
		else if(!g_strcasecmp(parm, "IconMediumOpened") ||
			!g_strcasecmp(parm, "IconMediumSelected")
		)
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->medium_icon_path[
					EDV_MIME_TYPE_ICON_STATE_OPENED
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconMediumInaccessible */
		else if(!g_strcasecmp(parm, "IconMediumInaccessible") ||
				!g_strcasecmp(parm, "IconMediumExtended")
		)
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->medium_icon_path[
					EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconMediumHidden */
		else if(!g_strcasecmp(parm, "IconMediumHidden"))
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->medium_icon_path[
					EDV_MIME_TYPE_ICON_STATE_HIDDEN
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}

		/* IconLargeStandard */
		else if(!g_strcasecmp(parm, "IconLargeStandard"))
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->large_icon_path[
					EDV_MIME_TYPE_ICON_STATE_STANDARD
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconLargeOpened */
		else if(!g_strcasecmp(parm, "IconLargeOpened") ||
			!g_strcasecmp(parm, "IconLargeSelected")
		)
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->large_icon_path[
					EDV_MIME_TYPE_ICON_STATE_OPENED
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconLargeInaccessible */
		else if(!g_strcasecmp(parm, "IconLargeInaccessible") ||
		 	!g_strcasecmp(parm, "IconLargeExtended")
		)
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->large_icon_path[
					EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}
		/* IconLargeHidden */
		else if(!g_strcasecmp(parm, "IconLargeHidden"))
		{
			gchar *path = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				gchar **p = &m->large_icon_path[
					EDV_MIME_TYPE_ICON_STATE_HIDDEN
				];
				g_free(*p);
				*p = STRDUP(path);
			}
			g_free(path);
		}

		/* Handler */
		else if(!g_strcasecmp(parm, "Handler"))
		{
			gchar *s = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				if(!g_strcasecmp(s, "Archiver"))
					m->handler = EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER;
				else if(!g_strcasecmp(s, "ImageBrowser"))
					m->handler = EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER;
				else if(!g_strcasecmp(s, "RecycleBin"))
					m->handler = EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN;
				else
					m->handler = EDV_MIME_TYPE_HANDLER_COMMAND;
			}
			g_free(s);
		}
		/* BeginCommand */
		else if(!g_strcasecmp(parm, "BeginCommand"))
		{
			gchar *name = (gchar *)FGetString(fp);
			if(m != NULL)
			{
				/* Create/set new context */
				cmd = edv_mime_type_command_new();
				if(cmd != NULL)
				{
					m->commands_list = g_list_append(
						m->commands_list,
						cmd
					);
					if(!STRISEMPTY(name))
						cmd->name = g_strdup(name);
				}
				else
				{
					break;
				}
			}
			g_free(name);
		}
		/* CommandCommand */
		else if(!g_strcasecmp(parm, "CommandCommand"))
		{
			gchar *command = (gchar *)FGetString(fp);
			if(cmd != NULL)
			{
				g_free(cmd->command);
				cmd->command = STRDUP(command);
			}
			g_free(command);
		}
		/* CommandShellCommand */
		else if(!g_strcasecmp(parm, "CommandShellCommand"))
		{
			gchar *shell_command = (gchar *)FGetString(fp);
			if(cmd != NULL)
			{
				g_free(cmd->shell_command);
				cmd->shell_command = STRDUP(shell_command);
			}
			g_free(shell_command);
		}
		/* CommandRunInTerminal */
		else if(!g_strcasecmp(parm, "CommandRunInTerminal"))
		{
			/* This parameter has no value */
			FSeekNextLine(fp);
			if(cmd != NULL)
			{
				cmd->flags |= EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL;
			}
		}
		/* EndCommand */
		else if(!g_strcasecmp(parm, "EndCommand"))
		{
			/* This parameter has no value */
			FSeekNextLine(fp);

			/* Reset the context */
			cmd = NULL;
		}
		/* Command (version 2.8.4 and older) */
		else if(!g_strcasecmp(parm, "Command"))
		{
			/* This parameter has been depreciated, we are handling
			 * it here for compatability purposes, see
			 * parameters BeginCommand and EndCommand
			 */
			gchar *command = (gchar *)FGetString(fp);
			if((m != NULL) && !STRISEMPTY(command))
			{
				cmd = edv_mime_type_command_new();
				if(cmd != NULL)
				{
					/* Seek s to the name and command deliminator */
					gchar *s_cmd = (gchar *)strchr(
						(char *)command,
						EDV_CFG_DELIMINATOR_CHAR
					);
					if(s_cmd != NULL)
					{
						gchar *s_end = s_cmd;
						gint len = (gint)(s_end - command);

						s_cmd++;

						cmd->name = (gchar *)g_malloc((len + 1) * sizeof(gchar));
						if(len > 0)
							memcpy(cmd->name, command, len * sizeof(gchar));
						cmd->name[len] = '\0';
						cmd->name = g_strstrip(cmd->name);

						cmd->command = STRDUP(s_cmd);
						cmd->command = g_strstrip(cmd->command);
					}
					else
					{
						/* No command deliminator, implying there is
						 * only a command
						 *
						 * Create an arbitrary name for this command
						 */
						cmd->name = g_strdup_printf(
							"Command #%i",
							g_list_length(m->commands_list) + 1
						);

						cmd->command = STRDUP(command);
						cmd->command = g_strstrip(cmd->command);
					}

					m->commands_list = g_list_append(
						m->commands_list,
						cmd
					);
				}

				/* Reset the context */
				cmd = NULL;
			}
			g_free(command);
		}

		/* AccessTime */
		else if(!g_strcasecmp(parm, "AccessTime"))
		{
			glong value[1];
			FGetValuesL(fp, (long *)value, 1);
			if(m != NULL)
				m->access_time = (gulong)value[0];
		}
		/* ModifyTime */
		else if(!g_strcasecmp(parm, "ModifyTime"))
		{
			glong value[1];
			FGetValuesL(fp, (long *)value, 1);
			if(m != NULL)
				m->modify_time = (gulong)value[0];
		}
		/* ChangeTime */
		else if(!g_strcasecmp(parm, "ChangeTime"))
		{
			glong value[1];
			FGetValuesL(fp, (long *)value, 1);
			if(m != NULL)
				m->change_time = (gulong)value[0];
		}

		/* EndMIMEType */
		else if(!g_strcasecmp(parm, "EndMIMEType"))
		{
			/* This parameter has no value */
			FSeekNextLine(fp);

			/* Is there a MIME Type currently in context to be
			 * added or updated?
			 */
			if(m != NULL)
			{
				gint mt_num = -1;

				/* Update MIME Types if they already exist in the
				 * list?
				 */
				if(update)
				{
					/* Check if this MIME Type is already in the list */
					EDVMIMEType *src_m = edv_mime_types_list_match_type(
						mime_types_list,
						&mt_num,
						m->type, FALSE
					);
					/* Found a matching MIME Type in the list? */
					if((mt_num > -1) && (src_m != NULL))
					{
						/* Only update if the MIME Type from file
						 * is newer than the MIME Type from the
						 * list?
						 */
						if(only_newer)
						{
							/* Is the MIME Type from the file the
							 * same age or older than the one from
							 * the list?
							 */
							if((m->modify_time <= src_m->modify_time) &&
							   (m != src_m)
							)
							{
								/* Delete it and set it NULL so
								 * that it neither gets added nor
								 * updated
								 */
								edv_mime_type_delete(m);
								m = NULL;
								mt_num = -1;
							}
						}
					}
				}

				/* Update this MIME Type that is already in the
				 * list?
				 */
				if((mt_num > -1) && (m != NULL))
				{
					/* Delete the MIME Type that is in the list
					 * and then replace it with the one from the
					 * file
					 */
					GList *glist = g_list_nth(
						mime_types_list,
						(guint)mt_num
					);
					if(glist != NULL)
					{
						edv_mime_type_delete(EDV_MIME_TYPE(glist->data));
						glist->data = m;
						if(modified_cb != NULL)
							modified_cb(
								mt_num,
								m,
								modified_data
							);
					}
					else
					{
						edv_mime_type_delete(m);
						m = NULL;
						mt_num = -1;
					}
				}
				/* Add this MIME Type into the list? */
				else if(m != NULL)
				{
					/* Append/insert this MIME Type into the list */
					if(cur_insert_index < 0)
					{
						/* Append this MIME Type to the list */
						mt_num = g_list_length(mime_types_list);
						mime_types_list = g_list_append(
							mime_types_list,
							m
						);
					}
					else
					{
						/* Insert this MIME Type into the list */
						mt_num = cur_insert_index;
						mime_types_list = g_list_insert(
							mime_types_list,
							m,
							cur_insert_index
						);
						cur_insert_index++;	/* Increment the insert position */
					}
					if(added_cb != NULL)
						added_cb(
							mt_num,
							m,
							added_data
						);
				}
			}

			/* Reset the context */
			m = NULL;
		}
		/* Unsupported parameter */
		else
		{
			/* Ignore unsupported parameter */
			FSeekNextLine(fp);
		}
	}

	/* Delete the current MIME Type if it was not added to the list */
	edv_mime_type_delete(m);

	/* Delete the parameter */
	g_free(parm);

	/* Close the MIME Types file */
	(void)fclose(fp);

	/* Report the final progress */
	if((progress_cb != NULL) && !aborted)
	{
		if(progress_cb(
			progress_data,
			file_size, file_size
		))
			aborted = TRUE;
	}

	return(mime_types_list);
}

/*
 *	Saves the MIME Types list to the MIME Types file.
 *
 *	The path specifies the MIME Types file.
 *
 *	The mime_types_list specifies the MIME Types list.
 *
 *	If include_read_only is TRUE then MIME Types that are marked
 *	as read only will also be saved.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.   
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 */
void edv_mime_types_list_file_save(
	GList *mime_types_list,
	const gchar *path,
	const gboolean include_read_only,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
)
{
	FILE *fp;
	gboolean aborted = FALSE;
	gint mt_num, total;
	const gchar *s, *icon_path;
	gchar *parent;
	GList *glist;
	EDVMIMEType *m;

	if((mime_types_list == NULL) || STRISEMPTY(path))
		return;

	/* Get parent directory and create it as needed */
	parent = g_dirname(path);
	if(parent != NULL)
	{
		edv_directory_create(
			parent,
			TRUE,				/* Create parents */
			NULL				/* No new paths list return */
		);
		g_free(parent);
	}

	/* Open the MIME Types file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
		return;

	total = g_list_length(mime_types_list);

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
		if(progress_cb(
			progress_data,
			0l, (gulong)total
		))
			aborted = TRUE;
	}

	/* Save each MIME Type */
	for(glist = mime_types_list, mt_num = 0;
		glist != NULL;
		glist = g_list_next(glist), mt_num++
	)
	{
		if(aborted)
			break;

		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		/* Skip MIME Types that are marked read only, meaning they
		 * should not be saved to file since they are created
		 * internally or are loaded from a global configuration
		 */
		if(!include_read_only && m->read_only)
			continue;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				progress_data,
				(gulong)(mt_num + 1), (gulong)total
			))
			{
				aborted = TRUE;
				break;
			}
		}


		/* Begin saving this MIME Type */

		/* BeginMIMEType */
		(void)fprintf(
			fp,
			"BeginMIMEType = %i\n",
			(int)m->mt_class
		);

		/* Type */
		if(!STRISEMPTY(m->type))
			(void)fprintf(
				fp,
				"\tType = %s\n",
				(const char *)m->type
			);

		/* Value */
		if(!STRISEMPTY(m->value))
			(void)fprintf(
				fp,
				"\tValue = %s\n",
				(const char *)m->value
			);

		/* Description */
		if(!STRISEMPTY(m->description))
			(void)fprintf(
				fp,
				"\tDescription = %s\n",
				(const char *)m->description
			);

		/* IconSmallStandard */
		icon_path = m->small_icon_path[EDV_MIME_TYPE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconSmallStandard = %s\n",
				(const char *)icon_path
			);
		/* IconSmallOpened */
		icon_path = m->small_icon_path[EDV_MIME_TYPE_ICON_STATE_OPENED];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconSmallOpened = %s\n",
				(const char *)icon_path
			);
		/* IconSmallInaccessible */
		icon_path = m->small_icon_path[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconSmallInaccessible = %s\n",
				(const char *)icon_path
			);
		/* IconSmallHidden */
		icon_path = m->small_icon_path[EDV_MIME_TYPE_ICON_STATE_HIDDEN];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconSmallHidden = %s\n",
				(const char *)icon_path
			);

		/* IconMediumStandard */
		icon_path = m->medium_icon_path[EDV_MIME_TYPE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconMediumStandard = %s\n",
				(const char *)icon_path
			);
		/* IconMediumOpened */
		icon_path = m->medium_icon_path[EDV_MIME_TYPE_ICON_STATE_OPENED];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconMediumOpened = %s\n",
				(const char *)icon_path
			);
		/* IconMediumInaccessible */
		icon_path = m->medium_icon_path[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconMediumInaccessible = %s\n",
				(const char *)icon_path
			);
		/* IconMediumHidden */
		icon_path = m->medium_icon_path[EDV_MIME_TYPE_ICON_STATE_HIDDEN];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconMediumHidden = %s\n",
				(const char *)icon_path
			);

		/* IconLargeStandard */
		icon_path = m->large_icon_path[EDV_MIME_TYPE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconLargeStandard = %s\n",
				(const char *)icon_path
			);
		/* IconLargeOpened */
		icon_path = m->large_icon_path[EDV_MIME_TYPE_ICON_STATE_OPENED];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconLargeOpened = %s\n",
				(const char *)icon_path
			);
		/* IconLargeInaccessible */
		icon_path = m->large_icon_path[EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconLargeInaccessible = %s\n",
				(const char *)icon_path
			);
		/* IconLargeHidden */
		icon_path = m->large_icon_path[EDV_MIME_TYPE_ICON_STATE_HIDDEN];
		if(!STRISEMPTY(icon_path))
			(void)fprintf(
				fp,
				"\tIconLargeHidden = %s\n",
				(const char *)icon_path
			);

		/* Handler */
		switch(m->handler)
		{
		  case EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER:
			s = "Archiver";
			break;
		  case EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER:
			s = "ImageBrowser";
			break;
		  case EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN:
			s = "RecycleBin";
			break;
		  default:	/* EDV_MIME_TYPE_HANDLER_COMMAND */
			s = "Command";
			break;
		}
		(void)fprintf(
			fp,
			"\tHandler = %s\n",
			(const char *)s
		);

		/* Commands */
		if(m->commands_list != NULL)
		{
			gint i;
			GList *glist;
			EDVMIMETypeCommandFlags flags;
			EDVMIMETypeCommand *cmd;

			for(glist = m->commands_list, i = 0;
				glist != NULL;
				glist = g_list_next(glist), i++
			)
			{
				cmd = EDV_MIME_TYPE_COMMAND(glist->data);
				if(cmd == NULL)
					continue;

				flags = cmd->flags;

				/* BeginCommand */
				if(STRISEMPTY(cmd->name))
					(void)fprintf(
						fp,
						"\tBeginCommand = Command #%i\n",
						(int)(i + 1)
					);
				else
					(void)fprintf(
						fp,
						"\tBeginCommand = %s\n",
						(const char *)cmd->name
					);

				/* CommandCommand */
				if(!STRISEMPTY(cmd->command))
					(void)fprintf(
						fp,
						"\t\tCommandCommand = %s\n",
						(const char *)cmd->command
					);

				/* CommandShellCommand */
				if(!STRISEMPTY(cmd->shell_command))
					(void)fprintf(
						fp,
						"\t\tCommandShellCommand = %s\n",
						(const char *)cmd->shell_command
					);

				/* CommandRunInTerminal */
				if(flags & EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL)
					(void)fprintf(
						fp,
						"\t\tCommandRunInTerminal\n"
					);

				/* EndCommand */
				(void)fprintf(
					fp,
					"\tEndCommand\n"
				);
			}
		}

		/* AccessTime */
		if(m->access_time > 0l)
			(void)fprintf(
				fp,
				"\tAccessTime = %ld\n",
				(unsigned long)m->access_time
		    );
		/* ModifyTime */
		if(m->modify_time > 0l)
			(void)fprintf(
				fp,
				"\tModifyTime = %ld\n",
				(unsigned long)m->modify_time
			);
		/* ChangeTime */
		if(m->change_time > 0l)
			(void)fprintf(
				fp,
				"\tChangeTime = %ld\n",
				(unsigned long)m->change_time
			);

		/* EndMIMEType */
		(void)fprintf(
			fp,
			"EndMIMEType\n"
		);
	}

	/* Close the MIME Types file */
	if(fclose(fp))
		return;

	/* Report the final progress */
	if((progress_cb != NULL) && !aborted)
	{
		if(progress_cb(
			progress_data,
			(gulong)total, (gulong)total
		))
			aborted = TRUE;
	}
}
