#include <gtk/gtk.h>

#include "cfg.h"

#include "cdialog.h"

#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_utils_gtk.h"
#include "mime_types_list_win.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "edv_mime_type_install.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


gint edv_mime_type_install_wizard(
	EDVCore *core,
	const gint pos,
	EDVMIMETypeClass mt_class,
	const gchar *value,
	const gchar *type,
	const gchar *description,
	GtkWidget *toplevel
);

gint edv_mime_type_install(
	EDVCore *core,
	EDVMIMEType *m,
	const gint pos,
	const gboolean interactive,
	const gboolean verbose,
	GtkWidget *toplevel
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Queries the user to install a new MIME Type.
 *
 *	The pos specifies the index position in the MIME Types list
 *	to install the new MIME Type at. If pos is -1 then the new
 *	MIME Type will be appended.
 *
 *	Returns the index of the new installed MIME Type or one of the
 *	following error codes:
 *
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error or memory allocation error.
 *	-4	User aborted.
 *	-5	Another operation is in progress.
 *	-7	Another MIME Type already exists with the same type.
 */
gint edv_mime_type_install_wizard(
	EDVCore *core,
	const gint pos,
	EDVMIMETypeClass mt_class,
	const gchar *value,
	const gchar *type,
	const gchar *description,
	GtkWidget *toplevel
)
{
	gint mt_num;
	EDVMIMEType *m;
	edv_mime_types_list_win_struct *lw;

	if(core == NULL)
		return(-2);

	if(CDialogIsQuery())
		return(-5);

	/* Create a new MIME Type */
	m = edv_mime_type_new_values();
	if(m == NULL)
		return(-3);

	m->mt_class = mt_class;
	m->value = STRDUP(value);
	m->type = STRDUP(type);
	m->description = STRDUP(description);




	/* Map the MIME Types window */
	edv_map_mime_types(
		core,
		NULL,				/* No initial MIME Type */
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);

	/* Install the new MIME Type and select it on the MIME
	 * Types list
	 */
	mt_num = edv_mime_type_install(
		core,
		m,
		pos,
		FALSE,				/* Not interactive */
		FALSE,				/* Not verbose */
		toplevel
	);
	if(mt_num < 0)
	{
		const gint error_code = mt_num;
		switch(error_code)
		{
		  case -1:
			edv_play_sound_error(core);
			edv_message_error(
"Install MIME Type Failed",
"General error.",
				NULL,
				toplevel
			);
			break;

		  case -2:
			edv_play_sound_error(core);
			edv_message_error(
"Install MIME Type Failed",
"Invalid value.",
				NULL,
				toplevel
			);
			break;

		  case -3:
			edv_play_sound_error(core);
			edv_message_error(
"Install MIME Type Failed",
"Memory allocation error.",
				NULL,
				toplevel
			);
			break;

		  case -4:
			edv_play_sound_warning(core);
			edv_message_warning(
"Install MIME Type Canceled",
"Install MIME Type canceled.",
				NULL,
				toplevel
			);
			break;

		  case -5:
			edv_play_sound_warning(core);
			edv_message_warning(
"Install MIME Type Failed",
"Another operation is already in progress.",
				NULL,
				toplevel
			);
			break;
		}

		return(error_code);
	}

	/* Edit the new MIME Type */
	lw = core->mime_types_list_win;
	EDVMimeTypesListWinEditMimeType(lw, mt_num);

	return(mt_num);
}


/*
 *	Installs a new MIME Type.
 *
 *	The core specifies the core and the list of MIME Types to
 *	install the MIME Type to.
 *
 *	The m specifies the MIME Type to be installed. It will be
 *	transfered and therefore it should not be referenced again
 *	after this call.
 *
 *	The pos specifies the index position in the MIME Types list
 *	to install the new MIME Type at. If pos is -1 then the new
 *	MIME Type will be appended.
 *
 *	Returns the index of the new installed MIME Type or one of the
 *	following error codes:
 *
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error or memory allocation error.
 *	-4	User aborted.
 *	-5	Another operation is in progress.
 */
gint edv_mime_type_install(
	EDVCore *core,
	EDVMIMEType *m,
	const gint pos,
	const gboolean interactive,
	const gboolean verbose,
	GtkWidget *toplevel
)
{
	const gchar *s;
	gint mt_num = 0;
	CfgList *cfg_list;
	EDVMIMEType *existing_mime_type;
	edv_mime_types_list_win_struct *lw;

	if((core == NULL) || (m == NULL))
	{
		edv_mime_type_delete(m);
		return(-2);
	}

	cfg_list = core->cfg_list;
	lw = core->mime_types_list_win;

	/* Check if a MIME Type with the same type already exists */
	existing_mime_type = NULL;
	if(!STRISEMPTY(m->type))
	{
		gchar *type = g_strdup(m->type);
		GList *glist;
		EDVMIMEType *src_m;

		for(mt_num = 0,
			glist = core->mime_types_list;
			glist != NULL;
			mt_num++,
			glist = g_list_next(glist)
		)
		{
			src_m = EDV_MIME_TYPE(glist->data);
			if(src_m == NULL)
				continue;

			if(src_m->type == NULL)
				continue;

			if(!g_strcasecmp(src_m->type, type))
			{
				existing_mime_type = src_m;
				break;
			}
		}

		g_free(type);
	}

	/* Confirm install or update? */
	if(interactive)
	{
		gint response;
		gchar *msg;

		if(CDialogIsQuery())
		{
			edv_mime_type_delete(m);
			return(-5);
		}

		msg = g_strdup_printf(
"%s:\n\
\n\
    %s\n\
\n\
%s",
			(existing_mime_type != NULL) ?
				"Update MIME Type" : "Install MIME Type",
			m->type,
			m->description
		);
		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		if(existing_mime_type != NULL)
			response = CDialogGetResponse(
"Update MIME Type",
				msg,
"You are being asked if you want to update the MIME Type.\n\
\n\
MIME Types are used for recognizing different kinds of file\n\
formats and determining how to handle them.\n\
\n\
For more information on MIME Types see Help->MIME Types",
				CDIALOG_ICON_INSTALL,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_YES
			);
		else
			response = CDialogGetResponse(
"Install MIME Type",
				msg,
"You are being asked if you want to install a new MIME Type.\n\
\n\
MIME Types are used for recognizing different kinds of file\n\
formats and determining how to handle them.\n\
\n\
For more information on MIME Types see Help->MIME Types",
				CDIALOG_ICON_INSTALL,
				CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
				CDIALOG_BTNFLAG_YES
			);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		if(response != CDIALOG_RESPONSE_YES)
		{
			edv_mime_type_delete(m);
			return(-4);
		}
	}

	/* Does the MIME Type already exist? */
	if(existing_mime_type != NULL)
	{
		/* Update the existing MIME Type with the values from
		 * the specified MIME Type
		 */
		EDVMIMEType	*tar_m = existing_mime_type,
									*src_m = m;
		if(tar_m != src_m)
		{
			gint i, n;

			tar_m->mt_class = src_m->mt_class;

			g_free(tar_m->value);
			tar_m->value = STRDUP(src_m->value);

			g_free(tar_m->type);
			tar_m->type = STRDUP(src_m->type);

			g_free(tar_m->description);
			tar_m->description = STRDUP(src_m->description);

			n = EDV_MIME_TYPE_TOTAL_ICON_STATES;
			for(i = 0; i < n; i++)
			{
				g_free(tar_m->small_icon_path[i]);
				tar_m->small_icon_path[i] = STRDUP(src_m->small_icon_path[i]);
				g_free(tar_m->medium_icon_path[i]);
				tar_m->medium_icon_path[i] = STRDUP(src_m->medium_icon_path[i]);
				g_free(tar_m->large_icon_path[i]);
				tar_m->large_icon_path[i] = STRDUP(src_m->large_icon_path[i]);
			}

			tar_m->handler = src_m->handler;

			if(tar_m->commands_list != NULL)
			{
				g_list_foreach(
					tar_m->commands_list,
					(GFunc)edv_mime_type_command_delete,
					NULL
				);
				g_list_free(tar_m->commands_list);
				tar_m->commands_list = NULL;
			}
			if(src_m->commands_list != NULL)
			{
				GList *glist;
				for(glist = src_m->commands_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					tar_m->commands_list = g_list_append(
						tar_m->commands_list,
						edv_mime_type_command_copy(EDV_MIME_TYPE_COMMAND(glist->data))
					);
				}
			}

			tar_m->access_time = src_m->access_time;
			tar_m->modify_time = src_m->modify_time;
			tar_m->change_time = src_m->change_time;

		}

		/* Delete the specified MIME Type and transfer the
		 * existing MIME Type pointer to its place
		 */
		edv_mime_type_delete(m);
		m = existing_mime_type;

		/* Re-realize the MIME Type so its GTK resources reflect
		 * the updated values
		 */
		edv_mime_type_realize(
			m,
			TRUE				/* Force re-realize */
		);

		/* Notify about the MIME Type being updated */
		edv_emit_mime_type_modified(core, mt_num, m);
	}
	else
	{
		/* Append or insert the specified MIME Type to the MIME
		 * Types list
		 */
		if(pos < 0)
		{
			mt_num = g_list_length(core->mime_types_list);
			core->mime_types_list = g_list_append(
				core->mime_types_list,
				m
			);
		}
		else
		{
			mt_num = pos;
			core->mime_types_list = g_list_insert(
				core->mime_types_list,
				m,
				pos
			);
		}

		/* Notify about the new MIME Type being installed */
		edv_emit_mime_type_added(core, mt_num, m);
	}

	/* If the MIME Types List Window is currently mapped then
	 * select the installed or updated MIME Type
	 */
	if(EDVMimeTypesListWinIsMapped(lw))
	{
		EDVMimeTypesListWinSetBusy(lw, TRUE);
		EDVMimeTypesListWinSelect(lw, mt_num);
		EDVMimeTypesListWinSetBusy(lw, FALSE);
	}

	/* Save the MIME Types file */
	s = EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES);
	if(!STRISEMPTY(s))
	{
		gchar *path = g_strdup(s);
		if(path != NULL)
		{
			const gboolean object_existed = edv_path_lexists(path);
			EDVVFSObject *obj;
			edv_mime_types_list_file_save(
				core->mime_types_list,
				path,
				FALSE,		/* Do not save read only
						 * MIME Types */
				NULL, core
			);
			obj = edv_vfs_object_lstat(path);
			if(obj != NULL)
			{
				if(object_existed)
					edv_emit_vfs_object_modified(
						core,
						path,
						path,
						obj
					);
				else
					edv_emit_vfs_object_added(
						core,
						path,
						obj
					);
				edv_vfs_object_delete(obj);
			}
			g_free(path);
 	    }
	}

	return(mt_num);
}
