#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "edv_types.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"


/* MIME Type Commands */
const gchar *edv_mime_type_match_command_value_by_name(
	EDVMIMEType *m,
	const gchar *name
);
EDVMIMETypeCommand *edv_mime_type_match_command_by_name(
	EDVMIMEType *m,
	const gchar *name
);

EDVMIMETypeCommand *edv_mime_type_command_new(void);
EDVMIMETypeCommand *edv_mime_type_command_copy(EDVMIMETypeCommand *cmd);
void edv_mime_type_command_delete(EDVMIMETypeCommand *cmd);


/* MIME Types */
EDVMIMEType *edv_mime_type_new_values(void);
EDVMIMEType *edv_mime_type_copy(EDVMIMEType *m);
static void edv_mime_type_load_icons_nexus(
	EDVPixmap **icons_list,
	edv_pixmap_data **datas_list,
	const gchar **paths_list,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
);
void edv_mime_type_realize(
	EDVMIMEType *m,
	const gboolean force_rerealize
);
void edv_mime_type_unrealize(EDVMIMEType *m);
void edv_mime_type_delete(EDVMIMEType *m);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the command value of the MIME Type command that matches
 *	the specified command name.
 */
const gchar *edv_mime_type_match_command_value_by_name(
	EDVMIMEType *m,
	const gchar *name
)
{
	GList *glist;
	EDVMIMETypeCommand *cmd;

	if((m == NULL) || STRISEMPTY(name))
		return(NULL);

	for(glist = m->commands_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
			continue;

		if(cmd->name == NULL)
			continue;

		if(!g_strcasecmp(cmd->name, name))
			return(cmd->command);
	}

	return(NULL);
}

/*
 *	Gets the MIME Type command that matches the specified command
 *	name.
 */
EDVMIMETypeCommand *edv_mime_type_match_command_by_name(
	EDVMIMEType *m,
	const gchar *name
)
{
	GList *glist;
	EDVMIMETypeCommand *cmd;

	if((m == NULL) || STRISEMPTY(name))
		return(NULL);

	for(glist = m->commands_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
			continue;

		if(cmd->name == NULL)
			continue;

		if(!g_strcasecmp(cmd->name, name))
			return(cmd);
	}

	return(NULL);
}


/*
 *	Creates a new MIME Type command.
 */
EDVMIMETypeCommand *edv_mime_type_command_new(void)
{
	return(EDV_MIME_TYPE_COMMAND(g_malloc0(
		sizeof(EDVMIMETypeCommand)
	)));
}

/*
 *	Coppies the MIME Type command.
 */
EDVMIMETypeCommand *edv_mime_type_command_copy(EDVMIMETypeCommand *cmd)
{
	EDVMIMETypeCommand	*tar,
									*src = cmd;
	if(src == NULL)
		return(NULL);

	tar = edv_mime_type_command_new();
	if(tar == NULL)
		return(NULL);

	tar->name = STRDUP(src->name);
	tar->command = STRDUP(src->command);
	tar->shell_command = STRDUP(src->shell_command);
	tar->flags = src->flags;

	return(tar);
}

/*
 *	Deletes the MIME Type command.
 */
void edv_mime_type_command_delete(EDVMIMETypeCommand *cmd)
{
	if(cmd == NULL)
		return;

	g_free(cmd->name);
	g_free(cmd->command);
	g_free(cmd->shell_command);
	g_free(cmd);
}


/*
 *	Creates a new MIME Type.
 */
EDVMIMEType *edv_mime_type_new_values(void)
{
	return(EDV_MIME_TYPE(g_malloc0(sizeof(EDVMIMEType))));
}

/*
 *	Coppies the MIME Type.
 */
EDVMIMEType *edv_mime_type_copy(EDVMIMEType *m)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;
	GList *glist;
	EDVMIMEType	*tar,
							*src = m;
	if(src == NULL)
		return(NULL);

	tar = edv_mime_type_new_values();
	if(tar == NULL)
		return(NULL);

	tar->mt_class = src->mt_class;
	tar->value = STRDUP(src->value);
	tar->type = STRDUP(src->type);
	tar->description = STRDUP(src->description);

	tar->realized = src->realized;
	tar->read_only = src->read_only;

	for(i = 0; i < nicon_states; i++)
	{
		/* Small */
		tar->small_icon[i] = edv_pixmap_ref(src->small_icon[i]);
		tar->small_icon_path[i] = STRDUP(src->small_icon_path[i]);

		/* Medium */
		tar->medium_icon[i] = edv_pixmap_ref(src->medium_icon[i]);
		tar->medium_icon_path[i] = STRDUP(src->medium_icon_path[i]);

		/* Large */
		tar->large_icon[i] = edv_pixmap_ref(src->large_icon[i]);
		tar->large_icon_path[i] = STRDUP(src->large_icon_path[i]);
	}

	tar->handler = src->handler;

	tar->commands_list = NULL;
	for(glist = src->commands_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
		tar->commands_list = g_list_append(
			tar->commands_list,
			edv_mime_type_command_copy(EDV_MIME_TYPE_COMMAND(
				glist->data
			))
		);

	tar->access_time = src->access_time;
	tar->modify_time = src->modify_time;
	tar->change_time = src->change_time;

	return(tar);
}

/*
 *	Loads the pixmap and mask pairs list.
 *
 *	The pixmaps_list and masks_list specifies the pixmap and mask
 *	pairs list, any existing (non NULL) pixmap and mask pairs in
 *	this list will be unref'ed.
 *
 *	The data specifies the XPM data. If data is not NULL then the
 *	pixmap and mask pair will be loaded from the XPM data.
 *
 *	The file specifies the path to the XPM file. If file is not NULL
 *	then the pixmap and mask pair will be loaded from the XPM file.
 *
 *	If both data and file are NULL then the existing pixmap and
 *	mask pair will be unref'ed and no new pixmap and mask pair will
 *	be loaded.
 *
 *	The req_width and req_height specifies the requested size in
 *	pixels.
 *
 *	If allow_resize is TRHE then the pixmap and mask pair will be
 *	resized to the requested size as needed.
 */
static void edv_mime_type_load_icons_nexus(
	EDVPixmap **icons_list,
	edv_pixmap_data **datas_list,
	const gchar **paths_list,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;

	/* Unref the existing EDVPixmap icons */
	for(i = 0; i < nicon_states; i++)
		icons_list[i] = edv_pixmap_unref(icons_list[i]);

	/* Load the icons from data? */
	if(datas_list != NULL)
	{
		EDVPixmap *p;

		/* Load each icon from data */
		for(i = 0; i < nicon_states; i++)
		{
			/* Open a new EDVPixmap from this data
			 *
			 * We do not use edv_load_pixmap_from_data() because it
			 * may give us an existing EDVPixmap and we need to
			 * potentially resize it which would cause problems
			 * for other parts of the code that want the original
			 * size of the EDVPixmap
			 */
			icons_list[i] = p = edv_pixmap_new_from_data(
				datas_list[i],
				NULL			/* No name */
			);
			if(p == NULL)
				continue;

			if(allow_resize)
				edv_pixmap_resize(
					p,
					req_width, req_height
				);
		}
	}
	/* Open the icons from file? */
	else if(paths_list != NULL)
	{
		EDVPixmap *p;

		/* Open each icon from file */
		for(i = 0; i < nicon_states; i++)
		{
			/* Open a new EDVPixmap from this file
			 *
			 * We do not use edv_open_pixmap_from_file() because it
			 * may give us an existing EDVPixmap and we need to
			 * potentially resize it which would cause problems
			 * for other parts of the code that want the original
			 * size of the EDVPixmap
			 */
			icons_list[i] = p = edv_pixmap_new_from_file(paths_list[i]);
			if(p == NULL)
				continue;

			if(allow_resize)
				edv_pixmap_resize(
					p,
					req_width, req_height
				);
		}
	}
}

/*
 *	Opens all the icon EDVPixmaps from their respective icon file
 *	paths on the MIME Type.
 *
 *	If force_rerealize is TRUE and the MIME Type is already realized
 *	then all the existing icon EDVPixmaps will be unrefed and then
 *	reopened from their respective icon file paths.
 */
void edv_mime_type_realize(
	EDVMIMEType *m,
	const gboolean force_rerealize
)
{
	const gchar **paths_list;

	if(m == NULL)
		return;

	/* Not forcing a re-realize? */
	if(!force_rerealize)
	{
		/* Skip if this MIME Type was already realized */
		if(m->realized)
			return;
	}

	/* Begin realizing this MIME Type */

	/* (Re)open the icons from their icon files
	 *
	 * Small icons
	 */
	paths_list = (const gchar **)m->small_icon_path;
	if(paths_list != NULL)
		edv_mime_type_load_icons_nexus(
			m->small_icon,
			NULL,			/* No data */
			paths_list,
			20, 20,
			TRUE			/* Allow resize */
		);

	/* Medium icons */
	paths_list = (const gchar **)m->medium_icon_path;
	if(paths_list != NULL)
		edv_mime_type_load_icons_nexus(
			m->medium_icon,
			NULL,			/* No data */
			paths_list,
			32, 32,
			TRUE			/* Allow resize */
		);

	/* Large icons */
	paths_list = (const gchar **)m->large_icon_path;
	if(paths_list != NULL)
		edv_mime_type_load_icons_nexus(
			m->large_icon,
			NULL,			/* No data */
			paths_list,
			48, 48,
			TRUE			/* Allow resize */
		);

	/* Mark this MIME Type as realized */
	m->realized = TRUE;
}

/*
 *	Unrefs all the icon EDVPixmaps on the MIME Type.
 */
void edv_mime_type_unrealize(EDVMIMEType *m)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;

	if(m == NULL)
		return;

	/* Skip if not already marked as realized */
	if(!m->realized)
		return;

	for(i = 0; i < nicon_states; i++)
	{
		m->small_icon[i] = edv_pixmap_unref(m->small_icon[i]);
		m->medium_icon[i] = edv_pixmap_unref(m->medium_icon[i]);
		m->large_icon[i] = edv_pixmap_unref(m->large_icon[i]);
	}

	/* Mark this MIME Type as no longer realized */
	m->realized = FALSE;
}

/*
 *	Deletes the MIME Type.
 */
void edv_mime_type_delete(EDVMIMEType *m)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;

	if(m == NULL)
		return;

	for(i = 0; i < nicon_states; i++)
	{
		m->small_icon[i] = edv_pixmap_unref(m->small_icon[i]);
		g_free(m->small_icon_path[i]);

		m->medium_icon[i] = edv_pixmap_unref(m->medium_icon[i]);
		g_free(m->medium_icon_path[i]);

		m->large_icon[i] = edv_pixmap_unref(m->large_icon[i]);
		g_free(m->large_icon_path[i]);
	}

	if(m->commands_list != NULL)
	{
		g_list_foreach(
			m->commands_list,
			(GFunc)edv_mime_type_command_delete,
			NULL
		);
		g_list_free(m->commands_list);
	}

	g_free(m->value);
	g_free(m->type);
	g_free(m->description);

	g_free(m);
}
