#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "cfg.h"
#include "cfg_fio.h"

#include "edv_main.h"
#include "edv_types.h"
#include "libendeavour2-base/edv_interps.h"
#include "edv_mime_types_list.h"
#include "edv_devices_list.h"
#include "edv_core.h"
#include "edv_cb.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Global variable declarations go here:
 */
EDVCore		*edv_core_ptr;


gint edv_main(const gint argc, const gchar **argv);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Runs Endeavour Mark II normally.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_main(const gint argc, const gchar **argv)
{
	gint error_code;
	CfgList *cfg_list;
	EDVCore *core;

	/* Reset the global contexts */
	edv_core_ptr = NULL;

	/* Set up the UNIX signal callbacks */
#ifdef SIGHUP
	signal(SIGHUP, edv_signal_cb);
#endif
#ifdef SIGINT
	signal(SIGINT, edv_signal_cb);
#endif
#ifdef SIGTERM
	signal(SIGTERM, edv_signal_cb);
#endif
#ifdef SIGQUIT
	signal(SIGQUIT, edv_signal_cb);
#endif
#ifdef SIGSEGV
	signal(SIGSEGV, edv_signal_cb);
#endif
#ifdef SIGUSR1
	signal(SIGUSR1, edv_signal_cb);
#endif
#ifdef SIGUSR2
	signal(SIGUSR2, edv_signal_cb);
#endif
#ifdef SIGPIPE
	signal(SIGPIPE, edv_signal_cb);
#endif

	/* Create the Endeavour Mark II core, initialize GTK and its
	 * resources, display the splash, open the configuration files,
	 * display any startup messages, create the initial window,
	 * and set the main callbacks
	 *
	 * If the InterPS lock link existed (if another process of this
	 * program was running) then an InterPS command would have been
	 * sent and NULL would be returned with error_code == 0
	 */
	edv_core_ptr = core = edv_core_new_init(
		argc, argv,
		&error_code
	);
	if(core == NULL)
		return(error_code);

	/* Enter the main GTK loop */
	gtk_main();

	/* Remove the timeout callbacks, InterPS lock link, and any
	 * InterPS commands so that this process is marked as no
	 * longer accepting or processing InterPS commands and no
	 * further InterPS commands should be sent to this process
	 *
	 * This may result in commands not being processed if commands
	 * were queued in the time between our last timeout callback
	 * and this point
	 */
	edv_core_remove_timeouts(core);
	if(edv_interps_get_lock(core->cfg_list) == core->pid)
	{
		edv_interps_remove_commands(core->cfg_list);
		edv_interps_remove_lock(core->cfg_list);
	}

	cfg_list = core->cfg_list;

	/* Query the user to unmount any mounted removable devices? */
	if(EDV_GET_B(EDV_CFG_PARM_QUERY_UNMOUNT_BEFORE_EXIT))
		edv_query_unmount_before_exit(core);

	/* Save the MIME Types List */
	edv_mime_types_list_file_save(
		core->mime_types_list,
		EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES),
		FALSE,				/* Do not save MIME Types
						 * marked read_only */
		NULL, core
	);

	/* Save the Devices List */
	edv_devices_list_file_save(
		core->devices_list,
		EDV_GET_S(EDV_CFG_PARM_FILE_DEVICES),
		NULL, core
	);

	/* Save the configuration */
	CFGFileSave(
		core->cfg_path,
		core->cfg_list
	);

	/* Unset any main callbacks, remove the InterPS lock link,
	 * delete all the windows and lists, shut down GTK and its
	 * resources if it was initialized, and delete the rest of the
	 * core
	 */
	edv_core_ptr = NULL;			/* Unset the global context */
	edv_core_delete(core);
/*	core = NULL; */

	return(0);
}
