#include <stdlib.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_interps.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_utils_gtk.h"
#include "edv_help.h"
#include "edv_mime_type_install.h"
#include "edv_open.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "edv_interps_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


static void edv_interps_op_process_command_iterate(
	EDVCore *core,
	const gchar *cmd
);

void edv_interps_op_process_commands(
	EDVCore *core,
	gchar **cmd_list
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Processes the InterPS command.
 *
 *	The cmd specifies the InterPS command.
 */
static void edv_interps_op_process_command_iterate(
	EDVCore *core,
	const gchar *cmd
)
{
	gint argc;
	gchar **argv;
	const gchar *op;
	CfgList *cfg_list = core->cfg_list;

	if(STRISEMPTY(cmd))
		return;

	/* Explode command into arguments
	 *
	 * Note that we use ExecExplodeCommand() to handle arguments
	 * wraped in '"' characters
	 */
	argv = edv_strexp_args(
		cmd,
		FALSE,				/* Do not parse escapes */
		TRUE				/* Parse quotes */
	);
	if(argv == NULL)
		return;

	/* Get the operation (first argument) */
	op = argv[0];
	if(op == NULL)
	{
		g_strfreev(argv);
		return;
	}

	/* Count the number of arguments */
	for(argc = 0; argv[argc] != NULL; argc++);

	/* Begin handling the operation
	 *
	 * New Window */
	if(!g_strcasecmp(op, "new_window"))
	{
		/* Format:
		 *
		 * <EDVWindowType> [path] [extra] [geometry]
		 *
		 * Note that the "new_window" command is always sent from
		 * another Endeavour process and not by another program
		 * using the Endeavour API library because Endeavour
		 * might not be running when an Endeavour window needs to
		 * be created/mapped and therefore Endeavour should
		 * always be runned in order to create/map a window
		 */
		if(argc >= 2)
		{
			const EDVWindowType win_type = (EDVWindowType)ATOI(argv[1]);
			const gchar	*startup_path = (argc > 2) ? argv[2] : NULL,
					*window_extra_arg = (argc > 3) ? argv[3] : NULL,
					*geometry_str = (argc > 4) ? argv[4] : NULL;
			gint	x, y,
				width, height;
			const GdkGeometryFlags geometry_flags = gdk_parse_geometry(
				geometry_str,
				&x, &y,
				&width, &height
			);
			GdkRectangle geometry;

			geometry.x = (geometry_flags & GDK_GEOMETRY_X) ? x : 0;
			geometry.y = (geometry_flags & GDK_GEOMETRY_Y) ? y : 0;
			geometry.width = (geometry_flags & GDK_GEOMETRY_WIDTH) ? width : 0;
			geometry.height = (geometry_flags & GDK_GEOMETRY_HEIGHT) ? height : 0;

			/* Create the new window */
			edv_new_window(
				core,
				win_type,
				startup_path,
				window_extra_arg,
				geometry_flags,
				(geometry_flags != 0) ? &geometry : NULL,
				NULL			/* No toplevel */
			);
		}
	}
	/* Set Cfg Integer */
	else if(!g_strcasecmp(op, "set_cfg_i"))
	{
		if(argc >= 3)
		{
			const gchar *parm = argv[1];
			const gint val = ATOI(argv[2]);
			EDV_SET_I(parm, val);
		}
	}
	/* Set Cfg Unsigned Integer */
	else if(!g_strcasecmp(op, "set_cfg_ui"))
	{
		if(argc >= 3)
		{
			const gchar *parm = argv[1];
			const guint val = (guint)ATOI(argv[2]);
			EDV_SET_I(parm, (gint)val);
		}
	}
	/* Set Cfg Long */
	else if(!g_strcasecmp(op, "set_cfg_l"))
	{
		if(argc >= 3)
		{
			const gchar *parm = argv[1];
			const glong val = ATOL(argv[2]);
			EDV_SET_L(parm, val);
		}
	}
	/* Set Cfg Unsigned Long */
	else if(!g_strcasecmp(op, "set_cfg_ul"))
	{
		if(argc >= 3)
		{
			const gchar *parm = argv[1];
			const gulong val = ATOL(argv[2]);
			EDV_SET_UL(parm, val);
		}
	}
	/* Set Cfg Float */
	else if(!g_strcasecmp(op, "set_cfg_f"))
	{
		if(argc >= 3)
		{
			const gchar *parm = argv[1];
			const gfloat val = ATOF(argv[2]);
			EDV_SET_F(parm, val);
		}
	}
	/* Set Cfg Double */
	else if(!g_strcasecmp(op, "set_cfg_d"))
	{
		if(argc >= 3)
		{
			const gchar *parm = argv[1];
			const gdouble val = (gdouble)ATOF(argv[2]);
			EDV_SET_D(parm, val);
		}
	}
	/* Set Cfg String */
	else if(!g_strcasecmp(op, "set_cfg_s"))
	{
		if(argc >= 3)
		{
			const gchar *parm = argv[1];
			const gchar *val = argv[2];
			EDV_SET_S(parm, val);
		}
	}
	/* Write Protect Changed Notify */
	else if(!g_strcasecmp(op, "write_protect_changed_notify"))
	{
		edv_emit_master_write_protect_changed(core);
	}
	/* VFS Object Added Notify */
	else if(!g_strcasecmp(op, "vfs_object_added_notify"))
	{
		if(argc >= 2)
		{
			const gchar *path = argv[1];
			if(!STRISEMPTY(path))
			{
				EDVVFSObject *obj = edv_vfs_object_lstat(path);
				if(obj != NULL)
				{
					edv_emit_vfs_object_added(
						core,
						path,
						obj
					);
					edv_vfs_object_delete(obj);
				}
			}
		}
	}
	/* VFS Object Modified Notify */
	else if(!g_strcasecmp(op, "vfs_object_modified_notify"))
	{
		if(argc >= 2)
		{
			const gchar	*path = argv[1],
							*new_path = (argc > 2) ? argv[2] : NULL;
			if(!STRISEMPTY(path))
			{
				EDVVFSObject *obj = edv_vfs_object_lstat(
					STRISEMPTY(new_path) ? path : new_path
				);
				if(obj != NULL)
				{
					edv_emit_vfs_object_modified(
						core,
						path,
						new_path,
						obj
					);
					edv_vfs_object_delete(obj);
				}
			}
		}
	}
	/* VFS Object Removed Notify */
	else if(!g_strcasecmp(op, "vfs_object_removed_notify"))
	{
		if(argc >= 2)
		{
			const gchar *path = argv[1];
			if(!STRISEMPTY(path))
				edv_emit_vfs_object_removed(
					core,
					path
				);
		}
	}
	/* VFS Object Mounted Notify or Object Unmounted Notify */
	else if(!g_strcasecmp(op, "vfs_object_mounted_notify") ||
		    !g_strcasecmp(op, "vfs_object_unmounted_notify")
	)
	{
		if(argc >= 2)
		{
			const gchar *path = argv[1];
			gint dev_num;
			GList *glist;
			EDVDevice *d;

			/* The path could be a mount path or device path, see
			 * which one it is by matching it in our list of device
			 * references
			 */
			for(glist = core->devices_list, dev_num = 0;
				glist != NULL;
				glist = g_list_next(glist), dev_num++
			)
			{
				d = EDV_DEVICE(glist->data);
				if(d == NULL)
					continue;

				if(!STRISEMPTY(d->mount_path) &&
				   !STRISEMPTY(path)
				)
				{
					if(!strcmp((const char *)d->mount_path, (const char *)path))
						break;
				}
				if(!STRISEMPTY(d->device_path) &&
				   !STRISEMPTY(path)
				)
				{
					if(!strcmp((const char *)d->device_path, (const char *)path))
						break;
				}
			}
			/* Got match? */
			if(glist != NULL)
			{
				/* Need to update mount states and device capacities */
				edv_devices_list_update_mount_states(core->devices_list);
				edv_devices_list_update_statistics(core->devices_list);
				/* Emit mount or unmount signal */
				edv_emit_device_mount(
					core,
					dev_num,
					d,
					EDV_DEVICE_IS_MOUNTED(d)
				);
			}
		}
	}
	/* Recycled Object Added Notify */
	else if(!g_strcasecmp(op, "recycled_object_added_notify"))
	{
		if(argc >= 2)
		{
			const guint index = ATOI(argv[1]);
			edv_emit_recycled_object_added(
				core,
				index
			);
		}
	}
	/* Recycled Object Modified Notify */
	else if(!g_strcasecmp(op, "recycled_object_modified_notify"))
	{
		if(argc >= 2)
		{
			const guint index = ATOI(argv[1]);
			edv_emit_recycled_object_modified(
				core,
				index
			);
		}
	}
	/* Recycled Object Removed Notify */
	else if(!g_strcasecmp(op, "recycled_object_removed_notify"))
	{
		if(argc >= 2)
		{
			const guint index = ATOI(argv[1]);
			edv_emit_recycled_object_removed(
				core,
				index
			);
		}
	}
	/* Archive Object Added Notify */
	else if(!g_strcasecmp(op, "archive_object_added_notify"))
	{
		if(argc >= 3)
		{
#if 0
			edv_emit_archive_object_added(
				core,
				argv[1],			/* Archive path */
				argv[2],			/* Object path */
				NULL
			);
#endif
		}
	}
	/* Archive Object Modified Notify */
	else if(!g_strcasecmp(op, "archive_object_modified_notify"))
	{
		if(argc >= 4)
		{
#if 0
			edv_emit_archive_object_modified)
				core,
				argv[1],			/* Archive path */
				argv[2],			/* Object path */
				argv[3],			/* Object new path */
				NULL
			);
#endif
		}
	}
	/* Archive Object Removed Notify */
	else if(!g_strcasecmp(op, "archive_object_removed_notify"))
	{
		if(argc >= 2)
		{
			edv_emit_archive_object_removed(
				core,
				argv[1],			/* Archive path */
				argv[2]			/* Object path */
			);
		}
	}
	/* Reconfigured Notify */
	else if(!g_strcasecmp(op, "reconfigured_notify"))
	{
		edv_emit_reconfigured(core);
	}

	/* Append History */
	else if(!g_strcasecmp(op, "append_history"))
	{
		/* Arguments:
		 *
		 * <type> <time_start> <time_end> <status>
		 * [source] [target] [comments]
		 */
		if(argc >= 5)
		{
			EDVHistoryType type = (EDVHistoryType)ATOI(argv[1]);
			gulong time_start = (gulong)ATOL(argv[2]);
			gulong time_end = (gulong)ATOL(argv[3]);
			gint status = ATOI(argv[4]);
			const gchar *source = (argc > 5) ? argv[5] : NULL;
			const gchar *target = (argc > 6) ? argv[6] : NULL;
			const gchar *comments = (argc > 7) ? argv[7] : NULL;

			/* Append history */
			edv_append_history(
				core,
				type,
				time_start,
				time_end,
				status,
				source,
				target,
				comments
			);
		}
	}
	/* Install MIME Type */
	else if(!g_strcasecmp(op, "install_mimetype"))
	{
		/* Arguments:
		 *
		 * <class> <type>
		 * [value] [description]
		 * [handler]
		 * [icon_small_std_file] [icon_small_sel_file]
		 *  [icon_small_ext_file] [icon_small_hid_file]
		 * [icon_medium_std_file] [icon_medium_sel_file]
		 *  [icon_medium_ext_file] [icon_medium_hid_file]
		 * [icon_large_std_file] [icon_large_sel_file]
		 *  [icon_large_ext_file] [icon_large_hid_file]
		 * [command_name] [command] [...]
		 */
		if(argc >= 3)
		{
			GtkWidget *toplevel = NULL;
			edv_mime_types_list_win_struct *lw = core->mime_types_list_win;
			EDVMIMEType *m = edv_mime_type_new_values();
			if(m != NULL)
			{
				m->mt_class = (EDVMIMETypeClass)ATOI(argv[1]);
				m->value = STRDUP((argc > 3) ? argv[3] : NULL);
				m->type = STRDUP(argv[2]);
				m->description = STRDUP((argc > 4) ? argv[4] : NULL);

				/* Handler (arg 5) */
				m->handler = (argc > 5) ?
					ATOI(argv[5]) : EDV_MIME_TYPE_HANDLER_COMMAND;

				/* Icon files (args 6 to 17) */
				if(argc > 17)
				{
					m->small_icon_path[
						EDV_MIME_TYPE_ICON_STATE_STANDARD
					] = STRDUP(argv[6]);
					m->small_icon_path[
						EDV_MIME_TYPE_ICON_STATE_OPENED
					] = STRDUP(argv[7]);
					m->small_icon_path[
						EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
					] = STRDUP(argv[8]);
					m->small_icon_path[
						EDV_MIME_TYPE_ICON_STATE_HIDDEN
					] = STRDUP(argv[9]);

					m->medium_icon_path[
						EDV_MIME_TYPE_ICON_STATE_STANDARD
					] = STRDUP(argv[10]);
					m->medium_icon_path[
						EDV_MIME_TYPE_ICON_STATE_OPENED
					] = STRDUP(argv[11]);
					m->medium_icon_path[
						EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
					] = STRDUP(argv[12]);
					m->medium_icon_path[
						EDV_MIME_TYPE_ICON_STATE_HIDDEN
					] = STRDUP(argv[13]);

					m->large_icon_path[
						EDV_MIME_TYPE_ICON_STATE_STANDARD
					] = STRDUP(argv[14]);
					m->large_icon_path[
						EDV_MIME_TYPE_ICON_STATE_OPENED
					] = STRDUP(argv[15]);
					m->large_icon_path[
						EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
					] = STRDUP(argv[16]);
					m->large_icon_path[
						EDV_MIME_TYPE_ICON_STATE_HIDDEN
					] = STRDUP(argv[17]);
				}

				/* Commands (args 18 to ...) */
				if(argc > 18)
				{
					gint i = 18;
					EDVMIMETypeCommand *cmd;
					while(i < argc)
					{
						cmd = edv_mime_type_command_new();
						if(cmd == NULL)
							break;

						cmd->name = (argc > i) ?
							STRDUP(argv[i]) : NULL;
						i++;
						cmd->command = (argc > i) ?
							STRDUP(argv[i]) : NULL;
						i++;

						m->commands_list = g_list_append(
							m->commands_list,
							cmd
						);
					}
				}

				if(EDVMimeTypesListWinIsMapped(lw))
					toplevel = lw->toplevel;

				/* Install the MIME Type
				 *
				 * The MIME Type passed to this function will be
				 * installed (transfered) and should not be referenced
				 * again
				 */
				edv_mime_type_install(
					core,
					m,
					-1,			/* Append */
					TRUE,			/* Interactive */
					TRUE,			/* Verbose */
					toplevel
				);
			}
		}
	}

	/* Help */
	else if(!g_strcasecmp(op, "help"))
	{
		const gchar *subject = (argc > 1) ? argv[1] : NULL;
		edv_help(
			core,
			subject,		/* Subject */
		    NULL			/* No toplevel */
		);
	}
	/* About */
	else if(!g_strcasecmp(op, "about"))
	{
		const gchar	*page_name = (argc > 1) ? argv[1] : NULL,
					*geometry_str = (argc > 2) ? argv[2] : NULL;
		gint x, y, width, height;
		const GdkGeometryFlags geometry_flags = gdk_parse_geometry(
			geometry_str,
			&x, &y,
			&width, &height
		);
		GdkRectangle geometry;

		geometry.x = (geometry_flags & GDK_GEOMETRY_X) ? x : 0;
		geometry.y = (geometry_flags & GDK_GEOMETRY_Y) ? y : 0;
		geometry.width = (geometry_flags & GDK_GEOMETRY_WIDTH) ? width : 0;
		geometry.height = (geometry_flags & GDK_GEOMETRY_HEIGHT) ? height : 0;

		edv_map_about_dialog_page(
			core,
			page_name,			/* Page Name */
			geometry_flags,
			(geometry_flags != 0) ? &geometry : NULL,
			NULL				/* No toplevel */
		);
	}
	/* All else display error message */
	else
	{
		gchar *msg = g_strdup_printf(
"Unsupported InterPS command:\n\
\n\
    %s\n\
\n\
Was received by this application.",
			op
		);
		edv_play_sound_warning(core);
		edv_message_warning(
"InterPS Command Warning",
msg,
"InterPS (Inter Process) commands are sent between different\n\
processes of this same application to ensure that only one copy of\n\
this application's process exists per user.\n\
\n\
If you are writing a program to send InterPS to this application,\n\
then you should contact the authors for assistance, otherwise you\n\
should regard this message as an internal programming error.\n",
			NULL
		);
		g_free(msg);
	}

	/* Delete the arguments */
	g_strfreev(argv);
}

/*
 *	Processes the specified InterPS commands.
 *
 *	The cmd_list specifies the list of command strings. The last
 *	pointer in the array must be NULL to mark the end of the list.
 */
void edv_interps_op_process_commands(
	EDVCore *core,
	gchar **cmd_list
)
{
	gint i;

	if((core == NULL) || (cmd_list == NULL))
		return;

	for(i = 0; cmd_list[i] != NULL; i++)
		edv_interps_op_process_command_iterate(core, cmd_list[i]);
}
