#include <glib.h>

#include "imgio.h"
#include "imgio_formats.h"

#include "guirgbimg.h"

#include "libendeavour2-base/edv_path.h"
#include "edv_image_io.h"


typedef struct _EDVImageIOProgressData	EDVImageIOProgressData;
#define EDV_IMAGE_IO_PROGRESS_DATA(p)	((EDVImageIOProgressData *)(p))


static int edv_image_io_progress_cb(
	void *progress_data,
	const int i, const int m,
	const int width, const int height,
	const int bpl, const int bpp,
	const u_int8_t *data
);


/* Check Type */
gboolean edv_image_is_supported_extension(const gchar *ext);

/* Open RGBA */
GList *edv_image_open_rgba(
	const gchar *path,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color,
	gint (*progress_cb)(
		const gint, const gint,		/* Width, height */
		const gint, const gint,		/* BPP, BPL */
		const guint8 *,			/* Image data */
		const gulong, const gulong,	/* Progress min, max */
		gpointer
	),
	gpointer progress_data
);

/* Open RGBA Details */
GList *edv_image_open_rgba_details(
	const gchar *path,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn,          /* In seconds since EPOCH */
	gint (*progress_cb)(
		const gint, const gint,         /* Width, height */
		const gint, const gint,         /* BPP, BPL */
		const guint8 *,                 /* RGBA image data */
		const gulong, const gulong,     /* Progress min, max */
		gpointer
	),
	gpointer progress_data
);

/* Open RGBA Thumb */
guint8 *edv_image_open_rgba_thumb(
	const gchar *path,
	const gint req_width, const gint req_height,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	guint8 *bg_color,
	gint *orig_width_rtn, gint *orig_height_rtn,
	gint *orig_nframes_rtn,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn,		/* In seconds since EPOCH */
	gint (*progress_cb)(
		const gint, const gint,		/* Width, height */
		const gint, const gint,		/* BPP, BPL */
		const guint8 *,			/* Image data */
		const gulong, const gulong,	/* Progress min, max */
		gpointer
	),
	gpointer progress_data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Progress Data:
 */
struct _EDVImageIOProgressData {
	gint (*progress_cb)(
		const gint, const gint,		/* Width, height */
		const gint, const gint,		/* BPP, BPL */
		const guint8 *,			/* RGBA image data */
		const gulong, const gulong,	/* Progress min, max */
		gpointer
	);
	gpointer progress_data;
};


/*
 *	Progress callback.
 */
static int edv_image_io_progress_cb(
	void *progress_data,
	const int i, const int m,
	const int width, const int height,
	const int bpl, const int bpp,
	const u_int8_t *data
)
{
	EDVImageIOProgressData *d = EDV_IMAGE_IO_PROGRESS_DATA(progress_data);
	if(d == NULL)
		return(FALSE);

	/* Call the user set callback? */
	if(d->progress_cb != NULL)
		return(!d->progress_cb(
			width, height,
			bpp, bpl,
			data,
			i, m,
			d->progress_data
		));
	else
		return(TRUE);
}


/*
 *	Checks if the extension belongs to a supported image format.
 *
 *	The ext specifies the string describing the extension (includes
 *	the "." character).
 */
gboolean edv_image_is_supported_extension(const gchar *ext)
{
	return(ImgIsSupportedExtension(ext));
}


/* 
 *	Opens the image file to RGBA image data.
 *
 *	The path specifies the image file to be opened.
 *
 *	The width_rtn and height_rtn specifies the image geometry
 *	return values.
 *
 *	The bpl_rtn specifies the bytes per line return value.
 *
 *	The nframes_rtn specifies the number of frames return value.
 *
 *	The delay_list_rtn specifies the delay GList return value.
 *	The value in the GList will be a gulong.
 *
 *	The bg_color specifies the buffer for the return value of
 *	the 4 bytes RGBA background color.
 *
 *	Returns a list of guint8 * RGBA image datas or NULL on error.
 */
GList *edv_image_open_rgba(
	const gchar *path,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color,
	gint (*progress_cb)(
		const gint, const gint,		/* Width, height */
		const gint, const gint,		/* BPP, BPL */
		const guint8 *,			/* Image data */
		const gulong, const gulong,	/* Progress min, max */
		gpointer
	),
	gpointer progress_data
)
{
	return(edv_image_open_rgba_details(
		path,
		width_rtn, height_rtn,
		bpl_rtn,
		nframes_rtn,
		delay_list_rtn,
		bg_color,
		NULL,
		NULL, NULL, NULL, NULL,
		NULL,
		progress_cb, progress_data
	));
}

/* 
 *	Opens the image file to RGBA image data.
 *
 *	The path specifies the image file to be opened.
 *
 *	The width_rtn and height_rtn specifies the image geometry
 *	return values.
 *
 *	The bpl_rtn specifies the bytes per line return value.
 *
 *	The nframes_rtn specifies the number of frames return value.
 *
 *	The delay_list_rtn specifies the delay GList return value.
 *	The value in the GList will be a gulong.
 *
 *	The bg_color specifies the buffer for the return value of
 *	the 4 bytes RGBA background color.
 *
 *	Returns a list of guint8 * RGBA image datas or NULL on error.
 */
GList *edv_image_open_rgba_details(
	const gchar *path,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn,          /* In seconds since EPOCH */
	gint (*progress_cb)(
		const gint, const gint,         /* Width, height */
		const gint, const gint,         /* BPP, BPL */
		const guint8 *,                 /* RGBA image data */
		const gulong, const gulong,     /* Progress min, max */
		gpointer
	),
	gpointer progress_data
)
{
	gint		bpl = 0,
			nframes = 0,
			x = 0, y = 0,
			width = 0, height = 0,
			base_width = 0, base_height = 0;
	gchar		*creator = NULL,
			*title = NULL,
			*author = NULL,
			*comments = NULL;
	guint8		**imgio_rgba_list = NULL,
			def_alpha_value = 0xFF,
			def_bg_color[4];
	gulong		*imgio_delay_list = NULL,
			modified_time_sec = 0l;
	GList *rgba_list;
	EDVImageIOProgressData cb_data;

	/* Reset the returns */
	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;
	if(bpl_rtn != NULL)
		*bpl_rtn = 0;
	if(nframes_rtn != NULL)
		*nframes_rtn = 0;
	if(delay_list_rtn != NULL)
		*delay_list_rtn = NULL;
	if(play_time_ms_rtn != NULL)
		*play_time_ms_rtn = 0l;
	if(creator_rtn != NULL)
		*creator_rtn = NULL;
	if(title_rtn != NULL)
		*title_rtn = NULL;
	if(author_rtn != NULL)
		*author_rtn = NULL;
	if(comments_rtn != NULL)
		*comments_rtn = NULL;
	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = 0l;

	if(STRISEMPTY(path))
		return(NULL);

	/* Set the callback data */
	cb_data.progress_cb = progress_cb;
	cb_data.progress_data = progress_data;

	/* If no background color was specified then use the default
	 * background color
	 */
	if(bg_color == NULL)
	{
		bg_color = def_bg_color;
		bg_color[0] = 0xFF;
		bg_color[1] = 0xFF;
		bg_color[2] = 0xFF;
		bg_color[3] = 0xFF;
	}

	rgba_list = NULL;

	/* Open the image in RGBA */
	(void)ImgFileOpenRGBA(
		path,
		&width, &height,
		&bpl,
		&imgio_rgba_list,
		&imgio_delay_list,
		&nframes,
		bg_color,			/* 4 bytes RGBA */
		&x, &y,
		&base_width, &base_height,
		&creator, &title, &author, &comments,
		&modified_time_sec,
		def_alpha_value,
		edv_image_io_progress_cb, &cb_data
	);
	if(nframes > 0)
	{
		gint i;

		/* Update the return values */
		if(width_rtn != NULL)
			*width_rtn = width;
		if(height_rtn != NULL)
			*height_rtn = height;
		if(bpl_rtn != NULL)
			*bpl_rtn = bpl;
		if(nframes_rtn != NULL)
			*nframes_rtn = nframes;

		/* Add/convert each frame to the return lists */
		for(i = 0; i < nframes; i++)
		{
			/* Add this frame to the return list */
			rgba_list = g_list_append(
				rgba_list,
				imgio_rgba_list[i]
			);
/*			imgio_rgba_list[i] = NULL; */

			/* Add the delay to the return list if a delay list
			 * was obtained
			 */
			if(imgio_delay_list != NULL)
			{
				const gulong delay_ms = imgio_delay_list[i];
				if(delay_list_rtn != NULL)
					*delay_list_rtn = g_list_append(
						*delay_list_rtn,
						(gpointer)((guint)delay_ms)
					);
				if(play_time_ms_rtn != NULL)
					*play_time_ms_rtn = (*play_time_ms_rtn) + delay_ms;
			}
		}

		if(creator_rtn != NULL)
		{
			*creator_rtn = creator;
			creator = NULL;
		}
		if(title_rtn != NULL)
		{
			*title_rtn = title;
			title = NULL;
		}
		if(author_rtn != NULL)
		{
			*author_rtn = author;
			author = NULL;
		}
		if(comments_rtn != NULL)
		{
			*comments_rtn = comments;
			comments = NULL;
		}
		if(modified_time_sec_rtn != NULL)
			*modified_time_sec_rtn = modified_time_sec;
	}

	g_free(imgio_rgba_list);
	g_free(imgio_delay_list);

	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(rgba_list);
}

/*
 *	Opens the image in RGBA for a thumb.
 *
 *	The image file will be opened and read at a smaller size, if
 *	possible, which is designed to improve efficiency of opening
 *	the image file. If the size of the original image is smaller
 *	than the requested size then the image will not be enlarged,
 *	and thus, be opened as its actual size. Not all image formats
 *	can be opened at a smaller size, in which case the image
 *	data may still be opened at a larged size.
 *
 *	The path specifies the image to be opened.
 *
 *	The req_width and req_height specifies the requested thumb
 *	size. Note that the actual opened image data size may still be
 *	larged in cases where opening the image as a thumb is not
 *	supported by the image's format.
 *
 *	The width_rtn and height_rtn specifies the return values for
 *	the image size.
 *
 *	The bpl_rtn specifies the return value for the bytes per line.
 *
 *	The bg_color specifies the buffer for the return value of
 *	the 4 bytes RGBA background color.
 *
 *	The orig_width_rtn and orig_height_rtn specifies the return
 *	values for the image's original size.
 *
 *	The orig_nframes_rtn specifies the reutrn value for the image's
 *	original number of frames.
 *
 *	Returns the RGBA image data or NULL on error.
 */
guint8 *edv_image_open_rgba_thumb(
	const gchar *path,
	const gint req_width, const gint req_height,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	guint8 *bg_color,
	gint *orig_width_rtn, gint *orig_height_rtn,
	gint *orig_nframes_rtn,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn,		/* In seconds since EPOCH */
	gint (*progress_cb)(
		const gint, const gint,		/* Width, height */
		const gint, const gint,		/* BPP, BPL */
		const guint8 *,			/* Image data */
		const gulong, const gulong,	/* Progress min, max */
		gpointer
	),
	gpointer progress_data
)
{
	gboolean	specific_loader_used = FALSE;
	gint		x = 0, y = 0,
			width = 0, height = 0,
			bpl = 0,
			base_width = 0, base_height = 0,
			orig_width = 0, orig_height = 0,
			nframes = 0;
	gchar		*creator = NULL,
			*title = NULL,
			*author = NULL,
			*comments = NULL;
	const gchar *name;
	guint8		**imgio_rgba_list = NULL,
			*rgba_rtn,
			def_alpha_value = 0xFF,
			def_bg_color[4];
	gulong		play_time_ms = 0l,
			*imgio_delay_list = NULL,
			modified_time_sec = 0l;
	EDVImageIOProgressData cb_data;

	/* Reset the returns */
	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;
	if(bpl_rtn != NULL)
		*bpl_rtn = 0;
	if(orig_width_rtn != NULL)
		*orig_width_rtn = 0;
	if(orig_height_rtn != NULL)
		*orig_height_rtn = 0;
	if(orig_nframes_rtn != NULL)
		*orig_nframes_rtn = 0;
	if(play_time_ms_rtn != NULL)
		*play_time_ms_rtn = 0l;
	if(creator_rtn != NULL)
		*creator_rtn = NULL;
	if(title_rtn != NULL)
		*title_rtn = NULL;
	if(author_rtn != NULL)
		*author_rtn = NULL;
	if(comments_rtn != NULL)
		*comments_rtn = NULL;
	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = 0l;

	if(STRISEMPTY(path))
		return(NULL);

	name = g_basename(path);

	/* Set the callback data */
	cb_data.progress_cb = progress_cb;
	cb_data.progress_data = progress_data;

	/* If no background color was specified then set a default
	 * background color
	 */
	if(bg_color == NULL)
	{
		bg_color = def_bg_color;
		bg_color[0] = 0xFF;
		bg_color[1] = 0xFF;
		bg_color[2] = 0xFF;
		bg_color[3] = 0xFF;
	}

	rgba_rtn = NULL;

	/* Open the image in RGBA using the more efficient thumb
	 * opening functions if available based on the extension
	 */
#ifdef HAVE_LIBGIF
	if(edv_name_has_extension(name, IMAGE_FORMAT_EXTENSION_GIF))
	{
		gboolean user_aborted = FALSE;
		(void)ImgFileOpenGIFRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			&rgba_rtn,
			bg_color,		/* 4 bytes RGBA */
			&orig_width, &orig_height,
			&nframes,
			&play_time_ms,
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			def_alpha_value,
			edv_image_io_progress_cb, &cb_data,
			&user_aborted
		);
		specific_loader_used = TRUE;
	}
#endif	/* HAVE_LIBGIF */
#ifdef HAVE_LIBJPEG
	if(edv_name_has_extension(name, IMAGE_FORMAT_EXTENSION_JFIF " "
					IMAGE_FORMAT_EXTENSION_JPE " "
					IMAGE_FORMAT_EXTENSION_JPEG " "
					IMAGE_FORMAT_EXTENSION_JPG
	))
	{
		gboolean user_aborted = FALSE;
		(void)ImgFileOpenJPEGRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			&rgba_rtn,
			&orig_width, &orig_height,
			edv_image_io_progress_cb, &cb_data,
			&user_aborted
		);
		nframes = 1;
		specific_loader_used = TRUE;
	}
#endif	/* HAVE_LIBJPEG */
#ifdef HAVE_LIBMNG
	if(edv_name_has_extension(name, IMAGE_FORMAT_EXTENSION_JNG " "
					IMAGE_FORMAT_EXTENSION_MNG
	))
	{
		gboolean user_aborted = FALSE;
		(void)ImgFileOpenMNGRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			&rgba_rtn,
			bg_color,		/* 4 bytes RGBA */
			&orig_width, &orig_height,
			&nframes,
			&play_time_ms,
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			&modified_time_sec,	/* In seconds since EPOCH */
			def_alpha_value,
			edv_image_io_progress_cb, &cb_data,
			&user_aborted
		);
		specific_loader_used = TRUE;
	}
#endif	/* HAVE_LIBMNG */
#ifdef HAVE_LIBPNG
	if(edv_name_has_extension(name, IMAGE_FORMAT_EXTENSION_PNG))
	{
		gboolean user_aborted = FALSE;
		(void)ImgFileOpenPNGRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			&rgba_rtn,
			bg_color,		/* 4 bytes RGBA */
			&orig_width, &orig_height,
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			&modified_time_sec,	/* In seconds since EPOCH */
			def_alpha_value,
			edv_image_io_progress_cb, &cb_data,
			&user_aborted
		);
		nframes = 1;
		specific_loader_used = TRUE;
	}
#endif	/* HAVE_LIBPNG */
#ifdef HAVE_LIBTIFF
	if(edv_name_has_extension(name, IMAGE_FORMAT_EXTENSION_TIF " "
				        IMAGE_FORMAT_EXTENSION_TIFF
	))
	{
		gboolean user_aborted = FALSE;
		(void)ImgFileOpenTIFFRGBAThumb(
			path,
			req_width, req_height,
			&width, &height,
			&bpl,
			&rgba_rtn,
			bg_color,		/* 4 bytes RGBA */
			&orig_width, &orig_height,
			&nframes,
			&play_time_ms,
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			def_alpha_value,
			edv_image_io_progress_cb, &cb_data,
			&user_aborted
		);
		specific_loader_used = TRUE;
	}
#endif	/* HAVE_LIBTIFF */
	if(!specific_loader_used)
	{
		(void)ImgFileOpenRGBA(
			path,
			&width, &height,
			&bpl,
			&imgio_rgba_list,
			&imgio_delay_list,
			&nframes,
			bg_color,		/* 4 bytes RGBA */
			&x, &y,
			&base_width, &base_height,
			&creator, &title, &author, &comments,
			&modified_time_sec,	/* In seconds since EPOCH */
			def_alpha_value,
			edv_image_io_progress_cb, &cb_data
		);
		if(nframes > 0)
		{
			gint i;

			orig_width = width;
			orig_height = height;

			/* Get the first frame only */
			rgba_rtn = imgio_rgba_list[0];

			/* Delete the rest of the frames */
			for(i = 1; i < nframes; i++)
				g_free(imgio_rgba_list[i]);
		}
	}

	/* Update the return values */
	if(width_rtn != NULL)
		*width_rtn = width;
	if(height_rtn != NULL)
		*height_rtn = height;
	if(bpl_rtn != NULL)
		*bpl_rtn = bpl;
	if(orig_width_rtn != NULL)
		*orig_width_rtn = orig_width;
	if(orig_height_rtn != NULL)
		*orig_height_rtn = orig_height;
	if(orig_nframes_rtn != NULL)
		*orig_nframes_rtn = nframes;
	if(play_time_ms_rtn != NULL)
		*play_time_ms_rtn = play_time_ms;
	if(creator_rtn != NULL)
	{
		*creator_rtn = creator;
		creator = NULL;
	}
	if(title_rtn != NULL)
	{
		*title_rtn = title;
		title = NULL;
	}
	if(author_rtn != NULL)
	{
		*author_rtn = author;
		author = NULL;
	}
	if(comments_rtn != NULL)
	{
		*comments_rtn = comments;
		comments = NULL;
	}
	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = modified_time_sec;

	g_free(imgio_rgba_list);
	g_free(imgio_delay_list);

	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(rgba_rtn);
}
