#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <glib.h>

#include "edv_types.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"


/* User IDs List */
EDVUID *edv_uids_list_match_uid(
	GList *uids_list,
	const gint user_id,
	gint *index_rtn
);
EDVUID *edv_uids_list_match_gid(
	GList *uids_list,
	const gint group_id,
	gint *index_rtn
);
gint edv_uid_name_to_uid(
	GList *uids_list,
	const gchar *user_name
);
gchar *edv_uid_uid_to_name(
	GList *uids_list,
	const gint user_id,
	gint *index_rtn
);
gchar *edv_uid_gid_to_name(
	GList *uids_list,
	const gint group_id,
	gint *index_rtn
);

/* Group IDs List */
EDVGID *edv_gids_list_match_gid(
	GList *gids_list,
	const gint group_id,
	gint *index_rtn
);
gint edv_gid_name_to_gid(
	GList *gids_list,
	const gchar *group_name
);
gchar *edv_gid_gid_to_name(
	GList *gids_list,
	const gint group_id,
	gint *index_rtn
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Matches the UID with the specified user ID.
 */
EDVUID *edv_uids_list_match_uid(
	GList *uids_list,
	const gint user_id,
	gint *index_rtn
)
{
	gint i;
	GList *glist;
	EDVUID *uid;

	if(index_rtn != NULL)
		*index_rtn = -1;

	for(glist = uids_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		uid = EDV_UID(glist->data);
		if(uid == NULL)
			continue;

		if(uid->user_id == user_id)
		{
			if(index_rtn != NULL)
				*index_rtn = i;
			return(uid);
		}
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Matches the UID with the specified group ID.
 */
EDVUID *edv_uids_list_match_gid(
	GList *uids_list,
	const gint group_id,
	gint *index_rtn
)
{
	gint i;
	GList *glist;
	EDVUID *uid;

	if(index_rtn != NULL)
		*index_rtn = -1;

	for(glist = uids_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		uid = EDV_UID(glist->data);
		if(uid == NULL)
			continue;

		if(uid->group_id == group_id)
		{
			if(index_rtn != NULL)
				*index_rtn = i;
			return(uid);
		}
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Matches the UID with the specified user name.
 */
gint edv_uid_name_to_uid(
	GList *uids_list,
	const gchar *user_name
)
{
	if(STRISEMPTY(user_name))
		return(0);

	/* Match by number? */
	if(isdigit(*user_name))
	{
		return((gint)atoi((const char *)user_name));
	}
	else
	{
		GList *glist;
		EDVUID *uid;

		for(glist = uids_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			uid = EDV_UID(glist->data);
			if(uid == NULL)
				continue;

			if(uid->name == NULL)
				continue;

			if(!strcmp((const char *)uid->name, (const char *)user_name))
				return(uid->user_id);
		}
	}

	return(0);
}


/*
 *	Matches the user name with the specified user ID.
 *
 *	Returns a dynamically allocated string describing the user
 *	name.
 */
gchar *edv_uid_uid_to_name(
	GList *uids_list,
	const gint user_id,
	gint *index_rtn
)
{
	EDVUID *uid = edv_uids_list_match_uid(uids_list, user_id, index_rtn);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
		return(g_strdup(uid->name));
	}
	else if(user_id < 0)
	{
		return(g_strdup(""));
	}
	else
	{
		return(g_strdup_printf(
			"%i",
			user_id
		));
	}
}

/*
 *	Matches the user name with the specified group ID.
 *
 *	On failed matched a number string describing the user ID is
 *	returned, so this function never returns NULL.
 */
gchar *edv_uid_gid_to_name(
	GList *uids_list,
	const gint group_id,
	gint *index_rtn
)
{
	EDVUID *uid = edv_uids_list_match_gid(uids_list, group_id, index_rtn);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
		return(g_strdup(uid->name));
	}
	else if(group_id < 0)
	{
		return(g_strdup(""));
	}
	else
	{
		return(g_strdup_printf(
			"%i",
			group_id
		));
	}
}


/*
 *	Creates a new GID.
 */
EDVGID *edv_gid_new(void)
{
	return(EDV_GID(g_malloc0(sizeof(EDVGID))));
}

/*
 *	Deletes the GID.
 */
void edv_gid_delete(EDVGID *gid)
{
	if(gid == NULL)
		return;

	g_free(gid->name);
	g_free(gid->password);
	if(gid->group_members_list != NULL)
	{
		g_list_foreach(gid->group_members_list, (GFunc)g_free, NULL);
		g_list_free(gid->group_members_list);
	}
	g_free(gid);
}


/*
 *	Matches the GID with the specified group ID.
 */
EDVGID *edv_gids_list_match_gid(
	GList *gids_list,
	const gint group_id,
	gint *index_rtn
)
{
	gint i;
	GList *glist;
	EDVGID *gid;

	if(index_rtn != NULL)
		*index_rtn = -1;

	for(glist = gids_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		gid = EDV_GID(glist->data);
		if(gid == NULL)
			continue;

		if(gid->group_id == group_id)
		{
			if(index_rtn != NULL)
				*index_rtn = i;
			return(gid);
		}
	}

	return(NULL);
}

/*
 *	Matches the GID with the specified group name.
 */
gint edv_gid_name_to_gid(
	GList *gids_list,
	const gchar *group_name
)
{
	if(STRISEMPTY(group_name))
		return(0);

	/* Match by number? */
	if(isdigit(*group_name))
	{
		return((gint)atoi((const char *)group_name));
	}
	else
	{
		GList *glist;
		EDVGID *gid;

		for(glist = gids_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			gid = EDV_GID(glist->data);
			if(gid == NULL)
				continue;

			if(gid->name == NULL)
				continue;

			if(!strcmp((const char *)gid->name, (const char *)group_name))
				return(gid->group_id);
		}
	}

	return(0);
}


/*
 *	Matches the group name from the specified group ID.
 *
 *	On failed matched, a number string containing the group ID is
 *      returned, so this function never returns NULL.
 */
gchar *edv_gid_gid_to_name(
	GList *gids_list,
	const gint group_id,
	gint *index_rtn
)
{
	EDVGID *gid = edv_gids_list_match_gid(gids_list, group_id, index_rtn);
	if((gid != NULL) ? (gid->name != NULL) : FALSE)
	{
		return(g_strdup(gid->name));
	}
	else if(group_id < 0)
	{
		return(g_strdup(""));
	}
	else
	{
		return(g_strdup_printf(
			"%i",
			group_id
		));
	}
}
