#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#ifdef HAVE_LIBID3TAG
# include <id3tag.h>
#endif
#include <glib.h>

#include "imgio.h"

#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"

#include "edv_id3.h"
#include "edv_id3_ids_list.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0			/* 0 = off
						 * 1 - 3 = on */
#endif
#if (DEBUG_LEVEL > 0)
# warning "Debugging enabled."
#endif


#ifdef HAVE_LIBID3TAG
static union id3_field *edv_id3_libid3tag_get_frame_field_by_type(
	struct id3_frame *id3_frame,
	enum id3_field_type type
);
#endif

/* Open */
#ifdef HAVE_LIBID3TAG
static gint edv_id3_tag_open_iterate_field(
	EDVID3Tag *tag,
	EDVID3Frame *frame,
	const union id3_field *ffield,
	const EDVID3TagIOOptions io_options
);
#endif	/* HAVE_LIBID3TAG */
EDVID3Tag *edv_id3_tag_open_file(
        const gchar *path,
        const EDVID3TagIOOptions io_options
);

/* Save */
#ifdef HAVE_LIBID3TAG
static gint edv_id3_tag_save_stream_render_tag(
	FILE *fp,
	struct id3_tag *ftag
);
static gint edv_id3_tag_save_file_write(
	const gchar *path,
	struct id3_tag *ftag,
	EDVID3TagIOOptions io_options
);
static gint edv_id3_tag_save_iterate_field(
	struct id3_frame *fframe,
	EDVID3Frame *frame,
	EDVID3Field *field,
	const EDVID3TagIOOptions io_options
);
static gint edv_id3_tag_save_iterate_frame(
	struct id3_tag *ftag,
	EDVID3Frame *frame,
	const EDVID3TagIOOptions io_options
);
#endif	/* HAVE_LIBID3TAG */
gint edv_id3_tag_save_file(
        EDVID3Tag *tag,
        const gchar *path,
        const EDVID3TagIOOptions io_options
);

/* Strip */
gint edv_id3_tag_strip_file(const gchar *path);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


#ifdef HAVE_LIBID3TAG
/*
 *      Gets the id3_field on the id3_frame by type.
 */
static union id3_field *edv_id3_libid3tag_get_frame_field_by_type(
	struct id3_frame *id3_frame,
	enum id3_field_type type
)
{
	union id3_field *field;
	const gint n = id3_frame->nfields;
	gint i;

	for(i = 0; i < n; i++)
	{
		field = id3_frame_field(
			id3_frame,
			(unsigned int)i
		);
		if(field == NULL)
			continue;

		if(field->type == type)
			return(field);
	}

	return(NULL);
}
#endif  /* HAVE_LIBID3TAG */


#ifdef HAVE_LIBID3TAG
/*
 *	Called by edv_id3_tag_open_file() to parse the values
 *	from the id3_field and stores them on a EDVID3Field.
 *
 *	The tag specifies the current EDVID3Tag in context.
 *
 *	The frame specifies the current EDVID3Frame in context.
 *
 *	Returns 0 on success or non-zero on error.
 */
static gint edv_id3_tag_open_iterate_field(
	EDVID3Tag *tag,
	EDVID3Frame *frame,
	const union id3_field *ffield,
	const EDVID3TagIOOptions io_options
)
{
	gint		n,
			status = -7;
	gchar *s;
	const gchar *id = frame->id;
	EDVID3Field *field = NULL;

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_open_iterate_field() id=%s\n", id);
#endif

	/* Handle by the field's data type, if the type is supported
	 * then a new EDVID3Frame will be created (frame != NULL)
	 * and its type, data, and data_length will be set
	 */
	switch(id3_field_type(ffield))
	{
	    case ID3_FIELD_TYPE_TEXTENCODING:
		status = -2;
		switch(id3_field_gettextencoding(ffield))
		{
		    case ID3_FIELD_TEXTENCODING_ISO_8859_1:
			frame->text_encoding = EDV_ID3_TEXT_ENCODING_ISO_8859_1;
			status = 0;
			break;
		    case ID3_FIELD_TEXTENCODING_UTF_16:
			frame->text_encoding = EDV_ID3_TEXT_ENCODING_UTF_16;
			status = 0;
			break;
		    case ID3_FIELD_TEXTENCODING_UTF_16BE:
			frame->text_encoding = EDV_ID3_TEXT_ENCODING_UTF_16_BE;
			status = 0;
			break;
		    case ID3_FIELD_TEXTENCODING_UTF_8:
			frame->text_encoding = EDV_ID3_TEXT_ENCODING_UTF_8;
			status = 0;
			break;
		}
		break;

	    case ID3_FIELD_TYPE_LATIN1:
		/* Handle special cases
		 *
		 * Image
		 */
		if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_IMAGE))
		{
			const gchar *s = (const gchar *)id3_field_getlatin1(ffield);
			if(s != NULL)
			{
				field = edv_id3_frame_get_field_by_type(
					frame,
					EDV_ID3_FIELD_TYPE_MIME_TYPE
				);
				if(field == NULL)
					field = edv_id3_frame_append_field(frame);
				if(field != NULL)
				{
					field->type = EDV_ID3_FIELD_TYPE_MIME_TYPE;
					g_free(field->data);
					field->data = g_strdup(s);
					field->data_length = (field->data != NULL) ?
						((gint)strlen((char *)field->data) + 1) : 0;
					status = 0;
				}
				else
				{
					status = -3;
				}
			}
			else
			{
				status = -2;
			}
		}
		/* All other frames */
		else
		{
			const gchar *s = (const gchar *)id3_field_getlatin1(ffield);
			if(s != NULL)
			{
				field = edv_id3_frame_get_field_by_type(
					frame,
					EDV_ID3_FIELD_TYPE_STRING
				);
				if(field == NULL)
					field = edv_id3_frame_append_field(frame);
				if(field != NULL)
				{
					field->type = EDV_ID3_FIELD_TYPE_STRING;
					g_free(field->data);
					field->data = g_strdup(s);
					field->data_length = (field->data != NULL) ?
						((gint)strlen((char *)field->data) + 1) : 0;
					status = 0;
				}
				else
				{
					status = -3;
				}
			}
			else
			{
				status = -2;
			}
		}
		break;

	    case ID3_FIELD_TYPE_LATIN1FULL:
		s = (gchar *)id3_field_getfulllatin1(ffield);
		if(s != NULL)
		{
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_STRING;
				g_free(field->data);
				field->data = g_strdup(s);
				field->data_length = (field->data != NULL) ?
					((gint)strlen((char *)field->data) + 1) : 0;
				status = 0;
			}
			else
			{
				status = -3;
			}
		}
		else
		{
			status = -2;
		}
		break;

	    case ID3_FIELD_TYPE_LATIN1LIST:
		s = NULL;
		n = (gint)ffield->latin1list.nstrings;
		if(n > 0)
		{
			gint i;
			gchar *s_tmp;
			s = g_strdup("");
			for(i = 0; i < n; i++)
			{
				s_tmp = g_strconcat(
					s,
					(gchar *)ffield->latin1list.strings[i],
					NULL
				);
				g_free(s);
				s = s_tmp;
			}
		}
		if(s != NULL)
		{
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_STRING;
				g_free(field->data);
				field->data = g_strdup(s);
				field->data_length = (field->data != NULL) ?
					((gint)strlen((char *)field->data) + 1) : 0;
			}
			g_free(s);
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_STRING:
		s = (gchar *)id3_ucs4_utf8duplicate(id3_field_getstring(ffield));
		if(s != NULL)
		{
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_STRING;
				g_free(field->data);
				field->data = g_strdup(s);
				field->data_length = (field->data != NULL) ?
					((gint)strlen((char *)field->data) + 1) : 0;
			}
			g_free(s);
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_STRINGFULL:
		s = (gchar *)id3_ucs4_utf8duplicate(id3_field_getfullstring(ffield));
		if(s != NULL)
		{
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_STRING;
				g_free(field->data);
				field->data = g_strdup(s);
				field->data_length = (field->data != NULL) ?
					((gint)strlen((char *)field->data) + 1) : 0;
			}
			g_free(s);
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_STRINGLIST:
		s = NULL;
		n = (gint)id3_field_getnstrings(ffield);
		if(n > 0)
		{
			gint i;
			gchar	*s_next,
				*s_tmp;
			s = g_strdup("");
			for(i = 0; i < n; i++)
			{
				s_next = (gchar *)id3_ucs4_utf8duplicate(id3_field_getstrings(
					ffield,
					(unsigned int)i
				));
				s_tmp = g_strconcat(
					s,
					s_next,
					NULL
				);
				g_free(s_next);
				g_free(s);
				s = s_tmp;
			}
		}
		if(s != NULL)
		{
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_STRING
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_STRING;
				g_free(field->data);
				field->data = g_strdup(s);
				field->data_length = (field->data != NULL) ?
					((gint)strlen((char *)field->data) + 1) : 0;
			}
			g_free(s);
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_LANGUAGE:
		s = (gchar *)ffield->immediate.value;
		if(s != NULL)
		{
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_LANGUAGE
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_LANGUAGE;
				g_free(field->data);
				field->data = g_strdup(s);
				field->data_length = (field->data != NULL) ?
					((gint)strlen((char *)field->data) + 1) : 0;
			}
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_FRAMEID:
		/* Ignore */
		status = 0;
		break;

	    case ID3_FIELD_TYPE_DATE:
		/* Ignore */
		status = 0;
		break;

	    case ID3_FIELD_TYPE_INT8:
		{
			const gint8 x = (gint8)id3_field_getint(ffield);
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_INT8
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_INT8;
				field->data_length = (gint)sizeof(gint8);
				g_free(field->data);
				field->data = g_memdup(
					&x,
					field->data_length
				);
			}
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_INT16:
		{
			const gint16 x = (gint16)id3_field_getint(ffield);
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_INT16
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_INT16;
				field->data_length = (gint)sizeof(gint16);
				g_free(field->data);
				field->data = g_memdup(
					&x,
					field->data_length
				);
			}
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_INT24:		/* Treat as 32 bit */
	    case ID3_FIELD_TYPE_INT32:
		{
			const gint32 x = (gint32)id3_field_getint(ffield);
			field = edv_id3_frame_get_field_by_type(
				frame,
				EDV_ID3_FIELD_TYPE_INT32
			);
			if(field == NULL)
				field = edv_id3_frame_append_field(frame);
			if(field != NULL)
			{
				field->type = EDV_ID3_FIELD_TYPE_INT32;
				field->data_length = (gint)sizeof(gint32);
				g_free(field->data);
				field->data = g_memdup(
					&x,
					field->data_length
				);
			}
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_INT32PLUS:
		{
			id3_length_t data_length;
			const id3_byte_t *data = id3_field_getbinarydata(
				ffield,
				&data_length
			);
			if((data != NULL) && (data_length == 8))
			{
				field = edv_id3_frame_get_field_by_type(
					frame,
					EDV_ID3_FIELD_TYPE_INT64
				);
				if(field == NULL)
					field = edv_id3_frame_append_field(frame);
				if(field != NULL)
				{
					field->type = EDV_ID3_FIELD_TYPE_INT64;
					field->data_length = data_length;
					g_free(field->data);
					field->data = g_memdup(
						data,
						field->data_length
					);
				}
			}
		}
		status = 0;
		break;

	    case ID3_FIELD_TYPE_BINARYDATA:
		{
			id3_length_t data_length;
			const id3_byte_t *data = id3_field_getbinarydata(
				ffield,
				&data_length
			);
			if((data != NULL) && (data_length > 0))
			{
				field = edv_id3_frame_get_field_by_type(
					frame,
					EDV_ID3_FIELD_TYPE_DATA
				);
				if(field == NULL)
					field = edv_id3_frame_append_field(frame);
				if(field != NULL)
				{
					field->type = EDV_ID3_FIELD_TYPE_DATA;
					field->data_length = data_length;
					g_free(field->data);
					field->data = g_memdup(
						data,
						field->data_length
					);
				}
			}
		}
		status = 0;
		break;
	}

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_open_iterate_field() done status=%i\n", status);
#endif

	return(status);
}
#endif	/* HAVE_LIBID3TAG */

/*
 *	Opens the ID3 tag from the file.
 *
 *	The path specifies the file.
 *
 *	The io_options specifies any options to be used when opening
 *	the ID3 tags from the file.
 *
 *	Returns a new dynamically allocated EDVID3Tag opened from the
 *	file or NULL on error or if no tags were found.
 */
EDVID3Tag *edv_id3_tag_open_file(
	const gchar *path,
	const EDVID3TagIOOptions io_options
)
{
#if defined(HAVE_LIBID3TAG)
	struct id3_tag *ftag;
	struct id3_file *ffp;
	EDVID3Tag *tag = NULL;

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_open_file() path=%s\n", path);
#endif

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(tag);
	}

	/* Open the file as a libid3tag stream */
	ffp = id3_file_open(
		(const char *)path,
		ID3_FILE_MODE_READONLY
	);
	if(ffp == NULL)
		return(tag);

	/* Look for ID3 tags */
	ftag = id3_file_tag(ffp);
	if(ftag == NULL)
	{
		/* No ID3 tags found */
		(void)id3_file_close(ffp);
		return(tag);
	}

#if (DEBUG_LEVEL >= 1)
g_print(" version=0x%.8X flags=0x%.8X frames=%i pad_len=%ld\n",
 (guint32)ftag->version,
 (guint32)ftag->flags,
 ftag->nframes,
 (gulong)ftag->paddedsize
);
#endif

	/* Any ID3 tags with frames found? */
	if(ftag->nframes > 0)
	{
		struct id3_frame	*fframe,
					**fframes_list = ftag->frames;
		const gint n = (gint)ftag->nframes;
		gint i;
		const gchar *id;
		EDVID3Frame *frame;

		/* Create a new EDVID3Tag */
		tag = edv_id3_tag_new();
		if(tag == NULL)
		{
			(void)id3_file_close(ffp);
			return(tag);
		}

		/* set the EDVID3Tag's values from the id3_tag */
		tag->version_major = ID3_TAG_VERSION_MAJOR(ftag->version);
		tag->version_minor = ID3_TAG_VERSION_MINOR(ftag->version);
		tag->flags = ftag->flags;
		tag->extended_flags = ftag->extendedflags;
		tag->restriction_flags = ftag->restrictions;

		/* Parse all the frames on this id3_tag and append them
		 * to the new EDVID3Tag
		 */
		for(i = 0; i < n; i++)
		{
			fframe = fframes_list[i];
			if(fframe == NULL)
				continue;

			/* Get this frame's ID */
			id = (const gchar *)fframe->id;
			if(id == NULL)
				continue;

			/* Do not parse frames with no fields? */
			if((fframe->fields == NULL) || (fframe->nfields <= 0))
			{
				if(io_options & EDV_ID3_TAG_IO_EXCLUDE_EMPTY_FRAMES)
					continue;
			}

			/* Check if an EDVID3Frame has already been
			 * created for this ID on the EDVID3Tag
			 */
			frame = edv_id3_tag_get_frame_by_id(
				tag,
				id
			);
			if(frame == NULL)
			{
				/* No EDVID3Frame has been created
				 * yet, so create a new one
				 */
				frame = edv_id3_frame_new_with_values(
					id,
					EDV_ID3_TEXT_ENCODING_UNKNOWN,
					NULL,
					fframe->description
				);
				if(frame == NULL)
					break;

				tag->frames_list = g_list_append(
					tag->frames_list,
					frame
				);
			}

			/* Parse all the fields on this frame */
			if(fframe->nfields > 0)
			{
				const gint n = (gint)fframe->nfields;
				gint		i,
						status2;
				union id3_field *ffield;

				for(i = 0; i < n; i++)
				{
					ffield = id3_frame_field(
						fframe,
						(unsigned int)i
					);
					if(ffield == NULL)
						continue;

					status2 = edv_id3_tag_open_iterate_field(
						tag,
						frame,
						ffield,
						io_options
					);
				}
			}
		}
	}
	else
	{
		/* No ID3 tags found */
		(void)id3_file_close(ffp);
		errno = ENOENT;
		return(tag);
	}

	/* Close the libi3dtag stream */
	(void)id3_file_close(ffp);

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_open_file() done tag=%p\n", tag);
#endif

	return(tag);
#else
	return(NULL);
#endif	/* !HAVE_LIBID3TAG */
}


#ifdef HAVE_LIBID3TAG
/*
 *	Renders the id3_tag to a buffer and writes it to the stream at
 *	its current position.
 */
static gint edv_id3_tag_save_stream_render_tag(
	FILE *fp,
	struct id3_tag *ftag
)
{
	size_t		units_to_write,
			units_written;
	gulong		i,
			estimated_length,
			io_buf_len,
			rendered_len;
	guint8 *buf;
	EDVVFSObject *obj;

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_save_stream_render_tag()\n");
#endif

	/* Estimate the size of the buffer needed to hold the rendered
	 * the ID3 tag
	 */
	estimated_length = (gulong)id3_tag_render(
		ftag,
		NULL				/* Estimate size only
						 * do not render */
	);
	if((estimated_length == 0l) || (estimated_length == (gulong)-1))
	{
#if (DEBUG_LEVEL >= 2)
g_print("id3_tag_render(): Unable to estimate size.\n");
#endif
		return(-1);
	}


	/* Allocate the rendered ID3 tag buffer */
	buf = (guint8 *)g_malloc(estimated_length * sizeof(guint8));
	if(buf == NULL)
		return(-3);

	/* Render the ID3 tag buffer */
	rendered_len = (gulong)id3_tag_render(
		ftag,
		(id3_byte_t *)buf
	);
	if(rendered_len == 0l)
	{
#if (DEBUG_LEVEL >= 2)
g_print("id3_tag_render(): Unable to render.\n");
#endif
		g_free(buf);
		return(-1);
	}

	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len == 0l)
		io_buf_len = 1l;

	/* Write to the stream */
	i = 0l;
	while(i < rendered_len)
	{
		units_to_write = (size_t)MIN(
			io_buf_len,
			rendered_len - i
		) / sizeof(guint8);
		units_written = fwrite(
			buf + i,
			sizeof(guint8),
			units_to_write,
			fp
		);
		if(units_written != units_to_write)
		{
#if (DEBUG_LEVEL >= 2)
			const gint error_code = (gint)errno;
g_print("fwrite(): %s: i=%ld units_to_write=%ld units_written=%ld\n",
 g_strerror(error_code),
 i,
 (gulong)units_to_write,
 (gulong)units_written
);
#endif
			g_free(buf);
			return(-1);
		}
		i += (gulong)(units_written * sizeof(guint8));
	}

	/* Delete the rendered ID3 tag buffer */
	g_free(buf);

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_save_stream_render_tag() done status=%i\n", 0);
#endif

	return(0);
}
#endif	/* HAVE_LIBID3TAG */

#ifdef HAVE_LIBID3TAG
/*
 *	Inserts the ID3 tag to the file while overwriting any existing
 *	ID3 tags in the file and preserving all other data in the file.
 */
static gint edv_id3_tag_save_file_write(
	const gchar *path,
	struct id3_tag *ftag,
	EDVID3TagIOOptions io_options
)
{
typedef struct {
	gulong		position,
			length;
} EDVID3TagStreamPosition;
	FILE		*in_fp,
			*out_fp;
	size_t		units_to_read,
			units_read,
			units_to_write,
			units_written;
	gboolean	found_id3v1_tag,
			found_id3v2_tag;
	gint status;
	gulong		position,
			length,
			id3v1_position,
			id3v1_length,
			id3v2_position,
			id3v2_length,
			io_buf_len = 0l,
			in_fp_pos;
	gchar		*in_path,
			*out_path;
	guint8 *io_buf;
	GList		*glist,
			*tag_positions_list;
	EDVVFSObject *obj;
	EDVID3TagStreamPosition *tag_position;

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_save_file_write()\n");
#endif
	/* Get the input and output file paths */
	in_path = g_strdup(path);
	out_path = edv_tmp_name(NULL);
	if((in_path == NULL) || (out_path == NULL))
	{
		g_free(in_path);
		(void)edv_unlink(out_path);
		g_free(out_path);
		return(-1);
	}

#if (DEBUG_LEVEL >= 3)
g_print(" in_path=%s\n", in_path);
g_print(" out_path=%s\n", out_path);
#endif

	/* Open the output file for append writing */
	out_fp = fopen(
		(const char *)out_path,
		"ab"
	);
	if(out_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s\n",
 g_strerror(error_code),
 out_path
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	obj = edv_vfs_object_fstat((gint)fileno(out_fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len == 0l)
		io_buf_len = 1l;

	/* Insert the ID3v2 to the start of the output stream? */
	if(!(io_options & EDV_ID3_TAG_IO_APPEND))
	{
		/* Render an ID3v2 tag data from the id3_tag's values
		 * and write the rendered ID3v2 tag data to the output
		 * stream
		 */
		status = edv_id3_tag_save_stream_render_tag(
			out_fp,
			ftag
		);
		if(status != 0)
		{
			(void)fclose(out_fp);
			(void)edv_unlink(out_path);
			g_free(out_path);
			g_free(in_path);
			return(status);
		}
	}

	/* Allocate the write buffer */
	io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
	if(io_buf == NULL)
	{
		(void)fclose(out_fp);
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	/* Open the input file for reading */
	in_fp = fopen(
		(const char *)in_path,
		"rb"
	);
	if(in_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s\n",
 g_strerror(error_code),
 in_path
);
#endif
		(void)fclose(out_fp);
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	in_fp_pos = 0l;

	/* List the ID3 tag locations in the input file */
	tag_position = NULL;
	tag_positions_list = NULL;
	while(TRUE)
	{
		/* Look for the next ID3v2 tag */
		found_id3v2_tag = edv_id3_stream_locate_id3v2(
			in_fp,
			NULL,
			NULL,
			&id3v2_position,
			&id3v2_length
		);

		/* Reset the stream position and look for the next
		 * ID3v1 tag
		 */
		if(fseek(
			in_fp,
			(long)in_fp_pos,
			SEEK_SET
		) != 0)
			break;

		found_id3v1_tag = edv_id3_stream_locate_id3v1(
			in_fp,
			&id3v1_position,
			&id3v1_length
		);

		/* Get the next tag's position based on which type
		 * of tag was found closer to the stream position
		 * at the start of this iteration
		 */
		if(found_id3v1_tag && found_id3v2_tag)
		{
			if(id3v1_position < id3v2_position)
			{
				position = id3v1_position;
				length = id3v1_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v1 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif

			}
			else
			{
				position = id3v2_position;
				length = id3v2_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v2 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif
			}
		}
		else if(found_id3v1_tag)
		{
			position = id3v1_position;
			length = id3v1_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v1 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif
		}
		else if(found_id3v2_tag)
		{
			position = id3v2_position;
			length = id3v2_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v2 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif
		}
		else
		{
			/* No more tags found */
			break;
		}

		/* If a previous tag was found then check if this
		 * tag lies within or before it, in which case we
		 * need to break out of this potential infinite loop
		 */
		if(tag_position != NULL)
		{
			if(position < (tag_position->position + tag_position->length))
				break;
		}

		/* Record this tag position */
		tag_position = (EDVID3TagStreamPosition *)g_malloc(
			sizeof(EDVID3TagStreamPosition)
		);
		if(tag_position != NULL)
		{
			tag_positions_list = g_list_append(
				tag_positions_list,
				tag_position
			);
			tag_position->position = position;
			tag_position->length = length;
		}

		/* Seek the input stream just past this tag */
		in_fp_pos = position + length;
		if(fseek(
			in_fp,
			(long)in_fp_pos,
			SEEK_SET
		) != 0)
			break;
	}

	/* Seek the input stream back to the beginning and clear any
	 * errors
	 */
	rewind(in_fp);
	in_fp_pos = 0l;

	/* Copy the data from the input stream to the output stream */
	glist = tag_positions_list;
	while(!feof(in_fp))
	{
		/* Skip the tags from the input stream? */
		if(io_options & EDV_ID3_TAG_IO_TRUNCATE_EXISTING_TAGS)
		{
			/* Seek the input stream and calculate the
			 * units to read based on the current item
			 * in the tag positions list then put the next
			 * item in the tag positions list into context
			 */
			while(glist != NULL)
			{
				tag_position = (EDVID3TagStreamPosition *)glist->data;

				/* Not reached yet? */
				if(in_fp_pos < tag_position->position)
					break;

				/* Reached the next tag, seek the input
				 * stream past it and update the
				 * recorded position
				 */
				in_fp_pos = tag_position->position + tag_position->length;
#if (DEBUG_LEVEL >= 3)
g_print(" Skipping tag in put stream at position %ld, seeking to %ld.\n",
 tag_position->position,
 in_fp_pos
);
#endif
				if(fseek(
					in_fp,
					(long)in_fp_pos,
					SEEK_SET
				) != 0)
					break;

				/* Get the next upcomming tag and check
				 * again
				 */
				glist = g_list_next(glist);
			}

			/* Calculate the number of units to read */
			if(glist != NULL)
			{
				tag_position = (EDVID3TagStreamPosition *)glist->data;
				units_to_read = (size_t)MIN(
					io_buf_len,
					tag_position->position - in_fp_pos
				) / sizeof(guint8);
			}
			else
			{
				units_to_read = (size_t)io_buf_len / sizeof(guint8);
			}
			if(units_to_read <= 0l)
				break;
		}
		else
		{
			units_to_read = (size_t)io_buf_len / sizeof(guint8);
		}

		/* Read this block from the input stream */
		units_read = fread(
			io_buf,
			sizeof(guint8),
			units_to_read,
			in_fp
		);

		/* Update the recorded position of the input stream */
		in_fp_pos += (gulong)(units_read * sizeof(guint8));

		/* Write this block to the output stream */
		units_to_write = units_read;
		if(units_to_write > 0l)
		{
			units_written = fwrite(
				io_buf,
				sizeof(guint8),
				units_to_write,
				out_fp
			);
			if((units_written != units_to_write) ||
			   (ferror(out_fp))
			)
			{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fwrite(): %s: units_to_write=%ld units_written=%ld\n",
 g_strerror(error_code),
 (gulong)units_to_write,
 (gulong)units_written
);
#endif
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				g_free(out_path);
				(void)fclose(in_fp);
				g_free(in_path);
				g_free(io_buf);
				if(tag_positions_list != NULL)
				{
					g_list_foreach(tag_positions_list, (GFunc)g_free, NULL);
					g_list_free(tag_positions_list);
				}
				return(-1);
			}
		}

		/* Read error? */
		if(ferror(in_fp))
		{
#if (DEBUG_LEVEL >= 2)
g_print(" fread(): Error: units_to_read=%ld units_read=%ld\n",
 (gulong)units_to_read,
 (gulong)units_read
);
#endif
			(void)fclose(out_fp);
			(void)edv_unlink(out_path);
			g_free(out_path);
			(void)fclose(in_fp);
			g_free(in_path);
			g_free(io_buf);
			if(tag_positions_list != NULL)
			{
				g_list_foreach(tag_positions_list, (GFunc)g_free, NULL);
				g_list_free(tag_positions_list);
			}
			return(-1);
		}

		/* Nothing was read? */
		if(units_read <= 0l)
			break;
	}

	/* Delete the tag positions list */
	if(tag_positions_list != NULL)
	{
		g_list_foreach(tag_positions_list, (GFunc)g_free, NULL);
		g_list_free(tag_positions_list);
		tag_positions_list = NULL;
	}

	/* Close the input stream */
	(void)fclose(in_fp);

	/* Append the ID3v2 to the start of the output stream? */
	if(io_options & EDV_ID3_TAG_IO_APPEND)
	{
		/* Render an ID3v2 tag data from the id3_tag's values
		 * and write the rendered ID3v2 tag data to the output
		 * stream
		 */
		status = edv_id3_tag_save_stream_render_tag(
			out_fp,
			ftag
		);
		if(status != 0)
		{
			(void)fclose(out_fp);
			(void)edv_unlink(out_path);
			g_free(out_path);
			g_free(in_path);
			g_free(io_buf);
			return(status);
		}
	}

	/* Close the output stream */
	if(fclose(out_fp) != 0)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fclose(out_fp): %s.\n",
 g_strerror(error_code)
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		g_free(io_buf);
		return(-1);
	}


	/* Copy the data from the output path back to the input path,
	 * opening the output file as the input file and vice versa
	 */
	in_fp = fopen(
		(const char *)out_path,
		"rb"
	);
	if(in_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s (opening output file as input stream)\n",
 g_strerror(error_code),
 out_path
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		g_free(io_buf);
		return(-1);
	}

	out_fp = fopen(
		(const char *)in_path,
		"wb"
	);
	if(out_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s (opening input file as output stream)\n",
 g_strerror(error_code),
 in_path
);
#endif
		(void)fclose(in_fp);
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		g_free(io_buf);
		return(-1);
	}

	/* Copy the data from the output file (now the input stream)
	 * to the input file (now the output stream)
	 */
	while(!feof(in_fp))
	{
		/* Calculate the number of units to read */
		units_to_read = (size_t)io_buf_len / sizeof(guint8);

		/* Read this block from the input stream */
		units_read = fread(
			io_buf,
			sizeof(guint8),
			units_to_read,
			in_fp
		);

		/* Write this block to the output stream */
		units_to_write = units_read;
		if(units_to_write > 0l)
		{
			units_written = fwrite(
				io_buf,
				sizeof(guint8),
				units_to_write,
				out_fp
			);
			if((units_written != units_to_write) ||
			   (ferror(out_fp))
			)
			{
#if (DEBUG_LEVEL >= 2)
			const gint error_code = (gint)errno;
g_print(" fwrite(): %s: units_to_write=%ld units_written=%ld\n",
 g_strerror(error_code),
 (gulong)units_to_write,
 (gulong)units_written
);
#endif
				(void)fclose(in_fp);
				(void)edv_unlink(out_path);
				g_free(out_path);
				(void)fclose(out_fp);
				g_free(in_path);
				g_free(io_buf);
				return(-1);
			}
		}

		/* Read error? */
		if(ferror(in_fp))
		{
#if (DEBUG_LEVEL >= 2)
g_print(" fread(): Error: units_to_read=%ld units_read=%ld\n",
 (gulong)units_to_read,
 (gulong)units_read
);
#endif
			(void)fclose(in_fp);
			(void)edv_unlink(out_path);
			g_free(out_path);
			(void)fclose(out_fp);
			g_free(in_path);
			g_free(io_buf);
			return(-1);
		}

		/* Nothing was read? */
		if(units_read <= 0l)
			break;
	}

	/* Delete the write buffer */
	g_free(io_buf);

	/* Close the output file (now the input stream) */
	(void)fclose(in_fp);

	/* Close the input file (now the output stream) */
	if(fclose(out_fp) != 0)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fclose(out_fp): %s.\n",
 g_strerror(error_code)
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	/* Remove the output file */
	(void)edv_unlink(out_path);

	g_free(out_path);
	g_free(in_path);

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tag_save_file_write() done status=%i\n", status);
#endif

	return(status);
}
#endif	/* HAVE_LIBID3TAG */

#ifdef HAVE_LIBID3TAG
/*
 *	Called by edv_id3_tag_save_iterate_frame() to update the
 *	id3_fields on the id3_frame with the EDVID3Fields on the
 *	EDVID3Frame.
 */
static gint edv_id3_tag_save_iterate_field(
	struct id3_frame *fframe,
	EDVID3Frame *frame,
	EDVID3Field *field,
	const EDVID3TagIOOptions io_options
)
{
	union id3_field *ffield;
	gint status = -7;
	const gchar *id = frame->id;

#if (DEBUG_LEVEL >= 1)
g_print("  EDVID3FieldType=%i\n", field->type);
#endif
	switch(field->type)
	{
	    case EDV_ID3_FIELD_TYPE_UNKNOWN:
		/* EDVID3Frames with unknown data types are ignored */
		status = -2;
		break;

	    case EDV_ID3_FIELD_TYPE_INT8:
		ffield = edv_id3_libid3tag_get_frame_field_by_type(
			fframe,
			ID3_FIELD_TYPE_INT8
		);
		if(ffield != NULL)
		{
			const glong x = (glong)edv_id3_field_get_long(field);
			if(id3_field_setint(
				ffield,
				(signed long)x
			) != 0)
				status = -1;
			else
				status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   INT8=%ld\n", x);
#endif
		}
		break;

	    case EDV_ID3_FIELD_TYPE_INT16:
		ffield = edv_id3_libid3tag_get_frame_field_by_type(
			fframe,
			ID3_FIELD_TYPE_INT16
		);
		if(ffield != NULL)
		{
			const glong x = (glong)edv_id3_field_get_long(field);
			if(id3_field_setint(
				ffield,
				(signed long)x
			) != 0)
				status = -1;
			else
				status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   INT16=%ld\n", x);
#endif
		}
		break;

	    case EDV_ID3_FIELD_TYPE_INT32:
		ffield = edv_id3_libid3tag_get_frame_field_by_type(
			fframe,
			ID3_FIELD_TYPE_INT32
		);
		if(ffield != NULL)
		{
			const glong x = (glong)edv_id3_field_get_long(field);
			if(id3_field_setint(
				ffield,
				(signed long)x
			) != 0)
				status = -1;
			else
				status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   INT32=%ld\n", x);
#endif
		}
		break;

	    case EDV_ID3_FIELD_TYPE_INT64:
		ffield = edv_id3_libid3tag_get_frame_field_by_type(
			fframe,
			ID3_FIELD_TYPE_INT32PLUS
		);
		if(ffield != NULL)
		{
			const gint64 x = (gint64)edv_id3_field_get_long(field);

			/* ID3_FIELD_TYPE_INT32PLUS must be set using
			 * id3_field_setbinarydata()
			 */
			if(id3_field_setbinarydata(
				ffield,
				(id3_byte_t *)&x,
				(id3_length_t)sizeof(x)
			) != 0)
				status = -1;
			else
				status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   INT64=%ld\n", (gulong)x);
#endif
		}
		break;

	    case EDV_ID3_FIELD_TYPE_STRING:
		ffield = edv_id3_libid3tag_get_frame_field_by_type(
			fframe,
			ID3_FIELD_TYPE_TEXTENCODING
		);
		if(ffield != NULL)
		{
			enum id3_field_textencoding ftext_encoding = ID3_FIELD_TEXTENCODING_UTF_8;
			switch(frame->text_encoding)
			{
			    case EDV_ID3_TEXT_ENCODING_UNKNOWN:
				break;
			    case EDV_ID3_TEXT_ENCODING_UTF_8:
				ftext_encoding = ID3_FIELD_TEXTENCODING_UTF_8;
				break;
			    case EDV_ID3_TEXT_ENCODING_UTF_16:
				ftext_encoding = ID3_FIELD_TEXTENCODING_UTF_16;
				break;
			    case EDV_ID3_TEXT_ENCODING_UTF_16_BE:
				ftext_encoding = ID3_FIELD_TEXTENCODING_UTF_16BE;
				break;
			    case EDV_ID3_TEXT_ENCODING_ISO_8859_1:
				ftext_encoding = ID3_FIELD_TEXTENCODING_ISO_8859_1;
				break;
			}
			(void)id3_field_settextencoding(
				ffield,
				ftext_encoding
			);
#if (DEBUG_LEVEL >= 3)
g_print("   TEXT_ENCODING=%i\n", ftext_encoding);
#endif
		}

		/* Handle special case frames first
		 *
		 * Genre
		 */
		if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_GENRE))
		{
			const gchar *s = (field->data != NULL) ?
				(gchar *)field->data : "";
			id3_ucs4_t *ucs4;
			if(isdigit(*s))
			{
				ucs4 = id3_utf8_ucs4duplicate((const id3_utf8_t *)s);
			}
			else
			{
				int genre_index;
				ucs4 = id3_utf8_ucs4duplicate((const id3_utf8_t *)s);
				genre_index = id3_genre_number(ucs4);
				g_free(ucs4);
				if(genre_index >= 0)
				{
					gchar *s_tmp = g_strdup_printf(
						"%d",
						genre_index
					);
					ucs4 = id3_utf8_ucs4duplicate((id3_utf8_t *)s_tmp);
					g_free(s_tmp);
				}
				else
				{
					ucs4 = id3_utf8_ucs4duplicate("0");
				}
			}

			ffield = edv_id3_libid3tag_get_frame_field_by_type(
				fframe,
				ID3_FIELD_TYPE_STRINGLIST
			);
			if(ffield != NULL)
			{
				if(id3_field_setstrings(
					ffield,
					1,	/* One string */
					&ucs4
				) != 0)
					status = -1;
				else
					status = 0;
			}

			g_free(ucs4);
#if (DEBUG_LEVEL >= 3)
g_print("   STRING=\"%s\"\n", s);
#endif
		}
		/* Comment */
		else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_COMMENTS))
		{
			const gchar *s = (field->data != NULL) ?
				(gchar *)field->data : "";
			id3_ucs4_t *ucs4 = id3_utf8_ucs4duplicate((const id3_utf8_t *)s);

			/* Comments need to have an empty string set
			 * for the ID3_FIELD_TYPE_STRING id3_field
			 */
			ffield = edv_id3_libid3tag_get_frame_field_by_type(
				fframe,
				ID3_FIELD_TYPE_STRING
			);
			if(ffield != NULL)
			{
				id3_ucs4_t *ucs4_empty = id3_utf8_ucs4duplicate("");
				(void)id3_field_setstring(
					ffield,
					ucs4_empty
				);
				g_free(ucs4_empty);
			}

			ffield = edv_id3_libid3tag_get_frame_field_by_type(
				fframe,
				ID3_FIELD_TYPE_STRINGFULL
			);
			if(ffield != NULL)
			{
				if(id3_field_setfullstring(
					ffield,
					ucs4
				) != 0)
					status = -1;
				else
					status = 0;
			}

			g_free(ucs4);
#if (DEBUG_LEVEL >= 3)
g_print("   STRING=\"%s\"\n", s);
#endif
		}
		/* Image */
		else if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_IMAGE))
		{
			const gchar *s = (field->data != NULL) ?
				(gchar *)field->data : "";
			id3_ucs4_t *ucs4 = id3_utf8_ucs4duplicate((const id3_utf8_t *)s);

			ffield = edv_id3_libid3tag_get_frame_field_by_type(
				fframe,
				ID3_FIELD_TYPE_STRING
			);
			if(ffield != NULL)
			{
				if(id3_field_setstring(
					ffield,
					ucs4
				) != 0)
					status = -1;
				else
					status = 0;
			}

			g_free(ucs4);
#if (DEBUG_LEVEL >= 3)
g_print("   STRING=\"%s\"\n", s);
#endif
		}
		/* All other frames */
		else
		{
			const gchar *s = (field->data != NULL) ?
				(gchar *)field->data : "";
			id3_ucs4_t *ucs4 = id3_utf8_ucs4duplicate((const id3_utf8_t *)s);

			ffield = edv_id3_libid3tag_get_frame_field_by_type(
				fframe,
				ID3_FIELD_TYPE_STRINGLIST
			);
			if(ffield != NULL)
			{
				if(id3_field_setstrings(
					ffield,
					1,	/* One string */
					&ucs4
				) != 0)
					status = -1;
				else
					status = 0;
			}

			g_free(ucs4);
#if (DEBUG_LEVEL >= 3)
g_print("   STRING=\"%s\"\n", s);
#endif
		}
		break;

	    case EDV_ID3_FIELD_TYPE_DATA:
		/* Handle special case frames first
		 *
		 * Image
		 */
		if(!g_strcasecmp(id, EDV_ID3_FRAME_ID_IMAGE))
		{
/*
	APIC 5
	 0 0
	 1 1
	 2 10
	 3 4
	 4 15
*/
			ffield = edv_id3_libid3tag_get_frame_field_by_type(
				fframe,
				ID3_FIELD_TYPE_BINARYDATA
			);
			if(ffield != NULL)
			{
				if(id3_field_setbinarydata(
					ffield,
					(id3_byte_t *)field->data,
					(id3_length_t)field->data_length
				) != 0)
					status = -1;
				else
					status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   DATA=%i bytes\n", field->data_length);
#endif
			}
		}
		/* All other frames */
		else
		{
			ffield = edv_id3_libid3tag_get_frame_field_by_type(
				fframe,
				ID3_FIELD_TYPE_BINARYDATA
			);
			if(ffield != NULL)
			{
				if(id3_field_setbinarydata(
					ffield,
					(id3_byte_t *)field->data,
					(id3_length_t)field->data_length
				) != 0)
					status = -1;
				else
					status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   DATA=%i bytes\n", field->data_length);
#endif
			}
		}
		break;

	    case EDV_ID3_FIELD_TYPE_LANGUAGE:
		ffield = edv_id3_libid3tag_get_frame_field_by_type(
			fframe,
			ID3_FIELD_TYPE_LANGUAGE
		);
		if(ffield != NULL)
		{
			const gchar *s = (field->data != NULL) ?
				(gchar *)field->data : "";
			if(id3_field_setlanguage(
				ffield,
				s
			) != 0)
				status = -1;
			else
				status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   LANGUAGE=\"%s\"\n", s);
#endif
		}
		break;

	     case EDV_ID3_FIELD_TYPE_MIME_TYPE:
		/* MIME type field values are stored as
		 * ID3_FIELD_TYPE_LATIN1
		 */
		ffield = edv_id3_libid3tag_get_frame_field_by_type(
			fframe,
			ID3_FIELD_TYPE_LATIN1
		);
		if(ffield != NULL)
		{
			const gchar *s = (field->data != NULL) ?
				(gchar *)field->data : "";
			if(id3_field_setlatin1(
				ffield,
				(const id3_latin1_t *)s
			) != 0)
				status = -1;
			else
				status = 0;
#if (DEBUG_LEVEL >= 3)
g_print("   MIME TYPE=\"%s\"\n", s);
#endif
		}
		break;
	}

#if (DEBUG_LEVEL >= 1)
g_print("  Field done status=%i\n", status);
#endif

	return(status);
}
#endif	/* HAVE_LIBID3TAG */

#ifdef HAVE_LIBID3TAG
/*
 *	Called by edv_id3_tag_save_file() to update the
 *	corresponding id3_frame on the id3_tag with the value from
 *	the EDVID3Frame.
 */
static gint edv_id3_tag_save_iterate_frame(
	struct id3_tag *ftag,
	EDVID3Frame *frame,
	const EDVID3TagIOOptions io_options
)
{
	struct id3_frame *fframe;
	gint		status,
			status2;
	const gchar *id = frame->id;
	GList		*glist,
			*fields_list = frame->fields_list;
	EDVID3Field *field;

	if(STRISEMPTY(id))
		return(-2);

#if (DEBUG_LEVEL >= 1)
g_print(
 " Frame ID \"%s\" saving %i fields\n",
 id, g_list_length(fields_list)
);
#endif

	if(fields_list == NULL)
	{
		if(io_options & EDV_ID3_TAG_IO_EXCLUDE_EMPTY_FRAMES)
#if (DEBUG_LEVEL >= 1)
g_print("  No fields found on frame, skipping.\n");
#endif
			return(0);
	}

	/* Check if an id3_frame matching the specified EDVID3Frame's
	 * ID already exists
	 */
	fframe = id3_tag_findframe(
		ftag,
		id,
		0
	);
	if(fframe == NULL)
	{
		/* No matching id3_frame found, create a new one */
		fframe = id3_frame_new(id);
		if(fframe == NULL)
			return(-3);

		if(id3_tag_attachframe(
			ftag,
			fframe
		) != 0)
			return(-3);
	}

	status = 0;

	/* Update each id3_field on this id3_frame with the values
	 * from the EDVID3Fields on the EDVID3Frame
	 */
	for(glist = fields_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		field = EDV_ID3_FIELD(glist->data);
		if(field == NULL)
			continue;

		status2 = edv_id3_tag_save_iterate_field(
			fframe,
			frame,
			field,
			io_options
		);
		if(status2 != 0)
		{
			if(status == 0)
				status = status2;
		}
	}

#if (DEBUG_LEVEL >= 1)
g_print(
 " Frame ID \"%s\" done status=%i\n",
 id, status
);
#endif

	return(status);
}
#endif	/* HAVE_LIBID3TAG */

/*
 *	Adds/updates the ID3 tag to the file.
 *
 *	The tag specifies the EDVID3Tag to add/update to the file.
 *
 *	The path specifies the file.
 *
 *	The io_options specifies any options to be used when opening
 *	the ID3 tags from the file.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_id3_tag_save_file(
	EDVID3Tag *tag,
	const gchar *path,
	const EDVID3TagIOOptions io_options
)
{
#if defined(HAVE_LIBID3TAG)
	struct id3_tag *ftag;
	struct id3_file *ffp;
	gint		status,
			status2;
	GList *glist;
	EDVID3Frame *frame;

	if(STRISEMPTY(path) || (tag == NULL))
	{
		errno = EINVAL;
		return(-2);
	}

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tags_list_save_file() %s\n", path);
#endif
	/* Create a new id3_tag */
	ftag = id3_tag_new();
	if(ftag == NULL)
	{
		(void)id3_file_close(ffp);
		return(-1);
	}

	status = 0;

	/* Convert the values from the EDVID3Tag to the id3_tag */
	ftag->version = (unsigned int)(
		(((guint32)tag->version_major & 0x000000FF) << 8) |
		(((guint32)tag->version_minor & 0x000000FF) << 0)
	);
	ftag->flags = (int)tag->flags;
	ftag->extendedflags = (int)tag->extended_flags;
	ftag->restrictions = (int)tag->restriction_flags;

	/* Unsyncronization */
	if(tag->flags & EDV_ID3_TAG_FLAG_UNSYNCHRONISATION)
		ftag->options |= ID3_TAG_OPTION_UNSYNCHRONISATION;
	else
		ftag->options &= ~ID3_TAG_OPTION_UNSYNCHRONISATION;

	/* CRC 16 */
	if(tag->flags & EDV_ID3_TAG_FLAG_EXTENDED_HEADER)
	{
		if(tag->extended_flags & EDV_ID3_TAG_EXTENDED_FLAG_CRC_DATA_PRESENT)
			ftag->options |= ID3_TAG_OPTION_CRC;
		else
			ftag->options &= ~ID3_TAG_OPTION_CRC;
	}
	else
	{
		ftag->options &= ~ID3_TAG_OPTION_CRC;
	}

	/* Compress using GZip (zlib) */
	if(io_options & EDV_ID3_TAG_IO_COMPRESS_GZIP)
		ftag->options |= ID3_TAG_OPTION_COMPRESSION;
	else
		ftag->options &= ~ID3_TAG_OPTION_COMPRESSION;

	/* If the EDV_ID3_TAG_FLAG_FOOTER_PRESENT flag was set then
	 * we need to add the ID3_TAG_OPTION_APPENDEDTAG flag to
	 * the id3_tag's options in order to get it to render the
	 * footer
	 *
	 * We do not check for our EDV_ID3_TAG_IO_APPEND flag in the
	 * io_options to set the ID3_TAG_OPTION_APPENDEDTAG flag
	 * since we will check that later when we do the actual
	 * writing of the ID3v2 tag to the file
	 */
	if(tag->flags & EDV_ID3_TAG_FLAG_FOOTER_PRESENT)
		ftag->options |= ID3_TAG_OPTION_APPENDEDTAG;
	else
		ftag->options &= ~ID3_TAG_OPTION_APPENDEDTAG;

	/* ID3v2 */
	ftag->options &= ~ID3_TAG_OPTION_ID3V1;

#if (DEBUG_LEVEL >= 1)
g_print(
 "Generating ID3v2.%i.%i id3_tag.\n",
 tag->version_major,
 tag->version_minor
);
#endif
	/* Append and convert each EDVID3Frame on the EDVID3Tag
	 * to the id3_tag's id3_frames
	 */
	for(glist = tag->frames_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		frame = EDV_ID3_FRAME(glist->data);
		if(frame == NULL)
			continue;

		status2 = edv_id3_tag_save_iterate_frame(
			ftag,
			frame,
			io_options
		);
		if(status2 != 0)
		{
			/* Ignore these errors since many minor field
			 * conversion errors can be ignored without
			 * any bad consequences
			 */
		}
	}

	/* Insert the tag to the file */
	status2 = edv_id3_tag_save_file_write(
		path,
		ftag,
		io_options
	);
	if(status2 != 0)
	{
		if(status == 0)
			status = status2;
	}

	/* Delete the id3_tag */
	id3_tag_delete(ftag);

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tags_list_save_file(): done status=%i\n", status);
#endif

	return(status);
#else
	return(-7);
#endif	/* !HAVE_LIBID3TAG */
}


/*
 *	Removes all ID3 tags from the file.
 *
 *	The path specifies the file.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_id3_tags_list_strip_file(const gchar *path)
{
typedef struct {
	gulong		position,
			length;
} EDVID3TagStreamPosition;
	FILE		*in_fp,
			*out_fp;
	size_t		units_to_read,
			units_read,
			units_to_write,
			units_written;
	gboolean	found_id3v1_tag,
			found_id3v2_tag;
	gint status;
	gulong		position,
			length,
			id3v1_position,
			id3v1_length,
			id3v2_position,
			id3v2_length,
			io_buf_len = 0l,
			in_fp_pos;
	gchar		*in_path,
			*out_path;
	guint8 *io_buf;
	GList		*glist,
			*tag_positions_list;
	EDVVFSObject *obj;
	EDVID3TagStreamPosition *tag_position;

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tags_list_strip_file(): path=%s\n", path);
#endif

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Get the input and output file paths */
	in_path = g_strdup(path);
	out_path = edv_tmp_name(NULL);
	if((in_path == NULL) || (out_path == NULL))
	{
		g_free(in_path);
		(void)edv_unlink(out_path);
		g_free(out_path);
		return(-1);
	}

#if (DEBUG_LEVEL >= 3)
g_print(" in_path=%s\n", in_path);
g_print(" out_path=%s\n", out_path);
#endif

	/* Open the output file for append writing */
	out_fp = fopen(
		(const char *)out_path,
		"ab"
	);
	if(out_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s\n",
 g_strerror(error_code),
 out_path
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	obj = edv_vfs_object_fstat((gint)fileno(out_fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len == 0l)
		io_buf_len = 1l;

	/* Allocate the I/O buffer */
	io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
	if(io_buf == NULL)
	{
		(void)fclose(out_fp);
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	/* Open the input file for reading */
	in_fp = fopen(
		(const char *)in_path,
		"rb"
	);
	if(in_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s\n",
 g_strerror(error_code),
 in_path
);
#endif
		(void)fclose(out_fp);
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	in_fp_pos = 0l;

	/* List the ID3 tag locations in the input file */
	tag_position = NULL;
	tag_positions_list = NULL;
	while(TRUE)
	{
		/* Look for the next ID3v2 tag */
		found_id3v2_tag = edv_id3_stream_locate_id3v2(
			in_fp,
			NULL,
			NULL,
			&id3v2_position,
			&id3v2_length
		);

		/* Look for the next ID3v1 tag */
		if(fseek(
			in_fp,
			(long)in_fp_pos,
			SEEK_SET
		) != 0)
			break;

		found_id3v1_tag = edv_id3_stream_locate_id3v1(
			in_fp,
			&id3v1_position,
			&id3v1_length
		);

		/* Get the next tag's position based on which type
		 * of tag was found closer to the stream position
		 * at the start of this iteration
		 */
		if(found_id3v1_tag && found_id3v2_tag)
		{
			if(id3v1_position < id3v2_position)
			{
				position = id3v1_position;
				length = id3v1_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v1 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif

			}
			else
			{
				position = id3v2_position;
				length = id3v2_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v2 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif
			}
		}
		else if(found_id3v1_tag)
		{
			position = id3v1_position;
			length = id3v1_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v1 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif
		}
		else if(found_id3v2_tag)
		{
			position = id3v2_position;
			length = id3v2_length;
#if (DEBUG_LEVEL >= 3)
 g_print(" Existing ID3v2 tag found in input stream at %ld %ld\n",
   position, length
 );
#endif
		}
		else
		{
			/* No more tags found */
			break;
		}

		/* If a previous tag was found then check if this
		 * tag lies within or before it, in which case we
		 * need to break out of this potential infinite loop
		 */
		if(tag_position != NULL)
		{
			if(position < (tag_position->position + tag_position->length))
				break;
		}

		/* Record this tag position */
		tag_position = (EDVID3TagStreamPosition *)g_malloc(
			sizeof(EDVID3TagStreamPosition)
		);
		if(tag_position != NULL)
		{
			tag_positions_list = g_list_append(
				tag_positions_list,
				tag_position
			);
			tag_position->position = position;
			tag_position->length = length;
		}

		/* Seek the input stream just past this tag */
		in_fp_pos = position + length;
		if(fseek(
			in_fp,
			(long)in_fp_pos,
			SEEK_SET
		) != 0)
			break;
	}

	/* Seek the input stream back to the beginning and clear any
	 * errors
	 */
	rewind(in_fp);
	in_fp_pos = 0l;

	/* Copy the data from the input stream to the output stream */
	glist = tag_positions_list;
	while(!feof(in_fp))
	{
		/* Skip the tags from the input stream
		 *
		 * Seek the input stream and calculate the
		 * units to read based on the current item
		 * in the tag positions list then put the next
		 * item in the tag positions list into context
		 */
		while(glist != NULL)
		{
			tag_position = (EDVID3TagStreamPosition *)glist->data;

			/* Not reached yet? */
			if(in_fp_pos < tag_position->position)
				break;

			/* Reached the next tag, seek the input
			 * stream past it and update the
			 * recorded position
			 */
			in_fp_pos = tag_position->position + tag_position->length;
#if (DEBUG_LEVEL >= 3)
g_print(" Skipping tag in put stream at position %ld, seeking to %ld.\n",
 tag_position->position,
 in_fp_pos
);
#endif
			if(fseek(
				in_fp,
				(long)in_fp_pos,
				SEEK_SET
			) != 0)
				break;

			/* Get the next upcomming tag and check
			 * again
			 */
			glist = g_list_next(glist);
		}

		/* Calculate the number of units to read */
		if(glist != NULL)
		{
			tag_position = (EDVID3TagStreamPosition *)glist->data;
			units_to_read = (size_t)MIN(
				io_buf_len,
				tag_position->position - in_fp_pos
			) / sizeof(guint8);
		}
		else
		{
			units_to_read = (size_t)io_buf_len / sizeof(guint8);
		}
		if(units_to_read <= 0l)
			break;

		/* Read this block from the input stream */
		units_read = fread(
			io_buf,
			sizeof(guint8),
			units_to_read,
			in_fp
		);

		/* Update the recorded position of the input stream */
		in_fp_pos += (gulong)(units_read * sizeof(guint8));

		/* Write this block to the output stream */
		units_to_write = units_read;
		if(units_to_write > 0l)
		{
			units_written = fwrite(
				io_buf,
				sizeof(guint8),
				units_to_write,
				out_fp
			);
			if((units_written != units_to_write) ||
			   (ferror(out_fp))
			)
			{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fwrite(): %s: units_to_write=%ld units_written=%ld\n",
 g_strerror(error_code),
 (gulong)units_to_write,
 (gulong)units_written
);
#endif
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				g_free(out_path);
				(void)fclose(in_fp);
				g_free(in_path);
				g_free(io_buf);
				if(tag_positions_list != NULL)
				{
					g_list_foreach(tag_positions_list, (GFunc)g_free, NULL);
					g_list_free(tag_positions_list);
				}
				return(-1);
			}
		}

		/* Read error? */
		if(ferror(in_fp))
		{
#if (DEBUG_LEVEL >= 2)
g_print(" fread(): Error: units_to_read=%ld units_read=%ld\n",
 (gulong)units_to_read,
 (gulong)units_read
);
#endif
			(void)fclose(out_fp);
			(void)edv_unlink(out_path);
			g_free(out_path);
			(void)fclose(in_fp);
			g_free(in_path);
			g_free(io_buf);
			if(tag_positions_list != NULL)
			{
				g_list_foreach(tag_positions_list, (GFunc)g_free, NULL);
				g_list_free(tag_positions_list);
			}
			return(-1);
		}

		/* Nothing was read? */
		if(units_read <= 0l)
			break;
	}

	/* Delete the tag positions list */
	if(tag_positions_list != NULL)
	{
		g_list_foreach(tag_positions_list, (GFunc)g_free, NULL);
		g_list_free(tag_positions_list);
		tag_positions_list = NULL;
	}

	/* Close the input stream */
	(void)fclose(in_fp);

	/* Close the output stream */
	if(fclose(out_fp) != 0)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fclose(out_fp): %s.\n",
 g_strerror(error_code)
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		g_free(io_buf);
		return(-1);
	}


	/* Copy the output path back to the input path, opening the
	 * output file as the input file and vice versa
	 */
	in_fp = fopen(
		(const char *)out_path,
		"rb"
	);
	if(in_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s (opening output file as input stream)\n",
 g_strerror(error_code),
 out_path
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		g_free(io_buf);
		return(-1);
	}

	out_fp = fopen(
		(const char *)in_path,
		"wb"
	);
	if(out_fp == NULL)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fopen() %s: %s (opening input file as output stream)\n",
 g_strerror(error_code),
 in_path
);
#endif
		(void)fclose(in_fp);
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		g_free(io_buf);
		return(-1);
	}

	/* Copy the data from the output file (now the input stream)
	 * to the input file (now the output stream)
	 */
	while(!feof(in_fp))
	{
		/* Calculate the number of units to read */
		units_to_read = (size_t)io_buf_len / sizeof(guint8);

		/* Read this block from the input stream */
		units_read = fread(
			io_buf,
			sizeof(guint8),
			units_to_read,
			in_fp
		);

		/* Write this block to the output stream */
		units_to_write = units_read;
		if(units_to_write > 0l)
		{
			units_written = fwrite(
				io_buf,
				sizeof(guint8),
				units_to_write,
				out_fp
			);
			if((units_written != units_to_write) ||
			   (ferror(out_fp))
			)
			{
#if (DEBUG_LEVEL >= 2)
			const gint error_code = (gint)errno;
g_print(" fwrite(): %s: units_to_write=%ld units_written=%ld\n",
 g_strerror(error_code),
 (gulong)units_to_write,
 (gulong)units_written
);
#endif
				(void)fclose(in_fp);
				(void)edv_unlink(out_path);
				g_free(out_path);
				(void)fclose(out_fp);
				g_free(in_path);
				g_free(io_buf);
				return(-1);
			}
		}

		/* Read error? */
		if(ferror(in_fp))
		{
#if (DEBUG_LEVEL >= 2)
g_print(" fread(): Error: units_to_read=%ld units_read=%ld\n",
 (gulong)units_to_read,
 (gulong)units_read
);
#endif
			(void)fclose(in_fp);
			(void)edv_unlink(out_path);
			g_free(out_path);
			(void)fclose(out_fp);
			g_free(in_path);
			g_free(io_buf);
			return(-1);
		}

		/* Nothing was read? */
		if(units_read <= 0l)
			break;
	}

	/* Delete the IO buffer */
	g_free(io_buf);

	/* Close the output file (now the input stream) */
	(void)fclose(in_fp);

	/* Close the input file (now the output stream) */
	if(fclose(out_fp) != 0)
	{
#if (DEBUG_LEVEL >= 2)
const gint error_code = (gint)errno;
g_print(" fclose(out_fp): %s.\n",
 g_strerror(error_code)
);
#endif
		(void)edv_unlink(out_path);
		g_free(out_path);
		g_free(in_path);
		return(-1);
	}

	/* Remove the output file */
	(void)edv_unlink(out_path);

	g_free(out_path);
	g_free(in_path);

#if (DEBUG_LEVEL >= 1)
g_print("edv_id3_tags_list_save_strip_write() done status=%i\n", status);
#endif

	return(status);
}


/*
 *	Searches for the EDVID3Frame in the EDVID3Tags list by ID.
 *
 *	The tags_list specifies the GList of EDVID3Tag * tags
 *	to search through.
 *
 *	The id specifies the ID of the EDVID3Frame to search for.
 *
 *	Returns the pointer to the EDVID3Frame or NULL on error.
 */
EDVID3Frame *edv_id3_tags_list_find_frame_by_id(
	GList *tags_list,
	const gchar *id
)
{
	GList *glist;
	EDVID3Frame *frame;
	EDVID3Tag *tag;

	if(id == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = tags_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		tag = EDV_ID3_TAG(glist->data);
		if(tag == NULL)
			continue;

		frame = edv_id3_frames_list_find_by_id(
			tag->frames_list,
			id
		);
		if(frame != NULL)
			return(frame);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Searches for the EDVID3Tag in the EDVID3Tags list that has
 *	the EDVID3Frame.
 *
 *	The tags_list specifies the GList of EDVID3Tag * tags
 *	to search through.
 *
 *	The frame specifies the EDVID3Frame of the EDVID3Tag to
 *	search for.
 *
 *	Returns the pointer to the EDVID3Tag or NULL on error.
 */
EDVID3Tag *edv_id3_tags_list_find_tag_by_frame(
	GList *tags_list,
	EDVID3Frame *frame
)
{
	GList *glist;
	EDVID3Tag *tag;

	if(frame == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = tags_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		tag = EDV_ID3_TAG(glist->data);
		if(tag == NULL)
			continue;

		if(tag->frames_list != NULL)
		{
			GList *glist;
			EDVID3Frame *frame2;
			for(glist = tag->frames_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				frame2 = EDV_ID3_FRAME(glist->data);
				if(frame2 == frame)
					return(tag);
			}
		}
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Deletes the ID3 tags list.
 *
 *	The tags_list specifies the GList of EDVID3Tag * tags to
 *	delete.
 */
GList *edv_id3_tags_list_delete(GList *tags_list)
{
	if(tags_list == NULL)
		return(NULL);

	g_list_foreach(
		tags_list,
		(GFunc)edv_id3_tag_delete,
		NULL
	);
	g_list_free(tags_list);

	return(NULL);
}


/*
	"Three Centers Of One World"

	There are three adult MALEs who follow a religious belief
	known as Catholisim, they call themselves Catholics. In
	this belief, there are certain leaders of groups of
	other Catholics, these leaders are called Fathers.

	So there are these three Fathers, they have not mated with
	a FEMALE, popular amoung their belief is that mating without
	a contract, which they call a Marriage, is considered bad,
	which they called a Sin. Decided by only reasons know to
	them have also not entered into any such contract.

	Also popular amoung Catholics in general, is the unspoken
	agreement that a heightened priority of their efforts be put
	into the caring of maturating and developing persons, called
	Fetuses, that they mature to the point past maturation,
	called Birth and subsequently become Children. These Fetuses
	need not be related to them or even known to them in
	specific for Catholics to invest effort of their concearns
	towards.

	But its not been demostrated that Catholics express any
	such concearn towards Fetuses past maturation, called
	Children. Though it has been claimed by Catholics and
	commanded by Fathers that an equal amount of care should be
	expended on Children.

	The same amount of implicit care is observed for MALE
	Catholics to avoid having relationships with other MALE
	Catholics or MALE NON-Catholics. Most Catholics passively
	but diligently monitor MALE Catholics to identity any such
	relationship by means of detecting certain behavioral
	traits such as behaviors common to FEMALE Catholics as an
	identifier that the MALE Catholic is having a relationship
	with other MALE Catholics or MALE non-Catholics.

	The caring of children is a behavior often associated with
	FEMALE adults, both Catholic and NON-Catholic.

	The three fathers expend a large amount of their time and
	effort on the caring of children and not; mating, entering
	into contracts, or FEMALEs. You will also find that this
	last sentence stating that the above was written with the
	intention for artistic observation and creation with no
	intention to be a threat or attack to any group is often
	excluded from citations.
 */
