#include <errno.h>
#include <glib.h>

#include "edv_id3.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0			/* 0 = off
						 * 1 - 3 = on */
#endif
#if (DEBUG_LEVEL > 0)
# warning "Debugging enabled."
#endif


/* EDVID3Tag */
EDVID3Tag *edv_id3_tag_new(void);
EDVID3Tag *edv_id3_tag_copy(EDVID3Tag *tag);
void edv_id3_tag_delete(EDVID3Tag *tag);

EDVID3Frame *edv_id3_tag_get_frame_by_id(
	EDVID3Tag *tag,
	const gchar *id
);
gboolean edv_id3_tag_frame_exists(
	EDVID3Tag *tag,
	EDVID3Frame *frame
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVID3Tag.
 *
 *	Returns a new dynamically allocated EDVID3Tag with all its
 *	values zero'ed or NULL on error.
 */
EDVID3Tag *edv_id3_tag_new(void)
{
	return(EDV_ID3_TAG(g_malloc0(sizeof(EDVID3Tag))));
}

/*
 *	Coppies the EDVID3Tag.
 *
 *	The tag specifies the EDVID3Tag to copy.
 *
 *	Returns a new dynamically allocated copy of the EDVID3Tag or
 *	NULL on error.
 */
EDVID3Tag *edv_id3_tag_copy(EDVID3Tag *tag)
{
	EDVID3Tag	*src_tag = tag,
			*tar_tag;

	if(src_tag == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	tar_tag = edv_id3_tag_new();
	if(tar_tag == NULL)
		return(NULL);

	tar_tag->version_major = src_tag->version_major;
	tar_tag->version_minor = src_tag->version_minor;
	tar_tag->flags = src_tag->flags;
	tar_tag->extended_flags = src_tag->extended_flags;
	tar_tag->restriction_flags = src_tag->restriction_flags;
	if(src_tag->frames_list != NULL)
	{
		GList *glist;
		for(glist = src_tag->frames_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
			tar_tag->frames_list = g_list_append(
				tar_tag->frames_list,
				edv_id3_frame_copy(EDV_ID3_FRAME(glist->data))
			);
	}

	return(tar_tag);
}

/*
 *	Deletes the EDVID3Tag.
 */
void edv_id3_tag_delete(EDVID3Tag *tag)
{
	if(tag == NULL)
		return;

	tag->frames_list = edv_id3_frames_list_delete(tag->frames_list);
	g_free(tag);
}


/*
 *	Gets the EDVID3Frame on the EDVID3Tag by ID.
 *
 *	The tag specifies the EDVID3Tag.
 *
 *	The id specifies the ID.
 *
 *	Returns the pointer to the EDVID3Frame with the matching id
 *	or NULL on error.
 */
EDVID3Frame *edv_id3_tag_get_frame_by_id(
	EDVID3Tag *tag,
	const gchar *id
)
{
	if(tag == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	return(edv_id3_frames_list_find_by_id(
		tag->frames_list,
		id
	));
}

/*
 *	Checks if the EDVID3Frame is on the EDVID3Tag.
 */
gboolean edv_id3_tag_frame_exists(
	EDVID3Tag *tag,
	EDVID3Frame *frame
)
{
	if((tag == NULL) || (frame == NULL))
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(g_list_index(tag->frames_list, frame) > -1)
		return(TRUE);

	errno = EINVAL;

	return(FALSE);
}
