#include <errno.h>
#include <glib.h>

#include "edv_id3.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0			/* 0 = off
						 * 1 - 3 = on */
#endif
#if (DEBUG_LEVEL > 0)
# warning "Debugging enabled."
#endif


/* EDVID3Frame */
EDVID3Frame *edv_id3_frame_new(void);
EDVID3Frame *edv_id3_frame_new_with_values(
	const gchar *id,
	const EDVID3TextEncoding text_encoding,
	GList *fields_list,
	const gchar *description
);
EDVID3Frame *edv_id3_frame_copy(EDVID3Frame *frame);
void edv_id3_frame_delete(EDVID3Frame *frame);

gint edv_id3_frame_set_id(
	EDVID3Frame *frame,
	const gchar *id
);

gint edv_id3_frame_get_text_bit_size(EDVID3Frame *frame);

EDVID3Field *edv_id3_frame_get_field(
	EDVID3Frame *frame,
	const gint i
);
EDVID3Field *edv_id3_frame_get_field_by_type(
	EDVID3Frame *frame,
	const EDVID3FieldType type
);
EDVID3Field *edv_id3_frame_insert_field(
	EDVID3Frame *frame,
	const gint i
);
EDVID3Field *edv_id3_frame_append_field(EDVID3Frame *frame);

/* EDVID3Frames List */
EDVID3Frame *edv_id3_frames_list_find_by_id(
	GList *frames_list,
	const gchar *id
);
GList *edv_id3_frames_list_delete(GList *frames_list);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVID3Frame.
 *
 *	Returns a new dynamically allocated EDVID3Frame with all its
 *	values zero'ed or NULL on error.
 */
EDVID3Frame *edv_id3_frame_new(void)
{
	return(EDV_ID3_FRAME(g_malloc0(sizeof(EDVID3Frame))));
}

/*
 *	Creates a new EDVID3Frame with values.
 *
 *	The id specifies the string describing the ID.
 *
 *	The fields_list specifies a GList of EDVID3Field * fields that
 *	will be coppied to the new EDVID3Frame.
 *
 *	The description specifies a verbose string describing the
 *	EDVID3Frame.
 *
 *	Returns a new dynamically allocated EDVID3Frame with the
 *	specified values or NULL on error.
 */
EDVID3Frame *edv_id3_frame_new_with_values(
	const gchar *id,
	const EDVID3TextEncoding text_encoding,
	GList *fields_list,
	const gchar *description
)
{
	EDVID3Frame *frame = edv_id3_frame_new();
	if(frame == NULL)
		return(NULL);

	if(id != NULL)
		frame->id = g_strdup(id);

	frame->text_encoding = text_encoding;

	if(fields_list != NULL)
	{
		GList *glist;
		EDVID3Field *field;
		for(glist = fields_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			field = EDV_ID3_FIELD(glist->data);
			if(field == NULL)
				continue;

			frame->fields_list = g_list_append(
				frame->fields_list,
				edv_id3_field_copy(field)
			);
		}
	}

	if(description != NULL)
		frame->description = g_strdup(description);

	return(frame);
}

/*
 *	Coppies the EDVID3Frame.
 *
 *	The frame specifies the EDVID3Frame to copy.
 *
 *	Returns a new dynamically allocated copy of the EDVID3Frame or
 *	NULL on error.
 */
EDVID3Frame *edv_id3_frame_copy(EDVID3Frame *frame)
{
	EDVID3Frame	*src_frame = frame,
			*tar_frame;

	if(src_frame == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	tar_frame = edv_id3_frame_new_with_values(
		src_frame->id,
		src_frame->text_encoding,
		src_frame->fields_list,
		src_frame->description
	);
	if(tar_frame == NULL)
		return(NULL);

	return(tar_frame);
}

/*
 *	Deletes the EDVID3Frame.
 */
void edv_id3_frame_delete(EDVID3Frame *frame)
{
	if(frame == NULL)
		return;

	g_free(frame->id);
	frame->fields_list = edv_id3_fields_list_delete(frame->fields_list);
	g_free(frame->description);
	g_free(frame);
}


/*
 *	Sets the EDVID3Frame's ID.
 *
 *	The frame specifies the EDVID3Frame to set.
 *
 *	The id specifies the new ID to set. If id is NULL then the
 *	EDVID3Frame's ID will be cleared.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_id3_frame_set_id(
	EDVID3Frame *frame,
	const gchar *id
)
{
	if(frame == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	g_free(frame->id);
	frame->id = STRDUP(id);

	return(0);
}


/*
 *	Gets the number of bits per character set on the EDVID3Frame
 *	based on its text_encoding.
 *
 *	Returns the number of bits per character or 0 on error.
 */
gint edv_id3_frame_get_text_bit_size(EDVID3Frame *frame)
{
	if(frame == NULL)
	{
		errno = EINVAL;
		return(0);
	}

	switch(frame->text_encoding)
	{
	    case EDV_ID3_TEXT_ENCODING_UNKNOWN:
		break;
	    case EDV_ID3_TEXT_ENCODING_UTF_8:
		return(8);
		break;
	    case EDV_ID3_TEXT_ENCODING_UTF_16:
		return(16);
		break;
	    case EDV_ID3_TEXT_ENCODING_UTF_16_BE:
		return(16);
		break;
	    case EDV_ID3_TEXT_ENCODING_ISO_8859_1:/* Latin 8 bit */
		return(8);
		break;
	}

	errno = EINVAL;

	return(0);
}


/*
 *	Gets the EDVID3Field on the EDVID3Frame by index.
 *
 *	The frame specifies the EDVID3Frame.
 *
 *	The i specifies the index of the EDVID3Frame to get.
 *
 *	Returns the pointer to the EDVID3Field on the EDVID3Frame or
 *	NULL on error.
 */
EDVID3Field *edv_id3_frame_get_field(
	EDVID3Frame *frame,
	const gint i
)
{
	if((frame == NULL) || (i < 0))
	{
		errno = EINVAL;
		return(NULL);
	}

	return(EDV_ID3_FIELD(g_list_nth_data(
		frame->fields_list,
		(guint)i
	)));
}

/*
 *	Gets the EDVID3Field on the EDVID3Frame by type.
 *
 *	The frame specifies the EDVID3Frame.
 *
 *	The type specifies the EDVID3FieldType type, the type may not
 *	be EDV_ID3_FIELD_TYPE_ANY.
 *
 *	Returns the pointer to the EDVID3Field on the EDVID3Frame or
 *	NULL on error.
 */
EDVID3Field *edv_id3_frame_get_field_by_type(
	EDVID3Frame *frame,
	const EDVID3FieldType type
)
{
	if(frame == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	return(edv_id3_fields_list_find_by_type(
		frame->fields_list,
		type
	));
}


/*
 *	Inserts a new EDVID3Field on the EDVID3Frame.
 *
 *	The frame specifies the EDVID3Frame to insert a new
 *	EDVID3Field on.
 *
 *	The i specifies the index position to insert the new
 *	EDVID3Field at. If i is negative then the new EDVID3Field will
 *	be appended.
 *
 *	Returns a new dynamically allocated EDVID3Field with all of
 *	its values zero'ed or NULL on error.
 */
EDVID3Field *edv_id3_frame_insert_field(
	EDVID3Frame *frame,
	const gint i
)
{
	EDVID3Field *field;

	if(frame == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	field = edv_id3_field_new();
	if(field == NULL)
		return(NULL);

	if(i < 0)
		frame->fields_list = g_list_append(
			frame->fields_list,
			field
		);
	else
		frame->fields_list = g_list_insert(
			frame->fields_list,
			field,
			(guint)i
		);

	return(field);
}

/*
 *	Appends a new EDVID3Field on the EDVID3Frame.
 *
 *	The frame specifies the EDVID3Frame to append a new
 *	EDVID3Field on.
 *
 *	Returns a new dynamically allocated EDVID3Field with all of
 *	its values zero'ed or NULL on error.
 */
EDVID3Field *edv_id3_frame_append_field(EDVID3Frame *frame)
{
	return(edv_id3_frame_insert_field(
		frame,
		-1				/* Append */
	));
}


/*
 *	Searches for a EDVID3Frame in a EDVID3Frames list by ID.
 *
 *	The frames_list specifies the GList of EDVID3Frame * frames
 *	to search through.
 *
 *	The id specifies the ID to find.
 *
 *	Returns the pointer to the EDVID3Frame with the matching id
 *	or NULL on error.
 */
EDVID3Frame *edv_id3_frames_list_find_by_id(
	GList *frames_list,
	const gchar *id
)
{
	GList *glist;
	EDVID3Frame *frame;

	if(id == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = frames_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		frame = EDV_ID3_FRAME(glist->data);
		if(frame == NULL)
			continue;

		if(frame->id == NULL)
			continue;

		if(!g_strcasecmp(frame->id, id))
			return(frame);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Deletes the ID3 frames list.
 *
 *	The frames_list specifies the GList of EDVID3Frame * frames
 *	to delete.
 *
 *	This function always returns NULL.
 */
GList *edv_id3_frames_list_delete(GList *frames_list)
{
	if(frames_list == NULL)
		return(NULL);

	g_list_foreach(
		frames_list,
		(GFunc)edv_id3_frame_delete,
		NULL
	);
	g_list_free(frames_list);

	return(NULL);
}
