#include <errno.h>
#include <sys/types.h>
#include <glib.h>

#include "imgio.h"

#include "edv_id3.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0			/* 0 = off
						 * 1 - 3 = on */
#endif
#if (DEBUG_LEVEL > 0)
# warning "Debugging enabled."
#endif


/* EDVID3Field Image */
GList *edv_id3_field_get_image_rgba(
	EDVID3Field *field,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color
);
GList *edv_id3_field_get_image_rgba_details(
	EDVID3Field *field,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn		/* In seconds since EPOCH */
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the field's data as an RGBA image.
 *
 *	The field specifies the EDVID3Field, which must be of type
 *	EDV_ID3_FIELD_TYPE_DATA.
 *
 *	Returns a dynamically allocated GList of guint8 * RGBA image
 *	field datas or NULL on error.
 */
GList *edv_id3_field_get_image_rgba(
	EDVID3Field *field,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color
)
{
	gulong		play_time_ms = 0l,
			modified_time_sec = 0l;
	gchar		*creator = NULL,
			*title = NULL,
			*author = NULL,
			*comments = NULL;
	GList *rgba_list = edv_id3_field_get_image_rgba_details(
		field,
		width_rtn, height_rtn,
		bpl_rtn,
		nframes_rtn,
		delay_list_rtn,
		bg_color,
		&play_time_ms,
		&creator, &title,
		&author, &comments,
		&modified_time_sec
	);

	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(rgba_list);
}

GList *edv_id3_field_get_image_rgba_details(
	EDVID3Field *field,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn		/* In seconds since EPOCH */
)
{
	gconstpointer data;
	gint data_length;
	const guint8 def_alpha_value = 0xFF;
	guint8 def_bg_color[4];

	if(width_rtn != NULL)
		*width_rtn = 0;
	if(height_rtn != NULL)
		*height_rtn = 0;
	if(bpl_rtn != NULL)
		*bpl_rtn = 0;
	if(nframes_rtn != NULL)
		*nframes_rtn = 0;
	if(delay_list_rtn != NULL)
		*delay_list_rtn = NULL;
	if(play_time_ms_rtn != NULL)
		*play_time_ms_rtn = 0l;
	if(creator_rtn != NULL)
		*creator_rtn = NULL;
	if(title_rtn != NULL)
		*title_rtn = NULL;
	if(author_rtn != NULL)
		*author_rtn = NULL;
	if(comments_rtn != NULL)
		*comments_rtn = NULL;
	if(modified_time_sec_rtn != NULL)
		*modified_time_sec_rtn = 0l;


	/* If no background color was specified then use the default
	 * background color
	 */
	if(bg_color == NULL)
	{
		bg_color = def_bg_color;
		bg_color[0] = 0xFF;
		bg_color[1] = 0xFF;
		bg_color[2] = 0xFF;
		bg_color[3] = 0xFF;
	}

	if(field == NULL)
		return(NULL);

	if(field->type != EDV_ID3_FIELD_TYPE_DATA)
		return(NULL);

	data = field->data;
	data_length = field->data_length;
	if((data == NULL) || (data_length <= 0))
		return(NULL);

#ifdef HAVE_LIBGIF
	if(ImgBufferIsGIF(
		(const u_int8_t *)data,
		(int)data_length
	))
	{
		int	nframes = 0,
			x = 0, y = 0,
			base_width = 0, base_height = 0,
			user_aborted = 0;
		u_int8_t **rgba_list = NULL;
		unsigned long *delay_list = NULL;

		(void)ImgBufferReadGIFRGBA(
			(const void *)data,
			(const unsigned long)data_length,
			width_rtn, height_rtn,
			bpl_rtn,
			&rgba_list,
			&delay_list,
			&nframes,
			bg_color,
			&x, &y,
			&base_width, &base_height,
			creator_rtn, title_rtn,
			author_rtn, comments_rtn,
			def_alpha_value,
			NULL, NULL,
			&user_aborted
		);

		if(rgba_list != NULL)
		{
			gint i;
			GList	*gframes_list = NULL,
				*gdelay_list = NULL;

			for(i = 0; i < nframes; i++)
			{
				gframes_list = g_list_append(
					gframes_list,
					rgba_list[i]
				);
				if(delay_list != NULL)
				{
					gdelay_list = g_list_append(
						gdelay_list,
						(gpointer)delay_list[i]
					);
					if(play_time_ms_rtn != NULL)
						*play_time_ms_rtn = (*play_time_ms_rtn) + delay_list[i];
				}
			}

			if(nframes_rtn != NULL)
				*nframes_rtn = nframes;
			if(delay_list_rtn != NULL)
				*delay_list_rtn = gdelay_list;
			else
				g_list_free(gdelay_list);

			g_free(rgba_list);

			return(gframes_list);
		}

		return(NULL);
	}
#endif	/* HAVE_LIBGIF */
#ifdef HAVE_LIBJPEG
	if(ImgBufferIsJPEG(
		(const u_int8_t *)data,
		(int)data_length
	))
	{
		int user_aborted = 0;
		u_int8_t *rgba = NULL;

		(void)ImgBufferReadJPEGRGBA(
			(const void *)data,
			(const unsigned long)data_length,
			width_rtn, height_rtn,
			bpl_rtn,
			&rgba,
			NULL, NULL,
			&user_aborted
		);
		if(rgba != NULL)
		{
			GList *gframes_list = g_list_append(
				NULL,
				rgba
			);

			if(nframes_rtn != NULL)
				*nframes_rtn = 1;

			return(gframes_list);
		}

		return(NULL);
	}
#endif	/* HAVE_LIBJPEG */
#ifdef HAVE_LIBPNG
	if(ImgBufferIsPNG(
		(const u_int8_t *)data,
		(int)data_length
	))
	{
		int	x = 0, y = 0,
			base_width = 0, base_height = 0,
			user_aborted = 0;
		u_int8_t *rgba = NULL;

		(void)ImgBufferReadPNGRGBA(
			(const void *)data,
			(const unsigned long)data_length,
			width_rtn, height_rtn,
			bpl_rtn,
			&rgba,
			bg_color,
			&x, &y,
			&base_width, &base_height,
			creator_rtn, title_rtn,
			author_rtn, comments_rtn,
			modified_time_sec_rtn,
			def_alpha_value,
			NULL, NULL,
			&user_aborted
		);

		if(rgba != NULL)
		{
			GList *gframes_list = g_list_append(
				NULL,
				rgba
			);

			if(nframes_rtn != NULL)
				*nframes_rtn = 1;

			return(gframes_list);
		}

		return(NULL);
	}
#endif	/* HAVE_LIBPNG */

	return(NULL);
}


