#include <errno.h>
#include <glib.h>

#include "edv_id3.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0			/* 0 = off
						 * 1 - 3 = on */
#endif
#if (DEBUG_LEVEL > 0)
# warning "Debugging enabled."
#endif


/* EDVID3Field */
EDVID3Field *edv_id3_field_new(void);
EDVID3Field *edv_id3_field_new_with_values(
	const EDVID3FieldType type,
	gconstpointer data,
	const gint data_len
);
EDVID3Field *edv_id3_field_new_with_string_value(const gchar *s);
EDVID3Field *edv_id3_field_copy(EDVID3Field *field);
void edv_id3_field_delete(EDVID3Field *field);

gint edv_id3_field_get_number(EDVID3Field *field);
glong edv_id3_field_get_long(EDVID3Field *field);
const gchar *edv_id3_field_get_string(EDVID3Field *field);
gint edv_id3_field_set_string(
	EDVID3Field *field,
	const gchar *s
);

/* EDVID3Fields List */
EDVID3Field *edv_id3_fields_list_find_by_type(
	GList *fields_list,
	const EDVID3FieldType type
);
GList *edv_id3_fields_list_delete(GList *fields_list);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVID3Field.
 *
 *	Returns a new dynamically allocated EDVID3Field with all its
 *	values zero'ed or NULL on error.
 */
EDVID3Field *edv_id3_field_new(void)
{
	return(EDV_ID3_FIELD(g_malloc0(sizeof(EDVID3Field))));
}

/*
 *	Creates a new EDVID3Field with values.
 *
 *	The type specifies the EDVID3Field's value type.
 *
 *	If data is not NULL then it specifies the pointer to the data
 *	to be coppied to the new EDVID3Field.
 *
 *	The data_length specifies the length of data in bytes. If data
 *	is NULL then this value is ignored.
 *
 *	Returns a new dynamically allocated EDVID3Field with the
 *	specified values or NULL on error.
 */
EDVID3Field *edv_id3_field_new_with_values(
	const EDVID3FieldType type,
	gconstpointer data,
	const gint data_length
)
{
	EDVID3Field *field = edv_id3_field_new();
	if(field == NULL)
		return(NULL);

	field->type = type;

	if((data != NULL) && (data_length > 0))
	{
		field->data = g_memdup(
			data,
			data_length
		);
		if(field->data != NULL)
			field->data_length = data_length;
	}

	return(field);
}

/*
 *	Creates a new EDVID3Field with a string value.
 *
 *	The s pecifies the stirng.
 *
 *	Returns a new dynamically allocated EDVID3Field with the
 *	specified values or NULL on error.
 */
EDVID3Field *edv_id3_field_new_with_string_value(const gchar *s)
{
	EDVID3Field *field = edv_id3_field_new();
	if(field == NULL)
		return(NULL);

	field->type = EDV_ID3_FIELD_TYPE_STRING;

	if(s != NULL)
	{
		field->data = g_strdup(s);
		field->data_length = (field->data != NULL) ?
			((gint)strlen((char *)field->data) + 1) : 0;
	}

	return(field);
}

/*
 *	Coppies the EDVID3Field.
 *
 *	The field specifies the EDVID3Field to copy.
 *
 *	Returns a new dynamically allocated copy of the EDVID3Field or
 *	NULL on error.
 */
EDVID3Field *edv_id3_field_copy(EDVID3Field *field)
{
	EDVID3Field	*src_field = field,
			*tar_field;

	if(src_field == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	tar_field = edv_id3_field_new_with_values(
		src_field->type,
		src_field->data,
		src_field->data_length
	);
	if(tar_field == NULL)
		return(NULL);

	return(tar_field);
}

/*
 *	Deletes the EDVID3Field.
 */
void edv_id3_field_delete(EDVID3Field *field)
{
	if(field == NULL)
		return;

	g_free(field->data);
	g_free(field);
}


/*
 *	Gets the EDVID3Field's data as a number.
 *
 *	The field specifies the EDVID3Field, which must be of type
 *	EDV_ID3_FIELD_TYPE_INT8, EDV_ID3_FIELD_TYPE_INT16,
 *	EDV_ID3_FIELD_TYPE_INT32, or EDV_ID3_FIELD_TYPE_INT64.
 *
 *	Returns the EDVID3Field's value as a gint or 0 on error.
 */
gint edv_id3_field_get_number(EDVID3Field *field)
{
	gint data_length;
	gconstpointer data;
	EDVID3FieldType type;

	if(field == NULL)
	{
		errno = EINVAL;
		return(0);
	}

	data = field->data;
	data_length = field->data_length;
	if((data == NULL) || (data_length <= 0))
	{
		errno = ENOENT;
		return(0l);
	}

	type = field->type;

	if((type == EDV_ID3_FIELD_TYPE_INT8) &&
	   (data_length >= sizeof(gint8))
	)
		return((gint)(*(gint8 *)data));

	if((type == EDV_ID3_FIELD_TYPE_INT16) &&
	   (data_length >= sizeof(gint16))
	)
		return((gint)(*(gint16 *)data));

	if((type == EDV_ID3_FIELD_TYPE_INT32) &&
	   (data_length >= sizeof(gint32))
	)
		return((gint)(*(gint32 *)data));

	if((type == EDV_ID3_FIELD_TYPE_INT64) &&
	   (data_length >= sizeof(gint64))
	)
		return((gint)(*(gint64 *)data));

	/* Data type is not a number */
	errno = EINVAL;

	return(0);
}

/*
 *	Gets the field's data as a glong.
 *
 *	The field specifies the EDVID3Field, which must be of type
 *	EDV_ID3_FIELD_TYPE_INT8, EDV_ID3_FIELD_TYPE_INT16,
 *	EDV_ID3_FIELD_TYPE_INT32, or EDV_ID3_FIELD_TYPE_INT64.
 *
 *	Returns the field's value as a glong or 0 on error.
 */
glong edv_id3_field_get_long(EDVID3Field *field)
{
	gint data_length;
	gconstpointer data;
	EDVID3FieldType type;

	if(field == NULL)
	{
		errno = EINVAL;
		return(0l);
	}

	data = field->data;
	data_length = field->data_length;
	if((data == NULL) || (data_length <= 0))
	{
		errno = ENOENT;
		return(0l);
	}

	type = field->type;

	if((type == EDV_ID3_FIELD_TYPE_INT8) &&
	   (data_length >= sizeof(gint8))
	)
		return((glong)(*(gint8 *)data));

	if((type == EDV_ID3_FIELD_TYPE_INT16) &&
	   (data_length >= sizeof(gint16))
	)
		return((glong)(*(gint16 *)data));

	if((type == EDV_ID3_FIELD_TYPE_INT32) &&
	   (data_length >= sizeof(gint32))
	)
		return((glong)(*(gint32 *)data));

	if((type == EDV_ID3_FIELD_TYPE_INT64) &&
	   (data_length >= sizeof(gint64))
	)
		return((glong)(*(gint64 *)data));

	/* Data type is not a number */
	errno = EINVAL;

	return(0l);
}

/*
 *	Gets the field's data as a string.
 *
 *	The field specifies the EDVID3Field, which must be of type
 *	EDV_ID3_FIELD_TYPE_STRING, EDV_ID3_FIELD_TYPE_LANGUAGE, or
 *	EDV_ID3_FIELD_TYPE_MIME_TYPE.
 *
 *	Returns the pointer to the field's string or NULL on error.
 */
const gchar *edv_id3_field_get_string(EDVID3Field *field)
{
	gint data_length;
	gconstpointer data;
	EDVID3FieldType type;

	if(field == NULL)
	{
		errno = EINVAL;
		return(0l);
	}

	data = field->data;
	data_length = field->data_length;
	if((data == NULL) || (data_length <= 0))
	{
		errno = ENOENT;
		return(0l);
	}

	type = field->type;

	if((type == EDV_ID3_FIELD_TYPE_STRING) ||
	   (type == EDV_ID3_FIELD_TYPE_LANGUAGE) ||
	   (type == EDV_ID3_FIELD_TYPE_MIME_TYPE)
	)
		return((const gchar *)data);

	/* Data type is not a string */
	errno = EINVAL;

	return(NULL);
}

/*
 *	Sets the field's type to EDV_ID3_FIELD_TYPE_STRING and sets its
 *	data as a string.
 *
 *	The field specifies the EDVID3Field to set.
 *
 *	The s specifies the string to set. If s is NULL then the value
 *	will be cleared (the type will still be set).
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_id3_field_set_string(
	EDVID3Field *field,
	const gchar *s
)
{
	if(field == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	/* Delete the previous value */
	g_free(field->data);

	/* Explicitly set/change the field's type */
	field->type = EDV_ID3_FIELD_TYPE_STRING;

	/* Set the new value */
	if(s != NULL)
	{
		field->data = g_strdup(s);
		field->data_length = (field->data != NULL) ?
			((gint)strlen((char *)field->data) + 1) : 0;
	}
	else
	{
		field->data = NULL;
		field->data_length = 0;
	}

	return(0);
}


/*
 *	Searches for a EDVID3Field in a EDVID3Fields list by ID.
 *
 *	The fields_list specifies the GList of EDVID3Field * fields
 *	to search through.
 *
 *	The type specifies the EDVID3FieldType type, the type may not
 *      be EDV_ID3_FIELD_TYPE_ANY.
 *
 *	Returns the pointer to the first EDVID3Field with the matching
 *	type or NULL on error.
 */
EDVID3Field *edv_id3_fields_list_find_by_type(
	GList *fields_list,
	const EDVID3FieldType type
)
{
	GList *glist;
	EDVID3Field *field;

	if(type == EDV_ID3_FIELD_TYPE_ANY)
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = fields_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		field = EDV_ID3_FIELD(glist->data);
		if(field == NULL)
			continue;

		if(field->type == type)
			return(field);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Deletes the EDVID3Fields list.
 *
 *	The fields_list specifies the GList of EDVID3Field * fields
 *	to delete.
 *
 *	This function always returns NULL.
 */
GList *edv_id3_fields_list_delete(GList *fields_list)
{
	if(fields_list == NULL)
		return(NULL);

	g_list_foreach(
		fields_list,
		(GFunc)edv_id3_field_delete,
		NULL
	);
	g_list_free(fields_list);

	return(NULL);
}
