/*
				    ID3
 */

#ifndef EDV_ID3_H
#define EDV_ID3_H

#include <stdio.h>				/* FILE */
#include <glib.h>


typedef struct _EDVID3Genre		EDVID3Genre;
#define EDV_ID3_GENRE(p)		((EDVID3Genre *)(p))
typedef struct _EDVID3Tag		EDVID3Tag;
#define EDV_ID3_TAG(p)			((EDVID3Tag *)(p))
typedef struct _EDVID3Frame		EDVID3Frame;
#define EDV_ID3_FRAME(p)		((EDVID3Frame *)(p))
typedef struct _EDVID3Field		EDVID3Field;
#define EDV_ID3_FIELD(p)		((EDVID3Field *)(p))


/*
 *	Field Types:
 */
typedef enum {
	EDV_ID3_FIELD_TYPE_UNKNOWN,
	EDV_ID3_FIELD_TYPE_INT8,		/* gchar * */
	EDV_ID3_FIELD_TYPE_INT16,		/* gshort * */
	EDV_ID3_FIELD_TYPE_INT32,		/* gint * */
	EDV_ID3_FIELD_TYPE_INT64,		/* glong * */
	EDV_ID3_FIELD_TYPE_STRING,		/* gchar * */
	EDV_ID3_FIELD_TYPE_DATA,		/* guint8 * */
	EDV_ID3_FIELD_TYPE_LANGUAGE,		/* gchar * */
	EDV_ID3_FIELD_TYPE_MIME_TYPE		/* gchar * */
} EDVID3FieldType;
#define EDV_ID3_FIELD_TYPE_ANY		EDV_ID3_FIELD_TYPE_UNKNOWN


/*
 *	Text Encoding:
 */
typedef enum {
	EDV_ID3_TEXT_ENCODING_UNKNOWN,
	EDV_ID3_TEXT_ENCODING_UTF_8,
	EDV_ID3_TEXT_ENCODING_UTF_16,
	EDV_ID3_TEXT_ENCODING_UTF_16_BE,
	EDV_ID3_TEXT_ENCODING_ISO_8859_1
} EDVID3TextEncoding;


/*
 *	Genre Index:
 */
typedef enum {
	EDV_ID3_GENRE_UNKNOWN		= 0
} EDVID3GenreIndex;


/*
 *	Genre:
 */
struct _EDVID3Genre {
	gchar		*name;
	EDVID3GenreIndex	index;
};


/*
 *	ID3v2 Tag Flags:
 *
 *	These values must match the ID3v2 format's values.
 */
typedef enum {
	EDV_ID3_TAG_FLAG_UNSYNCHRONISATION	\
					= 0x80,
	EDV_ID3_TAG_FLAG_EXTENDED_HEADER	\
					= 0x40,
	EDV_ID3_TAG_FLAG_EXPERIMENTAL	= 0x20,
	EDV_ID3_TAG_FLAG_FOOTER_PRESENT	= 0x10,
	EDV_ID3_TAG_FLAG_KNOWN_FLAGS	= 0xf0
} EDVID3TagFlags;

/*
 *	ID3v2 Tag Extended Flags:
 *
 *	These values must match the ID3v2 format's values.
 */
typedef enum {
	EDV_ID3_TAG_EXTENDED_FLAG_IS_UPDATE	\
					= 0x40,
	EDV_ID3_TAG_EXTENDED_FLAG_CRC_DATA_PRESENT	\
					= 0x20,
	EDV_ID3_TAG_EXTENDED_FLAG_RESTRICTIONS	\
					= 0x10,
	EDV_ID3_TAG_EXTENDED_FLAG_KNOWN_FLAGS	\
					= 0x70
} EDVID3TagExtendedFlags;

/*
 *	ID3v2 Tag Restriction Flags:
 *
 *	These values must match the ID3v2 format's values.
 */
typedef enum {
	EDV_ID3_TAG_RESTRICTION_TAG_SIZE_MASK	\
					= 0xc0,
	EDV_ID3_TAG_RESTRICTION_TAG_SIZE_128_FRAMES_1_MB	\
					= 0x00,
	EDV_ID3_TAG_RESTRICTION_TAG_SIZE_64_FRAMES_128_KB	\
					= 0x40,
	EDV_ID3_TAG_RESTRICTION_TAG_SIZE_32_FRAMES_40_KB	\
					= 0x80,
	EDV_ID3_TAG_RESTRICTION_TAG_SIZE_32_FRAMES_4_KB	\
					= 0xc0,

	EDV_ID3_TAG_RESTRICTION_TEXT_ENCODING_MASK	\
					= 0x20,
	EDV_ID3_TAG_RESTRICTION_TEXT_ENCODING_NONE	\
					= 0x00,
	EDV_ID3_TAG_RESTRICTION_TEXT_ENCODING_LATIN1_UTF8	\
					= 0x20,

	EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_MASK	\
					= 0x18,
	EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_NONE	\
					= 0x00,
	EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_1024_CHARS	\
					= 0x08,
	EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_128_CHARS	\
					= 0x10,
	EDV_ID3_TAG_RESTRICTION_TEXT_SIZE_30_CHARS	\
					= 0x18,

	EDV_ID3_TAG_RESTRICTION_IMAGE_ENCODING_MASK	\
					= 0x04,
	EDV_ID3_TAG_RESTRICTION_IMAGE_ENCODING_NONE	\
					= 0x00,
	EDV_ID3_TAG_RESTRICTION_IMAGE_ENCODING_PNG_JPEG	\
					= 0x04,

	EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_MASK	\
					= 0x03,
	EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_NONE	\
					= 0x00,
	EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_256_256	\
					= 0x01,
	EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_64_64	\
					= 0x02,
	EDV_ID3_TAG_RESTRICTION_IMAGE_SIZE_64_64_EXACT	\
					= 0x03
} EDVID3TagRestrictionFlags;


/*
 *	Field:
 */
struct _EDVID3Field {
	EDVID3FieldType	type;			/* Specifies the type of data
						 * stored in member data */
	gpointer	data;			/* Field */
	gint		data_length;		/* Allocated size of the
						 * field (data) in bytes */
};
#define EDV_ID3_FIELD_IS_NUMBER(p)	(((p) != NULL) ?	\
 ((((p)->type == EDV_ID3_FIELD_TYPE_INT8) ||			\
   ((p)->type == EDV_ID3_FIELD_TYPE_INT16) ||			\
   ((p)->type == EDV_ID3_FIELD_TYPE_INT32) ||			\
   ((p)->type == EDV_ID3_FIELD_TYPE_INT64)			\
  ) ? TRUE : FALSE)						\
 : FALSE							\
)
#define EDV_ID3_FIELD_IS_STRING(p)	(((p) != NULL) ?	\
 ((((p)->type == EDV_ID3_FIELD_TYPE_STRING) ||			\
   ((p)->type == EDV_ID3_FIELD_TYPE_LANGUAGE) ||		\
   ((p)->type == EDV_ID3_FIELD_TYPE_MIME_TYPE)			\
  ) ? TRUE : FALSE)						\
 : FALSE							\
)
#define EDV_ID3_FIELD_IS_DATA(p)	(((p) != NULL) ?	\
 (((p)->type == EDV_ID3_FIELD_TYPE_DATA) ? TRUE : FALSE)	\
 : FALSE							\
)
#define EDV_ID3_FIELD_IS_LANGUAGE(p)	(((p) != NULL) ?	\
 (((p)->type == EDV_ID3_FIELD_TYPE_LANGUAGE) ? TRUE : FALSE)	\
 : FALSE							\
)
#define EDV_ID3_FIELD_IS_MIME_TYPE(p)	(((p) != NULL) ?	\
 (((p)->type == EDV_ID3_FIELD_TYPE_MIME_TYPE) ? TRUE : FALSE)	\
 : FALSE							\
)


/*
 *	Frame:
 */
struct _EDVID3Frame {
	gchar		*id;
	EDVID3TextEncoding	text_encoding;
	GList		*fields_list;		/* GList of EDVID3Field *
						 * fields */
	gchar		*description;		/* Verbose description of
						 * this frame */
};
#define EDV_ID3_FRAME_GET_ID(p)		(((p) != NULL) ? (p)->id : NULL)


/*
 *	Tag:
 */
struct _EDVID3Tag {
	gint		version_major,
			version_minor;
	EDVID3TagFlags	flags;
	EDVID3TagExtendedFlags	extended_flags;
	EDVID3TagRestrictionFlags	restriction_flags;
	GList		*frames_list;		/* GList of EDVID3Frame *
						 * frames */
};


/*
 *	Tag IO Options:
 */
typedef enum {
	EDV_ID3_TAG_IO_EXCLUDE_EMPTY_FRAMES	\
					= (1 << 0),	/* Any frames with
							 * no fields will not
							 * be opened or saved */

	EDV_ID3_TAG_IO_COMPRESS_GZIP		\
					= (1 << 8),	/* Use GZip compression
							 * when saving (may
							 * cause problems
                                                         * with some players) */
	EDV_ID3_TAG_IO_APPEND		= (1 << 9),	/* Append the ID3v2
							 * tag instead of
							 * inserting at the
							 * beginning when saving
							 * (may cause problems
							 * with some players) */
	EDV_ID3_TAG_IO_TRUNCATE_EXISTING_TAGS	\
					= (1 << 10)	/* Any existing ID3v*
							 * tags will be removed
							 * from the file when
							 * saving */
} EDVID3TagIOOptions;


/* edv_id3.c */
/* Utilities */
extern guint32 edv_id3_parse_syncsafe4(const guint32 x);
extern gboolean edv_id3_is_id3v2_id_valid(const gchar *id);
extern gchar *edv_id3_id_to_verbose_name(const gchar *id);

/* Length */
extern gulong edv_id3_length_parse(const gchar *s);
extern gchar *edv_id3_length_format(const gulong ms);

/* Checking */
extern gboolean edv_id3_stream_has_id3v1(FILE *fp);
extern gboolean edv_id3_stream_locate_id3v1(
	FILE *fp,
	gulong *position_rtn,
	gulong *length_rtn
);
extern gboolean edv_id3_file_has_id3v1(const gchar *path);
extern gboolean edv_id3_file_locate_id3v1(
	const gchar *path,
	gulong *position_rtn,
	gulong *length_rtn
);

extern gboolean edv_id3_stream_has_id3v2(FILE *fp);
extern gboolean edv_id3_stream_locate_id3v2(
	FILE *fp,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gulong *position_rtn,
	gulong *length_rtn
);
extern gboolean edv_id3_file_has_id3v2(const gchar *path);
extern gboolean edv_id3_file_locate_id3v2(
	const gchar *path,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gulong *position_rtn,
	gulong *length_rtn
);

/* EDVID3GenreIndex */
extern gchar *edv_id3_genre_index_to_name(const EDVID3GenreIndex i);
extern EDVID3GenreIndex edv_id3_genre_name_to_index(const gchar *name);

/* EDVID3Genre */
extern EDVID3Genre *edv_id3_genre_new(void);
extern EDVID3Genre *edv_id3_genre_copy(EDVID3Genre *genre);
extern void edv_id3_genre_delete(EDVID3Genre *genre);

/* EDVID3Genres List */
extern GList *edv_id3_genre_list_new(void);
extern GList *edv_id3_genre_list_delete(GList *genre_list);


/* edv_id3_field.c */
/* EDVID3Field */
extern EDVID3Field *edv_id3_field_new(void);
extern EDVID3Field *edv_id3_field_new_with_values(
	const EDVID3FieldType type,
	gconstpointer data,
	const gint data_len
);
extern EDVID3Field *edv_id3_field_new_with_string_value(const gchar *s);
extern EDVID3Field *edv_id3_field_copy(EDVID3Field *field);
extern void edv_id3_field_delete(EDVID3Field *field);

extern gint edv_id3_field_get_number(EDVID3Field *field);
extern glong edv_id3_field_get_long(EDVID3Field *field);
extern const gchar *edv_id3_field_get_string(EDVID3Field *field);
extern gint edv_id3_field_set_string(
	EDVID3Field *field,
	const gchar *s
);

/* EDVID3Fields List */
extern EDVID3Field *edv_id3_fields_list_find_by_type(
	GList *fields_list,
	const EDVID3FieldType type
);
extern GList *edv_id3_fields_list_delete(GList *fields_list);


/* edv_id3_field_image.c */
/* EDVID3Field image */
extern GList *edv_id3_field_get_image_rgba(
	EDVID3Field *field,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color
);
extern GList *edv_id3_field_get_image_rgba_details(
	EDVID3Field *field,
	gint *width_rtn, gint *height_rtn,
	gint *bpl_rtn,
	gint *nframes_rtn,
	GList **delay_list_rtn,
	guint8 *bg_color,
	gulong *play_time_ms_rtn,
	gchar **creator_rtn,
	gchar **title_rtn,
	gchar **author_rtn,
	gchar **comments_rtn,
	gulong *modified_time_sec_rtn		/* In seconds since EPOCH */
);


/* edv_id3_frame.c */
/* EDVID3Frame */
extern EDVID3Frame *edv_id3_frame_new(void);
extern EDVID3Frame *edv_id3_frame_new_with_values(
	const gchar *id,
	const EDVID3TextEncoding text_encoding,
	GList *fields_list,
	const gchar *description
);
extern EDVID3Frame *edv_id3_frame_new_with_string_value(
	const gchar *id,
	const gchar *s
);
extern EDVID3Frame *edv_id3_frame_copy(EDVID3Frame *frame);
extern void edv_id3_frame_delete(EDVID3Frame *frame);

extern gint edv_id3_frame_set_id(
	EDVID3Frame *frame,
	const gchar *id
);

extern gint edv_id3_frame_get_text_bit_size(EDVID3Frame *frame);

extern EDVID3Field *edv_id3_frame_get_field(
	EDVID3Frame *frame,
	const gint i
);
extern EDVID3Field *edv_id3_frame_get_field_by_type(
	EDVID3Frame *frame,
	const EDVID3FieldType type
);
extern EDVID3Field *edv_id3_frame_insert_field(
	EDVID3Frame *frame,
	const gint i
);
extern EDVID3Field *edv_id3_frame_append_field(EDVID3Frame *frame);

/* EDVID3Frames List */
extern EDVID3Frame *edv_id3_frames_list_find_by_id(
	GList *frames_list,
	const gchar *id
);
extern GList *edv_id3_frames_list_delete(GList *frames_list);


/* edv_id3_tag.c */
/* EDVID3Tag */
extern EDVID3Tag *edv_id3_tag_new(void);
extern EDVID3Tag *edv_id3_tag_copy(EDVID3Tag *tag);
extern void edv_id3_tag_delete(EDVID3Tag *tag);

extern EDVID3Frame *edv_id3_tag_get_frame_by_id(
	EDVID3Tag *tag,
	const gchar *id
);
extern gboolean edv_id3_tag_frame_exists(
	EDVID3Tag *tag,
	EDVID3Frame *frame
);

/* edv_id3_tag_fio.c */
extern EDVID3Tag *edv_id3_tag_open_file(
	const gchar *path,
	const EDVID3TagIOOptions io_options
);
extern gint edv_id3_tag_save_file(
	EDVID3Tag *tag,
	const gchar *path,
	const EDVID3TagIOOptions io_options
);
extern gint edv_id3_tag_strip_file(const gchar *path);


#endif	/* EDV_ID3_H */
