#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_LIBID3TAG
# include <id3tag.h>
#endif
#include <glib.h>

#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"

#include "edv_id3.h"
#include "edv_id3_ids_list.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL	0			/* 0 = off
						 * 1 - 3 = on */
#endif
#if (DEBUG_LEVEL > 0)
# warning "Debugging enabled."
#endif


/*
 *	Verbose/Conical ID3 Frame Names List:
 *
 *	Format: "Conical", "Verbose"
 */
#define EDV_ID3_FRAME_NAMES_LIST	{			\
{ EDV_ID3_FRAME_ID_ALBUM,	"Album" },			\
{ EDV_ID3_FRAME_ID_ARTIST,	"Artist" },			\
{ EDV_ID3_FRAME_ID_AUDIO_ENCRYPTION, "Audio Encryption" },	\
{ EDV_ID3_FRAME_ID_BAND,	"Band" },			\
{ EDV_ID3_FRAME_ID_BPM,		"BPM" },			\
{ EDV_ID3_FRAME_ID_BUFFER_SIZE,	"Buffer Size" },		\
{ EDV_ID3_FRAME_ID_CD_ID,	"CD ID" },			\
{ EDV_ID3_FRAME_ID_COMMENTS,	"Comments" },			\
{ EDV_ID3_FRAME_ID_COMMERCIAL,	"Commercial" },			\
{ EDV_ID3_FRAME_ID_COMPOSER, 	"Composer" },			\
{ EDV_ID3_FRAME_ID_CONDUCTOR,	"Conductor" },			\
{ EDV_ID3_FRAME_ID_CONTENT_GROUP, "Content Group" },		\
{ EDV_ID3_FRAME_ID_COPYRIGHT,	"Copyright" },			\
{ EDV_ID3_FRAME_ID_CODEC,	"Codec" },			\
{ EDV_ID3_FRAME_ID_CODEC_SETTINGS, "Codec Settings" },		\
{ EDV_ID3_FRAME_ID_DATE,	"Date" },			\
{ EDV_ID3_FRAME_ID_USER_TEXT,	"User Text" },			\
{ EDV_ID3_FRAME_ID_ENCRYPTION,	"Encryption" },			\
{ EDV_ID3_FRAME_ID_EQUALIZATION, "Equalization" },		\
{ EDV_ID3_FRAME_ID_EVENT_TIMMING, "Event Timming" },		\
{ EDV_ID3_FRAME_ID_FILE_OWNER	"File Owner" },			\
{ EDV_ID3_FRAME_ID_FILE_TYPE,	"File Type" },			\
{ EDV_ID3_FRAME_ID_GENERAL_OBJECT, "General Object" },		\
{ EDV_ID3_FRAME_ID_GENRE,	"Genre" },			\
{ EDV_ID3_FRAME_ID_GROUP,	"Group" },			\
{ EDV_ID3_FRAME_ID_IMAGE,	"Image" },			\
{ EDV_ID3_FRAME_ID_INITIAL_KEY, "Initial Key" },		\
{ EDV_ID3_FRAME_ID_INTERNET_RADIO_OWNER,"Internet Radio Owner" },\
{ EDV_ID3_FRAME_ID_INTERNET_RADIO_STATION,"Internet Radio Station" },\
{ EDV_ID3_FRAME_ID_INVOLVED_PEOPLE,"Involved People" },		\
{ EDV_ID3_FRAME_ID_ISRC,	"ISRC" },			\
{ EDV_ID3_FRAME_ID_LANGUAGE,	"Language" },			\
{ EDV_ID3_FRAME_ID_LENGTH,	"Length" },			\
{ EDV_ID3_FRAME_ID_LEAD_ARTIST,	"Lead Artist" },		\
{ EDV_ID3_FRAME_ID_LINKED_INFO,	"Linked Info" },		\
{ EDV_ID3_FRAME_ID_LYRICALIST,	"Lyricalist" },			\
{ EDV_ID3_FRAME_ID_MEDIA_TYPE,	"Media Type" },			\
{ EDV_ID3_FRAME_ID_MIX_ARTIST,	"Mix Artist" },			\
{ EDV_ID3_FRAME_ID_MPEG_LOOKUP,	"MPEG Lookup" },		\
{ EDV_ID3_FRAME_ID_OBSOLETE,	"Obsolete" },			\
{ EDV_ID3_FRAME_ID_ORIGINAL_ALBUM,"Original Album" },		\
{ EDV_ID3_FRAME_ID_ORIGINAL_ARTIST,"Original Artist" },		\
{ EDV_ID3_FRAME_ID_ORIGINAL_FILE_NAME,"Original File Name" },	\
{ EDV_ID3_FRAME_ID_ORIGINAL_LYRICALIST,"Original Lyricalist" },	\
{ EDV_ID3_FRAME_ID_ORIGINAL_YEAR,"Original Year" },		\
{ EDV_ID3_FRAME_ID_OWNER,	"Owner" },			\
{ EDV_ID3_FRAME_ID_PART_INSET,	"Part Inset" },			\
{ EDV_ID3_FRAME_ID_PLAY_COUNTER,"Play Counter" },		\
{ EDV_ID3_FRAME_ID_PLAY_LIST_DELAY,"Play List Delay" },		\
{ EDV_ID3_FRAME_ID_POPULARIMETER,"Popularimeter" },		\
{ EDV_ID3_FRAME_ID_POSITION_SYNC,"Position Sync" },		\
{ EDV_ID3_FRAME_ID_PUBLISHER,	"Publisher" },			\
{ EDV_ID3_FRAME_ID_PRIVATE,	"Private" },			\
{ EDV_ID3_FRAME_ID_RECORDING_DATES,"Recording Dates" },		\
{ EDV_ID3_FRAME_ID_REVERB,	"Reverb" },			\
{ EDV_ID3_FRAME_ID_SIZE,	"Size" },			\
{ EDV_ID3_FRAME_ID_SUBTITLE,	"Subtitle" },			\
{ EDV_ID3_FRAME_ID_SYNCED_LYRICS,"Synced Lyrics" },		\
{ EDV_ID3_FRAME_ID_SYNCED_TEMPO,"Synced Tempo" },		\
{ EDV_ID3_FRAME_ID_TERMS_OF_USE,"Terms Of Use" },		\
{ EDV_ID3_FRAME_ID_TIME,	"Time" },			\
{ EDV_ID3_FRAME_ID_TITLE,	"Title" },			\
{ EDV_ID3_FRAME_ID_TRACK,	"Track" },			\
{ EDV_ID3_FRAME_ID_UNIQUE_FILE_ID, "Unique File ID" },		\
{ EDV_ID3_FRAME_ID_UNSYNCED_LYRICS, "Unsynced Lyrics" },	\
{ EDV_ID3_FRAME_ID_VOLUME_ADJUST, "Volume Adjust" },		\
{ EDV_ID3_FRAME_ID_WEBSITE_ARTIST, "Website Artist" },		\
{ EDV_ID3_FRAME_ID_WEBSITE_AUDIO_FILE, "Website Audio File" },	\
{ EDV_ID3_FRAME_ID_WEBSITE_AUDIO_SOURCE, "Website Audio Source" },\
{ EDV_ID3_FRAME_ID_WEBSITE_COMMERCIAL, "Website Commercial" },	\
{ EDV_ID3_FRAME_ID_WEBSITE_COPYRIGHT, "Website Copyright" },	\
{ EDV_ID3_FRAME_ID_WEBSITE_PAYMENT, "Website Payment" },	\
{ EDV_ID3_FRAME_ID_WEBSITE_PUBLISHER, "Website Publisher" },	\
{ EDV_ID3_FRAME_ID_WEBSITE_RADIO_PAGE, "Website Radio Page" },	\
{ EDV_ID3_FRAME_ID_WEBSITE_EXTRA, "Website Extra" },		\
{ EDV_ID3_FRAME_ID_YEAR,	"Year" }			\
}


typedef struct _EDVID3IDData			EDVID3IDData;
#define EDV_ID3_ID_DATA(p)			((EDVID3IDData *)(p))


/* Utilities */
guint32 edv_id3_parse_syncsafe4(const guint32 x);
gboolean edv_id3_is_id3v2_id_valid(const gchar *id);
gchar *edv_id3_id_to_verbose_name(const gchar *id);

/* Length */
gulong edv_id3_length_parse(const gchar *s);
gchar *edv_id3_length_format(const gulong ms);

/* Checking */
gboolean edv_id3_stream_has_id3v1(FILE *fp);
gboolean edv_id3_stream_locate_id3v1(
	FILE *fp,
	gulong *position_rtn,
	gulong *length_rtn
);
gboolean edv_id3_file_has_id3v1(const gchar *path);
gboolean edv_id3_file_locate_id3v1(
	const gchar *path,
	gulong *position_rtn,
	gulong *length_rtn
);

gboolean edv_id3_stream_has_id3v2(FILE *fp);
gboolean edv_id3_stream_locate_id3v2(
	FILE *fp,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gulong *position_rtn,
	gulong *length_rtn
);
gboolean edv_id3_file_has_id3v2(const gchar *path);
gboolean edv_id3_file_locate_id3v2(
	const gchar *path,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gulong *position_rtn,
	gulong *length_rtn
);

/* EDVID3GenreIndex */
gchar *edv_id3_genre_index_to_name(const EDVID3GenreIndex i);
EDVID3GenreIndex edv_id3_genre_name_to_index(const gchar *name);

/* EDVID3Genre */
EDVID3Genre *edv_id3_genre_new(void);
EDVID3Genre *edv_id3_genre_copy(EDVID3Genre *genre);
void edv_id3_genre_delete(EDVID3Genre *genre);

/* EDVID3Genres List */
static gint edv_id3_genre_list_sort_cb(
	gconstpointer a,
	gconstpointer b
);
GList *edv_id3_genre_list_new(void);
GList *edv_id3_genre_list_delete(GList *genre_list);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	ID Data:
 */
struct _EDVID3IDData {
	gchar		*id,
			*name;			/* Verbose name */
};


/*
 *	Parses a 4 byte 7 bits per byte syncsafe integer to a full
 *	4 byte 32 bit integer.
 *
 *	The data specifies the data which must be 4 bytes long.
 */
guint32 edv_id3_parse_syncsafe4(const guint32 x)
{
	guint32 buf32 = x;
	guint8 *ptr8 = (guint8 *)&buf32;
	guint32 v = 0x00000000;

	if(ptr8 == NULL)
		return(v);

	v = /* (v << 7) | */ (ptr8[0] & 0x7F);
	v = (v << 7) | (ptr8[1] & 0x7F);
	v = (v << 7) | (ptr8[2] & 0x7F);
	v = (v << 7) | (ptr8[3] & 0x7F);

	return(v);
}

/*
 *	Checks if the ID is a known and valid ID3v2 ID.
 */
gboolean edv_id3_is_id3v2_id_valid(const gchar *id)
{
	gint i;
	const EDVID3IDData	*id_data,
				id_datas_list[] = EDV_ID3_FRAME_NAMES_LIST;
	const gint n = sizeof(id_datas_list) / sizeof(EDVID3IDData);

	if(id == NULL)
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* Search for the ID in the list */
	for(i = 0; i < n; i++)
	{
		id_data = &id_datas_list[i];
		if(!g_strcasecmp(id_data->id, id))
			return(TRUE);
	}

	errno = EINVAL;

	return(FALSE);
}

/*
 *	Converts the ID3 tag ID to a verbose name.
 *
 *	If the tag ID is not recognized then a copy of it will be
 *	returned as the verbose name instead.
 *	Returns a dynamically allocated string describing the verbose
 *	name or NULL on error.
 */
gchar *edv_id3_id_to_verbose_name(const gchar *id)
{
	gint i;
	const EDVID3IDData	*id_data,
				id_datas_list[] = EDV_ID3_FRAME_NAMES_LIST;
	const gint n = sizeof(id_datas_list) / sizeof(EDVID3IDData);

	if(id == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Search for the ID in the list */
	for(i = 0; i < n; i++)
	{
		id_data = &id_datas_list[i];
		if(!g_strcasecmp(id_data->id, id))
			return(STRDUP(id_data->name));
	}

	/* All else return a string describing just the ID */
	return(g_strdup(id));
}


/*
 *	Parses the length string in the format "[[h:]m:]s[.s]" to
 *	milliseconds (for use with the EDV_ID3_FRAME_ID_LENGTH frame).
 *
 *	Returns the length in milliseconds or 0l on error.
 */
gulong edv_id3_length_parse(const gchar *s)
{
	const gint ntime_values = 3;		/* 3 time values */
	gint i;
	gulong ms;
	gfloat          x,
			v[ntime_values];

	/* Parse/get up to 3 time values from s */
	for(i = 0; i < ntime_values; i++)
	{
		/* Seek past any spaces */
		while((*s == ' ') || (*s == '\t'))
			s++;

		/* Get this time value as a gfloat because there may
		 * be decimal precision described in the string
		 */
		x = (gfloat)atof((const char *)s);
		v[i] = (x >= 0.0f) ? x : 0.0f;

		/* Seek s to the next ':' character or to the end
		 * of the string
		 */
		while(*s != '\0')
		{
			if(*s == ':')
				break;
			s++;
		}
		/* Is there another value to parse after this one? */
		if(*s == ':')
		{
			s++;
			continue;
		}
		/* No additional values? */
		if(*s == '\0')
			break;
	}

	ms = 0l;
	switch(i)
	{
	    case 2:                             /* "h:m:s[.s]" */
		ms += (gulong)(v[0] * 60.0f * 60.0f * 1000.0f);
		ms += (gulong)(v[1] * 60.0f * 1000.0f);
		ms += (gulong)(v[2] * 1000.0f);
		break;
	    case 1:                             /* "m:s[.s]" */
		ms += (gulong)(v[0] * 60.0f * 1000.0f);
		ms += (gulong)(v[1] * 1000.0f);
		break;
	    case 0:                             /* "s[.s]" */
		ms += (gulong)(v[0] * 1000.0f);
		break;
	}

	return(ms);
}

/*
 *	Formats a length string in the format "[[h:]m:]s[.s]" from
 *	milliseconds (for use with the EDV_ID3_FRAME_ID_LENGTH frame).
 *
 *	Returns a dynamically allocated string describing the length
 *	or NULL on error.
 */
gchar *edv_id3_length_format(const gulong ms)
{
	/* 60 minutes or longer? */
	if(ms >= (60l * 60l * 1000l))
	{
		/* "h:m:s" */
		return(g_strdup_printf(
			"%ld:%02ld:%02ld",
			ms / 1000l / 60l / 60l,
			(ms % (60l * 60l * 1000l)) / 1000l / 60l,
			(ms % (60l * 1000l)) / 1000l
		));
	}
	/* 60 seconds or longer? */
	else if(ms >= (60l * 1000l))
	{
		/* "m:s" */
		return(g_strdup_printf(
			"%ld:%02ld",
			ms / 1000l / 60l,
			(ms % (60l * 1000l)) / 1000l
		));
	}
	/* 59 seconds or less */
	else
	{
		/* "s" */
		return(g_strdup_printf(
			"0:%04.1f",
			(gfloat)ms / 1000.0f
		));
	}

	return(g_strdup(""));
}


/*
 *	Checks if the stream has an ID3v1 tag.
 *
 *	The fp specifies the stream to check. The stream's position
 *	will be repositioned to an undefined position by this call
 *	and not restored.
 *
 *	Returns TRUE if an ID3v1 tag was found or FALSE on error.
 */
gboolean edv_id3_stream_has_id3v1(FILE *fp)
{
	return(edv_id3_stream_locate_id3v1(
		fp,
		NULL,
		NULL
	));
}

/*
 *	Checks if the stream has an ID3v1 tag and gets its version,
 *	position and length.
 *
 *	The fp specifies the stream to check. The stream's position
 *	will be repositioned to an undefined position by this call
 *	and not restored.
 *
 *	If position_rtn is not NULL then the position of the ID3v1
 *	tag found in the stream will be returned relative to the
 *	stream's origin (not relative to the position of the stream
 *	when it was passed to this function).
 *
 *	If length_rtn is not NULL then the length of the ID3v1 tag
 *	found in the stream will be returned in units of bytes.
 *
 *	Returns TRUE if an ID3v1 tag was found or FALSE on error.
 */
gboolean edv_id3_stream_locate_id3v1(
	FILE *fp,
	gulong *position_rtn,
	gulong *length_rtn
)
{
	size_t		units_to_read,
			units_read;
	gboolean status;
	guint8 *tag_buf;
	const guint8 *buf_ptr, *buf_end;
	gulong		fp_pos,
			fp_last_pos,
			start_position,
			length;
	const gulong tag_buf_len = 128l;	/* ID3v1 tag length is 128 bytes */
	EDVVFSObject *obj;

	if(position_rtn != NULL)
		*position_rtn = 0l;
	if(length_rtn != NULL)
		*length_rtn = 0l;

	if(fp == NULL)
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* Get the stream's statistics */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj == NULL)
		return(FALSE);

	/* Allocate the tag buffer */
	tag_buf = (guint8 *)g_malloc(tag_buf_len * sizeof(guint8));
	if(tag_buf == NULL)
	{
		edv_vfs_object_delete(obj);
		return(FALSE);
	}

	/* The ID3v1 tag is usually placed at the end of the stream
	 * minus 128 bytes
	 *
	 * Calculate the best seek position to start searching for the
	 * tag,
	 */
	fp_pos = (obj->size >= tag_buf_len) ?
		(obj->size - tag_buf_len) : 0l;
	fp_last_pos = (gulong)ftell(fp);
	if(fp_pos < fp_last_pos)
		fp_pos = fp_last_pos;

	/* Seek to the search position */
	if(fseek(
		fp,
		(long)fp_pos,
		SEEK_SET
	) != 0)
	{
		g_free(tag_buf);
		edv_vfs_object_delete(obj);
		return(FALSE);
	}

	/* Read the block into the tag buffer for searching */
	fp_last_pos = fp_pos;
	units_to_read = (size_t)tag_buf_len / sizeof(guint8);
	units_read = fread(
		tag_buf,
		sizeof(guint8),
		units_to_read,
		fp
	);
	fp_pos += (gulong)(units_read * sizeof(guint8));
	if((units_read <= 0l) || ferror(fp))
	{
		g_free(tag_buf);
		edv_vfs_object_delete(obj);
		return(FALSE);
	}

	/* Search
	 *
	 * Tag format:
	 *
	 * Position	Length	Description
	 * 0		3	3 bytes with ASCII value "TAG"
	 * 3		30	Title string
	 * 33		30	Artist string
	 * 63		30	Album string
	 * 93		4	Year string
	 * 97		30	Comment
	 * 127		1	Genre
	 */
	status = FALSE;
	start_position = 0l;
	length = 0l;
	for(buf_ptr = tag_buf,
	    buf_end = buf_ptr + (units_read * sizeof(guint8));
	    buf_ptr < buf_end;
	    buf_ptr++
	)
	{
		if(*buf_ptr == 'T')
		{
			/* Encountered the 'T' byte, this may
			 * potentially be the start of a tag
			 *
			 * Record the starting position and calculate
			 * the length
			 */
			start_position = fp_last_pos + (gulong)(buf_ptr -
				tag_buf);
			length = (gulong)(buf_end - buf_ptr);

			/* Must be at least 128 bytes long */
			if(length < 128l)
				continue;

			if(buf_ptr[1] != 'A')
				continue;

			if(buf_ptr[2] != 'G')
				continue;

			status = TRUE;
		}
	}

	/* Delete the tag buffer */
	g_free(tag_buf);

	/* Delete the stream's statistics */
	edv_vfs_object_delete(obj);

	/* Was a tag found? */
	if(status)
	{
		/* Set the return values */
		if(position_rtn != NULL)
			*position_rtn = start_position;
		if(length_rtn != NULL)
			*length_rtn = length;
	}
	else
	{
		errno = ENOENT;
	}

	return(status);
}

/*
 *	Checks if the file has an ID3v1 tag.
 *
 *	The path specifies the file to check.
 *
 *	Returns TRUE if an ID3v1 tag was found or FALSE on error.
 */
gboolean edv_id3_file_has_id3v1(const gchar *path)
{
	return(edv_id3_file_locate_id3v1(
		path,
		NULL,
		NULL
	));
}

/*
 *	Checks if the file has an ID3v1 tag and gets its version,
 *	position and length.
 *
 *	The path specifies the file to check.
 *
 *	If position_rtn is not NULL then the position of the ID3v1
 *	tag found in the file will be returned.
 *
 *	If length_rtn is not NULL then the length of the ID3v1 tag
 *	found in the file will be returned in units of bytes.
 *
 *	Returns TRUE if an ID3v1 tag was found or FALSE on error.
 */
gboolean edv_id3_file_locate_id3v1(
	const gchar *path,
	gulong *position_rtn,
	gulong *length_rtn
)
{
	FILE *fp;
	gboolean status;

	if(position_rtn != NULL)
		*position_rtn = 0l;
	if(length_rtn != NULL)
		*length_rtn = 0l;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* Open the file for reading */
	fp = fopen(
		(const char *)path,
		"rb"
	);
	if(fp == NULL)
		return(FALSE);

	/* Search the stream for the ID3v1 tag */
	status = edv_id3_stream_locate_id3v1(
		fp,
		position_rtn,
		length_rtn
	);

	/* Close the file */
	(void)fclose(fp);

	return(status);
}


/*
 *	Checks if the stream has an ID3v2 tag.
 *
 *	The fp specifies the stream to check. The stream's position
 *	will be repositioned to an undefined position by this call
 *	and not restored.
 *
 *	Returns TRUE if an ID3v2 tag was found or FALSE on error.
 */
gboolean edv_id3_stream_has_id3v2(FILE *fp)
{
	return(edv_id3_stream_locate_id3v2(
		fp,
		NULL,
		NULL,
		NULL,
		NULL
	));
}

/*
 *	Checks if the stream has an ID3v2 tag and gets its version,
 *	position and length.
 *
 *	The fp specifies the stream to check. The stream's position
 *	will be repositioned to an undefined position by this call
 *	and not restored.
 *
 *	If version_major_rtn is not NULL then the major version
 *	number will be returned.
 *
 *	If version_minor_rtn is not NULL then the minor version
 *	number will be returned.
 *
 *	If position_rtn is not NULL then the position of the ID3v2
 *	tag found in the stream will be returned relative to the
 *	stream's origin (not relative to the position of the stream
 *	when it was passed to this function).
 *
 *	If length_rtn is not NULL then the length of the ID3v2 tag
 *	found in the stream will be returned in units of bytes.
 *
 *	Returns TRUE if an ID3v2 tag was found or FALSE on error.
 */
gboolean edv_id3_stream_locate_id3v2(
	FILE *fp,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gulong *position_rtn,
	gulong *length_rtn
)
{
	size_t		units_to_read,
			units_read;
	gboolean status;
	guint8		*io_buf,
			*header_buf,
			version_major,
			version_minor;
	const guint8	*buf_ptr, *buf_end;
	EDVID3TagFlags tag_flags;
	guint32		ui32;
	gulong		io_buf_len,
			fp_pos, fp_last_pos,
			start_position,
			length;
	const gulong header_buf_len = 10l;	/* ID3v2 header is 10 bytes */
	EDVVFSObject *obj;

	if(version_major_rtn != NULL)
		*version_major_rtn = 0;
	if(version_minor_rtn != NULL)
		*version_minor_rtn = 0;
	if(position_rtn != NULL)
		*position_rtn = 0l;
	if(length_rtn != NULL)
		*length_rtn = 0l;

	if(fp == NULL)
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* Get the stream's statistics */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		io_buf_len = obj->block_size;
		edv_vfs_object_delete(obj);
	}
	if(io_buf_len == 0l)
		io_buf_len = 1l;

	/* Allocate the read buffer */
	io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
	if(io_buf == NULL)
		return(FALSE);

	/* Allocate the header buffer */
	header_buf = (guint8 *)g_malloc(header_buf_len * sizeof(guint8));
	if(header_buf == NULL)
	{
		g_free(io_buf);
		return(FALSE);
	}

	status = FALSE;
	start_position = fp_pos = (gulong)ftell(fp);
	version_major = 0x00;
	version_minor = 0x00;
	tag_flags = 0;
	length = 0l;

	/* Look for the ID3v2 tag with the following pattern:
	 *
	 * 0x49 0x44 0x33 0xyy 0xyy 0xxx 0xzz 0xzz 0xzz 0xzz
	 *
	 * Where 0xyy is less than 0xFF, 0xxx is the 'flags' byte
	 * and zz is less than 0x80
	 */
	while(!feof(fp) && !status)
	{
		/* Read the next block and update the current stream
		 * position
		 */
		fp_last_pos = fp_pos;
		units_to_read = (size_t)io_buf_len / sizeof(guint8);
		units_read = fread(
			io_buf,
			sizeof(guint8),
			units_to_read,
			fp
		);
		fp_pos += (gulong)(units_read * sizeof(guint8));
		if((units_read <= 0l) || ferror(fp))
			break;

		/* Search this block for an ID3v2 tag */
		for(buf_ptr = io_buf,
		    buf_end = buf_ptr + (units_read * sizeof(guint8));
		    buf_ptr < buf_end;
		    buf_ptr++
		)
		{
			/* Starting 'I' byte? */
			if(*buf_ptr == 0x49)
			{
				/* Found an 'I' byte, which means that
				 * this may potentially be the start
				 * of an ID3v2 tag
				 */
				gulong io_buf_len_remaining;

				/* Record this position as potentially
				 * being the start of the tag
				 */
				start_position = fp_last_pos +
					(gulong)(buf_ptr - io_buf);

				/* Check if we have enough data in the
				 * read buffer from the current
				 * position to fill the header buffer
				 * for checking/parsing
				 */
				io_buf_len_remaining = (gulong)(
					buf_end - buf_ptr
				);
				if(io_buf_len_remaining >= header_buf_len)
				{
					/* Copy the data from the
					 * read buffer to the header
					 * buffer
					 */
					(void)memcpy(
						header_buf,
						buf_ptr,
						(size_t)header_buf_len *
							sizeof(guint8)
					);
				}
				else
				{
					/* Not enough data in the
					 * read buffer to fill the
					 * header buffer
					 */
					size_t	units_to_read,
						units_read;

					/* Copy the remaining data in
					 * the read buffer to the
					 * header buffer
					 */
					(void)memcpy(
						header_buf,
						buf_ptr,
						(size_t)io_buf_len_remaining *
							sizeof(guint8)
					);

					/* Read the additional data
					 * needed from the stream to
					 * the header buffer
					 */
					units_to_read = (size_t)(
						header_buf_len -
							io_buf_len_remaining
					) / sizeof(guint8);
					units_read = fread(
						header_buf +
							io_buf_len_remaining,
						sizeof(guint8),
						units_to_read,
						fp
					);
					if((units_read < units_to_read) ||
					   ferror(fp)
					)
					{
						fp_pos += (gulong)(
							units_read *
								sizeof(guint8)
						);
						break;
					}

					/* Restore the stream position */
					if(fseek(
						fp,
						(long)fp_pos,
						SEEK_SET
					) != 0)
						break;
				}

				/* Look for the 'D' and '3' bytes */
				if(header_buf[1] != 0x44)
					continue;
				if(header_buf[2] != 0x33)
					continue;

				/* Version major and minor */
				version_major = header_buf[3];
				if(version_major == 0xFF)
					continue;
				version_minor = header_buf[4];
				if(version_minor == 0xFF)
					continue;

				/* Flags */
				tag_flags = (EDVID3TagFlags)header_buf[5];

				/* Frames length (4 bytes, 7 bits per
				 * byte, syncsafe)
				 */
				ui32 = *(guint32 *)(header_buf + 6);
				if((((ui32 & 0xFF000000) >> 24) >= 0x80) ||
				   (((ui32 & 0x00FF0000) >> 16) >= 0x80) ||
				   (((ui32 & 0x0000FF00) >> 8) >= 0x80) ||
				   (((ui32 & 0x000000FF) >> 0) >= 0x80)
				)
					continue;
				length = (gulong)edv_id3_parse_syncsafe4(ui32);
				/* Add the length of the header (and
				 * footer) to get the total length
				 * of the tag
				 */
				if(tag_flags & EDV_ID3_TAG_FLAG_FOOTER_PRESENT)
					length += (10l + 10l);
				else
					length += 10l;

				/* Mark that we found an ID3v2 tag */
				status = TRUE;

				/* Stop searching */
				break;
			}
		}
	}

	/* Delete the header buffer */
	g_free(header_buf);

	/* Delete the read buffer */
	g_free(io_buf);

	/* Was a tag found? */
	if(status)
	{
		/* Set the return values */
		if(version_major_rtn != NULL)
			*version_major_rtn = (gint)version_major;
		if(version_minor_rtn != NULL)
			*version_minor_rtn = (gint)version_minor;
		if(position_rtn != NULL)
			*position_rtn = start_position;
		if(length_rtn != NULL)
			*length_rtn = length;
	}
	else
	{
		errno = ENOENT;
	}

	return(status);
}

/*
 *	Checks if the file has an ID3v2 tag.
 *
 *	The path specifies the file to check.
 *
 *	Returns TRUE if an ID3v2 tag was found or FALSE on error.
 */
gboolean edv_id3_file_has_id3v2(const gchar *path)
{
	return(edv_id3_file_locate_id3v2(
		path,
		NULL,
		NULL,
		NULL,
		NULL
	));
}

/*
 *	Checks if the file has an ID3v2 tag and gets its version,
 *	position and length.
 *
 *	The path specifies the file to check.
 *
 *	If version_major_rtn is not NULL then the major version
 *	number will be returned.
 *
 *	If version_minor_rtn is not NULL then the minor version
 *	number will be returned.
 *
 *	If position_rtn is not NULL then the position of the ID3v2
 *	tag found in the file will be returned.
 *
 *	If length_rtn is not NULL then the length of the ID3v2 tag
 *	found in the file will be returned in units of bytes.
 *
 *	Returns TRUE if an ID3v2 tag was found or FALSE on error.
 */
gboolean edv_id3_file_locate_id3v2(
	const gchar *path,
	gint *version_major_rtn,
	gint *version_minor_rtn,
	gulong *position_rtn,
	gulong *length_rtn
)
{
	FILE *fp;
	gboolean status;

	if(version_major_rtn != NULL)
		*version_major_rtn = 0;
	if(version_minor_rtn != NULL)
		*version_minor_rtn = 0;
	if(position_rtn != NULL)
		*position_rtn = 0l;
	if(length_rtn != NULL)
		*length_rtn = 0l;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* Open the file for reading */
	fp = fopen(
		(const char *)path,
		"rb"
	);
	if(fp == NULL)
		return(FALSE);

	/* Search the stream for the ID3v2 tag */
	status = edv_id3_stream_locate_id3v2(
		fp,
		version_major_rtn,
		version_minor_rtn,
		position_rtn,
		length_rtn
	);

	/* Close the file */
	(void)fclose(fp);

	return(status);
}


/*
 *	Converts the EDVID3GenreIndex to a genre name string.
 *
 *	Returns a dynamically allocated string describing the
 *	genre name or NULL on error.
 */
gchar *edv_id3_genre_index_to_name(const EDVID3GenreIndex i)
{
#if defined(HAVE_LIBID3TAG)
	const id3_ucs4_t *ucs4;

	if(i < 0)
	{
		errno = EINVAL;
		return(NULL);
	}

	ucs4 = id3_genre_index((unsigned int)i);
	if(ucs4 == NULL)
		return(NULL);

	return((gchar *)id3_ucs4_utf8duplicate(ucs4));
#else
	return(NULL);
#endif
}

/*
 *	Converts the verbose genre name to a EDVID3GenreIndex.
 */
EDVID3GenreIndex edv_id3_genre_name_to_index(const gchar *name)
{
#if defined(HAVE_LIBID3TAG)
	id3_ucs4_t *ucs4;
	EDVID3GenreIndex genre_index;

	if(STRISEMPTY(name))
	{
		errno = EINVAL;
		return(EDV_ID3_GENRE_UNKNOWN);
	}

	ucs4 = id3_utf8_ucs4duplicate((id3_utf8_t const *)name);
	if(ucs4 == NULL)
		return(-3);

	genre_index = (EDVID3GenreIndex)id3_genre_number(ucs4);

	g_free(ucs4);

	return(genre_index);
#else
	if(STRISEMPTY(name))
	{
		errno = EINVAL;
		return(EDV_ID3_GENRE_UNKNOWN);
	}

	return(EDV_ID3_GENRE_UNKNOWN);
#endif
}


/*
 *	Creates a new EDVID3Frame.
 *
 *	Returns a new dynamically allocated EDVID3Frame with all its
 *	values zero'ed or NULL on error.
 */
EDVID3Genre *edv_id3_genre_new(void)
{
	return(EDV_ID3_GENRE(g_malloc0(sizeof(EDVID3Genre))));
}

/*
 *	Coppies the EDVID3Genre.
 *
 *	The genre specifies the EDVID3Genre to copy.
 *
 *	Returns a new dynamically allocated copy of the EDVID3Genre or
 *	NULL on error.
 */
EDVID3Genre *edv_id3_genre_copy(EDVID3Genre *genre)
{
	EDVID3Genre	*src_genre = genre,
			*tar_genre;

	if(src_genre == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	tar_genre = edv_id3_genre_new();
	if(tar_genre == NULL)
		return(NULL);

	tar_genre->name = STRDUP(src_genre->name);
	tar_genre->index = src_genre->index;

	return(tar_genre);
}

/*
 *	Deletes the EDVID3Genre.
 */
void edv_id3_genre_delete(EDVID3Genre *genre)
{
	if(genre == NULL)
		return;

	g_free(genre->name);
	g_free(genre);
}


/*
 *	Genre list sort callback.
 */
static gint edv_id3_genre_list_sort_cb(
	gconstpointer a,
	gconstpointer b
)
{
	EDVID3Genre	*genre_a = EDV_ID3_GENRE(a),
			*genre_b = EDV_ID3_GENRE(b);
	return((gint)strcmp((const char *)genre_a->name, (const char *)genre_b->name));
}

/*
 *	Creates a new genre list.
 *
 *	Returns a new dynamically allocated GList of EDVID3Genre * genres
 *	or NULL on error.
 */
GList *edv_id3_genre_list_new(void)
{
	gchar *name;
	GList *genre_list = NULL;
	EDVID3GenreIndex genre_index;
	EDVID3Genre *genre;

	for(genre_index = 0,
	    name = edv_id3_genre_index_to_name(genre_index);
	    name != NULL;
	    genre_index++,
	    g_free(name),
	    name = edv_id3_genre_index_to_name(genre_index)
	)
	{
		genre = edv_id3_genre_new();
		if(genre == NULL)
			continue;

		genre->name = g_strdup(name);
		genre->index = genre_index;

		genre_list = g_list_append(
			genre_list,
			genre
		);
	}

	genre_list = g_list_sort(
		genre_list,
		edv_id3_genre_list_sort_cb
	);

	return(genre_list);
}

/*
 *	Deletes the genre list.
 *
 *	This function always returns NULL.
 */
GList *edv_id3_genre_list_delete(GList *genre_list)
{
	if(genre_list == NULL)
		return(NULL);

	g_list_foreach(
		genre_list,
		(GFunc)edv_id3_genre_delete,
		NULL
	);
	g_list_free(genre_list);

	return(NULL);
}


/*
	"Three Centers Of One World"

	There are three adult MALEs who follow a religious belief
	known as Catholisim, they call themselves Catholics. In
	this belief, there are certain leaders of groups of
	other Catholics, these leaders are called Fathers.

	So there are these three Fathers, they have not mated with
	a FEMALE, popular amoung their belief is that mating without
	a contract, which they call a Marriage, is considered bad,
	which they called a Sin. Decided by only reasons know to
	them have also not entered into any such contract.

	Also popular amoung Catholics in general, is the unspoken
	agreement that a heightened priority of their efforts be put
	into the caring of maturating and developing persons, called
	Fetuses, that they mature to the point past maturation,
	called Birth and subsequently become Children. These Fetuses
	need not be related to them or even known to them in
	specific for Catholics to invest effort of their concearns
	towards.

	But its not been demostrated that Catholics express any
	such concearn towards Fetuses past maturation, called
	Children. Though it has been claimed by Catholics and
	commanded by Fathers that an equal amount of care should be
	expended on Children.

	The same amount of implicit care is observed for MALE
	Catholics to avoid having relationships with other MALE
	Catholics or MALE NON-Catholics. Most Catholics passively
	but diligently monitor MALE Catholics to identity any such
	relationship by means of detecting certain behavioral
	traits such as behaviors common to FEMALE Catholics as an
	identifier that the MALE Catholic is having a relationship
	with other MALE Catholics or MALE non-Catholics.

	The caring of children is a behavior often associated with
	FEMALE adults, both Catholic and NON-Catholic.

	The three fathers expend a large amount of their time and
	effort on the caring of children and not; mating, entering
	into contracts, or FEMALEs. You will also find that this
	last sentence stating that the above was written with the
	intention for artistic observation and creation with no
	intention to be a threat or attack to any group is often
	excluded from citations.
 */
