#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_history.h"
#include "edv_history_list.h"

#include "config.h"


/* Open History List File */
GList *edv_history_list_file_open(
	GList *history_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
);

/* Save History List File */
void edv_history_list_stream_append(
	FILE *fp,
	const EDVHistory *h
);
void edv_history_list_file_append(
	const gchar *path,
	const EDVHistory *h
);
void edv_history_list_file_save(
	GList *history_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Opens the History List from the History List File.
 *
 *	The history_list specifies the History List, a GList of
 *	EDVHistory * history events.
 *
 *	The path specifies the History List file.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns a GList of EDVHistory * events or NULL on error.
 */
GList *edv_history_list_file_open(
	GList *history_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
)
{
	FILE *fp;
	gboolean aborted = FALSE;
	gchar *parm;
	gulong file_size;
	EDVVFSObject *obj;
	EDVHistory *h = NULL;

	if(STRISEMPTY(path))
		return(history_list);

	/* Open the History List file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(history_list);

	/* Get the statistics */
	obj = edv_vfs_object_fstat((gint)fileno(fp));
	if(obj != NULL)
	{
		file_size = obj->size;
		edv_vfs_object_delete(obj);
	}
	else
	{
		file_size = 0l;
	}

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
		if(progress_cb(
			progress_data,
			0l, file_size
		))
			aborted = TRUE;
	}

	/* Begin reading the History List file */
	parm = NULL;
	while(!aborted)
	{
		/* Get the next parameter from the History List file */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Report the progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				progress_data,
				(gulong)ftell(fp), file_size
			))
			{
				aborted = TRUE;
				break;
			}
		}

		/* Begin handling the parameter */

		/* Start of history item? */
		if(!g_strcasecmp(parm, "Begin"))
		{
			gint value[1];
			FGetValuesI(fp, value, 1);

			h = edv_history_new();
			if(h == NULL)
				break;

			history_list = g_list_append(
				history_list,
				h
			);

			h->type = value[0];
		}
		/* TimeStart */
		else if(!g_strcasecmp(parm, "TimeStart"))
		{
			glong value[1];
			FGetValuesL(fp, value, 1);
			if(h != NULL)
			{
				h->start_time = (gulong)value[0];
			}
		}
		/* TimeEnd */
		else if(!g_strcasecmp(parm, "TimeEnd"))
		{
			glong value[1];
			FGetValuesL(fp, value, 1);
			if(h != NULL)
			{
				h->end_time = (gulong)value[0];
			}
		}
		/* Status */
		else if(!g_strcasecmp(parm, "Status"))
		{
			gint value[1];
			FGetValuesI(fp, value, 1);
			if(h != NULL)
			{
				h->status = value[0];
			}
		}
		/* Source */
		else if(!g_strcasecmp(parm, "Source"))
		{
			gchar *value = FGetString(fp);
			if(h != NULL)
			{
				g_free(h->source);
				h->source = STRDUP(value);
			}
			g_free(value);
		}
		/* Target */
		else if(!g_strcasecmp(parm, "Target"))
		{
			gchar *value = FGetString(fp);
			if(h != NULL)
			{
				g_free(h->target);
				h->target = STRDUP(value);
			}
			g_free(value);
		}
		/* Comments */
		else if(!g_strcasecmp(parm, "Comments"))
		{
			gchar *value = FGetString(fp);
			if(h != NULL)
			{
				g_free(h->comments);
				h->comments = STRDUP(value);
			}
			g_free(value);
		}
		/* End */
		else if(!g_strcasecmp(parm, "End"))
		{
			FSeekNextLine(fp);

			/* Reset contexts */
			h = NULL;
		}
		else
		{
			FSeekNextLine(fp);
		}
	}

	/* Delete the parameter */
	g_free(parm);

	/* Close the History List file */
	fclose(fp);

	/* Report the final progress */
	if((progress_cb != NULL) && !aborted)
	{
		if(progress_cb(
			progress_data,
			file_size, file_size
		))
			aborted = TRUE;
	}

	return(history_list);
}


/*
 *	Appends the history item to the stream.
 *
 *	The fp specifies the stream.
 *
 *	The h specifies the history item.
 */
void edv_history_list_stream_append(
	FILE *fp, const EDVHistory *h
)
{
	if((fp == NULL) || (h == NULL))
		return;

	(void)fprintf(
		fp,
		"Begin = %i\n",
		h->type
	);
	if(h->start_time > 0l)
		(void)fprintf(
			fp,
			"\tTimeStart = %ld\n",
			h->start_time
		);
	if(h->end_time > 0l)
		(void)fprintf(
			fp,
			"\tTimeEnd = %ld\n",
			h->end_time
		);
	(void)fprintf(
		fp,
		"\tStatus = %i\n",
		h->status
	);
	if(!STRISEMPTY(h->source))
	{
		gchar *s = edv_strsub(
			h->source,
			"\n",
			"\\n"
		);
		(void)fprintf(
			fp,
			"\tSource = %s\n",
			s
		);
		g_free(s);
	}
	if(!STRISEMPTY(h->target))
	{
		gchar *s = edv_strsub(
			h->target,
			"\n",
			"\\n"
		);
		(void)fprintf(
			fp,
			"\tTarget = %s\n",
			s
		);
		g_free(s);
	}
	if(!STRISEMPTY(h->comments))
	{
		gchar *s = edv_strsub(
			h->comments,
			"\n",
			"\\n"
		);
		(void)fprintf(
			fp,
			"\tComments = %s\n",
			s
		);
		g_free(s);
	}
	(void)fprintf(fp, "End\n");
}

/*
 *	Appends the history item to the history file.
 *
 *	The path specifies the history file.
 *
 *	The h specifies the history item.
 */
void edv_history_list_file_append(
	const gchar *path, const EDVHistory *h
)
{
	FILE *fp = !STRISEMPTY(path) ?
		fopen((const char *)path, "ab") : NULL;
	if(fp == NULL)
		return;

	edv_history_list_stream_append(fp, h);

	fclose(fp);
}

/*
 *	Saves the History List to the History List file.
 *
 *	The history_list specifies the History List, a GList of
 *	EDVHistory * history events.
 *
 *	The path specifies the History List file.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 */
void edv_history_list_file_save(
	GList *history_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
)
{
	FILE *fp;
	gboolean aborted = FALSE;
	gint i, nhistory_events;
	gchar *parent_path;
	GList *glist;
	EDVHistory *h;

	if((history_list == NULL) || STRISEMPTY(path))
		return;

	/* Get parent directory and create it as needed */
	parent_path = g_dirname(path);
	if(parent_path != NULL)
	{
		edv_directory_create(
			parent_path,
			TRUE,				/* Create parents */
			NULL				/* No new paths list return */
		);
		g_free(parent_path);
	}

	/* Open the history file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
		return;

	nhistory_events = g_list_length(history_list);

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
		if(progress_cb(
			progress_data,
			0l, (gulong)nhistory_events
		))
			aborted = TRUE;
	}

	/* Iterate through history items */
	for(glist = history_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		if(aborted)
			break;

		h = EDV_HISTORY(glist->data);
		if(h == NULL)
			continue;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				progress_data,
				(gulong)(i + 1),
				(gulong)nhistory_events
			))
			{
				aborted = TRUE;
				break;
			}
		}

		edv_history_list_stream_append(fp, h);
	}

	/* Close the History List file */
	fclose(fp);

	/* Report the final progress */
	if((progress_cb != NULL) && !aborted)
		progress_cb(
			progress_data,
			(gulong)nhistory_events, (gulong)nhistory_events
		);
}
