#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_utils_gtk.h"
#include "edv_help.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"



static gboolean edv_help_is_file_executable(const gchar *path);

void edv_help(
	EDVCore *core,
	const gchar *topic,
	GtkWidget *toplevel
);


/*
 *	Help Program Argument Types:
 *
 *	For the EDV_HELP_PROGRAM_LOCATIONS list.
 */
typedef enum {
	EDV_HELP_ARGUMENT_PATH,
	EDV_HELP_ARGUMENT_URL
} EDVHelpArgumentType;

/*
 *	Help Programs List:
 *
 *	The list item format:
 *
 *	<program_path>,
 *	<arguments>,
 *	<edv_help_prog_input_type>,
 *	<reserved>
 *
 *	The last set of four pointers in the list should be all NULL's.
 */
#if defined(DEBIAN)
#define EDV_HELP_PROGRAM_LOCATIONS	{	\
	/* Debian */				\
{	"/usr/bin/x-www-browser",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* GNOME Help Browser */		\
{	"/usr/bin/gnome-help-browser",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/local/bin/gnome-help-browser",	\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/bin/gnome-help-browser",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* Netscape */				\
{	"/usr/bin/netscape",			\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/local/bin/netscape",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/bin/netscape",			\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* Mozilla */				\
{	"/usr/bin/mozilla",			\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/local/bin/mozilla",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/bin/mozilla",				\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* Lynx */				\
{	"/usr/X11R6/bin/nxterm",		\
	"-e lynx",				\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/X11R6/bin/xterm",			\
	"-e lynx",				\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* End */				\
{	NULL, NULL, 0, 0			\
}						\
}
#else
#define EDV_HELP_PROGRAM_LOCATIONS	{	\
	/* GNOME Help Browser */		\
{	"/usr/bin/gnome-help-browser",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/local/bin/gnome-help-browser",	\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/bin/gnome-help-browser",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* Netscape */				\
{	"/usr/bin/netscape",			\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/local/bin/netscape",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/bin/netscape",			\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* Mozilla */				\
{	"/usr/bin/mozilla",			\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/local/bin/mozilla",		\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/bin/mozilla",				\
	"",					\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* Lynx */				\
{	"/usr/X11R6/bin/nxterm",		\
	"-e lynx",				\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
{	"/usr/X11R6/bin/xterm",			\
	"-e lynx",				\
	EDV_HELP_ARGUMENT_URL,			\
	0					\
},						\
						\
	/* End */				\
{	NULL, NULL, 0, 0			\
}						\
}
#endif

typedef struct {
	const gchar	*prog;
	const gchar	*arg;
	EDVHelpArgumentType	arg_type;
	gint		reserved;
} EDVHelpProgram;


/*
 *	Help Topics & Help File Locations List:
 */
#define EDV_HELP_FILE_LOCATIONS {				\
{ "Contents",		"help/index.html", 		0 },	\
{ "File Browser",	"help/file_vfs_browser.html",	0 },	\
{ "Image Browser",	"help/image_vfs_browser.html", 	0 },	\
{ "Archiver",		"help/archiver.html",		0 },	\
{ "Recycle Bin",	"help/recycle_bin.html",	0 },	\
{ "Keys List",		"help/keys_list.html",		0 },	\
{ "Disk Objects",	"help/disk_objects.html",	0 },	\
{ "MIME Types",		"help/mime_types.html",		0 },	\
{ "Devices",		"help/devices.html",		0 },	\
{ "Common Operations",	"help/common_operations.html",	0 },	\
{ "Front Ends",		"help/front_ends.html",		0 },	\
{ "Utilities",		"help/utilities.html",		0 },	\
{ "FAQ",		"help/faq.html",		0 },	\
{ "Contacts",		"help/contacts.html",		0 },	\
{ NULL,			NULL,				0 }	\
}
typedef struct {
	const gchar	*topic;
	const gchar	*file;
	gint		reserved;
} EDVHelpFile;


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Checks if the object is set executable.
 */
static gboolean edv_help_is_file_executable(const gchar *path)
{
        EDVPermissionFlags permissions;
        EDVVFSObject *obj = edv_vfs_object_stat(path);
        if(obj == NULL)
            return(FALSE);

        /* Not a file? */
        if(!EDV_VFS_OBJECT_IS_FILE(obj))
        {
            edv_vfs_object_delete(obj);
            return(FALSE);
        }

        /* Is the file executable? */
        permissions = obj->permissions;
        if((permissions & EDV_PERMISSION_UX) ||
           (permissions & EDV_PERMISSION_GX) ||
           (permissions & EDV_PERMISSION_OX)
        )
        {
            edv_vfs_object_delete(obj);
            return(TRUE);
        }

        edv_vfs_object_delete(obj);

        return(FALSE);
}


/*
 *	Help calling nexus.
 *
 *	If topic is NULL then the topic is automatically ste to
 *	"Contents".
 *
 *	Valid values for topic are defined in EDV_EDV_HELP_FILE_LOCATIONS.
 */
void edv_help(
	EDVCore *core,
	const gchar *topic,
	GtkWidget *toplevel
)
{
	gint i;
	const gchar	*s,
			*data_dir;
	gchar *help_file;
	CfgList *cfg_list;
	EDVHelpProgram	*hp_ptr,
			help_prog_list[] = EDV_HELP_PROGRAM_LOCATIONS,
			help_prog;
	EDVHelpFile	*hf_ptr,
			help_file_list[] = EDV_HELP_FILE_LOCATIONS;

	if(core == NULL)
	    return;

	cfg_list = core->cfg_list;

	/* If no topic is specified then set default topic */
	if(STRISEMPTY(topic))
	    topic = help_file_list[0].topic;
	if(STRISEMPTY(topic))
	    return;

	/* Get the global data directory */
	data_dir = EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL);
	if(STRISEMPTY(data_dir))
	{
	    gchar *msg = g_strdup_printf(
"The global data directory was not set.\n\
\n\
Please check the configuration file:\n\
\n\
    %s\n\
\n\
And make sure that the parameter:\n\
\n\
    %s\n\
\n\
is defined.",
		core->cfg_path,
		EDV_CFG_PARM_DIR_GLOBAL
	    );
	    edv_play_sound_warning(core);
	    edv_message_warning(
		"Help Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    return;
	}

	/* Get the help file based on the specified topic */
	for(i = 0, hf_ptr = NULL;
	    help_file_list[i].topic != NULL;
	    i++
	)
	{
	    if(!g_strcasecmp(help_file_list[i].topic, topic))
	    {
		hf_ptr = &help_file_list[i];
		break;
	    }
	}
	s = (hf_ptr != NULL) ? hf_ptr->file : NULL;
	if(s != NULL)
	    help_file = edv_paths_join(
		data_dir,
		s
	    );
	else
	    help_file = NULL;

	/* No help file matched for the specified topic? */
	if(help_file == NULL)
	{
	    gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de encontrar el tema de ayuda \"%s\"."
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable de trouver \"%s\" de sujet d'aide."
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, hilfe thema \"%s\" zu finden."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per trovare \"%s\" di soggetto di aiuto."
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam hulp onderwerp \"%s\" te vinden."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de achar \"%s\" de tema de ajuda."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls finne hjelpeemne \"%s\"."
#else
"Unable to find help topic \"%s\"."
#endif
		,
		topic
	    );
	    edv_play_sound_warning(core);
	    edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Ninguna Ayuda Disponible",
msg,
"El tema especificado de la ayuda no se encontr, para una lista de\n\
temas de ayuda por favor mirada en el contenido yendo a\n\
Ayudar->Contenido.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Aucune Aide Disponible",
msg,
"Le sujet spcifi d'aide n'a pas t trouv, pour une liste de\n\
sujets d'aide s'il vous plat regard dans les contenus en allant\n\
pour Aider->Contenus.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Keine Hilfe Verfgbar",
msg,
"Das angegebene hilfe thema wurde, fr eine liste von hilfe\n\
themen bitte aussehen im inhalt durch der gehen nicht\n\
Hilfe->Themen.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Nessuno Aiuto Disponibile",
msg,
"Il soggetto di aiuto specificato non  stato trovato, per un\n\
elenco di soggetti di aiuto per favore sguardo nel contenuto da\n\
andare di Aiutare->Contenuto.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Geen Hulp Verkrijgbaar",
msg,
"Het gespecificeerd hulp onderwerp werd door gaan, voor een lijst\n\
van hulp onderwerpen alstublieft blik in de inhoud niet\n\
Hulp->Gevonden.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Nenhuma Ajuda Disponvel",
msg,
"O tema especificado de ajuda no foi achado, para uma lista de\n\
temas de ajuda por favor olha no contedo por ir Ajudar->Contedo.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ingen Hjelp Tilgjengelig",
msg,
"Det spesifiserte hjelpeemne fant ikke, for en liste av hjelpeemner\n\
behager titt i innholdet ved  dra Hjelpe->Innhold.\n",
#else
"Help Failed",
msg,
"The specified help topic was not found, for a list of help topics\n\
please look in the contents by going to Help->Contents.\n",
#endif
		toplevel
	    );
	    g_free(msg);
	    return;
	}

	/* Help file does not exist or is not readable? */
	if(!edv_path_is_readable(help_file))
	{
	    gchar *msg = g_strdup_printf(
"Unable to find help documentation:\n\
\n\
    %s",
		help_file
	    );
	    edv_play_sound_warning(core);
	    edv_message_warning(
		"Help Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);

	    g_free(help_file);
	    return;
	}


	/* Search for a help program to use
	 *
	 * First check if the help program is specified in an
	 * environment variable
	 *
	 * If no help program is specified then use the first
	 * available help program found to exist in the help programs
	 * list
	 */

	/* Check if the HELPBROWSER environment variable is set */
	s = g_getenv(ENV_VAR_NAME_HELPBROWSER);
	if(STRISEMPTY(s))
	{
	    /* HELPBROWSER not set, fall back to the BROWSER environment
	     * variable
	     */
	    s = g_getenv(ENV_VAR_NAME_BROWSER);
	    if(STRISEMPTY(s))
	    {
#if 0
/* Do not warn, since a help browser can still be obtained from the
 * list of help browsers if no environment variable is set
 */
		edv_play_sound_warning(core);
		edv_message_warning(
"Help Warning",
"Help browser environment variables not set:\n\
\n\
    " ENV_VAR_NAME_HELPBROWSER "\n\
    " ENV_VAR_NAME_BROWSER "\n\
\n\
(Click on \"Help\" for more information)\n",
"The environment variable which specifies which help browser to use\n\
to display the help documentation is not set. To set this you need to\n\
edit your shell configuration file (this is usually the file named\n\
.bashrc in your home directory) and add the lines:\n\
\n\
" ENV_VAR_NAME_BROWSER "=/usr/bin/mozilla\n\
export " ENV_VAR_NAME_BROWSER "\n\
" ENV_VAR_NAME_HELPBROWSER "=/usr/bin/mozilla\n\
export " ENV_VAR_NAME_HELPBROWSER "\n\
\n\
Save the file and then restart this application.\n",
		    toplevel
		);
#endif
	    }
	}
	/* Environment variable set? */
	if(!STRISEMPTY(s))
	{
	    /* Found help program set from environment variable */
	    help_prog.prog = s;
	    help_prog.arg = "";
	    help_prog.arg_type = EDV_HELP_ARGUMENT_URL;
	    help_prog.reserved = 0;
	    hp_ptr = &help_prog;
	}
	else
	{
	    /* No run time help program specified, so look for help
	     * program specified in the help programs list
	     */
	    hp_ptr = NULL;
	    for(i = 0; help_prog_list[i].prog != NULL; i++)
	    {
		if(edv_help_is_file_executable(help_prog_list[i].prog))
		{
		    hp_ptr = &help_prog_list[i];
		    break;
		}
	    }
	}

	/* Found a help program? */
	if(hp_ptr != NULL)
	{
	    pid_t p;
	    gchar	*cmd = NULL,
			*shell_prog;
	    const gchar *shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	    );

	    /* Format the command */
	    cmd = edv_strcat(cmd, hp_ptr->prog);
	    if(hp_ptr->arg != NULL)
	    {
		cmd = edv_strcat(cmd, " ");
		cmd = edv_strcat(cmd, hp_ptr->arg);
	    }
	    switch(hp_ptr->arg_type)
	    {
	      case EDV_HELP_ARGUMENT_URL:
		cmd = edv_strcat(cmd, " file://");
		cmd = edv_strcat(cmd, help_file);
		break;

	      default:	/* EDV_HELP_ARGUMENT_PATH */
		cmd = edv_strcat(cmd, " ");
		cmd = edv_strcat(cmd, help_file);
		break;
	    }

	    /* Execute the help viewer */
	    p = edv_system_shell(
		cmd,
		shell_prog,
		shell_args
	    );
	    if(p <= 0)
	    {
		/* Unable to execute help viewer */
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to execute help viewer command:\n\
\n\
    %s\n\
\n\
%s.",
		    cmd,
		    g_strerror(error_code)
		);
		edv_play_sound_warning(core);
		edv_message_warning(
"Help Failed",
		    msg,
"An error was encountered while attempting to run the help\n\
viewer to display the requested document. Please verify that\n\
the help viewer and the requested document are both\n\
installed properly.",
		    toplevel
		);
		g_free(msg);
	    }

	    g_free(cmd);
	    g_free(shell_prog);
	}
	else
	{
	    gchar *msg = g_strdup_printf(
"Unable to find a help viewer to display help documentation:\n\
\n\
    %s",
		help_file
	    );
	    edv_play_sound_warning(core);
	    edv_message_warning(
		"Help Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	}

	g_free(help_file);
}
