#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "imgio.h"

#include "guirgbimg.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property_directory.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_image_io.h"
#include "edv_utils_gtk.h"
#include "edv_folder.h"
#include "edv_emit.h"

#include "config.h"

#include "images/icon_tools_32x32.xpm"


typedef struct _EDVFolderProgressData	EDVFolderProgressData;
#define EDV_FOLDER_PROGRESS_DATA(p)	((EDVFolderProgressData *)(p))


static gint edv_folder_resize_progress_cb(
	const gulong i, const gulong n,
	gpointer data
);
static gint edv_folder_open_image_progress_cb(
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const guint8 *rgba,
	const gulong i, const gulong n,
	gpointer data
);
#ifdef HAVE_LIBXPM
static int edv_folder_save_xpm_progress_cb(
        void *data,
        const int i, const int n,
        const int width, const int height,
        const int bpl, const int bpp,
        const u_int8_t *image_data
);
#endif

gint edv_folder_set_icons_from_image(
	EDVCore *core,
	const gchar *directory_path,
	const gchar *image_path,
	const gboolean show_progress,
	const gboolean interactive,
	GtkWidget *toplevel
);
gint edv_folder_remove_all_customizations(
	EDVCore *core,
	const gchar *directory_path,
	const gboolean show_progress,
	const gboolean interactive,
	GtkWidget *toplevel
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


struct _EDVFolderProgressData {

	gint		nstages,
			current_stage_index;
	gfloat		current_stage_coeff;	/* current_stage_index /
						 * nstages */
	const gchar	*image_path;

};
#define EDV_FOLDER_PROGRESS_DATA_INCREMENT_STAGE(p)	{	\
 (p)->current_stage_index++;					\
 if((p)->nstages > 0)						\
  (p)->current_stage_coeff = (gfloat)((p)->current_stage_index) / \
   (gfloat)((p)->nstages);					\
 gdk_flush();							\
}


/*
 *	GUIImageBufferResize() progress callback.
 */
static gint edv_folder_resize_progress_cb(
	const gulong i, const gulong n,
	gpointer data
)
{
	EDVFolderProgressData *progress_data = EDV_FOLDER_PROGRESS_DATA(data);
	const gfloat	this_stage_coeff = (n > 0l) ? (gfloat)i / (gfloat)n : 0.0f,
			overall_coeff = progress_data->current_stage_coeff +
		(this_stage_coeff / (gfloat)progress_data->nstages);
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			overall_coeff,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(1);
	}

	return(0);
}

/*
 *	Open image progress callback.
 */
static gint edv_folder_open_image_progress_cb(
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const guint8 *rgba,
	const gulong i, const gulong n,
	gpointer data
)
{
	EDVFolderProgressData *progress_data = EDV_FOLDER_PROGRESS_DATA(data);
	const gfloat	this_stage_coeff = (n > 0l) ? (gfloat)i / (gfloat)n : 0.0f,
			overall_coeff = progress_data->current_stage_coeff +
		(this_stage_coeff / (gfloat)progress_data->nstages);
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			overall_coeff,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(1);
	}

	return(0);
}

#ifdef HAVE_LIBXPM
/*
 *	Save XPM progress callback.
 */
static int edv_folder_save_xpm_progress_cb(
        void *data,
        const int i, const int n,
        const int width, const int height,
        const int bpl, const int bpp,
        const u_int8_t *image_data
)
{
	EDVFolderProgressData *progress_data = EDV_FOLDER_PROGRESS_DATA(data);
	const gfloat	this_stage_coeff = (n > 0l) ? (gfloat)i / (gfloat)n : 0.0f,
			overall_coeff = progress_data->current_stage_coeff +
		(this_stage_coeff / (gfloat)progress_data->nstages);
	if(ProgressDialogIsQuery())
	{
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			overall_coeff,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);
		if(ProgressDialogStopCount() > 0)
			return(0);
	}

	return(1);
}
#endif	/* HAVE_LIBXPM */

/*
 *	Opens the image file, creates a set of icon files from the
 *	image file, sets the icon paths in the directory's property
 *	file to refer to the set of created icon files, and notifies
 *	about the modification of the directory, directory's property
 *	file and icon files.
 *
 *	The directory_path specifies the directory.
 *
 *	The image_path specifies the image to open.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_folder_set_icons_from_image(
	EDVCore *core,
	const gchar *directory_path,
	const gchar *image_path,
	const gboolean show_progress,
	const gboolean interactive,
	GtkWidget *toplevel
)
{
	const gboolean maintain_aspect = TRUE;
	gboolean prop_path_existed;
	const gint bpp = 4;			/* RGBA */
	gint		width, height,
			bpl,
			nframes;
	const guint8 *src_rgba;
	guint8 bg_color[4];
	gchar *prop_path;
	GList		*frames_list,
			*delay_list;
	EDVFolderProgressData *progress_data;

	if((core == NULL) || STRISEMPTY(directory_path) ||
	   STRISEMPTY(image_path)
	)
	{
		errno = EINVAL;
		return(-2);
	}

	/* Get the path to the directory properties file */
	prop_path = g_strconcat(
		directory_path,
		G_DIR_SEPARATOR_S,
		EDV_NAME_DIRECTORY_PROPERTIES_FILE,
		NULL
	);

	/* Record the directory properties file's existance */
	prop_path_existed = edv_path_lexists(prop_path);

	/* Map the progress dialog */
	if(show_progress && !ProgressDialogIsQuery())
	{
		gchar	*p1 = edv_path_shorten(
			image_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
"Creating icons from image:\n\
\n\
    %s",
			p1
		);
		g_free(p1);
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
			"Creating Icons",
			msg,
			(const guint8 **)icon_tools_32x32_xpm,
			"Stop"
		);
		ProgressDialogUpdate(
			NULL,
			NULL,
			NULL,
			NULL,
			0.0f,
			EDV_PROGRESS_BAR_NTICKS,
			TRUE
		);
	        gdk_flush();
		g_free(msg);
	}

	/* Allocate the progress data */
	progress_data = EDV_FOLDER_PROGRESS_DATA(g_malloc0(
		sizeof(EDVFolderProgressData)
	));
	if(progress_data == NULL)
	{
		return(-3);
	}

	progress_data->image_path = image_path;
	progress_data->nstages++;		/* edv_image_open_rgba() */
#ifdef HAVE_LIBXPM
	progress_data->nstages += 3;		/* 3 * ImgFileSaveXPMRGBA() */
	progress_data->nstages += 3;		/* 3 * GUIImageBufferResize() */
#endif

	/* Open the image */
	frames_list = edv_image_open_rgba(
		image_path,
		&width, &height,
		&bpl,
		&nframes,
		&delay_list,
		bg_color,
		edv_folder_open_image_progress_cb, progress_data
	);
	EDV_FOLDER_PROGRESS_DATA_INCREMENT_STAGE(progress_data);

	/* Unmap the progress dialog which may have been mapped during
	 * the above operation
	 */
	if(show_progress)
	{
	        ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
	}

	/* Unable to open the image? */
	if(frames_list == NULL)
	{
		const gint error_code = (gint)errno;
		if(interactive)
		{
			gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
				ImgOpenGetError(),
				image_path
			);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Open Image Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
		g_list_free(delay_list);
		g_free(prop_path);
		g_free(progress_data);
		errno = (int)error_code;
		return(-1);
	}

	/* Use the first frame */
	src_rgba = (const guint8 *)g_list_nth_data(
		frames_list,
		0
	);
	if(src_rgba != NULL)
	{
#ifdef HAVE_LIBXPM

/* Resizes the src_rgba image data as needed and saves it to an XPM icon
 * file */
#define SET_ICON(_icon_width_,_icon_height_,_icon_size_) {	\
 int status;							\
 gboolean icon_path_existed;					\
 gchar *icon_path;						\
 const gint icon_bpl = (_icon_width_) * bpp;			\
								\
 /* Allocate the icon RGBA image data and copy/resize the	\
  * source image data to the icon image data			\
  */								\
 guint8 *icon_rgba = (guint8 *)g_malloc(			\
  (_icon_height_) * icon_bpl * sizeof(guint8)			\
 );								\
 if(icon_rgba != NULL) {					\
  if(maintain_aspect) {						\
   /* Calculate the aspect ratio from the opened image size */	\
   const gfloat aspect = (width > 0) ?				\
    ((gfloat)height / (gfloat)width) : 1.0f;			\
   /* Calculate the resized icon size with aspect ratio		\
    * maintained while not resizing if the image size is	\
    * smaller than the icon's size				\
    */								\
   gint rs_width = (width > (_icon_width_)) ?			\
         (_icon_width_) : width,				\
	rs_height = (gint)(rs_width * aspect),			\
	rs_bpl;							\
   guint8 *rs_rgba;						\
								\
   /* Clip the resized icon size with aspect ratio maintained */\
   if(rs_height > (_icon_height_)) {				\
    rs_height = (height > (_icon_height_)) ?			\
     (_icon_height_) : height;					\
    if(aspect > 0.0f)						\
     rs_width = (gint)(rs_height / aspect);			\
   }								\
   rs_bpl = rs_width * bpp;					\
								\
   /* Allocate the resized icon RGBA image data */		\
   rs_rgba = (guint8 *)g_malloc(				\
    rs_height * rs_bpl * sizeof(guint8)				\
   );								\
   if(rs_rgba != NULL) {					\
    /* Copy/resize the source RGBA image data to the resized	\
     * icon RGBA image data					\
     */								\
    GUIImageBufferResize(					\
     bpp,							\
     src_rgba,							\
     width, height, bpl,					\
     rs_rgba,							\
     rs_width, rs_height, rs_bpl,				\
     edv_folder_resize_progress_cb, progress_data		\
    );								\
    EDV_FOLDER_PROGRESS_DATA_INCREMENT_STAGE(progress_data);	\
    /* Clear the icon RGBA image data (make transparent) */	\
    (void)memset(						\
     icon_rgba,							\
     0x00,							\
     (_icon_height_) * icon_bpl * sizeof(guint8)		\
    );								\
    /* Copy the resized icon RGBA image data to the icon RGBA	\
     * image data						\
     */								\
    GUIImageBufferCopyArea(					\
     bpp,							\
     rs_rgba,							\
     rs_width, rs_height, rs_bpl,				\
     icon_rgba,							\
     (_icon_width_), (_icon_height_), icon_bpl,			\
     ((_icon_width_) - rs_width) / 2,				\
     ((_icon_height_) - rs_height) / 2,				\
     TRUE,				/* Blend */		\
     NULL, NULL							\
    );								\
    /* Delete the resized icon RGBA image data */		\
    g_free(rs_rgba);						\
   }								\
  } else {							\
   /* Copy/resize the source RGBA image data to the icon RGBA	\
    * image data while not maintaining aspect			\
    */								\
   GUIImageBufferResize(					\
    bpp,							\
    src_rgba,							\
    width, height, bpl,						\
    icon_rgba,							\
    (_icon_width_), (_icon_height_), icon_bpl,			\
    edv_folder_resize_progress_cb, progress_data		\
   );								\
   EDV_FOLDER_PROGRESS_DATA_INCREMENT_STAGE(progress_data);	\
  }								\
 }								\
								\
 /* Format the path to this icon file */			\
 icon_path = g_strdup_printf(					\
  "%s%c.icon_%ix%i.xpm",					\
  directory_path,						\
  G_DIR_SEPARATOR,						\
  (_icon_width_), (_icon_height_)				\
 );								\
								\
 /* Record the icon file's existance */				\
 icon_path_existed = edv_path_lexists(icon_path);		\
								\
/* TODO Confirm overwrite */					\
								\
 /* Save the icon RGBA image data to an XPM icon file */	\
 status = ImgFileSaveXPMRGBA(					\
  icon_path,							\
  (_icon_width_), (_icon_height_),				\
  icon_bpl,							\
  icon_rgba,							\
  bg_color,							\
  0, 0,			/* X, y */				\
  (_icon_width_), (_icon_height_),	/* Base size */		\
  g_basename(icon_path),	/* C ID */			\
  NULL,			/* No comments */			\
  2,			/* Color */				\
  -1,			/* Max colors (-1 no limit) */		\
  0x80,			/* Threshold */				\
  edv_folder_save_xpm_progress_cb, progress_data		\
 );								\
 EDV_FOLDER_PROGRESS_DATA_INCREMENT_STAGE(progress_data);	\
								\
 /* Delete the icon RGBA image data  */				\
 g_free(icon_rgba);						\
								\
 if(status != 0) {						\
  /* Save failed */						\
  if(interactive) {						\
   gchar *msg = g_strdup_printf(				\
"%s:\n\
\n\
    %s",							\
    ImgSaveGetError(),						\
    icon_path							\
   );								\
   edv_play_sound_warning(core);				\
   edv_message_warning(						\
    "Save Image Failed",					\
    msg,							\
    NULL,							\
    toplevel							\
   );								\
   g_free(msg);							\
  }								\
 } else {							\
  /* Save successful, notify about the icon file being added	\
   * or modified						\
   */								\
  EDVVFSObject *obj = edv_vfs_object_lstat(icon_path);		\
  if(obj != NULL) {						\
   if(icon_path_existed)					\
    edv_emit_vfs_object_modified(				\
     core,							\
     icon_path,							\
     icon_path,							\
     obj							\
    );								\
   else								\
    edv_emit_vfs_object_added(					\
     core,							\
     icon_path,							\
     obj							\
    );								\
   edv_vfs_object_delete(obj);					\
  }								\
								\
  /* Set/update the icon path in the directory properties file */\
  edv_property_directory_set_icon_path(				\
   directory_path,						\
   (_icon_size_),						\
   icon_path							\
  );								\
 }								\
								\
 g_free(icon_path);						\
}

		SET_ICON(48, 48, EDV_ICON_SIZE_48);
		SET_ICON(32, 32, EDV_ICON_SIZE_32);
		SET_ICON(20, 20, EDV_ICON_SIZE_20);

#undef SET_ICON

#endif	/* HAVE_LIBXPM */
	}


	/* Delete the opened image data */
	g_list_foreach(frames_list, (GFunc)g_free, NULL);
/*	src_rgba = NULL; */
	g_list_free(frames_list);
/*	frames_list = NULL; */
	g_list_free(delay_list);
/* 	delay_list = NULL; */

	/* Unmap the progress dialog which may have been mapped during
	 * the above operation
	 */
	if(show_progress)
	{
	        ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
	}

	/* Notify about the properties file being added or modified */
	if(!STRISEMPTY(prop_path))
	{
		EDVVFSObject *obj = edv_vfs_object_lstat(prop_path);
		if(obj != NULL)
		{
			if(prop_path_existed)
				edv_emit_vfs_object_modified(
					core,
					prop_path,
					prop_path,
					obj
				);
			else
				edv_emit_vfs_object_added(
					core,
					prop_path,
					obj
				);
			edv_vfs_object_delete(obj);
		}
	}

	/* Notify about the directory being modified */
	if(!STRISEMPTY(directory_path))
	{
		EDVVFSObject *obj = edv_vfs_object_lstat(directory_path);
		if(obj != NULL)
		{
			edv_emit_vfs_object_modified(
			       core,
			       directory_path,
			       directory_path,
			       obj
			);
			edv_vfs_object_delete(obj);
		}
	}

	g_free(prop_path);
	g_free(progress_data);

	return(0);
}

/*
 *	Removes the directory's properties file and icon files, and
 *	notifies about the removal.
 *
 *	The directory_path specifies the directory.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_folder_remove_all_customizations(
	EDVCore *core,
	const gchar *directory_path,
	const gboolean show_progress,
	const gboolean interactive,
	GtkWidget *toplevel
)
{
	gchar *prop_path;

	if((core == NULL) || STRISEMPTY(directory_path))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Get the path to the directory properties file */
	prop_path = g_strconcat(
		directory_path,
		G_DIR_SEPARATOR_S,
		EDV_NAME_DIRECTORY_PROPERTIES_FILE,
		NULL
	);
	if(!edv_path_lexists(prop_path))
	{
		g_free(prop_path);
		return(0);
	}

	/* Remove the properties file and icon files, this removing
	 * all the customizations from this directory
	 */
	if(edv_unlink(prop_path))
	{
		const gint error_code = (gint)errno;
		if(interactive)
		{
			/* Failed to remove the properties file */
			gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
				g_strerror(error_code),
				prop_path
			);
			edv_play_sound_error(core);
			edv_message_error(
				"Remove Customizations Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
	        g_free(prop_path);
		errno = (int)error_code;
		return(-1);
	}
	else
	{
		EDVVFSObject *obj;

		/* Remove the icon files */
#define REMOVE_ICON_FILE(_icon_width_,_icon_height_,_icon_size_) { \
 gchar *icon_path = g_strdup_printf(				\
  "%s%c.icon_%ix%i.xpm",					\
  directory_path,						\
  G_DIR_SEPARATOR,						\
  (_icon_width_), (_icon_height_)				\
 );								\
 if(edv_unlink(icon_path)) {					\
								\
 } else {							\
  edv_emit_vfs_object_removed(					\
   core,							\
   icon_path							\
  );								\
 }								\
								\
 g_free(icon_path);						\
}
		REMOVE_ICON_FILE(48, 48, EDV_ICON_SIZE_48);
		REMOVE_ICON_FILE(32, 32, EDV_ICON_SIZE_32);
		REMOVE_ICON_FILE(20, 20, EDV_ICON_SIZE_20);
#undef REMOVE_ICON_FILE

		/* Notify that the directory properties file has been
		 * removed
		 */
		edv_emit_vfs_object_removed(
			core,
			prop_path
		);

		/* Notify that this directory has been modified */
		obj = edv_vfs_object_lstat(directory_path);
		if(obj != NULL)
		{
			edv_emit_vfs_object_modified(
				core,
				directory_path,
				directory_path,
				obj
			);
			edv_vfs_object_delete(obj);
		}
	}

	g_free(prop_path);

	return(0);
}
