#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "pulist.h"

#include "cfg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_date_parse.h"
#include "edv_find.h"
#include "edv_utils_gtk.h"
#include "edv_find_bar.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_stop_20x20.xpm"


typedef struct _EDVFindBar		EDVFindBar;
#define EDV_FIND_BAR(p)			((EDVFindBar *)(p))
#define EDV_FIND_BAR_KEY		"EDV/FindBar"


/*
 *	Flags:
 */
typedef enum {
	EDV_FIND_BAR_MAPPED		= (1 << 0),
	EDV_FIND_BAR_REALIZED		= (1 << 1)
} EDVFindBarFlags;


/* Callbacks */
static void edv_find_bar_destroy_cb(gpointer data);
static void edv_find_bar_realize_cb(GtkWidget *widget, gpointer data);
static void edv_find_bar_show_cb(GtkWidget *widget, gpointer data);
static void edv_find_bar_hide_cb(GtkWidget *widget, gpointer data);
static gint edv_find_bar_progress_cb(
	const gchar *path, const gfloat progress,
	gpointer data
);
static void edv_find_bar_match_cb(
	const gchar *path,
	GList *properties_list,
	gpointer data
);
static void edv_find_bar_match_excerpt_cb(
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
);

static void edv_find_bar_find_cb(GtkWidget *widget, gpointer data);
static void edv_find_bar_stop_cb(GtkWidget *widget, gpointer data);
static gint edv_find_bar_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

static void edv_find_bar_find_by_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
);


/* Search */
void edv_find_bar_set_search(
	GtkWidget *w,
	const gchar *s,
	const gboolean record_history
);

/* Find By */
EDVFindBarFindBy edv_find_bar_current_find_by(GtkWidget *w);

/* Find Bar */
static EDVFindBar *EDVFindBarGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
);
GtkWidget *edv_find_bar_new(
	EDVCore *core, 
	const EDVLocationType location_type,
	const gchar *(*get_location_cb)(
		GtkWidget *,                    /* Find Bar */
		gpointer                        /* get_location_data */
	),
	gpointer get_location_data,
	void (*start_cb)(
		GtkWidget *,                    /* Find Bar */
		gpointer                        /* start_data */
	),
	gpointer start_data,
	void (*end_cb)(
		GtkWidget *,                    /* Find Bar */
		const gint,                     /* Number of matches */
		gpointer                        /* end_data */
	),
	gpointer end_data,
	void (*match_cb)(
		GtkWidget *,                    /* Find Bar */
		const gchar *,                  /* Name/path of object */ 
		GList *,                        /* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,                     /* Line index */
		const gchar *,                  /* Excerpt */
		gpointer                        /* match_data */
	),
	gpointer match_data,
	void (*status_message_cb)(
		GtkWidget *,                    /* Find Bar */
		const gchar *,                  /* Message */
		gpointer                        /* status_message_data */
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
		GtkWidget *,                    /* Find Bar */
		const gfloat,                   /* Progress */
		gpointer                        /* status_progress_data */
	),
	gpointer status_progress_data
);
void edv_find_bar_update_display(GtkWidget *w);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Find Bar:
 */
struct _EDVFindBar {

	GtkWidget	*toplevel;
	gboolean	processing;
	gint		freeze_count,
			stop_count;
	EDVCore	*core;
	EDVFindBarFlags	flags;

	GtkWidget	*find_by_pulistbox,
			*search_combo,
			*case_sensitive_check,
			*stop_btn;

	EDVLocationType	location_type;

	/* Get current location callback */
	const gchar	*(*get_location_cb)(
		GtkWidget *,			/* EDVFindBar */
		gpointer			/* get_location_data */
	);
	gpointer	get_location_data;

	/* Find start callback */
	void		(*start_cb)(
		GtkWidget *,			/* EDVFindBar */
		gpointer			/* start_data */
	);
	gpointer	start_data;

	/* Find end callback */
	void		(*end_cb)(
		GtkWidget *,			/* EDVFindBar */
		const gint,			/* Total Matches */
		gpointer			/* end_data */
	);
	gpointer	end_data;

	/* Matched path callback
	 *
	 * Note, if the role is set to EDV_LOCATION_TYPE_RECYCLE_BIN
	 * then the path is a string describing the index of the
	 * recycled object
	 */
	void		(*match_cb)(
		GtkWidget *,			/* EDVFindBar */
		const gchar *,			/* Name/path of object */ 
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	);
	gpointer	match_data;

	/* Status message callback */
	void		(*status_message_cb)(
		GtkWidget *,			/* EDVFindBar */
		const gchar *,			/* Message */
		gpointer			/* status_message_data */
	);
	gpointer	status_message_data;

	/* Status progress callback */
	void		(*status_progress_cb)(
		GtkWidget *,			/* EDVFindBar */
		const gfloat,			/* Progress */
		gpointer			/* status_progress_data */
	);

	gpointer	status_progress_data;

};


/*
 *	Toplevel GtkObject data "destroy" signal callback.
 */
static void edv_find_bar_destroy_cb(gpointer data)
{
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if(bar == NULL)
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	bar->freeze_count--;

	g_free(bar);
}

/*
 *	Toplevel GtkWidget "realize" signal callback.
 */
static void edv_find_bar_realize_cb(GtkWidget *widget, gpointer data)
{
	GdkWindow *window;
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	if(bar->flags & EDV_FIND_BAR_REALIZED)
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	window = widget->window;
	if(window == NULL)
	{

	}

	/* Mark the EDVFindBar as realized */
	bar->flags |= EDV_FIND_BAR_REALIZED;

	bar->freeze_count--;
}

/*
 *	Toplevel GtkWidget "show" signal callback.
 */
static void edv_find_bar_show_cb(GtkWidget *widget, gpointer data)
{
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	/* Mark the EDVFindBar as mapped */
	bar->flags |= EDV_FIND_BAR_MAPPED;
}

/*
 *	Toplevel GtkWidget "hide" signal callback.
 */
static void edv_find_bar_hide_cb(GtkWidget *widget, gpointer data)
{
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	/* Handle this signal even if currently frozen */

	/* Mark the EDVFindBar as not mapped */
	bar->flags &= ~EDV_FIND_BAR_MAPPED;
}

/*
 *	Progress callback, used for edv_find_vfs_object_by_name() and similar
 *	functions.
 */
static gint edv_find_bar_progress_cb(
	const gchar *path, const gfloat progress,
	gpointer data
)
{
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if(bar == NULL)
		return(0);

	/* User aborted? */
	if(bar->stop_count > 0)
		return(1);

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	if(bar->status_message_cb != NULL)
		bar->status_message_cb(
			bar->toplevel,
			path,
			bar->status_message_data
		);
	if(bar->status_progress_cb != NULL)
		bar->status_progress_cb(
			bar->toplevel,
			progress,
			bar->status_progress_data
		);

	gtk_events_process();

	bar->freeze_count--;

	return(0);
}

/*
 *	Match callback.
 */
static void edv_find_bar_match_cb(
	const gchar *path,
	GList *properties_list,
	gpointer data
)
{
	edv_find_bar_match_excerpt_cb(
		path,
		properties_list,
		-1,				/* No line index */
		NULL,				/* No excerpt */
		data
	);
}

/*
 *	Match excerpt callback.
 */
static void edv_find_bar_match_excerpt_cb(
	const gchar *path,
	GList *properties_list,
	const gint line_index,
	const gchar *excerpt,
	gpointer data
)
{
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if(bar == NULL)
		return;

	/* Handle this signal even if currently frozen */

	bar->freeze_count++;

	if(bar->match_cb != NULL)
		bar->match_cb(
			bar->toplevel,		/* EDVFindBar */
			path,			/* Path */
			properties_list,	/* Properties list */
			line_index,		/* Line Index */
			excerpt,		/* Excerpt */
			bar->match_data		/* Data */
		);

	gtk_events_process();

	bar->freeze_count--;
}

/*
 *	Find GtkEntry "activate" signal callback.
 *
 *	This starts the search process.
 */
static void edv_find_bar_find_cb(GtkWidget *widget, gpointer data)
{
	gboolean case_sensitive = FALSE;
	gint nmatches;
	const gchar *s;
	gchar		*location,
			*search_string;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVFindBarFindBy operation;
	EDVCore *core;
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if(bar == NULL)
		return;

	if(bar->freeze_count > 0)
		return;

	core = bar->core;
	cfg_list = core->cfg_list;

	bar->freeze_count++;
	bar->processing = TRUE;

	/* Reset the find values */
	bar->stop_count = 0;
	edv_find_bar_update_display(bar->toplevel);

	/* Get the operation */
	operation = edv_find_bar_current_find_by(bar->toplevel);

	/* Get/copy the location */
	s = NULL;
	switch(bar->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	  case EDV_LOCATION_TYPE_ARCHIVE:
		if(bar->get_location_cb != NULL)
			s = bar->get_location_cb(
				bar->toplevel,
				bar->get_location_data
			);
		break;
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
		s = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
		break;
	}
	location = STRDUP((s != NULL) ? s : "/");

	/* Get/copy the search string */
	s = NULL;
	w = bar->search_combo;
	if(w != NULL)
	{
		GtkCombo *combo = GTK_COMBO(w);
		GtkEntry *entry = GTK_ENTRY(combo->entry);
		s = gtk_entry_get_text(entry);
	}
	search_string = g_strdup((s != NULL) ? s : "");
	if(search_string == NULL)
	{
		g_free(location);
		bar->processing = FALSE;
		bar->freeze_count--;
		return;
	}

	/* No wild cards in the search string and finding by name? */
	if((strchr((char *)search_string, '*') == NULL) &&
	   (strchr((char *)search_string, '?') == NULL) &&
	   (operation == EDV_FIND_BAR_FIND_BY_NAME)
	)
	{
		/* Tack on two '*' characters to the beginning and end of
		 * the search string.
		 */
		gchar *s = g_strdup_printf("*%s*", search_string);
		g_free(search_string);
		search_string = s;
	}
	/* Record the search string history */
	edv_find_bar_set_search(
		bar->toplevel,
		search_string,
		TRUE
	);

	/* Case sensitive */
	case_sensitive = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		bar->case_sensitive_check
	);


	/* Notify about the start of the find */
	if(bar->start_cb != NULL)
		bar->start_cb(
			bar->toplevel,
			bar->start_data
		);


	/* Perform the search by the operation */
	nmatches = 0;
	switch(operation)
	{
	  case EDV_FIND_BAR_FIND_BY_NAME:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_name(
				location,
				search_string,
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				case_sensitive,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;

		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_name(
				location,	/* Recycled objects index file */
				search_string,
				case_sensitive,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;

		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_name(
				core,
				location,	/* Archive */
				search_string,
				case_sensitive,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_CONTENT:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_content(
				location,
				search_string,
				EDV_FIND_STRING_MATCH_LITERAL,
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				case_sensitive,	/* Case sensitive */
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_excerpt_cb, bar
			);
			break;

		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_content(
				location,	/* Recycled objects index file */
				search_string,
				EDV_FIND_STRING_MATCH_LITERAL,
				case_sensitive,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_excerpt_cb, bar
			);
			break;

		  case EDV_LOCATION_TYPE_ARCHIVE:
			/* Not supported */
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_EQUAL_TO:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_size(
				location,
				ATOL(search_string),	/* Size */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_size(
				location,	/* Recycled objects index file */
				ATOL(search_string),	/* Size */
				EDV_FIND_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_size(
				core,
				location,	/* Archive */
				ATOL(search_string),	/* Size */
				EDV_FIND_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_NOT_EQUAL_TO:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_size(
				location,
				ATOL(search_string),	/* Size */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_NOT_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_size(
				location,	/* Recycled objects index file */
				ATOL(search_string),	/* Size */
				EDV_FIND_NOT_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_size(
				core,
				location,	/* Archive */
				ATOL(search_string),	/* Size */
				EDV_FIND_NOT_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_LESS_THAN:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_size(
				location,
				ATOL(search_string),	/* Size */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_LESS_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_size(
				location,	/* Recycled objects index file */
				ATOL(search_string),	/* Size */
				EDV_FIND_LESS_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_size(
				core,
				location,	/* Archive */
				ATOL(search_string),	/* Size */
				EDV_FIND_LESS_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_GREATER_THAN:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_size(
				location,
				ATOL(search_string),	/* Size */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_GREATER_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_size(
				location,	/* Recycled objects index file */
				ATOL(search_string),	/* Size */
				EDV_FIND_GREATER_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_size(
				core,
				location,	/* Archive */
				ATOL(search_string),	/* Size */
				EDV_FIND_GREATER_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_EQUAL_TO:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_modify_time(
				location,
				edv_date_parse_epoch(search_string),	/* Time */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_modify_time(
				location,	/* Recycled objects index file */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_modify_time(
				core,
				location,	/* Archive */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_modify_time(
				location,
				edv_date_parse_epoch(search_string),	/* Time */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_NOT_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_modify_time(
				location,	/* Recycled objects index file */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_NOT_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_modify_time(
				core,
				location,	/* Archive */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_NOT_EQUAL_TO,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_LESS_THAN:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_modify_time(
				location,
				edv_date_parse_epoch(search_string),	/* Time */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_LESS_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_modify_time(
				location,	/* Recycled objects index file */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_LESS_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_modify_time(
				core,
				location,	/* Archive */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_LESS_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_GREATER_THAN:
		switch(bar->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
			nmatches = edv_find_vfs_object_by_modify_time(
				location,
				edv_date_parse_epoch(search_string),	/* Time */
				FALSE,		/* Do not recurse */
				FALSE,		/* Do not follow links */
				EDV_FIND_GREATER_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_RECYCLE_BIN:
			nmatches = edv_find_recycled_object_by_modify_time(
				location,	/* Recycled objects index file */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_GREATER_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		  case EDV_LOCATION_TYPE_ARCHIVE:
			nmatches = edv_find_archive_object_by_modify_time(
				core,
				location,	/* Archive */
				edv_date_parse_epoch(search_string),	/* Time */
				EDV_FIND_GREATER_THAN,
				edv_find_bar_progress_cb, bar,
				edv_find_bar_match_cb, bar
			);
			break;
		}
		break;
	}


	/* Tally */
	if(bar->status_progress_cb != NULL)
		bar->status_progress_cb(
			bar->toplevel,
			0.0f,
			bar->status_progress_data
		);

	if(nmatches > 0)
	{
		if(bar->status_message_cb != NULL)
		{
			gchar *msg = g_strdup_printf(
				"Matched %i %s",
				nmatches,
				(nmatches == 1) ? "object" : "objects"
			);
			bar->status_message_cb(
				bar->toplevel,
				msg,
				bar->status_message_data
			);
			g_free(msg);
		}
	}
	else
	{
		if(bar->status_message_cb != NULL)
			bar->status_message_cb(
				bar->toplevel,
				"No objects found",
				bar->status_message_data
			);
	}

	bar->processing = FALSE;
	bar->freeze_count--;

	edv_find_bar_update_display(bar->toplevel);


	/* Call end find callback to notify about end of find */
	if(bar->end_cb != NULL)
		bar->end_cb(
			bar->toplevel,
			nmatches,
			bar->end_data
		);


	g_free(location);
	g_free(search_string);
}

/*
 *	Stop callback.
 */
static void edv_find_bar_stop_cb(GtkWidget *widget, gpointer data)
{
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if(bar == NULL)
		return;

	bar->stop_count++;
}

/*
 *	Any GtkWidget "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
static gint edv_find_bar_crossing_cb(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	GtkWidget *w;
	const gchar *msg = NULL;
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if((widget == NULL) || (crossing == NULL) || (bar == NULL))
		return(status);

	/* Get event type */
	etype = (gint)crossing->type;
	w = widget;

	/* Handle by event type */
	switch(etype)
	{
	  case GDK_ENTER_NOTIFY:
		if(w == bar->case_sensitive_check)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Especifica si los iguales de cuerda no deben ser el casos sensibles ni"
#elif defined(PROG_LANGUAGE_FRENCH)
"Spcifie si les allumettes de ficelle doivent tre des cas sensibles ou pas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gibt an, wenn Schnur Gegenstcke Fall empfindlich oder nicht sein sollen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Non specifica se i fiammiferi di cordicella dovrebbero essere dei casi sensibili o"
#elif defined(PROG_LANGUAGE_DUTCH)
"Specificeert indien koord stellen geval gevoelig of niet zouden moeten zijn"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Especifica se partidas de barbante devem ser caso sensvel ou no"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesifiserer om snorkamper er tilfelle sensitiv eller ikke"
#else
"Specifies if string matches should be case sensitive or not"
#endif
			;
		else if(w == bar->stop_btn)
			msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Especifica si el proceso de la bsqueda debe recurse en subdirectorios"
#elif defined(PROG_LANGUAGE_FRENCH)
"Spcifie si le procd de recherche devrait recurse dans subdirectories"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gibt an, wenn das suche verfahren recurse in unterverzeichnisse soll"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Specifica se il processo di ricerca dovrebbe il recurse nelle directory secondarie"
#elif defined(PROG_LANGUAGE_DUTCH)
"Specificeert indien het zoektocht proces recurse in subdirectories zal"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Especifica se o processo de busca devia recurse em subdirectories"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesifiserer om letingssom prosessen skal recurse inn i subdirectories"
#else
"Specifies if the search process should recurse into subdirectories"
#endif
			;
		break;
	}

	if(bar->status_message_cb != NULL)
		bar->status_message_cb(
			bar->toplevel,
			msg,
			bar->status_message_data
		);

	return(status);
}


/*
 *	Find by method changed callback.
 */
static void edv_find_bar_find_by_changed_cb(
	GtkWidget *widget, const gint i, gpointer data
)
{
	const gchar *tip;
	GtkWidget *w;
	GtkCombo *combo;
	EDVFindBar *bar = EDV_FIND_BAR(data);
	if((widget == NULL) || (bar == NULL))
		return;

	if(bar->freeze_count > 0)
		return;

	bar->freeze_count++;

	combo = GTK_COMBO(bar->search_combo);
	w = combo->entry;
	tip = NULL;
	switch(edv_find_bar_current_find_by(bar->toplevel))
	{
	  case EDV_FIND_BAR_FIND_BY_NAME:
		tip = "Enter a string describing the full or partial name\
 of the object(s) that you are looking for. Wildcards such as * and ? are\
 allowed.";
		break;

	  case EDV_FIND_BAR_FIND_BY_CONTENT:
		tip = "Enter a string of characters that you are looking for\
 inside an object(s).";
		break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_EQUAL_TO:
		tip = "Enter the size (in bytes) of the object(s) that you are\
 looking for.";
		break;
	  case EDV_FIND_BAR_FIND_BY_SIZE_NOT_EQUAL_TO:
		tip = "Enter the size (in bytes) of the object(s) that you are\
 not looking for.";
		break;
	  case EDV_FIND_BAR_FIND_BY_SIZE_LESS_THAN:
		tip = "Enter the size (in bytes) of the object(s) that you are\
 looking for that are smaller in size.";
		break;
	  case EDV_FIND_BAR_FIND_BY_SIZE_GREATER_THAN:
		tip = "Enter the size (in bytes) of the object(s) that you are\
 looking for that are bigger in size.";
		break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_EQUAL_TO:
		tip = "Enter the time (in the format of YYYY/MMM/DD [HH:MM:SS])\
 of the object(s) that you are looking for that were modified on.";
		break;
	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
		tip = "Enter the time (in the format of YYYY/MMM/DD [HH:MM:SS])\
 of the object(s) that you are looking for that were not modified on.";
		break;
	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_LESS_THAN:
		tip = "Enter the time (in the format of YYYY/MMM/DD [HH:MM:SS])\
 of the object(s) that you are looking for that were modified earlier than.";
		break;
	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_GREATER_THAN:
		tip = "Enter the time (in the format of YYYY/MMM/DD [HH:MM:SS])\
 of the object(s) that you are looking for that were modified later than.";
		break;
	}
	GUISetWidgetTip(w, tip);

	edv_find_bar_update_display(bar->toplevel);

	bar->freeze_count--;
}


/*
 *	Sets the search string.
 *
 *	The s specifies the new search string.
 *
 *	If record_history is TRUE then the current search string will
 *	be recorded on the search list history.
 */
void edv_find_bar_set_search(
	GtkWidget *w,
	const gchar *s,
	const gboolean record_history
)
{
	const gchar *cur_s;
	gchar *new_s;
	GtkCombo *combo;
	EDVFindBar *bar = EDVFindBarGetWidgetData(
		w,
		"edv_find_bar_set_search"
	);
	if((bar == NULL) || (s == NULL))
		return;

	combo = GTK_COMBO(bar->search_combo);

	/* Copy the search string */
	new_s = STRDUP(s);
	if(new_s == NULL)
		return;

	/* Check for no change in value */
	cur_s = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if(cur_s != NULL)
	{
		/* No change in value? */
		if(!strcmp((const char *)cur_s, (const char *)new_s))
		{
			g_free(new_s);
			return;
		}
	}

	/* Get record the current search string on to the search
	 * history list?
	 */
	if(record_history)
	{
		gchar *old_s = STRDUP(cur_s);
		if(old_s != NULL)
		{
			GUIComboAddItem(GTK_WIDGET(combo), old_s);
			g_free(old_s);
		}
	}

	/* Set the new search string */
	gtk_entry_set_text(GTK_ENTRY(combo->entry), new_s);

	g_free(new_s);
}


/*
 *	Gets the current find by method.
 */
EDVFindBarFindBy edv_find_bar_current_find_by(GtkWidget *w)
{
	EDVFindBar *bar = EDVFindBarGetWidgetData(
		w,
		"edv_find_bar_current_find_by"
	);
	if(bar == NULL)
		return(EDV_FIND_BAR_FIND_BY_NAME);

	return((EDVFindBarFindBy)PUListBoxGetSelectedData(
		bar->find_by_pulistbox
	));
}


/*
 *	Gets the EDVFindBar data from the GtkWidget.
 */
static EDVFindBar *EDVFindBarGetWidgetData(
	GtkWidget *w,
	const gchar *func_name
) 
{
	const gchar *key = EDV_FIND_BAR_KEY;
	EDVFindBar *bar;

	if(w == NULL)
		return(NULL);

	bar = EDV_FIND_BAR(gtk_object_get_data(
		GTK_OBJECT(w),
		key
	));
	if(bar == NULL)
	{
		g_printerr(
"%s(): Warning: GtkWidget %p:\
 Unable to find the data that matches the key \"%s\".\n",
			func_name,
			w,
			key
		);
		return(NULL);
	}

	return(bar);
}

/*
 *	Creates a new find bar.
 */
GtkWidget *edv_find_bar_new(
	EDVCore *core, 
	const EDVLocationType location_type,
	const gchar *(*get_location_cb)(
			GtkWidget *,                    /* Find Bar */
			gpointer                        /* get_location_data */
	),
	gpointer get_location_data,
	void (*start_cb)(
			GtkWidget *,                    /* Find Bar */
			gpointer                        /* start_data */
	),
	gpointer start_data,
	void (*end_cb)(
			GtkWidget *,                    /* Find Bar */
			const gint,                     /* Number of matches */
			gpointer                        /* end_data */
	),
	gpointer end_data,
	void (*match_cb)(
			GtkWidget *,                    /* Find Bar */
			const gchar *,                  /* Name/path of object */ 
			GList *,                        /* Properties list, a GList of
											 * EDVProperty * properties */
			const gint,                     /* Line index */
			const gchar *,                  /* Excerpt */
			gpointer                        /* match_data */
	),
	gpointer match_data,
	void (*status_message_cb)(
			GtkWidget *,                    /* Find Bar */
			const gchar *,                  /* Message */
			gpointer                        /* status_message_data */
	),
	gpointer status_message_data,
	void (*status_progress_cb)(
			GtkWidget *,                    /* Find Bar */
			const gfloat,                   /* Progress */
			gpointer                        /* status_progress_data */
	),
	gpointer status_progress_data
)
{
	const gint border_minor = 2;
	gint i;
	gpointer combo_rtn;
	GList *glist;
	GtkWidget	*w,
					*parent2, *parent3,
					*toplevel,
					*pulist;
	GtkCombo *combo;
	EDVFindBar *bar;

	if(core == NULL)
		return(NULL);

	bar = EDV_FIND_BAR(g_malloc0(sizeof(EDVFindBar)));
	if(bar == NULL)
		return(NULL);

	bar->toplevel = toplevel = gtk_hbox_new(FALSE, border_minor);
/*
	bar->processing = FALSE;
	bar->freeze_count = 0;
	bar->stop_count = 0;
 */
	bar->core = core;
	bar->location_type = location_type;
	bar->get_location_cb = get_location_cb;
	bar->get_location_data = get_location_data;
	bar->start_cb = start_cb;
	bar->start_data = start_data;
	bar->end_cb = end_cb;
	bar->end_data = end_data;
	bar->match_cb = match_cb;
	bar->match_data = match_data;
	bar->status_message_cb = status_message_cb;
	bar->status_message_data = status_message_data;
	bar->status_progress_cb = status_progress_cb;
	bar->status_progress_data = status_progress_data;

	bar->freeze_count++;

	/* Toplevel GtkHBox */
	w = toplevel;
	gtk_widget_set_name(w, EDV_WIDGET_NAME_FIND_BAR);
	gtk_object_set_data_full(
		GTK_OBJECT(w), EDV_FIND_BAR_KEY,
		bar, edv_find_bar_destroy_cb
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(edv_find_bar_realize_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "show",
		GTK_SIGNAL_FUNC(edv_find_bar_show_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "hide",
		GTK_SIGNAL_FUNC(edv_find_bar_hide_cb), bar
	);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	parent2 = w;

	/* Find Operation GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Buscar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn"
#else
"Find"
#endif
		":"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	bar->find_by_pulistbox = w = PUListBoxNew(
		170, -1
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	PUListBoxSetChangedCB(
		w,
		edv_find_bar_find_by_changed_cb, bar
	);
	pulist = PUListBoxGetPUList(w);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
		PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	i = PUListAddItem(pulist, "Name");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_NAME);
	i = PUListAddItem(pulist, "Content");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_CONTENT);
	i = PUListAddItem(pulist, "Size Equal To");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_EQUAL_TO);
	i = PUListAddItem(pulist, "Size Not Equal To");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_NOT_EQUAL_TO);
	i = PUListAddItem(pulist, "Size Less Than");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_LESS_THAN);
	i = PUListAddItem(pulist, "Size Greater Than");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_GREATER_THAN);
	i = PUListAddItem(pulist, "Date Modified On");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_EQUAL_TO);
	i = PUListAddItem(pulist, "Date Modified Not On");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO);
	i = PUListAddItem(pulist, "Date Modified Before");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_LESS_THAN);
	i = PUListAddItem(pulist, "Date Modified After");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_GREATER_THAN);
	PUListBoxSetLinesVisible(w, MIN((i + 1), 10));
	gtk_widget_show(w);

	/* Search string GtkCombo */
	glist = NULL;
	w = GUIComboCreate(
/*
#if defined(PROG_LANGUAGE_SPANISH)
"Palabra"
#elif defined(PROG_LANGUAGE_FRENCH)
"Egaler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Anpassend"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Uguagliare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Passend"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Combinar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Passende"
#else
"Matching"
#endif
		":",
 */
		NULL,
		"",			/* No initial value */
		glist,		/* List */
		25,			/* Max items */
		&combo_rtn,
		bar,
		edv_find_bar_find_cb,
		NULL
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)combo_rtn;
	bar->search_combo = w;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);

	w = combo->entry;
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	/* Case sensitive GtkCheckButton */
	bar->case_sensitive_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Sensible May/min."
#elif defined(PROG_LANGUAGE_FRENCH)
"Reconnatre Sensible"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fall Empfindlich"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Imballare Sensibile"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sluit Gevoelig In"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Caso Sensvel"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfelle Sensitiv"
#else
"Case Sensitive"
#endif
	);
	GTK_TOGGLE_BUTTON(w)->active = FALSE;
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_bar_crossing_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique esto para discriminar Mayusculas y minusculas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier ceci pour galer des ficelles reconnaissent sensiblement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie dies, schnren fall empfindlich anzupassen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare questo per uguagliare le cordicelle imballano sensibilmente"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer dit om bij koorden geval gevoelig te passen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique isto combinar caso de barbantes sensivelmente"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk dette passe snortilfelle sensitivt"
#else
"Check this to match strings case sensitively"
#endif
	);
	gtk_widget_show(w);

	w = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Stop button */
	bar->stop_btn = w = GUIButtonPixmap(
		(guint8 **)icon_stop_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edv_find_bar_stop_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "enter_notify_event",
		GTK_SIGNAL_FUNC(edv_find_bar_crossing_cb), bar
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "leave_notify_event",
		GTK_SIGNAL_FUNC(edv_find_bar_crossing_cb), bar
	);
	GUISetWidgetTip(
		w,
#if defined(PROG_LANGUAGE_SPANISH)
"Pare el procedimiento actual del hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrter la procdure actuelle de dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halten Sie das jetzige Fund Verfahren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermare la corrente trova la procedura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stop de huidig vondst procedure"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pare a corrente achar procedimento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans den nvrendee funnprosedyre"
#else
"Stop the current find procedure"
#endif
	);
	gtk_widget_show(w);


	bar->freeze_count--;

	/* Set the intiial values */
	edv_find_bar_find_by_changed_cb(
		bar->find_by_pulistbox,
		PUListBoxGetSelected(bar->find_by_pulistbox),
		bar
	);

	return(bar->toplevel);
}


/*
 *	Updates the Find Bar's widgets to reflect current values.
 */
void edv_find_bar_update_display(GtkWidget *w)
{
	gboolean sensitive, processing;
	EDVFindBarFindBy find_by;
	EDVCore *core;
	EDVFindBar *bar = EDVFindBarGetWidgetData(
		w,
		"edv_find_bar_update_display"
	);
	if(bar == NULL)
		return;

	processing = bar->processing;
	find_by = edv_find_bar_current_find_by(bar->toplevel);
	core = bar->core;

	/* Find By */
	sensitive = !processing;
	gtk_widget_set_sensitive(bar->find_by_pulistbox, sensitive);

	/* Search String */
	sensitive = !processing;
	gtk_widget_set_sensitive(bar->search_combo, sensitive);

	/* Case Sensitive */
	if(processing)
	{
		sensitive = FALSE;
	}
	else
	{
		switch(find_by)
		{
		  case EDV_FIND_BAR_FIND_BY_NAME:
		  case EDV_FIND_BAR_FIND_BY_CONTENT:
			sensitive = TRUE;
			break;
		  case EDV_FIND_BAR_FIND_BY_SIZE_EQUAL_TO:
		  case EDV_FIND_BAR_FIND_BY_SIZE_NOT_EQUAL_TO:
		  case EDV_FIND_BAR_FIND_BY_SIZE_LESS_THAN:
		  case EDV_FIND_BAR_FIND_BY_SIZE_GREATER_THAN:
		  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_EQUAL_TO:
		  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
		  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_LESS_THAN:
		  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_GREATER_THAN:
			sensitive = FALSE;
			break;
		}
	}
	gtk_widget_set_sensitive(bar->case_sensitive_check, sensitive);

	/* Stop */
	sensitive = processing;
	gtk_widget_set_sensitive(bar->stop_btn, sensitive);
}
