#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifndef HAVE_REGEX
# define HAVE_REGEX
#endif
#if defined(HAVE_REGEX)
# include <regex.h>
#endif
#include <fnmatch.h>
#include <sys/stat.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_convert_obj.h"
#include "edv_archive_obj_stat.h"
#include "edv_find.h"
#include "endeavour2.h"


/* Grep */
static gchar *edv_find_grep_excerpt_file(
	const gchar *path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint *line_num_rtn,
	const gint excerpt_left_pad,
	const gint excerpt_right_pad
);
static gchar *edv_find_grep_excerpt_file_regex(
	const gchar *path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint *line_num_rtn,
	const gint excerpt_left_pad,
	const gint excerpt_right_pad
);


/* Archive Object */
static void edv_find_archive_object_by_name_iterate(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
);
gint edv_find_archive_object_by_name(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_archive_object_by_size_iterate(
	EDVCore *core,
	const gchar *arch_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_archive_object_by_size(
	EDVCore *core,
	const gchar *arch_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_archive_object_by_modify_time_iterate(
	EDVCore *core,
	const gchar *arch_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_archive_object_by_modify_time(
	EDVCore *core,
	const gchar *arch_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);


/* Recycled Object */
static void edv_find_recycled_object_by_name_iterate(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_recycled_object_by_name(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_recycled_object_by_content_iterate(
	const gchar *index_path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
);
gint edv_find_recycled_object_by_content(
	const gchar *index_path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_recycled_object_by_size_iterate(
	const gchar *index_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_recycled_object_by_size(
	const gchar *index_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_recycled_object_by_modify_time_iterate(
	const gchar *index_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_recycled_object_by_modify_time(
	const gchar *index_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);


/* VFS Object */
static void edv_find_vfs_object_by_name_iterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_vfs_object_by_name(
	const gchar *start_path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_vfs_object_by_content_iterate(
	const gchar *path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_vfs_object_by_content(
	const gchar *start_path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_vfs_object_by_size_iterate(
	const gchar *path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_vfs_object_by_size(
	const gchar *start_path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);

static void edv_find_vfs_object_by_modify_time_iterate(
	const gchar *path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint edv_find_vfs_object_by_modify_time(
	const gchar *start_path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
);


/*
 *	Maximum number of characters to include around excerpts
 *	(before and after):
 */
#define EDV_FIND_EXCERPT_PAD_MAX	40


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Opens the file and searches for the expression in the file's
 *	contents using literal matching.
 *
 *	The path specifies the full path to the file to search.
 *
 *	The expression specifies the string describing the string to
 *	search for.
 *
 *	If case_sensitive is TRUE then case sensitive matching will
 *	be made.
 *
 *	The line_num_rtn specifies the line number return. If a match
 *	is made then *line_num_rtn will be a positive number
 *	describing the line that the match occured on.
 *
 *	The excerpt_left_pad and excerpt_right_pad specifies the
 *	number of characters before and after (respectively) to add
 *	to the return string.
 *
 *	If a match is made then a dynamically allocated string will be
 *	returned containing the excerpt of the matched line and
 *	*line_num_rtn will be set to the line that contained the match.
 */
static gchar *edv_find_grep_excerpt_file(
	const gchar *path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint *line_num_rtn,
	const gint excerpt_left_pad,
	const gint excerpt_right_pad
)
{
	FILE *fp;
	gint c;
	gulong line_start_pos;
	const gchar *expre_ptr;
	gchar *excerpt;

	*line_num_rtn = 0;

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(NULL);

	/* Set the expression to the beginning */
	expre_ptr = expression;

	line_start_pos = 0l;
	excerpt = NULL;

	/* Search for the expression in the file */
	while(!feof(fp))
	{
		/* Get the next character */
		c = (gint)fgetc(fp);
		if((int)c == EOF)
			break;

		/* If not performing case sensitive search then make the
		 * character that was just read uppercase
		 */
		if(!case_sensitive)
			c = (gint)toupper((int)c);

		/* If this character marks the end of the line then record
		 * the last line starting position as one character ahead
		 * of the current position
		 */
		if(ISCR(c))
		{
			/* Update the line number return */
			*line_num_rtn = (*line_num_rtn) + 1;

			/* Record the line start position */
			line_start_pos = (gulong)ftell(fp);
		}

		/* Does this character matches the current character of the
		 * expression?
		 */
		if(((gchar)c == *expre_ptr) && ((gchar)c != '\0'))
		{
			/* Increment to the next character in the expression */
			expre_ptr++;
		}
		/* Reset expression back to the beginning as needed */
		else if(expre_ptr != expression)
		{
			expre_ptr = expression;
		}

		/* Matched the entire expression?
		 *
		 * Test if the expression pointer has been incremented all
		 * the way to the end which suggests that the entire
		 * expression has been matched
		 */
		if(*expre_ptr == '\0')
		{
			gchar *line;

			/* Seek to the start of the line */
			if(fseek(fp, (long)line_start_pos, SEEK_SET))
				break;

			/* Get/copy the current line for matching and editing */
			line = (gchar *)FGetStringLiteral(fp);
			if(line != NULL)
			{
				/* Seek to the start of the expression in the line */
				gchar *exp_start = (case_sensitive) ?
					(gchar *)strstr((char *)line, (const char *)expression) :
					(gchar *)strcasestr((char *)line, (const char *)expression);
				if(exp_start != NULL)
				{
					const gint	exp_start_i = (gint)(exp_start - line),
									exp_end_i = exp_start_i + STRLEN(expression);
					gchar *sp;

					/* Get the padded start of the line */
					gchar *s_start = line + MAX(
						(exp_start_i - excerpt_left_pad),
						0
					);

					/* Get the padded end of the line (may be past
					 * the actual end of the line)
					 */
					gchar *s_end = line + (exp_end_i + excerpt_right_pad);

					/* Iterate from the padded start to the end of
					 * the line string or padded end (whichever
					 * comes first
					 */
					for(sp = s_start; *sp != '\0'; sp++)
					{
						/* At the padded end? */
						if(sp >= s_end)
						{
							/* Terminate the line string here and break */
							*sp = '\0';
							break;
						}

						/* Non-ACSII character? */
						if(!isascii((int)(*sp)))
							*sp = ' ';	/* Replace with space */
					}

					/* Copy the edited line to the excerpt return
					 * from the padded start
					 */
					g_free(excerpt);
					excerpt = STRDUP(s_start);
				}

				/* Delete the copy of the current line */
				g_free(line);
			}

			/* Stop searching after the first match */
			break;

		}	/* Matched the entire expression? */
	}

	/* Close the file */
	(void)fclose(fp);

	return(excerpt);
}


/*
 *	Same as edv_find_grep_excerpt_file() except that it uses regex
 */
static gchar *edv_find_grep_excerpt_file_regex(
	const gchar *path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint *line_num_rtn,
	const gint excerpt_left_pad,
	const gint excerpt_right_pad
)
{
#if defined(HAVE_REGEX)
	FILE *fp;
	regex_t *regex_filter;
	regmatch_t regex_matches_list[1];
	gint line_num;
	gchar *line, *excerpt;

	*line_num_rtn = 0;

	/* Compile the regex search criteria */
	regex_filter = (regex_t *)g_malloc(sizeof(regex_t));
	if(regex_filter != NULL)
	{
		if(regcomp(
			regex_filter,
			(const char *)expression,
#ifdef REG_EXTENDED
			REG_EXTENDED |		/* Use POSIX extended regex */
#endif
			(case_sensitive ? 0 : REG_ICASE)
		))
		{
			g_free(regex_filter);
			return(NULL);
		}
	}
	else
	{
		return(NULL);
	}

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	{
		regfree(regex_filter);
		g_free(regex_filter);
		return(NULL);
	}

	line_num = 0;
	excerpt = NULL;

	/* Search for the expression in the file */
	while(!feof(fp))
	{
		line = (gchar *)FGetStringLiteral(fp);
		if(line == NULL)
			break;

		line_num = line_num + 1;

		/* Check if the expression is found on this line */
		if(regexec(
			regex_filter,
			line,
			sizeof(regex_matches_list) / sizeof(regmatch_t),
			regex_matches_list,
			0
		) == 0)
		{
			/* Got a match */
			regmatch_t *m = &regex_matches_list[0];
			gchar *sp, *s_start, *s_end;

			/* Set the line number that this match occured on */
			*line_num_rtn = line_num;

			/* Get the padded start of the line */
			s_start = line + MAX(
				((gint)m->rm_so - excerpt_left_pad),
				0
			);

			/* Get the padded end of the line (may be past the
			 * actual end of the line)
			 */
			s_end = line + ((gint)m->rm_eo + excerpt_right_pad);

			/* Iterate from the padded start to the end of the
			 * line string or padded end (whichever comes first)
			 */
			for(sp = s_start; *sp != '\0'; sp++)
			{
				/* At the padded end? */
				if(sp >= s_end)
				{
					/* Terminate the line string here and break */
					*sp = '\0';
					break;
				}

				/* Non-ACSII character? */
				if(!isascii((int)(*sp)))
					*sp = ' ';		/* Replace with space */
			}

			/* Copy the edited line to the excerpt return from the
			 * padded start
			 */
			g_free(excerpt);
			excerpt = STRDUP(s_start);

			/* Stop searching after the first match */
			g_free(line);
			break;
		}

		/* Delete the current line */
		g_free(line);
	}

	/* Close the file */
	(void)fclose(fp);

	/* Decompile and delete the regex search criteria */
	regfree(regex_filter);
	g_free(regex_filter);

	return(excerpt);
#else
	*line_num_rtn = 0;
	return(NULL);
#endif
}


/*
 *	Called by edv_find_archive_object_by_name().
 */
static void edv_find_archive_object_by_name_iterate(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
)
{
	gint		i,
			nobjs;
	GList		*glist,
			*obj_list;
	EDVArchiveObject *obj;

	if(*stop_find)
		return;

	/* Get the list of all the objects in the archive */
	obj_list = edv_archive_object_stat_list(
		core->cfg_list,
		arch_path,
		NULL,				/* Get all the objects */
		NULL,				/* No filter */
		NULL,				/* No password */
		NULL, NULL				/* No progress callback */
	);

	/* Iterate through all the objects in the archive */
	nobjs = g_list_length(obj_list);
	for(glist = obj_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		/* User aborted? */
		if(*stop_find)
			break;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				obj->name,
				(gfloat)i / (gfloat)nobjs,
				progress_data
			))
			{
				*stop_find = TRUE;
				continue;
			}
		}

		/* Match name */
		if(!STRISEMPTY(obj->name))
		{
			gchar *s = STRDUP(obj->name);
			if(!case_sensitive)
				g_strup(s);

			/* Names match? */
			if(!fnmatch((const char *)expression, (const char *)s, 0))
			{
				gchar *path = STRDUP(obj->path);

				/* Count this match */
				*nmatches = (*nmatches) + 1;

				/* Report this match */
				if(match_cb != NULL)
				{
					GList *properties_list = NULL;
					properties_list = edv_convert_archive_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						path,		/* Path */
						properties_list,	/* Properties list */
						match_data		/* Data */
					);
					(void)edv_properties_list_delete(properties_list);
				}
				g_free(path);
			}

			g_free(s);
		}
	}

	/* Delete the archive object stats */
	if(obj_list != NULL)
	{
		g_list_foreach(
			obj_list,
			(GFunc)edv_archive_object_delete,
			NULL
		);
		g_list_free(obj_list);
	}
}

/*
 *	Find an archive object by name.
 *
 *	The arch_path specifies the full path to the archive.
 *
 *	The expression specifies the string describing the object's
 *	name to find. Wildcards are allowed.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_archive_object_by_name(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *dexpression;

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(expression)
	)
		return(0);

	if(!strcmp((const char *)expression, "*"))
		return(0);

	/* Convert expression to upper case if not performing case
	 * sensitive matching
	 */
	dexpression = g_strdup(expression);
	if(!case_sensitive)
		g_strup(dexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_archive_object_by_name_iterate(
		core,
		arch_path,
		dexpression,
		case_sensitive,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	g_free(dexpression);

	return(nmatches);
}


/*
 *	Called by edv_find_archive_object_by_size().
 */
static void edv_find_archive_object_by_size_iterate(
	EDVCore *core,
	const gchar *arch_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint		i,
			nobjs;
	GList		*glist,
			*obj_list;
	EDVArchiveObject *obj;

	if(*stop_find)
		return;

	/* Get the list of all the objects in the archive */
	obj_list = edv_archive_object_stat_list(
		core->cfg_list,
		arch_path,
		NULL,				/* Get all the objects */
		NULL,				/* No filter */
		NULL,				/* No password */
		NULL, NULL			/* No progress callback */
	);

	/* Iterate through all the objects in the archive */
	nobjs = g_list_length(obj_list);
	for(glist = obj_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		/* User aborted? */
		if(*stop_find)
			break;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				obj->name,
				(gfloat)i / (gfloat)nobjs,
				progress_data
			))
			{
				*stop_find = TRUE;
				continue;
			}
		}

		/* Match size */
		if(TRUE)
		{
			gboolean got_match = FALSE;

			/* Match the size by the size relativity */
			switch(relativity)
			{
			  case EDV_FIND_EQUAL_TO:
				if(size == obj->size)
					got_match = TRUE;
				break;

			  case EDV_FIND_NOT_EQUAL_TO:
				if(size != obj->size)
					got_match = TRUE;
				break;

			  case EDV_FIND_LESS_THAN:
				if(size >= obj->size)
					got_match = TRUE;
				break;

			  case EDV_FIND_GREATER_THAN:
				if(size <= obj->size)
					got_match = TRUE;
				break;
			}
			if(got_match)
			{
				gchar *path = STRDUP(obj->path);

				/* Count this match */
				*nmatches = (*nmatches) + 1;

				/* Report this match */
				if(match_cb != NULL)
				{
					GList *properties_list = NULL;
					properties_list = edv_convert_archive_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						path,		/* Path */
						properties_list,	/* Properties list */
						match_data	/* Data */
					);
					(void)edv_properties_list_delete(properties_list);
				}

				g_free(path);
			}
		}
	}

	/* Delete the archive object stats */
	if(obj_list != NULL)
	{
		g_list_foreach(
			obj_list,
			(GFunc)edv_archive_object_delete,
			NULL
		);
		g_list_free(obj_list);
	}
}

/*
 *	Find an archive object by size.
 *
 *	The arch_path specifies the full path to the archive.
 *
 *	The size specifies the size in bytes to find.
 *
 *	The relativity specifies the size relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_archive_object_by_size(
	EDVCore *core,
	const gchar *arch_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(arch_path))
		return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_archive_object_by_size_iterate(
		core,
		arch_path,
		size,
		relativity,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	return(nmatches);
}


/*
 *	Called by edv_find_archive_object_by_modify_time().
 */
static void edv_find_archive_object_by_modify_time_iterate(
	EDVCore *core,
	const gchar *arch_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint		i,
			nobjs;
	GList		*glist,
			*obj_list;
	EDVArchiveObject *obj;

	if(*stop_find)
		return;

	/* Get the list of all the objects in the archive */
	obj_list = edv_archive_object_stat_list(
		core->cfg_list,
		arch_path,
		NULL,				/* Get all the objects */
		NULL,				/* No filter */
		NULL,				/* No password */
		NULL, NULL			/* No progress callback */
	);

	/* Iterate through all the objects in the archive */
	nobjs = g_list_length(obj_list);
	for(glist = obj_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		/* User aborted? */
		if(*stop_find)
			break;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				obj->name,
				(gfloat)i / (gfloat)nobjs,
				progress_data
			))
			{
				*stop_find = TRUE;
				continue;
			}
		}

		/* Match modify time */
		if(TRUE)
		{
			gboolean got_match = FALSE;

			/* Match the size by the size relativity */
			switch(relativity)
			{
			  case EDV_FIND_EQUAL_TO:
				if(t == obj->modify_time)
					got_match = TRUE;
				break;

			  case EDV_FIND_NOT_EQUAL_TO:
				if(t != obj->modify_time)
					got_match = TRUE;
				break;

			  case EDV_FIND_LESS_THAN:
				if(t >= obj->modify_time)
					got_match = TRUE;
				break;

			  case EDV_FIND_GREATER_THAN:
				if(t <= obj->modify_time)
					got_match = TRUE;
				break;
			}
			if(got_match)
			{
				gchar *path = STRDUP(obj->path);

				/* Count this match */
				*nmatches = (*nmatches) + 1;

				/* Report this match */
				if(match_cb != NULL)
				{
					GList *properties_list = NULL;
					properties_list = edv_convert_archive_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						path,		/* Path */
						properties_list,/* Properties list */
						match_data	/* Data */
					);
					(void)edv_properties_list_delete(properties_list);
				}

				g_free(path);
			}
		}
	}

	/* Delete the archive object stats */
	if(obj_list != NULL)
	{
		g_list_foreach(
			obj_list,
			(GFunc)edv_archive_object_delete,
			NULL
		);
		g_list_free(obj_list);
	}
}

/*
 *	Find an archive object by modify time.
 *
 *	The arch_path specifies the full path to the archive.
 *
 *	The t specifies the time in seconds since EPOCH to find.
 *
 *	The relativity specifies the time relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_archive_object_by_modify_time(
	EDVCore *core,
	const gchar *arch_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path of object in archive */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path of object in archive */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(arch_path))
		return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_archive_object_by_modify_time_iterate(
		core,
		arch_path,
		t,
		relativity,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	return(nmatches);
}


/*
 *	Called by edv_find_recycled_object_by_name().
 */
static void edv_find_recycled_object_by_name_iterate(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint		i,
			nobjs;
	gchar *recbin_path;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;

	if(*stop_find)
		return;

	/* Get the recycle bin directory */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);

	/* Get the total number of recycled objects */
	nobjs = edv_recycle_bin_index_get_total(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rp = edv_recycle_bin_index_open(index_path);
	for(obj = edv_recycle_bin_index_next(rp);
	    obj != NULL;
	    obj = edv_recycle_bin_index_next(rp)
	)
	{
		/* User aborted? */
		if(*stop_find)
			break;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				obj->name,
				(gfloat)i / (gfloat)nobjs,
				progress_data
			))
			{
				*stop_find = TRUE;
				i++;
				continue;
			}
		}

		/* Match name */
		if(!STRISEMPTY(obj->name))
		{
			gchar *s = STRDUP(obj->name);
			if(!case_sensitive)
				g_strup(s);

			/* Names match? */
			if(!fnmatch((const char *)expression, (const char *)s, 0))
			{
				/* Count this match */
				*nmatches = (*nmatches) + 1;

				/* Report this match */
				if(match_cb != NULL)
				{
					gchar *name = STRDUP(obj->name);
					GList *properties_list = NULL;
					properties_list = edv_convert_recycled_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						name,	/* Recycled object's name */
						properties_list,	/* Properties list */
						match_data	/* Data */
					);
					g_free(name);
					(void)edv_properties_list_delete(properties_list);
				}
			}

			g_free(s);
		}
	}

	edv_recycle_bin_index_close(rp);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by name.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The expression specifies the string describing the object's
 *	name to find. Wildcards are allowed.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_recycled_object_by_name(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *dexpression;

	if(STRISEMPTY(index_path) || STRISEMPTY(expression))
		return(0);
	if(!strcmp((const char *)expression, "*"))
		return(0);

	/* Convert expression to upper case if not matching case
	 * sensitive
	 */
	dexpression = g_strdup(expression);
	if(!case_sensitive)
		g_strup(dexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_recycled_object_by_name_iterate(
		index_path,
		dexpression,
		case_sensitive,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	g_free(dexpression);

	return(nmatches);
}


/*
 *	Called by edv_find_recycled_object_by_content() to perform one find
 *	iteration on the given path.
 */
static void edv_find_recycled_object_by_content_iterate(
	const gchar *index_path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
)
{
	gint		i,
			nobjs;
	gchar *recbin_path;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;

	if(*stop_find)
		return;

	/* Get the recycle bin directory */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);

	/* Get the total number of recycled objects */
	nobjs = edv_recycle_bin_index_get_total(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rp = edv_recycle_bin_index_open(index_path);
	for(obj = edv_recycle_bin_index_next(rp);
	    obj != NULL;
	    obj = edv_recycle_bin_index_next(rp)
	)
	{

		/* User aborted? */
		if(*stop_find)
			break;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				obj->name,
				(gfloat)i / (gfloat)nobjs,
				progress_data
			))
			{
				*stop_find = TRUE;
				i++;
				continue;
			}
		}

		/* Match content */
		if(TRUE)
		{
			gint line_num;
			gchar	*excerpt = NULL,
					*full_path = edv_recycle_bin_index_get_recycled_object_path(
				index_path,
				obj->index
			);

			/* Contents match? */
			switch(using)
			{
			  case EDV_FIND_STRING_MATCH_LITERAL:
				excerpt = edv_find_grep_excerpt_file(
					full_path,
					expression,
					case_sensitive,
					&line_num,
					EDV_FIND_EXCERPT_PAD_MAX,	/* Left excerpt pad */
					EDV_FIND_EXCERPT_PAD_MAX	/* Right excerpt pad */
				);
				break;
			  case EDV_FIND_STRING_MATCH_REGEX:
				excerpt = edv_find_grep_excerpt_file_regex(
					full_path,
					expression,
					case_sensitive,
					&line_num,
					EDV_FIND_EXCERPT_PAD_MAX,	/* Left excerpt pad */
					EDV_FIND_EXCERPT_PAD_MAX	/* Right excerpt pad */
				);
				break;
			}

			/* Got a match? */
			if(excerpt != NULL)
			{
				/* Count this match */
				*nmatches = (*nmatches) + 1;

				/* Report this match */
				if(match_cb != NULL)
				{
					gchar *name = STRDUP(obj->name);
					GList *properties_list = NULL;
					properties_list = edv_convert_recycled_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						name,	/* Recycled object's name */
						properties_list,	/* Properties list */
						line_num,	/* Line number */
						excerpt,	/* Excerpt */
						match_data	/* Data */
					);
					g_free(name);
					(void)edv_properties_list_delete(properties_list);
				}

				g_free(excerpt);
			}

			g_free(full_path);
		}
	}

	edv_recycle_bin_index_close(rp);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by content.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The expression specifies the string to find within the
 *	object's contents.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_recycled_object_by_content(
	const gchar *index_path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *dexpression;

	if(STRISEMPTY(index_path) || STRISEMPTY(expression))
		return(0);

	/* Convert expression to upper case if not performing case
	 * sensitive matching
	 */
	dexpression = g_strdup(expression);
	if(!case_sensitive)
		g_strup(dexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_recycled_object_by_content_iterate(
		index_path,
		dexpression,
		using,
		case_sensitive,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	g_free(dexpression);

	return(nmatches);
}


/*
 *	Called by edv_find_recycled_object_by_size().
 */
static void edv_find_recycled_object_by_size_iterate(
	const gchar *index_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint		i,
			nobjs;
	gchar *recbin_path;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;

	if(*stop_find)
		return;

	/* Get the recycle bin directory */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);

	/* Get the total number of recycled objects */
	nobjs = edv_recycle_bin_index_get_total(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rp = edv_recycle_bin_index_open(index_path);
	for(obj = edv_recycle_bin_index_next(rp);
		obj != NULL;
		obj = edv_recycle_bin_index_next(rp)
	)
	{
		/* User aborted? */
		if(*stop_find)
			break;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				obj->name,
				(gfloat)i / (gfloat)nobjs,
				progress_data
			))
			{
				*stop_find = TRUE;
				i++;
				continue;
			}
		}

		/* Match size */
		if(TRUE)
		{
			gboolean got_match = FALSE;

			/* Match the size by the size relativity */
			switch(relativity)
			{
			  case EDV_FIND_EQUAL_TO:
				if(size == obj->size)
					got_match = TRUE;
				break;

			  case EDV_FIND_NOT_EQUAL_TO:
				if(size != obj->size)
					got_match = TRUE;
				break;

			  case EDV_FIND_LESS_THAN:
				if(size >= obj->size)
					got_match = TRUE;
				break;

			  case EDV_FIND_GREATER_THAN:
				if(size <= obj->size)
					got_match = TRUE;
				break;
			}
			if(got_match)
			{
				/* Count this match */
				*nmatches = (*nmatches) + 1;

				/* Report this match */
				if(match_cb != NULL)
				{
					gchar *name = STRDUP(obj->name);
					GList *properties_list = NULL;
					properties_list = edv_convert_recycled_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						name,		/* Recycled object's name */
						properties_list,	/* Properties list */
						match_data		/* Data */
					);
					g_free(name);
					(void)edv_properties_list_delete(properties_list);
				}
			}
		}
	}

	edv_recycle_bin_index_close(rp);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by size.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The size specifies the size in bytes to find.
 *
 *	The relativity specifies the size relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_recycled_object_by_size(
	const gchar *index_path,
	const gulong size,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(index_path))
		return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_recycled_object_by_size_iterate(
		index_path,
		size,
		relativity,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	return(nmatches);
}


/*
 *	Called by edv_find_recycled_object_by_modify_time().
 */
static void edv_find_recycled_object_by_modify_time_iterate(
	const gchar *index_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint		i,
			nobjs;
	gchar *recbin_path;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;

	if(*stop_find)
		return;

	/* Get the recycle bin directory */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);

	/* Get the total number of recycled objects */
	nobjs = edv_recycle_bin_index_get_total(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rp = edv_recycle_bin_index_open(index_path);
	for(obj = edv_recycle_bin_index_next(rp);
		obj != NULL;
		obj = edv_recycle_bin_index_next(rp)
	)
	{
		/* User aborted? */
		if(*stop_find)
			break;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				obj->name,
				(gfloat)i / (gfloat)nobjs,
				progress_data
			))
			{
				*stop_find = TRUE;
				i++;
				continue;
			}
		}

		/* Match modify time */
		if(TRUE)
		{
			gboolean got_match = FALSE;

			/* Match the modify time by the time relativity */
			switch(relativity)
			{
			  case EDV_FIND_EQUAL_TO:
				if(t == obj->modify_time)
					got_match = TRUE;
				break;

			  case EDV_FIND_NOT_EQUAL_TO:
				if(t != obj->modify_time)
					got_match = TRUE;
				break;

			  case EDV_FIND_LESS_THAN:
				if(t >= obj->modify_time)
					got_match = TRUE;
				break;

			  case EDV_FIND_GREATER_THAN:
				if(t <= obj->modify_time)
					got_match = TRUE;
				break;
			}
			if(got_match)
			{
				/* Count this match */
				*nmatches = (*nmatches) + 1;

				/* Report this match */
				if(match_cb != NULL)
				{
					gchar *name = STRDUP(obj->name);
					GList *properties_list = NULL;
					properties_list = edv_convert_recycled_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						name,	/* Recycled object's name */
						properties_list,	/* Properties list */
						match_data	/* Data */
					);
					g_free(name);
					(void)edv_properties_list_delete(properties_list);
				}
			}
		}
	}

	edv_recycle_bin_index_close(rp);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by modify time.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The t specifies the modify time in seconds since EPOCH to find.
 *
 *	The relativity specifies the time relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_recycled_object_by_modify_time(
	const gchar *index_path,
	const gulong t,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Name of recycled object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Name of recycled object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(index_path))
		return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_recycled_object_by_modify_time_iterate(
		index_path,
		t,
		relativity,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	return(nmatches);
}


/*
 *	Called by edv_find_vfs_object_by_name().
 */
static void edv_find_vfs_object_by_name_iterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	GList *names_list;

	if(*stop_find)
		return;

	/* Search through the objects in this directory */
	names_list = edv_directory_list(
		path,
		TRUE,				/* Sorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gint nobjs = g_list_length(names_list);
		gint i;
		const gchar *name;
		gchar	*s,
			*full_path;
		GList *glist;
		EDVVFSObject *obj;

		for(glist = names_list, i = 0;
		    glist != NULL;
		    glist = g_list_next(glist), i++
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Report progress */
			if(progress_cb != NULL)
			{
				if(progress_cb(
					full_path,
					recursive ? -1.0f : ((gfloat)i / (gfloat)nobjs),
					progress_data
				))
				{
					g_free(full_path);
					*stop_find = TRUE;
					break;
				}
			}

			/* Get this object's statistics */
			if(follow_links)
				obj = edv_vfs_object_stat(full_path);
			else
				obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Get/copy the object's name and make it upper case
			 * if not matching case sensitive
			 */
			s = STRDUP(name);
			if(!case_sensitive)
				g_strup(s);

			/* Match the object's name */
			if(!fnmatch((const char *)expression, (const char *)s, 0))
			{
				*nmatches = (*nmatches) + 1;
				if(match_cb != NULL)
				{
					GList *properties_list = NULL;
					properties_list = edv_convert_vfs_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						full_path,	/* Path */
						properties_list,	/* Properties list */
						match_data	/* Data */
					);
					(void)edv_properties_list_delete(properties_list);
				}
			}

			g_free(s);

			/* Is it a directory and should we recurse into it? */
			if(recursive && EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				edv_find_vfs_object_by_name_iterate(
					full_path,
					expression,
					recursive,
					follow_links,
					case_sensitive,
					progress_cb, progress_data,
					match_cb, match_data,
					nmatches,
					stop_find
				);
			}

			edv_vfs_object_delete(obj);
			g_free(full_path);

			if(*stop_find)
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}

		g_list_free(names_list);
	}
}

/*
 *	Find an object by name.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The expression specifies the string describing the object's
 *	name to find. Wildcards are allowed.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_vfs_object_by_name(
	const gchar *start_path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *dexpression;

	if(STRISEMPTY(start_path) || STRISEMPTY(expression))
		return(0);
	if(!strcmp((const char *)expression, "*"))
		return(0);

	/* Convert the expression to upper case if not matching case
	 * sensitive
	 */
	dexpression = g_strdup(expression);
	if(!case_sensitive)
		g_strup(dexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_vfs_object_by_name_iterate(
		start_path,
		dexpression,
		recursive,
		follow_links,
		case_sensitive,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	g_free(dexpression);

	return(nmatches);
}


/*
 *	Called by edv_find_vfs_object_by_content().
 */
static void edv_find_vfs_object_by_content_iterate(
	const gchar *path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	GList *names_list;

	if(*stop_find)
		return;

	/* Search through the objects in this directory */
	names_list = edv_directory_list(
		path,
		TRUE,				/* Sorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gint nobjs = g_list_length(names_list);
		gint	i,
			line_index;
		const gchar *name;
		gchar	*excerpt,
			*full_path;
		GList *glist;
		EDVVFSObject *obj;

		for(glist = names_list, i = 0;
		    glist != NULL;
		    glist = g_list_next(glist), i++
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Report progress */
			if(progress_cb != NULL)
			{
				if(progress_cb(
					full_path,
					recursive ? -1.0f : ((gfloat)i / (gfloat)nobjs),
					progress_data
				))
				{
					g_free(full_path);
					*stop_find = TRUE;
					break;
				}
			}

			/* Get this object's statistics */
			if(follow_links)
				obj = edv_vfs_object_stat(full_path);
			else
				obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Match the object's contents */
			line_index = 0;
			excerpt = NULL;
			switch(using)
			{
			  case EDV_FIND_STRING_MATCH_LITERAL:
				excerpt = edv_find_grep_excerpt_file(
					full_path,
					expression,
					case_sensitive,
					&line_index,
					EDV_FIND_EXCERPT_PAD_MAX,	/* Left excerpt pad */
					EDV_FIND_EXCERPT_PAD_MAX	/* Right excerpt pad */
				);
				break;
			  case EDV_FIND_STRING_MATCH_REGEX:
				excerpt = edv_find_grep_excerpt_file_regex(
					full_path,
					expression,
					case_sensitive,
					&line_index,
					EDV_FIND_EXCERPT_PAD_MAX,	/* Left excerpt pad */
					EDV_FIND_EXCERPT_PAD_MAX	/* Right excerpt pad */
				);
				break;
			}

			/* Got match? */
			if(excerpt != NULL)
			{
				*nmatches = (*nmatches) + 1;
				if(match_cb != NULL)
				{
					GList *properties_list = NULL;
					properties_list = edv_convert_vfs_object_to_properties_list(
						properties_list,
						obj
					);
					match_cb(
						full_path,		/* Path */
						properties_list,	/* Properties list */
						line_index,		/* Line index */
						excerpt,		/* Excerpt */
						match_data		/* Data */
					);
					(void)edv_properties_list_delete(properties_list);
				}

				g_free(excerpt);
			}

			/* Is it a directory and should we recurse into it? */
			if(recursive && EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				edv_find_vfs_object_by_content_iterate(
					full_path,
					expression,
					using,
					recursive,
					follow_links,
					case_sensitive,
					progress_cb, progress_data,
					match_cb, match_data,
					nmatches,
					stop_find
				);
			}

			edv_vfs_object_delete(obj);
			g_free(full_path);

			if(*stop_find)
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}

		g_list_free(names_list);
	}
}

/*
 *	Find an object by its content.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The expression specifies the string to find within the
 *	object's contents.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_vfs_object_by_content(
	const gchar *start_path,
	const gchar *expression,
	const EDVFindStringMatch using,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		const gint,			/* Line index */
		const gchar *,			/* Excerpt */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *dexpression;

	if(STRISEMPTY(start_path) || STRISEMPTY(expression))
		return(0);

	/* Convert the expression to upper case if not matching
	 * case sensitive
	 */
	dexpression = g_strdup(expression);
	if(!case_sensitive)
		g_strup(dexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_vfs_object_by_content_iterate(
		start_path,
		dexpression,
		using,
		recursive,
		follow_links,
		case_sensitive,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	g_free(dexpression);

	return(nmatches);
}


/*
 *	Called by edv_find_vfs_object_by_size().
 */
static void edv_find_vfs_object_by_size_iterate(
	const gchar *path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	GList *names_list;

	if(*stop_find)
		return;

	/* Search through the objects in this directory */
	names_list = edv_directory_list(
		path,
		TRUE,				/* Sorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gint nobjs = g_list_length(names_list);
		gint i;
		const gchar *name;
		gchar *full_path;
		GList *glist;
		EDVVFSObject *obj;

		for(glist = names_list, i = 0;
		    glist != NULL;
		    glist = g_list_next(glist), i++
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Report progress */
			if(progress_cb != NULL)
			{
				if(progress_cb(
					full_path,
					recursive ? -1.0f : ((gfloat)i / (gfloat)nobjs),
					progress_data
				))
				{
					g_free(full_path);
					*stop_find = TRUE;
					break;
				}
			}

			/* Get this object's statistics */
			if(follow_links)
				obj = edv_vfs_object_stat(full_path);
			else
				obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Match the size by the size relativity */
			switch(relativity)
			{
			  case EDV_FIND_EQUAL_TO:
				if(size == obj->size)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;

			  case EDV_FIND_NOT_EQUAL_TO:
				if(size != obj->size)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;

			  case EDV_FIND_LESS_THAN:
				if(size >= obj->size)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;

			  case EDV_FIND_GREATER_THAN:
				if(size <= obj->size)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;
			}

			/* Is it a directory and should we recurse into it? */
			if(recursive && EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				edv_find_vfs_object_by_size_iterate(
					full_path,
					size,
					recursive,
					follow_links,
					relativity,
					progress_cb, progress_data,
					match_cb, match_data,
					nmatches,
					stop_find
				);
			}

			edv_vfs_object_delete(obj);
			g_free(full_path);

			if(*stop_find)
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}

		g_list_free(names_list);
	}
}

/*
 *	Find an object by size.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The size specifies the size in bytes to find.
 *
 *	The relativity specifies the size relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_vfs_object_by_size(
	const gchar *start_path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(start_path))
		return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_vfs_object_by_size_iterate(
		start_path,
		size,
		recursive,
		follow_links,
		relativity,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	return(nmatches);
}


/*
 *	Called by edv_find_vfs_object_by_modify_time().
 */
static void edv_find_vfs_object_by_modify_time_iterate(
	const gchar *path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	GList *names_list;

	if(*stop_find)
		return;

	/* Search through the objects in this directory */
	names_list = edv_directory_list(
		path,
		TRUE,				/* Sorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gint nobjs = g_list_length(names_list);
		gint i;
		const gchar *name;
		gchar *full_path;
		GList *glist;
		EDVVFSObject *obj;

		for(glist = names_list, i = 0;
			glist != NULL;
			glist = g_list_next(glist), i++
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			/* Get this object's full path */
			full_path = edv_paths_join(
				path,
				name
			);
			if(full_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Report progress */
			if(progress_cb != NULL)
			{
				if(progress_cb(
					full_path,
					recursive ? -1.0f : ((gfloat)i / (gfloat)nobjs),
					progress_data
				))
				{
					g_free(full_path);
					*stop_find = TRUE;
					break;
				}
			}

			/* Get this object's statistics */
			if(follow_links)
				obj = edv_vfs_object_stat(full_path);
			else
				obj = edv_vfs_object_lstat(full_path);
			if(obj == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Match the modify time by the relativity */
			switch(relativity)
			{
			  case EDV_FIND_EQUAL_TO:
				if(t == obj->modify_time)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;

			  case EDV_FIND_NOT_EQUAL_TO:
				if(t != obj->modify_time)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;

			  case EDV_FIND_LESS_THAN:
				if(t >= obj->modify_time)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;

			  case EDV_FIND_GREATER_THAN:
				if(t <= obj->modify_time)
				{
					*nmatches = (*nmatches) + 1;
					if(match_cb != NULL)
				    {
						GList *properties_list = NULL;
						properties_list = edv_convert_vfs_object_to_properties_list(
							properties_list,
							obj
						);
						match_cb(
							full_path,	/* Path */
							properties_list,	/* Properties list */
							match_data	/* Data */
						);
						(void)edv_properties_list_delete(properties_list);
					}
				}
				break;
			}

			/* Is it a directory and should we recurse into it? */
			if(recursive && EDV_VFS_OBJECT_IS_DIRECTORY(obj))
			{
				edv_find_vfs_object_by_modify_time_iterate(
					full_path,
					t,
					recursive,
					follow_links,
					relativity,
					progress_cb, progress_data,
					match_cb, match_data,
					nmatches,
					stop_find
				);
			}

			edv_vfs_object_delete(obj);
			g_free(full_path);

			if(*stop_find)
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}

		g_list_free(names_list);
	}
}

/*
 *	Find an object by modify time.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The t specifies the modify time in seconds since EPOCH to find.
 *
 *	The relativity specifies the time relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint edv_find_vfs_object_by_modify_time(
	const gchar *start_path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const EDVFindRelativity relativity,
	gint (*progress_cb)(
		const gchar *,			/* Path to VFS object */
		const gfloat,			/* Progress from 0.0 to 1.0 */
		gpointer			/* progress_data */
	),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *,			/* Path to VFS object */
		GList *,			/* Properties list, a GList of
						 * EDVProperty * properties */
		gpointer			/* match_data */
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(start_path))
		return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	edv_find_vfs_object_by_modify_time_iterate(
		start_path,
		t,
		recursive,
		follow_links,
		relativity,
		progress_cb, progress_data,
		match_cb, match_data,
		&nmatches,
		&stop_find
	);

	return(nmatches);
}
