/*
                       Endeavour Signal Emitters

	Endeavour Signals notify Endeavour's resources of when
	something has changed and to take action or update values
	based on that change.

	For example, the edv_emit_vfs_object_added() function emits a
	signal to all of Endeavour's windows who need to know of when
	a new VFS object has been added. The File Browser would be
	such a window that receives this signal and it would update
	its listing to show the new VFS object being added.

	All edv_emit_*() functions emit a signal as they are called,
	while edv_queue_emit*() functions will use the gtk_idle_add()
	function to add a GTK Idle callback to call its corresponding
	EDVEmit*() function when GTK is idle. The EDVQueueEmit*()
	functions are useful when you need its corresponding
	EDVEmit*() function called on at a lower stack or when the
	same signal maybe called multiple times and become redundant.

 */

#ifndef EDV_EMIT_H
#define EDV_EMIT_H

#include <gtk/gtk.h>

#include "edv_types.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "edv_mime_type.h"
#include "edv_device.h"
#include "edv_history.h"


/* Window Created/Deleted */
extern void edv_emit_window_created(
        EDVCore *core,
        const EDVWindowType win_type,
        gpointer win
);
extern void edv_emit_window_deleted(
        EDVCore *core,
        const EDVWindowType win_type,
        gpointer win
);

/* Reconfigured */
extern void edv_emit_reconfigured(EDVCore *core);
extern void edv_queue_emit_reconfigured(EDVCore *core);

/* Master Write Protect Changed */
extern void edv_emit_master_write_protect_changed(EDVCore *core);
extern void edv_queue_emit_master_write_protect_changed(EDVCore *core);

/* Delete Method Changed */
extern void edv_emit_delete_method_changed(EDVCore *core);
extern void edv_queue_emit_delete_method_changed(EDVCore *core);

/* MIME Type Added/Mdoified/Removed */
extern void edv_emit_mime_type_added(
	EDVCore *core,
	const gint mt_num, EDVMIMEType *mt
);
extern void edv_emit_mime_type_modified(
	EDVCore *core,
	const gint mt_num, EDVMIMEType *mt
);
extern void edv_emit_mime_type_removed(
	EDVCore *core,
	const gint mt_num
);

/* Device Added/Modified/Removed */
extern void edv_emit_device_added(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev
);
extern void edv_emit_device_modified(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev
);
extern void edv_emit_device_removed(
	EDVCore *core,
	const gint dev_num
);

/* Device Mount/Unmount */
extern void edv_emit_device_mount(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev,
	const gboolean is_mounted
);

/* History Event Added/Removed */
extern void edv_emit_history_added(
	EDVCore *core,
	const gint h_num, EDVHistory *h
);
extern void edv_emit_history_removed(
	EDVCore *core,
	const gint h_num
);

/* VFS Object Added/Modified/Removed */
extern void edv_emit_vfs_object_added(
	EDVCore *core,
	const gchar *path,
	EDVVFSObject *obj
);
extern void edv_emit_vfs_object_modified(
	EDVCore *core,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
extern void edv_emit_vfs_object_removed(
	EDVCore *core,
	const gchar *path
);

/* Recycled Object Added/Modified/Removed */
extern void edv_emit_recycled_object_added(
	EDVCore *core,
	const guint index
);
extern void edv_emit_recycled_object_modified(
	EDVCore *core,
	const guint index
);
extern void edv_emit_recycled_object_removed(
	EDVCore *core,
	const guint index
);

/* Archive Object Added/Modified/Removed */
extern void edv_emit_archive_object_added(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path,
	EDVArchiveObject *obj
);
extern void edv_emit_archive_object_modified(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path,
	const gchar *new_path,
	EDVArchiveObject *obj
);
extern void edv_emit_archive_object_removed(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path
);


#endif	/* EDV_EMIT_H */
