#include <gtk/gtk.h>

#include "cfg.h"

#include "guiutils.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "edv_mime_type.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_utils_gtk.h"
#include "vfs_browser.h"
#include "vfs_browser_cb.h"
#include "image_browser.h"
#include "image_browser_cb.h"
#include "archiver.h"
#include "archiver_cb.h"
#include "recycle_bin.h"
#include "recycle_bin_cb.h"
#include "prop_dlg.h"
#include "find_win.h"
#include "find_win_cb.h"
#include "history_win.h"
#include "edv_cb.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/* Window Created/Deleted */
void edv_emit_window_created(
	EDVCore *core,
	const EDVWindowType win_type,
	gpointer win
);
void edv_emit_window_deleted(
	EDVCore *core,
	const EDVWindowType win_type,
	gpointer win
);

/* Reconfigured */
void edv_emit_reconfigured(EDVCore *core);
void edv_queue_emit_reconfigured(EDVCore *core);

/* Master Write Protect Changed */
void edv_emit_master_write_protect_changed(EDVCore *core);
void edv_queue_emit_master_write_protect_changed(EDVCore *core);

/* Delete Method Changed */
void edv_emit_delete_method_changed(EDVCore *core);
void edv_queue_emit_delete_method_changed(EDVCore *core);

/* MIME Type Added/Mdoified/Removed */
void edv_mime_type_added_cb(
	const gint mt_num, EDVMIMEType *mt, gpointer data
);
void edv_emit_mime_type_added(
	EDVCore *core,
	const gint mt_num, EDVMIMEType *mt
);
void edv_mime_type_modified_cb(
	const gint mt_num, EDVMIMEType *mt, gpointer data
);
void edv_emit_mime_type_modified(
	EDVCore *core,
	const gint mt_num, EDVMIMEType *mt
);
void edv_mime_type_removed_cb(
	const gint mt_num, gpointer data
);
void edv_emit_mime_type_removed(
	EDVCore *core, const gint mt_num
);

/* Device Added/Modified/Removed */
void edv_emit_device_added(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev
);
void edv_emit_device_modified(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev
);
void edv_emit_device_removed(
	EDVCore *core,
	const  gint dev_num
);

/* Device Mount/Unmount */
void edv_emit_device_mount(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev,
	const gboolean is_mounted
);

/* History Event Added/Removed */
void edv_emit_history_added(
	EDVCore *core,
	const gint h_num, EDVHistory *h
);
void edv_emit_history_removed(
	EDVCore *core,
	const gint h_num
);

/* VFS Object Added/Modified/Removed */
void edv_emit_vfs_object_added(
	EDVCore *core,
	const gchar *path,
	EDVVFSObject *obj
);
void edv_emit_vfs_object_modified(
	EDVCore *core,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
);
void edv_emit_vfs_object_removed(
	EDVCore *core,
	const gchar *path
);

/* Recycled Object Added/Modified/Removed */
void edv_emit_recycled_object_added(
	EDVCore *core,
	const guint index
);
void edv_emit_recycled_object_modified(
	EDVCore *core,
	const guint index
);
void edv_emit_recycled_object_removed(
	EDVCore *core,
	const guint index
);

/* Archive Object Added/Modified/Removed */
void edv_emit_archive_object_added(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path,
	EDVArchiveObject *obj
);
void edv_emit_archive_object_modified(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path,
	const gchar *new_path,
	EDVArchiveObject *obj
);
void edv_emit_archive_object_removed(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Emits a window created signal.
 *
 *	The win_type specifies the type of window.
 *
 *	The win specifies an opaque pointer to the window.
 */
void edv_emit_window_created(
	EDVCore *core,
	const EDVWindowType win_type,
	gpointer win
)
{
	gint i;

	if((core == NULL) || (win_type == 0) || (win == NULL))
		return;

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_window_created_cb(
			core->browser[i],
			win_type,
			win
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_window_created_cb(
			core->imbr[i],
			win_type,
			win
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_window_created_cb(
			core->archiver[i],
			win_type,
			win
		);

	/* Recycle Bin */
	edv_recycle_bin_window_created_cb(
		core->recbin,
		win_type,
		win
	);

}

/*
 *	Emits a window deleted signal.
 *
 *	The win_type specifies the type of window.
 *
 *	The win specifies an opaque pointer to the window. Since the
 *	window has been deleted, this pointer must be assumed to no
 *	longer point to anything valid.
 */
void edv_emit_window_deleted(
	EDVCore *core,
	const EDVWindowType win_type,
	gpointer win
)
{
	gint i;

	if((core == NULL) || (win_type == 0) || (win == NULL))
		return;

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_window_deleted_cb(
			core->browser[i],
			win_type,
			win
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_window_deleted_cb(
			core->imbr[i],
			win_type,
			win
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_window_deleted_cb(
			core->archiver[i],
			win_type,
			win
		);

	/* Recycle Bin */
	edv_recycle_bin_window_deleted_cb(
		core->recbin,
		win_type,
		win
	);
}


/*
 *	Emits a reconfigured signal.
 *
 *	The reconfigured signal notifies all of Endeavour that the
 *	configuration has been reconfigured (changed) and that all
 *	displays and values need to be updated to reflect that.
 */
void edv_emit_reconfigured(EDVCore *core)
{
	gint i;
	gulong t;
	GtkRcStyle	*standard_rcstyle,
			*lists_rcstyle;
	CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Timeouts */
	t = EDV_GET_UL(EDV_CFG_PARM_CHECK_MOUNT_STATES_INT);
	if(t > 0l)
	{
		(void)GTK_TIMEOUT_REMOVE(core->check_mount_states_toid);
		core->check_mount_states_toid = gtk_timeout_add(
			t * 1000l,
			edv_check_mount_states_cb, core
		);
	}
	else
	{
		core->check_mount_states_toid = GTK_TIMEOUT_REMOVE(core->check_mount_states_toid);
	}


	/* Update the RC styles */
	edv_update_rc_styles(core);
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Update the global Tool Tips state */
	GUISetGlobalTipsState(EDV_GET_B(EDV_CFG_PARM_SHOW_TOOLTIPS));


	/* Update the popup lists */

	/* Users & Groups popup lists */
	core->users_pulist = PUListUnref(core->users_pulist);
	core->groups_pulist = PUListUnref(core->groups_pulist);
	edv_update_id_popup_lists(core);

	/* Devices popup list */
	core->devices_pulist = PUListUnref(core->devices_pulist);
	edv_update_devices_popup_list(core);

	/* Open With popup list */
	core->open_with_pulist = PUListUnref(core->open_with_pulist);
	edv_update_open_with_popup_list(core);


	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_reconfigured_cb(
			core->browser[i]
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_reconfigured_cb(
			core->imbr[i]
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_reconfigured_cb(
			core->archiver[i]
		);

	/* Recycle Bin Desktop Icon */
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_SHOW_DESKTOP_ICON))
		edv_map_recycle_bin_desktop_icon(core);
	else
		edv_recycle_bin_desktop_icon_unmap(core->recbin_deskicon);

	/* Recycle Bin */
	edv_recycle_bin_reconfigured_cb(core->recbin);

	/* Find Window */
	edv_find_win_reconfigured_cb(core->find_win);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_reconfigured_cb(
			core->prop_dlg[i]
		);

	/* History List Window */
	EDVHistoryWinReconfiguredNotifyCB(
		core->history_list_win
	);

	/* Devices List Window */
	EDVDevicesListWinReconfiguredNotifyCB(
		core->devices_list_win
	);

	/* MIME Types Window */
	EDVMimeTypesListWinReconfiguredNotifyCB(
		core->mime_types_list_win
	);
}

/*
 *	Queues a reconfigured signal.
 */
void edv_queue_emit_reconfigured(EDVCore *core)
{
	if(core == NULL)
		return;

	if(core->emit_reconfigured_idle_id != 0)
		return;

	core->emit_reconfigured_idle_id = gtk_idle_add_priority(
		G_PRIORITY_LOW,
		edv_reconfigured_idle_cb, core
	);
}


/*
 *	Emits a Master Write Protect state changed signal.
 *
 *	The write protect changed signal notifies when the Master
 *	Write Protect state has changed.
 */
void edv_emit_master_write_protect_changed(EDVCore *core)
{
	gboolean state;
	gint i;
	CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Get the current Master Write Protect state */
	state = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_master_write_protect_changed_cb(
			core->browser[i], state
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_master_write_protect_changed_cb(
			core->imbr[i], state
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_write_protect_changed_cb(
			core->archiver[i], state
		);

	/* Recycle Bin Desktop Icon */
	edv_recycle_bin_desktop_icon_update_display(core->recbin_deskicon);

	/* Recycle Bin */
	edv_recycle_bin_master_write_protect_changed_cb(core->recbin, state);

	/* Find Window */
	edv_find_win_write_protect_changed_cb(core->find_win, state);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_master_write_protect_changed_cb(
			core->prop_dlg[i], state
		);

	/* History Window */
	EDVHistoryWinWriteProtectChangedCB(
		core->history_list_win, state
	);
}

/*
 *	Queues a Master Write Protect state changed signal.
 */
void edv_queue_emit_master_write_protect_changed(EDVCore *core)
{
	if(core == NULL)
		return;

	if(core->emit_master_write_protect_changed_idle_id != 0)
		return;

	core->emit_master_write_protect_changed_idle_id = gtk_idle_add_priority(
		G_PRIORITY_LOW,
		edv_master_write_protect_changed_idle_cb, core
	);
}


/*
 *	Emits a Delete Method changed signal.
 */
void edv_emit_delete_method_changed(EDVCore *core)
{
	gint i;
	CfgList *cfg_list;
	EDVDeleteMethod delete_method;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	/* Get the current Delete Method */
	delete_method = (EDVDeleteMethod)EDV_GET_I(
		EDV_CFG_PARM_DELETE_METHOD
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_delete_method_changed_cb(
			core->browser[i],
			delete_method
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_delete_method_changed_cb(
			core->imbr[i],
			delete_method
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_delete_method_changed_cb(
			core->archiver[i],
			delete_method
		);

	/* Recycle Bin Desktop Icon */
	edv_recycle_bin_desktop_icon_update_display(core->recbin_deskicon);

	/* Recycle Bin */
	edv_recycle_bin_delete_method_changed_cb(
		core->recbin,
		delete_method
	);

	/* Find Window */
	edv_find_win_delete_method_changed_cb(
		core->find_win,
		delete_method
	);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_delete_method_changed_cb(
			core->prop_dlg[i],
			delete_method
		);

	/* History Window */
	EDVHistoryWinDeleteMethodChangedCB(
		core->history_list_win,
		delete_method
	);
}

/*
 *	Queues a Delete Method changed signal.
 */
void edv_queue_emit_delete_method_changed(EDVCore *core)
{
	if(core == NULL)
		return;

	if(core->emit_delete_method_changed_idle_id != 0)
		return;

	core->emit_delete_method_changed_idle_id = gtk_idle_add_priority(
		G_PRIORITY_LOW,
		edv_delete_method_changed_idle_cb, core
	);
}


/*
 *	Emits a MIME Type added signal.
 */
void edv_emit_mime_type_added(
	EDVCore *core,
	const gint mt_num, EDVMIMEType *mt
)
{
	gint i;

	if(core == NULL)
		return;

	/* Update MIME Types class list index hints due to change in
	 * the MIME Types list ordering
	 */
	edv_update_mime_type_hint_indicies(core);

	/* Reget list of application MIME Types on the `open with'
	 * popup list
	 */
	edv_update_open_with_popup_list(core);


	/* MIME Types Window */
	EDVMimeTypesListWinMimeTypeAddedCB(
		core->mime_types_list_win, mt_num, mt
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_mime_type_added_cb(
			core->browser[i], mt_num, mt
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_mime_type_added_cb(
			core->imbr[i], mt_num, mt
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_mime_type_added_cb(
			core->archiver[i], mt_num, mt
		);

	/* Recycle Bin */
	edv_recycle_bin_mime_type_added_cb(
		core->recbin, mt_num, mt
	);

	/* Property dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_mime_type_added_cb(
			core->prop_dlg[i], mt_num, mt
		);
}

/*
 *	Emits a MIME Type modified signal.
 */
void edv_emit_mime_type_modified(
	EDVCore *core,
	const gint mt_num, EDVMIMEType *mt
)
{
	gint i;

	if(core == NULL)
		return;

	/* Update MIME Types class list index hints due to change in
	 * the MIME Types list ordering
	 */
	edv_update_mime_type_hint_indicies(core);

	/* Reget list of application MIME Types on the `open with'
	 * popup list
	 */
	edv_update_open_with_popup_list(core);


	/* MIME Types Window */
	EDVMimeTypesListWinMimeTypeModifiedCB(
		core->mime_types_list_win, mt_num, mt
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_mime_type_modified_cb(
			core->browser[i], mt_num, mt
		);

	/* Image browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_mime_type_modified_cb(
			core->imbr[i], mt_num, mt
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_mime_type_modified_cb(
			core->archiver[i], mt_num, mt
		);

	/* Recycle Bin */
	edv_recycle_bin_mime_type_modified_cb(
		core->recbin, mt_num, mt
	);

	/* Property dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_mime_type_modified_cb(
			core->prop_dlg[i], mt_num, mt
		);
}

/*
 *	Emits a MIME Type removed signal.
 */
void edv_emit_mime_type_removed(
	EDVCore *core,
	const gint mt_num
)
{
	gint i;

	if(core == NULL)
		return;

	/* Update MIME Types class list index hints due to change in
	 * the MIME Types list ordering
	 */
	edv_update_mime_type_hint_indicies(core);

	/* Reget list of application MIME Types on the `open with'
	 * popup list
	 */
	edv_update_open_with_popup_list(core);


	/* MIME Types Window */
	EDVMimeTypesListWinMimeTypeRemovedCB(
		core->mime_types_list_win, mt_num
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_mime_type_removed_cb(
			core->browser[i], mt_num
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_mime_type_removed_cb(
			core->imbr[i], mt_num
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_mime_type_removed_cb(
			core->archiver[i], mt_num
		);

	/* Recycle Bin */
	edv_recycle_bin_mime_type_removed_cb(
		core->recbin, mt_num
	);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_mime_type_removed_cb(
			core->prop_dlg[i], mt_num
		);
}


/*
 *	Emits a device added signal.
 */
void edv_emit_device_added(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev
)
{
	gint i;

	if(core == NULL)
		return;

	/* Update the Devices Popup List */
	edv_update_devices_popup_list(core);


	/* Devices List Window */
	EDVDevicesListWinDeviceAddedCB(
		core->devices_list_win, dev_num, dev
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_device_added_cb(
			core->browser[i], dev_num, dev
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_device_added_cb(
			core->imbr[i], dev_num, dev
		);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_device_added_cb(
			core->prop_dlg[i], dev_num, dev
		);
}

/*
 *	Emits a device modified signal.
 */
void edv_emit_device_modified(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev
)
{
	gint i;

	if(core == NULL)
		return;

	/* Update the Devices Popup List */
	edv_update_devices_popup_list(core);


	/* Devices List Window */
	EDVDevicesListWinDeviceModifiedCB(
		core->devices_list_win, dev_num, dev
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_device_modified_cb(
			core->browser[i], dev_num, dev
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_device_modified_cb(
			core->imbr[i], dev_num, dev
		);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_device_modified_cb(
			core->prop_dlg[i], dev_num, dev
		);
}

/*
 *	Emits a device removed signal.
 */
void edv_emit_device_removed(
	EDVCore *core,
	const gint dev_num
)
{
	gint i;

	if(core == NULL)
		return;

	/* Update the Devices Popup List */
	edv_update_devices_popup_list(core);


	/* Devices List Window */
	EDVDevicesListWinDeviceRemovedCB(
		core->devices_list_win, dev_num
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_device_removed_cb(
			core->browser[i], dev_num
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_device_removed_cb(
			core->imbr[i], dev_num
		);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_device_removed_cb(
			core->prop_dlg[i], dev_num
		);
}


/*
 *	Emits a device mount/unmount signal.
 *
 *	The is_mounted specifies whether the device was mounted
 *	(is_mounted = TRUE) or unmounted (is_mounted = FALSE).
 */
void edv_emit_device_mount(
	EDVCore *core,
	const gint dev_num, EDVDevice *dev,
	const gboolean is_mounted
)
{
	gint i;

	if(core == NULL)
		return;

	/* Devices List */
	EDVDevicesListWinMountNotifyCB(
		core->devices_list_win,
		dev_num, dev,
		is_mounted
	);

	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_device_mount_cb(
			core->browser[i],
			dev_num, dev,
			is_mounted
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_device_mount_cb(
			core->imbr[i],
			dev_num, dev,
			is_mounted
		);

	/* Skip the Archivers */

	/* Skip the Recycle Bin */

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_device_mount_cb(
			core->prop_dlg[i],
			dev_num, dev,
			is_mounted
		);
}


/*
 *	Emits a history event added signal.
 */
void edv_emit_history_added(
	EDVCore *core,
	const gint h_num, EDVHistory *h
)
{
	if(core == NULL)
		return;

	/* History Window */
	EDVHistoryWinHistoryAddedCB(
		core->history_list_win,
		h_num, h
	);
}

/*
 *	Emits a history event removed signal.
 */
void edv_emit_history_removed(
	EDVCore *core,
	const gint h_num
)
{
	if(core == NULL)
		return;

	/* History Window */
	EDVHistoryWinHistoryRemovedCB(
		core->history_list_win,
		h_num
	);
}


/*
 *	Emits an object added signal.
 */
void edv_emit_vfs_object_added(
	EDVCore *core,
	const gchar *path,
	EDVVFSObject *obj
)
{
	gint i;
	gchar *dpath;

	if((core == NULL) || STRISEMPTY(path))
		return;

	dpath = g_strdup(path);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);


	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_vfs_object_added_cb(
			core->browser[i],
			dpath,
			obj
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_vfs_object_added_cb(
			core->imbr[i],
			dpath,
			obj
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_object_added_cb(
			core->archiver[i],
			dpath,
			obj
		);

	/* Skip the Recycle Bin */

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_vfs_object_added_cb(
			core->prop_dlg[i],
			dpath,
			obj
		);

	g_free(dpath);
}

/*
 *	Emits an object modified signal.
 *
 *	The path specifies the full path to the modified object. If
 *	new_path is not NULL then path specifies the object's previous
 *	location and name, otherwise it just specifies the full path
 *	to the modified object.
 *
 *	The new_path specifies the full path to the modified object's
 *	new location and name. If new_path is NULL then path will be
 *	used instead and the object's location and name are assumed to
 *	be unchanged.
 */
void edv_emit_vfs_object_modified(
	EDVCore *core,
	const gchar *path,
	const gchar *new_path,
	EDVVFSObject *obj
)
{
	gint i;
	gchar		*dpath,
			*dnew_path;

	if((core == NULL) || STRISEMPTY(path))
		return;

	dpath = g_strdup(path);
	if(dpath == NULL)
		return;

	dnew_path = g_strdup(new_path);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);


	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_vfs_object_modified_cb(
			core->browser[i],
			dpath,
			dnew_path,
			obj
		);

	/* Image browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_vfs_object_modified_cb(
			core->imbr[i],
			dpath,
			dnew_path,
			obj
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_object_modified_cb(
			core->archiver[i],
			dpath,
			dnew_path,
			obj
		);

	/* Skip the Recycle Bin */

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_vfs_object_modified_cb(
			core->prop_dlg[i],
			dpath,
			dnew_path,
			obj
		);

	g_free(dpath);
	g_free(dnew_path);
}

/*
 *	Emits an object removed signal.
 *
 *	The path specifies the full path to the object that has
 *	been removed. This object is assumed to no longer exist.
 */
void edv_emit_vfs_object_removed(
	EDVCore *core,
	const gchar *path
)
{
	gint i;
	gchar *dpath;

	if((core == NULL) || STRISEMPTY(path))
		return;

	/* Copy the specified path */
	dpath = g_strdup(path);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);


	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_vfs_object_removed_cb(
			core->browser[i],
			dpath
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_vfs_object_removed_cb(
			core->imbr[i],
			dpath
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		EDVArchiverObjectRemovedNotifyCB(
			core->archiver[i],
			dpath
		);

	/* Skip the Recycle Bin */

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_vfs_object_removed_cb(
			core->prop_dlg[i],
			dpath
		);

	g_free(dpath);
}


/*
 *	Emits a recycled object added signal.
 */
void edv_emit_recycled_object_added(
	EDVCore *core, const guint index
)
{
	gint i;
	const gchar *index_path;
	CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);

	/* Update the number of recycled objects */
	core->last_nrecycle_bin_items = edv_recycle_bin_index_get_total(index_path);


	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_recycled_object_added_cb(
			core->browser[i], index
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_recycled_object_added_cb(
			core->imbr[i], index
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_recycled_object_added_cb(
			core->archiver[i],
			index
		);

	/* Recycle Bin Desktop Icon */
	edv_recycle_bin_desktop_icon_update_display(core->recbin_deskicon);

	/* Recycle Bin */
	edv_recycle_bin_recycled_object_added_cb(
		core->recbin,
		index
	);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_recycled_object_added_cb(
			core->prop_dlg[i],
			index
		);
}

/*
 *	Emits a recycled object modified signal.
 */
void edv_emit_recycled_object_modified(
	EDVCore *core,
	const guint index
)
{
	gint i;
	const gchar *index_path;
	CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);

	/* Update the number of recycled objects */
	core->last_nrecycle_bin_items = edv_recycle_bin_index_get_total(index_path);


	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_recycled_object_modified_cb(
			core->browser[i],
			index
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_recycled_object_modified_cb(
			core->imbr[i],
			index
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_recycled_object_modified_cb(
			core->archiver[i],
			index
		);

	/* Recycle Bin Desktop Icon */
	edv_recycle_bin_desktop_icon_update_display(core->recbin_deskicon);

	/* Recycle Bin */
	edv_recycle_bin_recycled_object_modified_cb(
		core->recbin,
		index
	);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_recycled_object_modified_cb(
			core->prop_dlg[i],
			index
		);
}

/*
 *	Emits a recycled object removed signal.
 */
void edv_emit_recycled_object_removed(
	EDVCore *core, const guint index
)
{
	gint i;
	const gchar *index_path;
	const CfgList *cfg_list;

	if(core == NULL)
		return;

	cfg_list = core->cfg_list;

	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);

	/* Update number of recycled objects on the core structure */
	core->last_nrecycle_bin_items = edv_recycle_bin_index_get_total(index_path);


	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
		edv_vfs_browser_recycled_object_removed_cb(
			core->browser[i],
			index
		);

	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
		edv_image_browser_recycled_object_removed_cb(
			core->imbr[i],
			index
		);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_recycled_object_removed_cb(
			core->archiver[i],
			index
		);

	/* Recycle Bin Desktop Icon */
	edv_recycle_bin_desktop_icon_update_display(core->recbin_deskicon);

	/* Recycle Bin */
	edv_recycle_bin_recycled_object_removed_cb(
		core->recbin,
		index
	);

	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
		edv_prop_dlg_recycled_object_removed_cb(
			core->prop_dlg[i],
			index
		);
}


/*
 *	Emits an archive object added signal.
 *
 *	The arch_path specifies the full path to the archive that the
 *	object was added to.
 *
 *	The path and obj specifies the path to the added archive object.
 */
void edv_emit_archive_object_added(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path,
	EDVArchiveObject *obj
)
{
	gint i;
	gchar		*darch_path,
			*dpath;

	if(core == NULL)
		return;

	darch_path = STRDUP(arch_path);
	dpath = STRDUP(path);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);


	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_archive_object_added_cb(
			core->archiver[i],
			darch_path,
			dpath,
			obj
		);

	g_free(darch_path);
	g_free(dpath);
}

/*
 *	Emits an archive object modified signal.
 *
 *	The arch_path specifies the full path to the archive that the
 *	object was modified in.
 *
 *	The path and obj specifies the modified archive object. If
 *	new_path is not NULL then path specifies the path to
 *	the archive object's previous location and name, otherwise
 *	it just specifies the path to the modified archive object.
 *
 *	The new_path specifies the path to the modifed archive object's
 *	new location and name. If new_path is NULL then the location
 *	and name of the archive object is assumed to be unchanged.
 */
void edv_emit_archive_object_modified(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path,
	const gchar *new_path,
	EDVArchiveObject *obj
)
{
	gint i;
	gchar		*darch_path,
			*dpath,
			*dnew_path;

	if(core == NULL)
		return;

	darch_path = STRDUP(arch_path);
	dpath = STRDUP(path);
	dnew_path = STRDUP(new_path);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);

	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_archive_object_modified_cb(
			core->archiver[i],
			darch_path,
			dpath,
			dnew_path,
			obj
		);

	g_free(darch_path);
	g_free(dpath);
	g_free(dnew_path);
}

/*
 *	Emits an archive object removed signal.
 *
 *	The arch_path specifies the full path to the archive that the
 *	object was removed from.
 *
 *	The path specifies the path to the archive object that has
 *	been removed. This archive object is assumed to no longer exist.
 */
void edv_emit_archive_object_removed(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *path
)
{
	gint i;
	gchar		*darch_path,
			*dpath;

	if(core == NULL)
		return;

	darch_path = STRDUP(arch_path);
	dpath = STRDUP(path);

	/* Update the devices list */
	edv_devices_list_update_statistics(core->devices_list);

	/* Update the devices list window */
	EDVDevicesListWinUsageChangedCB(core->devices_list_win);


	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
		edv_archiver_archive_object_removed_cb(
			core->archiver[i],
			darch_path,
			dpath
		);

	g_free(darch_path);
	g_free(dpath);
}
