#include <stdlib.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "url.h"
#include "cfg.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycled_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_confirm.h"
#include "edv_status_bar.h"
#include "archive_options_dlg.h"
#include "edv_vfs_obj_op.h"
#include "edv_recycle_obj.h"
#include "edv_recover_obj.h"
#include "edv_archive_obj_stat.h"
#include "edv_archive_add.h"
#include "edv_archive_extract.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "edv_dnd.h"
#include "endeavour2.h"

#include "config.h"


gint edv_dnd_any_to_vfs(
	EDVCore *core,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	const gchar *tar_path,
	const gboolean verbose,
	GtkWidget *toplevel,
	GtkWidget *status_bar
);

gint edv_dnd_vfs_to_recycle_bin(
	EDVCore *core,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	const gboolean verbose,
	GtkWidget *toplevel,
	GtkWidget *status_bar
);

gint edv_dnd_vfs_to_archive(
	EDVCore *core,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	const gchar *arch_path,
	gchar **password_rtn,
	const gboolean verbose,
	GtkWidget *toplevel,
	GtkWidget *status_bar
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Process any DND operation to the VFS.
 *
 *	Modifications to the VFS, Recycle Bin, and archives may be
 *	notified by this call, so the calling function should not
 *	freeze any windows or else they will not process the signal.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *	supported location types are:
 *
 *	EDV_DND_INFO_TEXT_PLAIN		VFS
 *	EDV_DND_INFO_TEXT_URI_LIST	VFS
 *	EDV_DND_INFO_STRING		VFS
 *	EDV_DND_INFO_RECYCLED_OBJECT	Recycle Bin
 *	EDV_DND_INFO_ARCHIVE_OBJECT	Archive
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *	The tar_path specifies the target location.
 *
 *	If verbose is TRUE then error messages may be displayed.
 *
 *	The toplevel specifies the toplevel GtkWindow.
 *
 *	If status_bar is not NULL then the status bar message may
 *	be set throughout this call.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_dnd_any_to_vfs(
	EDVCore *core,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	const gchar *tar_path,
	const gboolean verbose,
	GtkWidget *toplevel,
	GtkWidget *status_bar
)
{
	gint		status,
			nurls;
	gchar *parent_path;
	GList *urls_list;
	GdkDragAction drag_action;

	if((core == NULL) || (dc == NULL) || (selection_data == NULL) ||
	   STRISEMPTY(tar_path)
	)
	{
		errno = EINVAL;
		return(-2);
	}

        /* Is there an operation already in progress? */
        if(core->op_level > 0)
        {
		errno = EBUSY;
                return(-6);
        }

	/* Check if the master write protect is enabled */
	if(edv_check_master_write_protect(core, verbose, toplevel))
	{
		errno = EPERM;
		return(-1);
	}

	drag_action = dc->action;

	parent_path = g_strdup(tar_path);
	if(parent_path == NULL)
		return(-3);

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 g_free(parent_path);				\
						\
 if(urls_list != NULL) {			\
  g_list_foreach(				\
   urls_list,					\
   (GFunc)url_delete,				\
   NULL						\
  );						\
  g_list_free(urls_list);			\
 }						\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

	/* If the target path does not lead to a directory then get
	 * its parent
	 */
	if(edv_path_exists(parent_path))
	{
		if(!edv_path_is_directory(parent_path))
		{
			gchar *s = g_dirname(parent_path);
			if(s != NULL)
			{
				g_free(parent_path);
				parent_path = s;
			}
		}
	}
	else
	{
		gchar *s = g_dirname(parent_path);
		if(s != NULL)
		{
			g_free(parent_path);
			parent_path = s;
		}
	}

	/* Decode the DDE buffer into a list of URLs */
	urls_list = url_decode(
		(const guint8 *)selection_data->data,
		selection_data->length
	);
	if(urls_list == NULL)
	{
		CLEANUP_RETURN(-1);
	}

	nurls = g_list_length(urls_list);

	/* External drag source checks go here, for example sources
	 * that have a url with a http or ftp protocol
	 */
	if(nurls >= 1)
	{
		URLStruct *url = URL(g_list_nth_data(
			urls_list,
			0
		));
		if(url != NULL)
		{
			const gchar *protocol = url->protocol;
			if(protocol != NULL)
			{
				if(!g_strcasecmp(protocol, "http") ||
				   !g_strcasecmp(protocol, "ftp") ||
				   !g_strcasecmp(protocol, "https")
				)
				{
					const gint status = edv_internet_download_object(
						core,
						url,
						parent_path,
						toplevel
					);
					CLEANUP_RETURN(status);
				}
			}
		}
	}

	status = -1;

	/* Source location is the VFS? */
	if((info == EDV_DND_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_INFO_STRING)
	)
	{
		gboolean yes_to_all = FALSE;
		gint nobjs_processed = 0;
		const gchar	*path,
				*error_title,
				*error_msg;
		GList	*glist,
			*new_paths_list;
		URLStruct *url;

		/* Confirm */
		if(nurls >= 1)
		{
			const gchar *src_path = NULL;
			const gint nsrc_paths = nurls;
			gint response = CDIALOG_RESPONSE_NOT_AVAILABLE;
			if(nurls == 1)
			{
				URLStruct *url = URL(g_list_nth_data(
					urls_list,
					0
				));
				if(url != NULL)
					src_path = url->path;
			}
			switch(drag_action)
			{
			  case GDK_ACTION_LINK:
				response = edv_confirm_link(
					core,
					toplevel,
					src_path, nsrc_paths,
					parent_path
				);
				break;
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_MOVE:
				response = edv_confirm_move(
					core,
					toplevel,
					src_path, nsrc_paths,
					parent_path
				);
				break;
			  case GDK_ACTION_COPY:
				response = edv_confirm_copy(
					core,
					toplevel,
					src_path, nsrc_paths,
					parent_path
				);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			if((response != CDIALOG_RESPONSE_YES_TO_ALL) &&
			   (response != CDIALOG_RESPONSE_YES) &&
			   (response != CDIALOG_RESPONSE_OK)
			)
			{
				errno = EINTR;
				CLEANUP_RETURN(-5);
			}
		}

		/* Copy, move, or link each object on the VFS */
		for(glist = urls_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			url = URL(glist->data);
			if(url == NULL)
				continue;

			path = url->path;
			if(STRISEMPTY(path))
				continue;

			/* Handle by the drag action type */
			new_paths_list = NULL;
			error_title = NULL;
			error_msg = NULL;
			switch(drag_action)
			{
			  case GDK_ACTION_COPY:
				status = edv_vfs_object_op_copy(
					core,
					path,			/* Source */
					parent_path,		/* Target */
					&new_paths_list,
					TRUE,			/* Archive */
					toplevel,
					verbose,		/* Show progress */
					TRUE,			/* Interactive */
					&yes_to_all
				);
				error_title = "Copy Object Error";
				break;

			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_MOVE:
				status = edv_vfs_object_op_move(
					core,
					path,			/* Source */
					parent_path,		/* Target */
					&new_paths_list,
					TRUE,			/* Archive */
					toplevel,
					verbose,		/* Show progress */
					TRUE,			/* Interactive */
					&yes_to_all
				);
				error_title = "Move Object Error";
				break;

			  case GDK_ACTION_LINK:
				status = edv_vfs_object_op_link(
					core,
					parent_path,		/* Create link here */
					path,			/* Link's target */
					&new_paths_list,
					toplevel,
					verbose,		/* Show progress */
					TRUE,			/* Interactive */
					&yes_to_all
				);
				error_title = "Link Object Error";
				break;

			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				status = -2;
				error_title = "Error";
				error_msg = "Unsupported drag operation";
				break;
			}

			/* Check for errors */
			if(error_msg == NULL)
				error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg) && verbose)
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					error_title,
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Report that the source object no longer exists? */
			if(!edv_path_lexists(path))
				edv_emit_vfs_object_removed(
					core,
					path
				);

			/* Count the VFS objects processed and notify
			 * about the new VFS objects
			 */
			if(new_paths_list != NULL)
			{
				const gchar *path;
				GList *glist;
				EDVVFSObject *obj;
				for(glist = new_paths_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					path = (const gchar *)glist->data;
					if(STRISEMPTY(path))
						continue;

					nobjs_processed++;

					obj = edv_vfs_object_lstat(path);
					if(obj != NULL)
					{
						edv_emit_vfs_object_added(
							core,
							path,
							obj
						);
						edv_vfs_object_delete(obj);
					}
				}

				g_list_foreach(
					new_paths_list,
					(GFunc)g_free,
					NULL
				);
				g_list_free(new_paths_list);
			}

			/* Stop the operation if the user aborted */
			if(status == -4)
				break;
		}

		/* Unmap the progress dialog which may have been
		 * mapped in the above operation
		 */
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);

		/* Set the status bar message */
		if(status_bar != NULL)
		{
			const gint nsrc_objs = nurls;
			gchar *msg = NULL;
			switch(drag_action)
			{
			  case GDK_ACTION_COPY:
				if(status == -4)
					msg = g_strdup(
"Copy canceled"
					);
				else if(nobjs_processed > 0)
					msg = g_strdup_printf(
"Coppied %i %s",
						nobjs_processed,
						(nobjs_processed == 1) ? "object" : "objects"
					);
				else
					msg = g_strdup_printf(
"Unable to copy %s",
						(nsrc_objs == 1) ? "object" : "objects"
					);
				break;
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_MOVE:
				if(status == -4)
					msg = g_strdup(
"Move canceled"
					);
				else if(nobjs_processed > 0)
					msg = g_strdup_printf(
"Moved %i %s",
						nobjs_processed,
						(nobjs_processed == 1) ? "object" : "objects"
					);
				else
					msg = g_strdup_printf(
"Unable to move %s",
						(nsrc_objs == 1) ? "object" : "objects"
					);
				break;
			  case GDK_ACTION_LINK:
				if(status == -4)
					msg = g_strdup(
"Link canceled"
					);
				else if(nobjs_processed > 0)
					msg = g_strdup_printf(
"Linked %i %s",
						nobjs_processed,
						(nobjs_processed == 1) ? "object" : "objects"
					);
				else
					msg = g_strdup_printf(
"Unable to link %s",
						(nsrc_objs == 1) ? "object" : "objects"
					);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			edv_status_bar_message(
				status_bar,
				msg,
				FALSE
			);
			g_free(msg);
		}

		/* Play the "completed" sound on success */
		if(status == 0)
			edv_play_sound_completed(core);
	}
	/* Source location is the recycle bin? */
	else if(info == EDV_DND_INFO_RECYCLED_OBJECT)
	{
		gboolean yes_to_all = FALSE;
		gint nobjs_recovered = 0;
		gulong index;
		gchar *new_path;
		const gchar *error_msg;
		GList *glist;
		URLStruct *url;

		/* Confirm */
		if(nurls >= 1)
		{
			const gchar *src_name = NULL;
			const gint nsrc_names = nurls;
			gint response = CDIALOG_RESPONSE_NOT_AVAILABLE;
			if(nurls == 1)
			{
				URLStruct *url = URL(g_list_nth_data(
					urls_list,
					0
				));
				if(url != NULL)
					src_name = url->path_arg;
			}
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_LINK:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_COPY:
				response = edv_confirm_recover(
					core,
					toplevel,
					src_name, nsrc_names,
					parent_path
				);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			if((response != CDIALOG_RESPONSE_YES_TO_ALL) &&
			   (response != CDIALOG_RESPONSE_YES) &&
			   (response != CDIALOG_RESPONSE_OK)
			)
			{
				errno = EINTR;
				CLEANUP_RETURN(-5);
			}
		}

		/* Recover each recycled object to the VFS */
		for(glist = urls_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{  
			url = URL(glist->data);
			if(url == NULL)
				continue;

			index = ATOL(url->path);

			/* Handle by the drag action type */
			new_path = NULL;
			error_msg = NULL;
			switch(drag_action)
			{
			    case GDK_ACTION_COPY:
				error_msg = "Recycled objects may not be coppied";
				break;
			    case GDK_ACTION_MOVE:
				status = edv_recover_object(
					core,
					index,		/* Recycled object to recover */
					parent_path,	/* Alternate recovery location */
					&new_path,	/* Recovered object path return */
					toplevel,
					verbose,	/* Show progress */
					TRUE,		/* Interactive */
					&yes_to_all
				);
				break;
			    case GDK_ACTION_LINK:
				error_msg = "Recycled objects may not be linked";
				break;
			    default:
				error_msg = "Unsupported drag operation";
				break;
			}

			/* Check for errors */
			if(error_msg == NULL)
				error_msg = edv_recover_object_get_error(core);
			if(!STRISEMPTY(error_msg) && verbose)
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Recover Recycled Object Error",
					error_msg,
					NULL,
					toplevel
				);
			}

	                /* Count the recovered object, notify about
			 * the recycled object being removed, and
			 * notify about the new recovered VFS object
			 */
			if(new_path != NULL)
			{
				EDVVFSObject *obj = edv_vfs_object_lstat(new_path);
				nobjs_recovered++;
				if(obj != NULL)
				{
					edv_emit_recycled_object_removed(
						core,
						index
					);
					edv_emit_vfs_object_added(
						core,
						new_path,
						obj
					);
					edv_vfs_object_delete(obj);
				}
				g_free(new_path);
			}

			/* Stop recovering if the user aborted */
			if(status == -4)
				break;
		}

		/* Unmap the progress dialog which may have been
		 * mapped in the above operation
		 */
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);

		/* Set the status bar message */
		if(status_bar != NULL)
		{
			const gint	nsrc_objs = nurls,
					nobjs_processed = nobjs_recovered;
			gchar *msg = NULL;
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_COPY:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_LINK:
				if(status == -4)
					msg = g_strdup(
"Recover canceled"
					);
				else if(nobjs_processed > 0)
					msg = g_strdup_printf(
"Recovered %i %s",
						nobjs_processed,
						(nobjs_processed == 1) ? "object" : "objects"
					);
				else
					msg = g_strdup_printf(
"Unable to recover %s",
						(nsrc_objs == 1) ? "object" : "objects"
					);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			edv_status_bar_message(
				status_bar,
				msg,
				FALSE
			);
			g_free(msg);
		}

		/* Play the "completed" sound on success */
		if(status == 0)
			edv_play_sound_completed(core);
	}
	/* Source location is an archive? */
	else if(info == EDV_DND_INFO_ARCHIVE_OBJECT)
	{
		gboolean	yes_to_all = FALSE,
				preserve_directories = TRUE,
				preserve_timestamps = TRUE;
		gint nobjs_extracted = 0;
		const gchar	*arch_path = NULL,
				*error_msg;
		gchar *password = NULL;
		GList	*obj_list,
			*new_paths_list;

		/* Confirm */
		if(nurls >= 2)
		{
			/* The first object is treated as the archive itself,
			 * and all subsequent objects are objects in the
			 * archive
			 */
			const gchar *src_path = NULL;
			const gint nsrc_paths = nurls - 1;
			gint response = CDIALOG_RESPONSE_NOT_AVAILABLE;
			if(nurls == 2)
			{
				URLStruct *url = URL(g_list_nth_data(
					urls_list,
					1			/* Second URL is the first
											 * object in the archive */
				));
				if(url != NULL)
					src_path = url->path;
			}
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_LINK:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_COPY:
				response = edv_confirm_archive_extract(
					core,
					toplevel,
					src_path, nsrc_paths,
					parent_path
				);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			if((response != CDIALOG_RESPONSE_YES_TO_ALL) &&
			   (response != CDIALOG_RESPONSE_YES) &&
			   (response != CDIALOG_RESPONSE_OK)
			)
			{
				errno = EINTR;
				CLEANUP_RETURN(-5);
			}
		}

		/* Get the path to the archive (first URL) */
		if(nurls >= 1)
		{
			URLStruct *url = URL(g_list_nth_data(
				urls_list,
				0
			));
			if(url != NULL)
				arch_path = url->path;
		}

		/* Query the user for extract from archive options */
		if(!edv_archive_options_dlg_query_extract(
			core,
			toplevel,
			arch_path,
			&password,
			&preserve_directories,
			&preserve_timestamps
		))
		{
			g_free(password); 
			errno = EINTR;
			CLEANUP_RETURN(-5);
		}

		/* Get the archive object statistics list */
		obj_list = NULL;
		if(nurls >= 2)
		{
			GList	*glist,
					*paths_list = NULL;
			URLStruct *url;

			/* Add all the subsequent URL paths to the paths_list */
			for(glist = g_list_next(urls_list);	/* Skip the first URL */
				glist != NULL;
				glist = g_list_next(glist)
			)
			{
				url = URL(glist->data);
				if(url == NULL)
					continue;

				if(url->path == NULL)
					continue;

				paths_list = g_list_append(
					paths_list,
					g_strdup(url->path)
				);
			}
			if(paths_list != NULL)
			{
				obj_list = edv_archive_object_stat_list(
					core->cfg_list,
					arch_path,
					paths_list,
					NULL,			/* No filter */
					password,
					NULL, NULL
				);
				g_list_foreach(
					paths_list,
					(GFunc)g_free,
					NULL
				);
				g_list_free(paths_list);
			}
		}

		/* Extract the object(s) from the archive */
		new_paths_list = NULL;
		status = edv_archive_extract(
			core,
			arch_path,
			obj_list,
			FALSE,			/* Not extract all */
			parent_path,
			&new_paths_list,
			password,
			toplevel,
			verbose,		/* Show progress */
			TRUE,			/* Interactive */
			&yes_to_all,
			preserve_directories,
			preserve_timestamps
		);

		/* Unmap the progress dialog which may have been
		 * mapped during the above operation
		 */
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);

		/* Check for errors */
		error_msg = edv_archive_extract_get_error(core);
		if(!STRISEMPTY(error_msg) && verbose)
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Extract Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

		/* Count the number of archive objects extracted and
		 * notify about the new VFS objects
		 */
		if(new_paths_list != NULL)
		{
			const gchar *path;
			GList *glist;
			EDVVFSObject *obj;

			for(glist = new_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
				path = (const gchar *)glist->data;
				if(STRISEMPTY(path))
					continue;

				nobjs_extracted++;

				obj = edv_vfs_object_lstat(path);
				if(obj != NULL)
				{
					edv_emit_vfs_object_added(
						core,
						path,
						obj
					);
					edv_vfs_object_delete(obj);
				}
			}

			/* Delete the list of the extracted object paths */
			g_list_foreach(
				new_paths_list,
				(GFunc)g_free,
				NULL
			);
			g_list_free(new_paths_list);
		}

		/* Set the status bar message */
		if(status_bar != NULL)
		{
			const gint	nsrc_objs = nurls,
					nobjs_processed = nobjs_extracted;
			gchar *msg = NULL;
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_COPY:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_LINK:
				if(status == -4)
					msg = g_strdup(
"Extract canceled"
					);
				else if(nobjs_processed > 0)
					msg = g_strdup_printf(
"Extracted %i %s",
						nobjs_processed,
						(nobjs_processed == 1) ? "object" : "objects"
					);
				else
					msg = g_strdup_printf(
"Unable to extract %s",
						(nsrc_objs == 1) ? "object" : "objects"
					);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			edv_status_bar_message(
				status_bar,
				msg,
				FALSE
			);
			g_free(msg);
		}

		g_free(password);

		/* Delete the archive object statistics list */
		if(obj_list != NULL)
		{
			g_list_foreach(
				obj_list,
				(GFunc)edv_archive_object_delete,
				NULL
			);
			g_list_free(obj_list);
		}

		/* Play the "completed" sound on success */
		if(status == 0)
			edv_play_sound_completed(core);
	}
	/* Unsupported source location type */
	else
	{
		if(verbose)
		{
			gchar	*target_name = gdk_atom_name(selection_data->target),
					*msg = g_strdup_printf(
"Unsupported target type:\n\
\n\
    %s",
				target_name
			);
			g_free(target_name);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Operation Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
		errno = EINVAL;
		status = -2;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Process DND operation from the VFS to the Recycle Bin.
 *
 *	Modifications to the VFS, Recycle Bin, and archives may be
 *	notified by this call, so the calling function should not
 *	freeze any windows or else they will not process the signal.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *	supported location types are:
 *
 *	EDV_DND_INFO_TEXT_PLAIN		VFS
 *	EDV_DND_INFO_TEXT_URI_LIST	VFS
 *	EDV_DND_INFO_STRING		VFS
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *	If verbose is TRUE then error messages may be displayed.
 *
 *	The toplevel specifies the toplevel GtkWindow.
 *
 *	If status_bar is not NULL then the status bar message may
 *	be set throughout this call.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_dnd_vfs_to_recycle_bin(
	EDVCore *core,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	const gboolean verbose,
	GtkWidget *toplevel,
	GtkWidget *status_bar
)
{
	gint		status,
			nurls;
	GList *urls_list;
	GdkDragAction drag_action;

	if((core == NULL) || (dc == NULL) || (selection_data == NULL))
	{
		errno = EINVAL;
		return(-2);
	}

        /* Is there an operation already in progress? */
        if(core->op_level > 0)
        {
		errno = EBUSY;
                return(-6);
        }

	/* Check if the master write protect is enabled */
	if(edv_check_master_write_protect(core, verbose, toplevel))
	{
		errno = EPERM;
		return(-1);
	}

	drag_action = dc->action;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 if(urls_list != NULL) {			\
  g_list_foreach(				\
   urls_list,					\
   (GFunc)url_delete,				\
   NULL						\
  );						\
  g_list_free(urls_list);			\
 }						\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}


	/* Decode the DDE buffer into a list of URLs */
	urls_list = url_decode(
		(const guint8 *)selection_data->data,
		selection_data->length
	);
	if(urls_list == NULL)
	{
		CLEANUP_RETURN(-1);
	}

	nurls = g_list_length(urls_list);

	status = -1;

	/* Source location is the VFS? */
	if((info == EDV_DND_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_INFO_STRING)
	)
	{
		gboolean yes_to_all = FALSE;
		gint nobjs_deleted = 0;
		const gchar	*path,
				*error_msg;
		GList	*glist,
			*index_list;
		URLStruct *url;

		/* Confirm */
		if(nurls >= 1)
		{
			const gchar *src_path = NULL;
			const gint nsrc_paths = nurls;
			gint response = CDIALOG_RESPONSE_NOT_AVAILABLE;
			if(nurls == 1)
			{
				URLStruct *url = URL(g_list_nth_data(
					urls_list,
					0
				));
				if(url != NULL)
					src_path = url->path;
			}
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_COPY:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_LINK:
				response = edv_confirm_delete(
					core,
					toplevel,
					src_path, nsrc_paths
				);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			if((response != CDIALOG_RESPONSE_YES_TO_ALL) &&
			   (response != CDIALOG_RESPONSE_YES) &&
			   (response != CDIALOG_RESPONSE_OK)
			)
			{
				errno = EINTR;
				CLEANUP_RETURN(-5);
			}
		}

		/* Delete each object from the VFS */
		for(glist = urls_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			url = URL(glist->data);
			if(url == NULL)
				continue;

			path = url->path;
			if(STRISEMPTY(path))
				continue;

			/* Handle by the drag action type */
			index_list = NULL;
			error_msg = NULL;
			switch(drag_action)
			{
			  case GDK_ACTION_COPY:
				status = -2;
				error_msg = "Objects may not be coppied to the Recycle Bin";
				break;

			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_MOVE:
				status = edv_recycle_object(
					core,
					path,
					&index_list,
					toplevel,
					TRUE,	/* Show progress */
					TRUE,	/* Interactive */
					&yes_to_all
				);
				break;

			  case GDK_ACTION_LINK:
				status = -2;
				error_msg = "Objects may not be linked to the Recycle Bin";
				break;

			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				status = -2;
				error_msg = "Unsupported drag operation";
				break;
			}

			/* Check for errors */
			if(error_msg == NULL)
				error_msg = edv_recycle_object_get_error(core);
			if(!STRISEMPTY(error_msg) && verbose)
			{
				/* Report the error */
				edv_play_sound_error(core);
				edv_message_error(
					"Delete Object Error",
					error_msg,
					NULL,
					toplevel
				);
			}

			/* Report that the source object no longer exists? */
			if(!edv_path_lexists(path))
				edv_emit_vfs_object_removed(
					core,
					path
				);

			/* Report the deleted objects */
			if(index_list != NULL)
			{
				gulong index;
				GList *glist;

				/* Report recycled objects added */
				for(glist = index_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					index = (gulong)glist->data;
					if(index == 0l)
						continue;

					nobjs_deleted++;

					edv_emit_recycled_object_added(
						core,
						index
					);
				}
			}

			/* Delete the recycle objects index list */
			g_list_free(index_list);

			/* User aborted? */
			if(status == -4)
				break;
		}

		/* Unmap the progress dialog which may have been
		 * mapped during the above operation
		 */
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);

		/* Set the status bar message */
		if(status_bar != NULL)
		{
			const gint nsrc_objs = nurls;
			gchar *msg = NULL;
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_COPY:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_LINK:
				if(status == -4)
					msg = g_strdup(
"Delete operation canceled"
					);
				else if(nobjs_deleted > 0)
					msg = g_strdup_printf(
"Deleted %i %s",
						nobjs_deleted,
						(nobjs_deleted == 1) ? "object" : "objects"
					);
				else
					msg = g_strdup_printf(
"Unable to delete %s",
						(nsrc_objs == 1) ? "object" : "objects"
					);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			edv_status_bar_message(
				status_bar,
				msg,
				FALSE
			);
			g_free(msg);
		}

		/* Play the "completed" sound on success */
		if(status == 0)
			edv_play_sound_completed(core);
	}
	/* Unsupported source location type */
	else
	{
		if(verbose)
		{
			gchar	*target_name = gdk_atom_name(selection_data->target),
				*msg = g_strdup_printf(
"Unsupported target type:\n\
\n\
    %s",
				target_name
			);
			g_free(target_name);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Delete Object Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
		errno = EINVAL;
		status = -2;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Process DND operation from the VFS to an Archive.
 *
 *	Modifications to the VFS, Recycle Bin, and archives may be
 *	notified by this call, so the calling function should not
 *	freeze any windows or else they will not process the signal.
 *
 *	The dc specifies the GdkDragContext.
 *
 *	The info specifies the source object's location type,
 *	supported location types are:
 *
 *	EDV_DND_INFO_TEXT_PLAIN		VFS
 *	EDV_DND_INFO_TEXT_URI_LIST	VFS
 *	EDV_DND_INFO_STRING		VFS
 *
 *	The selection_data specifies the DND data describing the
 *	source objects in the form of a null-separated list of
 *	URL strings which will be parsed into a list of URL strings.
 *
 *	The arch_path specifies the archive.
 *
 *	If password_rtn is not NULL then *password_rtn specifies the
 *	password to use to encrypt the objects that are being added
 *	to the archive. The value of *password_rtn may be modified
 * 	by this call.
 *
 *	If verbose is TRUE then error messages may be displayed.
 *
 *	The toplevel specifies the toplevel GtkWindow.
 *
 *	If status_bar is not NULL then the status bar message may
 *	be set throughout this call.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_dnd_vfs_to_archive(
	EDVCore *core,
	GdkDragContext *dc,
	const guint info,
	GtkSelectionData *selection_data,
	const gchar *arch_path,
	gchar **password_rtn,
	const gboolean verbose,
	GtkWidget *toplevel,
	GtkWidget *status_bar
)
{
	gint		status,
			nurls;
	GList *urls_list;
	GdkDragAction drag_action;

	if((core == NULL) || (dc == NULL) || (selection_data == NULL) ||
	   STRISEMPTY(arch_path)
	)
	{
		errno = EINVAL;
		return(-2);
	}

        /* Is there an operation already in progress? */
        if(core->op_level > 0)
        {
		errno = EBUSY;
                return(-6);
        }

	/* Check if the master write protect is enabled */
	if(edv_check_master_write_protect(core, verbose, toplevel))
	{
		errno = EPERM;
		return(-1);
	}

	drag_action = dc->action;

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 if(urls_list != NULL) {			\
  g_list_foreach(				\
   urls_list,					\
   (GFunc)url_delete,				\
   NULL						\
  );						\
  g_list_free(urls_list);			\
 }						\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}


	/* Decode the DDE buffer into a list of URLs */
	urls_list = url_decode(
		(const guint8 *)selection_data->data,
		selection_data->length
	);
	if(urls_list == NULL)
	{
		CLEANUP_RETURN(-1);
	}

	nurls = g_list_length(urls_list);

	status = -1;

	/* Source location is the VFS? */
	if((info == EDV_DND_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_INFO_STRING)
	)
	{
		/* Add VFS objects to an archive */
		gboolean	yes_to_all = FALSE,
				recurse = TRUE,
				dereference_links = FALSE;
		gint	compression = 50,	/* 0 to 100 */
			nobjs_added = 0;
		const gchar *error_msg;
		GList	*glist,
			*tar_paths_list,
			*new_paths_list;
		URLStruct *url;
		EDVVFSObject *arch_obj;

		/* Confirm add VFS objects to an archive */
		if(nurls >= 1)
		{
			const gchar *src_path = NULL;
			const gint nsrc_paths = nurls;
			gint response = CDIALOG_RESPONSE_NOT_AVAILABLE;
			if(nurls == 1)
			{
				URLStruct *url = URL(g_list_nth_data(
					urls_list,
					0
				));
				if(url != NULL)
					src_path = url->path;
			}
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_COPY:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_LINK:
				response = edv_confirm_archive_add(
					core,
					toplevel,
					src_path, nsrc_paths,
					arch_path
				);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}
			if((response != CDIALOG_RESPONSE_YES_TO_ALL) &&
			   (response != CDIALOG_RESPONSE_YES) &&
			   (response != CDIALOG_RESPONSE_OK)
			)
			{
				errno = EINTR;
				CLEANUP_RETURN(-5);
			}
		}

		/* Query the user for add to archive options */
		if(!edv_archive_options_dlg_qiery_add(
			core,
			toplevel,
			arch_path,
			password_rtn,
			&recurse,
			&compression,
			&dereference_links
		))
		{
			errno = EINTR;
			CLEANUP_RETURN(-5);
		}

		/* Record the archive's original statistics and check
		 * if it existed
		 */
		arch_obj = edv_vfs_object_stat(arch_path);

		/* Create the list of paths of the VFS objects to be
		 * added to the archive from the URLs list
		 */
		tar_paths_list = NULL;
		for(glist = urls_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			url = URL(glist->data);
			if(url == NULL)
				continue;

			if(STRISEMPTY(url->path))
				continue;

			tar_paths_list = g_list_append(
				tar_paths_list,
				g_strdup(url->path)
			);
		}

		/* Add the objects to the archive */
		error_msg = NULL;
		new_paths_list = NULL;
		switch(drag_action)
		{
		  case GDK_ACTION_DEFAULT:
		  case GDK_ACTION_COPY:
			status = edv_archive_add(
				core,
				arch_path,
				tar_paths_list,	/* List of objects to add
						 * to the archive */
				&new_paths_list,/* Return list of objects
						 * added to the archive */
				(password_rtn != NULL) ? *password_rtn : NULL,
				toplevel,
				TRUE,		/* Show progress */
				TRUE,		/* Interactive */
				&yes_to_all,
				recurse,
				compression,
				dereference_links
			);
			break;

		  case GDK_ACTION_MOVE:
			status = -2;
			error_msg = "Objects may not be moved to the archive";
			break;

		  case GDK_ACTION_LINK:
			status = -2;
			error_msg = "Objects may not be linked to the archive";
			break;

		  case GDK_ACTION_PRIVATE:
		  case GDK_ACTION_ASK:
			status = -2;
			error_msg = "Unsupported drag operation";
			break;
		}

		/* Unmap the progress dialog which may have been
		 * mapped during the above operation
		 */
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);

		/* Delete the list of objects to be added to the archive */
		if(tar_paths_list != NULL)
		{
			g_list_foreach(
				tar_paths_list,
				(GFunc)g_free,
				NULL
			);
			g_list_free(tar_paths_list);
		}

		/* Check for errors */
		if(error_msg == NULL)
			error_msg = edv_archive_add_get_error(core);
		if(!STRISEMPTY(error_msg) && verbose)
		{
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
				"Add Object Error",
				error_msg,
				NULL,
				toplevel
			);
		}

		/* Count the number of objects added to the archive */
		if(new_paths_list != NULL)
		{
			nobjs_added += g_list_length(new_paths_list);

			/* Delete the list of objects added to
			 * the archive
			 */
			g_list_foreach(
				new_paths_list,
				(GFunc)g_free,
				NULL
			);
			g_list_free(new_paths_list);
		}

		/* Notify about objects being added to this archive */
		if(arch_obj != NULL)
		{
			edv_vfs_object_delete(arch_obj);
			arch_obj = edv_vfs_object_lstat(arch_path);
			if(arch_obj != NULL)
				edv_emit_vfs_object_modified(
					core,
					arch_path,
					arch_path,
					arch_obj
				);
		}
		else
		{
			arch_obj = edv_vfs_object_lstat(arch_path);
			if(arch_obj != NULL)
				edv_emit_vfs_object_added(
					core,
					arch_path,
					arch_obj
				);
		}

		edv_vfs_object_delete(arch_obj);

		/* Set the status bar message */
		if(status_bar != NULL)
		{
			const gint nsrc_objs = nurls;
			gchar *msg = NULL;
			switch(drag_action)
			{
			  case GDK_ACTION_DEFAULT:
			  case GDK_ACTION_COPY:
			  case GDK_ACTION_MOVE:
			  case GDK_ACTION_LINK:
				if(status == -4)
					msg = g_strdup(
"Add operation canceled"
					);
				else if(nobjs_added > 0)
					msg = g_strdup_printf(
"Added %i %s",
						nobjs_added,
						(nobjs_added == 1) ? "object" : "objects"
					);
				else
					msg = g_strdup_printf(
"Unable to add %s",
						(nsrc_objs == 1) ? "object" : "objects"
					);
				break;
			  case GDK_ACTION_PRIVATE:
			  case GDK_ACTION_ASK:
				break;
			}

			edv_status_bar_message(
				status_bar,
				msg,
				FALSE
			);
			g_free(msg);
		}

		/* Play the "completed" sound on success */
		if(status == 0)
			edv_play_sound_completed(core);
	}
	/* Unsupported source location type */
	else
	{
		if(verbose)
		{
			gchar	*target_name = gdk_atom_name(selection_data->target),
					*msg = g_strdup_printf(
"Unsupported target type:\n\
\n\
    %s",
				target_name
			);
			g_free(target_name);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Add Object Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
		}
		errno = EINVAL;
		status = -2;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
