#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#if defined(__FreeBSD__)
# include <fstab.h>
# include <sys/param.h>
# include <sys/ucred.h>
# include <sys/mount.h>
#elif defined(__SOLARIS__)
# include <sys/vfs.h>
# include <sys/mnttab.h>
# include <sys/vfstab.h>
# include <sys/statvfs.h>
#else
# include <mntent.h>
# include <sys/vfs.h>
#endif
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_fs_type.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "config.h"


/* Devices List Matching */
EDVDevice *edv_devices_list_match_mount_path(
	GList *devices_list,
	gint *n,
	const gchar *mount_path
);
EDVDevice *edv_devices_list_match_device_path(
	GList *devices_list,
	gint *n,
	const gchar *device_path
);
EDVDevice *edv_devices_list_match_object(
	GList *devices_list,
	gint *n,
	const gchar *path
);

/* Open System Devices List File */
GList *edv_devices_list_open_from_system(
	GList *devices_list,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
);

/* Mount States */
void edv_devices_list_update_mount_states(GList *devices_list);

/* Statistics */
void edv_devices_list_update_statistics(GList *devices_list);

/* Open Devices List File */
GList *edv_devices_list_open(
	GList *devices_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
);
/* Save Devices List File */
void edv_devices_list_file_save(
	GList *devices_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Matches the Device by its mount path.
 *
 *	The devices_list specifies the devices list.
 *
 *	The n specifies the device index return value.
 *
 *	The mount_path specifies the string describing the full path
 *	to the directory that the device is mounted on.
 *
 *	Returns the device or NULL if the device could not be found or
 *	an error occured.
 */
EDVDevice *edv_devices_list_match_mount_path(
	GList *devices_list,
	gint *n,
	const gchar *mount_path
)
{
	gint i;
	GList *glist;
	EDVDevice *d;

	if(n != NULL)
		*n = -1;

	if(STRISEMPTY(mount_path))
		return(NULL);

	/* Iterate through the devices list */
	for(glist = devices_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(d->mount_path == NULL)
			continue;

		if(!strcmp((const char *)d->mount_path, (const char *)mount_path))
		{
			if(n != NULL)
				*n = i;
			return(d);
		}
	}

	return(NULL);
}

/*
 *	Matches the Device by its device path.
 *
 *	The list and total specifies the devices list.
 *
 *	The n specifies the device index return value.
 *
 *	The device_path specifies the string describing the full path
 *	to the device.
 *
 *	Returns the device or NULL if the device could not be found or
 *	an error occured.
 */
EDVDevice *edv_devices_list_match_device_path(
	GList *devices_list,
	gint *n,
	const gchar *device_path
)
{
	gint i;
	GList *glist;
	EDVDevice *d;

	if(n != NULL)
		*n = -1;

	if(STRISEMPTY(device_path))
		return(NULL);

	/* Iterate through the devices list */
	for(glist = devices_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(d->device_path == NULL)
			continue;

		if(!strcmp((const char *)d->device_path, (const char *)device_path))
		{
			if(n != NULL)
				*n = i;
			return(d);
		}
	}

	return(NULL);
}

/*
 *	Matches the Device that the object is on.
 *
 *	The list and total specifies the devices list.
 *
 *	The n specifies the device index return value.
 *
 *	The path specifies the string describing the full path to the
 *	object.
 *
 *	Returns the device that the object reside on or NULL if the
 *	device could not be found or an error occured.
 */
EDVDevice *edv_devices_list_match_object(
	GList *devices_list,
	gint *n,
	const gchar *path
)
{
	gint i, matched_dev_num, matched_mount_path_len;
	GList *glist;
	EDVDevice *d, *matched_d;

	if(n != NULL)
		*n = -1;

	if(STRISEMPTY(path))
		return(NULL);

	matched_mount_path_len = 0;
	matched_dev_num = -1;
	matched_d = NULL;

	for(glist = devices_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		if(d->mount_path == NULL)
			continue;

		if(strpfx((const char *)path, (const char *)d->mount_path))
		{
			/* Was there a previously matched device? */
			if(matched_d != NULL)
			{
				/* Is this device's mount path longer than
				 * the previously matched device's mount path?
				 */
				const gint len = STRLEN(d->mount_path);
				if(len > matched_mount_path_len)
				{
					matched_mount_path_len = len;
					matched_dev_num = i;
					matched_d = d;
				}
			}
			else
			{
				matched_mount_path_len = STRLEN(d->mount_path);
				matched_dev_num = i;
				matched_d = d;
			}
		}
	}

	if(n != NULL)
		*n = matched_dev_num;

	return(matched_d);
}


/*
 *	Opens the devices found in the system (/etc/fstab) list of
 *	devices and appends them to the Devices list.
 */
GList *edv_devices_list_open_from_system(
	GList *devices_list,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
)
{
	FILE *fp;
	struct stat stat_buf;
#if defined(__FreeBSD__)
	struct fstab *fs_ptr;
#elif defined(__SOLARIS__)
	struct vfstab *vfs_ptr;
	int mtback;
#else
	struct mntent *mt_ptr;
#endif
	gulong file_size;
	EDVDevice *d;

	/* Open the system's devices list file */
#if defined(__FreeBSD__)
	setfsent();
	fp = fopen("/etc/fstab", "rb");
#elif defined(__SOLARIS__)
	fp = fopen("/etc/vfstab", "rb");
#else
	fp = setmntent("/etc/fstab", "rb");
#endif
	if(fp == NULL)
		return(devices_list);

	/* Get file statistics */
	if(fstat(fileno(fp), &stat_buf))
		file_size = 0l;
	else
		file_size = (gulong)stat_buf.st_size;

	/* Report initial progress */
	if(progress_cb != NULL)
		progress_cb(
			progress_data,
			0l, file_size
		);

	/* Begin reading the system's devices list file */
#if defined(__FreeBSD__)
	fs_ptr = getfsent();
	while(fs_ptr != NULL)
#elif defined(__SOLARIS__)
	vfs_ptr = (struct vfstab *)g_malloc(sizeof(struct vfstab));
	mtback = getvfsent(fp, vfs_ptr);
	while(mtback != 0)
#else
	mt_ptr = getmntent(fp);
	while(mt_ptr != NULL)
#endif
	{
		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				progress_data,
				(gulong)ftell(fp), file_size
			))
				break;
		}

		/* Ignore devices that are marked "none"
		 *
		 * Since many devices may be named "none" it can get
		 * confusing, let the Endeavour Devices List file
		 * describe these instead
		 */
#if defined(__FreeBSD__)
		if(!g_strcasecmp(fs_ptr->fs_spec, "none"))
		{
			/* Get the next mount entry */
			fs_ptr = getfsent();
			continue;
		}
#elif defined(__SOLARIS__)
		if(!g_strcasecmp(vfs_ptr->vfs_special, "none"))
		{
			/* Get the next mount entry */
			mtback = getmntent(fp, vfs_ptr);
			continue;
		}
#else
		if(!g_strcasecmp(mt_ptr->mnt_fsname, "none"))
		{
			/* Get the next mount entry */
			mt_ptr = getmntent(fp);
			continue;
		}
# ifdef MNTTYPE_IGNORE
		if(!g_strcasecmp(mt_ptr->mnt_type, MNTTYPE_IGNORE))
		{
			/* Get the next mount entry */
			mt_ptr = getmntent(fp);
			continue;
		}
# endif
#endif

		/* Check if an existing Device already exists in the
		 * Devices List that matches the device path
		 */
		d = edv_devices_list_match_device_path(
			devices_list,
			NULL,				/* No index return */
#if defined(__FreeBSD__)
			fs_ptr->fs_spec			/* Device Path */
#elif defined(__SOLARIS__)
			vfs_ptr->vfs_special		/* Device Path */
#else
			mt_ptr->mnt_fsname		/* Device Path */
#endif
		);
		if(d != NULL)
		{
			/* Found an existing Device, update its values */

			/* Filesystem Type */
			g_free(d->fs_type_name);
#if defined(__FreeBSD__)
			d->fs_type_name = STRDUP(fs_ptr->fs_vfstype);
#elif defined(__SOLARIS__)
			d->fs_type_name = STRDUP(vfs_ptr->vfs_fstype);
#else
			d->fs_type_name = STRDUP(mt_ptr->mnt_type);
#endif
			d->fs_type_code = edv_fs_type_get_code_from_name(d->fs_type_name);

			/* Mount Path */
			g_free(d->mount_path);
#if defined(__FreeBSD__)
			d->mount_path = STRDUP(fs_ptr->fs_file);
#elif defined(__SOLARIS__)
			d->mount_path = STRDUP(vfs_ptr->vfs_mountp);
#else
			d->mount_path = STRDUP(mt_ptr->mnt_dir);
#endif
		}
		else
		{
			/* No such Device exists in the Devices List, so
			 * create a new one
			 */
			d = edv_device_new();
			if(d != NULL)
			{
				/* Append this new Device to the Devices List */
				devices_list = g_list_append(
					devices_list,
					d
				);

				/* Get the device path, mount path, and filesystem */
#if defined(__FreeBSD__)
				d->device_path = STRDUP(fs_ptr->fs_spec);
				d->mount_path = STRDUP(fs_ptr->fs_file);
				d->fs_type_name = STRDUP(fs_ptr->fs_vfstype);
#elif defined(__SOLARIS__)
				d->device_path = STRDUP(vfs_ptr->vfs_special);
				d->mount_path = STRDUP(vfs_ptr->vfs_mountp);
				d->fs_type_name = STRDUP(vfs_ptr->vfs_fstype);
#else
				d->device_path = STRDUP(mt_ptr->mnt_fsname);
				d->mount_path = STRDUP(mt_ptr->mnt_dir);
				d->fs_type_name = STRDUP(mt_ptr->mnt_type);
#endif
				d->fs_type_code = edv_fs_type_get_code_from_name(d->fs_type_name);

				/* Get the options */
#if defined(__FreeBSD__)
				/* Not yet supported */
#elif defined(__SOLARIS__)
				/* Not yet supported */
#else
				if(mt_ptr->mnt_opts != NULL)
				{
					gchar **options_list = g_strsplit(
						(const gchar *)mt_ptr->mnt_opts,
						",",
						-1
					);
					if(options_list != NULL)
					{
						gint i;
						gchar *opt;
						for(i = 0; options_list[i] != NULL; i++)
						{
							opt = g_strstrip(options_list[i]);
							if(opt == NULL)
								continue;

							options_list[i] = opt;
#ifdef MNTOPT_DEFAULTS
							if(!g_strcasecmp(opt, MNTOPT_DEFAULTS))
#else
							if(FALSE)
#endif
							{

							}
#ifdef MNTOPT_RO
							else if(!g_strcasecmp(opt, MNTOPT_RO))
#else
							else if(FALSE)
#endif
							{
								d->flags |= EDV_DEVICE_READ_ONLY;
							}
#ifdef MNTOPT_RW
							else if(!g_strcasecmp(opt, MNTOPT_RW))
#else
							else if(FALSE)
#endif
							{
								d->flags &= ~EDV_DEVICE_READ_ONLY;
							}
#ifdef MNTOPT_SUID
							else if(!g_strcasecmp(opt, MNTOPT_SUID))
#else
							else if(FALSE)
#endif
							{

							}
#ifdef MNTOPT_NOSUID
							else if(!g_strcasecmp(opt, MNTOPT_NOSUID))
#else
							else if(FALSE)
#endif
							{

							}
#ifdef MNTOPT_NOAUTO
							else if(!g_strcasecmp(opt, MNTOPT_NOAUTO))
#else
							else if(FALSE)
#endif
							{

							}
						}

						g_strfreev(options_list);
					}
				}
#endif

				/* Set the default icons */
				if(EDV_DEVICE_TOTAL_ICON_STATES >= 1)
				{
					/* Small icon? */
					if(d->small_icon_path[0] == NULL)
					{
						d->small_icon_path[0] = g_strdup(
							EDV_NAME_DEF_DEVICE_ICON_SMALL_FILE
						);
					}
					/* Medium icon? */
					if(d->medium_icon_path[0] == NULL)
					{
						d->medium_icon_path[0] = g_strdup(
							EDV_NAME_DEF_DEVICE_ICON_MEDIUM_FILE
						);
					}
					/* Large icon? */
					if(d->large_icon_path[0] == NULL)
					{
						d->large_icon_path[0] = g_strdup(
							EDV_NAME_DEF_DEVICE_ICON_LARGE_FILE
						);
					}
				}

				/* Set the default fsck program */
				g_free(d->command_check);
				d->command_check = g_strdup_printf(
					"%s %s",
					EDV_PATH_PROG_DEF_FSCK_FRONT,
					d->device_path
				);

				/* Set the default format program */
				g_free(d->command_format);
				d->command_format = g_strdup_printf(
					"%s %s",
					EDV_PATH_PROG_DEF_FORMAT_FRONT,
					d->device_path
				);
			}
		}

		/* Get the next mount entry */
#if defined(__FreeBSD__)
		fs_ptr = getfsent();
#elif defined(__SOLARIS__)
		mtback = getmntent(fp, vfs_ptr);
#else
		mt_ptr = getmntent(fp);
#endif
	}


	/* Close the system's devices list file */
#if defined(__FreeBSD__)
	endfsent();
	(void)fclose(fp);
#elif defined(__SOLARIS__)
	(void)fclose(fp);
	g_free(vfs_ptr);
#else
	endmntent(fp);
#endif

	/* Report the final progress */
	if(progress_cb != NULL)
		progress_cb(
			progress_data,
			file_size, file_size
		);

	return(devices_list);
}


/*
 *	Updates the mount states on the Devices List.
 */
void edv_devices_list_update_mount_states(GList *devices_list)
{
	g_list_foreach(
		devices_list,
		(GFunc)edv_device_update_mount_state,
		NULL
	);
}


/*
 *	Updates the statistics of each device in Devices List.
 *
 *	If a Device is marked as not mounted then it will be skipped
 *	and its statistics will be set to 0's.
 *
 *	edv_devices_list_update_mount_states() should be called prior to
 *	calling this function to update the device's mount state.
 */
void edv_devices_list_update_statistics(GList *devices_list)
{
	g_list_foreach(
		devices_list,
		(GFunc)edv_device_update_statistics,
		NULL
	);
}


/*
 *	Opens the Devices List from the Devices List file.
 *
 *	The devices_list specifies the Devices List, a GList of
 *	EDVDevice * devices.
 *
 *	The path specifies the Devices List file.
 *
 *	The progress_cb and progress_data specifies the progress
 *	callback.
 *
 *	Any existing devices in the list that match (if their
 *	device_path matches) devices from the file will be updated.
 *
 *	Returns the History List or NULL on error.
 *
 *	Reminder: There is a sister function in lib/edv_devices_list.c
 *	which should perform equvilently to this function.
 */
GList *edv_devices_list_open(
	GList *devices_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
)
{
	FILE *fp;
	struct stat stat_buf;
	gchar *parm;
	gulong file_size;
	EDVDevice *d;

	if(STRISEMPTY(path))
		return(devices_list);

	/* Open the Devices List file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(devices_list);

	/* Get the Devices List file's statistics */
	if(fstat(fileno(fp), &stat_buf))
		file_size = 0l;
	else
		file_size = (gulong)stat_buf.st_size;

	/* Report the initial progress? */
	if(progress_cb != NULL)
		progress_cb(
			progress_data,
			0l, file_size
		);

	/* Begin reading the Devices List file */
	d = NULL;
	parm = NULL;
	while(TRUE)
	{
		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				progress_data,
				(gulong)ftell(fp), file_size
			))
				break;
		}

		/* Get the next parameter from the Devices List file */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Begin handling by parameter */

		/* BeginDevice */
		if(!g_strcasecmp(parm, "BeginDevice"))
		{
			gchar *device_path = FGetString(fp);

			/* Check if an existing Device already exists in the
			 * Devices List that matches the device path
			 */
			d = edv_devices_list_match_device_path(
				devices_list,
				NULL,			/* No index return */
				device_path
			);
			if(d == NULL)
			{
				/* No such Device exists, create a new one */
				d = edv_device_new();
				if(d != NULL)
				{
					d->device_path = STRDUP(device_path);

					/* Append the new Device to the Devices List */
					devices_list = g_list_append(
						devices_list,
						d
					);
				}
			}

			g_free(device_path);
		}
		/* FileSystemTypeName */
		else if(!g_strcasecmp(parm, "FileSystemTypeName"))
		{
			gchar *fs_type_name = FGetString(fp);
			if(d != NULL)
			{
				d->fs_type_name = STRDUP(fs_type_name);
				d->fs_type_code = edv_fs_type_get_code_from_name(d->fs_type_name);
			}
			g_free(fs_type_name);
		}
		/* NoUnmount */
		else if(!g_strcasecmp(parm, "NoUnmount"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_NO_UNMOUNT;
				else
					d->flags &= ~EDV_DEVICE_NO_UNMOUNT;
			}
		}
		/* ReadOnly */
		else if(!g_strcasecmp(parm, "ReadOnly"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_READ_ONLY;
				else
					d->flags &= ~EDV_DEVICE_READ_ONLY;
			}
		}
		/* Unlisted */
		else if(!g_strcasecmp(parm, "Unlisted"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_UNLISTED;
				else
					d->flags &= ~EDV_DEVICE_UNLISTED;
			}
		}
		/* NoScan */
		else if(!g_strcasecmp(parm, "NoScan"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_NO_SCAN;
				else
					d->flags &= ~EDV_DEVICE_NO_SCAN;
			}
		}
		/* Name */
		else if(!g_strcasecmp(parm, "Name"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->name);
				d->name = STRDUP(v);
			}
			g_free(v);
		}
		/* MountPath */
		else if(!g_strcasecmp(parm, "MountPath"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->mount_path);
				d->mount_path = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandMount */
		else if(!g_strcasecmp(parm, "CommandMount"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_mount);
				d->command_mount = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandUnmount */
		else if(!g_strcasecmp(parm, "CommandUnmount"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_unmount);
				d->command_unmount = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandEject */
		else if(!g_strcasecmp(parm, "CommandEject"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_eject);
				d->command_eject = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandCheck */
		else if(!g_strcasecmp(parm, "CommandCheck"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_check);
				d->command_check = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandTools */
		else if(!g_strcasecmp(parm, "CommandTools"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_tools);
				d->command_tools = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandFormat */
		else if(!g_strcasecmp(parm, "CommandFormat"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_format);
				d->command_format = STRDUP(v);
			}
			g_free(v);
		}

		/* IconSmallStandard */
		else if(!g_strcasecmp(parm, "IconSmallStandard"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->small_icon_path[
					EDV_DEVICE_ICON_STATE_STANDARD
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconSmallSelected */
		else if(!g_strcasecmp(parm, "IconSmallSelected"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->small_icon_path[
					EDV_DEVICE_ICON_STATE_SELECTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconSmallUnmounted */
		else if(!g_strcasecmp(parm, "IconSmallUnmounted"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->small_icon_path[
					EDV_DEVICE_ICON_STATE_UNMOUNTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}

		/* IconMediumStandard */
		else if(!g_strcasecmp(parm, "IconMediumStandard"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->medium_icon_path[
					EDV_DEVICE_ICON_STATE_STANDARD
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconMediumSelected */
		else if(!g_strcasecmp(parm, "IconMediumSelected"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->medium_icon_path[
				    EDV_DEVICE_ICON_STATE_SELECTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconMediumUnmounted */
		else if(!g_strcasecmp(parm, "IconMediumUnmounted"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->medium_icon_path[
					EDV_DEVICE_ICON_STATE_UNMOUNTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}

		/* IconLargeStandard */
		else if(!g_strcasecmp(parm, "IconLargeStandard"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->large_icon_path[
				    EDV_DEVICE_ICON_STATE_STANDARD
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconLargeSelected */
		else if(!g_strcasecmp(parm, "IconLargeSelected"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->large_icon_path[
					EDV_DEVICE_ICON_STATE_SELECTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconLargeUnmounted */
		else if(!g_strcasecmp(parm, "IconLargeUnmounted"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->large_icon_path[
					EDV_DEVICE_ICON_STATE_UNMOUNTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}

		/* LastMountTime */
		else if(!g_strcasecmp(parm, "LastMountTime"))
		{
			glong v[1];
			FGetValuesL(fp, v, 1);
			if(d != NULL)
				d->last_mount_time = (gulong)v[0];
		}
		/* LastCheckTime */
		else if(!g_strcasecmp(parm, "LastCheckTime"))
		{
			glong v[1];
			FGetValuesL(fp, v, 1);
			if(d != NULL)
				d->last_check_time = (gulong)v[0];
		}

		/* End of a device block? */
		else if(!g_strcasecmp(parm, "EndDevice"))
		{
			FSeekNextLine(fp);

			/* Reset the contexts */
			d = NULL;
		}
		/* All else unsupported parameter or wrong context */
		else
		{
			FSeekNextLine(fp);
		}
	}

	/* Delete the parameter */
	g_free(parm);

	/* Close the Devices List file */
	(void)fclose(fp);

	return(devices_list);
}

/*
 *	Saves the Devices List to the Devices List file.
 *
 *	The devices_list specifies the Devices List, a GList of
 *	EDVDevice * devices.
 *
 *	The path specifies the Devices List file.
 *
 *	The progress_cb and progress_data specifies the progress
 *	callback.
 *
 *	Reminder: There is a sister function in lib/edv_devices_list.c
 *	which should perform equvilently to this function.
 */
void edv_devices_list_file_save(
	GList *devices_list,
	const gchar *path,
	gint (*progress_cb)(gpointer, const gulong, const gulong),
	gpointer progress_data
)
{
	FILE *fp;
	gint		dev_num,
			ndevices;
	gchar *parent_path;
	const gchar *lpath;
	GList *glist;
	EDVDevice *d;

	if((devices_list == NULL) || STRISEMPTY(path))
		return;

	/* Get the parent directory and create it as needed */
	parent_path = g_dirname(path);
	if(parent_path != NULL)
	{
		edv_directory_create(
			parent_path,
			TRUE,				/* Create parents */
			NULL				/* No new paths list return */
		);
		g_free(parent_path);
	}

	/* Open the Devices List file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
		return;

	ndevices = g_list_length(devices_list);

	/* Report the initial progress */
	if(progress_cb != NULL)
		progress_cb(
			progress_data,
			0l, (gulong)ndevices
		);

	/* Save each Device in the Devices List */
	for(glist = devices_list, dev_num = 0;
		glist != NULL;
		glist = g_list_next(glist), dev_num++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		/* Skip devices that are marked internal, meaning they
		 * should not be saved to file
		 */
		if(EDV_DEVICE_IS_INTERNAL(d))
			continue;

		/* Report progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(
				progress_data,
				(gulong)(dev_num + 1), (gulong)ndevices
			))
				break;
		}


		/* BeginDevice */
		(void)fprintf(
			fp,
			"BeginDevice = %s\n",
			d->device_path
		);

		/* FileSystemTypeName */
		(void)fprintf(
			fp,
			"\tFileSystemTypeName = %s\n",
			d->fs_type_name
		);
		/* NoUnmount */
		(void)fprintf(
			fp,
			"\tNoUnmount = %i\n",
			EDV_DEVICE_IS_NO_UNMOUNT(d)
		);
		/* ReadOnly */
		(void)fprintf(
			fp,
			"\tReadOnly = %i\n",
			EDV_DEVICE_IS_READ_ONLY(d)
		);
		/* Unlisted */
		(void)fprintf(
			fp,
			"\tUnlisted = %i\n",
			EDV_DEVICE_IS_UNLISTED(d)
		);
		/* NoScan */
		(void)fprintf(
			fp,
			"\tNoScan = %i\n",
			EDV_DEVICE_IS_NO_SCAN(d)
		);
		/* Name */
		if(!STRISEMPTY(d->name))
			(void)fprintf(
				fp,
				"\tName = %s\n",
				d->name
			);
		/* MountPath */
		if(!STRISEMPTY(d->mount_path))
			(void)fprintf(
				fp,
				"\tMountPath = %s\n",
				d->mount_path
			);

		/* CommandMount */
		if(!STRISEMPTY(d->command_mount))
			(void)fprintf(
				fp,
				"\tCommandMount = %s\n",
				d->command_mount
			);
		/* CommandUnmount */
		if(!STRISEMPTY(d->command_unmount))
			(void)fprintf(
				fp,
				"\tCommandUnmount = %s\n",
				d->command_unmount
			);
		/* CommandEject */
		if(!STRISEMPTY(d->command_eject))
			(void)fprintf(
				fp,
				"\tCommandEject = %s\n",
				d->command_eject
			);
		/* CommandCheck */
		if(!STRISEMPTY(d->command_check))
			(void)fprintf(
				fp,
				"\tCommandCheck = %s\n",
				d->command_check
			);
		/* CommandTools */
		if(!STRISEMPTY(d->command_tools))
			(void)fprintf(
				fp,
				"\tCommandTools = %s\n",
				d->command_tools
			);
		/* CommandFormat */
		if(!STRISEMPTY(d->command_format))
			(void)fprintf(
				fp,
				"\tCommandFormat = %s\n",
				d->command_format
			);

		/* IconSmallStandard */
		lpath = d->small_icon_path[EDV_DEVICE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconSmallStandard = %s\n",
				lpath
			);
		/* IconSmallSelected */
		lpath = d->small_icon_path[EDV_DEVICE_ICON_STATE_SELECTED];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconSmallSelected = %s\n",
				lpath
			);
		/* IconSmallUnmounted */
		lpath = d->small_icon_path[EDV_DEVICE_ICON_STATE_UNMOUNTED];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconSmallUnmounted = %s\n",
				lpath
			);

		/* IconMediumStandard */
		lpath = d->medium_icon_path[EDV_DEVICE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconMediumStandard = %s\n",
				lpath
			);
		/* IconMediumSelected */
		lpath = d->medium_icon_path[EDV_DEVICE_ICON_STATE_SELECTED];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconMediumSelected = %s\n",
				lpath
			);
		/* IconMediumUnmounted */
		lpath = d->medium_icon_path[EDV_DEVICE_ICON_STATE_UNMOUNTED];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconMediumUnmounted = %s\n",
				lpath
			);

		/* IconLargeStandard */
		lpath = d->large_icon_path[EDV_DEVICE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconLargeStandard = %s\n",
				lpath
			);
		/* IconLargeSelected */
		lpath = d->large_icon_path[EDV_DEVICE_ICON_STATE_SELECTED];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconLargeSelected = %s\n",
				lpath
			);
		/* IconLargeUnmounted */
		lpath = d->large_icon_path[EDV_DEVICE_ICON_STATE_UNMOUNTED];
		if(!STRISEMPTY(lpath))
			(void)fprintf(
				fp,
				"\tIconLargeUnmounted = %s\n",
				lpath
			);

		/* LastMountTime */
		if(d->last_mount_time > 0l)
			(void)fprintf(
				fp,
				"\tLastMountTime = %ld\n",
				d->last_mount_time
			);
		/* LastCheckTime */
		if(d->last_check_time > 0l)
			(void)fprintf(
				fp,
				"\tLastCheckTime = %ld\n",
				d->last_check_time
			);

		/* EndDevice */
		(void)fprintf(
			fp,
			"EndDevice\n"
		);
	}

	/* Close the Devices List file */
	if(fclose(fp))
		return;

	/* Report the final progress */
	if(progress_cb != NULL)
		progress_cb(
			progress_data,
			(gulong)ndevices, (gulong)ndevices
		);
}
