#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "edv_device.h"
#include "edv_utils_gtk.h"
#include "edv_device_mount.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_mount_32x32.xpm"
#include "images/icon_unmount_32x32.xpm"
#include "images/icon_eject_32x32.xpm"


static gboolean edv_device_mount_message_is_empty(const gchar *s);

const gchar *edv_device_mount_get_error(EDVCore *core);

gint edv_device_mount(
	EDVCore *core,
	EDVDevice *d,
	const gboolean show_progress,
	const gboolean verbose,
	GtkWidget *toplevel
);
gint edv_device_unmount(
	EDVCore *core,
	EDVDevice *d,
	const gboolean show_progress,
	const gboolean verbose,
	GtkWidget *toplevel
);
gint edv_device_eject(
	EDVCore *core,
	EDVDevice *d,
	const gboolean show_progress,
	const gboolean verbose,
	GtkWidget *toplevel
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(p)       (((p) != NULL) ? (gint)fclose(p) : -1)

#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)


static gboolean edv_device_mount_message_is_empty(const gchar *s)
{
	if(s == NULL)
		return(TRUE);

	while(*s != '\0')
	{
		if(!isspace((int)*s))
			return(FALSE);
		s++;
	}

	return(TRUE);
}

/*
 *	Gets the last error message or NULL if there was no error.
 */
const gchar *edv_device_mount_get_error(EDVCore *core)
{
	if(core == NULL)
		return(NULL);

	return(core->last_error_ptr);
}


/*
 *	Mounts the Device.
 *
 *	The d specifies the Device to mount. The Device's mount state
 *	and last_mount_time will be updated.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	displayed during this operation.
 *
 *	If verbose is TRUE then any messages printed by the mount
 *	command will be displayed.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_device_mount(
	EDVCore *core,
	EDVDevice *d,
	const gboolean show_progress,
	const gboolean verbose,
	GtkWidget *toplevel
)
{
	FILE		*cstdout,
			*cstderr;
	gboolean user_aborted;
	gint pid;
	gchar		*shell_prog = NULL,
			*cmd = NULL,
			*mount_path = NULL,
			*device_path = NULL,
			*cstdout_msg = NULL,
			*cstderr_msg = NULL;
	const gchar	*shell_cmd,
			*shell_args,
			*fs_type_name;
	CfgList *cfg_list;

	if(core == NULL)
		return(-2);

	core->last_error_ptr = NULL;

	if(d == NULL)
	{
		core->last_error_ptr = "Invalid value.";
		return(-2);
	}

	if(core->op_level > 0)
	{
		core->last_error_ptr = "An operation is already in progress.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(shell_prog);			\
 g_free(cmd);				\
 g_free(mount_path);			\
 g_free(device_path);			\
 g_free(cstdout_msg);			\
 g_free(cstderr_msg);			\
					\
 core->op_level--;			\
					\
 return(_v_);				\
}
	cfg_list = core->cfg_list;

	/* Get the shell program and the shell arguments */
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
	shell_args = edv_strarg(
	    shell_cmd,
	    &shell_prog,
	    TRUE,				/* Parse escapes */
	    TRUE				/* Parse quotes */
	);

	/* Copy the Device's mount path */
	mount_path = STRDUP(d->mount_path);
	if(mount_path == NULL)
	{
		core->last_error_ptr = "Mount path not set.";
		CLEANUP_RETURN(-2);
	}

	/* Copy the Device's path */
	device_path = STRDUP(d->device_path);
	if(device_path == NULL)
	{
		core->last_error_ptr = "Device path not set.";
		CLEANUP_RETURN(-2);
	}

	/* Get file system type as a string */
	fs_type_name = d->fs_type_name;
	if(fs_type_name == NULL)
	{
		core->last_error_ptr = "Unable to obtain the filesystem type.";
		CLEANUP_RETURN(-2);
	}

	while(TRUE)
	{
		/* Format the mount command */
		g_free(cmd);
		if(STRISEMPTY(d->command_mount))
			cmd = g_strdup_printf(
				"\"%s\" \"%s\"%s",
				EDV_PATH_PROG_DEF_MOUNT,
				mount_path,
				EDV_DEVICE_IS_READ_ONLY(d) ? " -r" : ""
			);
		else
			cmd = g_strdup(d->command_mount);

		/* Execute the mount command */
		pid = edv_system_shell_streams(
			cmd,
			shell_prog,
			shell_args,
			NULL,
			&cstdout,
			&cstderr
		);
		if(pid < 0)
		{
			(void)FCLOSE(cstdout);
			(void)FCLOSE(cstderr);
			core->last_error_ptr =
"Execution of the mount command failed.";
			CLEANUP_RETURN(-1);
		}

		/* Map the progress dialog? */
		if(show_progress)
		{
			gchar *msg = g_strdup_printf(
"Mounting \"%s\" to:\n\
\n\
    %s\n",
				(d->name != NULL) ? d->name : device_path,
				mount_path
			);
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Mounting",
				msg,
				(const guint8 **)icon_mount_32x32_xpm,
				"Stop"
			);
			g_free(msg);
			ProgressDialogUpdateUnknown(
				NULL, NULL, NULL, NULL,
				TRUE
			);
			gdk_flush();
		}

		/* Monitor the mount process */
		user_aborted = FALSE;
		while(edv_pid_exists(pid))
		{
			/* Show progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				ProgressDialogUpdateUnknown(
					NULL, NULL, NULL, NULL,
					TRUE
				);
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					user_aborted = TRUE;
					break;
				}
			}
			/* Collect any messages from the output streams */
			cstdout_msg = edv_stream_read_strbuf(
				cstdout,
				cstdout_msg,
				FALSE		/* Nonblocking */
			);
			cstderr_msg = edv_stream_read_strbuf(
				cstderr,
				cstderr_msg,
				FALSE		/* Nonblocking */
			);
			edv_usleep(8000l);
		}
		/* Collect any tailing messages from the output streams */
		cstdout_msg = edv_stream_read_strbuf(
			cstdout,
			cstdout_msg,
			TRUE			/* Block until EOF */
		);
		cstderr_msg = edv_stream_read_strbuf(
			cstderr,
			cstderr_msg,
			TRUE			/* Block until EOF */
		);

		/* Close the output streams */
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);

		/* Unmap the progress dialog */
		if(show_progress)
		{
			ProgressDialogBreakQuery(TRUE);
			ProgressDialogSetTransientFor(NULL);
		}

		/* Update this device's mount state */
		edv_device_update_mount_state(d);

		/* If the user aborted then do not go on to check if the
		 * mount was successful or not
		 */
		if(user_aborted)
			break;

		/* Mount successful? */
		if(EDV_DEVICE_IS_MOUNTED(d))
		{
			/* Update the last mount time */
			d->last_mount_time = edv_time();

			/* Display any messages collected from the
			 * output streams
			 */
			if(verbose)
			{
				if(!edv_device_mount_message_is_empty(cstderr_msg))
				{
					edv_play_sound_warning(core);
					edv_message_warning(
						"Mount Warning",
						cstderr_msg,
						NULL,
						toplevel
					);
				}
				if(!edv_device_mount_message_is_empty(cstdout_msg))
				{
					edv_play_sound_info(core);
					edv_message(
						"Mount Message",
						cstdout_msg,
						NULL,
						toplevel
					);
				}
			}
		}
		else
		{
			/* Mount failed */
			if(verbose)
			{
				/* Query the user to try again */
				const cdialog_btn_flags	buttons = CDIALOG_BTNFLAG_RETRY | CDIALOG_BTNFLAG_ABORT,
							default_button = CDIALOG_BTNFLAG_RETRY;
				gint response;
				CDialogSetTransientFor(toplevel);
				if(!edv_device_mount_message_is_empty(cstderr_msg))
				{
					response = CDialogGetResponse(
						"Mount Failed",
						cstderr_msg,
						NULL,
						CDIALOG_ICON_WARNING,
						buttons,
						default_button
					);
				}
				else
				{
					gchar *msg = g_strdup_printf(
"Unable to mount \"%s\" to:\n\
\n\
    %s\n\
\n\
Try again?",
						(d->name != NULL) ? d->name : device_path,
						mount_path
					);
					response = CDialogGetResponse(
						"Mount Failed",
						msg,
						NULL,
						CDIALOG_ICON_WARNING,
						buttons,
						default_button
					);
					g_free(msg);
				}
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_RETRY)
				{
					/* Clear the message buffers */
					g_free(cstdout_msg);
					cstdout_msg = NULL;
					g_free(cstderr_msg);
					cstderr_msg = NULL;
					/* Retry */
					continue;
				}
			}
		}

		break;
	}

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}

/*
 *	Unmounts the device.
 *
 *	The d specifies the Device to mount, the Device's mount state
 *	will be updated.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	displayed during this operation.
 *
 *	If verbose is TRUE then any messages printed by the unmount
 *	command will be displayed.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_device_unmount(
	EDVCore *core,
	EDVDevice *d,
	const gboolean show_progress,
	const gboolean verbose,
	GtkWidget *toplevel
)
{
	FILE            *cstdout,
			*cstderr;
	gboolean user_aborted;
	gint pid;
	gchar		*shell_prog = NULL,
			*cmd = NULL,
			*mount_path = NULL,
			*device_path = NULL,
			*cstdout_msg = NULL,
			*cstderr_msg = NULL;
	const gchar	*shell_cmd,
			*shell_args;
	CfgList *cfg_list;

	if(core == NULL)
		return(-2);

	core->last_error_ptr = NULL;

	if(d == NULL)
	{
		core->last_error_ptr = "Invalid value.";
		return(-2);
	}

	/* Device not allowed to be unmounted? */
	if(EDV_DEVICE_IS_NO_UNMOUNT(d))
	{
		core->last_error_ptr = "The device is marked \"no unmount\".";
		return(-1);
	}

	if(core->op_level > 0)
	{
		core->last_error_ptr = "An operation is already in progress.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(shell_prog);			\
 g_free(cmd);				\
 g_free(mount_path);			\
 g_free(device_path);			\
 g_free(cstdout_msg);			\
 g_free(cstderr_msg);			\
					\
 core->op_level--;			\
					\
 return(_v_);				\
}

	cfg_list = core->cfg_list;

	/* Get the shell program and the shell arguments */
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
	shell_args = edv_strarg(
	    shell_cmd,
	    &shell_prog,
	    TRUE,				/* Parse escapes */
	    TRUE				/* Parse quotes */
	);

	/* Get copy of mount path */
	mount_path = STRDUP(d->mount_path);
	if(mount_path == NULL)
	{
		core->last_error_ptr = "Mount path not set.";
		CLEANUP_RETURN(-2);
	}

	/* Copy the device path */
	device_path = STRDUP(d->device_path);
	if(device_path == NULL)
	{
		core->last_error_ptr = "Device path not set.";
		CLEANUP_RETURN(-2);
	}

	while(TRUE)
	{
		/* Format the unmount command */
		g_free(cmd);
		if(STRISEMPTY(d->command_unmount))
			cmd = g_strdup_printf(
				"\"%s\" \"%s\"",
				EDV_PATH_PROG_DEF_UMOUNT,
				mount_path
			);
		else
			cmd = g_strdup(d->command_unmount);

		/* Execute the unmount command */
		pid = edv_system_shell_streams(
			cmd,
			shell_prog,
			shell_args,
			NULL,
			&cstdout,
			&cstderr
		);
		if(pid < 0)
		{
			(void)FCLOSE(cstdout);
			(void)FCLOSE(cstderr);
			core->last_error_ptr =
"Execution of the unmount command failed.";
			CLEANUP_RETURN(-1);
		}

		/* Map the progress dialog? */
		if(show_progress)
		{
			gchar *msg = g_strdup_printf(
"Unmounting \"%s\" from:\n\
\n\
    %s\n",
				(d->name != NULL) ? d->name : device_path,
				mount_path
			);
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Unmounting",
				msg,
				(const guint8 **)icon_unmount_32x32_xpm,
				"Stop"
			);
			g_free(msg);
			ProgressDialogUpdateUnknown(
				NULL, NULL, NULL, NULL,
				TRUE
			);
			gdk_flush();
		}

		/* Monitor the unmount process */
		user_aborted = FALSE;
		while(edv_pid_exists(pid))
		{
			/* Show progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				ProgressDialogUpdateUnknown(
					NULL, NULL, NULL, NULL,
					TRUE
				);
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					user_aborted = TRUE;
					break;
				}
			}
			/* Collect any messages from the output streams */
			cstdout_msg = edv_stream_read_strbuf(
				cstdout,
				cstdout_msg,
				FALSE		/* Nonblocking */
			);
			cstderr_msg = edv_stream_read_strbuf(
				cstderr,
				cstderr_msg,
				FALSE		/* Nonblocking */
			);
			edv_usleep(8000l);
		}
		/* Collect any tailing messages from the output streams */
		cstdout_msg = edv_stream_read_strbuf(
			cstdout,
			cstdout_msg,
			TRUE			/* Block until EOF */
		);
		cstderr_msg = edv_stream_read_strbuf(
			cstderr,
			cstderr_msg,
			TRUE			/* Block until EOF */
		);

		/* Close the output streams */
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);

		/* Unmap the progress dialog? */
		if(show_progress)
		{
			ProgressDialogBreakQuery(TRUE);
			ProgressDialogSetTransientFor(NULL);
		}

		/* Update this device's mount state */
		edv_device_update_mount_state(d);

		/* If the user aborted then do not go on to check if the
		 * unmount was successful or not
		 */
		if(user_aborted)
			break;

		/* Unmount successful? */
		if(!EDV_DEVICE_IS_MOUNTED(d))
		{
			/* Update the last mount time */
			d->last_mount_time = edv_time();

			/* Display any messages collected from the
			 * output streams
			 */
			if(verbose)
			{
				if(!edv_device_mount_message_is_empty(cstderr_msg))
				{
					edv_play_sound_warning(core);
					edv_message_warning(
						"Mount Warning",
						cstderr_msg,
						NULL,
						toplevel
					);
				}
				if(!edv_device_mount_message_is_empty(cstdout_msg))
				{
					edv_play_sound_info(core);
					edv_message(
						"Mount Message",
						cstdout_msg,
						NULL,
						toplevel
					);
				}
			}
		}
		else
		{
			/* Unmount failed */
			if(verbose)
			{
				/* Query the user to try again */
				const cdialog_btn_flags	buttons = CDIALOG_BTNFLAG_RETRY | CDIALOG_BTNFLAG_ABORT,
							default_button = CDIALOG_BTNFLAG_RETRY;
				gint response;
				CDialogSetTransientFor(toplevel);
				if(!edv_device_mount_message_is_empty(cstderr_msg))
				{
					response = CDialogGetResponse(
						"Mount Failed",
						cstderr_msg,
						NULL,
						CDIALOG_ICON_WARNING,
						buttons,
						default_button
					);
				}
				else
				{
					gchar *msg = g_strdup_printf(
"Unable to unmount \"%s\" from:\n\
\n\
    %s\n\
\n\
Try again?",
						(d->name != NULL) ? d->name : device_path,
						mount_path
					);
					response = CDialogGetResponse(
						"Unmount Failed",
						msg,
						NULL,
						CDIALOG_ICON_WARNING,
						buttons,
						default_button
					);
					g_free(msg);
				}
				CDialogSetTransientFor(NULL);
				if(response == CDIALOG_RESPONSE_RETRY)
				{
					/* Clear the message buffers */
					g_free(cstdout_msg);
					cstdout_msg = NULL;
					g_free(cstderr_msg);
					cstderr_msg = NULL;
					/* Retry */
					continue;
				}
			}
		}

		break;
	}

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}

/*
 *	Ejects the media from the Device.
 *
 *	If the Device is already mounted then edv_device_unmount() must
 *	be called first or else this operation will fail.
 *
 *	The dev specifies the Device with the media to eject.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	displayed during this operation.
 *
 *	If verbose is TRUE then any messages printed by the eject
 *	command will be displayed.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_device_eject(
	EDVCore *core,
	EDVDevice *d,
	const gboolean show_progress,
	const gboolean verbose,
	GtkWidget *toplevel
)
{
	FILE		*cstdout,
			*cstderr;
	gboolean user_aborted;
	gint pid;
	gchar		*shell_prog = NULL,
			*cmd = NULL,
			*device_path = NULL,
			*cstdout_msg = NULL,
			*cstderr_msg = NULL;
	const gchar	*shell_cmd,
			*shell_args;
	CfgList *cfg_list;

	if(core == NULL)
		return(-2);

	core->last_error_ptr = NULL;

	if(d == NULL)
	{
		core->last_error_ptr = "Invalid value.";
		return(-2);
	}

	/* Media not allowed to be ejected from device? */
	if(EDV_DEVICE_IS_NO_UNMOUNT(d))
	{
		core->last_error_ptr = "The device is marked \"no unmount\".";
		return(-1);
	}

	if(core->op_level > 0)
	{
		core->last_error_ptr = "An operation is already in progress.";
		return(-6);
	}

	core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(shell_prog);			\
 g_free(cmd);				\
 g_free(device_path);			\
 g_free(cstdout_msg);			\
 g_free(cstderr_msg);			\
					\
 core->op_level--;			\
					\
 return(_v_);				\
}

	cfg_list = core->cfg_list;

	/* Get the shell program and the shell arguments */
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
	shell_args = edv_strarg(
	    shell_cmd,
	    &shell_prog,
	    TRUE,				/* Parse escapes */
	    TRUE				/* Parse quotes */
	);

	/* Copy the Device's path */
	device_path = STRDUP(d->device_path);
	if(device_path == NULL)
	{
		core->last_error_ptr = "Device path not set.";
		CLEANUP_RETURN(-2);
	}

	while(TRUE)
	{
		/* Format the eject command */
		g_free(cmd);
		if(STRISEMPTY(d->command_eject))
			cmd = g_strdup_printf(
				"\"%s\" \"%s\"",
				EDV_PATH_PROG_DEF_EJECT,
				device_path
			);
		else
			cmd = g_strdup(d->command_eject);

		/* Execute the eject command */
		pid = edv_system_shell_streams(
			cmd,
			shell_prog,
			shell_args,
			NULL,
			&cstdout,
			&cstderr
		);
		if(pid < 0)
		{
			(void)FCLOSE(cstdout);
			(void)FCLOSE(cstderr);
			core->last_error_ptr =
"Execution of the eject command failed.";
			CLEANUP_RETURN(-1);
		}

		/* Map the progress dialog? */
		if(show_progress)
		{
			gchar *msg = g_strdup_printf(
"Ejecting \"%s\"\n",
				(d->name != NULL) ? d->name : device_path
			);
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Ejecting",
				msg,
				(const guint8 **)icon_eject_32x32_xpm,
				"Stop"
			);
			g_free(msg);
			ProgressDialogUpdateUnknown(
				NULL, NULL, NULL, NULL,
				TRUE
			);
			gdk_flush();
		}

		/* Monitor the eject process */
		user_aborted = FALSE;
		while(edv_pid_exists(pid))
		{
			/* Show progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				ProgressDialogUpdateUnknown(
					NULL, NULL, NULL, NULL,
					TRUE
				);
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					user_aborted = TRUE;
					break;
				}
			}
			/* Collect any messages from the output streams */
			cstdout_msg = edv_stream_read_strbuf(
				cstdout,
				cstdout_msg,
				FALSE		/* Nonblocking */
			);
			cstderr_msg = edv_stream_read_strbuf(
				cstderr,
				cstderr_msg,
				FALSE		/* Nonblocking */
			);
			edv_usleep(8000l);
		}
		/* Collect any tailing messages from the output streams */
		cstdout_msg = edv_stream_read_strbuf(
			cstdout,
			cstdout_msg,
			TRUE			/* Block until EOF */
		);
		cstderr_msg = edv_stream_read_strbuf(
			cstderr,
			cstderr_msg,
			TRUE			/* Block until EOF */
		);

		/* Close the output streams */
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);

		/* Unmap the progress dialog? */
		if(show_progress)
		{
			ProgressDialogBreakQuery(TRUE);
			ProgressDialogSetTransientFor(NULL);
		}

		/* If the user aborted then do not go on to check if the
		 * eject was successful or not
		 */
		if(user_aborted)
			break;

		/* Display any messages collected from the output streams */
		if(verbose)
		{
		        if(!edv_device_mount_message_is_empty(cstderr_msg))
			{
				edv_play_sound_warning(core);
				edv_message_warning(
					"Mount Warning",
					cstderr_msg,
					NULL,
					toplevel
				);
			}
			if(!edv_device_mount_message_is_empty(cstdout_msg))
			{
				edv_play_sound_info(core);
				edv_message(
					"Mount Message",
					cstdout_msg,
					NULL,
					toplevel
				);
			}
		}

		break;
	}

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}
