#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(__FreeBSD__)
# include <fstab.h>
# include <sys/param.h>
# include <sys/ucred.h>
# include <sys/mount.h>
#elif defined(__SOLARIS__)
# include <sys/vfs.h>
# include <sys/mnttab.h>
# include <sys/vfstab.h>
# include <sys/statvfs.h>
#else
# include <mntent.h>
# include <sys/vfs.h>
#endif
#include <gtk/gtk.h>

#include "edv_types.h"
#include "edv_pixmap.h"
#include "edv_device.h"

#include "config.h"


/* Mount State */
void edv_device_update_mount_state(EDVDevice *d);

/* Statistics */
void edv_device_update_statistics(EDVDevice *d);

/* Devices */
EDVDevice *edv_device_new(void);
EDVDevice *edv_device_copy(EDVDevice *d);
static void edv_device_load_icons_nexus(
	EDVPixmap **icons_list,
	edv_pixmap_data **datas_list,
	const gchar **paths_list,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
);
void edv_device_realize(
	EDVDevice *d,
	const gboolean force_rerealize
);
void edv_device_unrealize(EDVDevice *d);
void edv_device_delete(EDVDevice *d);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Updates the Device's mount state.
 *
 *	Reminder: There is a version of this function in
 *	lib/edv_device.c that needs to perform equvilently to this
 *	function.
 */
void edv_device_update_mount_state(EDVDevice *d)
{
#if defined(__FreeBSD__)
	gint		i,
					mntsize;
	struct statfs *mt_buf;
#elif defined(__SOLARIS__)
	FILE *fp;
	gint mtback;
	struct mnttab *mt_ptr;
#else
	FILE *fp;
	struct mntent *mt_ptr;
#endif
	const gchar	*device_path,
					*mount_path;

	if(d == NULL)
		return;

	/* Reset the device's mount state */
	d->flags &= ~EDV_DEVICE_MOUNTED;

	/* Open the system's mounted devices list file */
#if defined(__FreeBSD__)
	/* Do nothing */
#elif defined(__SOLARIS__)
	fp = fopen(PATH_ETC_MNTTAB, "rb");
	if(fp == NULL)
		return;
#else
	fp = setmntent(PATH_PROC_MOUNTS, "rb");
	if(fp == NULL)
		return;
#endif

	/* Begin reading the system's mounted devices list file */
#if defined(__FreeBSD__)
	mntsize = getmntinfo(
		&mt_buf,
		MNT_NOWAIT
	);
	if(mntsize == 0)
		return;
	i = mntsize - 1;
	while(i >= 0)
#elif defined(__SOLARIS__)
	mt_ptr = (struct mnttab *)g_malloc(sizeof(struct mnttab));
	mtback = getmntent(
		fp,
		mt_ptr
	);
	while(mtback != 0)
#else
	mt_ptr = getmntent(fp);
	while(mt_ptr != NULL)
#endif
	{
		/* Check if this mount path's device path matches this
		 * Device's device path
		 */
		device_path = (const gchar *)(
#if defined(__FreeBSD__)
			mt_buf[i].f_mntfromname
#elif defined(__SOLARIS__)
			mt_ptr->mnt_special
#else
			mt_ptr->mnt_fsname
#endif
		);
		if(!STRISEMPTY(device_path) && !STRISEMPTY(d->device_path))
		{
			if(!strcmp(
				(const char *)device_path,
				(const char *)d->device_path)
			)
			{
				/* Mark this device as mounted */
				d->flags |= EDV_DEVICE_MOUNTED;
				break;
			}
		}
		 
		/* Also check if this mount path's mounted path matches
		 * this Device's mounted path
		 */
		mount_path = (const gchar *)(
#if defined(__FreeBSD__)
			mt_buf[i].f_mntonname
#elif defined(__SOLARIS__)
			mt_ptr->mnt_mountp
#else
			mt_ptr->mnt_dir
#endif
		);
		if(!STRISEMPTY(d->mount_path) && !STRISEMPTY(mount_path))
		{
			if(!strcmp(
				(const char *)d->mount_path,
				(const char *)mount_path)
			)
			{
				/* Mark this device as mounted */
				d->flags |= EDV_DEVICE_MOUNTED;
				break;
			}
		}

		/* Get the next mount entry */
#if defined(__FreeBSD__)
		i--;
#elif defined(__SOLARIS__)
		mtback = getmntent(fp, mt_ptr);
#else
		mt_ptr = getmntent(fp);
#endif
	}

	/* Close the system's mounted devices list file */
#if defined(__FreeBSD__)
	/* Do nothing */
#elif defined(__SOLARIS__)
	(void)fclose(fp);
	g_free(mt_ptr);
#else
	endmntent(fp);
#endif
}


/*
 *	Updates the Device's statistics.
 *
 *	If a Device is marked as not mounted then it will be skipped
 *	and its statistics will be set to 0's.
 *
 *	edv_device_update_mount_state() should be called prior to
 *	calling this function to update the device's mount state.
 *
 *	Reminder: There is a version of this function in
 *	lib/edv_device.c that needs to perform equvilently to this
 *	function.
 */
void edv_device_update_statistics(EDVDevice *d)
{
	const gchar *mount_path;

	if(d == NULL)
		return;

	mount_path = d->mount_path;
	if(EDV_DEVICE_IS_MOUNTED(d) && !STRISEMPTY(mount_path))
	{
#if defined(__SOLARIS__)
		struct statvfs buf;
		if(!statvfs((const char *)mount_path, &buf))
#else
		struct statfs buf;
		if(!statfs((const char *)mount_path, &buf))
#endif
		{
			/* Check if the block size transfer rate (which is
			 * really just the block size), is larger than or
			 * equal to the base of 1024 bytes per block
			 */
			const gulong block_size = (gulong)buf.f_bsize;
			if(block_size >= 1024l)
			{
				const gulong block_div = block_size / 1024l;

				d->blocks_total = (gulong)buf.f_blocks * block_div;
				d->blocks_available = (gulong)buf.f_bavail * block_div;
				d->blocks_free = (gulong)buf.f_bfree * block_div;
			}
			else if(block_size > 0l)
			{
				/* Block size is less than 1024 bytes but positive */
				const gulong block_div = 1024l / block_size;

				/* Note, block_div is in range of 1 to 1024 */
				d->blocks_total = (gulong)buf.f_blocks / block_div;
				d->blocks_available = (gulong)buf.f_bavail / block_div;
				d->blocks_free = (gulong)buf.f_bfree / block_div;
			}

			d->block_size = block_size;

			d->indicies_total = (gulong)buf.f_files;
#if defined(__SOLARIS__)
			d->indicies_available = (gulong)buf.f_favail;
#else
			d->indicies_available = (gulong)buf.f_ffree;
#endif
			d->indicies_free = (gulong)buf.f_ffree;

#if defined(__linux__)
/*		d->fs_id = (guint)*(int *)(&buf.f_fsid); */
#endif
			d->name_length_max = (gulong)buf.f_namelen;
		}
	}
	else
	{
		/* Reset this Device's statistics */
		d->blocks_total = 0l;
		d->blocks_available = 0l;
		d->blocks_free = 0l;
		d->block_size = 0l;
		d->indicies_total = 0l;
		d->indicies_available = 0l;
		d->indicies_free = 0l;
		d->name_length_max = 0l;
	}
}


/*
 *	Creates a new Device.
 */
EDVDevice *edv_device_new(void)
{
	return(EDV_DEVICE(g_malloc0(sizeof(EDVDevice))));
}

/*
 *	Coppies the Device.
 */
EDVDevice *edv_device_copy(EDVDevice *d)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	gint i;
	EDVDevice	*tar,
							*src = d;
	if(src == NULL)
		return(NULL);

	tar = edv_device_new();
	if(tar == NULL)
		return(NULL);

	tar->flags = src->flags;
	tar->name = STRDUP(src->name);
	tar->device_path = STRDUP(src->device_path);
	tar->mount_path = STRDUP(src->mount_path);
	tar->fs_type_name = STRDUP(src->fs_type_name);
	tar->fs_type_code = src->fs_type_code;

	for(i = 0; i < nicon_states; i++)
	{
		/* Small */
		tar->small_icon[i] = edv_pixmap_ref(src->small_icon[i]);
		tar->small_icon_path[i] = STRDUP(src->small_icon_path[i]);

		/* Medium */
		tar->medium_icon[i] = edv_pixmap_ref(src->medium_icon[i]);
		tar->medium_icon_path[i] = STRDUP(src->medium_icon_path[i]);

		/* Large */
		tar->large_icon[i] = edv_pixmap_ref(src->large_icon[i]);
		tar->large_icon_path[i] = STRDUP(src->large_icon_path[i]);
	}

	tar->command_mount = STRDUP(src->command_mount);
	tar->command_unmount = STRDUP(src->command_unmount);
	tar->command_eject = STRDUP(src->command_eject);

	tar->command_check = STRDUP(src->command_check);
	tar->command_tools = STRDUP(src->command_tools);
	tar->command_format = STRDUP(src->command_format);

	tar->blocks_total = src->blocks_total;
	tar->blocks_available = src->blocks_available;
	tar->blocks_free = src->blocks_free;

	tar->block_size = src->block_size;

	tar->indicies_total = src->indicies_total;
	tar->indicies_available = src->indicies_available;
	tar->indicies_free = src->indicies_free;

	tar->name_length_max = src->name_length_max;

	tar->last_mount_time = src->last_mount_time;
	tar->last_check_time = src->last_check_time;

	return(tar);
}

/*
 *	Loads the pixmap and mask pairs list.
 *
 *	The pixmaps_list and masks_list specifies the pixmap and mask
 *	pairs list, any existing (non NULL) pixmap and mask pairs in
 *	this list will be unref'ed first.
 *
 *	The data specifies the XPM data. If data is not NULL then the
 *	pixmap and mask pair will be loaded from the XPM data.
 *
 *	The file specifies the path to the XPM file. If file is not NULL
 *	then the pixmap and mask pair will be loaded from the XPM file.
 *
 *	If both data and file are NULL then the existing pixmap and
 *	mask pair will be unref'ed and no new pixmap and mask pair will
 *	be loaded.
 *
 *	The req_width and req_height specifies the requested size in
 *	pixels.
 *
 *	If allow_resize is TRHE then the pixmap and mask pair will be
 *	resized to the requested size as needed.
 */
static void edv_device_load_icons_nexus(
	EDVPixmap **icons_list,
	edv_pixmap_data **datas_list,
	const gchar **paths_list,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	gint i;

	/* Unref all the existing icons in the specified list */
	for(i = 0; i < nicon_states; i++)
		icons_list[i] = edv_pixmap_unref(icons_list[i]);

	/* Load the icons from data? */
	if(datas_list != NULL)
	{
		EDVPixmap *p;

		/* Load each icon from data */
		for(i = 0; i < nicon_states; i++)
		{
			/* Open a new EDVPixmap from this data
			 *
			 * We do not use edv_load_pixmap_from_data() because it
			 * may give us an existing EDVPixmap and we need to
			 * potentially resize it which would cause problems
			 * for other parts of the code that want the original
			 * size of the EDVPixmap
			 */
			icons_list[i] = p = edv_pixmap_new_from_data(
				datas_list[i],
				NULL			/* No name */
			);
			if(p == NULL)
				continue;

			if(allow_resize)
				(void)edv_pixmap_resize(
					p,
					req_width, req_height
				);
		}
	}
	/* Open the icons from file? */
	else if(paths_list != NULL)
	{
		EDVPixmap *p;

		/* Open each icon from file */
		for(i = 0; i < nicon_states; i++)
		{
			/* Open a new EDVPixmap from this file
			 *
			 * We do not use edv_open_pixmap_from_file() because it
			 * may give us an existing EDVPixmap and we need to
			 * potentially resize it which would cause problems
			 * for other parts of the code that want the original
			 * size of the EDVPixmap
			 */
			icons_list[i] = p = edv_pixmap_new_from_file(paths_list[i]);
			if(p == NULL)
				continue;

			if(allow_resize)
				(void)edv_pixmap_resize(
					p,
					req_width, req_height
				);
		}
	}
}

/*
 *	Opens all the icon EDVPixmaps from their respective icon file
 *	paths on the Device.
 *
 *	If force_rerealize is TRUE and the Device is already realized
 *	then all the existing icon EDVPixmaps will be unrefed and then
 *	reopened from their respective icon file paths.
 */
void edv_device_realize(
	EDVDevice *d,
	const gboolean force_rerealize
)
{
	const gchar **paths_list;

	if(d == NULL)
		return;

	/* Not forcing re-realize? */
	if(!force_rerealize)
	{
		/* Skip if this Device was already realized */
		if(EDV_DEVICE_IS_REALIZED(d))
			return;
	}

	/* Begin realizing this Device */

	/* (Re)open the icons from their icon files
	 *
	 * Small icons
	 */
	paths_list = (const gchar **)d->small_icon_path;
	if(paths_list != NULL)
		edv_device_load_icons_nexus(
			d->small_icon,
			NULL,			/* No data */
			paths_list,
			20, 20,
			TRUE			/* Allow resize */
		);

	/* Medium icons */
	paths_list = (const gchar **)d->medium_icon_path;
	if(paths_list != NULL)
		edv_device_load_icons_nexus(
			d->medium_icon,
			NULL,			/* No data */
			paths_list,
			32, 32,
			TRUE			/* Allow resize */
		);

	/* Large icons */
	paths_list = (const gchar **)d->large_icon_path;
	if(paths_list != NULL)
		edv_device_load_icons_nexus(
			d->large_icon,
			NULL,			/* No data */
			paths_list,
			48, 48,
			TRUE			/* Allow resize */
		);

	/* Mark this Device as realized */
	d->flags |= EDV_DEVICE_REALIZED;
}

/*
 *	Unrefs all the icon EDVPixmaps on the Device.
 */
void edv_device_unrealize(EDVDevice *d)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	gint i;

	if(d == NULL)
		return;

	/* Skip if not already marked as realized */
	if(!EDV_DEVICE_IS_REALIZED(d))
		return;

	for(i = 0; i < nicon_states; i++)
	{
		d->small_icon[i] = edv_pixmap_unref(d->small_icon[i]);
		d->medium_icon[i] = edv_pixmap_unref(d->medium_icon[i]);
		d->large_icon[i] = edv_pixmap_unref(d->large_icon[i]);
	}

	/* Mark this Device as not realized */
	d->flags &= ~EDV_DEVICE_REALIZED;
}

/*
 *	Deletes the Device.
 */
void edv_device_delete(EDVDevice *d)
{
	const gint nicon_states = EDV_DEVICE_TOTAL_ICON_STATES;
	gint i;

	if(d == NULL)
		return;

	for(i = 0; i < nicon_states; i++)
	{
		d->small_icon[i] = edv_pixmap_unref(d->small_icon[i]);
		g_free(d->small_icon_path[i]);

		d->medium_icon[i] = edv_pixmap_unref(d->medium_icon[i]);
		g_free(d->medium_icon_path[i]);

		d->large_icon[i] = edv_pixmap_unref(d->large_icon[i]);
		g_free(d->large_icon_path[i]);
	}

	g_free(d->command_mount);
	g_free(d->command_unmount);
	g_free(d->command_eject);

	g_free(d->command_check);
	g_free(d->command_tools);
	g_free(d->command_format);

	g_free(d->name);
	g_free(d->device_path);
	g_free(d->mount_path);
	g_free(d->fs_type_name);

	g_free(d);
}
