#include <string.h>
#include <time.h>
#include <glib.h>

#include "edv_date_format.h"


gchar *edv_date_string_format(
	const gulong t,
	const gchar *format,
	const EDVDateRelativity relativity
);
gchar *edv_date_format_duration(const gulong dt);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Format a date and time string.
 *
 *	The t specifies the time since EPOCH in seconds. If t is 0
 *	then an empty string will be returned.
 *
 *	The format specifies the date and time format string. If format
 *	is NULL then ctime() will be used to format the date and time
 *	string.
 *
 *	The relativity can be either EDV_DATE_RELATIVITY_ABSOLUTE or
 *	EDV_DATE_RELATIVITY_CURRENT. If the relativity is
 *	EDV_DATE_RELATIVITY_CURRENT then the format will be
 *	"<n> <month(s)/day(s)/hour(s)/minute(s)/second(s)> ago",
 *	except if t >= 6 months then EDV_DATE_RELATIVITY_ABSOLUTE
 *	will be used as the relativity.
 *
 *	Returns a dynamically allocated string describing the
 *	formatted date and time.
 */
gchar *edv_date_string_format(
	const gulong t,
	const gchar *format,
	const EDVDateRelativity relativity
)
{
	/* Handle by relativity */
	switch(relativity)
	{
	  case EDV_DATE_RELATIVITY_CURRENT:
		if(t > 0l)
		{
			const gulong dt = (gulong)time(NULL) - t;
			gulong ct;

			/* Less than one second? */
			if(dt < 1l)
			{
				return(g_strdup(
"less than a second ago"
				));
			}
			/* Less than one minute? */
			else if(dt < (1l * 60l))
			{
				ct = MAX((dt / 1l), 1l);
				return(g_strdup_printf(
					"%ld %s ago",
					ct, ((ct == 1l) ? "second" : "seconds")
				));
			}
			/* Less than one hour? */
			else if(dt < (60l * 60l))
			{
				ct = MAX((dt / 60l), 1l);
				return(g_strdup_printf(
					"%ld %s ago",
					ct, ((ct == 1l) ? "minute" : "minutes")
				));
			}
			/* Less than one day? */
			else if(dt < (24l * 60l * 60l))
			{
				ct = MAX((dt / 60l / 60l), 1l);
				return(g_strdup_printf(
					"%ld %s ago",
					ct, ((ct == 1l) ? "hour" : "hours")
				));
			}
			/* Less than one week? */
			else if(dt < (7l * 24l * 60l * 60l))
			{
				ct = MAX((dt / 60l / 60l / 24l), 1l);
				return(g_strdup_printf(
					"%ld %s ago",
					ct, ((ct == 1l) ? "day" : "days")
				));
			}
			/* Less than one month (30 days)? */
			else if(dt < (30l * 24l * 60l * 60l))
			{
				ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
				return(g_strdup_printf(
					"%ld %s ago",
					ct, ((ct == 1l) ? "week" : "weeks")
				));
			}
			/* Less than 6 months ago? */
			else if(dt < (6l * 30l * 24l * 60l * 60l))
			{
				ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
				return(g_strdup_printf(
					"%ld %s ago",
					ct, ((ct == 1l) ? "month" : "months")
				));
			}
			/* 6 months or longer ago */
			else
			{
				/* Recurse to return a date and time string
				 * describing absolute relativity
				 */
				return(edv_date_string_format(
					t,
					format,
					EDV_DATE_RELATIVITY_ABSOLUTE
				));
			}
		}
		break;

	  case EDV_DATE_RELATIVITY_ABSOLUTE:
		/* Date and time format string specified? */
		if(format != NULL)
		{
			time_t tv = (time_t)t;
			const struct tm *tm_ptr = localtime(&tv);
			if(tm_ptr != NULL)
			{
		                /* Calculate the length of the return
				 * string based on the number of
				 * tokens encountered
		                 */
		                gint len = 0;
		                const gchar *s = format;
				gchar *buf;
		                while(*s != '\0')
		                {
                		        if(*s == '%')
		                                len += 80;
		                        else
		                                len++;
		                        s++;
		                }
		                len++;		/* Count the null byte */

		                /* Allocate the return string */
		                buf = (gchar *)g_malloc(len * sizeof(gchar));
		                if(buf == NULL)
		                        return(NULL);

		                /* Format the date and time string
				 * relative to the Common Era of the
				 * current locale
		                 */
				if(strftime(
					(char *)buf, (size_t)len,
					(const char *)format,
					tm_ptr
				) <= 0l)
					*buf = '\0';

				return(buf);
			}
			else
			{
				return(g_strdup(""));
			}
		}
		else
		{
			/* No format string specified so use ctime() */
			time_t t2 = (time_t)t;
			gchar *buf = g_strdup(ctime(&t2));
			if(buf != NULL)
			{
				gchar *s = (gchar *)strchr(
					(char *)buf,
					'\n'
				);
				if(s != NULL)
					*s = '\0';
			}
			return(buf);
		}
		break;
	}

	return(g_strdup(""));
}

/*
 *	Formats a date and time string describing the time lapsed in
 *	the format of "<n1> <units1> <n2> <units2>".
 *
 *	The dt specifies the time lapsed in seconds. If dt is 0 then
 *	"less than a second" will be returned.
 *
 *	Returns a dynamically allocated string describing the time
 *	lapsed.
 */
gchar *edv_date_format_duration(const gulong dt)
{
	gulong		ct,
			ct2;

	/* Less than one second? */
	if(dt < 1l)
	{
		return(g_strdup(
"less than a second"
		));
	}
	/* Less than one minute? */
	else if(dt < (1l * 60l))
	{
		ct = MAX((dt / 1l), 1l);
		return(g_strdup_printf(
			"%ld %s",
			ct, ((ct == 1l) ? "second" : "seconds")
		));
	}
	/* Less than one hour? */
	else if(dt < (60l * 60l))
	{
		ct = MAX((dt / 60l), 1l);
		ct2 = MAX((dt / 1l), 1l) % 60l;
		if(ct2 != 0l)
			return(g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "minute" : "minutes"),
				ct2, ((ct2 == 1l) ? "second" : "seconds")
			));
		else
			return(g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "minute" : "minutes") 
			));
	}
	/* Less than one day? */
	else if(dt < (24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l), 1l);
		ct2 = MAX((dt / 60l), 1l) % 60l;
		if(ct2 != 0l)
			return(g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "hour" : "hours"),
				ct2, ((ct2 == 1l) ? "minute" : "minutes")
			));
		else
			return(g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "hour" : "hours")
			));
	}
	/* Less than one week? */
	else if(dt < (7l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l), 1l);
		ct2 = MAX((dt / 60l / 60l), 1l) % 24l;
		if(ct2 != 0l)
			return(g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "day" : "days"),
				ct2, ((ct2 == 1l) ? "hour" : "hours")
			));
		else
			return(g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "day" : "days") 
			));
	}
	/* Less than one month (30 days)? */
	else if(dt < (30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 7l;
		if(ct2 != 0l)
			return(g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "week" : "weeks"),
				ct2, ((ct2 == 1l) ? "day" : "days")
			));
		else
			return(g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "week" : "weeks")
			));
	}
#if 0
	/* Less than 6 months ago? */
	else if(dt < (6l * 30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
		return(g_strdup_printf(
			"%ld %s %ld %s",
			ct, ((ct == 1l) ? "month" : "months"),
			ct2, ((ct2 == 1l) ? "day" : "days")
		));
	}
#endif
	/* Less than a year (365 days)? */
	else if(dt < (12l * 30l * 24l * 60l * 60l))
	{
		ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
		if(ct2 != 0l)
			return(g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "month" : "months"),
				ct2, ((ct2 == 1l) ? "day" : "days")
			));
		else
			return(g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "month" : "months")
			));
	}
	/* A year or longer */
	else
	{
		ct = MAX((dt / 60l / 60l / 24l / 30l / 12l), 1l);
		ct2 = MAX((dt / 60l / 60l / 24l / 30l), 1l) % 12l;
		if(ct2 != 0l)
			return(g_strdup_printf(
				"%ld %s %ld %s",
				ct, ((ct == 1l) ? "year" : "years"),
				ct2, ((ct2 == 1l) ? "month" : "months")
			));
		else
			return(g_strdup_printf(
				"%ld %s",
				ct, ((ct == 1l) ? "year" : "years") 
			));
	}

	/* Never reached */
	return(g_strdup(""));
}
