#include <gdk/gdk.h>

#include "guiutils.h"
#include "edv_cursor.h"
#include "config.h"


EDVCursor *edv_cursor_new(void);
EDVCursor *edv_cursor_new_from_gdk_cursor(
	const EDVCursorCode code,
	GdkCursor *gdk_cursor
);
EDVCursor *edv_cursor_new_from_gdk_cursor_type(
	const EDVCursorCode code,
	const GdkCursorType gdk_cursor_type
);
void edv_cursor_set_gdk_cursor(
	EDVCursor *cursor,
	GdkCursor *gdk_cursor
);
void edv_cursor_delete(EDVCursor *cursor);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new cursor.
 */
EDVCursor *edv_cursor_new(void)
{
	return(EDV_CURSOR(
		g_malloc0(sizeof(EDVCursor))
	));
}

/*
 *	Creates a new cursor.
 *
 *	The code specifies the cursor code, which can be one of
 *	EDV_CURSOR_CODE_*.
 *
 *	The gdk_cursor specifies the GdkCursor. The GdkCursor will
 *	be passed on to the new cursor and as such, the GdkCursor
 *	should not be referenced again after this call.
 *
 *	Returns the new cursor or NULL on error.
 */
EDVCursor *edv_cursor_new_from_gdk_cursor(
	const EDVCursorCode code,
	GdkCursor *gdk_cursor
)
{
	EDVCursor *cursor = edv_cursor_new();
	if(cursor == NULL)
	{
		GDK_CURSOR_DESTROY(gdk_cursor);
		return(NULL);
	}

	cursor->code = code;
	cursor->cursor = gdk_cursor;

	return(cursor);
}

/*
 *	Creates a new cursor from the GdkCursorType.
 *
 *	The code specifies the cursor code, which can be one of
 *	EDV_CURSOR_CODE_*.
 *
 *	The gdk_cursor_type specifies the GdkCursorType, which is a
 *	code that specifies a predefined GDK cursor.
 *
 *	Returns the new cursor or NULL on error. If the GdkCursorType
 *	is invalid or the GdkCursor was unable to be created then
 *	NULL will also be returned.
 */
EDVCursor *edv_cursor_new_from_gdk_cursor_type(
	const EDVCursorCode code,
	const GdkCursorType gdk_cursor_type
)
{
	GdkCursor *gdk_cursor = gdk_cursor_new(gdk_cursor_type);
	if(gdk_cursor == NULL)
		return(NULL);

	return(edv_cursor_new_from_gdk_cursor(code, gdk_cursor));
}

/*
 *	Sets the cursor's GdkCursor.
 */
void edv_cursor_set_gdk_cursor(
	EDVCursor *cursor,
	GdkCursor *gdk_cursor
)
{
	if(cursor == NULL)
	{
		GDK_CURSOR_DESTROY(gdk_cursor);
		return;
	}

	GDK_CURSOR_DESTROY(cursor->cursor);
	cursor->cursor = gdk_cursor;
}

/*
 *	Deletes the cursor.
 */
void edv_cursor_delete(EDVCursor *cursor)
{
	if(cursor == NULL)
		return;

	GDK_CURSOR_DESTROY(cursor->cursor);
	g_free(cursor);
}
