#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "url.h"
#include "cfg.h"

#include "guiutils.h"
#include "pulist.h"
#include "fprompt.h"
#include "cdialog.h"
#include "csd.h"
#include "fsd.h"
#include "fb.h"
#include "progressdialog.h"
#include "pdialog.h"
#include "cfg_win.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_interps.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "about_dlg.h"
#include "vfs_browser.h"
#include "image_browser.h"
#include "archive_comments_dlg.h"
#include "archive_statistics_dlg.h"
#include "archiver.h"
#include "recycle_bin.h"
#include "mime_types_list_win.h"
#include "devices_list_win.h"
#include "history_win.h"
#include "prop_dlg.h"
#include "obj_op_dlg.h"
#include "find_win.h"
#include "run_dlg.h"
#include "edv_core.h"
#include "edv_cb.h"
#include "edv_emit.h"
#include "edv_op.h"
#include "edv_interps_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef enum {
	EDV_SORT_TYPE_TEXT,
	EDV_SORT_TYPE_NUMBER
} EDVSortType;


typedef struct _EDVInterPSIdleData	EDVInterPSIdleData;
#define EDV_INTERPS_IDLE_DATA(p)	((EDVInterPSIdleData *)(p))


/* UNIX Signal Callbacks */
void edv_signal_cb(int s);

/* Timeout Callbacks */
static gint edv_interps_commands_idle_cb(gpointer data);
gint edv_manage_timeout_cb(gpointer data);
gint edv_check_mount_states_cb(gpointer data);

/* Idle Callbacks */
gint edv_reconfigured_idle_cb(gpointer data);
gint edv_master_write_protect_changed_idle_cb(gpointer data);
gint edv_delete_method_changed_idle_cb(gpointer data);

/* New/Map Window Callbacks */
void edv_new_vfs_browser_cb(GtkWidget *widget, gpointer data);
void edv_new_image_browser_cb(GtkWidget *widget, gpointer data);
void edv_new_archiver_cb(GtkWidget *widget, gpointer data);
void edv_map_recycle_bin_cb(GtkWidget *widget, gpointer data);
void edv_mime_types_cb(GtkWidget *widget, gpointer data);
void edv_devices_cb(GtkWidget *widget, gpointer data);
void edv_options_cb(GtkWidget *widget, gpointer data);
void edv_customize_cb(GtkWidget *widget, gpointer data);

/* MIME Types Added/Mdoified/Removed Callbacks */
void edv_mime_type_added_cb(
	const gint mt_num, EDVMIMEType *m, gpointer data
);
void edv_mime_type_modified_cb(
	const gint mt_num, EDVMIMEType *m, gpointer data
);
void edv_mime_type_removed_cb(
	const gint mt_num, gpointer data
);

/* GtkCList Sort Callbacks */
static gint edv_clist_column_sort_nexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	const EDVSortType sort_type
);
gint EDVCListColumnSortStringCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
gint edv_clist_column_sort_number_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);

/* Refresh & Reset Callbacks */
void edv_refresh_cb(GtkWidget *widget, gpointer data);
void edv_reset_cb(GtkWidget *widget, gpointer data);

/* Clear Callbacks */
void edv_purge_all_recycled_objects_cb(GtkWidget *widget, gpointer data);
void edv_clear_all_history_cb(GtkWidget *widget, gpointer data);
void edv_clear_events_history_cb(GtkWidget *widget, gpointer data);
void edv_clear_location_bars_history_cb(GtkWidget *widget, gpointer data);
void edv_clear_run_history_cb(GtkWidget *widget, gpointer data);

/* GtkEntry DND Callbacks */
void edv_entry_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);

/* GtkEntry Complete Path Callback */
gint edv_entry_complete_path_cb(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);

/* Location Bar DND Callbacks */
void edv_location_bar_icon_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *context,
	GtkSelectionData *selection_data, guint info, guint time,
	gpointer data
);

/* File Selector Callbacks */
void edv_file_selector_object_craeted_cb(
	const gchar *path, gpointer data
);
void edv_file_selector_object_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
);
void edv_file_selector_object_deleted_cb(
	const gchar *path, gpointer data
);

/* Options Window Callbacks */
void edv_cfg_win_apply_cb(
	GtkWidget *w, CfgList *cfg_list, gpointer data
);


/*
 *	InterPS Idle Data:
 */
struct _EDVInterPSIdleData {
	EDVCore		*core;
	gchar		**cmds_list;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Signal callback.
 *
 *	No gtk/gdk calls can be made here, since this is actually
 *	called from a separate thread.
 */
void edv_signal_cb(int s)
{
	/* Get the current value of the global core context pointer */
	EDVCore *core = edv_core_ptr;

	/* If the core was not initialized then ignore the signal */
	if(core == NULL)
		return;

	switch(s)
	{
#ifdef SIGHUP
	  case SIGHUP:
		core->pending_flags |= EDV_CORE_PENDING_RESET;
		break;
#else
#warning SIGHUP is not defined, reset signaling will not function
#endif
#ifdef SIGINT
	  case SIGINT:
		core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
		break;
#endif
#ifdef SIGTERM
	  case SIGTERM:
		/* Immediate exit with success */
		exit(0);
		break;
#endif
#ifdef SIGQUIT
	  case SIGQUIT:
		core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
		break;
#endif
#ifdef SIGSEGV
	  case SIGSEGV:
		/* Immediate exit with error */
		exit(1);
		break;
#endif
#ifdef SIGUSR1
	  case SIGUSR1:
		/* If we are at an idle run level or higher then increment
		 * the InterPS signal count
		 *
		 * edv_manage_timeout_cb() will check the signal count and
		 * process any queued InterPS commands
		 */
		if(core->runlevel >= EDV_RUNLEVEL_IDLE)
			core->interps_signal_count++;
		break;
#else
#warning SIGUSR1 is not defined, InterPS signaling will not function
#endif

#ifdef SIGUSR2
	  case SIGUSR2:
		break;
#endif
#ifdef SIGPIPE
	  case SIGPIPE: 
		break;
#endif
	}
}


/*
 *	Process InterPS commands idle callback.
 */
static gint edv_interps_commands_idle_cb(gpointer data)
{
	EDVInterPSIdleData *d = EDV_INTERPS_IDLE_DATA(data);
	if(d == NULL)
		return(FALSE);

	/* Process the InterPS commands */
	if(d->cmds_list != NULL)
	{
		edv_interps_op_process_commands(d->core, d->cmds_list);
		g_strfreev(d->cmds_list);
	}

	g_free(d);

	return(FALSE);
}

/*
 *	Core timeout callback.
 */
gint edv_manage_timeout_cb(gpointer data)
{
	gboolean	conserve_memory,
			all_windows_unmapped;
	gint		i,
			nwindows_unmapped;
	GtkWidget *cfgwin;
	CfgList *cfg_list;
	AboutDlg *aboutdlg;
	EDVVFSBrowser *browser;
	EDVImageBrowser *imbr;
	EDVArchiver *archiver;
	EDVArchiveCommentsDlg *archive_comments_dlg;
	EDVArchiveStatisticsDlg *archive_statistics_dlg;
	edv_recbin_struct *recbin;
	edv_mime_types_list_win_struct *mimetypeswin;
	edv_devices_list_win_struct *deviceswin;
	edv_history_win_struct *histwin;
	EDVPropDlg *propdlg;
	EDVFindWin *find_win;
	edv_obj_op_dlg_struct *obj_op_dlg;
	EDVRunDlg *rundlg;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return(FALSE);

/* Removes all timeout callbacks and breaks out of the GTK main loop */
#define REMOVE_TIMEOUTS_AND_BREAK_MAIN {	\
 core->manage_toid = GTK_TIMEOUT_REMOVE(	\
  core->manage_toid				\
 );						\
						\
 gtk_main_quit();				\
}

/* Reschedual timeout callback to this function and return FALSE, since
 * this function need not be called again because a new timeout has
 * been set for this function
 */
#define RETURN_RESCHEDUAL	{		\
 core->manage_toid = gtk_timeout_add(		\
  1000l,		/* 1 second interval */	\
  edv_manage_timeout_cb, core			\
 );						\
						\
 return(FALSE);					\
}

/* Mark timeout callback to this function as no longer valid and
 * return FALSE to never call this function again
 */
#define RETURN_NO_RESCHEDUAL		{	\
 core->manage_toid = 0;				\
						\
 return(FALSE);					\
}

/* Returns TRUE, keeping the current timeout callback to this function */
#define RETURN_CURRENT_SCHEDUAL	{		\
 return(TRUE);					\
}

	/* Get the configuration values */
	cfg_list = core->cfg_list;
	conserve_memory = EDV_GET_B(EDV_CFG_PARM_CONSERVE_MEMORY);


	/* Need to close all windows? */
	if(!(core->run_flags & EDV_RUN_STAY_RUNNING) &&
	   (core->pending_flags & EDV_CORE_PENDING_CLOSE_ALL_WINDOWS)
	)
	{
		gboolean	still_processing = FALSE,
				has_changes = FALSE;

		/* Remove the close all windows flag */
		core->pending_flags &= ~EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;

		/* Check if any windows/dialogs are in query or are still
		 * processing
		 */
		if(CDialogIsQuery())
		{
			CDialogBreakQuery();
			RETURN_CURRENT_SCHEDUAL;
		}
		if(FileBrowserIsQuery())
		{
			FileBrowserBreakQuery();
			RETURN_CURRENT_SCHEDUAL;
		}
		if(ProgressDialogIsQuery())
		{
			ProgressDialogBreakQuery(FALSE);
			RETURN_CURRENT_SCHEDUAL;
		}
		if(FSDIsQuery())
		{
			FSDBreakQuery();
			RETURN_CURRENT_SCHEDUAL;
		}
		if(CSDIsQuery())
		{
			CSDBreakQuery();
			RETURN_CURRENT_SCHEDUAL;
		}
		if(PDialogIsQuery())
		{
			PDialogBreakQuery();
			RETURN_CURRENT_SCHEDUAL;
		}
		if(FPromptIsQuery())
		{
			FPromptBreakQuery();
			RETURN_CURRENT_SCHEDUAL;
		}
		/* All dialogs not in query */

		/* Begin checking through all windows to see if any are
		 * still processing or have changes
		 */

		/* About Dialog */
		aboutdlg = core->about_dlg;
		if(aboutdlg != NULL)
		{
			if(aboutdlg->freeze_count > 0)
				still_processing = TRUE;
		}
		/* File Browsers */
		for(i = 0; i < core->total_browsers; i++)
		{
			browser = core->browser[i];
			if(browser == NULL)
				continue;

			if(EDV_VFS_BROWSER_IS_PROCESSING(browser) ||
			   (browser->freeze_count > 0)
			)
				still_processing = TRUE;
		}
		/* Image Browsers */
		for(i = 0; i < core->total_imbrs; i++)
		{
			imbr = core->imbr[i];
			if(imbr == NULL)
				continue;

			if(EDV_IMAGE_BROWSER_IS_PROCESSING(imbr) ||
			   (imbr->freeze_count > 0)
			)
				still_processing = TRUE;
		}
		/* Archivers */
		for(i = 0; i < core->total_archivers; i++)
		{
			archiver = core->archiver[i];
			if(archiver == NULL)
				continue;

			if(EDV_ARCHIVER_IS_PROCESSING(archiver) ||
			   (archiver->freeze_count > 0)
			)
				still_processing = TRUE;

			if(edv_archive_comments_dlg_is_mapped(archiver->comments_dialog))
			{
				if(edv_archive_comments_dlg_has_changes(archiver->comments_dialog))
				{
					still_processing = TRUE;
					has_changes = TRUE;
				}
			}
		}
		/* Recycle Bin */
		recbin = core->recbin;
		if(recbin != NULL)
		{
			if(EDV_RECYCLE_BIN_IS_PROCESSING(recbin) ||
			   (recbin->freeze_count > 0)
			)
				still_processing = TRUE;
		}
		/* MIME Types Window */
		mimetypeswin = core->mime_types_list_win;
		if(mimetypeswin != NULL)
		{
			if(mimetypeswin->freeze_count > 0)
				still_processing = TRUE;
		}
		/* Devices List Window */
		deviceswin = core->devices_list_win;
		if(deviceswin != NULL)
		{
			if(deviceswin->freeze_count > 0)
				still_processing = TRUE;
		}
		/* History List Window */
		histwin = core->history_list_win;
		if(histwin != NULL)
		{
			if(histwin->processing || (histwin->freeze_count > 0))
				still_processing = TRUE;
		}
		/* Options Window */
		cfgwin = core->options_cfg_win;
		if(cfgwin != NULL)
		{
			if(CfgWinGetHasChanges(cfgwin))
				has_changes = TRUE;
		}
		/* Customize Window */
		cfgwin = core->customize_cfg_win;
		if(cfgwin != NULL)
		{
			if(CfgWinGetHasChanges(cfgwin))
				has_changes = TRUE;
		}
		/* Property Dialogs */
		for(i = 0; i < core->total_prop_dlgs; i++)
		{
			propdlg = core->prop_dlg[i];
			if(propdlg == NULL)
				continue;

			if(edv_prop_dlg_get_freeze_count(propdlg) > 0)
				still_processing = TRUE;
			if(edv_prop_dlg_get_has_changes(propdlg))
				has_changes = TRUE;
		}
		/* Find Window */
		find_win = core->find_win;
		if(find_win != NULL)
		{
			if(find_win->processing || (find_win->freeze_count > 0))
				still_processing = TRUE;
		}
		/* Object Operations Dialog */
		obj_op_dlg = core->obj_op_dlg;
		if(obj_op_dlg != NULL)
		{
			if(obj_op_dlg->processing || (obj_op_dlg->freeze_count > 0))
				still_processing = TRUE;
		}
		/* Run Dialog */
		rundlg = core->run_dlg;
		if(rundlg != NULL)
		{
			if(rundlg->processing || (rundlg->freeze_count > 0))
				still_processing = TRUE;
		}

		/* Begin checking results, to see if any windows are still
		 * processing or have unsaved changed data
		 */
		if(has_changes)
		{


		}

		/* Any windows still processing? */
		if(still_processing)
		{
			/* One or more window is still processing, cannot
			 * close now so we need to return
			 *
			 * The next call to this function won't close all
			 * windows since we set the global marker for that
			 * to FALSE
			 */
			RETURN_CURRENT_SCHEDUAL;
		}

		/* Remove all timeout callbacks and break out of the GTK
		 * main loop
		 */
		REMOVE_TIMEOUTS_AND_BREAK_MAIN;

		/* Return and do not call this function again */
		RETURN_NO_RESCHEDUAL;
	}


	/* Need to reset? */
	if(core->pending_flags & EDV_CORE_PENDING_RESET)
	{
		core->pending_flags &= ~EDV_CORE_PENDING_RESET;
		edv_reset(core);
	}


	/* Begin checking to see if at least one window is unmapped
	 * and delete some unmapped windows
	 *
	 * Reset all_windows_unmapped to TRUE, it will be set to FALSE
	 * if at least one window is mapped
	 */
	all_windows_unmapped = TRUE;

	nwindows_unmapped = 0;

	/* About Dialog */
	aboutdlg = core->about_dlg;
	if(aboutdlg != NULL)
	{
		/* Initialized and mapped? */
		if(about_dlg_is_mapped(aboutdlg))
		{
			all_windows_unmapped = FALSE;
		}
		else
		{
			/* Not mapped, delete it */
			core->about_dlg = NULL;
			about_dlg_delete(aboutdlg);
			aboutdlg = NULL;
			nwindows_unmapped++;
		}
	}
	/* File Browsers */
	for(i = 0; i < core->total_browsers; i++)
	{
		browser = core->browser[i];
		if(browser == NULL)
			continue;

		/* Mapped? */
		if(edv_vfs_browser_is_mapped(browser))
		{
			all_windows_unmapped = FALSE;
		}
		else
		{
			/* Not mapped, delete it */
			edv_delete_vfs_browser(core, i);
			nwindows_unmapped++;
			i = -1;			/* Start over from 0 */
			continue;
		}
	}
	/* Image Browsers */
	for(i = 0; i < core->total_imbrs; i++)
	{
		imbr = core->imbr[i];
		if(imbr == NULL)
			continue;

		/* Mapped? */
		if(edv_image_browser_is_mapped(imbr))
		{
			all_windows_unmapped = FALSE;
		}
		else
		{
			/* Not mapped, delete it */
			edv_delete_image_browser(core, i);
			nwindows_unmapped++;
			i = -1;			/* Start over from 0 */
			continue;
		}
	}
	/* Archivers */
	for(i = 0; i < core->total_archivers; i++)
	{
		archiver = core->archiver[i];
		if(archiver == NULL)
			continue;

		/* Archive Comments Dialog mapped? */
		archive_comments_dlg = archiver->comments_dialog;
		if(archive_comments_dlg != NULL)
		{
			/* Not mapped? */
			if(!edv_archive_comments_dlg_is_mapped(archive_comments_dlg))
			{
				/* Delete it */
				edv_archive_comments_dlg_delete(archive_comments_dlg);
				archiver->comments_dialog = NULL;
				nwindows_unmapped++;
			}
		}

		/* Archive Statistics Dialog mapped? */
		archive_statistics_dlg = archiver->statistics_dialog;
		if(archive_statistics_dlg != NULL)
		{
			/* Not mapped? */
			if(!edv_archive_statistics_dlg_is_mapped(archive_statistics_dlg))
			{
				/* Delete it */
				edv_archive_statistics_dlg_delete(archive_statistics_dlg);
				archiver->statistics_dialog = NULL;
				nwindows_unmapped++;
			}
		}

		/* Mapped? */
		if(edv_archiver_is_mapped(archiver))
		{
			all_windows_unmapped = FALSE;
		}
		else
		{
			/* Not mapped, delete it */
			edv_delete_archiver(core, i);
			nwindows_unmapped++;
			i = -1;				/* Start over from 0 */
			continue;
		}
	}
	/* Recycle Bin */
	recbin = core->recbin;
	if(recbin != NULL)
	{
		/* Mapped? */
		if(edv_recycle_bin_is_mapped(recbin))
		{
			all_windows_unmapped = FALSE;
		}
		else if(conserve_memory)
		{
			/* Not mapped and need to conserve memory, delete it */
			core->recbin = NULL;
			edv_recycle_bin_delete(recbin);
			recbin = NULL;
			nwindows_unmapped++;
		}
	}
	/* MIME Types Window */
	mimetypeswin = core->mime_types_list_win;
	if(mimetypeswin != NULL)
	{
		if(EDVMimeTypesListWinIsMapped(mimetypeswin))
		{
			all_windows_unmapped = FALSE;
		}
		else if(conserve_memory)
		{
			/* Not mapped and need to conserve memory, delete it */
			core->mime_types_list_win = NULL;
			EDVMimeTypesListWinDelete(mimetypeswin);
			mimetypeswin = NULL;
			nwindows_unmapped++;
		}
	}
	/* Devices List Window */
	deviceswin = core->devices_list_win;
	if(deviceswin != NULL)
	{
		if(EDVDevicesListWinIsMapped(deviceswin))
		{
			all_windows_unmapped = FALSE;
		}
		else if(conserve_memory)
		{
			/* Not mapped and need to conserve memory, delete it */
			core->devices_list_win = NULL;
			EDVDevicesListWinDelete(deviceswin);
			deviceswin = NULL;
			nwindows_unmapped++;
		}
	}
	/* History List Window */
	histwin = core->history_list_win;
	if(histwin != NULL)
	{
		/* Initialized and mapped? */
		if(EDVHistoryWinIsMapped(histwin))
		{
			all_windows_unmapped = FALSE;
		}
		else if(conserve_memory)
		{
			/* Not mapped and need to conserve memory, delete it */
			core->history_list_win = NULL;
			EDVHistoryWinDelete(histwin);
			histwin = NULL;
			nwindows_unmapped++;
		}
	}
	/* Options Window */
	cfgwin = core->options_cfg_win;
	if(cfgwin != NULL)
	{
		if(GTK_WIDGET_MAPPED(cfgwin))
		{
			all_windows_unmapped = FALSE;
		}
		else if(conserve_memory)
		{
			/* Not mapped and need to conserve memory, delete it */
			core->options_cfg_win = NULL;
			CfgWinUnref(cfgwin);
			cfgwin = NULL;
			nwindows_unmapped++;
		}
	}
	/* Customize Window */
	cfgwin = core->customize_cfg_win;
	if(cfgwin != NULL)
	{
		if(GTK_WIDGET_MAPPED(cfgwin))
		{
			all_windows_unmapped = FALSE;
		}
		else if(conserve_memory)
		{
			/* Not mapped and need to conserve memory, delete it */
			core->customize_cfg_win = NULL;
			CfgWinUnref(cfgwin);
			cfgwin = NULL;
			nwindows_unmapped++;
		}
	}
	/* Property Dialogs */
	for(i = 0; i < core->total_prop_dlgs; i++)
	{
		propdlg = core->prop_dlg[i];
		if(propdlg == NULL)
			continue;

		/* Initialized and mapped? */
		if(edv_prop_dlg_is_mapped(propdlg))
		{
			all_windows_unmapped = FALSE;
		}
		else
		{
			/* Not mapped, delete it */
			core->prop_dlg[i] = NULL;
			edv_prop_dlg_delete(propdlg);
			propdlg = NULL;
			nwindows_unmapped++;
		}
	}
	/* Find Window */
	find_win = core->find_win;
	if(find_win != NULL)
	{
		/* Initialized and mapped? */
		if(edv_find_win_is_mapped(find_win))
		{
			all_windows_unmapped = FALSE;
		}
		else
		{
			/* Never delete the find window since it holds data
			 * that needs to be preserved for the next use
			 */
		}
	}
	/* Object Operations Dialog */
	obj_op_dlg = core->obj_op_dlg;
	if(obj_op_dlg != NULL)
	{
		if(EDVObjOpDlgIsMapped(obj_op_dlg))
		{
			all_windows_unmapped = FALSE;
		}
		else
		{
			/* Never delete the object operations dialog since
			 * it holds the last used values that needs to be
			 * preserved for the next use
			 */
		}
	}
	/* Run Dialog */
	rundlg = core->run_dlg;
	if(rundlg != NULL)
	{
		if(edv_run_dlg_is_mapped(rundlg))
		{
			all_windows_unmapped = FALSE;
		}
		else if(conserve_memory)
		{
			/* Not mapped and need to conserve memory, delete it */
			core->run_dlg = NULL;
			edv_run_dlg_delete(rundlg);
			rundlg = NULL;
			nwindows_unmapped++;
		}
	}

	/* Were all the windows unmapped above? */
	if(all_windows_unmapped)
	{
		/* Begin exiting after all the windows have been unmapped? */
		if(!(core->run_flags & EDV_RUN_STAY_RUNNING))
		{
			/* Removes all timeout callbacks and breaks out of
			 * one GTK main loop
			 */
			REMOVE_TIMEOUTS_AND_BREAK_MAIN;

			/* Return and do not call this function again */
			RETURN_NO_RESCHEDUAL;
		}
	}


	/* Need to clean up memory? */
	if(nwindows_unmapped > 0)
	{
		/* Unref any EDVPixmaps in the list that have only one
		 * reference count remaining
		 */
		core->pixmaps_list = edv_pixmaps_list_clean(core->pixmaps_list);

	}

	/* Check for and handle any pending InterPS commands if the
	 * InterPS signal count is positive
	 */
	if(core->interps_signal_count > 0)
	{
		EDVInterPSIdleData *d;

		/* Get the commands from all the command files that are
		 * waiting to be read and remove each read command file
		 */
		gchar **cmds_list = edv_interps_get_commands(cfg_list);

		/* Reduce the InterPS signal count per this handling */
		core->interps_signal_count--;
		if(core->interps_signal_count < 0)
			core->interps_signal_count = 0;

		/* Schedual to process the InterPS commands */
		d = EDV_INTERPS_IDLE_DATA(g_malloc(
			sizeof(EDVInterPSIdleData)
		));
		if(d != NULL)
		{
			d->core = core;
			d->cmds_list = cmds_list;
			gtk_idle_add_priority(
				G_PRIORITY_LOW,
				edv_interps_commands_idle_cb, d
			);
		}
		else
		{
			if(cmds_list != NULL)
				g_strfreev(cmds_list);
		}
	}


	RETURN_RESCHEDUAL;

#undef RETURN_RESCHEDUAL
#undef RETURN_NO_RESCHEDUAL
#undef RETURN_CURRENT_SCHEDUAL
#undef REMOVE_TIMEOUTS_AND_BREAK_MAIN
}


/*
 *	Check devices timeout callback.
 */
gint edv_check_mount_states_cb(gpointer data)
{
	gboolean last_mount_state;
	gint i;
	GList *glist;
	EDVDevice *d;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return(FALSE);

/* TODO */

	/* Iterate through each device */
	for(glist = core->devices_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		/* Do not check devices that may not be unmounted */
		if(EDV_DEVICE_IS_NO_UNMOUNT(d))
			continue;

		/* Record this device's previous mount state */
		last_mount_state = EDV_DEVICE_IS_MOUNTED(d);

		/* Get this device's current mount state from the system */
		edv_device_update_mount_state(d);

		/* Is there a change in the mount state? */
		if(EDV_DEVICE_IS_MOUNTED(d) != last_mount_state)
		{
			/* Update this device's statistics */
			edv_device_update_statistics(d);

			/* Notify that this device has been mount/unmount */
			edv_emit_device_mount(
				core,
				i,
				d,
				EDV_DEVICE_IS_MOUNTED(d)
			);
		}
	}

	return(TRUE);
}


/*
 *	Reconfigued idle callback.
 */
gint edv_reconfigured_idle_cb(gpointer data)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return(FALSE);

	core->emit_reconfigured_idle_id = 0;

	edv_emit_reconfigured(core);

	return(FALSE);
}

/*
 *	Master Write Protect state changed idle callback.
 */
gint edv_master_write_protect_changed_idle_cb(gpointer data)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return(FALSE);

	core->emit_master_write_protect_changed_idle_id = 0;

	edv_emit_master_write_protect_changed(core);

	return(FALSE);
}

/*
 *	Delete Method changed idle callback.
 */
gint edv_delete_method_changed_idle_cb(gpointer data)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return(FALSE);

	core->emit_delete_method_changed_idle_id = 0;

	edv_emit_delete_method_changed(core);

	return(FALSE);
}


/*
 *	New File Browser callback.
 */
void edv_new_vfs_browser_cb(GtkWidget *widget, gpointer data)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_new_vfs_browser(
		core,
		NULL,
		NULL,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL
	);
}

/*
 *	New Image Browser callback.
 */
void edv_new_image_browser_cb(GtkWidget *widget, gpointer data)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_new_image_browser(
		core,
		NULL,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL
	);
}

/*
 *	New Archiver callback.
 */
void edv_new_archiver_cb(GtkWidget *widget, gpointer data)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_new_archiver(
		core,
		NULL,
		NULL,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL
	);
}

/*
 *	Map Recycle Bin callback.
 */
void edv_map_recycle_bin_cb(GtkWidget *widget, gpointer data)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_map_recycle_bin(
		core,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL
	);
}

/*
 *	Map MIME Types List callback.
 */
void edv_mime_types_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_map_mime_types(
		core,
		NULL,			/* No initial MIME Type */
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);
}

/*
 *	Map Devices List callback.
 */
void edv_devices_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_map_devices(
		core,
		NULL,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);
}

/*
 *	Map Options Window callback.
 */
void edv_options_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_map_options(
		core,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);
}

/*
 *	Map Customize Window callback.
 */
void edv_customize_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_map_customize(
		core,
		core->geometry_flags,
		(core->geometry_flags != 0) ? &core->geometry : NULL,
		toplevel
	);
}


/*
 *	MIME Type added callback.
 */
void edv_mime_type_added_cb(
	const gint mt_num, EDVMIMEType *m, gpointer data
) 
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_emit_mime_type_added(
		core,
		mt_num, m
	);
}

/*
 *	MIME Type modified callback.
 */
void edv_mime_type_modified_cb(
	const gint mt_num, EDVMIMEType *m, gpointer data
)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_emit_mime_type_modified(
		core,
		mt_num, m
	);
}

/*
 *	MIME Type removed callback.
 */
void edv_mime_type_removed_cb(
	const gint mt_num, gpointer data
)
{
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_emit_mime_type_removed(
		core,
		mt_num
	);
}


/*
 *	GtkCList column sort callback nexus.
 *
 *	This function only sorts by the cell's text value and not the
 *	row's data. Only cells of type GTK_CELL_TEXT and
 *	GTK_CELL_PIXTEXT will be sorted, all other types will
 *	reutrn -1.
 *
 *	The clist specifies the GtkCList.
 *
 *	The ptr1 specifies the first GtkCListRow.
 *
 *	The ptr2 specifies the second GtkCListRow.
 *
 *	The sort_type specifies what the text value describes (string,
 *	number, etc).
 */
static gint edv_clist_column_sort_nexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	const EDVSortType sort_type
)
{
	gint sort_column;
	const gchar	*text1 = NULL,
					*text2 = NULL;
	const GtkCListRow	*row1 = (const GtkCListRow *)ptr1,
							*row2 = (const GtkCListRow *)ptr2;

	if((clist == NULL) || (row1 == NULL) || (row2 == NULL))
		return(-1);

	/* Get the column number that we are to sort by and what
	 * type of sorting (ascending or descending)
	 */
	sort_column = clist->sort_column;
	if((sort_column < 0) || (sort_column >= clist->columns))
		return(-1);

	/* Get the first cell's text value */
	switch(row1->cell[sort_column].type)
	{
	  case GTK_CELL_TEXT:
		text1 = GTK_CELL_TEXT(row1->cell[sort_column])->text;
		break;
	  case GTK_CELL_PIXTEXT:
		text1 = GTK_CELL_PIXTEXT(row1->cell[sort_column])->text;
		break;
	  default:
		break;
	}

	/* Get the second cell's text value */
	switch(row2->cell[sort_column].type)
	{
	  case GTK_CELL_TEXT:
		text2 = GTK_CELL_TEXT(row2->cell[sort_column])->text;
		break;
	  case GTK_CELL_PIXTEXT:
		text2 = GTK_CELL_PIXTEXT(row2->cell[sort_column])->text;
		break;
	  default:
		break;
	}

	if(text2 == NULL)
		return((text1 != NULL) ? 1 : -1);
	if(text1 == NULL)
		return(-1);

	/* Sort by the text value's descriptive type */
	switch(sort_type)
	{
	  case EDV_SORT_TYPE_TEXT:
		/* The text value is plain text, sort alphabetically */
		return(strcmp(text1, text2));
		break;

	  case EDV_SORT_TYPE_NUMBER:
		/* The text value is a number, sort numerically */
		{
			/* Number strings may contain notation characters such
			 * as ',', '#', or '$', so copy them to tempory strings
			 * and then remove those characters them
			 */
			gchar	ns1[80], ns2[80];
			gint	n1, n2;

			/* Copy the strings to tempory and modifyable number
			 * strings
			 */
			strncpy(ns1, text1, sizeof(ns1));
			ns1[sizeof(ns1) - 1] = '\0';
			strncpy(ns2, text2, sizeof(ns2));
			ns2[sizeof(ns2) - 1] = '\0';

			/* Remove any notation characters */
			substr(ns1, ",", "");
			substr(ns1, "#", "");
			substr(ns1, "$", "");
			substr(ns2, ",", "");
			substr(ns2, "#", "");
			substr(ns2, "$", "");

			n1 = strtod(ns1, NULL),
			n2 = strtod(ns2, NULL);

			if(n1 >= n2)
				return((gint)(n1 > n2));
			else
				return(-1);
		}
		break;

	  default:
		return(-1);
		break;
	}
}

/*
 *	GtkCList column sort by string callback.
 */
gint EDVCListColumnSortStringCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_clist_column_sort_nexus(
		clist, ptr1, ptr2, EDV_SORT_TYPE_TEXT
	));
}

/*
 *	GtkCList column sort by numeric callback.
 */
gint edv_clist_column_sort_number_cb(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(edv_clist_column_sort_nexus(
		clist, ptr1, ptr2, EDV_SORT_TYPE_NUMBER
	));
}


/*
 *	Refresh Callback.
 */
void edv_refresh_cb(GtkWidget *widget, gpointer data)
{
	edv_refresh(EDV_CORE(data));
}


/*
 *	Purge Recycled Objects callback.
 */
void edv_purge_all_recycled_objects_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_purge_all_recycled_objects(
		core,
		FALSE,				/* Do not map the Recycle Bin */
		TRUE,				/* Show progress */
		TRUE,				/* Interactive */
		toplevel
	);
}

/*
 *	Clear All History callback.
 */
void edv_clear_all_history_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_clear_all_history(
		core,
		TRUE,				/* Confirm */
		toplevel
	);
}

/*
 *	Clear Events History callback.
 */
void edv_clear_events_history_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_clear_events_history(
		core,
		TRUE,				/* Confirm */
		toplevel
	);
}

/*
 *	Clear locations history callback.
 */
void edv_clear_location_bars_history_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_clear_location_bars_history(
		core,
		TRUE,				/* Confirm */
		toplevel
	);
}

/*
 *	Clear run history callback.
 */
void edv_clear_run_history_cb(GtkWidget *widget, gpointer data)
{
	GtkWidget *toplevel = (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL;
	EDVCore *core = EDV_CORE(data);
	if(core == NULL)
		return;

	edv_clear_run_history(
		core,
		TRUE,				/* Confirm */
		toplevel
	);
}


/*
 *	GtkEntry DND "drag_data_received" signal callback.
 *
 *	This is used as an opaque callback to support the dragging of any
 *	disk object to any GtkEntry widget. The value in the GtkEntry
 *	will be replaced with a space separated list of absolute disk
 *	object paths.
 *
 *	Note that only the text is updated, no "activate" signal will
 *	be emitted.
 */
void edv_entry_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	GtkEntry *entry;
	EDVCore *core = EDV_CORE(data);
	if((widget == NULL) || (dc == NULL) || (selection_data == NULL) ||
	   (core == NULL)
	)
		return;

	if((selection_data->data == NULL) ||
	   (selection_data->length <= 0)
	)
		return;

	/* Get the widget as a GtkEntry */
	if(GTK_IS_COMBO(widget))
	{
		GtkCombo *combo = GTK_COMBO(widget);
		entry = GTK_ENTRY(combo->entry);
	}
	else if(GTK_IS_ENTRY(widget))
	{
		entry = GTK_ENTRY(widget);
	}
	else
	{
		return;
	}

	/* Handle by the target type
	 *
	 * String
	 */
	if((info == EDV_DND_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_INFO_STRING)
	)
	{
		gint len = selection_data->length;
		const gchar *ss = (const gchar *)selection_data->data;
		gchar *s;

		if(ss[len - 1] == '\0')
			len--;

		s = (gchar *)g_malloc(len + (1 * sizeof(gchar)));
		if(s != NULL)
		{
			gint position;
			gchar *s2, *s_end;
			GtkEditable *editable = GTK_EDITABLE(entry);

			if(len > 0)
				memcpy(s, selection_data->data, len);
			s[len] = '\0';

			/* Remove any null deliminators within the string */
			s2 = s;
			s_end = s2 + len;
			while(s2 < s_end)
			{
				if(*s2 == '\0')
					*s2 = ' ';
				s2++;
			}

			/* Insert the string into the entry */
			position = gtk_editable_get_position(editable);
			if(len > 0)
				gtk_editable_insert_text(
					editable,
					s, len,
					&position
				);

			g_free(s);
		}
	}
}


/*
 *	GtkEntry complete path "key_press_event" or "key_release_event"
 *	signal callback.
 */
gint edv_entry_complete_path_cb(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = 0;
	gboolean press;
	GtkEntry *entry;
	EDVCore *core = EDV_CORE(data);
	if((widget == NULL) || (key == NULL) || (core == NULL))
		return(status);

	press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;

	/* Given widget must have or be a GtkEntry */
	if(GTK_IS_COMBO(widget))
	{
		GtkCombo *combo = GTK_COMBO(widget);
		entry = GTK_ENTRY(combo->entry);
	}
	else if(GTK_IS_ENTRY(widget))
	{
		entry = GTK_ENTRY(widget);
	}
	else
	{
		return(status);
	}

#define SIGNAL_EMIT_STOP	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	switch(key->keyval)
	{
	  case GDK_Tab:
		/* Skip this if the shift or ctrl keys are held */
		if((key->state & GDK_CONTROL_MASK) ||
		   (key->state & GDK_SHIFT_MASK)
		)
			return(status);

		if(press)
		{
			gchar *path = STRDUP(gtk_entry_get_text(entry));
			if(path != NULL)
			{
				gchar *prev_path = STRDUP(path);
				EDVCompletePathStatus status;

				/* Complete the path */
				path = EDVCompletePath(path, &status);

				/* Set the completed path to the entry */
				gtk_entry_set_text(entry, (path != NULL) ? path : "");
				gtk_entry_set_position(entry, -1);

				/* If there was no change in the path then beep */
				if((prev_path != NULL) && (path != NULL))
				{
					if(!strcmp((const char *)prev_path, (const char *)path))
						gdk_beep();
				}

				g_free(prev_path);
				g_free(path);
			}
		}
		SIGNAL_EMIT_STOP
		status = TRUE;
		break;
	}

#undef SIGNAL_EMIT_STOP

	return(status);
}


/*
 *	Location bar icon "drag_data_get" signal callback.
 *
 *	The data is a GtkCombo.
 */
void edv_location_bar_icon_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *context,
	GtkSelectionData *selection_data, guint info, guint time,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	guint8 *buf;
	gint buf_len;
	GList *url_list = NULL;
	GtkEntry *entry;
	GtkCombo *combo = (GtkCombo *)data;
	URLStruct *url;
	if((selection_data == NULL) || (combo == NULL))
		return;

	entry = GTK_ENTRY(combo->entry);
	if(entry == NULL)
		return;

	/* Generate a URL list from the current entry text */
	url = url_new();
	url->path = STRDUP(gtk_entry_get_text(entry));
	url_list = g_list_append(url_list, url);

	/* Encode the buffer from the URL list */
	buf = url_encode(url_list, &buf_len);
	if(buf != NULL)
	{
		gtk_selection_data_set(
			selection_data,
			GDK_SELECTION_TYPE_STRING,
			8,		/* 8 bits per character */
			buf,
			buf_len
		);
		data_sent = TRUE;
	}

	g_free(buf);

	g_list_foreach(url_list, (GFunc)url_delete, NULL);
	g_list_free(url_list);

	/* If failed to send data then send error response */
	if(!data_sent)
	{
		const gchar *s = "Error";
		gtk_selection_data_set(
			selection_data,
			GDK_SELECTION_TYPE_STRING,
			8,		/* 8 bits per character */
			s,
			STRLEN(s)
		);
		data_sent = TRUE;
	}
}


/*
 *	File Selector object created callback.
 */
void edv_file_selector_object_craeted_cb(
	const gchar *path, gpointer data
)
{
	EDVVFSObject *obj;
	EDVCore *core = EDV_CORE(data);
	if((core == NULL) || (path == NULL))
		return;

	obj = edv_vfs_object_lstat(path);
	if(obj != NULL)
	{
		edv_emit_vfs_object_added(
			core,
			path,
			obj
		);
		edv_vfs_object_delete(obj);
	}
}

/*
 *	File Selector object modified callback.
 */
void edv_file_selector_object_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
)
{
	EDVVFSObject *obj;
	EDVCore *core = EDV_CORE(data);
	if((core == NULL) || (new_path == NULL))
		return;

	if(old_path == NULL)
		old_path = new_path;

	obj = edv_vfs_object_lstat(new_path);
	if(obj != NULL)
	{
		edv_emit_vfs_object_modified(
			core,
			old_path,
			new_path,
			obj
		);
		edv_vfs_object_delete(obj);
	}
}

/*
 *	File Selector object deleted callback.
 */
void edv_file_selector_object_deleted_cb(
	const gchar *path, gpointer data
)
{
	EDVCore *core = EDV_CORE(data);
	if((core == NULL) || (path == NULL))
		return;

	edv_emit_vfs_object_removed(core, path);
}


/*
 *	Options Window apply callback.
 */
void edv_cfg_win_apply_cb(
	GtkWidget *w, CfgList *cfg_list, gpointer data
)
{
	EDVCore *core = EDV_CORE(data);
	if((cfg_list == NULL) || (core == NULL))
		return;

	/* Queue a notify about the change in the configuration */
	edv_queue_emit_reconfigured(core);
}
