#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>				/* localtime() */
#include <sys/types.h>
#include <fnmatch.h>				/* fnmatch() */
#ifdef HAVE_LIBZ
# include <zlib.h>
#endif
#ifdef HAVE_LIBBZ2
# include <bzlib.h>
#endif
#ifdef HAVE_LIBTAR
# include <fcntl.h>
# include <libtar.h>
#endif
#ifdef HAVE_LIBXAR
# include <xar/xar.h>
#endif
#ifdef HAVE_LIBZIP
# include <zip.h>
#endif
#include <glib.h>

#include "cfg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_archive_obj_stat.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Utils */
static void edv_archive_object_stat_cap_newline(gchar *s);
static gboolean edv_archive_object_stat_filter(
	GList *paths_list,
#if defined(HAVE_REGEX)
	regex_t *regex_filter,
#else
	const gchar *filter,
#endif
	const gchar *path
);
static gint edv_archive_object_stat_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
);

/* Archive Format-Specific Line Parsing */
static void edv_archive_object_stat_parse_line_arj(
	EDVArchiveObject *obj,
	const gchar *buf
);
static void edv_archive_object_stat_parse_line_lha(
	EDVArchiveObject *obj,
	const gchar *buf,
	const struct tm *cur_mtime_buf
);
static void edv_archive_object_stat_parse_line_rar(
	EDVArchiveObject *obj,
	const gchar *buf
);
static void edv_archive_object_stat_parse_line_rpm(
	EDVArchiveObject *obj,
	const gchar *buf
);
#ifndef HAVE_LIBTAR
static void edv_archive_object_stat_parse_line_tar(
	EDVArchiveObject *obj,
	const gchar *buf
);
#endif
#ifndef HAVE_LIBZIP
static void edv_archive_object_stat_parse_line_zip(
	EDVArchiveObject *obj,
	const gchar *buf
);
#endif

/* Archive Object statistics */
static EDVArchiveObject *edv_archive_object_stat_arj(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
);
static EDVArchiveObject *edv_archive_object_stat_lha(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
);
static EDVArchiveObject *edv_archive_object_stat_rar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
);
static EDVArchiveObject *edv_archive_object_stat_rpm(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
);
#ifdef HAVE_LIBTAR
/* static int edv_archive_object_stat_libtar_open_cb(const char *path, int oflags, int mode); */
# ifdef HAVE_LIBZ
static int edv_archive_object_stat_libtar_open_libz_cb(const char *path, int oflags, int mode);
# endif
# ifdef HAVE_LIBBZ2
static int edv_archive_object_stat_libtar_open_libbz2_cb(const char *path, int oflags, int mode);
static ssize_t edv_archive_object_stat_libtar_read_libbz2_cb(int fd, void *buf, size_t buf_len);
static ssize_t edv_archive_object_stat_libtar_write_libbz2_cb(int fd, const void *buf, size_t buf_len);
static int edv_archive_object_stat_libtar_close_libbz2_cb(int fd);
# endif
static EDVObjectType edv_archive_object_stat_libtar_get_type(TAR *tar);
static gchar *edv_archive_object_stat_libtar_get_path(TAR *tar);
static gint edv_archive_object_stat_libtar_next(TAR *tar);
static EDVArchiveObject *edv_archive_object_stat_libtar_new_object(
	TAR *tar,
	const gulong index
);
#endif	/* HAVE_LIBTAR */
static EDVArchiveObject *edv_archive_object_stat_tar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	gboolean is_compress_compressed,
	gboolean is_gzip_compressed,
	gboolean is_bzip2_compressed
);
#ifdef HAVE_LIBXAR
static EDVArchiveObject *edv_archive_object_stat_xar_new_object(
	CfgList *cfg_list,
	xar_t xar,
	xar_file_t xar_fp,
	const gulong index,
	const gchar *path,
	const glong gmt_offset_dst
);
#endif	/* HAVE_LIBXAR */
static EDVArchiveObject *edv_archive_object_stat_xar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
);
#ifdef HAVE_LIBZIP
static EDVArchiveObject *edv_archive_object_stat_libzip_new_object(
	CfgList *cfg_list,
	struct zip *archive,
	struct zip_stat *zip_stat_buf
);
#endif
static EDVArchiveObject *edv_archive_object_stat_zip(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
);
EDVArchiveObject *edv_archive_object_stat(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	const gchar *password
);

/* Archive Object List Statistics Sequentially */
#define EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES	\
	CfgList *cfg_list,				\
	const gchar *arch_path,				\
	const gulong arch_size_bytes,			\
	GList *paths_list,				\
	const gchar *filter,				\
	gint (*obj_rtn_cb)(				\
		const gchar *arch_path,			\
		EDVArchiveObject *obj,			\
		const gulong i, const gulong m,		\
		gpointer data				\
	),						\
	gpointer obj_rtn_data
static gint edv_archive_object_stat_list_sequential_arj(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES
);
static gint edv_archive_object_stat_list_sequential_lha(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES
);
static gint edv_archive_object_stat_list_sequential_rar(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES
);
static gint edv_archive_object_stat_list_sequential_rpm(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES
);
static gint edv_archive_object_stat_list_sequential_tar(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);
static gint edv_archive_object_stat_list_sequential_xar(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES
);
static gint edv_archive_object_stat_list_sequential_zip(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES
);
gint edv_archive_object_stat_list_sequential(
	CfgList *cfg_list,
	const gchar *arch_path,
	GList *paths_list,
	const gchar *filter,
	const gchar *password,
	gint (*obj_rtn_cb)(
		const gchar *arch_path,
		EDVArchiveObject *obj,
		const gulong i, const gulong m,
		gpointer data
	),
	gpointer obj_rtn_data
);

/* Archive Object List Statistics */
static gint edv_archive_object_stat_list_progress_cb(
	const gchar *arch_path,
	EDVArchiveObject *obj,
	const gulong i, const gulong m,
	gpointer data
);
GList *edv_archive_object_stat_list(
	CfgList *cfg_list,
	const gchar *arch_path,
	GList *paths_list,
	const gchar *filter,
	const gchar *password,
	gint (*progress_cb)(
		const gchar *,
		const gulong, const gulong,
		gpointer
	),
	gpointer progress_data
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define ISSPACE(c)	(((c) == ' ') || ((c) == '\t') ||	\
			 ((c) == '\v') || ((c) == '\f') ||	\
			 ((c) == '\r') || ((c) == '\n'))
#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))

#define STRCASEPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
			 !g_strncasecmp((s),(p),strlen(p)) : FALSE)
#define STRPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
			 !strncmp((s),(p),strlen(p)) : FALSE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Object return callback data:
 *
 *	Used by edv_archive_object_stat_list() in edv_archive_object_stat_list_progress_cb()
 *	to gather the list of objects returned by
 *	edv_archive_object_stat_list_sequential().
 */
typedef struct {
	GList		*objs_list;
	gint		(*progress_cb)(
			const gchar *,
			const gulong, const gulong,
			gpointer
	);
	gpointer	progress_data;
} EDVArchStatListObjRtnData;
#define EDV_ARCH_STAT_LIST_OBJ_RTN_DATA(p)	((EDVArchStatListObjRtnData *)(p))


#ifdef HAVE_LIBBZ2
/*
 *	BZip2 callback data:
 */
typedef struct {
	gint		fd;			/* Actual file descriptor */
	BZFILE		*bz2d;			/* BZip2 descriptor */
} bz2_cb_data_struct;
#define BZ2_CB_DATA(p)			((bz2_cb_data_struct *)(p))
#endif


/*
 *	Replaces the first newline character with a null byte.
 */
static void edv_archive_object_stat_cap_newline(gchar *s)
{
	while(*s != '\0')
	{
		if(ISCR(*s))
		{
			*s = '\0';
			return;
		}
		s++;
	}
}


/*
 *	Checks if the path passes the filter test.
 *
 *	The paths_list specifies the list of object paths. If the
 *	specified path matches a path in paths_list then TRUE
 *	will be returned. If paths_list is NULL then filter will
 *	be used instead.
 *
 *	The regex_filter or filter specifies the filter. If HAVE_REGEX
 *	is defined then it is the regex_filter and otherwise it is
 *	the filter. In either case, this is only used if paths_list is
 *	NULL.
 *
 *	The path specifies the path. If paths_list was NULL then only
 *	the name portion of the path will be used when comparing it
 *	to the filter.
 *
 *	Returns TRUE if the specified path passes the filter test.
 */
static gboolean edv_archive_object_stat_filter(
	GList *paths_list,
#if defined(HAVE_REGEX)
	regex_t *regex_filter,
#else
	const gchar *filter,
#endif
	const gchar *path
)
{
	/* If no path is specified then always return FALSE */
	if(STRISEMPTY(path))
		return(FALSE);

	/* Is the paths list specified? */
	if(paths_list != NULL)
	{
		const gchar *fpath;
		GList *glist;

		/* Iterate through the list of paths */
		for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
		{
			fpath = (const gchar *)glist->data;
			if(STRISEMPTY(fpath))
				continue;

			/* Paths match? (case sensitive) */
			if(!strcmp((const char *)fpath, (const char *)path))
				return(TRUE);
		}
	}
	else
	{
		/* No paths list was specified, use the filter instead */
		const gchar *name;

		/* If the filter is NULL then it is always a match */
#if defined(HAVE_REGEX)
		if(regex_filter == NULL)
			return(TRUE);
#else
		if(STRISEMPTY(filter))
			return(TRUE);
#endif

		/* Get the name portion of the path */
		name = (const gchar *)strrchr((const char *)path, G_DIR_SEPARATOR);
		if(name != NULL)
			name++;
		else
			name = path;

		/* Filter check */
#if defined(HAVE_REGEX)
		if(regexec(
			regex_filter,
			name,
			0, NULL,
			0
		) == 0)
			return(TRUE);
#else
		if(fnmatch(filter, name, 0) == 0)
			return(TRUE);
#endif
	}

	return(FALSE);
}

/*
 *	Executes the command using the shell specified by the
 *	configuration and opens and returns the requested streams.
 */
static gint edv_archive_object_stat_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
)
{
	gchar		*shell_prog;
	const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Execute the list archive command */
	gint pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		cstdin_rtn,
		cstdout_rtn,
		cstderr_rtn
	);

	g_free(shell_prog);

	return(pid);
}


/*
 *	Parses the null terminated string specified by buf which is
 *	from an ARJ Packager list output.
 *
 *	The given string contain four lines per object description
 *	per ARJ Packager list output format.
 */
static void edv_archive_object_stat_parse_line_arj(
	EDVArchiveObject *obj,
	const gchar *buf
)
{
	struct tm mtime_buf;
	gint c;
	const gchar *s = buf;

	if((obj == NULL) || (s == NULL))
		return;

	(void)memset(
		&mtime_buf,
		0x00,
		sizeof(mtime_buf)
	);
	mtime_buf.tm_isdst = -1;

	/* Sample format line:

002) dir/file.ext
 11 UNIX           1293        638 0.493 03-08-21 00:16:06 -rw-r--r-- ---  +1
							   DTA   03-08-25 20:00:36
							   DTC   03-08-21 00:16:06

	*/

	/* Index (ignored) */
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Full Path & Name */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISSPACE(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		/* Get full path */
		g_free(obj->path);
		obj->path = (gchar *)g_malloc(
			(len + 1) * sizeof(gchar)
		);
		if(len > 0)
			(void)memcpy(
				obj->path,
				s_start,
				len * sizeof(gchar)
			);
		obj->path[len] = '\0';

		/* Strip any tailing deliminators */
		edv_path_strip(obj->path);

		/* Get name from path */
		g_free(obj->name);
		obj->name = STRDUP(g_basename(obj->path));
	}
	while(ISSPACE(*s))
		s++;

	/* Rev (ignored) */
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Platform (ignored) */
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Size */
	obj->size = (gulong)atol((const char *)s);
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Storage Size */
	obj->storage_size = (gulong)atol((const char *)s);
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Compression Ratio */
	obj->compression_ratio = 1.0f - ATOF(s);
	if(obj->compression_ratio > 1.0f)
		obj->compression_ratio = 1.0f;
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Year-month-date */
	if(*s != '\0')
	{
		gint year_2digit;

		/* Get 2-digit year (use "imperfect" Y2K fix) */
		if(*s == '0')
			s++;
		year_2digit = atoi((const char *)s);
		mtime_buf.tm_year = (int)((year_2digit < 60) ?
			(year_2digit + 100) : year_2digit
		);
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* Month */
		if(*s == '0')
			s++;
		mtime_buf.tm_mon = (int)atoi((const char *)s) - 1;
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* Day */
		if(*s == '0')
			s++;
		mtime_buf.tm_mday = (int)atoi((const char *)s);
		while(!ISSPACE(*s) && (*s != '\0'))
			s++;
	}
	while(ISSPACE(*s))
		s++;

	/* Hour:minutes:seconds */
	if(*s != '\0')
	{
		/* Hour */
		if(*s == '0')
			s++;
		mtime_buf.tm_hour = (int)atoi((const char *)s) - 1;
		while((*s != ':') && (*s != '\0'))
			s++;
		if(*s == ':')
			s++;

		/* Minutes */
		if(*s == '0')
			s++;
		mtime_buf.tm_min = (int)atoi((const char *)s);
		while((*s != ':') && (*s != '\0'))
			s++;
		if(*s == ':')
			s++;

		/* Seconds */
		if(*s == '0')
			s++;
		mtime_buf.tm_sec = (int)atoi((const char *)s);
		while(!ISSPACE(*s) && (*s != '\0'))
			s++;
	}

	/* Parse time compoents and generate system time since we now
	 * have enough information to do that
	 */
	obj->access_time = 0l;
	obj->modify_time = (gulong)mktime(&mtime_buf);
	obj->change_time = 0l;

	while(ISSPACE(*s))
		s++;


	/* Type */
	if(*s != '\0')
	{
		switch(*s)
		{
		    case 'd':
			obj->type = EDV_OBJECT_TYPE_DIRECTORY;
			break;
		    case 'l':
			obj->type = EDV_OBJECT_TYPE_LINK;
			break;
		    case 'p':
			obj->type = EDV_OBJECT_TYPE_FIFO;
			break;
		    case 'b':
			obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
			break;
		    case 'c':
			obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
			break;
		    case 's':
			obj->type = EDV_OBJECT_TYPE_SOCKET;
			break;
		    default:
			obj->type = EDV_OBJECT_TYPE_FILE;
			break;
		}
		s++;
	}

	/* Permissions */
	obj->permissions = 0x00000000;

	/* Owner read/write/execute */
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_UR : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_UW : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_UX : 0;
		s++;
	}

	/* Group read/write/execute */
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_GR : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_GW : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_GX : 0;
		s++;
	}

	/* Anonymous read/write/execute */
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_OR : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_OW : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_OX : 0;
		s++;
	}

	while(ISSPACE(*s))
		s++;

	/* GUI (ignored) */
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* BPMGS (use as method) */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISSPACE(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		g_free(obj->storage_method);
		obj->storage_method = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(len > 0)
			(void)memcpy(
				obj->storage_method,
				s_start,
				len
			);
		obj->storage_method[len] = '\0';
	}
	while(ISSPACE(*s))
		s++;

	/* Ignore other dates and a whole bunch of other stuff */
}

/*
 *	Parses the null terminated string specified by buf which is
 *	from a LHA Archive list output.
 */
static void edv_archive_object_stat_parse_line_lha(
	EDVArchiveObject *obj,
	const gchar *buf,
	const struct tm *cur_mtime_buf
)
{
	struct tm mtime_buf;
	gint c;
	const gchar *s = buf;

	if((obj == NULL) || (s == NULL))
		return;

	if(cur_mtime_buf != NULL)
		(void)memcpy(
			&mtime_buf,
			cur_mtime_buf,
			sizeof(mtime_buf)
		);
	else
		(void)memset(
			&mtime_buf,
			0x00,
			sizeof(mtime_buf)
		);

	/* Sample format line:

-rw-r--r-- 500/500 1058 2643 40.0% -lh5- bf09 Aug 26 18:32 dir/file.ext

	 */

	/* Type */
	if(*s != '\0')
	{
		switch(*s)
		{
		    case 'd':
			obj->type = EDV_OBJECT_TYPE_DIRECTORY;
			break;
		    case 'l':
			obj->type = EDV_OBJECT_TYPE_LINK;
			break;
		    case 'p':
			obj->type = EDV_OBJECT_TYPE_FIFO;
			break;
		    case 'b':
			obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
			break;
		    case 'c':
			obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
			break;
		    case 's':
			obj->type = EDV_OBJECT_TYPE_SOCKET;
			break;
		    default:
			obj->type = EDV_OBJECT_TYPE_FILE;
			break;
		}
		s++;
	}

	/* Permissions */
	obj->permissions = 0x00000000;

	/* Owner read/write/execute */
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_UR : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_UW : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_UX : 0;
		s++;
	}

	/* Group read/write/execute */
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_GR : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_GW : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_GX : 0;
		s++;
	}

	/* Anonymous read/write/execute */
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_OR : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_OW : 0;
		s++;
	}
	c = *s;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_OX : 0;
		s++;
	}
	while(ISBLANK(*s))
		s++;

	/* Owner/group */
	if(*s != '\0')
	{
		gchar *s_delim;
		const gchar *s_end = (const gchar *)strpbrk(
			(const char *)s, " \t"
		);
		const gint len = (s_end != NULL) ? (gint)(s_end - s) : STRLEN(s);
		gchar *owner_group_str = (gchar *)g_malloc(
			(len + 1) * sizeof(gchar)
		);
		if(len > 0)
			memcpy(owner_group_str, s, len);
		owner_group_str[len] = '\0';

		s_delim = (gchar *)strchr((char *)owner_group_str, '/');
		if(s_delim != NULL)
		{
			*s_delim = '\0';

			/* Get the owner */
			g_free(obj->owner_name);
			obj->owner_name = STRDUP(owner_group_str);

			/* Get the group */
			g_free(obj->group_name);
			obj->group_name = STRDUP(s_delim + 1);
		}
		else
		{
			/* Get the owner */
			g_free(obj->owner_name);
			obj->owner_name = STRDUP(owner_group_str);
		}

		g_free(owner_group_str);

		s += len;
	}
	while(ISBLANK(*s))
		s++;

	/* Storage Size */
	obj->storage_size = (gulong)atol((const char *)s);
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Size */
	obj->size = (gulong)atol((const char *)s);
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Compression Ratio */
	obj->compression_ratio = CLIP(ATOF(s) / 100.0f, 0.0f, 1.0f);
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Method */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISBLANK(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		g_free(obj->storage_method);
		obj->storage_method = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(len > 0)
			memcpy(obj->storage_method, s_start, len);
		obj->storage_method[len] = '\0';
	}
	while(ISBLANK(*s))
		s++;

	/* CRC */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISBLANK(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		g_free(obj->crc);
		obj->crc = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(len > 0)
			memcpy(obj->crc, s_start, len);
		obj->crc[len] = '\0';
		g_strup(obj->crc);
	}
	while(ISBLANK(*s))
		s++;

	/* Month name */
	if(STRCASEPFX((const char *)s, "jan"))
		mtime_buf.tm_mon = 0;
	else if(STRCASEPFX((const char *)s, "feb"))
		mtime_buf.tm_mon = 1;
	else if(STRCASEPFX((const char *)s, "mar"))
		mtime_buf.tm_mon = 2;
	else if(STRCASEPFX((const char *)s, "apr"))
		mtime_buf.tm_mon = 3;
	else if(STRCASEPFX((const char *)s, "may"))
		mtime_buf.tm_mon = 4;
	else if(STRCASEPFX((const char *)s, "jun"))
		mtime_buf.tm_mon = 5;
	else if(STRCASEPFX((const char *)s, "jul"))
		mtime_buf.tm_mon = 6;
	else if(STRCASEPFX((const char *)s, "aug"))
		mtime_buf.tm_mon = 7;
	else if(STRCASEPFX((const char *)s, "sep"))
		mtime_buf.tm_mon = 8;
	else if(STRCASEPFX((const char *)s, "oct"))
		mtime_buf.tm_mon = 9;
	else if(STRCASEPFX((const char *)s, "nov"))
		mtime_buf.tm_mon = 10;
	else if(STRCASEPFX((const char *)s, "dec"))
		mtime_buf.tm_mon = 11;
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Day */
	mtime_buf.tm_mday = (int)atoi((const char *)s);
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Year or time */
	if(*s != '\0')
	{
		gchar *s2 = STRDUP(s), *s3;
		for(s3 = s2; *s3 != '\0'; s3++)
		{
			if(ISBLANK(*s3))
			{
				*s3 = '\0';
				break;
			}
		}

		s3 = s2;

		/* Time? */
		if(strchr((const char *)s3, ':') != NULL)
		{
			/* Hour */
			if(*s3 == '0')
				s3++;
			mtime_buf.tm_hour = (int)atoi((const char *)s3);
			while((*s3 != ':') && (*s3 != '\0'))
				s3++;
			if(*s3 == ':')
				s3++;

			/* Minutes */
			if(*s3 == '0')
				s3++;
			mtime_buf.tm_min = (int)atoi((const char *)s3);
		}
		else
		{
			/* Year */
			mtime_buf.tm_year = (int)atoi((const char *)s3) - 1900;
		}
		g_free(s2);

		while(!ISBLANK(*s) && (*s != '\0'))
			s++;
	}
	while(ISBLANK(*s))
		s++;

	/* Parse time compoents and generate system time since we now
	 * have enough information to do that
	 */
	obj->access_time = (gulong)mktime(&mtime_buf);
	obj->modify_time = obj->access_time;
	obj->change_time = obj->access_time;

	/* Full Path & Name */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISSPACE(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		/* Get full path */
		g_free(obj->path);
		obj->path = (gchar *)g_malloc(
			(len + 1) * sizeof(gchar)
		);
		if(len > 0)
			memcpy(obj->path, s_start, len);
		obj->path[len] = '\0';

		/* Strip any tailing deliminators */
		edv_path_strip(obj->path);

		/* Get name from path */
		g_free(obj->name);
		obj->name = STRDUP(g_basename(obj->path));
	}
	while(ISSPACE(*s))
		s++;
}

/*
 *	Parses the null terminated string specified by buf which is
 *	from a RAR Archive list output.
 *
 *	The given string must contain two lines per object description
 *	per RAR Archive list output format.
 */
static void edv_archive_object_stat_parse_line_rar(
	EDVArchiveObject *obj,
	const gchar *buf
)
{
	struct tm mtime_buf;
	gchar		*method,
			*version;
	const gchar *s = buf;

	if((obj == NULL) || (s == NULL))
		return;

	(void)memset(
		&mtime_buf,
		0x00,
		sizeof(mtime_buf)
	);
	mtime_buf.tm_isdst = -1;

	/* Sample format line:

 *dir/file.ext
   1604      780  48% 12-03-03 14:50 -rw-r--r-- 5005A169 m3b 2.9

		Or

 *dir/file.ext
   1604      780  48% 12-03-03 14:50 .....A 5005A169 m3b 2.9

	 */

	/* Encryption */
	if(*s != '\0')
	{
		/* If the first character is a '*' then it means that
		 * the archive object is encrypted
		 */
		if(*s == '*')
		{
			/* Advanced Encryption Standard Rijndael */
			g_free(obj->encryption_name);
			obj->encryption_name = g_strdup("AES Rijndael");
			s++;
		}
	}

	/* Full Path & Name */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_end;

		/* Seek past the '*' character */
		while(*s == '*')
			s++;

		/* Find the end deliminator */
		s_end = (const gchar *)strchr((const char *)s, '\n');
		if(s_end == NULL)
			s_end = (const gchar *)strchr((const char *)s, '\r');
		if(s_end == NULL)
			s_end = (const gchar *)strchr((const char *)s, ' ');
		if(s_end == NULL)
			s_end = (const gchar *)strchr((const char *)s, '\t');
		len = (s_end != NULL) ?
			(gint)(s_end - s) : (gint)strlen((const char *)s);

		g_free(obj->path);
		obj->path = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(obj->path != NULL)
		{
			if(len > 0)
				(void)memcpy(
					obj->path,
					s,
					(size_t)len
				);
			obj->path[len] = '\0';

			/* Strip any tailing deliminators */
			edv_path_strip(obj->path);

			/* Get the name from the path */
			g_free(obj->name);
			obj->name = STRDUP(g_basename(obj->path));
		}

		s += len;
	}
	while(ISSPACE(*s))
		s++;

	/* Size */
	obj->size = (gulong)atol((const char *)s);
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Storage Size */
	obj->storage_size = (gulong)atol((const char *)s);
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Compression Ratio */
	obj->compression_ratio = CLIP(
		(1.0f - ((gfloat)atoi((const char *)s) / 100.0f)),
		0.0f, 1.0f
	);
	while(!ISSPACE(*s) && (*s != '\0'))
		s++;
	while(ISSPACE(*s))
		s++;

	/* Day-month-year */
	if(*s != '\0')
	{
		gint year_2digit;

		/* Day */
		if(*s == '0')
			s++;
		mtime_buf.tm_mday = atoi((const char *)s);
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* Month */
		if(*s == '0')
			s++;
		mtime_buf.tm_mon = atoi((const char *)s) - 1;
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* Get 2-digit year (use "imperfect" Y2K fix) */
		if(*s == '0')
			s++;
		year_2digit = atoi((const char *)s);
		mtime_buf.tm_year = (int)((year_2digit < 60) ?
			(year_2digit + 100) : year_2digit
		);
		while(!ISSPACE(*s) && (*s != '\0'))
			s++;
	}
	while(ISSPACE(*s))
		s++;

	/* Hour:minutes */
	if(*s != '\0')
	{
		/* Hour */
		if(*s == '0')
			s++;
		mtime_buf.tm_hour = (atoi((const char *)s) - 1);
		while((*s != ':') && (*s != '\0'))
			s++;
		if(*s == ':')
			s++;

		/* Minutes */
		if(*s == '0')
			s++;
		mtime_buf.tm_min = atoi((const char *)s);
		while(!ISSPACE(*s) && (*s != '\0'))
			s++;
	}

	/* Parse time compoents and generate system time since we now
	 * have enough information to do that
	 */
	obj->access_time = 0l;
	obj->modify_time = (gulong)mktime(&mtime_buf);
	obj->change_time = 0l;

	while(ISSPACE(*s))
		s++;

	/* Attributes */
	if(s != '\0')
	{
		const gchar *s_end = (const gchar *)strpbrk(
			(const char *)s,
			" \t\n"
		);
		const gint len = (s_end != NULL) ?
			(gint)(s_end - s) : (gint)strlen((const char *)s);

		/* Determine the type of attributes based on the length
		 * of the string
		 *
		 * UNIX?
		 */
		if(len == 10)
		{
			/* Type */
			switch(s[0])
			{
			    case 'd':
				obj->type = EDV_OBJECT_TYPE_DIRECTORY;
				break;
			    case 'l':
				obj->type = EDV_OBJECT_TYPE_LINK;
				break;
			    case 'p':
				obj->type = EDV_OBJECT_TYPE_FIFO;
				break;
			    case 'b':
				obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
				break;
			    case 'c':
				obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
				break;
			    case 's':
				obj->type = EDV_OBJECT_TYPE_SOCKET;
				break;
			    default:
				obj->type = EDV_OBJECT_TYPE_FILE;
				break;
			}

			/* Permissions */
			obj->permissions = 0x00000000;

			/* User read/write/execute */
			if(s[1] == 'r')
				obj->permissions |= EDV_PERMISSION_UR;
			if(s[2] == 'w')
				obj->permissions |= EDV_PERMISSION_UW;
			if(s[3] == 'x')
				obj->permissions |= EDV_PERMISSION_UX;

			/* Group read/write/execute */
			if(s[4] == 'r')
				obj->permissions |= EDV_PERMISSION_GR;
			if(s[5] == 'w')
				obj->permissions |= EDV_PERMISSION_GW;
			if(s[6] == 'x')
				obj->permissions |= EDV_PERMISSION_GX;

			/* Other read/write/execute */
			if(s[7] == 'r')
				obj->permissions |= EDV_PERMISSION_OR;
			if(s[8] == 'w')
				obj->permissions |= EDV_PERMISSION_OW;
			if(s[9] == 'x')
				obj->permissions |= EDV_PERMISSION_OX;
		}
		/* DOS */
		else if(len == 6)
		{
			obj->type = EDV_OBJECT_TYPE_FILE;
			obj->permissions = EDV_PERMISSION_UR |
					   EDV_PERMISSION_UW |
					   EDV_PERMISSION_GR |
					   EDV_PERMISSION_GW |
					   EDV_PERMISSION_OR |
					   EDV_PERMISSION_OW;
		}
		/* Other */
		else
		{
			obj->type = EDV_OBJECT_TYPE_FILE;
			obj->permissions = EDV_PERMISSION_UR |
					   EDV_PERMISSION_UW |
					   EDV_PERMISSION_GR |
					   EDV_PERMISSION_GW |
					   EDV_PERMISSION_OR |
					   EDV_PERMISSION_OW;
		}

		s += len;
	}
	while(ISSPACE(*s))
		s++;

	/* CRC */
	if(*s != '\0')
	{
		const gchar *s_end = (const gchar *)strpbrk(
			(const char *)s,
			" \t\n"
		);
		const gint len = (s_end != NULL) ?
			(gint)(s_end - s) : (gint)strlen((const char *)s);

		g_free(obj->crc);
		obj->crc = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(obj->crc != NULL)
		{
			if(len > 0)
				(void)memcpy(
					obj->crc,
					s,
					(size_t)len
				);
			obj->crc[len] = '\0';
		}

		s += len;
	}
	while(ISSPACE(*s))
		s++;

	/* Method */
	if(*s != '\0')
	{
		const gchar *s_end = (const gchar *)strpbrk(
			(const char *)s,
			" \t\n"
		);
		const gint len = (s_end != NULL) ?
			(gint)(s_end - s) : (gint)strlen((const char *)s);

		method = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(method != NULL)
		{
			if(len > 0)
				(void)memcpy(
					method,
					s,
					(size_t)len
				);
			method[len] = '\0';
		}

		s += len;
	}
	else
	{
		method = NULL;
	}
	while(ISSPACE(*s))
		s++;

	/* Version */
	if(*s != '\0')
	{
		const gchar *s_end = (const gchar *)strpbrk(
			(const char *)s,
			" \t\n"
		);
		const gint len = (s_end != NULL) ?
			(gint)(s_end - s) : (gint)strlen((const char *)s);

		version = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(version != NULL)
		{
			if(len > 0)
				(void)memcpy(
					version,
					s,
					(size_t)len
				);
			version[len] = '\0';
		}

		s += len;
	}
	else
	{
		version = NULL;
	}
	/* For now we combine the method and version values as the
	 * storage_method
	 */
	if(!STRISEMPTY(method))
	{
		g_free(obj->storage_method);
		if(STRISEMPTY(version))
			obj->storage_method = g_strdup(method);
		else
			obj->storage_method = g_strconcat(
				method,
				" ",
				version,
				NULL
			);
	}
	g_free(method);
	g_free(version);
	while(ISSPACE(*s))
		s++;
}

/*
 *	Parses the null terminated string specified by buf which is
 *	from a RedHat Package Manager list output.
 */
static void edv_archive_object_stat_parse_line_rpm(
	EDVArchiveObject *obj,
	const gchar *buf
)
{
	struct tm mtime_buf;
	gint c;
	const gchar *buf_ptr = buf;

	if((obj == NULL) || (buf_ptr == NULL))
		return;

	(void)memset(
		&mtime_buf,
		0x00,
		sizeof(struct tm)
	);
	mtime_buf.tm_isdst = -1;

	/* Sample format line:

-rw-r--r--    1 root    root            27666 Jan  4  2000 /dir/file.ext

	 */

	/* Type and permissions string (this is one string with no
	 * deliminators
	 */

	/* Type */
	c = *buf_ptr;
	if(c != '\0')
	{
		switch(c)
		{
		    case 'd':
			obj->type = EDV_OBJECT_TYPE_DIRECTORY;
			break;
		    case 'l':
			obj->type = EDV_OBJECT_TYPE_LINK;
			break;
		    case 'p':
			obj->type = EDV_OBJECT_TYPE_FIFO;
			break;
		    case 'b':
			obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
			break;
		    case 'c':
			obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
			break;
		    case 's':
			obj->type = EDV_OBJECT_TYPE_SOCKET;
			break;
		    default:
			obj->type = EDV_OBJECT_TYPE_FILE;
			break;
		}
		buf_ptr++;
	}

	/* Permissions */
	obj->permissions = 0x00000000;

	/* Owner read/write/execute */
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_UR : 0;
		buf_ptr++;
	}
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_UW : 0;
		buf_ptr++;
	}
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_UX : 0;
		buf_ptr++;
	}

	/* Group read/write/execute */
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_GR : 0;
		buf_ptr++;
	}
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_GW : 0;
		buf_ptr++;
	}
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_GX : 0;
		buf_ptr++;
	}

	/* Anonymous read/write/execute */
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_OR : 0;
		buf_ptr++;
	}
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_OW : 0;
		buf_ptr++;
	}
	c = *buf_ptr;
	if(c != '\0')
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_OX : 0;
		buf_ptr++;
	}

	while(ISBLANK(*buf_ptr))
		buf_ptr++;


	/* This is number, not sure what it is but just seek past it */
	while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
		buf_ptr++;
	while(ISBLANK(*buf_ptr))
		buf_ptr++;


	/* Owner and group strings, deliminated by a space in between */
	if(*buf_ptr != '\0')
	{
		gchar	*s,
			*s2;

		/* Get owner */
		g_free(obj->owner_name);
		obj->owner_name = s = g_strdup(buf_ptr);
		s2 = strchr(s, ' ');
		if(s2 != NULL)
			*s2 = '\0';
		s2 = strchr(s, '\t');
		if(s2 != NULL)
			*s2 = '\0';

		/* Seek to group */
		while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
			buf_ptr++;
		while(ISBLANK(*buf_ptr))
			buf_ptr++;

		/* Get group */
		g_free(obj->group_name);
		obj->group_name = s = g_strdup(buf_ptr);
		s2 = strchr(s, ' ');
		if(s2 != NULL)
			*s2 = '\0';
		s2 = strchr(s, '\t');
		if(s2 != NULL)
			*s2 = '\0';

		/* Seek past group to first blank character */
		while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
			buf_ptr++;
	}


	while(ISBLANK(*buf_ptr))
		buf_ptr++;

	/* Size */
	obj->size = (gulong)atol((const char *)buf_ptr);

	while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
		buf_ptr++;
	while(ISBLANK(*buf_ptr))
		buf_ptr++;


	/* These next three strings are month, day, and year
	 * separated by blank characters but the names are verbose and
	 * thus too difficult to parse
	 */
	/* Skip month */
	while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
		buf_ptr++;
	while(ISBLANK(*buf_ptr))
		buf_ptr++;

	/* Skip day */
	while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
		buf_ptr++;
	while(ISBLANK(*buf_ptr))
		buf_ptr++;

	/* Skip year */
	while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
		buf_ptr++;
	while(ISBLANK(*buf_ptr))
		buf_ptr++;

	/* Full Path & Name */
	if(*buf_ptr != '\0')
	{
		gint len;
		const gchar *s = buf_ptr;

		while(!ISSPACE(*buf_ptr) && (*buf_ptr != '\0'))
			buf_ptr++;

		len = (gint)(buf_ptr - s);

		/* Get full path */
		g_free(obj->path);
		obj->path = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(obj->path != NULL)
		{
			if(len > 0)
				(void)memcpy(
					obj->path,
					s,
					(size_t)len
				);
			obj->path[len] = '\0';

			/* Strip any tailing deliminators */
			edv_path_strip(obj->path);

			/* Get name from path */
			g_free(obj->name);
			obj->name = STRDUP(g_basename(obj->path));
		}
	}
	while(ISSPACE(*buf_ptr))
		buf_ptr++;

	/* If this object is a link then we need to parse this
	 * link's target value
	 */
	if(obj->type == EDV_OBJECT_TYPE_LINK)
	{
		gint len;
		const gchar *s;

		/* Seek past "->" deliminator */
		while(!ISSPACE(*buf_ptr) && (*buf_ptr != '\0'))
			buf_ptr++;
		while(ISSPACE(*buf_ptr))
			buf_ptr++;

		/* Now at this link's target value */
		s = buf_ptr;
		while(!ISSPACE(*buf_ptr) && (*buf_ptr != '\0'))
			buf_ptr++;

		len = (gint)(buf_ptr - s);

		g_free(obj->link_target);
		obj->link_target = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(obj->link_target != NULL)
		{
			if(len > 0)
				(void)memcpy(
					obj->link_target,
					s,
					(size_t)len
				);
			obj->link_target[len] = '\0';
		}

		while(ISSPACE(*buf_ptr))
			buf_ptr++;
	}
}

#ifndef HAVE_LIBTAR
/*
 *	Parses the null terminated string specified by buf which is
 *	from a Tape Archiver list output.
 */
static void edv_archive_object_stat_parse_line_tar(
	EDVArchiveObject *obj,
	const gchar *buf
)
{
	struct tm mtime_buf;
	gchar c;
	const gchar *s = buf;

	if((obj == NULL) || (s == NULL))
		return;

	memset(&mtime_buf, 0x00, sizeof(struct tm));
	mtime_buf.tm_isdst = -1;

	/* Sample format line:

-rw-r--r-- learfox/learfox 279 2006-06-17 02:40 file.txt

	   or

lrwxrwxrwx root/root   0 2001-11-19 02:57 file.ext -> link.dest

	 */

	/* Type and permissions string (this is one string with no
	 * deliminators
	 */

	/* Type */
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		if(c == 'd')
			obj->type = EDV_OBJECT_TYPE_DIRECTORY;
		else if(c == 'l')
			obj->type = EDV_OBJECT_TYPE_LINK;
		else if(c == 'p')
			obj->type = EDV_OBJECT_TYPE_FIFO;
		else if(c == 'b')
			obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
		else if(c == 'c')
			obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
		else if(c == 's')
			obj->type = EDV_OBJECT_TYPE_SOCKET;
		else
			obj->type = EDV_OBJECT_TYPE_FILE;

		s++;
	}

	/* Permissions */
	obj->permissions = 0x00000000;

	/* Owner read/write/execute */
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_UR : 0;
		s++;
	}
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_UW : 0;
		s++;
	}
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_UX : 0;
		s++;
	}

	/* Group read/write/execute */
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_GR : 0;
		s++;
	}
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_GW : 0;
		s++;
	}
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_GX : 0;
		s++;
	}

	/* Anonymous read/write/execute */
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'r') ? EDV_PERMISSION_OR : 0;
		s++;
	}
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'w') ? EDV_PERMISSION_OW : 0;
		s++;
	}
	c = *s;
	if(!ISBLANK(c) && (c != '\0'))
	{
		obj->permissions |= (c == 'x') ? EDV_PERMISSION_OX : 0;
		s++;
	}

	/* Seek past any subsequent characters as needed */
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;

	while(ISBLANK(*s))
		s++;

	/* Owner and group string, separated by a '/' in between */
	if(*s != '\0')
	{
		gchar *s_delim;
		const gchar *s_end = (const gchar *)strpbrk(
			(const char *)s,
			" \t\n"
		);
		const int len = (s_end != NULL) ? (gint)(s_end - s) : STRLEN(s);
		gchar *owner_group_str = (gchar *)g_malloc(
			(len + 1) * sizeof(gchar)
		);
		if(len > 0)
			memcpy(owner_group_str, s, len);
		owner_group_str[len] = '\0';

		/* Get position of owner and group name deliminator */
		s_delim = (gchar *)strchr((char *)owner_group_str, '/');
		if(s_delim != NULL)
		{
			*s_delim = '\0';	/* Put null char at end of owner name */

			/* Get the owner's name */
			g_free(obj->owner_name);
			obj->owner_name = STRDUP(owner_group_str);

			/* Get the group's name (it is valid after the null char) */
			g_free(obj->group_name);
			obj->group_name = STRDUP(s_delim + 1);
		}
		else
		{
			/* Get the owner's name */
			g_free(obj->owner_name);
			obj->owner_name = STRDUP(owner_group_str);
		}

		g_free(owner_group_str);

		s += len;
	}
	while(ISBLANK(*s))
		s++;


	/* Size */
	obj->size = (gulong)atol((const char *)s);

	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;


	/* Year-month-day */
	if(*s != '\0')
	{
		/* Get year (input value starts from 1900) */
		if(*s == '0')
			s++;
		mtime_buf.tm_year = (int)(atoi((const char *)s) - 1900);
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* Get month */
		if(*s == '0')
			s++;
		mtime_buf.tm_mon = (int)(atoi((const char *)s) - 1);
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* Get day */
		if(*s == '0')
			s++;
		mtime_buf.tm_mday = (int)atoi((const char *)s);
		while(!ISBLANK(*s) && (*s != '\0'))
			s++;
	}

	while(ISBLANK(*s))
		s++;


	/* Hour:minutes */
	if(*s != '\0')
	{
		/* Get hour */
		if(*s == '0')
			s++;
		mtime_buf.tm_hour = (int)(atoi((const char *)s) - 1);
		while((*s != ':') && (*s != '\0'))
			s++;
		if(*s == ':')
			s++;

		/* Get minutes */
		if(*s == '0')
			s++;
		mtime_buf.tm_min = (int)atoi((const char *)s);
		while(!ISBLANK(*s) && (*s != '\0'))
			s++;
	}

	/* Parse time compoents and generate system time since we now
	 * have enough information to do that
	 */
	obj->access_time = (gulong)mktime(&mtime_buf);
	obj->modify_time = obj->access_time;
	obj->change_time = obj->access_time;

	while(ISBLANK(*s))
		s++;

	/* Full Path & Name */
	if(*s != '\0')
	{
		/* Link? */
		if(obj->type == EDV_OBJECT_TYPE_LINK)
		{
			const gchar *s_delim = (const gchar *)strstr(
				(const char *)s, " -> "
			);
			if(s_delim != NULL)
			{
				const gint len = (gint)(s_delim - s);

				g_free(obj->path);
				obj->path = (gchar *)g_malloc(
					(len + 1) * sizeof(gchar)
				);
				if(len > 0)
					memcpy(obj->path, s, len);
				obj->path[len] = '\0';

				/* Strip any tailing deliminators */
				edv_path_strip(obj->path);

				/* Get the name from the path */
				g_free(obj->name);
				obj->name = STRDUP(g_basename(obj->path));

				/* Seek past the deliminator and get this
				 * link's target value
				 */
				s = s_delim + strlen(" -> ");
				if(*s != '\0')
				{
					const gchar *s_end = (const gchar *)strpbrk(
						(const char *)s,
						"\n\r"
					);
					const gint len = (s_end != NULL) ?
						(gint)(s_end - s) : STRLEN(s);

					g_free(obj->link_target);
					obj->link_target = (gchar *)g_malloc(
						(len + 1) * sizeof(gchar)
					);
					if(len > 0)
						memcpy(obj->link_target, s, len);
					obj->link_target[len] = '\0';

					s += len;
				}
			}
			else
			{
				/* Link with a missing deliminator, so just get
				 * the full path & name
				 */
				const gchar *s_end = (const gchar *)strpbrk(
					(const char *)s,
					"\n\r"
				);
				const gint len = (s_end != NULL) ?
					(gint)(s_end - s) : STRLEN(s);

				g_free(obj->path);
				obj->path = (gchar *)g_malloc(
					(len + 1) * sizeof(gchar)
				);
				if(len > 0)
					memcpy(obj->path, s, len);
				obj->path[len] = '\0';

				/* Strip any tailing deliminators */
				edv_path_strip(obj->path);

				/* Get the name from the path */
				g_free(obj->name);
				obj->name = STRDUP(g_basename(obj->path));

				s += len;
			}
		}
		/* All else assume file */
		else
		{
			const gchar *s_end = (const gchar *)strpbrk(
				(const char *)s,
				"\n\r"
			);
			const gint len = (s_end != NULL) ?
				(gint)(s_end - s) : STRLEN(s);

			g_free(obj->path);
			obj->path = (gchar *)g_malloc(
				(len + 1) * sizeof(gchar)
			);
			if(len > 0)
				memcpy(obj->path, s, len);
			obj->path[len] = '\0';

			/* Strip any tailing deliminators */
			edv_path_strip(obj->path);

			/* Get the name from the path */
			g_free(obj->name);
			obj->name = STRDUP(g_basename(obj->path));

			s += len;
		}
	}
}
#endif	/* !HAVE_LIBTAR */

#ifndef HAVE_LIBZIP
/*
 *	Parses the null terminated string specified by buf which is
 *	from a PKZip list output
 */
static void edv_archive_object_stat_parse_line_zip(
	EDVArchiveObject *obj,
	const gchar *buf
)
{
	struct tm mtime_buf;
	const gchar *s = buf;

	if((obj == NULL) || (s == NULL))
		return;

	memset(&mtime_buf, 0x00, sizeof(struct tm));
	mtime_buf.tm_isdst = -1;

	/* Sample format line:

    2371  Defl:X      688  71%  08-24-03 20:33  9c2d86e6  file.ext

	 */

	/* Size */
	obj->size = (gulong)atol((const char *)s);
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Compression Method */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISBLANK(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		g_free(obj->storage_method);
		obj->storage_method = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(len > 0)
			memcpy(obj->storage_method, s_start, len);
		obj->storage_method[len] = '\0';
	}
	while(ISBLANK(*s))
		s++;

	/* Storage Size */
	obj->storage_size = (gulong)atol((const char *)s);
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Compression Ratio */
	obj->compression_ratio = (gfloat)atoi((const char *)s) / 100.0f;
	while(!ISBLANK(*s) && (*s != '\0'))
		s++;
	while(ISBLANK(*s))
		s++;

	/* Month-day-year */
	if(*s != '\0')
	{
		gint year_2digit;

		/* Get month */
		if(*s == '0')
			s++;
		mtime_buf.tm_mon = (int)(atoi((const char *)s) - 1);
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* Get day */
		if(*s == '0')
			s++;
		mtime_buf.tm_mday = (int)atoi((const char *)s);
		while((*s != '-') && (*s != '\0'))
			s++;
		if(*s == '-')
			s++;

		/* The year is relative to 1900, note that there is a Y2K
		 * problem here but we are compensating for it
		 */
		if(*s == '0')
			s++;
		year_2digit = (int)atoi((const char *)s);
		mtime_buf.tm_year = (int)((year_2digit < 60) ?
			(year_2digit + 100) : year_2digit
		);
		while(!ISBLANK(*s) && (*s != '\0'))
			s++;
	}
	while(ISBLANK(*s))
		s++;

	/* Hour:minutes */
	if(*s != '\0')
	{
		/* Get hour */
		if(*s == '0')
			s++;
		mtime_buf.tm_hour = (int)(atoi((const char *)s) - 1);
		while((*s != ':') && (*s != '\0'))
			s++;
		if(*s == ':')
			s++;

		/* Get minutes */
		if(*s == '0')
			s++;
		mtime_buf.tm_min = (int)atoi((const char *)s);
		while(!ISBLANK(*s) && (*s != '\0'))
			s++;
	}

	/* Parse time compoents and generate system time since we now
	 * have enough information to do that
	 */
	obj->access_time = (gulong)mktime(&mtime_buf);
	obj->modify_time = obj->access_time;
	obj->change_time = obj->access_time;

	while(ISBLANK(*s))
		s++;

	/* CRC */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISBLANK(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		g_free(obj->crc);
		obj->crc = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(len > 0)
			memcpy(obj->crc, s_start, len);
		obj->crc[len] = '\0';
	}
	while(ISBLANK(*s))
		s++;

	/* Full Path & Name */
	if(*s != '\0')
	{
		gint len;
		const gchar *s_start = s;

		while(!ISSPACE(*s) && (*s != '\0'))
			s++;

		len = (gint)(s - s_start);

		/* Get full path */
		g_free(obj->path);
		obj->path = (gchar *)g_malloc(
			(len + 1) * sizeof(gchar)
		);
		if(len > 0)
			memcpy(obj->path, s_start, len);
		obj->path[len] = '\0';

		/* Determine object type from the full path, if it has a
		 * tailing deliminator then it is a directory, otherwise it
		 * is a file
		 */
		if(len > 1)
		{
			switch(obj->path[len - 1])
			{
			  case G_DIR_SEPARATOR:
				obj->type = EDV_OBJECT_TYPE_DIRECTORY;
				break;
			  case '@':
				obj->type = EDV_OBJECT_TYPE_LINK;
				break;
			  default:
				obj->type = EDV_OBJECT_TYPE_FILE;
				break;
			}
		}
		else
		{
			obj->type = EDV_OBJECT_TYPE_FILE;
		}

		/* Strip any tailing deliminators */
		edv_path_strip(obj->path);

		/* Get name from path */
		g_free(obj->name);
		obj->name = STRDUP(g_basename(obj->path));
	}
	while(ISSPACE(*s))
		s++;

	/* The PKZip format does not support permissions */
	obj->permissions = 0;

}
#endif	/* !HAVE_LIBZIP */


/*
 *	Get ARJ archive object statistics.
 */
static EDVArchiveObject *edv_archive_object_stat_arj(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
)
{
	FILE		*cstdout,
			*cstderr;
	gint pid;
	const gchar *prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);
	EDVArchiveObject *obj;

	/* Format the list ARJ archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" v -i -y \"%s\" \"%s\"",
		prog_arj,
		arch_path,
		path
	);
	if(cmd == NULL)
		return(NULL);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(NULL);
	}

	obj = NULL;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		gboolean got_header = FALSE;
		const gchar *delim_header = "------------";
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE		/* Block */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					got_header = TRUE;
					break;
				}
				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Able to read past the header? */
		if(got_header)
		{
			/* Read the next 4 lines togeather */
			gint i;
			for(i = 0; (i < 4) && !feof(fp); i++)
			{
				FDISCARD(cstderr);
				(void)edv_stream_read_lineptr(
					fp,
					&buf,
					TRUE	/* Block */
				);
			}
			if(buf != NULL)
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				/* Parse the lines */
				obj = edv_archive_object_new();
				edv_archive_object_stat_parse_line_arj(
					obj,
					s
				);

				g_free(buf);
				buf = NULL;
			}
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(obj);
}

/*
 *	Get LHA archive object statistics.
 */
static EDVArchiveObject *edv_archive_object_stat_lha(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
)
{
	FILE		*cstdout,
			*cstderr;
	gint pid;
	const gchar *prog_lha = EDV_GET_S(EDV_CFG_PARM_PROG_LHA);
	EDVArchiveObject *obj;

	/* Format the list LHA archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" v \"%s\" \"%s\"",
		prog_lha,
		arch_path,
		path
	);
	if(cmd == NULL)
		return(NULL);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(NULL);
	}

	obj = NULL;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		time_t cur_time = (time_t)edv_time();
		const struct tm *tm_ptr;
		struct tm cur_mtime_buf;
		gboolean got_header = FALSE;
		const gchar *delim_header = "----------";
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Get current time */
		tm_ptr = localtime(&cur_time);
		if(tm_ptr != NULL)
			(void)memcpy(&cur_mtime_buf, tm_ptr, sizeof(struct tm));
		else
			(void)memset(&cur_mtime_buf, 0x00, sizeof(struct tm));

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE            /* Block */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					got_header = TRUE;
					break;
				}
				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Able to read past the header? */
		if(got_header)
		{
			/* Read the next line */
			gint i;
			for(i = 0; (i < 1) && !feof(fp); i++)
			{
				FDISCARD(cstderr);
				(void)edv_stream_read_lineptr(
					fp,
					&buf,
					TRUE    /* Block */
				);
			}
			if(buf != NULL)
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				/* Parse the line */
				obj = edv_archive_object_new();
				edv_archive_object_stat_parse_line_lha(
					obj,
					s,
					&cur_mtime_buf
				);

				g_free(buf);
				buf = NULL;
			}
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(obj);
}

/*
 *	Get RAR archive object statistics.
 */
static EDVArchiveObject *edv_archive_object_stat_rar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
)
{
	FILE		*cstdout,
			*cstderr;
	gint pid;
	const gchar *prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);
	EDVArchiveObject *obj;

	/* Format the list RAR archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" v -y -c- \"%s\" \"%s\"",
		prog_rar,
		arch_path,
		path
	);
	if(cmd == NULL)
		return(NULL);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(NULL);
	}

	obj = NULL;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		gboolean got_header = FALSE;
		const gchar *delim_header = "------------";
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE            /* Block */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
		                        buf = NULL;
					got_header = TRUE;
					break;
				}
				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Able to read past the header? */
		if(got_header)
		{
			/* Read the next 2 lines togeather */
			gint i;
			for(i = 0; (i < 2) && !feof(fp); i++)
			{
				FDISCARD(cstderr);
				(void)edv_stream_read_lineptr(
					fp,
					&buf,
					TRUE    /* Block */
				);
			}
			if(buf != NULL)
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				obj = edv_archive_object_new();
				edv_archive_object_stat_parse_line_rar(
					obj,
					s
				);

				g_free(buf);
				buf = NULL;
			}
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(obj);
}

/*
 *	Get RPM archive object statistics.
 */
static EDVArchiveObject *edv_archive_object_stat_rpm(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
)
{
	FILE		*cstdout,
			*cstderr;
	gint pid;
	const gchar *prog_rpm = EDV_GET_S(EDV_CFG_PARM_PROG_RPM);
	EDVArchiveObject *obj;

	/* Format the list RPM archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" -q -l -p -v \"%s\"",
		prog_rpm,
		arch_path
	);
	if(cmd == NULL)
		return(NULL);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(NULL);
	}

	obj = NULL;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Begin reading each line and get stats of the line
		 * that describes the specified path
		 */
		while(!feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE		/* Block */
			))
			{
				gint i;
				gchar	*s = buf,
					*s2;
				while(ISBLANK(*s))
					s++;

				/* Skip no more than 8 fields to position s2 at the
				 * file name argument
				 */
				s2 = s;
				for(i = 0; i < 8; i++)
				{
					while(!ISBLANK(*s2) && (*s2 != '\0'))
						s2++;
					while(ISBLANK(*s2))
						s2++;
				}

				/* Does this line describe the specified
				 * path?
				 */
				if(STRPFX(s2, path))
				{
					obj = edv_archive_object_new();
					edv_archive_object_stat_parse_line_rpm(
						obj,
						s
					);
					break;
				}

				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(obj);
}

#ifdef HAVE_LIBTAR
#if 0
/*
 *	libtar open callback.
 */
static int edv_archive_object_stat_libtar_open_cb(const char *path, int oflags, int mode)
{
	const gint fd = (gint)open(
		(const char *)path,
		(int)oflags,
		(mode_t)mode
	);
	if(fd < 0)
		return(-1);

	if((oflags & O_CREAT) && edv_fchmod(fd, mode))
	{
		const gint error_code = (gint)errno;
		(void)close(fd);
		errno = (int)error_code;
		return(-1);
	}

	return((int)fd);
}
#endif

# ifdef HAVE_LIBZ
/*
 *	libtar GZip open callback.
 */
static int edv_archive_object_stat_libtar_open_libz_cb(const char *path, int oflags, int mode)
{
	const char *libz_oflags;
	gint fd;
	gzFile zd;

	switch(oflags & O_ACCMODE)
	{
	    case O_WRONLY:
		libz_oflags = "wb";
		break;
	    case O_RDONLY:
		libz_oflags = "rb";
		break;
	    default:
	    case O_RDWR:
		errno = EINVAL;
		return(-1);
		break;
	}

	fd = (gint)open(path, oflags, mode);
	if(fd < 0)
		return(-1);

	if((oflags & O_CREAT) && edv_fchmod(fd, mode))
	{
		const gint error_code = (gint)errno;
		(void)close(fd);
		errno = (int)error_code;
		return(-1);
	}

	zd = gzdopen((int)fd, libz_oflags);
	if(zd == NULL)
	{
		errno = ENOMEM;
		return(-1);
	}

	/* Return the gzFile descriptor as the descriptor */
	return((int)zd);
}
# endif	/* HAVE_LIBZ  */

# ifdef HAVE_LIBBZ2
/*
 *	libtar BZip2 open callback.
 */
static int edv_archive_object_stat_libtar_open_libbz2_cb(const char *path, int oflags, int mode)
{
	gint fd;
	const char *libbz2_oflags;
	bz2_cb_data_struct *d;

	switch(oflags & O_ACCMODE)
	{
	  case O_WRONLY:
		libbz2_oflags = "wb";
		break;
	  case O_RDONLY:
		libbz2_oflags = "rb";
		break;
	  default:
	  case O_RDWR:
		errno = EINVAL;
		return(-1);
		break;
	}

	d = BZ2_CB_DATA(g_malloc0(
		sizeof(bz2_cb_data_struct)
	));
	if(d == NULL)
		return(-1);

	d->fd = fd = (gint)open(path, oflags, mode);
	if(fd < 0)
	{
		const gint error_code = (gint)errno;
		g_free(d);
		errno = (int)error_code;
		return(-1);
	}

	if(oflags & O_CREAT)
	{
		if(edv_fchmod(fd, mode))
		{
			const gint error_code = (gint)errno;
			(void)close(fd);
			g_free(d);
			errno = (int)error_code;
			return(-1);
		}
	}

	d->bz2d = BZ2_bzdopen((int)fd, libbz2_oflags);
	if(d->bz2d == NULL)
	{
		(void)close(fd);
		g_free(d);
		errno = ENOMEM;
		return(-1);
	}

	/* Return the libbz2 callback data as the descriptor */
	return((int)d);
}

/*
 *	libtar BZip2 read callback.
 */
static ssize_t edv_archive_object_stat_libtar_read_libbz2_cb(int fd, void *buf, size_t buf_len)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	return(BZ2_bzread(d->bz2d, buf, buf_len));
}

/*
 *	libtar BZip2 write callback.
 */
static ssize_t edv_archive_object_stat_libtar_write_libbz2_cb(int fd, const void *buf, size_t buf_len)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	return(BZ2_bzwrite(d->bz2d, (void *)buf, buf_len));
}

/*
 *	libtar BZip2 close callback.
 */
static int edv_archive_object_stat_libtar_close_libbz2_cb(int fd)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	BZ2_bzclose(d->bz2d);
	g_free(d);
	return(0);
}

# endif	/* HAVE_LIBBZ2 */

/*
 *	Gets the Tape Archive's current object's type.
 */
static EDVObjectType edv_archive_object_stat_libtar_get_type(TAR *tar)
{
	if(TH_ISREG(tar))
		return(EDV_OBJECT_TYPE_FILE);
	else if(TH_ISDIR(tar))
		return(EDV_OBJECT_TYPE_DIRECTORY);
	else if(TH_ISLNK(tar) || TH_ISSYM(tar))
		return(EDV_OBJECT_TYPE_LINK);
	else if(TH_ISBLK(tar))
		return(EDV_OBJECT_TYPE_DEVICE_BLOCK);
	else if(TH_ISCHR(tar))
		return(EDV_OBJECT_TYPE_DEVICE_CHARACTER);
	else if(TH_ISFIFO(tar))
		return(EDV_OBJECT_TYPE_FIFO);
	else
		return(EDV_OBJECT_TYPE_UNKNOWN);
}

/*
 *	Gets the Tape Archive's current object's path.
 */
static gchar *edv_archive_object_stat_libtar_get_path(TAR *tar)
{
	gchar *path;
	struct tar_header *tar_obj = &tar->th_buf;

	if(tar_obj->gnu_longname != NULL)
		path = g_strdup(tar_obj->gnu_longname);
	else if(tar_obj->prefix[0] != '\0')
		path = g_strdup_printf(
			"%.155s/%.100s",
			tar_obj->prefix,
			tar_obj->name
		);
	else
		path = g_strdup_printf(
			"%.100s", 
			tar_obj->name
		);

	/* Remove any tailing deliminators */
	edv_path_strip(path);

	return(path);
}

/*
 *	Seeks to the next object in the Tape Archive.
 */
static gint edv_archive_object_stat_libtar_next(TAR *tar)
{
	/* No need to seek to the next object if the current object
	 * is a file
	 */
	if(!TH_ISREG(tar))
		return(0);

	/* Seek past this file */
	if(tar_skip_regfile(tar) != 0)
		return(-1);

	return(0);
}

/*
 *	Creates a new archive object based on the current Tape Archive
 *	object.
 */
static EDVArchiveObject *edv_archive_object_stat_libtar_new_object(
	TAR *tar,
	const gulong index
)
{
	struct tar_header *tar_obj = &tar->th_buf;
	gchar *path;
	EDVArchiveObject *obj = edv_archive_object_new();
	if(obj == NULL)
		return(NULL);

	obj->type = edv_archive_object_stat_libtar_get_type(tar);
	obj->index = index;
	path = edv_archive_object_stat_libtar_get_path(tar);
	if(path != NULL)
	{
		obj->name = STRDUP(g_basename(path));
		obj->path = STRDUP(path);
		g_free(path);
	}
	obj->size = (gulong)th_get_size(tar);
/*	obj->storage_size = 0l; */
	if(obj->type == EDV_OBJECT_TYPE_LINK)
		obj->link_target = STRDUP(th_get_linkname(tar));
	obj->permissions = edv_stat_mode_to_edv_permissions(
		th_get_mode(tar)
	);
	obj->access_time = 0l;
	obj->modify_time = (gulong)th_get_mtime(tar);
	obj->change_time = 0l;
	obj->owner_name = STRDUP(tar_obj->uname);
	obj->group_name = STRDUP(tar_obj->gname);
	obj->device_type = (th_get_devmajor(tar) << 8) |
		th_get_devminor(tar);
/*	obj->encryption_name = NULL;	*/
/*	obj->compression_ratio = 0.0f; */
/*	obj->storage_method */
	obj->crc = g_strdup_printf(
		"%.8X",
		(guint32)th_get_crc(tar)
	);

	return(obj);
}
#endif	/* HAVE_LIBTAR */

/*
 *	Get Tape archive object statistics.
 */
static EDVArchiveObject *edv_archive_object_stat_tar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	gboolean is_compress_compressed,
	gboolean is_gzip_compressed,
	gboolean is_bzip2_compressed
)
{
#ifdef HAVE_LIBTAR
#ifdef HAVE_LIBZ
	tartype_t tar_io_z_cb = {
		(openfunc_t)edv_archive_object_stat_libtar_open_libz_cb,
		(closefunc_t)gzclose,
		(readfunc_t)gzread,
		(writefunc_t)gzwrite
	};
#endif
#ifdef HAVE_LIBBZ2
	tartype_t tar_io_bz2_cb = {
		(openfunc_t)edv_archive_object_stat_libtar_open_libbz2_cb,
		(closefunc_t)edv_archive_object_stat_libtar_close_libbz2_cb,
		(readfunc_t)edv_archive_object_stat_libtar_read_libbz2_cb,
		(writefunc_t)edv_archive_object_stat_libtar_write_libbz2_cb
	};
#endif
	tartype_t *tar_io_cbs;
	TAR *tar = NULL;
	gulong index;
	gchar *vpath;
	EDVArchiveObject *obj = NULL;

	/* Select the IO callbacks based on the compression format */
	if(is_compress_compressed)
	{
		tar_io_cbs = NULL;
	}
	else if(is_gzip_compressed)
	{
#ifdef HAVE_LIBZ
		tar_io_cbs = &tar_io_z_cb;
#else
		tar_io_cbs = NULL;
#endif
	}
	else if(is_bzip2_compressed)
	{
#ifdef HAVE_LIBBZ2
		tar_io_cbs = &tar_io_bz2_cb;
#else
		tar_io_cbs = NULL;
#endif
	}
	else
	{
		tar_io_cbs = NULL;
	}

	/* Open the Tape Archive for reading */
	if(tar_open(
		&tar,
		(char *)arch_path,
		tar_io_cbs,
		O_RDONLY, 0,
		TAR_GNU | TAR_NOOVERWRITE
	) == -1)
		return(obj);
	if(tar == NULL)
		return(obj);

	/* Begin reading each object in the Tape Archive */
	index = 1l;
	while(th_read(tar) == 0)
	{
		/* Get the path of this object in the Tape Archive */
		vpath = edv_archive_object_stat_libtar_get_path(tar);
		if(vpath == NULL)
		{
			if(edv_archive_object_stat_libtar_next(tar))
				return(obj);
			index++;
			continue;
		}

		/* Is this the object that we are looking for? */
		if(!strcmp((const char *)vpath, (const char *)path))
		{
			/* Create a new archive object based on the current
			 * Tape Archive object
			 */
			obj = edv_archive_object_stat_libtar_new_object(
				tar,
				index
			);
			if(obj != NULL)
			{
				if(is_compress_compressed)
					obj->storage_method = g_strdup("Tar Compress");
				else if(is_gzip_compressed)
					obj->storage_method = g_strdup("Tar GZip");
				else if(is_bzip2_compressed)
					obj->storage_method = g_strdup("Tar BZip2");
				else
					obj->storage_method = g_strdup("Tar");
			}
			g_free(vpath);
			break;
		}

		g_free(vpath);

		/* If this is object in the archive is of type file then
		 * the tar pointer must be seeked over to the next object
		 */
		if(edv_archive_object_stat_libtar_next(tar))
			return(obj);

		index++;
	}

	/* Close the Tape Archive */
	tar_close(tar);

	return(obj);
#else
	FILE            *cstdout,
			*cstderr;
	gint pid;
	gchar *cmd;
	const gchar	*prog_bunzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BUNZIP2),
			*prog_tar = EDV_GET_S(EDV_CFG_PARM_PROG_TAR);
	EDVArchiveObject *obj;

	/* Format the list TAR archive command */
	if(is_compress_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -Z -t -v -f \"%s\" \"%s\"",
			prog_tar,
			arch_path,
			path
		);
	else if(is_gzip_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -z -t -v -f \"%s\" \"%s\"",
			prog_tar,
			arch_path,
			path
		);
	else if(is_bzip2_compressed)
		cmd = g_strdup_printf(
			"\"%s\" \"--use-compress-program=%s\" -t -v -f \"%s\" \"%s\"",
			prog_tar,
			prog_bunzip2,
			arch_path,
			path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" -t -v -f \"%s\" \"%s\"",
			prog_tar,
			arch_path,
			path
		);
	if(cmd == NULL)
		return(NULL);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(NULL);
	}

	obj = NULL;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Begin reading each line and get stats of the line
		 * that describes the specified path
		 */
		while(!feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE		/* Block */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				obj = edv_archive_object_new();
				edv_archive_object_stat_parse_line_tar(
					obj,
					s
				);

				/* Set the method since it is not obtained from
				 * the line information
				 */
				if(is_compress_compressed)
				{
					g_free(obj->storage_method);
					obj->storage_method = STRDUP("Tar Compress");
				}
				else if(is_gzip_compressed)
				{
					g_free(obj->storage_method);
					obj->storage_method = STRDUP("Tar GZip");
				}
				else if(is_bzip2_compressed)
				{
					g_free(obj->storage_method);
					obj->storage_method = STRDUP("Tar BZip2");
				}
				else
				{
					g_free(obj->storage_method);
					obj->storage_method = STRDUP("Tar");
				}

				g_free(buf);
				buf = NULL;

				/* This is the object we want so don't
				 * bother reading any more
				 */
				break;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(obj);
#endif	/* !HAVE_LIBTAR */
}

#ifdef HAVE_LIBXAR
/*
 *	Gets the current object in the X Archive.
 */
static EDVArchiveObject *edv_archive_object_stat_xar_new_object(
	CfgList *cfg_list,
	xar_t xar,
	xar_file_t xar_fp,
	const gulong index,
	const gchar *path,
	const glong gmt_offset_dst
)
{
	const gchar	*key,
			*val;
	xar_iter_t xar_iter;

	/* Create a new archive object */
	EDVArchiveObject *obj = edv_archive_object_new();
	if(obj == NULL)
		return(NULL);

	obj->index = index;

	obj->name = STRDUP(g_basename(path));
	obj->path = STRDUP(path);

	/* Create a new X Archive property iterator */
	xar_iter = xar_iter_new(xar);
	if(xar_iter == 0)
	{
		const gint error_code = (gint)errno;
		edv_archive_object_delete(obj);
		errno = (int)error_code;
		return(NULL);
	}

	/* Iterate through each property and add each property's
	 * value to the archive object
	 */
	for(key = (const gchar *)xar_prop_first(xar_fp, xar_iter);
		key != NULL;
		key = (const gchar *)xar_prop_next(xar_iter)
	)
	{
		val = NULL;
		xar_prop_get(xar_fp, (const char *)key, (const char **)&val);

		/* If the property has no value then skip it */
		if(STRISEMPTY(val))
			continue;

		/* Name */
		if(!g_strcasecmp(key, "name"))
		{
			const gchar *name = val;
			g_free(obj->name);
			obj->name = STRDUP(name);
		}
		/* Type */
		else if(!g_strcasecmp(key, "type"))
		{
			const gchar *type = val;
			if(!g_strcasecmp(type, "file"))
			{
				obj->type = EDV_OBJECT_TYPE_FILE;
			}
			else if(!g_strcasecmp(type, "directory"))
			{
				obj->type = EDV_OBJECT_TYPE_DIRECTORY;
			}
			else if(!g_strcasecmp(type, "symlink") ||
				!g_strcasecmp(type, "hard link") ||
				!g_strcasecmp(type, "hardlink") ||
				!g_strcasecmp(type, "link")
			)
			{
				obj->type = EDV_OBJECT_TYPE_LINK;
			}
			else if(!g_strcasecmp(type, "block special") ||
				!g_strcasecmp(type, "block device") ||
				!g_strcasecmp(type, "blockspecial") ||
				!g_strcasecmp(type, "blockdevice")
			)
			{
					obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
			}
			else if(!g_strcasecmp(type, "character special") ||
				!g_strcasecmp(type, "character device") ||
				!g_strcasecmp(type, "characterspecial") ||
				!g_strcasecmp(type, "characterdevice")
			)
			{
				obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
			}
			else if(!g_strcasecmp(type, "fifo"))
			{
				obj->type = EDV_OBJECT_TYPE_FIFO;
			}
			else if(!g_strcasecmp(type, "socket"))
			{
				obj->type = EDV_OBJECT_TYPE_SOCKET;
			}
			else
			{
				obj->type = EDV_OBJECT_TYPE_UNKNOWN;
			}
		}
		/* Size */
		else if(!g_strcasecmp(key, "data/size") ||
			!g_strcasecmp(key, "size")
		)
		{
			obj->size = (gulong)atol((const char *)val);
		}
		/* Storage Size */
		else if(!g_strcasecmp(key, "data/length") ||
			!g_strcasecmp(key, "Storage Size")
		)
		{
			obj->storage_size = (gulong)atol((const char *)val);
		}
		/* Link Target */
		else if(!g_strcasecmp(key, "link"))
		{
			const gchar *link = val;
			g_free(obj->link_target);
			obj->link_target = STRDUP(link);
		}
		/* Permissions (Mode) */
		else if(!g_strcasecmp(key, "mode") ||
			!g_strcasecmp(key, "permissions")
		)
		{
			const gchar *permissions = val;
			const gint len = STRLEN(permissions);

			obj->permissions = 0;

			/* Is the value's format is numeric? */
			if((len > 0) ? isdigit(*permissions) : FALSE)
			{
				/* Numeric
				 *
				 * Check if the format is nnnn
				 */
				if(len >= 4)
				{
					/* nnnn */
					/* SetUID, SetGID, and Sticky */
					guint8 v = (guint8)(
						(gchar)permissions[0] - '0'
					);
					if(v & (1 << 0))
						obj->permissions |= EDV_PERMISSION_STICKY;
					if(v & (1 << 1))
						obj->permissions |= EDV_PERMISSION_SETGID;
					if(v & (1 << 2))
						obj->permissions |= EDV_PERMISSION_SETUID;

					/* User */
					v = (guint8)(
						(gchar)permissions[1] - '0'
					);
					if(v & (1 << 0))
						obj->permissions |= EDV_PERMISSION_UX;
					if(v & (1 << 1))
						obj->permissions |= EDV_PERMISSION_UW;
					if(v & (1 << 2))
						obj->permissions |= EDV_PERMISSION_UR;

					/* Group */
					v = (guint8)(
						(gchar)permissions[2] - '0'
					);
					if(v & (1 << 0))
						obj->permissions |= EDV_PERMISSION_GX;
					if(v & (1 << 1))
						obj->permissions |= EDV_PERMISSION_GW;
					if(v & (1 << 2))
						obj->permissions |= EDV_PERMISSION_GR;

					/* Other */
					v = (guint8)(
						(gchar)permissions[3] - '0'
					);
					if(v & (1 << 0))
						obj->permissions |= EDV_PERMISSION_OX;
					if(v & (1 << 1))
						obj->permissions |= EDV_PERMISSION_OW;
					if(v & (1 << 2))
						obj->permissions |= EDV_PERMISSION_OR;
				}
				/* nnn */
				else if(len == 3)
				{
					/* User */
					guint8 v = (guint8)(
						(gchar)permissions[0] - '0'
					);
					if(v & (1 << 0))
						obj->permissions |= EDV_PERMISSION_UX;
					if(v & (1 << 1))
						obj->permissions |= EDV_PERMISSION_UW;
					if(v & (1 << 2))
						obj->permissions |= EDV_PERMISSION_UR;

					/* Group */
					v = (guint8)(
						(gchar)permissions[1] - '0'
					);
					if(v & (1 << 0))
						obj->permissions |= EDV_PERMISSION_GX;
					if(v & (1 << 1))
						obj->permissions |= EDV_PERMISSION_GW;
					if(v & (1 << 2))
						obj->permissions |= EDV_PERMISSION_GR;

					/* Other */
					v = (guint8)(
						(gchar)permissions[2] - '0'
					);
					if(v & (1 << 0))
						obj->permissions |= EDV_PERMISSION_OX;
					if(v & (1 << 1))
						obj->permissions |= EDV_PERMISSION_OW;
					if(v & (1 << 2))
						obj->permissions |= EDV_PERMISSION_OR;
				}
				/* n */
				else if(len >= 1)
				{
					const guint8 v = (guint8)(
						(gchar)permissions[0] - '0'
					);
					if(v & (1 << 0))
					{
						obj->permissions |= EDV_PERMISSION_UX;
						obj->permissions |= EDV_PERMISSION_GX;
						obj->permissions |= EDV_PERMISSION_OX;
					}
					if(v & (1 << 1))
					{
						obj->permissions |= EDV_PERMISSION_UW;
						obj->permissions |= EDV_PERMISSION_GW;
						obj->permissions |= EDV_PERMISSION_OW;
					}
					if(v & (1 << 2))
					{
						obj->permissions |= EDV_PERMISSION_UR;
						obj->permissions |= EDV_PERMISSION_GR;
						obj->permissions |= EDV_PERMISSION_OR;
					}
				}
			}
			else
			{
				/* Alphabet
				 *
				 * trwxrwxrwx
				 */
				if(len >= 10)
				{
					if(permissions[1] == 'r')
						obj->permissions |= EDV_PERMISSION_UR;
					if(permissions[2] == 'w')
						obj->permissions |= EDV_PERMISSION_UW;
					if(permissions[3] == 'x')
						obj->permissions |= EDV_PERMISSION_UX;
					if(permissions[4] == 'r')
						obj->permissions |= EDV_PERMISSION_GR;
					if(permissions[5] == 'w')
						obj->permissions |= EDV_PERMISSION_GW;
					if(permissions[6] == 'x')
						obj->permissions |= EDV_PERMISSION_GX;
					if(permissions[7] == 'r')
						obj->permissions |= EDV_PERMISSION_OR;
					if(permissions[8] == 'w')
						obj->permissions |= EDV_PERMISSION_OW;
					if(permissions[9] == 'x')
						obj->permissions |= EDV_PERMISSION_OX;
				}
				/* rwxrwxrwx */
				else if(len == 9)
				{
					if(permissions[0] == 'r')
						obj->permissions |= EDV_PERMISSION_UR;
					if(permissions[1] == 'w')
						obj->permissions |= EDV_PERMISSION_UW;
					if(permissions[2] == 'x')
						obj->permissions |= EDV_PERMISSION_UX;
					if(permissions[3] == 'r')
						obj->permissions |= EDV_PERMISSION_GR;
					if(permissions[4] == 'w')
						obj->permissions |= EDV_PERMISSION_GW;
					if(permissions[5] == 'x')
						obj->permissions |= EDV_PERMISSION_GX;
					if(permissions[6] == 'r')
						obj->permissions |= EDV_PERMISSION_OR;
					if(permissions[7] == 'w')
						obj->permissions |= EDV_PERMISSION_OW;
					if(permissions[8] == 'x')
						obj->permissions |= EDV_PERMISSION_OX;
				}
				/* rwx */
				else if(len >= 3)
				{
					if(permissions[0] == 'r')
					{
						obj->permissions |= EDV_PERMISSION_UR;
						obj->permissions |= EDV_PERMISSION_GR;
						obj->permissions |= EDV_PERMISSION_OR;
					}
					if(permissions[1] == 'w')
					{
						obj->permissions |= EDV_PERMISSION_UW;
						obj->permissions |= EDV_PERMISSION_GW;
						obj->permissions |= EDV_PERMISSION_OW;
					}
					if(permissions[2] == 'x')
					{
						obj->permissions |= EDV_PERMISSION_UX;
						obj->permissions |= EDV_PERMISSION_GX;
						obj->permissions |= EDV_PERMISSION_OX;
					}
				}
			}
		}
		/* Owner (User) */
		else if(!g_strcasecmp(key, "user") ||
			!g_strcasecmp(key, "owner")
		)
		{
			const gchar *user = val;
			g_free(obj->owner_name);
			obj->owner_name = STRDUP(user);
		}
		/* Group */
		else if(!g_strcasecmp(key, "group"))
		{
			const gchar *group = val;
			g_free(obj->group_name);
			obj->group_name = STRDUP(group);
		}
		/* Access Time */
		else if(!g_strcasecmp(key, "atime"))
		{
			const gchar *atime = val;
			struct tm tm_buf;

			memset(&tm_buf, 0x00, sizeof(struct tm));
			tm_buf.tm_isdst = -1;

			/* Year */
			if(atime != NULL)
			{
				tm_buf.tm_year = (int)atoi((const char *)atime) - 1900;
				atime = (const gchar *)strchr(atime, '-');
				if(atime != NULL)
					atime++;
			}
			/* Month */
			if(atime != NULL)
			{
				tm_buf.tm_mon = (int)atoi((const char *)atime) - 1;
				atime = (const gchar *)strchr(atime, '-');
				if(atime != NULL)
					atime++;
			}
			/* Day */
			if(atime != NULL)
			{
				tm_buf.tm_mday = (int)atoi((const char *)atime);
				atime = (const gchar *)strchr(atime, 'T');
				if(atime != NULL)
					atime++;
			}
			/* Hours */
			if(atime != NULL)
			{
				tm_buf.tm_hour = (int)atoi((const char *)atime);
				atime = (const gchar *)strchr(atime, ':');
				if(atime != NULL)
					atime++;
			}
			/* Minutes */
			if(atime != NULL)
			{
				tm_buf.tm_min = (int)atoi((const char *)atime);
				atime = (const gchar *)strchr(atime, ':');
				if(atime != NULL)
					atime++;
			}
			/* Seconds */
			if(atime != NULL)
			{
				tm_buf.tm_sec = (int)atoi((const char *)atime);
			}

			obj->access_time = (gulong)((glong)mktime(&tm_buf) -
				gmt_offset_dst);
		}
		/* Modify Time */
		else if(!g_strcasecmp(key, "mtime"))
		{
			const gchar *mtime = val;
			struct tm tm_buf;

			memset(&tm_buf, 0x00, sizeof(struct tm));
			tm_buf.tm_isdst = -1;

			/* Year */
			if(mtime != NULL)
			{
				tm_buf.tm_year = (int)atoi((const char *)mtime) - 1900;
				mtime = (const gchar *)strchr(mtime, '-');
				if(mtime != NULL)
					mtime++;
			}
			/* Month */
			if(mtime != NULL)
			{
				tm_buf.tm_mon = (int)atoi((const char *)mtime) - 1;
				mtime = (const gchar *)strchr(mtime, '-');
				if(mtime != NULL)
					mtime++;
			}
			/* Day */
			if(mtime != NULL)
			{
				tm_buf.tm_mday = (int)atoi((const char *)mtime);
				mtime = (const gchar *)strchr(mtime, 'T');
				if(mtime != NULL)
					mtime++;
			}
			/* Hours */
			if(mtime != NULL)
			{
				tm_buf.tm_hour = (int)atoi((const char *)mtime);
				mtime = (const gchar *)strchr(mtime, ':');
				if(mtime != NULL)
					mtime++;
			}
			/* Minutes */
			if(mtime != NULL)
			{
				tm_buf.tm_min = (int)atoi((const char *)mtime);
				mtime = (const gchar *)strchr(mtime, ':');
				if(mtime != NULL)
					mtime++;
			}
			/* Seconds */
			if(mtime != NULL)
			{
				tm_buf.tm_sec = (int)atoi((const char *)mtime);
			}

			obj->modify_time = (gulong)((glong)mktime(&tm_buf) -
				gmt_offset_dst);
		}
		/* Change Time */
		else if(!g_strcasecmp(key, "ctime"))
		{
			const gchar *ctime = val;
			struct tm tm_buf;

			memset(&tm_buf, 0x00, sizeof(struct tm));
			tm_buf.tm_isdst = -1;

			/* Year */
			if(ctime != NULL)
			{
				tm_buf.tm_year = (int)atoi((const char *)ctime) - 1900;
				ctime = (const gchar *)strchr(ctime, '-');
				if(ctime != NULL)
					ctime++;
			}
			/* Month */
			if(ctime != NULL)
			{
				tm_buf.tm_mon = (int)atoi((const char *)ctime) - 1;
				ctime = (const gchar *)strchr(ctime, '-');
				if(ctime != NULL)
					ctime++;
			}
			/* Day */
			if(ctime != NULL)
			{
				tm_buf.tm_mday = (int)atoi((const char *)ctime);
				ctime = (const gchar *)strchr(ctime, 'T');
				if(ctime != NULL)
					ctime++;
			}
			/* Hours */
			if(ctime != NULL)
			{
				tm_buf.tm_hour = (int)atoi((const char *)ctime);
				ctime = (const gchar *)strchr(ctime, ':');
				if(ctime != NULL)
					ctime++;
			}
			/* Minutes */
			if(ctime != NULL)
			{
				tm_buf.tm_min = (int)atoi((const char *)ctime);
				ctime = (const gchar *)strchr(ctime, ':');
				if(ctime != NULL)
					ctime++;
			}
			/* Seconds */
			if(ctime != NULL)
			{
				tm_buf.tm_sec = (int)atoi((const char *)ctime);
			}

			obj->change_time = (gulong)((glong)mktime(&tm_buf) -
				gmt_offset_dst);
		}
		/* Compression Method (Encoding) */
		else if(!g_strcasecmp(key, "data/encoding") ||
			!g_strcasecmp(key, "encoding") ||
			!g_strcasecmp(key, "method")
		)
		{
			const gchar *encoding = val;
			g_free(obj->storage_method);
			obj->storage_method = STRDUP(encoding);
		}
		/* CRC (Archived Checksum) */
		else if(!g_strcasecmp(key, "data/archived-checksum") ||
			!g_strcasecmp(key, "data/checksum") ||
			!g_strcasecmp(key, "checksum") ||
			!g_strcasecmp(key, "crc")
		)
		{
			const gchar *checksum = val;
			g_free(obj->crc);
			obj->crc = STRDUP(checksum);
			g_strup(obj->crc);
		}
	}

	/* Delete the X Archive property iterator */
	xar_iter_free(xar_iter);

	/* Calculate the compression ratio */
	if(obj->size > 0l)
	{
		obj->compression_ratio = (1.0f -
			((gfloat)obj->storage_size / (gfloat)obj->size)
		);
		if(obj->compression_ratio < 0.0f)
			obj->compression_ratio = 0.0f;
	}
	else
		obj->compression_ratio = 0.0f;

	return(obj);
}
#endif	/* HAVE_LIBXAR */

/*
 *	Get XArchive archive object statistics.
 */
static EDVArchiveObject *edv_archive_object_stat_xar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
)
{
#ifdef HAVE_LIBXAR
	time_t t;
	struct tm *tm_local;
	xar_t xar;
	xar_iter_t i1;
	xar_file_t xar_fp;
	gulong index;
	glong gmt_offset_dst;
	gchar *cur_path;
	EDVArchiveObject *obj;

	/* Open the X Archive for reading */
	xar = xar_open(arch_path, READ);
	if(xar == NULL)
		return(NULL);

	/* Create a new X Archive object iterator */
	i1 = xar_iter_new(xar);
	if(i1 == 0)
	{
		const gint error_code = (gint)errno;
		(void)xar_close(xar);
		errno = (int)error_code;
		return(NULL);
	}

	/* Initialize the time globals timezone & daylight */
	t = (time_t)edv_time();
	tm_local = localtime(&t);

	/* Calculate the GMT offset with DST */
	gmt_offset_dst = (glong)(timezone - ((daylight == 1) ? (60 * 60) : 0));

	/* Iterate through each object in the X Archive */
	obj = NULL;
	for(xar_fp = xar_file_first(xar, i1),
		index = 1l;
		xar_fp != NULL;
		xar_fp = xar_file_next(i1),
		index++
	)
	{
		/* Get the path of this object within the X Archive */
		cur_path = (gchar *)xar_get_path(xar_fp);
		if(cur_path == NULL)
			continue;

		/* Not the object that we are looking for? */
		if(strcmp((const char *)cur_path, (const char *)path))
		{
			g_free(cur_path);
			continue;
		}

		g_free(cur_path);

		/* Get this object from the X Archive */
		obj = edv_archive_object_stat_xar_new_object(
			cfg_list,
			xar,
			xar_fp,
			index,
			path,
			gmt_offset_dst
		);

		break;
	}

	/* Delete the iterator and close the X Archive */
	xar_iter_free(i1);
	(void)xar_close(xar);

	return(obj);
#else
	return(NULL);
#endif
}

#ifdef HAVE_LIBZIP
/*
 *	Gets the current object in the PKZip Archive.
 */
static EDVArchiveObject *edv_archive_object_stat_libzip_new_object(
	CfgList *cfg_list,
	struct zip *archive,
	struct zip_stat *zip_stat_buf
)
{
	gchar		*path,
			*path_with_postfix;

	/* Create a new archive object */
	EDVArchiveObject *obj = edv_archive_object_new();
	if(obj == NULL)
		return(NULL);

	/* Get the index */
	obj->index = (gulong)(zip_stat_buf->index + 1);	/* Offset by + 1 */

	/* Name and Path */
	path_with_postfix = STRDUP(zip_stat_buf->name);
	path = STRDUP(path_with_postfix);
	if(path != NULL)
	{
		const gint len = strlen(path);

		/* Determine the type by the last character of the name */
		switch((len > 1) ? path[len - 1] : '\0')
		{
		  case G_DIR_SEPARATOR:
			obj->type = EDV_OBJECT_TYPE_DIRECTORY;
			path[len - 1] = '\0';
			break;
		  case '@':
			obj->type = EDV_OBJECT_TYPE_LINK;
			path[len - 1] = '\0';
			break;
		  default:
			obj->type = EDV_OBJECT_TYPE_FILE;
			break;
		}

		obj->path = g_strdup(path);
		obj->name = STRDUP(g_basename(path));
	}

	/* Permissions */
	obj->permissions = 0;			/* Not supported by the PKZip format */

	/* Time Stamps */
	obj->access_time = (gulong)zip_stat_buf->mtime;
	obj->modify_time = (gulong)zip_stat_buf->mtime;
	obj->change_time = (gulong)zip_stat_buf->mtime;

	/* Size */
	obj->size = (gulong)zip_stat_buf->size;

	/* Storage Size */
	obj->storage_size = (gulong)zip_stat_buf->comp_size;

	/* Compression Ratio */
	if(obj->size > 0l)
	{
		obj->compression_ratio = (1.0f -
			((gfloat)obj->storage_size / (gfloat)obj->size)
		);
		if(obj->compression_ratio < 0.0f)
			obj->compression_ratio = 0.0f;
	}
	else
		obj->compression_ratio = 0.0f;

	/* Compression Method */
	switch(zip_stat_buf->comp_method)
	{
#ifdef ZIP_CM_STORE
	    case ZIP_CM_STORE:
		obj->storage_method = g_strdup("Store");
		break;
#endif
#ifdef ZIP_CM_SHRINK
	    case ZIP_CM_SHRINK:
		obj->storage_method = g_strdup("Shrink");
		break;
#endif
#ifdef ZIP_CM_REDUCE_1
	    case ZIP_CM_REDUCE_1:
		obj->storage_method = g_strdup("Reduce 1");
		break;
#endif
#ifdef ZIP_CM_REDUCE_2
	    case ZIP_CM_REDUCE_2:
		obj->storage_method = g_strdup("Reduce 2");
		break;
#endif
#ifdef ZIP_CM_REDUCE_3
	    case ZIP_CM_REDUCE_3:
		obj->storage_method = g_strdup("Reduce 3");
		break;
#endif
#ifdef ZIP_CM_REDUCE_4
	    case ZIP_CM_REDUCE_4:
		obj->storage_method = g_strdup("Reduce 4");
		break;
#endif
#ifdef ZIP_CM_IMPLODE
	    case ZIP_CM_IMPLODE:
		obj->storage_method = g_strdup("Implode");
		break;
#endif
#ifdef ZIP_CM_DEFLATE
	    case ZIP_CM_DEFLATE:
		obj->storage_method = g_strdup("Deflate");
		break;
#endif
#ifdef ZIP_CM_DEFLATE64
	    case ZIP_CM_DEFLATE64:
		obj->storage_method = g_strdup("Deflate 64");
		break;
#endif
#ifdef ZIP_CM_PKWARE_IMPLODE
	    case ZIP_CM_PKWARE_IMPLODE:
		obj->storage_method = g_strdup("PKWare Implode");
		break;
#endif
	    default:
		obj->storage_method = g_strdup("Other");
		break;
	}

	/* Encryption Method */
	if(zip_stat_buf->encryption_method != ZIP_EM_NONE)
	{
		switch(zip_stat_buf->encryption_method)
		{
#ifdef ZIP_EM_TRAD_PKWARE
		    case ZIP_EM_TRAD_PKWARE:
			obj->encryption_name = g_strdup("PKWare");
			break;
#endif
#ifdef ZIP_EM_DES
		    case ZIP_EM_DES:
			obj->encryption_name = g_strdup("DES");
			break;
#endif
#ifdef ZIP_EM_RC2_OLD
		    case ZIP_EM_RC2_OLD:
			obj->encryption_name = g_strdup("RC2 (Old)");
			break;
#endif
#ifdef ZIP_EM_3DES_168
		    case ZIP_EM_3DES_168:
			obj->encryption_name = g_strdup("3DES 168");
			break;
#endif
#ifdef ZIP_EM_3DES_112
		    case ZIP_EM_3DES_112:
			obj->encryption_name = g_strdup("3DES 112");
			break;
#endif
#ifdef ZIP_EM_AES_128
		    case ZIP_EM_AES_128:
			obj->encryption_name = g_strdup("AES 128");
			break;
#endif
#ifdef ZIP_EM_AES_192
		    case ZIP_EM_AES_192:
			obj->encryption_name = g_strdup("AES 192");
			break;
#endif
#ifdef ZIP_EM_AES_256
		    case ZIP_EM_AES_256:
			obj->encryption_name = g_strdup("AES 256");
			break;
#endif
#ifdef ZIP_EM_RC2
		    case ZIP_EM_RC2:
			obj->encryption_name = g_strdup("RC2");
			break;
#endif
#ifdef ZIP_EM_RC4
		    case ZIP_EM_RC4:
			obj->encryption_name = g_strdup("RC4");
			break;
#endif
#ifdef ZIP_EM_UNKNOWN
		    case ZIP_EM_UNKNOWN:
			obj->encryption_name = g_strdup("Unknown");
			break;
#endif
		    default:
			obj->encryption_name = g_strdup("Other");
			break;
		}
	}

	/* CRC */
	obj->crc = g_strdup_printf(
		"%.8X",
		(guint32)zip_stat_buf->crc
	);

	/* If this object is a link then get the target value */
	if(obj->type == EDV_OBJECT_TYPE_LINK)
	{
		/* Open this object in the archive for reading */
		struct zip_file *zfp = zip_fopen(
			archive,
			path_with_postfix,
			0
		);
		if(zfp != NULL)
		{
			/* Get the link's target value */
			const gint len = MAX((gint)obj->size, 0);

			g_free(obj->link_target);
			obj->link_target = (gchar *)g_malloc(
				(len + 1) * sizeof(gchar)
			);
			if(obj->link_target != NULL)
			{
				const gint bytes_read = (gint)zip_fread(
					zfp,
					obj->link_target,
					len
				);
				if(bytes_read >= 0)
				{
					if(bytes_read < len)
						obj->link_target[bytes_read] = '\0';
					else
						obj->link_target[len] = '\0';
				}
			}

			/* Close this object in the archive */
			(void)zip_fclose(zfp);
		}
	}

	g_free(path_with_postfix);
	g_free(path);

	return(obj);
}
#endif	/* HAVE_LIBZIP */

/*
 *	Get PKZip archive object statistics.
 */
static EDVArchiveObject *edv_archive_object_stat_zip(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path
)
{
#ifdef HAVE_LIBZIP
	int zip_error;
	struct zip *archive;
	struct zip_stat zip_stat_buf;
	gint i;
	EDVArchiveObject *obj;

	/* Open the PKZip archive */
	archive = zip_open(arch_path, 0, &zip_error);
	if(archive == NULL)
		return(NULL);

	/* Find the object in the PKZip archive by the specified path */
	i = zip_name_locate(
		archive,
		path,
		0
	);
	if(i < 0)
	{
		/* Unable to find the object
		 *
		 * If the object is a link or a directory then it
		 * may need a postfix, try directory first
		 */
		gchar *path2 = g_strconcat(
			path,
			G_DIR_SEPARATOR_S,
			NULL
		);
		if(path2 == NULL)
		{
			(void)zip_close(archive);
			return(NULL);
		}
		i = zip_name_locate(
			archive,
			path2,
			0
		);
		g_free(path2);
		if(i < 0)
		{
			/* Try link */
			gchar *path2 = g_strconcat(
				path,
				"@",
				NULL
			);
			if(path2 == NULL)
			{
				(void)zip_close(archive);
				return(NULL);
			}
			i = zip_name_locate(archive, path2, 0);
			g_free(path2);
			if(i < 0)
			{
				/* Object could not be found */
				(void)zip_close(archive);
				errno = ENOENT;
				return(NULL);
			}
		}
	}

	/* Get the stats for this object in the PKZip archive */
	if(zip_stat_index(
		archive,
		i,
		0,
		&zip_stat_buf
	))
	{
		(void)zip_close(archive);
		return(NULL);
	}

	/* Get this object from the PKZip archive */
	obj = edv_archive_object_stat_libzip_new_object(
		cfg_list,
		archive,
		&zip_stat_buf
	);

	/* Close the PKZip archive */
	(void)zip_close(archive);

	return(obj);
#else
	FILE            *cstdout,
			*cstderr;
	gint pid;
	const gchar *prog_unzip = EDV_GET_S(EDV_CFG_PARM_PROG_UNZIP);
	EDVArchiveObject *obj;

	/* Format the list PKZip archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" -l -v \"%s\" \"%s\"",
		prog_unzip,
		arch_path,
		path
	);
	if(cmd == NULL)
		return(NULL);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(NULL);
	}

	obj = NULL;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		gboolean got_header = FALSE;
		const gchar *delim_header = "--------";
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE            /* Block */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					got_header = TRUE;
					break;
				}
				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Able to read past the header? */
		if(got_header)
		{
			/* Read the next line */
			gint i;
			for(i = 0; (i < 1) && !feof(fp); i++)
			{
				FDISCARD(cstderr);
				(void)edv_stream_read_lineptr(
					fp,
					&buf,
					TRUE	/* Block */
				);
			}
			if(buf != NULL)
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				obj = edv_archive_object_new();
				edv_archive_object_stat_parse_line_zip(
					obj,
					s
				);

				g_free(buf);
				buf = NULL;
			}
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(obj);
#endif	/* HAVE_LIBZIP */
}

/*
 *	Get archive object statistics.
 *
 *	The cfg_list specifies the CfgList configuration list.
 *
 *	The arch_path specifies the path to the archive object.
 *
 *	The path specifies the path to the object in the archive.
 *
 *	If password is not NULL then it specifies the password.
 *
 *	Returns a dynamically allocated EDVArchiveObject describing
 *	the archive object statistics or NULL on error.
 */
EDVArchiveObject *edv_archive_object_stat(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	const gchar *password
)
{
	gint error_code;
	const gchar *arch_name;
	EDVVFSObject *arch_obj;
	EDVArchiveObject *obj;

	if(STRISEMPTY(arch_path) || STRISEMPTY(path))
	{
		errno = EINVAL;
		return(NULL);
	}

	arch_name = g_basename(arch_path);

	/* Get the VFS statistics of the archive object */
	arch_obj = edv_vfs_object_stat(arch_path);
	if(arch_obj == NULL)
		return(NULL);

	if(EDV_VFS_OBJECT_IS_DIRECTORY(arch_obj))
	{
		edv_vfs_object_delete(arch_obj);
		errno = EISDIR;
		return(NULL);
	}

	/* Get the statistics by the archive's extension */

	/* ARJ Archive */
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		obj = edv_archive_object_stat_arj(
			cfg_list,
			arch_path,
			path
		);
	}
	/* LHA Archive */
	else if(edv_name_has_extension(arch_name, ".lha"))
	{
		obj = edv_archive_object_stat_lha(
			cfg_list,
			arch_path,
			path
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		obj = edv_archive_object_stat_rar(
			cfg_list,
			arch_path,
			path
		);
	}
	/* RedHat Package Manager Package */
	else if(edv_name_has_extension(arch_name, ".rpm"))
	{
		obj = edv_archive_object_stat_rpm(
			cfg_list,
			arch_path,
			path
		);
	}
	/* Tape Archive (Compressed) */
	else if(edv_name_has_extension(arch_name, ".tar.Z"))
	{
		obj = edv_archive_object_stat_tar(
			cfg_list,
			arch_path,
			path,
			TRUE,				/* Is compress compressed */
			FALSE,				/* Not gzip compressed */
			FALSE				/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (GZip) */
	else if(edv_name_has_extension(arch_name, ".tgz .tar.gz"))
	{
		obj = edv_archive_object_stat_tar(
			cfg_list,
			arch_path,
			path,
			FALSE,			/* Not compress compressed */
			TRUE,			/* Is gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (BZip2) */
	else if(edv_name_has_extension(arch_name, ".tar.bz2"))
	{
		obj = edv_archive_object_stat_tar(
			cfg_list,
			arch_path,
			path,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			TRUE			/* Is bzip2 compressed */
		);
	}
	/* Tape Archive */
	else if(edv_name_has_extension(arch_name, ".tar"))
	{
		obj = edv_archive_object_stat_tar(
			cfg_list,
			arch_path,
			path,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* X Archive */
	else if(edv_name_has_extension(arch_name, ".xar"))
	{
		obj = edv_archive_object_stat_xar(
			cfg_list,
			arch_path,
			path
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		obj = edv_archive_object_stat_zip(
			cfg_list,
			arch_path,
			path
		);
	}
	else
	{
		obj = NULL;
		errno = EINVAL;
	}

	error_code = (gint)errno;

	edv_vfs_object_delete(arch_obj);

	errno = (int)error_code;

	return(obj);
}


/*
 *	Get a list of ARJ archive object statistics.
 */
static gint edv_archive_object_stat_list_sequential_arj(EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES)
{
	FILE		*cstdout,
			*cstderr;
	gint            pid,
			status;
	const gchar *prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);

	/* Format the list ARJ archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" v -i -y \"%s\"",
		prog_arj,
		arch_path
	);
	if(cmd == NULL)
		return(-3);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(-1);
	}

	status = 0;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		const gchar *delim_header = "------------";
		gint line_counter;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE           /* Nonblocking */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					break;
				}

				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Begin reading each set of four lines describing each
		 * object in the ARJ archive
		 */
		line_counter = 0;
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE           /* Nonblocking */
			))
				line_counter++;

			/* Ending deliminator reached? If so do not parse this
			 * line or any subsequent lines
			 */
			if(buf != NULL)
			{
				if(*buf == '-')
					break;
			}

			/* Got four lines? */
			if(line_counter >= 4)
			{
				gint i;
				gchar   *s = buf,
					*path;
				while(ISBLANK(*s))
					s++;

				/* Seek path to the full path value
				 * (skipping 1 argument)
				 */
				path = s;
				for(i = 0; i < 1; i++)
				{
					while(!ISBLANK(*path) && (*path != '\0'))
						path++;
					while(ISBLANK(*path))
						path++;
				}

				/* Copy the path value and cap the first
				 * newline character
				 */
				path = g_strdup(path);
				edv_archive_object_stat_cap_newline(path);

				/* Check if the path matches any of the paths in
				 * the paths list or filter
				 */
				if(edv_archive_object_stat_filter(
					paths_list,
#if defined(HAVE_REGEX)
					regex_filter,
#else
					filter,
#endif
					path
				))
				{
					EDVArchiveObject *obj = edv_archive_object_new();
					edv_archive_object_stat_parse_line_arj(
						obj,
						s
					);

					/* Return this object */
					if(obj_rtn_cb(
						arch_path,
						obj,
						0l, 0l,
						obj_rtn_data
					))
					{
						g_free(path);
						status = -4;
						break;
					}
				}

				g_free(path);
				g_free(buf);
				buf = NULL;
				line_counter = 0;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(status);
}

/*
 *	Get a list of LHA archive object statistics.
 */
static gint edv_archive_object_stat_list_sequential_lha(EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES)
{
	FILE		*cstdout,
			*cstderr;
	gint		pid,
			status;
	const gchar *prog_lha = EDV_GET_S(EDV_CFG_PARM_PROG_LHA);

	/* Format the list LHA archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" v \"%s\"",
		prog_lha,
		arch_path
	);
	if(cmd == NULL)
		return(-3);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(-1);
	}

	status = 0;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		time_t cur_time = (time_t)edv_time();
		const struct tm *tm_ptr;
		struct tm cur_mtime_buf;
		const gchar	*delim_header = "----------",
				*delim_footer = "----------";
		gint line_counter;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Get the current time */
		tm_ptr = localtime(&cur_time);
		if(tm_ptr != NULL)
			(void)memcpy(&cur_mtime_buf, tm_ptr, sizeof(struct tm));
		else
			(void)memset(&cur_mtime_buf, 0x00, sizeof(struct tm));

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE           /* Nonblocking */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					break;
				}

				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Begin reading each line describing each object in
		 * the LHA archive
		 */
		line_counter = 0;
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
				line_counter++;

			/* Footer deliminator reached? */
			if(STRPFX(buf, delim_footer))
				break;

			/* Got a line? */
			if(line_counter >= 1)
			{
				gint i;
				gchar   *s = buf,
					*path;
				while(ISBLANK(*s))
					s++;

				/* Seek path to the path value
				 * (skipping 10 arguments)
				 */
				path = s;
				for(i = 0; i < 10; i++)
				{
					while(!ISBLANK(*path) && (*path != '\0'))
						path++;
					while(ISBLANK(*path))
						path++;
				}

				/* Copy the path value and cap the first
				 * newline character
				 */
				path = g_strdup(path);
				edv_archive_object_stat_cap_newline(path);

				/* Check if the path matches any of the
				 * paths in the paths list or filter
				 */
				if(edv_archive_object_stat_filter(
					paths_list,
#if defined(HAVE_REGEX)
					regex_filter,
#else                            
					filter,
#endif
					path
				))
				{
					EDVArchiveObject *obj = edv_archive_object_new();
					edv_archive_object_stat_parse_line_lha(
						obj,
						s,
						&cur_mtime_buf
					);

					/* Report this object */
					if(obj_rtn_cb(
						arch_path,
						obj,
						0l, 0l,
						obj_rtn_data
					))
					{
						g_free(path);
						status = -4;
						break;
					}
				}

				g_free(path);
				g_free(buf);
				buf = NULL;
				line_counter = 0;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(status);
}

/*
 *	Get a list of RAR archive object statistics.
 */
static gint edv_archive_object_stat_list_sequential_rar(EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES)
{
	FILE		*cstdout,
			*cstderr;
	gint		pid,
			status;
	const gchar *prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);

	/* Format the list RAR archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" v -y -c- \"%s\"",
		prog_rar,
		arch_path
	);
	if(cmd == NULL)
		return(-3);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(-1);
	}

	status = 0;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		const gchar	*delim_header = "------------",
				*delim_footer = "------------";
		gint line_counter;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					break;
				}

				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Begin reading each set of two lines describing each
		 * object in the RAR archive
		 */
		line_counter = 0;
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
				line_counter++;

			/* Footer deliminator reached? */
			if(STRPFX(buf, delim_footer))
				break;

			/* Got two lines? */
			if(line_counter >= 2)
			{
				gchar	*s = buf,
					*path;
				while(ISBLANK(*s))
					s++;

				/* Seek path to the path value */
				path = s;
				while(*path == '*')
					path++;

				/* Copy the path value and cap the first
				 * newline character
				 */
				path = g_strdup(path);
				edv_archive_object_stat_cap_newline(path);

				/* Check if the path matches any of the
				 * paths in the paths list or filter
				 */
				if(edv_archive_object_stat_filter(
					paths_list,
#if defined(HAVE_REGEX)
					regex_filter,
#else                            
					filter,
#endif
					path
				))
				{
					EDVArchiveObject *obj = edv_archive_object_new();
					edv_archive_object_stat_parse_line_rar(
						obj,
						s
					);

					/* Report this object */
					if(obj_rtn_cb(
						arch_path,
						obj,
						0l, 0l,
						obj_rtn_data
					))
					{
						g_free(path);
						status = -4;
						break;
					}
				}

				g_free(path);
				g_free(buf);
				buf = NULL;
				line_counter = 0;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(status);
}

/*
 *	Get a list of RPM archive object statistics.
 */
static gint edv_archive_object_stat_list_sequential_rpm(EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES)
{
	FILE		*cstdout,
			*cstderr;
	gint		pid,
			status;
	const gchar *prog_rpm = EDV_GET_S(EDV_CFG_PARM_PROG_RPM);

	/* Format the list RPM archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" -q -l -p -v \"%s\"",
		prog_rpm,
		arch_path
	);
	if(cmd == NULL)
		return(-3);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(-1);
	}

	status = 0;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Begin reading each line describing each object in
		 * the RedHat Package Manager package
		 */
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gint i;
				gchar	*s = buf,
					*path;
				while(ISBLANK(*s))
					s++;

				/* Seek path to the path value,
				 * skipping past 8 values
				 */
				path = s;
				for(i = 0; i < 8; i++)
				{
					while(!ISBLANK(*path) && (*path != '\0'))
						path++;
					while(ISBLANK(*path))
						path++;
				}

				/* Copy the path value and cap the first
				 * newline character
				 */
				path = g_strdup(path);
				edv_archive_object_stat_cap_newline(path);

				/* Check if the path matches any of the
				 * paths in the paths list or filter
				 */
				if(edv_archive_object_stat_filter(
					paths_list,
#if defined(HAVE_REGEX)
					regex_filter,
#else                            
					filter,
#endif
					path
				))
				{
					EDVArchiveObject *obj = edv_archive_object_new();
					edv_archive_object_stat_parse_line_rpm(
						obj,
						s
					);

					/* Report this object */
					if(obj_rtn_cb(
						arch_path,
						obj,
						0l, 0l,
						obj_rtn_data
					))
					{
						g_free(path);
						status = -4;
						break;
					}
				}

				g_free(path);
				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(status);
}

/*
 *	Get a list of Tape archive object statistics.
 */
static gint edv_archive_object_stat_list_sequential_tar(
	EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
#ifdef HAVE_LIBTAR
#ifdef HAVE_LIBZ
	tartype_t tar_io_z_cb = {
		(openfunc_t)edv_archive_object_stat_libtar_open_libz_cb,
		(closefunc_t)gzclose,
		(readfunc_t)gzread,
		(writefunc_t)gzwrite
	};
#endif
#ifdef HAVE_LIBBZ2
	tartype_t tar_io_bz2_cb = {
		(openfunc_t)edv_archive_object_stat_libtar_open_libbz2_cb,
		(closefunc_t)edv_archive_object_stat_libtar_close_libbz2_cb,
		(readfunc_t)edv_archive_object_stat_libtar_read_libbz2_cb,
		(writefunc_t)edv_archive_object_stat_libtar_write_libbz2_cb
	};
#endif
	tartype_t *tar_io_cbs;
	TAR *tar = NULL;
	gint status;
	gulong		index,
			fpos;
	gchar *vpath;

	/* Select the IO callbacks based on the compression format */
	if(is_compress_compressed)
	{
		tar_io_cbs = NULL;
	}
	else if(is_gzip_compressed)
	{
#ifdef HAVE_LIBZ
		tar_io_cbs = &tar_io_z_cb;
#else
		tar_io_cbs = NULL;
#endif
	}
	else if(is_bzip2_compressed)
	{
#ifdef HAVE_LIBBZ2
		tar_io_cbs = &tar_io_bz2_cb;
#else
		tar_io_cbs = NULL;
#endif
	}
	else
	{
		tar_io_cbs = NULL;
	}

	/* Open the Tape Archive for reading */
	if(tar_open(
		&tar,
		(char *)arch_path,
		tar_io_cbs,
		O_RDONLY, 0,
		TAR_GNU | TAR_NOOVERWRITE
	) == -1)
		return(-1);

	if(tar == NULL)
		return(-1);

	/* Begin reading each object in the Tape Archive */
	status = 0;
	fpos = 0l;
	index = 1l;
	while(th_read(tar) == 0)
	{
		/* Get the current position in the Tape Archive in bytes */
		if(is_compress_compressed)
			fpos = 0l;
		else if(is_gzip_compressed)
			fpos = (gulong)gzseek(
				(gzFile)tar_fd(tar),
				0,
				SEEK_CUR
			);
		else if(is_bzip2_compressed)
		{
			bz2_cb_data_struct *d = BZ2_CB_DATA(tar_fd(tar));
			fpos = (gulong)lseek(
				(int)d->fd,
				0,
				SEEK_CUR
			);
		}
		else
			fpos = (gulong)lseek(
				(int)tar_fd(tar),
				0,
				SEEK_CUR
			);

		/* Report progress */
		if(obj_rtn_cb(
			arch_path,
			NULL,
			fpos, arch_size_bytes,
			obj_rtn_data
		))
		{
			status = -4;
			break;
		}

		/* Get the path of this object in the Tape Archive */
		vpath = edv_archive_object_stat_libtar_get_path(tar);
		if(vpath == NULL)
		{
			/* Unable to get the path of this object */
			if(edv_archive_object_stat_libtar_next(tar))
			{
				status = -1;
				return(status);
			}
			index++;
			continue;
		}

		/* Check if the path matches any of the paths in
		 * the paths list or filter
		 */
		if(edv_archive_object_stat_filter(
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else                            
			filter,
#endif
			vpath
		))
		{
			/* Create a new archive object based on the current
			 * Tape Archive object
			 */
			EDVArchiveObject *obj = edv_archive_object_stat_libtar_new_object(
				tar,
				index
			);
			if(obj != NULL)
			{
				if(is_compress_compressed)
					obj->storage_method = STRDUP("Tar Compress");
				else if(is_gzip_compressed)
					obj->storage_method = STRDUP("Tar GZip");
				else if(is_bzip2_compressed)
					obj->storage_method = STRDUP("Tar BZip2");
				else
					obj->storage_method = STRDUP("Tar");

				/* Report this object */
				if(obj_rtn_cb(
					arch_path,
					obj,
					fpos, arch_size_bytes,
					obj_rtn_data
				))
				{
					g_free(vpath);
					status = -4;
					break;
				}
			}
			else
			{
				g_free(vpath);
				status = -3;
				break;
			}
		}

		g_free(vpath);

		/* If this is object in the archive is of type file then
		 * the tar pointer must be seeked over to the next object
		 */
		if(edv_archive_object_stat_libtar_next(tar))
		{
			status = -1;
			return(status);
		}

		index++;
	}

	/* Close the Tape Archive */
	if(tar_close(tar) != 0)
	{
		if((status == 0) || (status == -4))
			status = -1;
	}

	/* Report the final progress */
	if(status == 0)
	{
		if(obj_rtn_cb(
			arch_path,
			NULL,
			arch_size_bytes, arch_size_bytes,
			obj_rtn_data
		))
			status = -4;
	}

	return(status);
#else
	FILE            *cstdout,
			*cstderr;
	gint            pid,
			status;
	gchar *cmd;
	const gchar	*prog_tar = EDV_GET_S(EDV_CFG_PARM_PROG_TAR),
			*prog_bunzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BUNZIP2);

	/* Format the list Tape archive command */
	if(is_compress_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -Z -t -v -f \"%s\"",
			prog_tar,
			arch_path
		);
	else if(is_gzip_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -z -t -v -f \"%s\"",
			prog_tar,
			arch_path
		);
	else if(is_bzip2_compressed)
		cmd = g_strdup_printf(
			"\"%s\" \"--use-compress-program=%s\" -t -v -f \"%s\"",
			prog_tar,
			prog_bunzip2,
			arch_path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" -t -v -f \"%s\"",
			prog_tar,
			arch_path
		);
	if(cmd == NULL)
		return(-3);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(-1);
	}

	status = 0;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Begin reading each line describing each object in
		 * the Tape Archive
		 */
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gint i;
				gchar   *s = buf,
					*path;
				while(ISBLANK(*s))
					s++;

				/* Seek path to the path value,
				 * skipping 5 values
				 */
				path = s;
				for(i = 0; i < 5; i++)
				{
					while(!ISBLANK(*path) && (*path != '\0'))
						path++;
					while(ISBLANK(*path))
						path++;
				}

				/* Copy the path value and cap the " -> "
				 * or the newline character.
				 */
				path = g_strdup(path);
				if(path != NULL)
				{
					gchar *s = (gchar *)strstr((char *)path, " -> ");
					if(s == NULL)
						s = (gchar *)strpbrk((char *)path, "\n\r");
					if(s != NULL)
						*s = '\0';
				}

				/* Check if the path matches any of the
				 * paths in the paths list or filter
				 */
				if(edv_archive_object_stat_filter(
					paths_list,
#if defined(HAVE_REGEX)
					regex_filter,
#else                            
					filter,
#endif
					path
				))
				{
					EDVArchiveObject *obj = edv_archive_object_new();
					edv_archive_object_stat_parse_line_tar(
						obj,
						s
					);

					/* Set the method since it is not obtained
					 * from the line information
					 */
					if(is_compress_compressed)
					{
						g_free(obj->storage_method);
						obj->storage_method = g_strdup("Tar Compress");
					}
					else if(is_gzip_compressed)
					{
						g_free(obj->storage_method);
						obj->storage_method = g_strdup("Tar GZip");
					}
					else if(is_bzip2_compressed)
					{
						g_free(obj->storage_method);
						obj->storage_method = g_strdup("Tar BZip2");
					}
					else
					{
						g_free(obj->storage_method);
						obj->storage_method = g_strdup("Tar");
					}

					/* Report this object */
					if(obj_rtn_cb(
						arch_path,
						obj,
						0l, 0l,
						obj_rtn_data
					))
					{
						g_free(path);
						status = -4;
						break;
					}
				}

				g_free(path);
				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(status);
#endif	/* !HAVE_LIBTAR */
}

/*
 *	Get a list of XArchive archive object statistics.
 */
static gint edv_archive_object_stat_list_sequential_xar(EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES)
{
#ifdef HAVE_LIBXAR
	time_t t;
	struct tm *tm_local;
	xar_t xar;
	xar_iter_t i1;
	xar_file_t xar_fp;
	gint status;
	glong gmt_offset_dst;
	gulong		i,
			nobjs;
	gchar *path;
	EDVArchiveObject *obj;

	/* Open the X Archive for reading */
	xar = xar_open(arch_path, READ);
	if(xar == NULL)
		return(-1);

	/* Create a new X Archive object iterator */
	i1 = xar_iter_new(xar);
	if(i1 == 0)
	{
		(void)xar_close(xar);
		return(-3);
	}

	/* Count the number of objects in the X Archive */
	for(xar_fp = xar_file_first(xar, i1), nobjs = 0l;
	    xar_fp != NULL;
	    xar_fp = xar_file_next(i1), nobjs++
	);

	/* Initialize time globals timezone & daylight */
	t = (time_t)edv_time();
	tm_local = localtime(&t);

	/* Calculate the GMT offset with DST */
	gmt_offset_dst = (glong)(timezone - ((daylight == 1) ? (60 * 60) : 0));

	/* Iterate through each object in the X Archive */
	status = 0;
	for(xar_fp = xar_file_first(xar, i1), i = 0l;
	    xar_fp != NULL;
	    xar_fp = xar_file_next(i1), i++
	)
	{
		/* Report progress */
		if(obj_rtn_cb(
			arch_path,
			NULL,
			i, nobjs,
			obj_rtn_data
		))
		{
			status = -4;
			break;
		}

		/* Get the path of this object within the X Archive */
		path = (gchar *)xar_get_path(xar_fp);

		/* Is this object's path not in the list of paths? */
		if(!edv_archive_object_stat_filter(
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			path
		))
		{
			g_free(path);
			continue;
		}

		/* Get the values for this object */
		obj = edv_archive_object_stat_xar_new_object(
			cfg_list,
			xar,
			xar_fp,
			i + 1l,			/* Index */
			path,
			gmt_offset_dst
		);

		g_free(path);

		/* Report this object */
		if(obj_rtn_cb(
			arch_path,
			obj,
			i + 1l, nobjs,
			obj_rtn_data
		))
		{
			status = -4;
			break;
		}
	}

	/* Delete the iterator and close the X Archive */
	xar_iter_free(i1);
	(void)xar_close(xar);

	return(status);
#else
	errno = EINVAL;
	return(-2);
#endif
}

/*
 *	Get a list of PKZip archive object statistics.
 */
static gint edv_archive_object_stat_list_sequential_zip(EDV_ARCHIVE_OBJECT_STAT_LIST_ANY_PROTOTYPES)
{
#ifdef HAVE_LIBZIP
	int zip_error;
	struct zip *archive;
	struct zip_stat zip_stat_buf;
	gint		i,
			status,
			len,
			nobjs;
	gchar *clean_path;
	const gchar *path;
	EDVObjectType type;
	EDVArchiveObject *obj;

	/* Open the PKZip archive */
	archive = zip_open(arch_path, 0, &zip_error);
	if(archive == NULL)
		return(-1);

	/* Get the number of objects in the PKZip archive */
	nobjs = zip_get_num_files(archive);
	if(nobjs <= 0)
	{
		(void)zip_close(archive);
		return(-1);
	}

	/* Iterate through each object in the PKZip archive */
	status = 0;
	for(i = 0; i < nobjs; i++)
	{
		/* Report progress */
		if(obj_rtn_cb(
			arch_path,
			NULL,
			(gulong)i, (gulong)nobjs,
			obj_rtn_data
		))
		{
			status = -4;
			break;
		}

		/* Get the stats for this object in the PKZip archive, if
		 * unable to obtain the stats then just continue on to the
		 * next object (because sometimes an index may refer to a
		 * deleted or non-existant object)
		 */
		if(zip_stat_index(archive, i, 0, &zip_stat_buf))
			continue;

		path = (const gchar *)zip_stat_buf.name;
		if(STRISEMPTY(path))
			continue;

		/* Get the "clean path" (the path without the tailing
		 * directory deliminator or tailing type character)
		 */
		clean_path = g_strdup(path);
		len = STRLEN(clean_path);
		switch((len > 1) ? clean_path[len - 1] : '\0')
		{
		    case G_DIR_SEPARATOR:	/* Directory */
			clean_path[len - 1] = '\0';
			type = EDV_OBJECT_TYPE_DIRECTORY;
			break;
		    case '@':			/* Link */
			clean_path[len - 1] = '\0';
			type = EDV_OBJECT_TYPE_LINK;
			break;
		    default:			/* File */
			type = EDV_OBJECT_TYPE_FILE;
			break;
		}

		/* Is this object's path not in the list of paths (use the
		 * "clean path" for proper matching)?
		 */
		if(!edv_archive_object_stat_filter(
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			clean_path
		))
		{
			g_free(clean_path);
			continue;
		}

		g_free(clean_path);

		/* Get this object from the PKZip archive */
		obj = edv_archive_object_stat_libzip_new_object(
			cfg_list,
			archive,
			&zip_stat_buf
		);

		/* Report this object */
		if(obj_rtn_cb(
			arch_path,
			obj,
			(gulong)(i + 1), (gulong)nobjs,
			obj_rtn_data
		))
		{
			status = -4;
			break;
		}
	}

	/* Close the PKZip archive */
	(void)zip_close(archive);

	/* Check the PKZip archive for errors */
	if(status == 0)
	{
		archive = zip_open(arch_path, ZIP_CHECKCONS, &zip_error);
		if(archive != NULL)
		{
			(void)zip_close(archive);
		}
		else
		{
			status = -1;
		}
	}

	return(status);
#else
	FILE            *cstdout,
			*cstderr;
	gint            pid,
			status;
	const gchar *prog_unzip = EDV_GET_S(EDV_CFG_PARM_PROG_UNZIP);

	/* Format the list PKZip archive command */
	gchar *cmd = g_strdup_printf(
		"\"%s\" -l -v \"%s\"",
		prog_unzip,
		arch_path
	);
	if(cmd == NULL)
		return(-3);

	/* Execute the list archive command */
	pid = edv_archive_object_stat_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	g_free(cmd);
	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(-1);
	}

	status = 0;

	/* Read the archive list from the standard output stream */
	if(cstdout != NULL)
	{
		const gchar *delim_header = "--------";
		gint line_counter;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					break;
				}

				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Begin reading each line describing each object in
		 * the PKZip archive
		 */
		line_counter = 0;
		while(!feof(fp))
		{
			/* Report progress */
			if(obj_rtn_cb(
				arch_path,
				NULL,
				0l, 0l,
				obj_rtn_data
			))
			{
				status = -4;
				break;
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
				line_counter++;

			/* Ending deliminator reached? If so do not parse
			 * this line or any subsequent lines
			 */
			if(buf != NULL)
			{
				if(*buf == '-')
					break;
			}

			/* Got a line */
			if(line_counter >= 1)
			{
				gint i;
				gchar	*s = buf,
					*path;
				while(ISBLANK(*s))
					s++;

				/* Seek path to the path value,
				 * skipping 7 values
				 */
				path = s;
				for(i = 0; i < 7; i++)
				{
					while(!ISBLANK(*path) && (*path != '\0'))
						path++;
					while(ISBLANK(*path))
						path++;
				}

				/* Copy the path value and cap the first
				 * newline character
				 */
				path = g_strdup(path);
				edv_archive_object_stat_cap_newline(path);

				/* Check if the path matches any of the
				 * paths in the paths list or filter
				 */
				if(edv_archive_object_stat_filter(
					paths_list,
#if defined(HAVE_REGEX)
					regex_filter,
#else
					filter,
#endif
					path
				))
				{
					EDVArchiveObject *obj = edv_archive_object_new();
					edv_archive_object_stat_parse_line_zip(
						obj,
						s
					);

					/* Report this object */
					if(obj_rtn_cb(
						arch_path,
						obj,
						0l, 0l,
						obj_rtn_data
					))
					{
						g_free(path);
						status = -4;
						break;
					}
				}

				g_free(path);
				g_free(buf);
				buf = NULL;
				line_counter = 0;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(status);
#endif
}

/*
 *	Gets the statistics of each object in the archive sequentially.
 *
 *	The paths_list specifies a list of (const gchar *) paths that
 *	represent objects within the archive to obtain stats for. If
 *	paths_list is NULL then the filter will be used to determine
 *	which objects in the archive to obtain the stats for.
 *
 *	The filter specifies the object name filter to obtain the
 *	object stats for. Only the object's name (not its path within
 *	the archive) will be compared with the filter. The filter is
 *	only used if paths_list is NULL.
 *
 *	The password specifies the password to use in order to obtain
 *	stats from encrypted archives.
 *
 *	The obj_rtn_cb and obj_rtn_data specifies the object return
 *	callback function which is called each time to pass a object
 *	obtained from the archive. The EDVArchiveObject *obj
 *	is passed to obj_rtn_cb() and is not referenced again by
 *	this function so obj_rtn_cb() is responsible for deleting the
 *	object. When the EDVArchiveObject *obj is passed as
 *	NULL then the purpose of obj_rtn_cb() is to only report
 *	progress and check for abort. If obj_rtn_cb() returns any
 *	non-zero function then the operation will be aborted, otherwise
 *	the operation continues.
 *
 *	The returns 0 on success or non-zero on error.
 */
gint edv_archive_object_stat_list_sequential(
	CfgList *cfg_list,
	const gchar *arch_path,
	GList *paths_list,
	const gchar *filter,
	const gchar *password,
	gint (*obj_rtn_cb)(
		const gchar *arch_path,
		EDVArchiveObject *obj,
		const gulong i, const gulong m,
		gpointer data
	),
	gpointer obj_rtn_data
)
{
#ifdef HAVE_REGEX
	regex_t *regex_filter;
#endif
	gint		status,
			error_code;
	const gchar *arch_name;
	EDVVFSObject *arch_obj;

	if(STRISEMPTY(arch_path) || (obj_rtn_cb == NULL))
	{
		errno = EINVAL;
		return(-2);
	}

	arch_name = g_basename(arch_path);

	/* Get the VFS statistics of the archive object */
	arch_obj = edv_vfs_object_stat(arch_path);
	if(arch_obj == NULL)
		return(-1);

	if(EDV_VFS_OBJECT_IS_DIRECTORY(arch_obj))
	{
		edv_vfs_object_delete(arch_obj);
		errno = EISDIR;
		return(-2);
	}

#if defined(HAVE_REGEX)
	/* Compile the regex search criteria */
	if(STRISEMPTY(filter) ?
		FALSE : strcmp((const char *)filter, "*")
	)
	{
		regex_filter = (regex_t *)g_malloc(sizeof(regex_t));
		if(regcomp(
			regex_filter,
			filter,
#ifdef REG_EXTENDED
			REG_EXTENDED |		/* Use POSIX extended regex */
#endif
			REG_NOSUB		/* Do not report subpattern matches */
		))
		{
			g_free(regex_filter);
			regex_filter = NULL;
		}
	}
	else
	{
		regex_filter = NULL;
	}
#else
	if(STRISEMPTY(filter))
	{
		filter = NULL;
	}
	else
	{
		if(!strcmp((const char *)filter, "*"))
			filter = NULL;
	}
#endif

	/* Get the statistics of the objects in the archive based on
	 * the archive's format
	 */

	/* ARJ Archive */
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		status = edv_archive_object_stat_list_sequential_arj(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data
		);
	}
	/* LHA Archive */
	else if(edv_name_has_extension(arch_name, ".lha"))
	{
		status = edv_archive_object_stat_list_sequential_lha(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		status = edv_archive_object_stat_list_sequential_rar(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data
		);
	}
	/* RedHat Package Manager Package */
	else if(edv_name_has_extension(arch_name, ".rpm"))
	{
		status = edv_archive_object_stat_list_sequential_rpm(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data
		);
	}
	/* Tape Archive (Compressed) */
	else if(edv_name_has_extension(arch_name, ".tar.Z"))
	{
		status = edv_archive_object_stat_list_sequential_tar(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data,
			TRUE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (GZip) */
	else if(edv_name_has_extension(arch_name, ".tgz .tar.gz"))
	{
		status = edv_archive_object_stat_list_sequential_tar(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data,
			FALSE,			/* Not compress compressed */
			TRUE,			/* Is gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (BZip2) */
	else if(edv_name_has_extension(arch_name, ".tar.bz2"))
	{
		status = edv_archive_object_stat_list_sequential_tar(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			TRUE			/* Is bzip2 compressed */
		);
	}
	/* Tape Archive */
	else if(edv_name_has_extension(arch_name, ".tar"))
	{
		status = edv_archive_object_stat_list_sequential_tar(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* X Archive */
	else if(edv_name_has_extension(arch_name, ".xar"))
	{
		status = edv_archive_object_stat_list_sequential_xar(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		status = edv_archive_object_stat_list_sequential_zip(
			cfg_list,
			arch_path,
			arch_obj->size,
			paths_list,
#if defined(HAVE_REGEX)
			regex_filter,
#else
			filter,
#endif
			obj_rtn_cb, obj_rtn_data
		);
	}
	else
	{
		status = -2;
		errno = EINVAL;
	}

	error_code = (gint)errno;

#ifdef HAVE_REGEX
	/* Delete the regex search criteria */
	if(regex_filter != NULL)
	{
		regfree(regex_filter);
		g_free(regex_filter);
	}
#endif

	edv_vfs_object_delete(arch_obj);

	errno = (int)error_code;

	return(status);
}

/*
 *	Archive stat list sequential progress callback.
 *
 *	If obj is not NULL then it specifies a new EDVArchiveObject
 *	that we should take and append to the list. The calling
 *	function does not refer to it again after this call.
 */
static gint edv_archive_object_stat_list_progress_cb(
	const gchar *arch_path,
	EDVArchiveObject *obj,
	const gulong i, const gulong m,
	gpointer data
)
{
	EDVArchStatListObjRtnData *d = EDV_ARCH_STAT_LIST_OBJ_RTN_DATA(data);

	if(obj != NULL)
		d->objs_list = g_list_append(d->objs_list, obj);

	if(d->progress_cb != NULL)
		return(d->progress_cb(
			arch_path,
			i, m,
			d->progress_data
		));
	else
		return(0);
}

/*
 *	Returns a list of archive object stats for the objects found in
 *	the archive arch_path.
 *
 *	The paths_list specifies a list of (const gchar *) paths that
 *	represent objects within the archive to obtain stats for. If
 *	paths_list is NULL then the filter will be used to determine
 *	which objects in the archive to obtain the stats for.
 *
 *	The filter specifies the object name filter to obtain the
 *	object stats for. Only the object's name (not its path within
 *	the archive) will be compared with the filter. The filter is
 *	only used if paths_list is NULL.
 *
 *	The password specifies the password to use in order to obtain
 *	stats from encrypted archives.
 *
 *	The returned list of archive object stats must be deleted by the
 *	calling function.
 *
 *	Can return NULL if the archive is not supported, error, or no
 *	objects were found in the archive.
 */
GList *edv_archive_object_stat_list(
	CfgList *cfg_list,
	const gchar *arch_path,
	GList *paths_list,
	const gchar *filter,
	const gchar *password,
	gint (*progress_cb)(
		const gchar *,
		const gulong, const gulong,
		gpointer
	),
	gpointer progress_data
)
{
	gint error_code;
	GList *objs_list;
	EDVArchStatListObjRtnData *d = EDV_ARCH_STAT_LIST_OBJ_RTN_DATA(g_malloc0(
		sizeof(EDVArchStatListObjRtnData)
	));
	if(d == NULL)
		return(NULL);

	d->objs_list = NULL;
	d->progress_cb = progress_cb;
	d->progress_data = progress_data;

	edv_archive_object_stat_list_sequential(
		cfg_list,
		arch_path,
		paths_list,
		filter,
		password,
		edv_archive_object_stat_list_progress_cb, d
	);
	error_code = (gint)errno;

	objs_list = d->objs_list;
	g_free(d);

	errno = (int)error_code;

	return(objs_list);
}
