#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <glib.h>

#include "cfg.h"
#include "cfg_fio.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_property.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_archive_list.h"
#include "edv_archive_obj_stat.h"

/* Instruct edv_cfg_list.h to #define EDV_CONFIGURATION_LIST */
#define NEED_EDV_CFG_LIST_SOURCE
#include "edv_cfg_list.h"

#include "config.h"


static void edv_archive_list_signal_cb(int s);
static gint edv_archive_list_next_object_cb(
        const gchar *arch_path,
        EDVArchiveObject *obj,
        const gulong i, const gulong n,
        gpointer data
);

static void edv_archive_list_send_object_data(
	const gulong i, const gulong n,
	EDVArchiveObject *obj
);

gint edv_archive_list(const gint argc, const gchar **argv);


static gint edv_archive_list_stop_count;


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	System signal callback.
 */
static void edv_archive_list_signal_cb(int s)
{
	switch(s)
	{
	    case SIGINT:
	    case SIGTERM:
		edv_archive_list_stop_count++;
		break;
	}
}

/*
 *	Next object return callback.
 */
static gint edv_archive_list_next_object_cb(
        const gchar *arch_path,
        EDVArchiveObject *obj,
        const gulong i, const gulong n,
        gpointer data
)
{
	/* Interrupted? */
	if(edv_archive_list_stop_count > 0)
	{
		edv_archive_object_delete(obj);
		return(-4);
	}

	/* If an object was returned then the calling function expects
	 * us to take it since it does not want it back
	 *
	 * Since we are sending object data instead of taking it
	 * we will send the data first and then delete the object
	 */
	if(obj != NULL)
	{
		edv_archive_list_send_object_data(
			i, n,
			obj
		);
		edv_archive_object_delete(obj);
	}

	/* Keep listing */
	return(0);
}


/*
 *	Send the object's data to standard output where the main
 *	Endeavour Mark II process will read it from
 *
 *	Parsed by edv_archiver_subprocess_list_parse_list_value()
 *	in module archiver_subprocess.c.
 */
static void edv_archive_list_send_object_data(
	const gulong i, const gulong n,
	EDVArchiveObject *obj
)
{
	g_print(
"%ld\t%ld\t\
%i\t%ld\t\
%s\t%s\t\
%ld\t%ld\t\
%s\t\
%i\t\
%ld\t%ld\t%ld\t\
%s\t%s\t\
%i\t\
%s\t%f\t%s\t%s",
		i,
		n,

		(gint)obj->type,
		obj->index,

		(obj->path != NULL) ? obj->path : "",
		(obj->name != NULL) ? obj->name : "",

		obj->size,
		obj->storage_size,

		(obj->link_target != NULL) ? obj->link_target : "",

		(gint)obj->permissions,

		obj->access_time,
		obj->modify_time,
		obj->change_time,

		(obj->owner_name != NULL) ? obj->owner_name : "",
		(obj->group_name != NULL) ? obj->group_name : "",

		obj->device_type,

		(obj->encryption_name != NULL) ? obj->encryption_name : "",
		obj->compression_ratio,
		(obj->storage_method != NULL) ? obj->storage_method : "",
		(obj->crc != NULL) ? obj->crc : ""
	);

	/* Meta data */
	if(obj->meta_data_list != NULL)
	{
		GList *glist;
		EDVProperty *prop;

		for(glist = obj->meta_data_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			prop = EDV_PROPERTY(glist->data);
			if(prop == NULL)
				continue;

			if(STRISEMPTY(prop->name) || STRISEMPTY(prop->value))
				continue;

			g_print(
				"\t%s=%s",
				prop->name,
				prop->value
			);
		}
	}

	/* End of object */
	g_print("\n");
}


/*
 *	Runs Endeavour Mark II in noninteractive mode to print a list
 *	of objects in an archive to standard output for the main
 *	Endeavour Mark II process to read from.
 */
gint edv_archive_list(const gint argc, const gchar **argv)
{
	gint		i,
			status;
	const gchar *arg;
	gchar		*cwd = edv_getcwd(),
			*cfg_path = NULL,
			*arch_path = NULL,
			*filter = NULL,
			*password = NULL;

	/* Create the default configuration list */
	const CfgList src_cfg_list[] = EDV_CONFIGURATION_LIST;
	CfgList *cfg_list = CFGItemListCopyList(src_cfg_list);

	/* Reset */
	edv_archive_list_stop_count = 0;

        /* Set up the UNIX signal callbacks */
#ifdef SIGHUP
        signal(SIGHUP, edv_archive_list_signal_cb);
#endif
#ifdef SIGINT
        signal(SIGINT, edv_archive_list_signal_cb);
#endif
#ifdef SIGTERM
        signal(SIGTERM, edv_archive_list_signal_cb);
#endif

#define CLEANUP_RETURN(_v_)	{		\
 const gint error_code = (gint)errno;		\
						\
 g_free(cwd);					\
						\
 g_free(cfg_path);				\
 CFGItemListDeleteList(cfg_list);		\
						\
 g_free(arch_path);				\
 g_free(filter);				\
 g_free(password);				\
						\
 errno = (int)error_code;			\
						\
 return(_v_);					\
}

        /* Parse arguments */
        for(i = 1; i < argc; i++)
        {
                arg = argv[i];
                if(arg == NULL)
                        continue;

                /* Alternate configuration file */
                if(!g_strcasecmp(arg, "--config") ||
                   !g_strcasecmp(arg, "-config") ||
                   !g_strcasecmp(arg, "--rcfile") ||
                   !g_strcasecmp(arg, "-rcfile") ||
                   !g_strcasecmp(arg, "--f") ||
                   !g_strcasecmp(arg, "-f")
                )
                {
                        i++;
                        if(i < argc)
                        {
                                arg = argv[i];
				g_free(cfg_path);
                                cfg_path = g_strdup(arg);
                        }
                        else
                        {
                                g_printerr(
                                        "%s: Requires argument.\n",
					arg
                                );
				CLEANUP_RETURN(-2);
                        }
                }
		/* Archive path */
		else if((*arg != '-') && (*arg != '+'))
		{
			if(arch_path == NULL)
				arch_path = edv_path_evaluate(
					cwd,
					arg
				);
			else if(filter == NULL)
				filter = g_strdup(arg);
			else if(password == NULL)
				password = g_strdup(arg);
		}
	}

	/* Use the default configuration file path? */
	if(cfg_path == NULL)
	{
                const gchar *home = g_getenv(ENV_VAR_NAME_HOME);
                cfg_path = g_strconcat(
                        (home != NULL) ? home : "/",
                        G_DIR_SEPARATOR_S,
                        EDV_NAME_DEF_USER_DATA_DIR,
                        G_DIR_SEPARATOR_S,
                        EDV_NAME_DEF_CONFIG_FILE,
                        NULL
                );
		if(cfg_path == NULL)
		{
			const gint error_code = (gint)errno;
			g_printerr(
				"%s: %s.\n",
				argv[0],
				g_strerror(error_code)
			);
			CLEANUP_RETURN(-3);
		}
	}

	/* No archive specified? */
	if(arch_path == NULL)
	{
		g_printerr(
			"%s: No archive specified.\n",
			argv[0]
		);
		CLEANUP_RETURN(-2);
	}


	/* Open the configuration list from the configuration file,
	 * setting the value of each configuration parameter read
	 * from the configuration file
         */
	if(CFGFileOpen(
		cfg_path,
		cfg_list
	))
        {
		/* An error occured while opening the configuration file */
		const gint error_code = (gint)errno;
                g_printerr(
			"%s: %s\n",
			cfg_path,
                        g_strerror(error_code)
                );
		CLEANUP_RETURN(-1);
	}

	/* List the archive */
        status = edv_archive_object_stat_list_sequential(
                cfg_list,
                arch_path,
                NULL,				/* Get all the objects */
                filter,
                password,
                edv_archive_list_next_object_cb, NULL
        );

	/* Print any error messages */
	if((status != 0) && (status != -4))
	{
		const gint error_code = (gint)errno;
		g_printerr(
			"%s.\n",
			g_strerror(error_code)
		);
	}

	CLEANUP_RETURN(status);
}
