#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_archive_fix.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_archive_fix_get_error(EDVCore *core);
static void edv_archive_fix_set_error(
	EDVCore *core,
	const gchar *msg
);

/* Execute */
static gint edv_archive_fix_execute(
        CfgList *cfg_list,
        const gchar *cmd,
        FILE **cstdin_rtn,
        FILE **cstdout_rtn,
        FILE **cstderr_rtn
);

/* Progress Dialog */
static gint edv_archive_fix_map_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *arch_path,
        const gchar *path,
        const gfloat progress,
        GtkWidget *toplevel,
        const gboolean force_remap
);

/* Delete Object From Archive */
static gint edv_archive_fix_arj(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_fix_lha(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_fix_rar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_fix_tar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);
static gint edv_archive_fix_zip(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
gint edv_archive_fix(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	gboolean const show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Gets the last error message.
 */
const gchar *edv_archive_fix_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}


/*
 *	Coppies the error message to the core's last_error_buf
 *	and sets the core's last_error_ptr to point to it.
 */
static void edv_archive_fix_set_error(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Executes the command using the shell specified by the
 *	configuration and opens and returns the requested streams.
 */
static gint edv_archive_fix_execute(
        CfgList *cfg_list,
        const gchar *cmd,
        FILE **cstdin_rtn,
        FILE **cstdout_rtn,
        FILE **cstderr_rtn
)
{
        gchar           *shell_prog;
        const gchar     *shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
                        *shell_args = edv_strarg(
                shell_cmd,
                &shell_prog,
                TRUE,				/* Parse escapes */
                TRUE				/* Parse quotes */
        );

        /* Execute the list archive command */
        gint pid = edv_system_shell_streams(
                cmd,
                shell_prog,
                shell_args,
                cstdin_rtn,
                cstdout_rtn,
                cstderr_rtn
        );

        g_free(shell_prog);

        return(pid);
}


/*
 *      Updates/maps the progress dialog.
 */
static gint edv_archive_fix_map_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *arch_path,
        const gchar *path,
        const gfloat progress,
        GtkWidget *toplevel,
        const gboolean force_remap
) 
{
	if((arch_path != NULL) && (path != NULL))
	{
		gchar	*arch_path_shortened = edv_path_shorten(
			arch_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*path_shortened = edv_path_shorten(
			path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
"Fixing:\n\
\n\
    %s\n\
\n\
In:\n\
\n\
    %s\n"
			,
			path_shortened,
			arch_path_shortened
		);
		g_free(arch_path_shortened);
		g_free(path_shortened);
		edv_progress_dialog_map_archive_fix_animated(
			cfg_list,
			msg,
			progress,
			toplevel,
			force_remap
		);
		g_free(msg);
	}
	else
	{
                if(progress >= 0.0f)
                        ProgressDialogUpdate(
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                progress,
                                EDV_PROGRESS_BAR_NTICKS,
                                TRUE
                        );
                else
                        ProgressDialogUpdateUnknown(
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                TRUE
                        );
        }

	return(ProgressDialogStopCount());
}


/*
 *	Fixes the ARJ archive.
 */
static gint edv_archive_fix_arj(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	core->last_error_ptr = "There is no support for fixing ARJ archives.";
	return(-2);
}

/*
 *	Fixes the LHA archive.
 */
static gint edv_archive_fix_lha(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	core->last_error_ptr = "There is no support for fixing LHA archives.";
	return(-1);
}

/*
 *	Fixes the RAR archive.
 */
static gint edv_archive_fix_rar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid;
	gchar		*parent_path,
			*cwd = edv_getcwd(),
			*cmd = NULL,
			*rebuilt_arch_path;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 if(cwd != NULL) {			\
  (void)edv_setcwd(cwd);		\
  g_free(cwd);				\
 }					\
					\
 return(_v_);				\
}

	/* RAR needs the current working directory to be in the same
	 * directory as the archive
	 */
	parent_path = g_dirname(arch_path);
	if(parent_path != NULL)
	{
		(void)edv_setcwd(parent_path);
		g_free(parent_path);
	}

	/* Format the fix archive command */
	cmd = g_strdup_printf(
		"\"%s\" r -y \"%s\"",
		prog_rar,
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the fix command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the fix archive command */
	pid = edv_archive_fix_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the fix archive command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gchar	*cstdout_buf = NULL,
			*cstderr_buf = NULL;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_fix_map_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				-1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Update the progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_fix_map_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					-1.0f,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Check if the process has finished */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Read the next line from standard error */
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				const gchar *s = cstderr_buf;
				while(ISBLANK(*s))
					s++;

				/* Record the first error message */
				if(!STRISEMPTY(s) &&
				   (core->last_error_ptr == NULL)
				) 
				{
					edv_archive_fix_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
						status = -1;
				}

				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}

			/* Read the next line from standard output */
			if(edv_stream_read_strptrbrk(
				cstdout,
				&cstdout_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				/* Display the message */
				const gchar *s = cstdout_buf;
				while(ISBLANK(*s))
					s++;
				if(show_progress && !STRISEMPTY(s))
				{
					(void)edv_archive_fix_map_update_progress_dialog(
						cfg_list,
						arch_path,
						s,
						-1.0f,
						toplevel,
						FALSE
					);
				}

				g_free(cstdout_buf);
				cstdout_buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(cstdout) || !feof(cstderr));

		g_free(cstdout_buf);
		g_free(cstderr_buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* RAR creates a new fixed archive with a "rebuilt." prefixed
	 * to the specified archive's name
	 *
	 * We will rename the specified archive to a name with a
	 * ".bak" postfix and rename the new fixed archive to the
	 * original name
	 */
	parent_path = g_dirname(arch_path),
	rebuilt_arch_path = g_strconcat(
		parent_path,
		G_DIR_SEPARATOR_S,
		"rebuilt.",
		g_basename(arch_path),
		NULL
	);
	if(edv_path_exists(rebuilt_arch_path))
	{
		/* Remove the previous ".bak" archive and rename the
		 * specified archive to the ".bak" archive
		 */
		gchar *bak_arch_path = g_strconcat(
			arch_path,
			".bak",
			NULL
		);
		(void)edv_unlink(bak_arch_path);
		if(edv_rename(
			arch_path,
			bak_arch_path
		) == 0)
		{
			/* Rename the new fixed archive to the
			 * specified archive
			 */
			if(edv_rename(
				rebuilt_arch_path,
				arch_path
			) == 0)
			{
				/* Restore the permissions */
				EDVVFSObject *obj = edv_vfs_object_stat(bak_arch_path);
				if(obj != NULL)
				{
					(void)edv_permissions_set(
						arch_path,
						obj->permissions
					);
					(void)edv_chown(
						arch_path,
						obj->owner_id,
						obj->group_id
					);
					edv_vfs_object_delete(obj);
				}
			}
		}
		g_free(bak_arch_path);
	}
	g_free(parent_path);
	g_free(rebuilt_arch_path);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Fixes the Tape archive.
 */
static gint edv_archive_fix_tar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
	core->last_error_ptr = "There is no support for fixing TAR archives.";
	return(-2);
}

/*
 *	Fixes the PKZip archive.
 */
static gint edv_archive_fix_zip(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_zip = EDV_GET_S(EDV_CFG_PARM_PROG_ZIP);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the fix archive command */
	cmd = g_strdup_printf(
		"\"%s\" -FF -v \"%s\"",
		prog_zip,
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the fix command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the fix archive command */
	pid = edv_archive_fix_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the fix archive command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gchar   *cstdout_buf = NULL,
			*cstderr_buf = NULL;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_fix_map_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				-1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Update the progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_fix_map_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					-1.0f,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Check if the process has finished */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Read the next line from standard error */
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n",
				FALSE,          /* Exclude end character */
				FALSE           /* Nonblocking */
			))
			{
				const gchar *s = cstderr_buf;
				while(ISBLANK(*s))
					s++;

				/* Record the first error message */
				if(!STRISEMPTY(s) &&
				   (core->last_error_ptr == NULL)
				) 
				{
					edv_archive_fix_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
						status = -1;
				}

				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}

			/* Read the next line from standard output */
			if(edv_stream_read_strptrbrk(
				cstdout,
				&cstdout_buf,
				"\n",
				FALSE,          /* Exclude end character */
				FALSE           /* Nonblocking */
			))
			{
				/* Display the message */
				const gchar *s = cstdout_buf;
				while(ISBLANK(*s))
					s++;
				if(show_progress && !STRISEMPTY(s))
				{
					(void)edv_archive_fix_map_update_progress_dialog(
						cfg_list,
						arch_path,
						s,
						-1.0f,
						toplevel,
						FALSE
					);
				}

				g_free(cstdout_buf);
				cstdout_buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(cstdout) || !feof(cstderr));

		g_free(cstdout_buf);
		g_free(cstderr_buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Fixes the archive.
 *
 *	The arch_path specifies the archive.
 */
gint edv_archive_fix(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = edv_time();
	gint status;
	const gchar *arch_name;

#define CLEANUP_RETURN(_v_)	{		\
						\
 return(_v_);					\
}

	if(ProgressDialogIsQuery())
	{
		edv_archive_fix_set_error(
			core,
"An operation is already in progress, please try again later."
		);
		return(-6);
	}

	/* Clear the last error message */
	edv_archive_fix_set_error(core, NULL);

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   (yes_to_all == NULL)
	)
	{
		edv_archive_fix_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Begin deleting the source object from the archive object
	 * arch_path. The deleting method will be determined by taking
	 * the extension of the archive object's name
	 */

	/* ARJ Archive */
	arch_name = g_basename(arch_path);
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		status = edv_archive_fix_arj(
			core, arch_path,
			toplevel, show_progress, interactive, yes_to_all
		);
	}
	/* LHA Archive */
	else if(edv_name_has_extension(arch_name, ".lha"))
	{
		status = edv_archive_fix_lha(
			core, arch_path,
			toplevel, show_progress, interactive, yes_to_all
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		status = edv_archive_fix_rar(
			core, arch_path,
			toplevel, show_progress, interactive, yes_to_all
		);
	}
	/* Tape Archive (Compressed) */
	else if(edv_name_has_extension(arch_name, ".tar.Z"))
	{
		status = edv_archive_fix_tar(
			core, arch_path,
			toplevel, show_progress, interactive, yes_to_all,
			TRUE,		/* Is compress compressed */
			FALSE,		/* Not gzip compressed */
			FALSE		/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (GZip) */
	else if(edv_name_has_extension(arch_name, ".tgz .tar.gz"))
	{
		status = edv_archive_fix_tar(
			core, arch_path,
			toplevel, show_progress, interactive, yes_to_all,
			FALSE,		/* Not compress compressed */
			TRUE,		/* Is gzip compressed */
			FALSE		/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (BZip2) */
	else if(edv_name_has_extension(arch_name, ".tar.bz2"))
	{
		status = edv_archive_fix_tar(
			core, arch_path,
			toplevel, show_progress, interactive, yes_to_all,
			FALSE,		/* Not compress compressed */
			FALSE,		/* Not gzip compressed */
			TRUE		/* Is bzip2 compressed */
		);
	}
	/* Tape Archive */
	else if(edv_name_has_extension(arch_name, ".tar"))
	{
		status = edv_archive_fix_tar(
			core, arch_path,
			toplevel, show_progress, interactive, yes_to_all,
			FALSE,		/* Not compress compressed */
			FALSE,		/* Not gzip compressed */
			FALSE		/* Not bzip2 compressed */
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		status = edv_archive_fix_zip(
			core, arch_path, password,
			toplevel, show_progress, interactive, yes_to_all
		);
	}
	else
	{
		core->last_error_ptr = "Unsupported archive format.";
		status = -2;
	}

	/* Record history */
	if(status != 0)
	{
		edv_append_history(
			core,
			EDV_HISTORY_ARCHIVE_FIX,
			time_start,
			edv_time(),
			status,
			arch_path,			/* Source */
			NULL,				/* No target */
			core->last_error_ptr	/* Comment */
		);
	}
	else
	{
		edv_append_history(
			core,
			EDV_HISTORY_ARCHIVE_FIX,
			time_start,
			edv_time(),
			status,
			arch_path,			/* Source */
			NULL,				/* No target */
			core->last_error_ptr	/* Comment */
		);
	}

	/* Need to flush disk changes since the archive may have been
	 * modified on another process and the changes have not reached
	 * our process yet
	 */
	(void)edv_sync();

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
