#ifdef HAVE_LIBZIP
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <errno.h>
# include <zip.h>
# include <gtk/gtk.h>
# include "cdialog.h"
# include "progressdialog.h"
# include "edv_types.h"
# include "libendeavour2-base/edv_utils.h"
# include "libendeavour2-base/edv_path.h"
# include "libendeavour2-base/edv_directory.h"
# include "libendeavour2-base/edv_link.h"
# include "libendeavour2-base/edv_process.h"
# include "libendeavour2-base/edv_vfs_obj.h"
# include "libendeavour2-base/edv_vfs_obj_stat.h"
# include "libendeavour2-base/edv_archive_obj.h"
# include "edv_utils_gtk.h"
# include "edv_progress.h"
# include "edv_archive_extract.h"
# include "edv_archive_extract_zip.h"
# include "endeavour2.h"
# include "config.h"
#else
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <errno.h>
# include <signal.h>
# include <gtk/gtk.h>
# include "cdialog.h"
# include "progressdialog.h"
# include "edv_types.h"
# include "libendeavour2-base/edv_utils.h"
# include "libendeavour2-base/edv_path.h"
# include "libendeavour2-base/edv_stream.h"
# include "libendeavour2-base/edv_process.h"
# include "libendeavour2-base/edv_archive_obj.h"
# include "edv_utils_gtk.h"
# include "edv_progress.h"
# include "edv_archive_extract.h"
# include "edv_archive_extract_zip.h"
# include "endeavour2.h"
# include "edv_cfg_list.h"
# include "config.h"
#endif


gint edv_archive_extract_zip(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list,
	const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define STRCASEPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
			 !g_strncasecmp((s),(p),strlen(p)) : FALSE)
#define STRPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
			 !strncmp((s),(p),strlen(p)) : FALSE)

#define FCLOSE(_fp_)	(((_fp_) != NULL) ? (gint)fclose(_fp_) : -1)
#define INTERRUPT(i)	(((i) > 0) ? (gint)kill((int)(i), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Extract object from a PKZip archive.
 *
 *	Inputs assumed valid.
 */
gint edv_archive_extract_zip(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list,
	const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
#ifdef HAVE_LIBZIP
	struct zip *archive;
	gint		status,
					zip_error_code;
	gulong		cur_size,
					total_size;
	gchar		*src_path,
					*tar_path,
					*parent;
	GList *glist;
	CfgList *cfg_list = core->cfg_list;
	EDVObjectType type;
	EDVArchiveObject *obj;

	/* Open the PKZip archive for reading */
	archive = zip_open(arch_path, 0, &zip_error_code);
	if(archive == NULL)
	{
		const gint sys_error_code = (gint)errno;
		gchar *msg, err_msg[1024];
		zip_error_to_str(
			err_msg, sizeof(err_msg),
			zip_error_code, sys_error_code
		);
		msg = g_strdup_printf(
"Unable to open the PKZip Archive for reading:\n\
\n\
    %s\n\
\n\
%s.",
			arch_path, err_msg
		);
		edv_archive_extract_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Iterate through the list of objects to extract in order
	 * to calculate the total uncompressed size
	 */
	total_size = 0l;
	for(glist = objs_list; glist != NULL; glist = g_list_next(glist))
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		/* Count only the sizes of files and links */
		type = obj->type;
		if((type == EDV_OBJECT_TYPE_FILE) ||
		   (type == EDV_OBJECT_TYPE_LINK)
		)
			total_size += obj->size;
	}

	/* Iterate through the list of objects to extract and
	 * extract each one
	 */
	cur_size = 0l;
	status = 0;
	for(glist = objs_list; glist != NULL; glist = g_list_next(glist))
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		if(STRISEMPTY(obj->path))
			continue;

		src_path = g_strdup(obj->path);
		if(src_path == NULL)
			continue;

		/* Format the path of the object to extract */
		if(preserve_directories)
			tar_path = edv_paths_join(
				dest_path,
				src_path
			);
		else
			tar_path = edv_paths_join(
				dest_path,
				g_basename(src_path)
			);
		if(STRISEMPTY(tar_path))
		{
			g_free(src_path);
			g_free(tar_path);
			core->last_error_ptr =
"Unable to generate the extracted object's path.";
			status = -1;
			break;
		}

		edv_path_simplify(tar_path);

		/* Update the progress dialog message to display the
		 * current object being extracted
		 */
		if(show_progress)
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				src_path,
				tar_path,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : -1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(src_path);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

		/* Need to create the parent directory(ies)? */
		parent = g_dirname(tar_path);
		if(parent != NULL)
		{
			EDVVFSObject *parent_obj = edv_vfs_object_lstat(parent);
			const gint error_code = (gint)errno;
			if((parent_obj == NULL) && (error_code == ENOENT))
			{
				/* Create each parent directory(ies) and add
				 * them to the list of extracted objects
				 */
				if(edv_directory_create(
					parent,
					TRUE,	/* Create parents */
					new_paths_list_rtn
				))
				{
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"Unable to create the parent directories to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
						tar_path,
						arch_path,
						g_strerror(error_code)
					);
					edv_archive_extract_set_error(core, msg);
					g_free(msg);
					g_free(parent);
					g_free(src_path);
					g_free(tar_path);
					status = -1;
					break;
				}
			}
			edv_vfs_object_delete(parent_obj);
			g_free(parent);
		}

		/* Report the progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(src_path);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

#define QUERY_CONTINUE_EXTRACT	{			\
 /* Count this object even after a failed extraction */	\
 if(new_paths_list_rtn != NULL)				\
  *new_paths_list_rtn = g_list_append(			\
   *new_paths_list_rtn, STRDUP(tar_path)		\
  );							\
							\
 /* Need to query the user? */				\
 if(interactive && !(*yes_to_all)) {			\
  gint response;					\
  gchar *msg = g_strdup_printf(				\
"An error occured while extracting the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue extracting subsequent objects?",		\
   src_path,						\
   tar_path						\
  );							\
  edv_play_sound_error(core);				\
  CDialogSetTransientFor(toplevel);			\
  response = CDialogGetResponse(			\
   "Extract Failed",					\
   msg,							\
   NULL,						\
   CDIALOG_ICON_ERROR,					\
   CDIALOG_BTNFLAG_YES |				\
    CDIALOG_BTNFLAG_YES_TO_ALL |			\
   CDIALOG_BTNFLAG_NO,					\
   CDIALOG_BTNFLAG_YES					\
  );							\
  g_free(msg);						\
  CDialogSetTransientFor(NULL);				\
							\
  /* Stop extracting? */				\
  if((response == CDIALOG_RESPONSE_NO) ||		\
     (response == CDIALOG_RESPONSE_CANCEL) ||		\
     (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
  )							\
  {							\
   g_free(src_path);					\
   g_free(tar_path);					\
   break;						\
  }							\
							\
  /* Yes to all? */					\
  if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
   *yes_to_all = TRUE;					\
 }							\
							\
 g_free(src_path);					\
 g_free(tar_path);					\
 continue;						\
}

		/* Directory? */
		if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
		{
			/* Object already exist and is not a directory? */
			EDVVFSObject *obj_lvalues = edv_vfs_object_lstat(tar_path);
			if(obj_lvalues != NULL)
			{
				if(!EDV_VFS_OBJECT_IS_DIRECTORY(obj_lvalues))
				{
					/* Remove the existing object */
					if(edv_archive_extract_remove(tar_path))
					{
						const gint error_code = (gint)errno;
						gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
							tar_path,
							src_path,
							arch_path,
							g_strerror(error_code)
						);
						edv_archive_extract_set_error(core, msg);
						g_free(msg);
						status = -1;
						QUERY_CONTINUE_EXTRACT
					}
				}
				edv_vfs_object_delete(obj_lvalues);
			}
			/* Extract the directory by creating it as needed */
			if(edv_directory_create(
				tar_path,
				FALSE,			/* Do not create parents */
				NULL			/* No new paths list return */
			))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path,
					arch_path,
					g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Append the target path to the list of extracted objects */
			if(new_paths_list_rtn != NULL)
				*new_paths_list_rtn = g_list_append(
					*new_paths_list_rtn,
					g_strdup(tar_path)
				);
		}
		/* Link? */
		else if(obj->type == EDV_OBJECT_TYPE_LINK)
		{
			struct zip_file *src_fp;
			gint bytes_read;
			const gulong value_len = MAX(obj->size, 0l);
			gchar	*value,
				*src_path_pc;

			/* Open the source file in the PKZip archive */
			src_path_pc = g_strconcat(src_path, "@", NULL);
			src_fp = zip_fopen(
				archive,
				src_path_pc,
				0
			);
			g_free(src_path_pc);
			if(src_fp == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					src_path, arch_path, zip_strerror(archive)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Allocate the link's target value */
			value = (gchar *)g_malloc((value_len + 1) * sizeof(gchar));
			if(value == NULL)
			{
				core->last_error_ptr = "Memory allocation error.";
				zip_fclose(src_fp);
				g_free(src_path);
				g_free(tar_path);
				status = -3;
				break;
			}

			/* Read the link's target value from the source
			 * file in the PKZip archive
			 */
			if(value_len > 0)
			{
				bytes_read = (gint)zip_fread(src_fp, value, value_len);
				if(bytes_read < 0)
				{
					gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
						src_path, arch_path, zip_file_strerror(src_fp)
					);
					edv_archive_extract_set_error(core, msg);
					g_free(msg);
					g_free(value);
					zip_fclose(src_fp);
					status = -1;
					QUERY_CONTINUE_EXTRACT
				}
			}
			else
			{
				bytes_read = 0;
			}

			if((gulong)bytes_read < value_len)
				value[bytes_read] = '\0';
			else
				value[value_len] = '\0';

			/* Close the source and target files */
			zip_fclose(src_fp);

			/* Remove the target object in case it exists */
			if(edv_archive_extract_remove(tar_path))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path,
					src_path,
					arch_path,
					g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				g_free(value);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Create the link */
			if(edv_link_create(
				tar_path,			/* Path */
				value			/* Target */
			) != 0)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path, arch_path, g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				g_free(value);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Delete the link's target value */
			g_free(value);

			cur_size += (gulong)bytes_read;

			/* Append the target path to the list of extracted objects */
			if(new_paths_list_rtn != NULL)
				*new_paths_list_rtn = g_list_append(
					*new_paths_list_rtn,
					STRDUP(tar_path)
				);
		}
		/* All else extract the file */
		else
		{
			FILE *tar_fp;
			struct zip_file *src_fp;
			gint tar_fd;
			const guint m = edv_get_umask();
			gulong block_size;
			EDVVFSObject *obj;

			/* Open the source file in the PKZip archive */
			src_fp = zip_fopen(
				archive,
				src_path,
				0
			);
			if(src_fp == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					src_path,
					arch_path,
					zip_strerror(archive)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Remove the target object in case it exists */
			if(edv_archive_extract_remove(tar_path))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path,
					src_path,
					arch_path,
					g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				zip_fclose(src_fp);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Open the target file for writing */
			tar_fp = fopen((const char *)tar_path, "wb");
			if(tar_fp == NULL)
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path,
					arch_path,
					g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				zip_fclose(src_fp);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Append the target path to the list of extracted objects */
			if(new_paths_list_rtn != NULL)
				*new_paths_list_rtn = g_list_append(
					*new_paths_list_rtn,
					g_strdup(tar_path)
				);

			tar_fd = (gint)fileno(tar_fp);

			/* Set the file permissions */
			(void)edv_permissions_set_fd(
				tar_fd,
				edv_stat_mode_to_edv_permissions(~m) &
				(EDV_PERMISSION_UR | EDV_PERMISSION_UW |
				 EDV_PERMISSION_GR | EDV_PERMISSION_GW |
				 EDV_PERMISSION_OR | EDV_PERMISSION_OW
				)
			);

			/* Calculate the block size */
			obj = edv_vfs_object_fstat(tar_fd);
			if(obj != NULL)
			{
				block_size = obj->block_size;
				edv_vfs_object_delete(obj);
			}
			else
			{
				block_size = 0l;
			}
			if(block_size > 0l)
			{
				/* Extract one block size at a time */
				gint	bytes_read,
					bytes_written;
				const gint read_buf_len = (gint)block_size;
				guint8 *read_buf = (guint8 *)g_malloc(
					block_size * sizeof(guint8)
				);
				if(read_buf == NULL)
				{
					core->last_error_ptr = "Memory allocation error.";
					zip_fclose(src_fp);
					(void)fclose(tar_fp);
					g_free(src_path);
					g_free(tar_path);
					status = -3;
					break;
				}

				while(TRUE)
				{
					/* Read one block from the source file in the archive */
					bytes_read = (gint)zip_fread(
						src_fp,
						read_buf, (int)read_buf_len
					);
					if(bytes_read <= 0)
					{
						if(bytes_read < 0)
						{
							gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
								src_path,
								arch_path,
								zip_file_strerror(src_fp)
							);
							edv_archive_extract_set_error(core, msg);
							g_free(msg);
							status = -1;
						}
						break;
					}

					/* Write the block to the file */
					bytes_written = (gint)fwrite(
						read_buf,
						sizeof(guint8),
						(size_t)bytes_read,
						tar_fp
					);

					/* Add to the current size */
					if(bytes_written > 0)
						cur_size += (gulong)bytes_written;

					/* Write error? */
					if(bytes_written != bytes_read)
					{
						const gint error_code = (gint)errno;
						gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
							tar_path,
							arch_path,
							g_strerror(error_code)
						);
						edv_archive_extract_set_error(core, msg);
						g_free(msg);
						status = -1;
						break;
					}

					/* Update progress */
					if(show_progress && ProgressDialogIsQuery())
					{
						if(edv_archive_extract_update_progress_dialog(
							cfg_list,
							NULL,
							NULL,
							(total_size > 0l) ?
			  ((gfloat)cur_size / (gfloat)total_size) : 0.0f,
							toplevel,
							FALSE
						) > 0)
						{
							status = -4;
							break;
						}
					}
				}

				/* Delete the read buffer */
				g_free(read_buf);
			}
			else
			{
				/* File system IO block size not available,
				 * extract one character at a time
				 */
				gint bytes_read, bytes_written;
				const gint buf_len = 1;
				guint8 buf[1];

				while(TRUE)
				{
					/* Read one character from the source file in the archive */
					bytes_read = (gint)zip_fread(src_fp, buf, (int)buf_len);
					if(bytes_read <= 0)
					{
						if(bytes_read < 0)
						{
							gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
								src_path,
								arch_path,
								zip_file_strerror(src_fp)
							);
							edv_archive_extract_set_error(core, msg);
							g_free(msg);
							status = -1;
						}
						break;
					}

					/* Write the character to the file */
					if(fputc((int)buf[0], tar_fp) != EOF)
						bytes_written = 1;
					else
						bytes_written = 0;

					/* Add to the current size */
					if(bytes_written > 0)
						cur_size += (gulong)bytes_written;

					/* Write error? */
					if(bytes_written != bytes_read)
					{
						const gint error_code = (gint)errno;
						gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
							tar_path,
							arch_path,
							g_strerror(error_code)
						);
						edv_archive_extract_set_error(core, msg);
						g_free(msg);
						status = -1;
						break;
					}

					/* Update progress */
					if(show_progress && ProgressDialogIsQuery())
					{
						if(edv_archive_extract_update_progress_dialog(
							cfg_list,
							NULL,
							NULL,
							(total_size > 0l) ?
			  ((gfloat)cur_size / (gfloat)total_size) : 0.0f,
							toplevel,
							FALSE
						) > 0)
						{
							status = -4;
							break;
						}
					}
				}
			}

			/* Close the source and target files */
			zip_fclose(src_fp);
			(void)fclose(tar_fp);

			/* Did an error occure or user aborted while writing
			 * the extracted object?
			 */
			if(status != 0)
			{
				if(status == -4)
				{
					break;
				}
				else
				{
					QUERY_CONTINUE_EXTRACT
				}
			}
		}

		/* Report the progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(src_path);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

		/* Preserve timestamps? */
		if(preserve_timestamps &&
		   (obj->type != EDV_OBJECT_TYPE_LINK)
		)
		{
			(void)edv_utime(
				tar_path,
				obj->access_time,
				obj->modify_time
			);
		}

		/* Report the final progress for this object? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(src_path);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

		g_free(src_path);
		g_free(tar_path);

#undef QUERY_CONTINUE_EXTRACT
	}

	/* Close the PKZip archive */
	if(zip_close(archive))
	{
		if((status == 0) || (status == -4))
		{
			gchar *msg = g_strdup_printf(
"Unable to close the PKZip Archive:\n\
\n\
    %s\n\
\n\
%s.",
				arch_path,
				zip_strerror(archive)
			);
			edv_archive_extract_set_error(core, msg);
			g_free(msg);
			status = -1;
		}
	}

	return(status);
#else
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs_extracted;
	gchar		*cmd = NULL,
			*pwd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*src_path,
			*prog_unzip = EDV_GET_S(EDV_CFG_PARM_PROG_UNZIP);

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = edv_getcwd();
	if(edv_setcwd(dest_path))
	{
		core->last_error_ptr =
"Unable to change working directory to the destination directory.";
		CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	cmd = g_strdup_printf(
		"\"%s\" -o -X \"%s\"",
		prog_unzip,
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
		gchar *s;
		GList *glist;
		EDVArchiveObject *obj;

		for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
				continue;

			if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
			{
				const gint len = STRLEN(src_path);

				s = g_strconcat(cmd, " \"", src_path, NULL);
				g_free(cmd);
				cmd = s;

				/* If directory does not have a tailing
				 * deliminator then we must append one or else
				 * it will not get matched
				 */
				if(len > 1)
				{
					if(src_path[len - 1] != G_DIR_SEPARATOR)
					{
						s = g_strconcat(cmd, G_DIR_SEPARATOR_S, NULL);
						g_free(cmd);
						cmd = s;
					}
				}
				s = g_strconcat(cmd, "\"", NULL);
				g_free(cmd);
				cmd = s;
			}
			else
			{
				s = g_strconcat(cmd, " \"", src_path, "\"", NULL);
				g_free(cmd);
				cmd = s;
			}
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the extract objects from archive command */
	pid = edv_archive_extract_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the extract command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	/* Begin extracting */
	status = 0;
	nobjs_extracted = 0;

	/* Read the archive extract messages from the output streams */
	if(cstdout != NULL)
	{
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;
		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "creating:") ||
				   STRCASEPFX(s, "updating:") ||
				   STRCASEPFX(s, "inflating:") ||
				   STRCASEPFX(s, "extracting:") ||
				   STRCASEPFX(s, "linking:")
				)
				{
					gchar	*s2,
						*path;

					/* Seek s past the prefix to the path value */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					/* Cap the first blank character after the path */
					s2 = strpbrk(s, " \t\n\r");
					if(s2 != NULL)
						*s2 = '\0';

					path = edv_paths_join(
						dest_path,
						s
					);
					edv_path_simplify(path);

					/* Append this path to the list of paths
					 * extracted from the archive
					 */
					if(new_paths_list_rtn != NULL)
						*new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							STRDUP(path)
						);

					/* Update the progress dialog's label? */
					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						(if(edv_archive_extract_update_progress_dialog(
							cfg_list,
							s,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}

					nobjs_extracted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
		if(edv_archive_extract_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBZIP */
}
