#ifdef HAVE_LIBXAR
# include <stdlib.h>
# include <string.h>
# include <errno.h>
/* The X Archive library does not allow public error callback
 * setting... yet, when it does you can uncomment the below
 */
/* # define USE_LIBXAR_ERROR_CB */
# include <xar/xar.h>
# include <gtk/gtk.h>
# include "cdialog.h"
# include "progressdialog.h"
# include "edv_types.h"
# include "libendeavour2-base/edv_utils.h"
# include "libendeavour2-base/edv_path.h"
# include "libendeavour2-base/edv_directory.h"
# include "libendeavour2-base/edv_archive_obj.h"
# include "edv_date_format.h"
# include "edv_utils_gtk.h"
# include "edv_progress.h"
# include "edv_archive_extract.h"
# include "edv_archive_extract_xar.h"
# include "endeavour2.h"
# include "config.h"
#else
# include "edv_types.h"
# include "endeavour2.h"
# include "config.h"
#endif


#if defined(HAVE_LIBXAR) && defined(USE_LIBXAR_ERROR_CB)
/* Warning, this does not work yet becuase libxar does not support it */
typedef struct _EDVArchXArErrorData	EDVArchXArErrorData;
#define EDV_ARCH_XAR_ERROR_DATA(p)	((EDVArchXArErrorData *)(p))

static int32_t EDVArchExtractXArErrorCB(
	int32_t sev, int32_t err, xar_errctx_t ctx, void *data
);
#endif


gint edv_archive_extract_xar(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);


#if defined(HAVE_LIBXAR) && defined(USE_LIBXAR_ERROR_CB)
/* 
 *	X Archive Library Error Data:
 */
struct _EDVArchXArErrorData {
	EDVCore *core;
	const gchar	*arch_path;
};
#endif


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


#if defined(HAVE_LIBXAR) && defined(USE_LIBXAR_ERROR_CB)
/*
 *	X Archive library error callback.
 */
static int32_t EDVArchExtractXArErrorCB(
	int32_t sev, int32_t err, xar_errctx_t ctx, void *data
)
{
	xar_file_t xar_fp;
	gint error_code;
	gchar *path;
	const gchar *error_msg;
	EDVArchXArErrorData *d = EDV_ARCH_XAR_ERROR_DATA(data);
	if(d == NULL)
		return(0);

	xar_fp = xar_err_get_file(ctx);
	error_msg = (const gchar *)xar_err_get_string(ctx);
	error_code = (gint)xar_err_get_errno(ctx);

	switch(sev)
	{
	  case XAR_SEVERITY_DEBUG:
	  case XAR_SEVERITY_INFO:
	  case XAR_SEVERITY_WARNING:
	  case XAR_SEVERITY_NORMAL:
		/* Ignore these */
		break;

	  case XAR_SEVERITY_NONFATAL:
	  case XAR_SEVERITY_FATAL:
		path = (xar_fp != NULL) ? (gchar *)xar_get_path(xar_fp) : NULL;
		if(path != NULL)
		{
			gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s, %s.",
				path,
				d->arch_path,
				error_msg,
				g_strerror(error_code)
			);
			edv_archive_extract_set_error(core, msg);
			g_free(msg);
			g_free(path);
		}
		else
		{
			gchar *msg = g_strdup_printf(
"Unable to extract an object from the archive:\n\
\n\
    %s\n\
\n\
%s, %s.",
				d->arch_path,
				error_msg,
				g_strerror(error_code)
			);
			edv_archive_extract_set_error(core, msg);
			g_free(msg);
		}
		break;
	}

	return(0);
}
#endif	/* HAVE_LIBXAR && USE_LIBXAR_ERROR_CB */


/*
 *	Extract object from an X Archive.
 */
gint edv_archive_extract_xar(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list,
	const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
#ifdef HAVE_LIBXAR
	xar_t xar;
	xar_iter_t i1;
	xar_file_t xar_fp;
	gint status;
	gulong		cur_size,
					total_size;
	gchar		*p1,
					*src_path,
					*tar_path,
					*parent,
					*pwd;
	GList *glist;
	CfgList *cfg_list = core->cfg_list;
	EDVArchiveObject *obj;
#ifdef USE_LIBXAR_ERROR_CB
	EDVArchXArErrorData *d;
#endif

	/* Open the X Archive for reading */
	xar = xar_open(arch_path, READ);
	if(xar == NULL)
	{
		gchar *msg = g_strdup_printf(
"Unable to open the X Archive for reading:\n\
\n\
    %s",
			arch_path
		);
		edv_archive_extract_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

#ifdef USE_LIBXAR_ERROR_CB
	/* Set the error callback */
	d = EDV_ARCH_XAR_ERROR_DATA(g_malloc0(
		sizeof(EDVArchXArErrorData)
	));
	if(d != NULL)
	{
		d->core = core;
		d->arch_path = arch_path;
		xar_register_errhandler(
			xar,
			EDVArchExtractXArErrorCB, d
		);
	}
#endif

	/* Iterate through the list of objects to extract in order
	 * to calculate the total uncompressed size
	 */
	total_size = 0l;
	for(glist = objs_list; glist != NULL; glist = g_list_next(glist))
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		total_size += obj->size;
	}

	/* Record the previous working directory */
	pwd = edv_getcwd();

	/* Set the extraction directory as the current working
	 * directory
	 */
	edv_setcwd(dest_path);

	/* Iterate through the list of objects to extract and extract
	 * each one
	 */
	cur_size = 0l;
	status = 0;
	for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		/* Get the path of the object in the archive */
		src_path = STRDUP(obj->path);
		if(STRISEMPTY(src_path))
		{
			g_free(src_path);
			continue;
		}

		/* Format the path of the object to extract
		 *
		 * Note that if preserve_directories is FALSE it will have
		 * no affect because the libxar API always extracts
		 * preserving directories
		 */
		tar_path = edv_paths_join(
			dest_path,
			src_path
		);
		if(STRISEMPTY(tar_path))
		{
			core->last_error_ptr =
"Unable to generate the extracted object's path.";
			g_free(src_path);
			g_free(tar_path);
			status = -1;
			break;
		}

		edv_path_simplify(tar_path);

		/* Update the progress dialog to display the current
		 * object being extracted?
		 */
		if(show_progress)
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				src_path,
				tar_path,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(src_path);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

#define QUERY_CONTINUE_EXTRACT	{			\
 /* Count this object even after a failed extraction */	\
 cur_size += obj->size;					\
 if(new_paths_list_rtn != NULL)				\
  *new_paths_list_rtn = g_list_append(			\
   *new_paths_list_rtn,					\
   STRDUP(tar_path)					\
  );							\
													\
 /* Need to query the user? */				\
 if(interactive && !(*yes_to_all)) {			\
  gint response;					\
  gchar *msg = g_strdup_printf(				\
"An error occured while extracting the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue extracting subsequent objects?",		\
   src_path,						\
   tar_path						\
  );							\
  edv_play_sound_error(core);				\
  CDialogSetTransientFor(toplevel);			\
  response = CDialogGetResponse(			\
   "Extract Failed",					\
   msg,							\
   NULL,						\
   CDIALOG_ICON_ERROR,					\
   CDIALOG_BTNFLAG_YES |				\
    CDIALOG_BTNFLAG_YES_TO_ALL |			\
   CDIALOG_BTNFLAG_NO,					\
   CDIALOG_BTNFLAG_YES					\
  );							\
  g_free(msg);						\
  CDialogSetTransientFor(NULL);				\
													\
  /* Stop extracting? */				\
  if((response == CDIALOG_RESPONSE_NO) ||		\
     (response == CDIALOG_RESPONSE_CANCEL) ||		\
     (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
  )							\
  {							\
   g_free(src_path);					\
   g_free(tar_path);					\
   break;						\
  }							\
													\
  /* Yes to all? */					\
  if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
   *yes_to_all = TRUE;					\
 }							\
							\
 g_free(src_path);					\
 g_free(tar_path);					\
 continue;						\
}

		/* Need to create the parent directory(ies)? */
		parent = g_dirname(tar_path);
		if(parent != NULL)
		{
			if(!edv_path_lexists(parent))
			{
				/* Create each parent directory(ies) and add
				 * them to the list of extracted objects
				 */
				if(edv_directory_create(
					parent,
					TRUE,		/* Create parents */
					new_paths_list_rtn
				))
				{
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"Unable to create the parent directories to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
						tar_path,
						arch_path,
						g_strerror(error_code)
					);
					edv_archive_extract_set_error(core, msg);
					g_free(msg);
					g_free(parent);
					g_free(src_path);
					g_free(tar_path);
					status = -1;
					break;
				}
			}
			g_free(parent);
		}

		/* Report the progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(src_path);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

		/* Create an X Archive iterator and find this object in
		 * the X Archive
		 */
		i1 = xar_iter_new(xar);
		if(i1 == 0)
		{
			core->last_error_ptr =
"Unable to create a new X Archive iterator.";
			g_free(src_path);
			g_free(tar_path);
			status = -3;
			break;
		}
		for(xar_fp = xar_file_first(xar, i1);
			xar_fp != NULL;
			xar_fp = xar_file_next(i1)
		)
		{
			p1 = (gchar *)xar_get_path(xar_fp);
			if(p1 == NULL)
				continue;

			if(!strcmp((const char *)src_path, (const char *)p1))
			{
				g_free(p1);
				break;
			}

			g_free(p1);
		}
		xar_iter_free(i1);

		/* No such object in the X Archive? */
		if(xar_fp == NULL)
		{
			g_free(src_path);
			g_free(tar_path);
			continue;
		}

		/* Report the progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(src_path);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

		/* Remove the existing destination object if it exists */
		if(edv_archive_extract_remove(tar_path))
		{
			const gint error_code = (gint)errno;
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path,
					src_path,
					arch_path,
					g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
			}
			status = -1;
			QUERY_CONTINUE_EXTRACT
		}

		/* Extract this object */
		if(xar_extract(xar, xar_fp) != 0)
		{
			/* Extract failed */
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s",
					src_path,
					arch_path
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
			}
			status = -1;
			QUERY_CONTINUE_EXTRACT
		}

		/* Count this object's size after extraction */
		cur_size += obj->size;

		/* Append the target path to the list of extracted objects */
		if(new_paths_list_rtn != NULL)
			*new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn,
				STRDUP(tar_path)
			);

		g_free(src_path);
		g_free(tar_path);

		/* Report the progress for this object? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				status = -4;
				break;
			}
		}

#undef QUERY_CONTINUE_EXTRACT
	}

	/* Close the X Archive */
	if(xar_close(xar))
	{
		/* No previous error and no user abort? */
		if((status == 0) || (status != -4))
		{
			if(core->last_error_ptr == NULL)
			{
				gchar *msg = g_strdup_printf(
"Unable to close the X Archive:\n\
\n\
    %s",
					arch_path
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
			}
			status = -1;
		}
	}

	/* Restore the previous working directory */
	if(pwd != NULL)
	{
		(void)edv_setcwd(pwd);
		g_free(pwd);
	}

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
		if(edv_archive_extract_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

#ifdef USE_LIBXAR_ERROR_CB
	g_free(d);
#endif

	return(status);
#else
	core->last_error_ptr = "Unsupported archive format.";
	return(-2);
#endif	/* HAVE_LIBXAR */
}
