#ifdef HAVE_LIBTAR
# include <string.h>
# include <errno.h>
# include <fcntl.h>				/* open() */
# include <sys/types.h>				/* open() */
# include <sys/stat.h>				/* open() */
# ifdef HAVE_LIBZ
#  include <zlib.h>
# endif
# ifdef HAVE_LIBBZ2
#  include <bzlib.h>
# endif
# include <libtar.h>
# include <gtk/gtk.h>
# include "cdialog.h"
# include "progressdialog.h"
# include "edv_types.h"
# include "libendeavour2-base/edv_utils.h"
# include "libendeavour2-base/edv_path.h"
# include "libendeavour2-base/edv_directory.h"
# include "libendeavour2-base/edv_archive_obj.h"
# include "edv_utils_gtk.h"
# include "edv_progress.h"
# include "edv_archive_extract.h"
# include "edv_archive_extract_tar.h"
# include "endeavour2.h"
# include "config.h"
#else
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <errno.h>
# include <signal.h>
# include <gtk/gtk.h>
# include "progressdialog.h"
# include "cfg.h"
# include "edv_types.h"
# include "libendeavour2-base/edv_utils.h"
# include "libendeavour2-base/edv_path.h"
# include "libendeavour2-base/edv_stream.h"
# include "libendeavour2-base/edv_process.h"
# include "libendeavour2-base/edv_archive_obj.h"
# include "edv_utils_gtk.h"
# include "edv_progress.h"
# include "edv_archive_extract.h"
# include "edv_archive_extract_tar.h"
# include "endeavour2.h"
# include "edv_cfg_list.h"
# include "config.h"
#endif


#ifdef HAVE_LIBTAR
# ifdef HAVE_LIBZ
static int edv_archive_extract_libtar_open_libz_cb(const char *path, int oflags, int mode);
# endif
# ifdef HAVE_LIBBZ2
static int edv_archive_extract_libtar_open_libbz2_cb(const char *path, int oflags, int mode);
static ssize_t edv_archive_extract_libtar_read_libbz2_cb(int fd, void *buf, size_t buf_len);
static ssize_t edv_archive_extract_libtar_write_libbz2_cb(int fd, const void *buf, size_t buf_len);
static int edv_archive_extract_libtar_close_libbz2_cb(int fd);
# endif
static EDVObjectType edv_archive_extract_libtar_get_type(TAR *tar);
static gchar *edv_archive_extract_libtar_get_path(TAR *tar);
static gint edv_archive_extract_libtar_next(TAR *tar);
#endif  /* HAVE_LIBTAR */

gint edv_archive_extract_tar(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list,
	const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define STRCASEPFX(s,p) ((((s) != NULL) && ((p) != NULL)) ?     \
 !g_strncasecmp((s), (p), strlen(p)) : FALSE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


#if defined(HAVE_LIBTAR) && defined(HAVE_LIBBZ2)
typedef struct {
	int             fd;
	BZFILE          *bz2d;
} bz2_cb_data_struct;
#define BZ2_CB_DATA(p)                  ((bz2_cb_data_struct *)(p))
#endif


#ifdef HAVE_LIBTAR
# ifdef HAVE_LIBZ
/*
 *	libtar GZip open callback.
 */
static int edv_archive_extract_libtar_open_libz_cb(const char *path, int oflags, int mode)
{
	const char *libz_oflags;
	gint fd;
	gzFile zd;

	switch(oflags & O_ACCMODE)
	{
	  case O_WRONLY:
		libz_oflags = "wb";
		break;
	  case O_RDONLY:
		libz_oflags = "rb";
		break;
	  default:
	  case O_RDWR:
		errno = EINVAL;
		return(-1);
		break;
	}

	fd = (gint)open(path, oflags, mode);
	if(fd < 0)
		return(-1);

	if((oflags & O_CREAT) && edv_fchmod(fd, mode))
	{
		const gint error_code = (gint)errno;
		(void)close(fd);
		errno = (int)error_code;
		return(-1);
	}

	zd = gzdopen(fd, libz_oflags);
	if(zd == NULL)
	{
		errno = ENOMEM;
		return(-1);
	}

	/* Return the gzFile descriptor as the descriptor */
	return((int)zd);
}
# endif	/* HAVE_LIBZ */

# ifdef HAVE_LIBBZ2
/*
 *	libtar BZip2 open callback.
 */
static int edv_archive_extract_libtar_open_libbz2_cb(const char *path, int oflags, int mode)
{
	const char *libbz2_oflags;
	gint fd;
	bz2_cb_data_struct *d;

	switch(oflags & O_ACCMODE)
	{
	  case O_WRONLY:
		libbz2_oflags = "wb";
		break;
	  case O_RDONLY:
		libbz2_oflags = "rb";
		break;
	  default:
	  case O_RDWR:
		errno = EINVAL;
		return(-1);
		break;
	}

	d = BZ2_CB_DATA(g_malloc0(
		sizeof(bz2_cb_data_struct)
	));
	if(d == NULL)
		return(-1);

	d->fd = fd = (gint)open(path, oflags, mode);
	if(fd < 0)
	{
		const gint error_code = (gint)errno;
		g_free(d);
		errno = (int)error_code;
		return(-1);
	}

	if(oflags & O_CREAT)
	{
		if(edv_fchmod(fd, mode))
		{
			const gint error_code = (gint)errno;
			(void)close(d->fd);
			g_free(d);
			errno = (int)error_code;
			return(-1);
		}
	}

	d->bz2d = BZ2_bzdopen(d->fd, libbz2_oflags);
	if(d->bz2d == NULL)
	{
		(void)close(fd);
		g_free(d);
		errno = ENOMEM;
		return(-1);
	}

	/* Return the libbz2 callback data as the descriptor */
	return((int)d);
}

/*
 *      libtar BZip2 read callback.
 */
static ssize_t edv_archive_extract_libtar_read_libbz2_cb(int fd, void *buf, size_t buf_len)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	return(BZ2_bzread(d->bz2d, buf, buf_len));
}

/*
 *      libtar BZip2 write callback.
 */
static ssize_t edv_archive_extract_libtar_write_libbz2_cb(int fd, const void *buf, size_t buf_len)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	return(BZ2_bzwrite(d->bz2d, (void *)buf, buf_len));
}

/*
 *      libtar BZip2 close callback.
 */
static int edv_archive_extract_libtar_close_libbz2_cb(int fd)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	BZ2_bzclose(d->bz2d);
	g_free(d);
	return(0);
}
# endif /* HAVE_LIBBZ2 */

/*
 *	Gets the Tape Archive's current object's type.
 */
static EDVObjectType edv_archive_extract_libtar_get_type(TAR *tar)
{
	if(TH_ISREG(tar))
		return(EDV_OBJECT_TYPE_FILE);
	else if(TH_ISDIR(tar))
		return(EDV_OBJECT_TYPE_DIRECTORY);
	else if(TH_ISLNK(tar) || TH_ISSYM(tar))
		return(EDV_OBJECT_TYPE_LINK);
	else if(TH_ISBLK(tar))
		return(EDV_OBJECT_TYPE_DEVICE_BLOCK);
	else if(TH_ISCHR(tar))
		return(EDV_OBJECT_TYPE_DEVICE_CHARACTER);
	else if(TH_ISFIFO(tar))
		return(EDV_OBJECT_TYPE_FIFO);
	else
		return(EDV_OBJECT_TYPE_UNKNOWN);
}

/*
 *      Gets the Tape Archive's current object's path.
 */
static gchar *edv_archive_extract_libtar_get_path(TAR *tar)
{
	struct tar_header *tar_obj = &tar->th_buf;
	gchar *path;

	if(tar_obj->gnu_longname != NULL)
		path = g_strdup(tar_obj->gnu_longname);
	else if(tar_obj->prefix[0] != '\0')
		path = g_strdup_printf(
			"%.155s/%.100s",
			tar_obj->prefix,
			tar_obj->name
		);
	else
		path = g_strdup_printf(
			"%.100s",
			tar_obj->name
		);
	if(path == NULL)
		return(NULL);

	/* Remove any tailing deliminators */
	edv_path_simplify(path);

	return(path);
}

/*
 *      Seeks to the next object in the Tape Archive.
 */
static gint edv_archive_extract_libtar_next(TAR *tar)
{
	/* No need to seek to the next object if the current object
	 * is a file
	 */
	if(!TH_ISREG(tar))
		return(0);

	/* Seek past this file */
	if(tar_skip_regfile(tar) != 0)
		return(-1);

	return(0);
}
#endif  /* HAVE_LIBTAR */


/*
 *	Extract object from a Tape Archive.
 */
gint edv_archive_extract_tar(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list,
	const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
#ifdef HAVE_LIBTAR
#ifdef HAVE_LIBZ
	tartype_t tar_io_z_cb = {
		(openfunc_t)edv_archive_extract_libtar_open_libz_cb,
		(closefunc_t)gzclose,
		(readfunc_t)gzread,
		(writefunc_t)gzwrite
	};
#endif
#ifdef HAVE_LIBBZ2
	tartype_t tar_io_bz2_cb = {
		(openfunc_t)edv_archive_extract_libtar_open_libbz2_cb,
		(closefunc_t)edv_archive_extract_libtar_close_libbz2_cb,
		(readfunc_t)edv_archive_extract_libtar_read_libbz2_cb,
		(writefunc_t)edv_archive_extract_libtar_write_libbz2_cb
	};
#endif
	tartype_t *tar_io_cbs;
	TAR *tar = NULL;
	gint status;
	gulong		cur_size,
			total_size,
			cur_time;
	CfgList *cfg_list = core->cfg_list;

	/* Select/set the IO callbacks based on the compression format */
	if(is_compress_compressed)
	{
		tar_io_cbs = NULL;
	}
	else if(is_gzip_compressed)
	{
#ifdef HAVE_LIBZ
		tar_io_cbs = &tar_io_z_cb;
#else
		tar_io_cbs = NULL;
#endif
	}
	else if(is_bzip2_compressed)
	{
#ifdef HAVE_LIBBZ2
		tar_io_cbs = &tar_io_bz2_cb;
#else
		tar_io_cbs = NULL;
#endif
	}
	else
	{
		tar_io_cbs = NULL;
	}

	/* Open the Tape Archive for reading */
	if(tar_open(
		&tar,
		(char *)arch_path,
		tar_io_cbs,
		O_RDONLY, 0,
		TAR_GNU | TAR_NOOVERWRITE
	) == -1)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to open the Tape Archive for reading:\n\
\n\
    %s\n\
\n\
%s.",
			arch_path,
			g_strerror(error_code)
		);
		edv_archive_extract_set_error(core, msg);
		g_free(msg);
		return(-1);
	}
	if(tar == NULL)
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to open the Tape Archive for reading:\n\
\n\
    %s\n\
\n\
%s.",
			arch_path,
			g_strerror(error_code)
		);
		edv_archive_extract_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Calculate the total size of all the objects to extract */
	total_size = 0l;
	if(objs_list != NULL)
	{
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			total_size += obj->size;
		}
	}

	/* Extract each object */
	status = 0;
	cur_size = 0l;
	cur_time = edv_time();
	if(TRUE)
	{
		gchar	*parent_path,
			*tar_path,
			*src_path;
		EDVObjectType type;

		/* Reading the next object in the Tape Archive */
		while(th_read(tar) == 0)
		{
			/* Get the path of this object in the archive */
			src_path = edv_archive_extract_libtar_get_path(tar);
			if(src_path == NULL)
			{
				/* Unable to get the path of this object in the
				 * archive, try to seek to the next object in
				 * the archive
				 */
				if(edv_archive_extract_libtar_next(tar))
				{
					const gint error_code = (gint)errno;
					gchar *msg = g_strdup_printf(
"An error occured while reading from the Tape Archive:\n\
\n\
    %s\n\
\n\
%s.",
						arch_path,
						g_strerror(error_code)
					);
					edv_archive_extract_set_error(core, msg);
					g_free(msg);
					status = -1;
					return(status);
				}
				continue;
			}

			/* If not extracting all the objects then check if
			 * this object is in the list of objects to extract
			 */
			if(!extract_all)
			{
				GList *glist;
				EDVArchiveObject *obj;

				for(glist = objs_list;
					glist != NULL;
					glist = g_list_next(glist)
				)
				{
					obj = EDV_ARCHIVE_OBJECT(glist->data);
					if(obj == NULL)
						continue;

					if(obj->path == NULL)
						continue;

					/* Should this object be extracted? */
					if(!strcmp((const char *)src_path, (const char *)obj->path))
						break;
				}
				/* Do not extract this object? */
				if(glist == NULL)
				{
					/* Seek to the next object in the archive */
					if(edv_archive_extract_libtar_next(tar))
					{
						const gint error_code = (gint)errno;
						gchar *msg = g_strdup_printf(
"An error occured while reading the object:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
							src_path,
							arch_path,
							g_strerror(error_code)
						);
						edv_archive_extract_set_error(core, msg);
						g_free(msg);
						g_free(src_path);
						status = -1;
						return(status);
					}
					g_free(src_path);
					continue;
				}
			}

			/* Format the path of the object to extract */
			if(preserve_directories)
				tar_path = edv_paths_join(
					dest_path,
					src_path
				);
			else
				tar_path = edv_paths_join(
					dest_path,
					g_basename(src_path)
				);
			if(STRISEMPTY(tar_path))
			{
				g_free(src_path);
				g_free(tar_path);
				core->last_error_ptr =
"Unable to generate the extracted object's path.";
				status = -1;
				break;
			}

			/* Remove any tailing deliminators */
			edv_path_simplify(tar_path);

			/* Get this object's type */
			type = edv_archive_extract_libtar_get_type(tar);

			/* Update the progress dialog message to display
			 * the current object being extracted
			 */
			if(show_progress)
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					src_path,
					tar_path,
					(total_size > 0l) ?
						((gfloat)cur_size / (gfloat)total_size) : 0.0f,
					toplevel,
					FALSE
				) > 0)
				{
					g_free(src_path);
					g_free(tar_path);
					status = -4;
					break;
				}
			}

#define QUERY_CONTINUE_EXTRACT	{			\
							\
 /* Need to query the user? */				\
 if(interactive && !(*yes_to_all)) {			\
  gint response;					\
  gchar *msg = g_strdup_printf(				\
"An error occured while extracting the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue extracting subsequent objects?",		\
   src_path,						\
   tar_path						\
  );							\
  edv_play_sound_error(core);				\
  CDialogSetTransientFor(toplevel);			\
  response = CDialogGetResponse(			\
   "Extract Failed",					\
   msg,							\
   NULL,						\
   CDIALOG_ICON_ERROR,					\
   CDIALOG_BTNFLAG_YES |				\
    CDIALOG_BTNFLAG_YES_TO_ALL |			\
   CDIALOG_BTNFLAG_NO,					\
   CDIALOG_BTNFLAG_YES					\
  );							\
  g_free(msg);						\
  CDialogSetTransientFor(NULL);				\
							\
  /* Stop extracting? */				\
  if((response == CDIALOG_RESPONSE_NO) ||		\
     (response == CDIALOG_RESPONSE_CANCEL) ||		\
     (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
  )							\
  {							\
   g_free(src_path);					\
   g_free(tar_path);					\
   break;						\
  }							\
							\
  if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
   *yes_to_all = TRUE;					\
 }							\
							\
 g_free(src_path);					\
 g_free(tar_path);					\
 continue;						\
}

			/* Need to create the parent directory(ies)? */
			parent_path = g_dirname(tar_path);
			if(parent_path != NULL)
			{
				if(!edv_path_lexists(parent_path))
				{
					/* Create each parent directory
					 * and add them to the list of
					 * extracted objects
					 */
					if(edv_directory_create(
						parent_path,
						TRUE,	/* Create parents */
						new_paths_list_rtn
					))
					{
						const gint error_code = (gint)errno;
						gchar *msg = g_strdup_printf(
"Unable to create the parent directories to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
							tar_path,
							arch_path,
							g_strerror(error_code)
						);
						edv_archive_extract_set_error(core, msg);
						g_free(msg);
						g_free(parent_path);
						g_free(src_path);
						g_free(tar_path);
						status = -1;
						break;
					}
				}
				g_free(parent_path);
			}

			/* Report the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					(total_size > 0l) ?
						((gfloat)cur_size / (gfloat)total_size) : 0.0f,
					toplevel,
					FALSE
				) > 0)
				{
					g_free(src_path);
					g_free(tar_path);
					status = -4;
					break;
				}
			}

			/* Count and report this object as being extracted
			 *
			 * (Actual extraction will occure below)
			 */
			cur_size += (gulong)th_get_size(tar);

			if(new_paths_list_rtn != NULL)
				*new_paths_list_rtn = g_list_append(
					*new_paths_list_rtn,
					STRDUP(tar_path)
				);

			/* Remove the target object in case it already exists */
			if(edv_archive_extract_remove(tar_path))
			{
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path,
					src_path,
					arch_path,
					g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Extract this object and seek to the next object */
			if(tar_extract_file(
				tar,
				tar_path
			) != 0)
			{
				/* Extract error */
				const gint error_code = (gint)errno;
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
					tar_path,
					arch_path,
					g_strerror(error_code)
				);
				edv_archive_extract_set_error(core, msg);
				g_free(msg);
				status = -1;
				QUERY_CONTINUE_EXTRACT
			}

			/* Report the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					(total_size > 0l) ?
						((gfloat)cur_size / (gfloat)total_size) : 0.0f,
					toplevel,
					FALSE
				) > 0)
				{
					g_free(src_path);
					g_free(tar_path);
					status = -4;
					break;
				}
			}

			/* Do not preserve timestamps? */
			if(!preserve_timestamps &&
			   (type != EDV_OBJECT_TYPE_LINK)
			)
			{
				/* Set the current time for this object */
				edv_utime(
					tar_path,
					cur_time,		/* Access time */
					cur_time		/* Modify time */
				);
			}

			/* Report the final progress for this object? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					(total_size > 0l) ?
						((gfloat)cur_size / (gfloat)total_size) : 0.0f,
					toplevel,
					FALSE
				) > 0)
				{
					g_free(src_path);
					g_free(tar_path);
					status = -4;
					break;
				}
			}

			g_free(tar_path);
			g_free(src_path);

#undef QUERY_CONTINUE_EXTRACT
		}
	}

	/* Close the Tape Archive */
	if(tar_close(tar) != 0)
	{
		if((status == 0) || (status == -4))
		{
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to close the Tape Archive:\n\
\n\
    %s\n\
\n\
%s.",
				arch_path,
				g_strerror(error_code)
			);
			edv_archive_extract_set_error(core, msg);
			g_free(msg);
			status = -1;
		}
	}

	return(status);
#else
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs_extracted;
	gchar		*cmd = NULL,
			*pwd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*src_path,
			*prog_tar = EDV_GET_S(EDV_CFG_PARM_PROG_TAR),
			*prog_bunzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BUNZIP2);

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = edv_getcwd();
	if(edv_setcwd(dest_path))
	{
		core->last_error_ptr =
"Unable to change working directory to the destination directory.";
		CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	if(is_compress_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -Z -x%s -v -f \"%s\"",
			prog_tar,
			preserve_timestamps ? "" : " -m",
			arch_path
		);
	else if(is_gzip_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -z -x%s -v -f \"%s\"",
			prog_tar,
			preserve_timestamps ? "" : " -m",
			arch_path
		);
	else if(is_bzip2_compressed)
		cmd = g_strdup_printf(
			"\"%s\" \"--use-compress-program=%s\" -x%s -v -f \"%s\"",
			prog_tar,
			prog_bunzip2,
			preserve_timestamps ? "" : " -m",
			arch_path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" -x%s -v -f \"%s\"",
			prog_tar,
			preserve_timestamps ? "" : " -m",
			arch_path
		);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
		gchar *s;
		GList *glist;
		EDVArchiveObject *obj;

		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
				continue;

			if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
			{
				const gint len = strlen(src_path);

				s = g_strconcat(
					cmd,
					" \"",
					src_path,
					NULL
				);
				if(s != NULL)
				{
					g_free(cmd);
					cmd = s;
				}

				/* If directory does not have a tailing
				 * deliminator then we must append one or else
				 * it will not get matched
				 */
				if(len > 1)
				{
					if(src_path[len - 1] != G_DIR_SEPARATOR)
					{
						s = g_strconcat(
							cmd,
							G_DIR_SEPARATOR_S,
							NULL
						);
						if(s != NULL)
						{
							g_free(cmd);
							cmd = s;
						}
					}
				}
				s = g_strconcat(
					cmd,
					"\"",
					NULL
				);
				if(s != NULL)
				{
					g_free(cmd);
					cmd = s;
				}
			}
			else
			{
				s = g_strconcat(
					cmd,
					" \"",
					src_path,
					"\"",
					NULL
				);
				if(s != NULL)
				{
					g_free(cmd);
					cmd = s;
				}
			}
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the extract objects from archive command */
	pid = edv_archive_extract_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the extract command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	/* Begin extracting */
	status = 0;
	nobjs_extracted = 0;

	/* Read the archive extract messages from the output streams */
	if(cstdout != NULL)
	{
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;
		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE           /* Nonblocking */
			))
			{
				gchar	*s = buf,
					*path;
				while(ISBLANK(*s))
					s++;

				path = edv_paths_join(
					dest_path,
					s
				);
				edv_path_simplify(path);

				/* Append this path to the list of paths
				 * extracted from the archive
				 */
				if(new_paths_list_rtn != NULL)
					*new_paths_list_rtn = g_list_append(
						*new_paths_list_rtn,
						STRDUP(path)
					);

				/* Update the progress dialog's label? */
				if(show_progress &&
				   !STRISEMPTY(path)
				)
				{
					if(edv_archive_extract_update_progress_dialog(
						cfg_list,
						s,
						path,
						progress,
						toplevel,
						FALSE
					) > 0)
					{
						(void)INTERRUPT(pid);
						pid = 0;
						status = -4;
						break;
					}
				}

				nobjs_extracted++;
				progress = (nobjs > 0) ?
					((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

				g_free(path);

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
		if(edv_archive_extract_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBTAR */
}
