#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "cfg.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_date_format.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_archive_extract.h"
#include "edv_archive_extract_tar.h"
#include "edv_archive_extract_xar.h"
#include "edv_archive_extract_zip.h"
#include "edv_op.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_replace_file_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_archive_extract_get_error(EDVCore *core);
void edv_archive_extract_set_error(
	EDVCore *core,
	const gchar *msg
);

/* Execute */
gint edv_archive_extract_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
);

/* Progress Dialog */
gint edv_archive_extract_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *src_path,
	const gchar *tar_path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);

/* Confirm Overwrite */
static gint edv_archive_extract_confirm_overwrite(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVArchiveObject *src_obj_stat,
	EDVVFSObject *tar_obj_stat,
	GtkWidget *toplevel
);

/* Remove Recursive */
gint edv_archive_extract_remove(const gchar *path);

/* Extract From Archive */
static gint edv_archive_extract_arj(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);
static gint edv_archive_extract_lha(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);
static gint edv_archive_extract_rar(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);
gint edv_archive_extract(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l
#define EDV_ARCHIVE_EXTRACT_MAX_ERROR_MESSAGE_LINES	\
					40

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define STRCASEPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
 !g_strncasecmp((s), (p), strlen(p)) : FALSE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Gets the last error message.
 */
const gchar *edv_archive_extract_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Coppies the error message to the core's last_error_buf
 *	and sets the core's last_error_ptr to point to it.
 */
void edv_archive_extract_set_error(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Executes the command using the shell specified by the
 *	configuration and opens and returns the requested streams.
 */
gint edv_archive_extract_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
)
{
	gchar		*shell_prog;
	const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Execute the list archive command */
	const gint pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		cstdin_rtn,
		cstdout_rtn,
		cstderr_rtn
	);

	g_free(shell_prog);

	return(pid);
}


/*
 *	Updates/maps the progress dialog.
 */
gint edv_archive_extract_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *src_path,
	const gchar *tar_path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
) 
{
	if((src_path != NULL) && (tar_path != NULL))
	{
		gchar	*src_path_shortened = edv_path_shorten(
			src_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*tar_path_shortened = edv_path_shorten(
			tar_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
			,
			src_path_shortened,
			tar_path_shortened
		);
		g_free(src_path_shortened);
		g_free(tar_path_shortened);
		edv_progress_dialog_map_archive_extract_animated(
			cfg_list,
			msg,
			progress,
			toplevel,
			force_remap
		);
		g_free(msg);
	}
	else
	{
		if(progress >= 0.0f)
			ProgressDialogUpdate(
				NULL,
				NULL,
				NULL,
				NULL,
				progress,
				EDV_PROGRESS_BAR_NTICKS,
				TRUE
			);
		else
			ProgressDialogUpdateUnknown(
				NULL,
				NULL,
				NULL,
				NULL,
				TRUE
			);
	}

	return(ProgressDialogStopCount());
}


/*
 *	Maps the confirmation dialog and queries user for replacing an
 *	object.
 *
 *	Returns one of CDIALOG_RESPONSE_*.
 */
static gint edv_archive_extract_confirm_overwrite(
	EDVCore *core,
	const gchar *src_path,
	const gchar *tar_path,
	EDVArchiveObject *src_obj_stat,
	EDVVFSObject *tar_obj_stat,
	GtkWidget *toplevel
)
{
	gint response;
	gchar		*msg,
					*src_date,
					*tar_date;
	EDVDateRelativity relativity;
	const gchar *format;
	gulong		src_size,
					tar_size;
	CfgList *cfg_list;

	if(core == NULL)
		return(CDIALOG_RESPONSE_NOT_AVAILABLE);

	cfg_list = core->cfg_list;

	/* Get date relativity and format */
	relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(
		EDV_CFG_PARM_DATE_FORMAT
	);

	/* Get date strings for source and target objects */
	src_date = edv_date_string_format(
		(src_obj_stat != NULL) ? src_obj_stat->modify_time : 0l,
		format, relativity
	);
	tar_date = edv_date_string_format(
		(tar_obj_stat != NULL) ? tar_obj_stat->modify_time : 0l,
		format, relativity
	);

	/* Get sizes of source and target objects in units of bytes */
	src_size = (src_obj_stat != NULL) ? src_obj_stat->size : 0l;
	tar_size = (tar_obj_stat != NULL) ? tar_obj_stat->size : 0l;

	/* Format confirm overwrite message */
	msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Reemplace:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Con:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Remplacer:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Avec:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ersetzen Sie:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Mit:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Sostituire:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Con:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vervang:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Met:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Reponha:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Com:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Erstatt:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Med:\n\
\n\
    %s (%ld byte%s) %s\n"
#else
"Replace:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
With:\n\
\n\
    %s (%ld byte%s) %s\n"
#endif
		,
		tar_path, tar_size, (tar_size == 1) ? "" : "s", tar_date,
		src_path, src_size, (src_size == 1) ? "" : "s", src_date
	);

	edv_play_sound_warning(core);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponseIconData(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Superposer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overwrite"
#else
"Confirm Overwrite"
#endif
		, msg, NULL,
		(guint8 **)icon_replace_file_32x32_xpm,
/* Note that we are unable to have a "No" option, it must be yes or
 * cancel the entire operation
 */
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
		CDIALOG_BTNFLAG_CANCEL,
		CDIALOG_BTNFLAG_CANCEL
	);
	CDialogSetTransientFor(NULL);

	g_free(msg);
	g_free(tar_date);
	g_free(src_date);

	return(response);
}


/*
 *	Removes the object.
 *
 *	If the object is a directory then all objects within that
 *	directory will be removed.
 *
 *	Returns 0 on success or if the object does not exist. Returns
 *	non-zero on error.
 */
gint edv_archive_extract_remove(const gchar *path)
{
	gint status = edv_directory_remove(
		path,
		TRUE,				/* Recursive */
		TRUE,				/* Force and remove all objects */
		NULL,				/* No removed paths list return */
		NULL, NULL
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		if(error_code == ENOENT)
			return(0);
		else
			return(status);
	}
	else
	{
		return(status);
	}
}


/*
 *	Extract objects from an ARJ archive.
 */
static gint edv_archive_extract_arj(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		pid,
			status,
			nobjs_extracted;
	gchar		*cmd = NULL,
			*pwd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*src_path,
			*prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = edv_getcwd();
	if(edv_setcwd(dest_path))
	{
		core->last_error_ptr =
"Unable to change working directory to the destination directory.";
		CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	cmd = g_strdup_printf(
		"\"%s\" %c -i -r -y \"%s\"",
		prog_arj,
		preserve_directories ? 'x' : 'e',
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
				continue;

			if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
			{
				const gint len = STRLEN(src_path);

				cmd = edv_strcat(cmd, " \"");
				cmd = edv_strcat(cmd, src_path);
				/* If directory does not have a tailing deliminator
				 * then we must append one or else it will not get
				 * matched
				 */
				if(len > 1)
				{
					if(src_path[len - 1] != G_DIR_SEPARATOR)
					{
						gchar delim_str[2];
						delim_str[0] = G_DIR_SEPARATOR;
						delim_str[1] = '\0';
						cmd = edv_strcat(cmd, delim_str);
					}
				}
				cmd = edv_strcat(cmd, "\"");
			}
			else
			{
				cmd = edv_strcat(cmd, " \"");
				cmd = edv_strcat(cmd, src_path);
				cmd = edv_strcat(cmd, "\"");
			}
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the extract objects from archive command */
	pid = edv_archive_extract_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the extract command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;
	nobjs_extracted = 0;

	/* Read the archive extract messages from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Update the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				"",
				arch_path,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE           /* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Extracting"))
				{
					gchar	*s2,
						*path;

					/* Seek s past the prefix to the path value */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					/* Cap the first blank character after the path */
					for(s2 = s; *s2 != '\0'; s2++)
					{
						if(ISBLANK(*s2))
						{
							*s2 = '\0';
							break;
						}
					}

					path = edv_paths_join(
						dest_path,
						s
					);
					edv_path_simplify(path);

					/* Append this path to the list of paths
					 * extracted from the archive
					 */
					if(new_paths_list_rtn != NULL)
						*new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							STRDUP(path)
						);


					/* Update the progress dialog's label? */
					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_extract_update_progress_dialog(
							cfg_list,
							s,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}

					nobjs_extracted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output stremas */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
		if(edv_archive_extract_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Extract objects from a LHA archive.
 */
static gint edv_archive_extract_lha(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		pid,
			status,
			nobjs_extracted;
	gchar		*cmd = NULL,
			*pwd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*src_path,
			*prog_lha = EDV_GET_S(EDV_CFG_PARM_PROG_LHA);

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = edv_getcwd();
	if(edv_setcwd(dest_path))
	{
		core->last_error_ptr =
"Unable to change working directory to the destination directory.";
		CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	cmd = g_strdup_printf(
		"\"%s\" -%cvf \"%s\"",
		prog_lha,
		preserve_directories ? 'x' : 'e',
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
				continue;

			if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
			{
				const gint len = STRLEN(src_path);

				cmd = edv_strcat(cmd, " \"");
				cmd = edv_strcat(cmd, src_path);
				/* If directory does not have a tailing deliminator
				 * then we must append one or else it will not get
				 * matched
				 */
				if(len > 1)
				{
					if(src_path[len - 1] != G_DIR_SEPARATOR)
					{
						gchar delim_str[2];
						delim_str[0] = G_DIR_SEPARATOR;
						delim_str[1] = '\0';
						cmd = edv_strcat(cmd, delim_str);
					}
				}
				cmd = edv_strcat(cmd, "\"");
			}
			else
			{
				cmd = edv_strcat(cmd, " \"");
				cmd = edv_strcat(cmd, src_path);
				cmd = edv_strcat(cmd, "\"");
			}
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the extract objects from archive command */
	pid = edv_archive_extract_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the extract command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;
	nobjs_extracted = 0;

	/* Read the archive extract messages from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Update the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				"",
				arch_path,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush any error messages */
			FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE           /* Nonblocking */
			))
			{
				gchar	*s = buf,
					*s2;
				while(ISBLANK(*s))
					s++;

				s2 = (gchar *)strstr((char *)s, " :");

				if(STRCASEPFX(s, "Making directory \""))
				{
					gchar *path;

					s += strlen("Making directory \"");
					s2 = (gchar *)strrchr((char *)s, '"');
					if(s2 != NULL)
						*s2 = '\0';

					path = edv_paths_join(
						dest_path,
						s
					);
					edv_path_simplify(path);

					/* Append this path to the list of paths
					 * extracted from the archive
					 */
					if(new_paths_list_rtn != NULL)
						*new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							STRDUP(path)
						);
				}
				else if(s2 != NULL)
				{
					gchar *path;

					s2 = (gchar *)strpbrk((char *)s, " \t");
					if(s2 != NULL)
						*s2 = '\0';

					path = edv_paths_join(
						dest_path,
						s
					);
					edv_path_simplify(path);

					/* Append this path to the list of paths
					 * extracted from the archive
					 */
					if(new_paths_list_rtn != NULL)
						*new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							STRDUP(path)
						);

					/* Update the progress dialog's label? */
					if(show_progress &&
					   !STRISEMPTY(path))
					{
						if(edv_archive_extract_update_progress_dialog(
							cfg_list,
							s,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}

					nobjs_extracted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output stremas */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
		if(edv_archive_extract_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Extract objects from a RAR archive.
 */
static gint edv_archive_extract_rar(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		pid,
			status,
			nobjs_extracted;
	gchar		*cmd = NULL,
			*pwd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*src_path,
			*prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);

#define CLEANUP_RETURN(_v_)	{		\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(cmd);					\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  (void)edv_setcwd(pwd);			\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = edv_getcwd();
	if(edv_setcwd(dest_path))
	{
		core->last_error_ptr =
"Unable to change working directory to the destination directory.";
		CLEANUP_RETURN(-1);
	}

	/* Format the extract object(s) from the RAR archive command
	 *
	 * RAR requires that a password be specified otherwise it will
	 * query for a password and block if an archive object seems
	 * to be encrypted, so if no password was specified to us
	 * then we need to explicitly pass the -p- option to tell RAR
	 * never to query for a password even when one may be needed
	 */
	if(STRISEMPTY(password))
		cmd = g_strdup_printf(
			"\"%s\" %c%s -p- -kb -o+ -y -c- \"%s\"",
			prog_rar,
			preserve_directories ? 'x' : 'e',
			preserve_timestamps ? " -tsm -tsc -tsa" : "",
			arch_path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" %c%s \"-p%s\" -kb -o+ -y -c- \"%s\"",
			prog_rar,
			preserve_directories ? 'x' : 'e',
			preserve_timestamps ? " -tsm -tsc -tsa" : "",
			password,
			arch_path
		);
	if(cmd == NULL)
	{
		core->last_error_ptr =
"Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
				continue;

			/* Do not put tailing deliminators on directories
			 * for directory objects in RAR archives
			 */

			cmd = edv_strcat(cmd, " \"");
			cmd = edv_strcat(cmd, src_path);
			cmd = edv_strcat(cmd, "\"");
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr =
"Unable to generate the extract command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the extract objects from archive command */
	pid = edv_archive_extract_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the extract command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;
	nobjs_extracted = 0;

	/* Read the archive extract messages from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		const gint max_errors = EDV_ARCHIVE_EXTRACT_MAX_ERROR_MESSAGE_LINES;
		gfloat progress = 0.0f;
		gint nerrors = 0;
		gchar	*buf = NULL,
			*error_buf = NULL;
		FILE *fp = cstdout;

		/* Update the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_extract_update_progress_dialog(
				cfg_list,
				"",
				arch_path,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_extract_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Extracting  ") ||
				   STRCASEPFX(s, "Creating  ")
				)
				{
					gchar	*s2,
						*path;

					/* Seek s past the prefix to the path value */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					/* Cap s at the end deliminator "  "
					 *
					 * The RAR archiver has no formal deliminator
					 * to denote the end of the object's name and
					 * the "OK" string for each line
					 */
					s2 = strstr(s, "  ");
					if(s2 == NULL)
						s2 = strchr(s, ' ');
					if(s2 == NULL)
						s2 = strchr(s, '\t');
					if(s2 != NULL)
						*s2 = '\0';

					path = edv_paths_join(
						dest_path,
						s
					);
					edv_path_simplify(path);

					/* Append this path to the list of paths
					 * extracted from the archive
					 */
					if(new_paths_list_rtn != NULL)
						*new_paths_list_rtn = g_list_append(
							*new_paths_list_rtn,
							STRDUP(path)
						);

					/* Update the progress dialog's label? */
					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_extract_update_progress_dialog(
							cfg_list,
							s,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}

					nobjs_extracted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			/* Check for errors and read no more than
			 * max_errors error messages from the error
			 * stream
			 */
			if(nerrors < max_errors)
			{
				if(edv_stream_read_lineptr(
					cstderr,
					&error_buf,
					FALSE		/* Nonblocking */
				))
					nerrors++;
			}
			else
			{
				FDISCARD(cstderr);
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Record any errors */
		if(nerrors > 0)
		{
			if(core->last_error_ptr == NULL)
				edv_archive_extract_set_error(
					core,
					error_buf
				);
			if(status == 0)
				status = -1;
		}

		g_free(error_buf);
		g_free(buf);
	}

	/* Close the output stremas */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
		if(edv_archive_extract_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Extracts objects from the archive.
 *
 *	The arch_path specifies the archive.
 *
 *	The objs_list specifies the list of EDVArchiveObject *
 *	objects to extract from the archive.
 *
 *	The dest_path specifies the absolute path of the location to
 *	extract the objects in the archive to.
 *
 *	If new_paths_list_rtn is not NULL then a list of gchar * paths
 *	describing the objects that have been extracted from the archive
 *	will be returned. The calling function must delete the returned
 *	list and each string.
 */
gint edv_archive_extract(
	EDVCore *core,
	const gchar *arch_path,
	GList *objs_list,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	const gulong time_start = edv_time();
	gint		status,
			nobjs;
	const gchar *arch_name;
	gchar *ldest_path = NULL;
	GList *glist;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(ldest_path);				\
						\
 return(_v_);					\
}
	if(ProgressDialogIsQuery())
	{
		edv_archive_extract_set_error(
			core,
"An operation is already in progress, please try again later."
		);
		return(-6);
	}

	/* Clear the last error message */
	edv_archive_extract_set_error(core, NULL);

	if(new_paths_list_rtn != NULL)
		*new_paths_list_rtn = NULL;

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(dest_path) || (yes_to_all == NULL)
	)
	{
		edv_archive_extract_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Nothing to extract? */
	if(objs_list == NULL)
		return(0);

	arch_name = g_basename(arch_path);

	/* Make a copy of the destination path and simplify it */
	ldest_path = STRDUP(dest_path);
	edv_path_simplify(ldest_path);

	if(!g_path_is_absolute(ldest_path))
	{
		gchar *msg = g_strdup_printf(
"Extract to location is not an absolute path:\n\
\n\
    %s",
			ldest_path
		);
		edv_archive_extract_set_error(core, msg);
		g_free(msg);
		CLEANUP_RETURN(-2);
	}

	/* Get the total number of objects to extract */
	nobjs = g_list_length(objs_list);

	/* Do overwrite check? */
	if(interactive && !(*yes_to_all))
	{
		/* Do overwrite check by iterating through the given list of
		 * objects in the archive and check an object at the would
		 * be extracted location exists
		 */
		gboolean got_cancel = FALSE;
		gchar *tar_path;
		const gchar *src_path;
		EDVArchiveObject *obj;
		EDVVFSObject *tar_lobj;

		/* Iterate through the list of objects to extract and check
		 * if their extracted locations already exist, prompt the
		 * user to overwrite or cancel in each case
		 */
		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			/* Get the path of the object within the archive */
			src_path = obj->path;
			if(STRISEMPTY(src_path))
				continue;

			/* Generate the path of the would be extracted location
			 * for this object in the archive
			 */
			if(preserve_directories)
				tar_path = edv_paths_join(
					ldest_path,
					src_path
				);
			else
				tar_path = edv_paths_join(
					ldest_path,
					g_basename(src_path)
				);
			if(tar_path == NULL)
				continue;

			edv_path_simplify(tar_path);

			/* Check if an object already exists at the would be
			 * extracted location, if an object exists then prompt
			 * for extract overwrite
			 */
			tar_lobj = edv_vfs_object_lstat(tar_path);
			if(tar_lobj != NULL)
			{
				status = edv_archive_extract_confirm_overwrite(
					core,
					src_path,	/* Source object in archive path*/
					tar_path,	/* Target extracted object path */
					obj,		/* Source object in archive stats */
					tar_lobj,	/* Target extracted object stats */
					toplevel
				);
				edv_vfs_object_delete(tar_lobj);
			}
			else
				status = CDIALOG_RESPONSE_YES;

			g_free(tar_path);

			/* Check user response */
			switch(status)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
				*yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
				break;
			  default:		/* All else assume cancel */
				got_cancel = TRUE;
				break;
			}
			if(*yes_to_all)
				break;
			if(got_cancel)
				break;
		}

		/* User aborted? */
		if(got_cancel)
		{
			CLEANUP_RETURN(-4);
		}
	}

	/* Check if the destination path does not exist, if it does not
	 * then we first need to update the list of new path returns
	 * with all non-existent compoent directories of the destination
	 * path and create all compoent directories of the destination
	 * path
	 */
	if(edv_directory_create(
		ldest_path,
		TRUE,				/* Create parents */
		new_paths_list_rtn
	))
	{
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to create the extract to location:\n\
\n\
    %s\n\
\n\
%s.",
			ldest_path,
			g_strerror(error_code)
		);
		edv_archive_extract_set_error(core, msg);
		g_free(msg);
		CLEANUP_RETURN(-1);
	}


	/* Begin extracting the source object to the destination specified
	 * by dest_obj from the archive object arch_path
	 *
	 * The extracting method will be determined by taking the
	 * extension of the archive object's name
	 */

	/* ARJ Archive */
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		status = edv_archive_extract_arj(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps
		);
	}
	/* LHA Archive */
	else if(edv_name_has_extension(arch_name, ".lha"))
	{
		status = edv_archive_extract_lha(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		status = edv_archive_extract_rar(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps
		);
	}
	/* Tape Archive (Compressed) */
	else if(edv_name_has_extension(arch_name, ".tar.Z"))
	{
		status = edv_archive_extract_tar(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps,
			TRUE,			/* Is compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (GZip) */
	else if(edv_name_has_extension(arch_name, ".tgz .tar.gz"))
	{
		status = edv_archive_extract_tar(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps,
			FALSE,			/* Not compress compressed */
			TRUE,			/* Is gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (BZip2) */
	else if(edv_name_has_extension(arch_name, ".tar.bz2"))
	{
		status = edv_archive_extract_tar(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			TRUE			/* Is bzip2 compressed */
		);
	}
	/* Tape Archive */
	else if(edv_name_has_extension(arch_name, ".tar"))
	{
		status = edv_archive_extract_tar(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* X Archive */
	else if(edv_name_has_extension(arch_name, ".xar"))
	{
		status = edv_archive_extract_xar(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		status = edv_archive_extract_zip(
			core,
			arch_path,
			objs_list, nobjs,
			extract_all,
			ldest_path,
			new_paths_list_rtn,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			preserve_directories,
			preserve_timestamps
		);
	}
	else
	{
		core->last_error_ptr = "Unsupported archive format.";
		status = -2;
	}

	/* Record history */
	edv_append_history(
		core,
		EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT,
		time_start,
		edv_time(),
		status,
		arch_path,			/* Source */
		ldest_path,			/* Target */
		core->last_error_ptr		/* Comment */
	);

	/* Need to flush disk changes since the archive may have been
	 * modified on another process and the changes have not reached
	 * our process yet
	 */
	(void)edv_sync();

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
