#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#ifdef HAVE_LIBZIP
# include <zip.h>
#endif
#include <gtk/gtk.h>

#include "cfg.h"

#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_utils_gtk.h"
#include "edv_progress.h"
#include "edv_archive_delete.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_archive_delete_get_error(EDVCore *core);
static void edv_archive_delete_set_error(
	EDVCore *core,
	const gchar *msg
);

/* Execute */
static gint edv_archive_delete_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
);

/* Progress Dialog */
static gint edv_archive_delete_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);

#ifdef HAVE_LIBZIP
static struct zip *edv_archive_delete_libzip_open_writing(
	const gchar *arch_path,
	gint *zip_error_code, gint *sys_error_code
);
#endif

/* Delete Object From Archive */
static gint edv_archive_delete_arj(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_delete_lha(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_delete_rar(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_delete_tar(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);
static gint edv_archive_delete_zip(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
gint edv_archive_delete(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define STRCASEPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?	\
 !g_strncasecmp((s), (p), strlen(p)) : FALSE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Gets the last error message.
 */
const gchar *edv_archive_delete_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}

/*
 *	Coppies the error message to the core's last_error_buf
 *	and sets the core's last_error_ptr to point to it.
 */
static void edv_archive_delete_set_error(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *      Executes the command using the shell specified by the
 *      configuration and opens and returns the requested streams.
 */
static gint edv_archive_delete_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
)
{
	gchar		*shell_prog;
	const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Execute the list archive command */
	const gint pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		cstdin_rtn,
		cstdout_rtn,
		cstderr_rtn
	);

	g_free(shell_prog);

	return(pid);
}


/*
 *	Updates/maps the progress dialog.
 */
static gint edv_archive_delete_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
	if((arch_path != NULL) && (path != NULL))
	{
		gchar	*arch_path_shortened = edv_path_shorten(
			arch_path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*path_shortened = edv_path_shorten(
			path,
			EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Borrar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrappen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anular:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryking:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Deleting:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n"
#endif   
			,
			path_shortened,
			arch_path_shortened
		);
		g_free(arch_path_shortened);
		g_free(path_shortened);
		edv_progress_dialog_map_archive_delete_animated(
			cfg_list,
			msg,
			progress,
			toplevel,
			force_remap
		);
		g_free(msg);
	}
	else
	{
		if(progress >= 0.0f)
			ProgressDialogUpdate(
				NULL,
				NULL,
				NULL,
				NULL,
				progress,
				EDV_PROGRESS_BAR_NTICKS,
				TRUE
			);
		else
			ProgressDialogUpdateUnknown(
				NULL,
				NULL,
				NULL,
				NULL,
				TRUE
			);
	}

	return(ProgressDialogStopCount());
}


#ifdef HAVE_LIBZIP
/*
 *      Opens the PKZip archive for writing.
 *
 *	Returns the handle to the PKZip archive.
 */
static struct zip *edv_archive_delete_libzip_open_writing(
	const gchar *arch_path,
	gint *zip_error_code, gint *sys_error_code
)
{
	struct zip *archive;
	EDVVFSObject *obj;

	/* If the PKZip archive exists but has a size of 0 bytes then
	 * it must be deleted first or else libzip will return an
	 * error stating that it is not a valid PKZip archive
	 */
	obj = edv_vfs_object_stat(arch_path);
	if(obj != NULL)
	{
		if(obj->size == 0l)
			(void)edv_unlink(arch_path);
		edv_vfs_object_delete(obj);
	}

	archive = zip_open(arch_path, ZIP_CREATE, zip_error_code);
	if(sys_error_code != NULL)
		*sys_error_code = errno;

	return(archive);
}
#endif	/* HAVE_LIBZIP */


/*
 *	Deletes the object(s) from the ARJ archive.
 */
static gint edv_archive_delete_arj(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the delete objects from archive command */
	cmd = g_strdup_printf(
		"\"%s\" d -i -y \"%s\"",
		prog_arj,
		arch_path
	);
	nobjs = 0;
	if(cmd != NULL)
	{
		gchar *s;
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			s = g_strconcat(
				cmd,
				" \"",
				obj->path,
				"\"",
				NULL
			);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}

			nobjs++;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the delete objects from archive command */
	pid = edv_archive_delete_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Begin monitoring the deleting processing until it
	 * finishes or until user abort
	 */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gint nobjs_deleted = 0;
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_delete_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

	                /* Flush any error messages */
	                FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Deleting"))
				{
					gchar *path;

					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					path = g_strdup(s);

					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_delete_update_progress_dialog(
							cfg_list,
							arch_path,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							g_free(path);
							status = -4;
							break;
						}
					}

					nobjs_deleted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_deleted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Deletes the object(s) from the LHA archive.
 */
static gint edv_archive_delete_lha(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_lha = EDV_GET_S(EDV_CFG_PARM_PROG_LHA);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the delete objects from archive command */
	cmd = g_strdup_printf(
		"\"%s\" -df \"%s\"",
		prog_lha,
		arch_path
	);
	nobjs = 0;
	if(cmd != NULL)
	{
		gchar *s;
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			s = g_strconcat(
				cmd,
				" \"",
				obj->path,
				"\"",
				NULL
			);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}

			nobjs++;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the delete objects from archive command */
	pid = edv_archive_delete_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Begin monitoring the deleting processing until it
	 * finishes or until user abort
	 */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gint nobjs_deleted = 0;
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_delete_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

	                /* Flush any error messages */
	                FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "delete"))
				{
					gchar *path;

					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					path = g_strdup(s);

					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_delete_update_progress_dialog(
							cfg_list,
							arch_path,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							g_free(path);
							status = -4;
							break;
						}
					}

					nobjs_deleted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_deleted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Deletes the object(s) from the RAR archive.
 */
static gint edv_archive_delete_rar(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			nobjs,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the delete objects from archive command */
	cmd = g_strdup_printf(
	    "\"%s\" d \"-p%s\" -y -c- \"%s\"",
	    prog_rar,
		(STRISEMPTY(password)) ? "-" : password,
	    arch_path
	);
	nobjs = 0;
	if(cmd != NULL)
	{
		gchar *s;
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			s = g_strconcat(
				cmd,
				" \"",
				obj->path,
				"\"",
				NULL
			);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}

			nobjs++;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the delete objects from archive command */
	pid = edv_archive_delete_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Begin monitoring the deleting processing until it
	 * finishes or until user abort
	 */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gint nobjs_deleted = 0;
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_delete_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

	                /* Flush any error messages */
	                FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Deleting from"))
				{
					/* Ignore */
				}
				else if(STRCASEPFX(s, "Deleting"))
				{
					gchar *path;

					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					path = g_strdup(s);

					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_delete_update_progress_dialog(
							cfg_list,
							arch_path,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							g_free(path);
							status = -4;
							break;
						}
					}

					nobjs_deleted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_deleted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Deletes the object(s) from the Tape Archive.
 */
static gint edv_archive_delete_tar(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			pid,
			nobjs;
	gchar		*cmd = NULL,
			*arch_uncompressed_path = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*prog_tar = EDV_GET_S(EDV_CFG_PARM_PROG_TAR),
			*prog_compress = EDV_GET_S(EDV_CFG_PARM_PROG_COMPRESS),
			*prog_uncompress = EDV_GET_S(EDV_CFG_PARM_PROG_UNCOMPRESS),
			*prog_gzip = EDV_GET_S(EDV_CFG_PARM_PROG_GZIP),
			*prog_gunzip = EDV_GET_S(EDV_CFG_PARM_PROG_GUNZIP),
			*prog_bzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BZIP2),
			*prog_bunzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BUNZIP2);
	GTimer *timer = g_timer_new();

#define CLEANUP_RETURN(_v_)	{		\
 g_timer_destroy(timer);			\
 (void)FCLOSE(cstdout);				\
 (void)FCLOSE(cstderr);				\
 g_free(cmd);					\
 g_free(arch_uncompressed_path);		\
						\
 return(_v_);					\
}

	/* If compression is specified then create a temporary
	 * uncompressed archive
	 */
	if(is_compress_compressed || is_gzip_compressed || is_bzip2_compressed)
		arch_uncompressed_path = edv_tmp_name(EDV_GET_S(EDV_CFG_PARM_DIR_TMP));

	/* Format the decompress archive command as needed */
	if(is_compress_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -c \"%s\" > \"%s\"",
			prog_uncompress,
			arch_path,
			arch_uncompressed_path
		);
	else if(is_gzip_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -d -q -c \"%s\" > \"%s\"",
			prog_gunzip,
			arch_path,
			arch_uncompressed_path
		);
	else if(is_bzip2_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -d -q -c \"%s\" > \"%s\"",
			prog_bunzip2,
			arch_path,
			arch_uncompressed_path
		);
	else
		cmd = NULL;

	/* Need to decompress the archive? */
	status = 0;
	if(cmd != NULL)
	{
		/* Execute the decompress archive command */
		pid = edv_archive_delete_execute(
			cfg_list,
			cmd,
			NULL,
			NULL,			/* Ignore standard output
						 * which has been redirected
						 * to the archive output */
			&cstderr
		);
		if(pid < 0)
		{
			core->last_error_ptr = "Unable to execute decompress command.";
			CLEANUP_RETURN(-1);
		}

		/* Delete the command */
		g_free(cmd);
		cmd = NULL;

		/* Map the progress dialog */
		if(show_progress)
		{
			gchar	*arch_path_shortened = edv_path_shorten(
				arch_path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
			),	*msg = g_strdup_printf(
"Decompressing Archive:\n\
\n\
    %s\n",
				arch_path_shortened
			);
			g_free(arch_path_shortened);
			edv_progress_dialog_map_archive_delete_animated(
				cfg_list,
				msg,
				-1.0f,
				toplevel,
				TRUE
			);
			g_free(msg);
		}

		/* Wait for the decompress archive process to finish */
		g_timer_start(timer);
		while(edv_pid_exists(pid))
		{
			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				/* Update the label every second */
				if(g_timer_elapsed(timer, NULL) >= 1.0)
				{
					gchar *msg;
					EDVVFSObject *obj = edv_vfs_object_stat(arch_uncompressed_path);
					if(obj != NULL)
					{
						const gulong size = obj->size;
						gchar   *arch_path_shortened = edv_path_shorten(
							arch_path,
							EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
						),
							*size_str = STRDUP(edv_str_size_delim(size));
						msg = g_strdup_printf(
"Decompressing Archive:\n\
\n\
    %s (%s %s)\n",
							arch_path_shortened,
							size_str,
							(size == 1l) ? "byte" : "bytes"
						);
						edv_vfs_object_delete(obj);
						g_free(arch_path_shortened);
						g_free(size_str);
					}
					else
					{
						msg = NULL;
					}
					ProgressDialogUpdateUnknown(
						NULL,
						msg,
						NULL,
						NULL,
						TRUE
					);
					g_timer_start(timer);
					g_free(msg);
				}
				else
				{
					ProgressDialogUpdateUnknown(
						NULL,
						NULL,
						NULL,
						NULL,
						TRUE
					);
				}
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush the output streams */
			FDISCARD(cstderr);

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		/* Unmap the progress dialog */
		if(show_progress)
		{
			ProgressDialogBreakQuery(TRUE);
			ProgressDialogSetTransientFor(NULL);
		}

		g_timer_stop(timer);

		/* Close the output streams */
		(void)FCLOSE(cstderr);
		cstderr = NULL;
	}
	/* Error or user aborted? */
	if(status != 0)
	{
		(void)edv_unlink(arch_uncompressed_path);
		CLEANUP_RETURN(status);
	}


	/* Format the delete object from archive command */
	cmd = g_strdup_printf(
		"\"%s\" --delete -v -f \"%s\"",
		prog_tar,
		(arch_uncompressed_path != NULL) ?
			arch_uncompressed_path : arch_path
	);
	nobjs = 0;
	if(cmd != NULL)
	{
		gchar *s;
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			s = g_strconcat(
				cmd,
				" \"",
				obj->path,
				"\"",
				NULL
			);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}

			nobjs++;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the delete objects from archive command */
	pid = edv_archive_delete_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	/* Read the archive delete messages from the output streams
	 *
	 * Note that the tar program does not provide any output when
	 * deleting objects from the archive so no output will ever
	 * be printed
	 */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		/* Map the progress dialog? */
		if(show_progress)
		{
			gchar	*arch_path_shortened = edv_path_shorten(
				arch_path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
			),
				*msg = g_strdup_printf(
"Deleting Objects From Archive:\n\
\n\
    %s\n",
				arch_path_shortened
			);
			g_free(arch_path_shortened);
			edv_progress_dialog_map_archive_delete_animated(
				cfg_list,
				msg,
				-1.0f,
				toplevel,
				TRUE
			);
			g_free(msg);
		}

		/* Wait for the decompress archive process to finish */
		while(edv_pid_exists(pid))
		{
			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				ProgressDialogUpdateUnknown(
					NULL,
					NULL,
					NULL,
					NULL,
					TRUE
				);
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush the output streams */
			FDISCARD(cstdout);
			FDISCARD(cstderr);

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}
	}
	/* Error or user aborted? */
	if(status != 0)
	{
		(void)edv_unlink(arch_uncompressed_path);
		CLEANUP_RETURN(status);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
		if(edv_archive_delete_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			1.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}


	/* Format the compress archive command as needed */
	if(is_compress_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -c \"%s\" > \"%s\"",
			prog_compress,
			arch_uncompressed_path,
			arch_path
		);
	else if(is_gzip_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -c \"%s\" > \"%s\"",
			prog_gzip,
			arch_uncompressed_path,
			arch_path
		);
	else if(is_bzip2_compressed)
		cmd = g_strdup_printf(
			"\"%s\" -z -c -q \"%s\" > \"%s\"",
			prog_bzip2,
			arch_uncompressed_path,
			arch_path
		);
	else
		cmd = NULL;

	/* Need to compress the archive? */
	if(cmd != NULL)
	{
		/* Execute the compress archive command */
		pid = edv_archive_delete_execute(
			cfg_list,
			cmd,
			NULL,
			NULL,			/* Ignore standard output
						 * which has been redirected
						 * to the archive output */
			&cstderr
		);
		if(pid < 0)
		{
			core->last_error_ptr = "Unable to execute the compress command.";
			CLEANUP_RETURN(-1);
		}

		/* Delete the command */
		g_free(cmd);
		cmd = NULL;

		/* Unmap the progress dialog from the above operation */
		if(show_progress)
		{
			ProgressDialogBreakQuery(TRUE);
			ProgressDialogSetTransientFor(NULL);
		}

		/* Map the progress dialog? */
		if(show_progress)
		{
			gchar	*arch_path_shortened = edv_path_shorten(
				arch_path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
			),
				*msg = g_strdup_printf(
"Compressing Archive:\n\
\n\
    %s\n",
				arch_path_shortened
			);
			g_free(arch_path_shortened);
			edv_progress_dialog_map_archive_delete_animated(
				cfg_list,
				msg,
				-1.0f,
				toplevel,
				TRUE
			);
			g_free(msg);
		}

		/* Wait for the decompress archive process to finish */
		g_timer_start(timer);
		while(edv_pid_exists(pid))
		{
			/* Update the progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				/* Update the label every second */
				if(g_timer_elapsed(timer, NULL) >= 1.0)
				{
					gchar *msg;
					EDVVFSObject *obj = edv_vfs_object_stat(arch_path);
					if(obj != NULL)
					{
						const gulong size = obj->size;
						gchar *arch_path_shortened = edv_path_shorten(
							arch_path,
							EDV_PROGRESS_DLG_PATH_MAX_CHARS - 10
						),
							*size_str = STRDUP(edv_str_size_delim(size));
						msg = g_strdup_printf(
"Compressing Archive:\n\
\n\
    %s (%s %s)\n",
							arch_path_shortened,
							size_str,
							(size == 1l) ? "byte" : "bytes"
						);
						edv_vfs_object_delete(obj);
						g_free(arch_path_shortened);
						g_free(size_str);
					}
					else
					{
						msg = NULL;
					}
					ProgressDialogUpdateUnknown(
						NULL,
						msg,
						NULL,
						NULL,
						TRUE
					);
					g_timer_start(timer);
					g_free(msg);
				}
				else
				{
					ProgressDialogUpdateUnknown(
						NULL,
						NULL,
						NULL,
						NULL,
						TRUE
					);
				}
				if(ProgressDialogStopCount() > 0)
				{
					INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Flush the output streams */
			FDISCARD(cstderr);

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		g_timer_stop(timer);

		/* Remove the uncompressed copy of the archive */
		(void)edv_unlink(arch_uncompressed_path);

		/* Close the output streams */
		(void)FCLOSE(cstderr);
		cstderr = NULL;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Deletes the object(s) from the PKZip archive.
 */
static gint edv_archive_delete_zip(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
#ifdef HAVE_LIBZIP
	struct zip *archive;
	gint		n,
			status,
			obj_num,
			zip_error_code, sys_error_code;
	const gint nobjs = g_list_length(obj_list);
	const gchar *path;
	gchar *tar_path;
	GList *glist;
	CfgList *cfg_list = core->cfg_list;
	EDVArchiveObject *obj;

	/* Delete each the objects from the archive */
	status = 0;
	obj_num = 0;
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
			continue;

		path = obj->path;
		if(STRISEMPTY(path))
			continue;

		obj_num++;

		/* Display the object being deleted? */
		if(show_progress)
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				arch_path,
				path,
				(nobjs > 0) ?
					((gfloat)obj_num / (gfloat)nobjs) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				status = -4;
				break;
			}
		}

		/* Open the PKZip archive for writing */
		archive = edv_archive_delete_libzip_open_writing(
			arch_path,
			&zip_error_code, &sys_error_code
		);
		if(archive == NULL)
		{
			gchar	*msg,
				err_msg[1024];
			zip_error_to_str(
				err_msg, sizeof(err_msg),
				zip_error_code, sys_error_code
			);
			msg = g_strdup_printf(
"Unable to open the PKZip archive for writing:\n\
\n\
    %s\n\
\n\
%s.",
				arch_path,
				err_msg
			);
			edv_archive_delete_set_error(core, msg);
			g_free(msg);
			status = -1;
			break;
		}

		/* Report progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(nobjs > 0) ?
					((gfloat)obj_num / (gfloat)nobjs) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)zip_close(archive);
				status = -4;
				break;
			}
		}

		/* Format the target path, the path of the object within
		 * the PKZip archive
		 *
		 * If the object is a directory then a tailing deliminator
		 * must be appended or else it will not be matched
		 */
		if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
			tar_path = g_strconcat(
				path,
				G_DIR_SEPARATOR_S,
				NULL
			);
		else
			tar_path = g_strdup(path);
		if(tar_path == NULL)
		{
			(void)zip_close(archive);
			core->last_error_ptr = "Memory allocation error.";
			status = -3;
			break;
		}

		/* Look for the object in the PKZip archive */
		n = zip_name_locate(archive, tar_path, 0);
		if(n < 0)
		{
			/* Object not in the archive, ignore */
			(void)zip_close(archive);
			g_free(tar_path);
			continue;
		}

		/* Report progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(nobjs > 0) ?
					((gfloat)obj_num / (gfloat)nobjs) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)zip_close(archive);
				g_free(tar_path);
				status = -4;
				break;
			}
		}

		/* Delete this object in the PKZip archive */
		if(zip_delete(
			archive,
			n
		) != 0)
		{
			/* Unable to delete this object */
			gchar *msg = g_strdup_printf(
"Unable to delete:\n\
\n\
    %s\n\
\n\
From the archive:\n\
\n\
    %s\n\
\n\
%s.",
				tar_path,
				arch_path,
				zip_strerror(archive)
			);
			edv_archive_delete_set_error(core, msg);
			g_free(msg);
			(void)zip_close(archive);
			g_free(tar_path);
			status = -1;
			break;
		}

		/* Write/flush changes to the PKZip archive and close it */
		if(zip_close(archive) != 0)
		{
			/* If not interrupted during the write and close then
			 * set the status to indicate error, otherwise the error
			 * was that the user aborted
			 */
			if(status != -4)
			{
				gchar *msg = g_strdup_printf(
"Unable to close the PKZip archive:\n\
\n\
    %s\n\
\n\
%s.",
					arch_path,
					zip_strerror(archive)
				);
				edv_archive_delete_set_error(core, msg);
				g_free(msg);
				g_free(tar_path);
				status = -1;
				break;
			}
		}

		/* Report the final progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				(nobjs > 0) ?
					((gfloat)obj_num / (gfloat)nobjs) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(tar_path);
				status = -4;
				break;
			}
		}

		g_free(tar_path);
	}

	return(status);
#else
	FILE		*cstdout = NULL,
			*cstderr = NULL;
	gint		status,
			nobjs,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_zip = EDV_GET_S(EDV_CFG_PARM_PROG_ZIP);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the delete from archive command */
	cmd = g_strdup_printf(
		"\"%s\" -d \"%s\"",
		prog_zip,
		arch_path
	);
	nobjs = 0;
	if(cmd != NULL)
	{
		gchar *s;
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			s = g_strconcat(
				cmd,
				" \"",
				obj->path,
				"\"",
				NULL
			);
			if(s != NULL)
			{
				g_free(cmd);
				cmd = s;
			}

			nobjs++;
		}
	}
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the delete from archive command */
	pid = edv_archive_delete_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the delete command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Begin monitoring the deleting processing until it
	 * finishes or until user abort
	 */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gint nobjs_deleted = 0;
		gfloat progress = 0.0f;
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_delete_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Check if the process is no longer running */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Update progress? */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_delete_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					progress,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

	                /* Flush any error messages */
	                FDISCARD(cstderr);

			/* Read the next line */
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "deleting:"))
				{
					gchar *path;

					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					path = g_strdup(s);

					if(show_progress &&
					   !STRISEMPTY(path)
					)
					{
						if(edv_archive_delete_update_progress_dialog(
							cfg_list,
							arch_path,
							path,
							progress,
							toplevel,
							FALSE
						) > 0)
						{
							g_free(path);
							status = -4;
							break;
						}
					}

					nobjs_deleted++;
					progress = (nobjs > 0) ?
						((gfloat)nobjs_deleted / (gfloat)nobjs) : 0.0f;

					g_free(path);
				}

				g_free(buf);
				buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(fp));

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBZIP */
}

/*
 *	Deletes the object(s) from the archive.
 *
 *	The arch_path specifies the archive.
 *
 *	The obj_list specifies the list of objects to delete from the
 *	archive.
 */
gint edv_archive_delete(
	EDVCore *core,
	const gchar *arch_path,
	GList *obj_list,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint status;
	const gulong time_start = edv_time();
	const gchar *arch_name;

#define CLEANUP_RETURN(_v_)	{	\
					\
 return(_v_);				\
}

	if(ProgressDialogIsQuery())
	{
		edv_archive_delete_set_error(
			core,
"An operation is already in progress, please try again later."
		);
		return(-6);
	}

	/* Clear the last error message */
	edv_archive_delete_set_error(core, NULL);

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   (yes_to_all == NULL)
	)
	{
		edv_archive_delete_set_error(core, "Invalid value.");
		return(-2);
	}

	/* No objects to delete? */
	if(obj_list == NULL)
		return(0);

	/* Begin deleting the source object from the archive object
	 * arch_path. The deleting method will be determined by taking
	 * the extension of the archive object's name
	 */

	/* ARJ Archive */
	arch_name = g_basename(arch_path);
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		status = edv_archive_delete_arj(
			core,
			arch_path,
			obj_list,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	/* LHA Archive */
	else if(edv_name_has_extension(arch_name, ".lha"))
	{
		status = edv_archive_delete_lha(
			core,
			arch_path,
			obj_list,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		status = edv_archive_delete_rar(
			core,
			arch_path,
			obj_list,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	/* Tape Archive (Compressed) */
	else if(edv_name_has_extension(arch_name, ".tar.Z"))
	{
		status = edv_archive_delete_tar(
			core,
			arch_path,
			obj_list,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			TRUE,			/* Is compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (GZip) */
	else if(edv_name_has_extension(arch_name, ".tgz .tar.gz"))
	{
		status = edv_archive_delete_tar(
			core,
			arch_path,
			obj_list,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			FALSE,			/* Not compress compressed */
			TRUE,			/* Is gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (BZip2) */
	else if(edv_name_has_extension(arch_name, ".tar.bz2"))
	{
		status = edv_archive_delete_tar(
			core,
			arch_path,
			obj_list,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			TRUE			/* Is bzip2 compressed */
		);
	}
	/* Tape Archive */
	else if(edv_name_has_extension(arch_name, ".tar"))
	{
		status = edv_archive_delete_tar(
			core,
			arch_path,
			obj_list,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			FALSE,			/* Not compress compressed */
			FALSE,			/* Not gzip compressed */
			FALSE			/* Not bzip2 compressed */
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		status = edv_archive_delete_zip(
			core,
			arch_path,
			obj_list,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	else
	{
		core->last_error_ptr = "Unsupported archive format.";
		status = -2;
	}

	/* Record history */
	if(status != 0)
	{
		EDVArchiveObject *obj = (obj_list != NULL) ?
			EDV_ARCHIVE_OBJECT(obj_list->data) : NULL;
		edv_append_history(
			core,
			EDV_HISTORY_ARCHIVE_OBJECT_DELETE,
			time_start,
			edv_time(),
			status,
			arch_path,		/* Source */
			(obj != NULL) ? obj->path : NULL, /* Target */
			core->last_error_ptr	/* Comment */
		);
	}
	else
	{
		GList *glist;
		EDVArchiveObject *obj;
		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
				continue;

			edv_append_history(
				core,
				EDV_HISTORY_ARCHIVE_OBJECT_DELETE,
				time_start,
				edv_time(),
				status,
				arch_path,	/* Source */
				obj->path,	/* Target */
				core->last_error_ptr		/* Comment */
			);
		}
	}

	/* Need to flush disk changes since the archive may have been
	 * modified on another process and the changes have not reached
	 * our process yet
	 */
	(void)edv_sync();

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
