#include <stdio.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_LIBZIP
# include <zip.h>
#endif
#include <glib.h>

#include "cfg.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_archive_comment.h"

#include "edv_cfg_list.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Execute */
static gint edv_archive_comment_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
);

/* Get Archive Comment */
static gchar *edv_archive_comment_get_arj(
	CfgList *cfg_list,
	const gchar *arch_path
);
static gchar *edv_archive_comment_get_rar(
	CfgList *cfg_list,
	const gchar *arch_path
);
static gchar *edv_archive_comment_get_zip(
	CfgList *cfg_list,
	const gchar *arch_path
);
gchar *edv_archive_comment_get(
	CfgList *cfg_list,
	const gchar *arch_path
);

/* Set Archive Comment */
static gint edv_archive_comment_set_write_comment_file(
	const gchar *path,
	const gchar *prefix,
	const gchar *postfix,
	const gchar *comment
);
static gint edv_archive_comment_set_arj(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
);
static gint edv_archive_comment_set_rar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
);
#ifdef HAVE_LIBZIP   
static struct zip *edv_archive_comment_libzip_open_writing(
	const gchar *arch_path,
	gint *zip_error
);
#endif
static gint edv_archive_comment_set_zip(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
);
gint edv_archive_comment_set(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define STRCASEPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?     \
			 !g_strncasecmp((s),(p),strlen(p)) : FALSE)
#define STRPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?     \
			 !strncmp((s),(p),strlen(p)) : FALSE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Executes the command using the shell specified by the
 *	configuration and opens and returns the requested streams.
 */
static gint edv_archive_comment_execute(
	CfgList *cfg_list,
	const gchar *cmd,
	FILE **cstdin_rtn,
	FILE **cstdout_rtn,
	FILE **cstderr_rtn
)
{
	gchar		*shell_prog;
	const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Execute the list archive command */
	const gint pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		cstdin_rtn,
		cstdout_rtn,
		cstderr_rtn
	);

	g_free(shell_prog);

	return(pid);
}


/*
 *	Gets the archive comment from the ARJ archive.
 */
static gchar *edv_archive_comment_get_arj(
	CfgList *cfg_list,
	const gchar *arch_path
)
{
	FILE		*cstdout,
			*cstderr;
	gint pid;
	const gchar *prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);
	gchar		*cmd,
			*comment = NULL;

	/* Format the get archive comment command */
	cmd = g_strdup_printf(
		"\"%s\" l -y -i \"%s\" \"nosuchfile\"",
		prog_arj,
		arch_path
	);
	if(cmd == NULL)
		return(comment);

	/* Execute the get archive comment command */
	pid = edv_archive_comment_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(cmd);

	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(comment);
	}

	/* Read the comment from the output stream */
	if(cstdout != NULL)
	{
		gboolean got_header = FALSE;
		const gchar	*delim_header = "Archive created:",
				*delim_footer = "     0 file(s)";
		gchar *buf = NULL;
		FILE *fp = cstdout;

		/* Read past the header or to the end of file */
		while(!feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE		/* Block */
			))
			{
				if(STRPFX(buf, delim_header))
				{
					g_free(buf);
					buf = NULL;
					got_header = TRUE;
					break;
				}
				g_free(buf);
				buf = NULL;
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		g_free(buf);
		buf = NULL;

		/* Able to read past the header? */
		if(got_header)
		{
			/* Read all subsequent lines as the comment */
			while(!feof(fp))
			{
				FDISCARD(cstderr);
				if(edv_stream_read_lineptr(
					fp,
					&buf,
					TRUE	/* Block */
				))
				{
					/* Footer deliminator encountered? */
					if(STRPFX(buf, delim_footer))
						break;

					/* Add this line to the comment */
					comment = edv_strcat(
						comment,
						buf
					);

	                                g_free(buf);
	                                buf = NULL;
	                        }
				else
					edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		        }
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(comment);
}

/*
 *	Gets the archive comment from the RAR archive.
 */
static gchar *edv_archive_comment_get_rar(
	CfgList *cfg_list,
	const gchar *arch_path
)
{
	FILE		*cstdout,
			*cstderr,
			*fp;
	const gchar *prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);
	gint pid;
	gchar		*cmd,
			*comment_path,
			*comment = NULL;

	/* Generate a temporary file to write the RAR comment to */
	comment_path = edv_tmp_name(EDV_GET_S(EDV_CFG_PARM_DIR_TMP));
	if(comment_path == NULL)
		return(comment);

	/* Format the write RAR comment to file command */
	cmd = g_strdup_printf(
		"\"%s\" cw -y -inul \"%s\" \"%s\"",
		prog_rar,
		arch_path,
		comment_path
	);
	if(cmd == NULL)
	{
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(comment);
	}

	/* Execute the get archive comment command */
	pid = edv_archive_comment_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(cmd);

	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(comment);
	}

	/* Wait for the process to finish */
	while(edv_pid_exists(pid))
	{
		FDISCARD(cstdout);
		FDISCARD(cstderr);
		edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
	}

	/* Open the archive comment file for reading */
	fp = fopen((const char *)comment_path, "rb");
	if(fp != NULL)
	{
		gchar *buf;
		while(!feof(fp))
		{
			buf = edv_stream_read_str(
				fp,
				TRUE		/* Block */
			);
			if(buf != NULL)
			{
				comment = edv_strcat(
					comment,
					buf
				);
				g_free(buf);
			}
			else
				edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}
		fclose(fp);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	/* Delete the archive comment file */
	(void)edv_unlink(comment_path);
	g_free(comment_path);

	return(comment);
}

/*
 *	Gets the archive comment from the PKZip archive.
 */
static gchar *edv_archive_comment_get_zip(
	CfgList *cfg_list,
	const gchar *arch_path
)
{
#ifdef HAVE_LIBZIP
	struct zip *archive;
	gint		zip_error,
			comment_len = 0;
	const gchar *comment;
	gchar *dcomment;

	/* Open the PKZip archive for reading */
	archive = zip_open(arch_path, 0, &zip_error);
	if(archive == NULL)
		return(NULL);

	/* Get the PKZip archive comment */
	comment = zip_get_archive_comment(
		archive,
		&comment_len,
		0
	);
	if((comment == NULL) || (comment_len <= 0))
	{
		zip_close(archive);
		return(NULL);
	}

	/* Make a copy of the comment and null terminate it */
	dcomment = (gchar *)g_malloc((comment_len + 1) * sizeof(gchar));
	if(dcomment == NULL)
	{
		zip_close(archive);
		return(NULL);
	}
	(void)memcpy(
		dcomment,
		comment,
		comment_len * sizeof(gchar)
	);
	dcomment[comment_len] = '\0';

	/* Close the PKZip archive */
	(void)zip_close(archive);

	return(dcomment);
#else
	FILE		*cstdout,
			*cstderr;
	const gchar *prog_unzip = EDV_GET_S(EDV_CFG_PARM_PROG_UNZIP);
	gint pid;
	gchar		*cmd,
			*comment = NULL;

	/* Format the get archive comment command */
	cmd = g_strdup_printf(
		"\"%s\" -z \"%s\"",
		prog_unzip,
		arch_path
	);
	if(cmd == NULL)
		return(comment);

	/* Execute the get archive comment command */
	pid = edv_archive_comment_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(cmd);

	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		return(comment);
	}

	/* Read the comment from the output stream */
	if(cstdout != NULL)
	{
		FILE *fp = cstdout;
		gint nlines_read;
		gchar *buf = NULL;

		/* Read past the header by skipping 1 line */
		nlines_read = 0;
		while((nlines_read < 1) && !feof(fp))
		{
			FDISCARD(cstderr);
			if(edv_stream_read_lineptr(
				fp,
				&buf,
				TRUE		/* Block */
			))
			{
				g_free(buf);
		                buf = NULL;
				nlines_read++;
			}
			else
	                        edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
		}

		g_free(buf);
		buf = NULL;

		/* Read past the first line? */
		if(nlines_read >= 1)
		{
			while(!feof(fp))
			{
	                        FDISCARD(cstderr);
		                buf = edv_stream_read_str(
			                fp,
					TRUE	/* Block */
	                        );
				if(buf != NULL)
		                {
					/* Add this line to the comment */
					comment = edv_strcat(
						comment,
						buf
					);
					g_free(buf);
				        buf = NULL;
				}
				else
		                        edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
			}

			g_free(buf);
			buf = NULL;
		}

		/* Flush any error messages */
		FDISCARD(cstderr);

		g_free(buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	return(comment);
#endif
}

/*
 *	Gets the archive's comment.
 *
 *	The arch_path specifies the archive.
 *
 *	Returns a dynamically allocated string describing the archive's
 *	comment or NULL if the archive does not have a comment or on
 *	error.
 */
gchar *edv_archive_comment_get(
	CfgList *cfg_list,
	const gchar *arch_path
)
{
	const gchar *arch_name;
	gchar *comment;
	EDVVFSObject *obj;

	if((cfg_list == NULL) || STRISEMPTY(arch_path))
	{
		errno = EINVAL;
		return(NULL);
	}

	arch_name = g_basename(arch_path);

	/* Check if the archive exists and is a file */
	obj = edv_vfs_object_stat(arch_path);
	if(obj == NULL)
		return(NULL);

	if(!EDV_VFS_OBJECT_IS_FILE(obj))
	{
		edv_vfs_object_delete(obj);
		errno = EINVAL;
		return(NULL);
	}

	/* ARJ Archive */
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		comment = edv_archive_comment_get_arj(
			cfg_list,
			arch_path
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		comment = edv_archive_comment_get_rar(
			cfg_list,
			arch_path
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		comment = edv_archive_comment_get_zip(
			cfg_list,
			arch_path
		);
	}
	else
	{
		comment = NULL;
		errno = EINVAL;
	}

	edv_vfs_object_delete(obj);

	return(comment);
}


/*
 *	Writes a comment file.
 */
static gint edv_archive_comment_set_write_comment_file(
	const gchar *path,
	const gchar *prefix,
	const gchar *postfix,
	const gchar *comment
)
{
	FILE *fp = fopen(path, "wb");
	if(fp == NULL)
		return(-1);

	if(!STRISEMPTY(prefix))
		fwrite(prefix, sizeof(gchar), strlen(prefix), fp);

	if(!STRISEMPTY(comment))
		fwrite(comment, sizeof(gchar), strlen(comment), fp);

	if(!STRISEMPTY(postfix))
		fwrite(postfix, sizeof(gchar), strlen(postfix), fp);

	if(fclose(fp))
		return(-1);

	return(0);
}

/*
 *	Set comment for the ARJ Archive.
 */
static gint edv_archive_comment_set_arj(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
)
{
	FILE		*cstdout,
			*cstderr;
	gint pid;
	gchar		*comment_path,
			*cmd;
	const gchar *prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);

	/* Create a new temporary file as the comment file */
	comment_path = edv_tmp_name(EDV_GET_S(EDV_CFG_PARM_DIR_TMP));
	if(comment_path == NULL)
		return(-1);

	/* Write the comment to a spool file for archive program to
	 * read from
	 */
	if(edv_archive_comment_set_write_comment_file(
		comment_path,
		NULL,				/* No prefix */
		NULL,				/* No postfix */
		comment				/* Comment */
	))
	{
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(-1);
	}

	/* Format the set archive comment from file command */
	cmd = g_strdup_printf(
		"\"%s\" c -y -i \"%s\" \"-z%s\"",
		prog_arj,
		arch_path,
		comment_path
	);
	if(cmd == NULL)
	{
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(-1);
	}

	/* Execute the set archive comment command */
	pid = edv_archive_comment_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(cmd);

	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(-1);
	}

	/* Wait for the set archive comment process to finish */
	while(edv_pid_exists(pid))
	{
		FDISCARD(cstdout);
		FDISCARD(cstderr);
		edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	/* Remove the comment file */
	(void)edv_unlink(comment_path);
	g_free(comment_path);

	return(0);
}

/*
 *      Set comment for the RAR Archive.
 */
static gint edv_archive_comment_set_rar(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
)
{
	FILE		*cstdout,
			*cstderr;
	gint pid;
	gchar		*comment_path,
			*cmd;
	const gchar *prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);

	/* Create a new temporary file as the comment file */
	comment_path = edv_tmp_name(EDV_GET_S(EDV_CFG_PARM_DIR_TMP));
	if(comment_path == NULL)
		return(-1);

	/* Write the comment to a spool file for archive program to
	 * read from
	 */
	if(edv_archive_comment_set_write_comment_file(
		comment_path,
		NULL,				/* No prefix */
		NULL,				/* No postfix */
		comment				/* Comment */
	))
	{
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(-1);
	}

	/* Format the set archive comment from file command */
	cmd = g_strdup_printf(
		"\"%s\" c -y -inul \"-z%s\" \"%s\"",
		prog_rar,
		comment_path,
		arch_path
	);

	/* Execute the set archive comment command */
	pid = edv_archive_comment_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(cmd);

	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(-1);
	}

	/* Wait for the set archive comment process to finish */
	while(edv_pid_exists(pid))
	{
		FDISCARD(cstdout);
		FDISCARD(cstderr);
		edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	/* Remove the comment file */
	(void)edv_unlink(comment_path);
	g_free(comment_path);

	return(0);
}

#ifdef HAVE_LIBZIP
/*
 *	Opens the PKZip archive for writing.
 *
 *	Returns the handle to the PKZip archive.
 */
static struct zip *edv_archive_comment_libzip_open_writing(
	const gchar *arch_path,
	gint *zip_error
)
{
	/* If the PKZip archive exists but has a size of 0 bytes then
	 * it must be deleted first or else libzip will return an
	 * error stating that it is not a valid PKZip archive
	 */
	EDVVFSObject *obj = edv_vfs_object_stat(arch_path);
	if(obj != NULL)
	{
		if(obj->size == 0l)
			(void)edv_unlink(arch_path);
		edv_vfs_object_delete(obj);
	}

	return(zip_open(
		arch_path,
		ZIP_CREATE,
		zip_error
	));
}
#endif	/* HAVE_LIBZIP */

/*
 *	Set comment for the PKZip archive.
 */
static gint edv_archive_comment_set_zip(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
)
{
#ifdef HAVE_LIBZIP
	struct zip *archive;
	gint zip_error;

	/* Open the PKZip archive for writing */
	archive = edv_archive_comment_libzip_open_writing(arch_path, &zip_error);
	if(archive == NULL)
		return(-1);

	/* Set the PKZip archive comment */
	if(zip_set_archive_comment(archive, comment, STRLEN(comment)))
	{
		zip_close(archive);
		return(-1);
	}

	/* Write/flush changes to the PKZip archive and close it */
	if(zip_close(archive))
		return(-1);

	return(0);
#else
	FILE		*cstdout,
			*cstderr;
	gint pid;
	gchar		*cat_prog,
			*comment_path,
			*cmd;
	const gchar *prog_zip = EDV_GET_S(EDV_CFG_PARM_PROG_ZIP);

	/* Create a new temporary file as the comment file */
	comment_path = edv_tmp_name(EDV_GET_S(EDV_CFG_PARM_DIR_TMP));
	if(comment_path == NULL)
		return(-1);

	/* Write the comment to a spool file for archive program to
	 * read from
	 */
	if(edv_archive_comment_set_write_comment_file(
		comment_path,
		NULL,                           /* No prefix */
		NULL,                           /* No postfix */
		comment                         /* Comment */
	))
	{
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(-1);
	}

	/* Find the cat program */
	cat_prog = edv_which("cat");

	/* Format the set archive comment from file command */
	cmd = g_strdup_printf(
		"\"%s\" \"%s\" | \"%s\" -z \"%s\"",
		cat_prog,
		comment_path,
		prog_zip,
		arch_path
	);

	g_free(cat_prog);

	/* Execute the set archive comment command */
	pid = edv_archive_comment_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(cmd);

	if(pid < 0)
	{
		(void)FCLOSE(cstdout);
		(void)FCLOSE(cstderr);
		(void)edv_unlink(comment_path);
		g_free(comment_path);
		return(-1);
	}

	/* Wait for the set archive comment process to finish */
	while(edv_pid_exists(pid))
	{
		FDISCARD(cstdout);
		FDISCARD(cstderr);
		edv_usleep(EDV_ITERATION_SLEEP_MIN_US);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	/* Remove the comment file */
	(void)edv_unlink(comment_path);
	g_free(comment_path);

	return(0);
#endif
}

/*
 *	Sets the archive's comment.
 *
 *	The arch_path specifies the archive.
 *
 *	The comment specifies the comment.
 */
gint edv_archive_comment_set(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *comment
)
{
	gint status;
	const gchar *arch_name;
	EDVVFSObject *obj;

	if((cfg_list == NULL) || STRISEMPTY(arch_path))
	{
		errno = EINVAL;
		return(-2);
	}

	arch_name = g_basename(arch_path);

	/* Check if the archive exists and is a file */
	obj = edv_vfs_object_stat(arch_path);
	if(obj == NULL)
		return(-1);

	if(!EDV_VFS_OBJECT_IS_FILE(obj))
	{
		edv_vfs_object_delete(obj);
		errno = EINVAL;
		return(-2);
	}

	/* ARJ Archive */
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		status = edv_archive_comment_set_arj(
			cfg_list,
			arch_path,
			comment
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		status = edv_archive_comment_set_rar(
			cfg_list,
			arch_path,
			comment
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		status = edv_archive_comment_set_zip(
			cfg_list,
			arch_path,
			comment
		);
	}
	else
	{
		status = -2;
		errno = EINVAL;
	}

	edv_vfs_object_delete(obj);

	return(status);
}
