#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#ifdef HAVE_LIBXAR
# include <xar/xar.h>
#endif
/* Do not use libzip to check archives because zip_open() will return
 * error immediately if there is a problem and give no verbose means
 * to check for errors
 */
#undef HAVE_LIBZIP
#ifdef HAVE_LIBZIP
# include <zip.h>
#endif
#include <gtk/gtk.h>

#include "cfg.h"

#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_stream.h"
#include "libendeavour2-base/edv_process.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "edv_archive_check.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_archive_check_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_archive_check_get_error(EDVCore *core);
static void edv_archive_check_set_error(
	EDVCore *core,
	const gchar *msg
);

/* Execute */
static gint edv_archive_check_execute(
        CfgList *cfg_list,
        const gchar *cmd,
        FILE **cstdin_rtn,
        FILE **cstdout_rtn,
        FILE **cstderr_rtn
);

/* Progress Dialog */
static gint edv_archive_check_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
);

/* Delete Object From Archive */
static gint edv_archive_check_arj(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_check_lha(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_check_rar(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_check_tar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);
#ifdef HAVE_LIBXAR
static gint edv_archive_check_xar_iterate(
	EDVCore *core,
	xar_t xar,
	xar_file_t xar_fp,
	const guint index,
	const gchar *path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gfloat progress
);
#endif
static gint edv_archive_check_xar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
static gint edv_archive_check_zip(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);
gint edv_archive_check(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	gboolean const show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define EDV_ITERATION_SLEEP_MIN_US	8000l

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define ISSPACE(c)	(((c) == ' ') || ((c) == '\t') ||       \
					 ((c) == '\v') || ((c) == '\f') ||      \
					 ((c) == '\r') || ((c) == '\n'))
#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))

#define STRCASEPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ?     \
 !g_strncasecmp((s), (p), strlen(p)) : FALSE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)
#define INTERRUPT(p)	(((p) > 0) ? (gint)kill((int)(p), SIGINT) : -1)

#define FDISCARD(p)	{			\
 if((p) != NULL) {				\
  const gint fd = (gint)fileno(p);		\
  while(edv_poll_read(fd)) {			\
   if(fgetc(p) == EOF)				\
    break;					\
  }						\
 }						\
}


/*
 *	Gets the last error message.
 */
const gchar *edv_archive_check_get_error(EDVCore *core)
{
	return((core != NULL) ? core->last_error_ptr : NULL);
}


/*
 *	Coppies the error message to the core's last_error_buf
 *	and sets the core's last_error_ptr to point to it.
 */
static void edv_archive_check_set_error(
	EDVCore *core,
	const gchar *msg
)
{
	if(core == NULL)
		return;

	g_free(core->last_error_buf);
	core->last_error_ptr = core->last_error_buf = STRDUP(msg);
}


/*
 *	Executes the command using the shell specified by the
 *	configuration and opens and returns the requested streams.
 */
static gint edv_archive_check_execute(
        CfgList *cfg_list,
        const gchar *cmd,
        FILE **cstdin_rtn,
        FILE **cstdout_rtn,
        FILE **cstderr_rtn
)
{
        gchar		*shell_prog;
        const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
                        *shell_args = edv_strarg(
                shell_cmd,
                &shell_prog,
                TRUE,				/* Parse escapes */
                TRUE				/* Parse quotes */
        );

        /* Execute the list archive command */
        gint pid = edv_system_shell_streams(
                cmd,
                shell_prog,
                shell_args,
                cstdin_rtn,
                cstdout_rtn,
                cstderr_rtn
        );

        g_free(shell_prog);

        return(pid);
}


/*
 *	Updates/maps the progress dialog.
 */
static gint edv_archive_check_update_progress_dialog(
	CfgList *cfg_list,
	const gchar *arch_path,
	const gchar *path,
	const gfloat progress,
	GtkWidget *toplevel,
	const gboolean force_remap
)
{
        if((arch_path != NULL) && (path != NULL))
        {
		gchar	*arch_path_shortened = edv_path_shorten(
				arch_path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*path_shortened = edv_path_shorten(
				path,
				EDV_PROGRESS_DLG_PATH_MAX_CHARS
		),
			*msg = g_strdup_printf(
"Checking:\n\
\n\
    %s\n\
\n\
In:\n\
\n\
    %s\n"
				,
				path_shortened,
				arch_path_shortened
		);
		g_free(arch_path_shortened);
		g_free(path_shortened);
	        if(ProgressDialogIsQuery())
		{
	                if(progress >= 0.0f)
	                        ProgressDialogUpdate(
	                                NULL,
	                                msg,
	                                NULL,
	                                NULL,
        	                        progress,
	                                EDV_PROGRESS_BAR_NTICKS,
	                                TRUE
	                        );
	                else
	                        ProgressDialogUpdateUnknown(
	                                NULL,
	                                msg,
	                                NULL,
	                                NULL,
	                                TRUE
	                        );
		}
		else
        	{
			ProgressDialogSetTransientFor(toplevel);
			ProgressDialogMap(
				"Checking",
				msg,
				(const guint8 **)icon_archive_check_32x32_xpm,
				"Stop"
			);
		}
                g_free(msg);
        }
        else
        {
                if(progress >= 0.0f)
                        ProgressDialogUpdate(
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                progress,
                                EDV_PROGRESS_BAR_NTICKS,
                                TRUE
                        );
                else
                        ProgressDialogUpdateUnknown(
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                TRUE
                        );
        }

        return(ProgressDialogStopCount());
}


/*
 *	Checks the ARJ archive.
 */
static gint edv_archive_check_arj(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint            status,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_arj = EDV_GET_S(EDV_CFG_PARM_PROG_ARJ);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the check archive command */
	cmd = g_strdup_printf(
		"\"%s\" t -i -y \"%s\"",
		prog_arj,
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the check archive command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the check archive command */
	pid = edv_archive_check_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the check archive command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gchar   *cstdout_buf = NULL,
			*cstderr_buf = NULL;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_check_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				-1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Update the progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_check_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					-1.0f,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Check if the process has finished */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Read the next line from standard error */
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				const gchar *s = cstderr_buf;
				while(ISBLANK(*s))
					s++;

				/* Record the first error message */
				if(!STRISEMPTY(s) &&
				   (core->last_error_ptr == NULL)
				)
				if(!STRISEMPTY(s))
				{
					edv_archive_check_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
					       status = -1;
				}

				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}

			/* Read the next line from standard output */
			if(edv_stream_read_strptrbrk(
				cstdout,
				&cstdout_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = cstdout_buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Testing"))
				{
					gchar *s2;
					const gchar *path;

					/* Seek s past the prefix */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					/* Cap the space character after path */
					s2 = strstr(s, "  ");
					if(s2 == NULL)
						s2 = strpbrk(s, " \t\r\n");
					if(s2 != NULL)
						*s2 = '\0';

					path = s;

					if(show_progress && !STRISEMPTY(path))
					{
						if(edv_archive_check_update_progress_dialog(
							cfg_list,
							arch_path,
							path,
							-1.0f,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}
				}
				else if(STRCASEPFX(s, "Error"))
				{
					/* Seek s past the prefix */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					edv_archive_check_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
						status = -1;
				}

				g_free(cstdout_buf);
				cstdout_buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(cstdout) || !feof(cstderr));

		g_free(cstdout_buf);
		g_free(cstderr_buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Checks the LHA archive.
 */
static gint edv_archive_check_lha(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint            status,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_lha = EDV_GET_S(EDV_CFG_PARM_PROG_LHA);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the check archive command */
	cmd = g_strdup_printf(
		"\"%s\" -tq1 \"%s\"",
		prog_lha,
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the check archive command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the check archive command */
	pid = edv_archive_check_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the check archive command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gchar   *cstdout_buf = NULL,
			*cstderr_buf = NULL;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_check_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				-1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Update the progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_check_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					-1.0f,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Check if the process has finished */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Read the next line from standard error */
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				const gchar *s = cstderr_buf;
				while(ISBLANK(*s))
					s++;

				/* Record the first error message */
				if(!STRISEMPTY(s) &&
				   (core->last_error_ptr == NULL)
				)
				{
					edv_archive_check_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
					       status = -1;
				}

				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}

			/* Read the next line from standard output */
			if(edv_stream_read_strptrbrk(
				cstdout,
				&cstdout_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				gchar	*s = cstdout_buf,
					*s2;
				const gchar *path;
				while(ISBLANK(*s))
					s++;

				/* Cap the space character after the path */
				s2 = strpbrk(s, " \t\n\r");
				if(s2 != NULL)
					*s2 = '\0';

				path = s;

				/* Update the progress dialog's label? */
				if(show_progress && !STRISEMPTY(path))
				{
					if(edv_archive_check_update_progress_dialog(
						cfg_list,
						arch_path,
						path,
						-1.0f,
						toplevel,
						FALSE
					) > 0)
					{
						(void)INTERRUPT(pid);
						pid = 0;
						status = -4;
						break;
					}
				}

				g_free(cstdout_buf);
				cstdout_buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(cstdout) || !feof(cstderr));

		g_free(cstdout_buf);
		g_free(cstderr_buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Checks the RAR archive.
 */
static gint edv_archive_check_rar(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint            status,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_rar = EDV_GET_S(EDV_CFG_PARM_PROG_RAR);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the check archive command */
	cmd = g_strdup_printf(
		"\"%s\" t \"-p%s\" -y \"%s\"",
		prog_rar,
		(STRISEMPTY(password)) ? "-" : password,
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the check archive command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the check archive command */
	pid = edv_archive_check_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the check archive command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gchar   *cstdout_buf = NULL,
			*cstderr_buf = NULL;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_check_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				-1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Update the progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_check_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					-1.0f,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Check if the process has finished */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Read the next line from standard error */
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				const gchar *s = cstderr_buf;
				while(ISBLANK(*s))
					s++;

				/* Record the first error message */
				if(!STRISEMPTY(s) &&
				   (core->last_error_ptr == NULL)
				)
				{
					edv_archive_check_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
					       status = -1;
				}

				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}

			/* Read the next line from standard output */
			if(edv_stream_read_strptrbrk(
				cstdout,
				&cstdout_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = cstdout_buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "Testing  "))
				{
					gchar *s2;
					const gchar *path;

					/* Seek s past the prefix */
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					/* Cap the space character after the path */
					s2 = strstr(s, "  ");
					if(s2 == NULL)
						s2 = strpbrk(s, " \t\r\n");
					if(s2 != NULL)
						*s2 = '\0';

					path = s;

					/* Update the progress dialog's label? */
					if(show_progress && !STRISEMPTY(path))
					{
						if(edv_archive_check_update_progress_dialog(
							cfg_list,
							arch_path,
							path,
							-1.0f,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}
				}

				g_free(cstdout_buf);
				cstdout_buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(cstdout) || !feof(cstderr));

		g_free(cstdout_buf);
		g_free(cstderr_buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Checks the Tape archive.
 */
static gint edv_archive_check_tar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint            status,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar	*prog_tar = EDV_GET_S(EDV_CFG_PARM_PROG_TAR),
			*prog_uncompress = EDV_GET_S(EDV_CFG_PARM_PROG_UNCOMPRESS),
			*prog_gunzip = EDV_GET_S(EDV_CFG_PARM_PROG_GUNZIP),
			*prog_bunzip2 = EDV_GET_S(EDV_CFG_PARM_PROG_BUNZIP2);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the check archive command */
	if(is_compress_compressed)
		cmd = g_strdup_printf(
			"\"%s\" \"--use-compress-program=%s\" -t -f \"%s\"",
			prog_tar,
			prog_uncompress,
			arch_path
		);
	else if(is_gzip_compressed)
		cmd = g_strdup_printf(
			"\"%s\" \"--use-compress-program=%s\" -t -f \"%s\"",
			prog_tar,
			prog_gunzip,
			arch_path
		);
	else if(is_bzip2_compressed)
		cmd = g_strdup_printf(
			"\"%s\" \"--use-compress-program=%s\" -t -f \"%s\"",
			prog_tar,
			prog_bunzip2,
			arch_path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" -t -f \"%s\"",
			prog_tar,
			arch_path
		);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the check archive command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the check archive command */
	pid = edv_archive_check_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the check archive command.";
		CLEANUP_RETURN(-1);
	}

	/* Delete the command */
	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gchar   *cstdout_buf = NULL,
			*cstderr_buf = NULL;
		do {
			/* Update the progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_check_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					-1.0f,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Check if the process has finished */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Read the next line from standard error */
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				const gchar *s = cstderr_buf;
				while(ISBLANK(*s))
					s++;

				/* Record the first error message */
				if(!STRISEMPTY(s) &&
				   (core->last_error_ptr == NULL)
				)
				{
					edv_archive_check_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
					       status = -1;
				}

				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}

			/* Read the next line from standard output */
			if(edv_stream_read_strptrbrk(
				cstdout,
				&cstdout_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				gchar	*s = cstdout_buf,
					*path;
				while(ISBLANK(*s))
					s++;

				path = s;

				/* Update the progress dialog's label? */
				if(show_progress && !STRISEMPTY(path))
				{
					if(edv_archive_check_update_progress_dialog(
						cfg_list,
						arch_path,
						path,
						-1.0f,
						toplevel,
						FALSE
					) > 0)
					{
						(void)INTERRUPT(pid);
						pid = 0;
						status = -4;
						break;
					}
				}

				g_free(cstdout_buf);
				cstdout_buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(cstdout) || !feof(cstderr));

		g_free(cstdout_buf);
		g_free(cstderr_buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

#ifdef HAVE_LIBXAR
static gint edv_archive_check_xar_iterate(
	EDVCore *core,
	xar_t xar,
	xar_file_t xar_fp,
	const guint index,
	const gchar *path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gfloat progress
)
{
	gint status;
	const gchar	*key,
			*val;
	xar_iter_t xar_iter;
	CfgList *cfg_list = core->cfg_list;

	/* Create a new X Archive property iterator */
	xar_iter = xar_iter_new(xar);
	if(xar_iter == 0)
	{
		edv_archive_check_set_error(
			core,
"Unable to create a new X Archive iterator"
		);
		return(-3);
	}

	status = 0;

	/* Iterate through each property and add each property's
	 * value to the archive object
	 */
	for(key = (const gchar *)xar_prop_first(xar_fp, xar_iter);
	    key != NULL;
	    key = (const gchar *)xar_prop_next(xar_iter)
	)
	{
		/* Update the progress */
		if(show_progress && ProgressDialogIsQuery())
		{
			if(edv_archive_check_update_progress_dialog(
				cfg_list,
				NULL,
				NULL,
				progress,
				toplevel,
				FALSE
			) > 0)
			{
				status = -4;
				break;
			}
		}

		val = NULL;
		if(xar_prop_get(
			xar_fp,
			(const char *)key,
			(const char **)&val
		) != 0)
		{
			const gint error_code = (gint)errno;
			edv_archive_check_set_error(
				core,
				g_strerror(error_code)
			);
			status = -1;
			break;
		}

		/* If the property has no value then skip it */
		if(STRISEMPTY(val))
			continue;
	}

	/* Update the progress */
	if(show_progress && ProgressDialogIsQuery())
	{
		if(edv_archive_check_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			progress,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}

	/* Delete the X Archive property iterator */
	xar_iter_free(xar_iter);

	/* Generate a temporary output file name to extract to and test
	 * for any errors
	 */
	if(status == 0)
	{
		gchar *tar_path = edv_tmp_name(EDV_GET_S(EDV_CFG_PARM_DIR_TMP));
		if(tar_path != NULL)
		{
			if(xar_extract_tofile(
				xar,
				xar_fp,
				tar_path
			) != 0)
			{
				const gint error_code = (gint)errno;
				edv_archive_check_set_error(
					core,
					g_strerror(error_code)
				);
				status = -1;
			}
			(void)edv_unlink(tar_path);
			g_free(tar_path);
		}
	}

	return(status);
}
#endif	/* HAVE_LIBXAR */

/*
 *	Checks the X Archive.
 */
static gint edv_archive_check_xar(
	EDVCore *core,
	const gchar *arch_path,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
#ifdef HAVE_LIBXAR
	xar_t xar;
	xar_iter_t i1;
	xar_file_t xar_fp;
	gint            i,
			status,
			status2,
			nobjs;
	gchar *path;
	CfgList *cfg_list = core->cfg_list;

	/* Open the X Archive for reading */
	xar = xar_open(arch_path, READ);
	if(xar == NULL)
	{
		edv_archive_check_set_error(
			core,
"Unable to open the file for reading"
		);
		return(-1);
	}

	/* Create a new X Archive object iterator */
	i1 = xar_iter_new(xar);
	if(i1 == 0)
	{
		(void)xar_close(xar);
		edv_archive_check_set_error(
			core,
"Unable to create a new X Archive iterator"
		);
		return(-3);
	}

	/* Count the number of objects in the X Archive */
	for(xar_fp = xar_file_first(xar, i1), nobjs = 0;
	    xar_fp != NULL;
	    xar_fp = xar_file_next(i1), nobjs++
	);

	/* Iterate through each object in the X Archive */
	status = 0;
	for(xar_fp = xar_file_first(xar, i1),
	    i = 0;
	    xar_fp != NULL;
	    xar_fp = xar_file_next(i1),
	    i++
	)
	{
		/* Get the path of this object within the X Archive */
		path = (gchar *)xar_get_path(xar_fp);
		if(path == NULL)
			continue;

		/* Update the progress dialog to display the current
		 * object being extracted?
		 */
		if(show_progress)
		{
			if(edv_archive_check_update_progress_dialog(
				cfg_list,
				arch_path,
				path,
				(nobjs > 0l) ?
					((gfloat)i / (gfloat)nobjs) : 0.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(path);
				status = -4;
				break;
			}
		}

		/* Check this object */
		status2 = edv_archive_check_xar_iterate(
			core,
			xar,
			xar_fp,
			i,
			path,
			toplevel,
			show_progress,
			(nobjs > 0l) ?
				((gfloat)i / (gfloat)nobjs) : 0.0f
		);
		if(status2 != 0)
		{
			if((status == 0) || (status == -4))
				status = status2;
		}

		g_free(path);

		if(status != 0)
			break;
	}

	/* Delete the iterator and close the X Archive */
	xar_iter_free(i1);
	(void)xar_close(xar);

	return(status);
#else
	return(-2);
#endif
}

/*
 *	Checks the PKZip archive.
 */
static gint edv_archive_check_zip(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
#ifdef HAVE_LIBZIP
	struct zip *archive;
	struct zip_stat zip_stat_buf;
	struct zip_file *zip_fp;
	gint		status,
			i,
			nobjs,
			len,
			read_buf_len,
			bytes_read,
			zip_error_code;
	gulong		cur_size,
			total_size;
	const gchar *name;
	gchar *clean_name;
	guint8 *read_buf;
	EDVObjectType type;
	EDVVFSObject *arch_obj;

	arch_obj = edv_vfs_object_stat(arch_path);
	if(arch_obj == NULL)
	{
		const gint error_code = (int)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			arch_path
		);
		edv_archive_check_set_error(core, msg);
		g_free(msg);
		return(-1);
	}

	/* Open the PKZip archive for reading */
	archive = zip_open(arch_path, 0, &zip_error_code);
	if(archive == NULL)
	{
		const gint sys_error_code = (gint)errno;
		gchar *msg, err_msg[1024];
		zip_error_to_str(
			err_msg, sizeof(err_msg),
			zip_error_code, sys_error_code
		);
		msg = g_strdup_printf(
"Unable to open the PKZip Archive for reading:\n\
\n\
    %s\n\
\n\
%s.",
			arch_path,
			err_msg
		);
		edv_archive_check_set_error(core, msg);
		g_free(msg);
		edv_vfs_object_delete(arch_obj);
		return(-1);
	}

	/* Get the total number of objects in the archive */
	nobjs = zip_get_num_files(archive);

	/* Iterate through each object and calculate the total size */
	total_size = 0l;
	for(i = 0; i < nobjs; i++)
	{
		if(zip_stat_index(archive, i, 0, &zip_stat_buf) == 0)
			total_size += (gulong)zip_stat_buf.size;
	}

	/* Iterate through each object */
	status = 0;
	cur_size = 0l;
	for(i = 0; i < nobjs; i++)
	{
		if(zip_stat_index(archive, i, 0, &zip_stat_buf))
		{
			gchar *msg = g_strdup_printf(
"Unable to obtain the statistics for the PKZip Archive object #%i.\n\
\n\
%s.",
				i + 1,
				zip_strerror(archive)
			);
			edv_archive_check_set_error(core, msg);
			g_free(msg);
			status = -1;
			break;
		}

		/* Get the object's name, clean name (without the type
		 * postfix character), and type
		 */
		name = zip_stat_buf.name;
		clean_name = STRDUP(name);
		len = STRLEN(clean_name);
		switch((len > 1) ? clean_name[len - 1] : '\0')
		{
		  case G_DIR_SEPARATOR:
			type = EDV_OBJECT_TYPE_DIRECTORY;
			clean_name[len - 1] = '\0';
			break;
		  case '@':
			type = EDV_OBJECT_TYPE_LINK;
			clean_name[len - 1] = '\0';
			break;
		  default:
			type = EDV_OBJECT_TYPE_FILE;
			break;
		}

		/* Update the progress dialog to display the current
		 * object being extracted?
		 */
		if(show_progress)
		{
			if(edv_archive_check_update_progress_dialog(
				cfg_list,
				arch_path,
				clean_name,
				(total_size > 0l) ?
					((gfloat)cur_size / (gfloat)total_size) : -1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				g_free(clean_name);
				status = -4;
				break;
			}
		}

		/* Open the file in the PKZip archive */
		zip_fp = zip_fopen_index(archive, i, 0);
		if(zip_fp == NULL)
		{
			gchar *msg = g_strdup_printf(
"Unable to open the PKZip Archive object #%i for reading.\n\
\n\
%s.",
				i + 1, zip_strerror(archive)
			);
			edv_archive_check_set_error(core, msg);
			g_free(msg);
			g_free(clean_name);
			status = -1;
			break;
		}

		/* Allocate the read buffer */
		read_buf_len = MAX((gint)arch_obj->block_size, 1);
		read_buf = (guint8 *)g_malloc(read_buf_len * sizeof(guint8));
		if(read_buf == NULL)
		{
			core->last_error_ptr = "Memory allocation error.";
			g_free(clean_name);
			zip_fclose(zip_fp);
			status = -3;
			break;
		}

		/* Read the file in the PKZip archive and check for errors */
		while(TRUE)
		{
			bytes_read = (gint)zip_fread(
				zip_fp,
				read_buf,
				(int)read_buf_len
			);
			if(bytes_read <= 0)
			{
				if(bytes_read < 0)
				{
					gchar *msg = g_strdup_printf(
"Unable to read from the PKZip Archive object #%i.\n\
\n\
%s.",
						i + 1,
						zip_file_strerror(zip_fp)
					);
					edv_archive_check_set_error(core, msg);
					g_free(msg);
					status = -1;
				}
				break; 
			}

			cur_size += (gulong)bytes_read;

			/* Update progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_check_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					(total_size > 0l) ?
						((gfloat)cur_size / (gfloat)total_size) : 0.0f,
					toplevel,
					FALSE
				) > 0)
				{
					status = -4;
					break;
				}
			}
		}

		/* Delete the read buffer */
		g_free(read_buf);

		/* Close the file in the PKZip archive */
		zip_fclose(zip_fp);

		g_free(clean_name);

		if(status != 0)
			break;
	}

	/* Close the PKZip archive */
	if(zip_close(archive))
	{
		if((status == 0) || (status == -4))
		{
			gchar *msg = g_strdup_printf(
"Unable to close the PKZip Archive:\n\
\n\
    %s\n\
\n\
%s.",
				arch_path, zip_strerror(archive)
			);
			edv_archive_check_set_error(core, msg);
			g_free(msg);
			status = -1;
		}
	}

	/* Report the final progress */
	if(show_progress && ProgressDialogIsQuery() && (status == 0))
	{
		if(edv_archive_check_update_progress_dialog(
			cfg_list,
			NULL,
			NULL,
			(total_size > 0l) ?
				((gfloat)cur_size / (gfloat)total_size) : 0.0f,
			toplevel,
			FALSE
		) > 0)
			status = -4;
	}    

	edv_vfs_object_delete(arch_obj);

	return(status);
#else
	FILE            *cstdout = NULL,
			*cstderr = NULL;
	gint            status,
			pid;
	gchar *cmd = NULL;
	CfgList *cfg_list = core->cfg_list;
	const gchar *prog_zip = EDV_GET_S(EDV_CFG_PARM_PROG_ZIP);

#define CLEANUP_RETURN(_v_)	{	\
 (void)FCLOSE(cstdout);			\
 (void)FCLOSE(cstderr);			\
 g_free(cmd);				\
					\
 return(_v_);				\
}

	/* Format the check archive command */
	cmd = g_strdup_printf(
		"\"%s\" -T -v \"%s\"",
		prog_zip,
		arch_path
	);
	if(cmd == NULL)
	{
		core->last_error_ptr = "Unable to generate the check archive command.";
		CLEANUP_RETURN(-1);
	}

	/* Execute the check archive command */
	pid = edv_archive_check_execute(
		cfg_list,
		cmd,
		NULL,
		&cstdout,
		&cstderr
	);
	if(pid < 0)
	{
		core->last_error_ptr = "Unable to execute the check archive command.";
		CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	status = 0;

	/* Read from the output streams */
	if((cstdout != NULL) && (cstderr != NULL))
	{
		gchar   *cstdout_buf = NULL,
			*cstderr_buf = NULL;

		/* Map the progress dialog? */
		if(show_progress)
		{
			if(edv_archive_check_update_progress_dialog(
				cfg_list,
				arch_path,
				"",
				-1.0f,
				toplevel,
				FALSE
			) > 0)
			{
				(void)INTERRUPT(pid);
				pid = 0;
				status = -4;
			}
		}

		do {
			/* Update the progress */
			if(show_progress && ProgressDialogIsQuery())
			{
				if(edv_archive_check_update_progress_dialog(
					cfg_list,
					NULL,
					NULL,
					-1.0f,
					toplevel,
					FALSE
				) > 0)
				{
					(void)INTERRUPT(pid);
					pid = 0;
					status = -4;
					break;
				}
			}

			/* Check if the process has finished */
			if(pid > 0)
			{
				if(!edv_pid_exists(pid))
					pid = 0;
			}

			/* Read the next line from standard error */
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				const gchar *s = cstderr_buf;
				while(ISBLANK(*s))
					s++;

				/* Record the first error message */
				if(!STRISEMPTY(s) &&
				   (core->last_error_ptr == NULL)
				)
				{
					edv_archive_check_set_error(
						core,
						s
					);

					if((status == 0) || (status == -4))
					       status = -1;
				}

				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}

			/* Read the next line from standard output */
			if(edv_stream_read_strptrbrk(
				cstdout,
				&cstdout_buf,
				"\n",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				gchar *s = cstdout_buf;
				while(ISBLANK(*s))
					s++;

				if(STRCASEPFX(s, "testing:"))
				{
					gchar *s2;
					const gchar *path;

					while(ISBLANK(*s))
						s++;
					while(!ISBLANK(*s) && (*s != '\0'))
						s++;
					while(ISBLANK(*s))
						s++;

					s2 = strstr(s, "  ");
					if(s2 == NULL)
						s2 = strpbrk(s, " \t\n\r");
					if(s2 != NULL)
						*s2 = '\0';

					path = s;

					/* Update the progress dialog's label? */
					if(show_progress && !STRISEMPTY(path))
					{
						if(edv_archive_check_update_progress_dialog(
							cfg_list,
							arch_path,
							path,
							-1.0f,
							toplevel,
							FALSE
						) > 0)
						{
							(void)INTERRUPT(pid);
							pid = 0;
							status = -4;
							break;
						}
					}
				}

				g_free(cstdout_buf);
				cstdout_buf = NULL;
			}

			edv_usleep(EDV_ITERATION_SLEEP_MIN_US);

		} while((pid > 0) || !feof(cstdout) || !feof(cstderr));

		g_free(cstdout_buf);
		g_free(cstderr_buf);
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	cstdout = NULL;
	(void)FCLOSE(cstderr);
	cstderr = NULL;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBZIP */
}

/*
 *	Checks the archive.
 *
 *	The arch_path specifies the archive.
 */
gint edv_archive_check(
	EDVCore *core,
	const gchar *arch_path,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
/*	const gulong time_start = (gulong)time(NULL); */
	gint status;
	const gchar *arch_name;

#define CLEANUP_RETURN(_v_)	{	\
									\
 return(_v_);				\
}

	if(ProgressDialogIsQuery())
	{
		edv_archive_check_set_error(
			core,
"An operation is already in progress, please try again later."
		);
		return(-6);
	}

	/* Clear the last error message */
	edv_archive_check_set_error(core, NULL);

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   (yes_to_all == NULL)
	)
	{
		edv_archive_check_set_error(core, "Invalid value.");
		return(-2);
	}

	/* Begin deleting the source object from the archive object
	 * arch_path. The deleting method will be determined by taking
	 * the extension of the archive object's name
	 */

	/* ARJ Archive */
	arch_name = g_basename(arch_path);
	if(edv_name_has_extension(arch_name, ".arj"))
	{
		status = edv_archive_check_arj(
			core,
			arch_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	/* LHA Archive */
	else if(edv_name_has_extension(arch_name, ".lha"))
	{
		status = edv_archive_check_lha(
			core,
			arch_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	/* RAR Archive */
	else if(edv_name_has_extension(arch_name, ".rar"))
	{
		status = edv_archive_check_rar(
			core,
			arch_path,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	/* Tape Archive (Compressed) */
	else if(edv_name_has_extension(arch_name, ".tar.Z"))
	{
		status = edv_archive_check_tar(
			core,
			arch_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			TRUE,				/* Is compress compressed */
			FALSE,				/* Not gzip compressed */
			FALSE				/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (GZip) */
	else if(edv_name_has_extension(arch_name, ".tgz .tar.gz"))
	{
		status = edv_archive_check_tar(
			core,
			arch_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			FALSE,				/* Not compress compressed */
			TRUE,				/* Is gzip compressed */
			FALSE				/* Not bzip2 compressed */
		);
	}
	/* Tape Archive (BZip2) */
	else if(edv_name_has_extension(arch_name, ".tar.bz2"))
	{
		status = edv_archive_check_tar(
			core,
			arch_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			FALSE,				/* Not compress compressed */
			FALSE,				/* Not gzip compressed */
			TRUE				/* Is bzip2 compressed */
		);
	}
	/* Tape Archive */
	else if(edv_name_has_extension(arch_name, ".tar"))
	{
		status = edv_archive_check_tar(
			core,
			arch_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all,
			FALSE,				/* Not compress compressed */
			FALSE,				/* Not gzip compressed */
			FALSE				/* Not bzip2 compressed */
		);
	}
	/* X Archive */
	else if(edv_name_has_extension(arch_name, ".xar"))
	{
		status = edv_archive_check_xar(
			core,
			arch_path,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	/* PKZip Archive */
	else if(edv_name_has_extension(arch_name, ".zip .xpi .jar"))
	{
		status = edv_archive_check_zip(
			core,
			arch_path,
			password,
			toplevel,
			show_progress,
			interactive,
			yes_to_all
		);
	}
	else
	{
		core->last_error_ptr = "Unsupported archive format.";
		status = -2;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
